package org.eclipse.swt.custom;

import java.util.*;

import org.eclipse.swt.events.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

/**
 * @author Marc Strapetz
 */
public class QStyledText extends StyledText {

	// Fields =================================================================

	private final Map<Integer, Action> keyToAction = new HashMap<>();

	private int[] lineSpacings;
	private int[] cachedLineYOffsets;

	private boolean handlingAction;

	// Setup ==================================================================

	public QStyledText(Composite parent, int style) {
		super(parent, style);

		renderer.dispose();
		renderer = new Renderer(getDisplay(), this);
		renderer.setContent(content);
		renderer.setFont(getFont(), tabLength);

		addVerifyListener(new VerifyListener() {
			@Override
			public void verifyText(VerifyEvent e) {
				invalidateCaches();
			}
		});
	}

	// Implemented ============================================================

	@Override
	public void setText(String text) {
		super.setText(text);
		invalidateCaches();
	}

	@Override
	public void invokeAction(int actionId) {
		if (!handlingAction) {
			final Action action = keyToAction.get(actionId);
			if (action != null) {
				handlingAction = true;
				try {
					action.perform(this);
				}
				finally {
					handlingAction = false;
				}
				return;
			}
		}

		super.invokeAction(actionId);
	}

	// Accessing ==============================================================

	public int getLineHeightAndSpacing(int lineIndex) {
		return renderer.getLineHeight(lineIndex);
	}

	public int getLineSpacing(int lineIndex) {
		if (lineSpacings == null || lineIndex >= lineSpacings.length) {
			return getLineSpacing();
		}

		return lineSpacings[lineIndex];
	}

	public void setLineSpacing(int lineIndex, int lineSpacing) {
		setVariableLineHeight();

		final int lineCount = getLineCount();
		if (lineSpacings == null) {
			lineSpacings = new int[lineCount];
		}
		else if (lineSpacings.length != lineCount) {
			final int[] newLineSpacings = new int[lineCount];
			System.arraycopy(lineSpacings, 0, newLineSpacings, 0, Math.min(lineSpacings.length, newLineSpacings.length));
			lineSpacings = newLineSpacings;
		}

		lineSpacings[lineIndex] = lineSpacing;
	}

	public void clearLineSpacings() {
		lineSpacings = null;

		if (fixedLineHeight) {
			return;
		}
		fixedLineHeight = true;
		renderer.calculateIdle();
	}

	public final int getLineHeight(int lineIndex, boolean withSpacing) {
		if (!refreshCaches()) {
			return getLineHeight() + (withSpacing ? getLineSpacing() : 0);
		}

		int height = cachedLineYOffsets[lineIndex + 1] - cachedLineYOffsets[lineIndex];
		if (!withSpacing) {
			height -= lineSpacings != null && lineIndex < lineSpacings.length ? lineSpacings[lineIndex] : getLineSpacing();
		}

		return height;
	}

	public int getOverallHeight() {
		final int lineCount = getLineCount();
		if (!refreshCaches()) {
			return (getLineHeight() + getLineSpacing()) * lineCount;
		}

		return cachedLineYOffsets[lineCount];
	}

	public void registerAction(int key, Action action) {
		if (action == null) {
			throw new NullPointerException();
		}

		keyToAction.put(key, action);
		setKeyBinding(key, key);
	}

	// Utils ==================================================================

	private boolean refreshCaches() {
		if (lineSpacings == null) {
			cachedLineYOffsets = null;
			return false;
		}

		if (cachedLineYOffsets != null) {
			return true;
		}

		final int lineCount = getLineCount();
		cachedLineYOffsets = new int[lineCount + 1];

		cachedLineYOffsets[0] = 0;
		for (int lineIndex = 0; lineIndex < lineCount; lineIndex++) {
			final int lineHeight = getLineHeight(getOffsetAtLine(lineIndex));
			final int lineSpacing = lineIndex < lineSpacings.length ? lineSpacings[lineIndex] : getLineSpacing();
			cachedLineYOffsets[lineIndex + 1] = cachedLineYOffsets[lineIndex] + lineHeight + lineSpacing;
		}

		return true;
	}

	private void invalidateCaches() {
		cachedLineYOffsets = null;
	}

	// Inner Classes ==========================================================

	private class Renderer extends StyledTextRenderer {
		private Renderer(Device device, StyledText styledText) {
			super(device, styledText);
		}

		@Override
		TextLayout getTextLayout(int lineIndex) {
			int lineSpacing = styledText.lineSpacing;
			if (lineSpacings != null && lineIndex < lineSpacings.length) {
				lineSpacing = lineSpacings[lineIndex];
			}

			return getTextLayout(lineIndex, getOrientation(), getWrapWidth(), lineSpacing);
		}
	}

	public interface Action {
		void perform(StyledText text);
	}
}