/*******************************************************************************
 * Copyright (c) 2005, 2013 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Hannes Erven <hannes@erven.at> - Bug 293841 - [FieldAssist] NumLock keyDown event should not close the proposal popup [with patch]
 *     Syntevo GmbH    - major refactoring
 *******************************************************************************/
package com.syntevo.q.swt;

import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;

final class ContentProposalPopup {

	// Fields =================================================================

	private final QContentProposalAdapter proposalAdapter;
	private final Control control;
	private final QControlContentAdapter contentAdapter;
	private final Shell parentShell;
	private final Shell shell;
	private final PopupCloserListener popupCloser;
	private final Listener parentDeactivateListener;
	private final Table table;

	private List<? extends QContentProposal> proposals;
	private boolean listenToDeactivate;
	private boolean listenToParentDeactivate;

	// Setup ==================================================================

	/**
	 * Constructs a new instance of this popup, specifying the control for
	 * which this popup is showing content, and how the proposals should be
	 * obtained and displayed.
	 */
	public ContentProposalPopup(QContentProposalAdapter proposalAdapter, Control control, QControlContentAdapter contentAdapter, List<? extends QContentProposal> proposals) {
		this.proposalAdapter = proposalAdapter;
		this.control = control;
		this.contentAdapter = contentAdapter;
		this.proposals = proposals;

		parentShell = control.getShell();

		shell = new Shell(parentShell, SWT.NO_FOCUS | SWT.ON_TOP);

		final Listener listener = new Listener() {
			@Override
			public void handleEvent(Event event) {
				handleShellEvent(event);
			}
		};
		shell.addListener(SWT.Activate, listener);
		shell.addListener(SWT.Deactivate, listener);
		shell.addListener(SWT.Close, listener);

		parentDeactivateListener = new Listener() {
			@Override
			public void handleEvent(Event event) {
				if (listenToParentDeactivate) {
					asyncClose();
				}
				else {
					// Our first deactivate, now start listening on the Mac.
					listenToParentDeactivate = listenToDeactivate;
				}
			}
		};
		parentShell.addListener(SWT.Deactivate, parentDeactivateListener);

		final GridLayout layout = new GridLayout();
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		shell.setLayout(layout);

		table = new Table(shell, SWT.H_SCROLL | SWT.V_SCROLL);
		table.setHeaderVisible(false);
		table.addListener(SWT.DefaultSelection, new Listener() {
			@Override
			public void handleEvent(Event event) {
				final QContentProposal proposal = getSelectedProposal();
				acceptProposal(proposal);
			}
		});

		updateTable();

		shell.setVisible(true);

		popupCloser = new PopupCloserListener();
		// Listeners on this popup's table and scroll bar
		table.addListener(SWT.FocusOut, popupCloser);
		final ScrollBar scrollbar = table.getVerticalBar();
		if (scrollbar != null) {
			scrollbar.addListener(SWT.Selection, popupCloser);
		}

		// Listeners on this popup's shell
		shell.addListener(SWT.Deactivate, popupCloser);
		shell.addListener(SWT.Close, popupCloser);

		// Listeners on the target control
		control.addListener(SWT.MouseDoubleClick, popupCloser);
		control.addListener(SWT.MouseDown, popupCloser);
		control.addListener(SWT.Dispose, popupCloser);
		control.addListener(SWT.FocusOut, popupCloser);
		// Listeners on the target control's shell
		parentShell.addListener(SWT.Move, popupCloser);
		parentShell.addListener(SWT.Resize, popupCloser);
		parentShell.addListener(SWT.Deactivate, popupCloser);
	}

	// Accessing ==============================================================

	public void handleTraverseAndKeyDownEvent(Event e) {
		if (table.isDisposed()) {
			return;
		}

		final char key = e.character;

		// Traverse events are handled depending on whether the
		// event has a character.
		if (e.type == SWT.Traverse) {
			// let TAB work as normally
			if (key == SWT.TAB) {
				return;
			}

			// If the traverse event contains a legitimate character,
			// then we must set doit false so that the widget will
			// receive the key event. We return immediately so that
			// the character is handled only in the key event.
			// See https://bugs.eclipse.org/bugs/show_bug.cgi?id=132101
			if (key != 0) {
				e.doit = false;
				return;
			}
			// Traversal does not contain a character. Set doit true
			// to indicate TRAVERSE_NONE will occur and that no key
			// event will be triggered. We will check for navigation
			// keys below.
			e.detail = SWT.TRAVERSE_NONE;
			e.doit = true;
		}
		else {
			// Default is to only propagate when configured that way.
			// Some keys will always set doit to false anyway.
			e.doit = true;
		}

		// Check for special keys involved in cancelling, accepting, or
		// filtering the proposals.
		switch (key) {
		case 0:
			// No character. Check for navigation keys.
			handleNoCharacter(e);
			break;

		case SWT.ESC:
			e.doit = false;
			close();
			break;

		case SWT.LF:
		case SWT.CR:
			e.doit = false;
			final QContentProposal proposal = getSelectedProposal();
			if (proposal != null) {
				acceptProposal(proposal);
			}
			else {
				close();
			}
			break;

		case SWT.BS:
			// There is no filtering provided by us, but some
			// clients provide their own filtering based on content.
			// Recompute the proposals if the cursor position
			// will change (is not at 0).
			final int pos = contentAdapter.getCaretPosition();
			// We rely on the fact that the contents and pos do not yet
			// reflect the result of the BS. If the contents were
			// already empty, then BS should not cause
			// a recompute.
			if (pos > 0) {
				asyncRecomputeProposals();
			}
			break;

		default:
			// If the key is a defined unicode character, and not one of
			// the special cases processed above, update the filter text
			// and filter the proposals.
			if (Character.isDefined(key)) {
				// Recompute proposals after processing this event.
				asyncRecomputeProposals();
			}
			break;
		}
	}

	public Shell getShell() {
		return shell;
	}

	public void close() {
		if (!table.isDisposed()) {
			table.removeListener(SWT.FocusOut, popupCloser);
			final ScrollBar scrollbar = table.getVerticalBar();
			if (scrollbar != null) {
				scrollbar.removeListener(SWT.Selection, popupCloser);
			}

			shell.removeListener(SWT.Deactivate, popupCloser);
			shell.removeListener(SWT.Close, popupCloser);
		}

		if (!control.isDisposed()) {
			control.removeListener(SWT.MouseDoubleClick, popupCloser);
			control.removeListener(SWT.MouseDown, popupCloser);
			control.removeListener(SWT.Dispose, popupCloser);
			control.removeListener(SWT.FocusOut, popupCloser);

			parentShell.removeListener(SWT.Move, popupCloser);
			parentShell.removeListener(SWT.Resize, popupCloser);
			parentShell.removeListener(SWT.Deactivate, popupCloser);
		}

		if (shell.isDisposed()) {
			return;
		}

		parentShell.removeListener(SWT.Deactivate, parentDeactivateListener);

		// If we "close" the shell recursion will occur.
		// Instead, we need to "dispose" the shell to remove it from the
		// display.
		shell.dispose();
	}

	// Utils ==================================================================

	private void handleShellEvent(Event event) {
		if (event.type == SWT.Activate) {
			// ignore this event if we have launched a child
			if (event.widget == shell) {
				listenToDeactivate = true;
				// Typically we start listening for parent deactivate after
				// we are activated, except on the Mac, where the deactivate
				// is received after activate.
				// See https://bugs.eclipse.org/bugs/show_bug.cgi?id=100668
				listenToParentDeactivate = !isMac();
			}
		}
		else if (event.type == SWT.Deactivate) {
			// Close if we are deactivating and have no child shells. If we
			// have child shells, we are deactivating due to their opening.
			// On X, we receive this when a menu child (such as the system
			// menu) of the shell opens, but I have not found a way to
			// distinguish that case here. Hence bug #113577 still exists.
			if (listenToDeactivate && event.widget == shell) {
				asyncClose();
			}
			else {
				// We typically ignore deactivates to work around
				// platform-specific event ordering. Now that we've ignored
				// whatever we were supposed to, start listening to
				// deactivates. Example issues can be found in
				// https://bugs.eclipse.org/bugs/show_bug.cgi?id=123392
				listenToDeactivate = true;
			}
		}
		else if (event.type == SWT.Close) {
			event.doit = false;
			close();
		}
	}

	private void handleNoCharacter(Event e) {
		final int oldSelection = table.getSelectionIndex();
		final int visibleRowCount = Math.max((table.getClientArea().height - table.getHeaderHeight()) / table.getItemHeight() - 1, 1);
		final int maxIndex = table.getItemCount() - 1;
		int newSelection = oldSelection;
		boolean dontPropagateIfKeyDown = false;

		switch (e.keyCode) {
		case SWT.ARROW_UP:
			newSelection--;
			dontPropagateIfKeyDown = true;
			break;

		case SWT.ARROW_DOWN:
			newSelection++;
			dontPropagateIfKeyDown = true;
			break;

		case SWT.PAGE_DOWN:
			newSelection += visibleRowCount;
			dontPropagateIfKeyDown = true;
			break;

		case SWT.PAGE_UP:
			newSelection -= visibleRowCount;
			dontPropagateIfKeyDown = true;
			break;

		case SWT.HOME:
			newSelection = 0;
			dontPropagateIfKeyDown = true;
			break;

		case SWT.END:
			newSelection = maxIndex;
			dontPropagateIfKeyDown = true;
			break;

		// If received as a Traverse, these should propagate
		// to the control as keydown. If received as a keydown,
		// proposals should be recomputed since the cursor
		// position has changed.
		case SWT.ARROW_LEFT:
		case SWT.ARROW_RIGHT:
			if (e.type == SWT.Traverse) {
				e.doit = false;
			}
			else {
				e.doit = true;
				final String contents = contentAdapter.getText();
				// If there are no contents, changes in cursor
				// position have no effect. Note also that we do
				// not affect the filter text on ARROW_LEFT as
				// we would with BS.
				if (contents.length() > 0) {
					asyncRecomputeProposals();
				}
			}
			break;

		// Any unknown keycodes will cause the popup to close.
		// Modifier keys are explicitly checked and ignored because
		// they are not complete yet (no character).
		default:
			if (e.keyCode != SWT.CAPS_LOCK && e.keyCode != SWT.NUM_LOCK
			    && e.keyCode != SWT.MOD1
			    && e.keyCode != SWT.MOD2
			    && e.keyCode != SWT.MOD3
			    && e.keyCode != SWT.MOD4) {
				close();
			}
			return;
		}

		if (dontPropagateIfKeyDown && e.type == SWT.KeyDown) {
			// don't propagate to control
			e.doit = false;
		}

		// If any of these navigation events caused a new selection,
		// then handle that now and return.
		if (oldSelection >= 0 && maxIndex >= 0) {
			newSelection = Math.min(newSelection, maxIndex);
			newSelection = Math.max(newSelection, 0);
			if (newSelection != oldSelection) {
				table.setSelection(newSelection);
			}
		}
	}

	private boolean isMac() {
		final String ws = SWT.getPlatform();
		return "carbon".equals(ws) || "cocoa".equals(ws);
	}

	private void asyncClose() {
		// workaround for https://bugs.eclipse.org/bugs/show_bug.cgi?id=152010
		shell.getDisplay().asyncExec(new Runnable() {
			@Override
			public void run() {
				close();
			}
		});
	}

	private Rectangle getClosestMonitorBounds(Rectangle shellBounds, Display display) {
		final int shellCenterX = shellBounds.x + shellBounds.width / 2;
		final int shellCenterY = shellBounds.y + shellBounds.height / 2;

		int closestDistance = Integer.MAX_VALUE;

		final Monitor[] monitors = display.getMonitors();
		Rectangle closestMonitorBounds = monitors[0].getClientArea();

		for (Monitor monitor : monitors) {
			final Rectangle clientArea = monitor.getClientArea();
			if (clientArea.contains(shellCenterX, shellCenterY)) {
				return clientArea;
			}

			final int dX = clientArea.x + clientArea.width / 2 - shellCenterX;
			final int dy = clientArea.y + clientArea.height / 2 - shellCenterY;
			final int distance = dX * dX + dy * dy;
			if (distance < closestDistance) {
				closestDistance = distance;
				closestMonitorBounds = clientArea;
			}
		}
		return closestMonitorBounds;
	}

	private void recomputeProposals() {
		proposals = proposalAdapter.getProposals();
		if (proposals.isEmpty()) {
			close();
			return;
		}

		if (table.isDisposed()) {
			return;
		}

		updateTable();
	}

	private void updateTable() {
		final int newSize = proposals.size();
		table.setRedraw(false);
		try {
			table.setItemCount(newSize);
			final TableItem[] items = table.getItems();
			for (int i = 0; i < items.length; i++) {
				final TableItem item = items[i];
				final QContentProposal proposal = proposals.get(i);
				proposalAdapter.initializeItem(proposal, item);
			}
			// Default to the first selection if there is content.
			if (items.length > 0) {
				table.setSelection(0);
			}
		}
		finally {
			table.setRedraw(true);
		}

		// Get our control's location in display coordinates.
		final Point location = control.getDisplay().map(control.getParent(), null, control.getLocation());
		final Rectangle insertionBounds = contentAdapter.getInsertionBounds();
		final int initialX = location.x + 3 + insertionBounds.x;
		final int initialY = location.y + insertionBounds.y + insertionBounds.height;

		final GridData data = new GridData(GridData.FILL_BOTH);
		data.heightHint = Math.min(10, proposals.size()) * table.getItemHeight();
		data.widthHint = SWT.DEFAULT;
		table.setLayoutData(data);
		shell.pack();
		final Point popupSize = shell.getSize();

		// Constrain to the display
		final Rectangle constrainedBounds = new Rectangle(initialX, initialY, popupSize.x, popupSize.y);

		final Rectangle monitorBounds = getClosestMonitorBounds(constrainedBounds, control.getDisplay());
		constrainedBounds.width = Math.min(constrainedBounds.width, monitorBounds.width);
		constrainedBounds.height = Math.min(constrainedBounds.height, monitorBounds.height);

		constrainedBounds.x = Math.max(monitorBounds.x, Math.min(constrainedBounds.x, monitorBounds.x + monitorBounds.width - constrainedBounds.width));
		constrainedBounds.y = Math.max(monitorBounds.y, Math.min(constrainedBounds.y, monitorBounds.y + monitorBounds.height - constrainedBounds.height));

		// If there has been an adjustment causing the popup to overlap
		// with the control, then put the popup above the control.
		if (constrainedBounds.y < initialY) {
			shell.setLocation(initialX, location.y - popupSize.y);
		}
		else {
			shell.setLocation(initialX, initialY);
		}
	}

	/*
	 * Return the current selected proposal.
	 */
	private QContentProposal getSelectedProposal() {
		final int i = table.getSelectionIndex();
		if (i < 0 || i >= proposals.size()) {
			return null;
		}
		return proposals.get(i);
	}

	private void acceptProposal(QContentProposal proposal) {
		// Close before accepting the proposal. This is important
		// so that the cursor position can be properly restored at
		// acceptance, which does not work without focus on some controls.
		// See https://bugs.eclipse.org/bugs/show_bug.cgi?id=127108
		close();

		proposalAdapter.setWatchModify(false);

		if (proposal == null || control.isDisposed()) {
			return;
		}

		proposalAdapter.proposalAccepted(proposal);
	}

	/*
	 * In an async block, request the proposals. This is used when clients
	 * are in the middle of processing an event that affects the widget
	 * content. By using an async, we ensure that the widget content is up
	 * to date with the event.
	 */
	private void asyncRecomputeProposals() {
		if (table.isDisposed()) {
			recomputeProposals();
		}
		else {
			table.getDisplay().asyncExec(new Runnable() {
				@Override
				public void run() {
					if (table.isDisposed()) {
						return;
					}

					recomputeProposals();
				}
			});
		}
	}

	// Inner Classes ==========================================================

	/*
	 * The listener we install on the popup and related controls to
	 * determine when to close the popup. Some events (move, resize, close,
	 * deactivate) trigger closure as soon as they are received, simply
	 * because one of the registered listeners received them. Other events
	 * depend on additional circumstances.
	 */
	private final class PopupCloserListener implements Listener {
		private boolean scrollbarClicked;

		@Override
		public void handleEvent(Event e) {
			// If focus is leaving an important widget or the field's
			// shell is deactivating
			if (e.type == SWT.FocusOut || e.type == SWT.Deactivate) {
				scrollbarClicked = false;
				/*
				 * Ignore this event if it's only happening because focus is
				 * moving between the popup shells, their controls, or a
				 * scrollbar. Do this in an async since the focus is not
				 * actually switched when this event is received.
				 */
				final Display display = e.display;
				display.asyncExec(new Runnable() {
					@Override
					public void run() {
						if (table.isDisposed()) {
							return;
						}

						if (scrollbarClicked || shell.isFocusControl() || table.isFocusControl()) {
							return;
						}
						// Workaround a problem on X and Mac, whereby at
						// this point, the focus control is not known.
						// This can happen, for example, when resizing
						// the popup shell on the Mac.
						// Check the active shell.
						final Shell activeShell = display.getActiveShell();
						if (activeShell == shell) {
							return;
						}
						close();
					}
				});
				return;
			}

			// Scroll bar has been clicked. Remember this for focus event
			// processing.
			if (e.type == SWT.Selection) {
				scrollbarClicked = true;
				return;
			}
			// For all other events, merely getting them dictates closure.
			close();
		}
	}
}
