package com.syntevo.plugin.trac.commit.messagesource;

import java.util.*;

import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.trac.transport.*;


/**
 * @author syntevo GmbH
 */
final class TracIssueByRelevanceComparator implements Comparator<TracIssue> {
	private final List<TracIssueVersion> unreleasedVersions;
	private final List<TracIssueMilestone> incompleteMilestones;

	public TracIssueByRelevanceComparator(@NotNull List<TracIssueVersion> unreleasedVersions, @NotNull List<TracIssueMilestone> incompleteMilestones) {
		this.unreleasedVersions = unreleasedVersions;
		this.incompleteMilestones = incompleteMilestones;
	}

	@SuppressWarnings("SuspiciousMethodCalls")
	@Override
	public int compare(TracIssue o1, TracIssue o2) {
		if (o1 == null || o2 == null) {
			return CompareUtils.compareForNull(o1, o2);
		}

		final TracIssueStatus firstStatus = o1.getStatus();
		final TracIssueStatus secondStatus = o2.getStatus();

		if (firstStatus.getPriority() != secondStatus.getPriority()) {
			if (firstStatus.getPriority() < secondStatus.getPriority()) {
				return -1;
			}

			return 1;
		}

		final String firstMilestone = o1.getMilestoneName();
		final String secondMilestone = o2.getMilestoneName();
		final String firstVersion = o1.getVersionName();
		final String secondVersion = o2.getVersionName();

		final int compMilestone = compareVersionOrMilestone(getIncompletedMilestoneIndexByName(firstMilestone), getIncompletedMilestoneIndexByName(secondMilestone));

		if (compMilestone != 0) {
			return compMilestone;
		}

		final int compVersion = compareVersionOrMilestone(getUnreleasedVersionIndexByName(firstVersion), getUnreleasedVersionIndexByName(secondVersion));

		if (compVersion != 0) {
			return compVersion;
		}

		if (o1.getId() > o2.getId()) {
			return -1;
		}

		return 1;
	}

	private int getIncompletedMilestoneIndexByName(@Nullable String name) {
		if (name == null || name.isEmpty()) {
			return -1;
		}

		for (int cnt = 0; cnt < incompleteMilestones.size(); cnt++) {
			final TracIssueMilestone milestone = incompleteMilestones.get(cnt);

			if (milestone != null && milestone.getName().equals(name)) {
				return cnt;
			}
		}

		return -1;
	}

	private int getUnreleasedVersionIndexByName(@Nullable String name) {
		if (name == null || name.isEmpty()) {
			return -1;
		}

		for (int cnt = 0; cnt < unreleasedVersions.size(); cnt++) {
			final TracIssueVersion version = unreleasedVersions.get(cnt);

			if (version != null && version.getName().equals(name)) {
				return cnt;
			}
		}

		return -1;
	}

	public static int compareVersionOrMilestone(int idx1, int idx2) {
		if (idx1 == idx2) {
			return 0;
		}

		if (idx1 == -1) {
			return 1;
		}

		if (idx2 == -1) {
			return -1;
		}

		if (idx1 < idx2) {
			return -1;
		}

		return 1;
	}
}