package com.syntevo.plugin.trac.commit.messagesource;

import java.util.*;
import java.util.List;

import org.eclipse.swt.widgets.*;
import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.*;
import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.smartsvn.command.commit.*;
import com.syntevo.openapi.deprecated.smartsvn.settings.*;
import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.trac.commit.*;
import com.syntevo.plugin.trac.transport.*;

/**
 * @author syntevo GmbH
 */
final class TracCommitMessageSource implements ICommitMessageSource, TracIssueTable {

	// Fields =================================================================

	private final DialogDisplayer dialogDisplayer;
	private final TracIssueLoader issueLoader;
	private final TracQueryConfiguration queryConfiguration;
	private final CommitBugtraqProperties bugtraqProperties;

	private List<TracIssue> pendingIssues = new ArrayList<>();
	private TracCommitMessageSourceDialog currentDialog;
	private final TracUIConnection uiConnection;
	private static final String DEFAULT_COMMIT_MESSAGE_PATTERN = "#" + TracConfigureCommitMessagePatternDialog.PLACEHOLDER_BUGID + ": " + TracConfigureCommitMessagePatternDialog.PLACEHOLDER_MESSAGE;

	// Setup ==================================================================

	public TracCommitMessageSource(@NotNull String baseUrl, @NotNull SettingsServices settingsServices, @NotNull CommitBugtraqProperties bugtraqProperties,
	                               @NotNull GuiSpacings spacings, @NotNull DialogDisplayer dialogDisplayer) {
		this.bugtraqProperties = bugtraqProperties;
		this.dialogDisplayer = dialogDisplayer;
		issueLoader = new TracIssueLoader(this);
		issueLoader.start();

		final boolean loadAll = "true".equals(System.getProperty("smartsvn.plugin.trac.load-all-issues", "false"));
		uiConnection = new TracUIConnection(baseUrl, settingsServices, false, spacings, dialogDisplayer);
		queryConfiguration = new TracQueryConfiguration(uiConnection, loadAll);

		final TracQueryConfiguration silentQueryConfiguration = new TracQueryConfiguration(new TracUIConnection(baseUrl, settingsServices, true, spacings, dialogDisplayer), loadAll);
		issueLoader.load(silentQueryConfiguration, true);
	}

	// Implemented ============================================================

	@Override
	@NotNull
	public String getMenuItemName() {
		return "Get from Trac...";
	}

	@Override
	public void getCommitMessage(@NotNull final Consumer<String> consumer) {
		String commitMessagePattern = uiConnection.getCommitMessagePattern();

		if (commitMessagePattern == null) {
			commitMessagePattern = DEFAULT_COMMIT_MESSAGE_PATTERN;
		}

		final TracCommitMessageSourceDialog dialog = new TracCommitMessageSourceDialog(pendingIssues, issueLoader, queryConfiguration,
		                                                                               commitMessagePattern, bugtraqProperties, dialogDisplayer);
		currentDialog = dialog;

		issueLoader.load(queryConfiguration, false);
		dialogDisplayer.showAsync(currentDialog, new IDialogResultHandler() {
			@Override
			public void handleDialogResult(int value) {
				if (value != AbstractDialog.RESULT_OK) {
					return;
				}

				if (bugtraqProperties.getMessage() == null && !CompareUtils.areEqual(uiConnection.getCommitMessagePattern(), dialog.getCommitMessagePattern())) {
					uiConnection.storeCommitMessagePattern(dialog.getCommitMessagePattern());
				}

				consumer.consumer(dialog.getCommitMessage());
			}
		});
	}

	@Override
	public void cleanup() {
		issueLoader.abortAndCleanup();
	}

	@Override
	public void setIssues(@NotNull final List<TracIssue> issues) {
		Display.getDefault().asyncExec(new Runnable() {
			@Override
			public void run() {
				pendingIssues = new ArrayList<>(issues);
				if (currentDialog != null) {
					currentDialog.setIssues(pendingIssues);
				}
			}
		});
	}
}