package com.syntevo.plugin.jira.transport;

import java.io.*;
import java.net.*;
import java.util.*;
import javax.net.ssl.*;
import javax.xml.soap.*;

import org.w3c.dom.Node;
import org.w3c.dom.*;

import com.syntevo.plugin.common.bugtracker.transport.*;
import com.syntevo.plugin.jira.*;

/**
 * @author syntevo GmbH
 */
public final class JiraSoapClient extends JiraClient {

	// Constants ==============================================================

	private static final String JIRA_IN0 = "in0";
	private static final String JIRA_IN1 = "in1";
	private static final String JIRA_IN2 = "in2";
	private static final String JIRA_IN3 = "in3";
	private static final String JIRA_NAME = "name";
	private static final String JIRA_ARCHIVED = "archived";
	private static final String JIRA_RELEASED = "released";
	private static final String JIRA_FALSE = "false";
	private static final String JIRA_TRUE = "true";
	private static final String JIRA_REMOTE_FIELD_VALUE = "RemoteFieldValue";
	private static final String JIRA_GET_ISSUE = "getIssue";
	private static final String JIRA_GET_ISSUE_RESPONSE = "getIssueResponse";
	private static final String JIRA_GET_ISSUE_RETURN = "getIssueReturn";
	private static final String JIRA_GET_PROJECT_BY_KEY = "getProjectByKey";
	private static final String JIRA_GET_PROJECT_BY_KEY_RESPONSE = "getProjectByKeyResponse";
	private static final String JIRA_GET_PROJECT_BY_KEY_RETURN = "getProjectByKeyReturn";
	private static final String JIRA_GET_VERSIONS = "getVersions";
	private static final String JIRA_GET_VERSIONS_RESPONSE = "getVersionsResponse";
	private static final String JIRA_GET_VERSIONS_RETURN = "getVersionsReturn";
	private static final String JIRA_LOGIN = "login";
	private static final String JIRA_LOGIN_RESPONSE = "loginResponse";
	private static final String JIRA_LOGIN_RETURN = "loginReturn";
	private static final String JIRA_LOGOUT = "logout";
	private static final String JIRA_LOGOUT_RESPONSE = "logoutResponse";
	private static final String JIRA_LOGOUT_RETURN = "logoutReturn";
	private static final String JIRA_PROGRESS_WORKFLOW_ACTION = "progressWorkflowAction";
	private static final String JIRA_PROGRESS_WORKFLOW_ACTION_RESPONSE = "progressWorkflowActionResponse";
	private static final String JIRA_PROGRESS_WORKFLOW_ACTION_RETURN = "progressWorkflowActionReturn";
	private static final String SOAP_ENC_ARRAY = "soapenc:Array";
	private static final String SOAP_ENC_ARRAYTYPE = "soapenc:arrayType";
	private static final String SOAP_MULTIREF = "multiRef";
	private static final String SOAP_HREF = "href";
	private static final String SOAP_ID = "id";
	private static final String SOAP_FAULT = "Fault";
	private static final String SOAP_FAULTSTRING = "faultstring";
	private static final String SOAP_VALUES = "values";
	private static final String XSI_TYPE = "xsi:type";
	private static final String XSD_STRING = "xsd:string";
	private static final String RESOLVED_CONSTANT = System.getProperty("smartsvn.plugin.jira.resolvedConstant", System.getProperty("smartsvn.plugin.jira.resolved-constant", "5"));

	// Fields =================================================================

	private final String endpoint;

	private String loginToken;
	private String loggedInUsername;
	private SOAPConnection connection;
	private MessageFactory messageFactory;
	private SOAPFactory soapFactory;

	// Setup ==================================================================

	public JiraSoapClient(BugtrackerConnection connection) {
		super(connection);

		this.endpoint = connection.getUrl() + "rpc/soap/jirasoapservice-v2";
	}

	// Accessing ==============================================================

	public synchronized String getLoggedInUsername() {
		return loggedInUsername;
	}

	public synchronized boolean login() throws SOAPException, IOException {
		final String username = getBugtrackerConnection().getUsername();
		if (username == null) {
			return false;
		}

		final String password = getBugtrackerConnection().getPassword();
		final SOAPConnectionFactory factory = SOAPConnectionFactory.newInstance();
		this.connection = factory.createConnection();
		this.messageFactory = MessageFactory.newInstance();
		this.soapFactory = SOAPFactory.newInstance();

		final SOAPMessage requestMessage = createMessage(messageFactory);
		final SOAPBodyElement request = createBody(JIRA_LOGIN, requestMessage, soapFactory);
		addString(request, JIRA_IN0, username, soapFactory);
		addString(request, JIRA_IN1, password, soapFactory);

		final Node response;
		try {
			final SOAPBody responseBody = sendMessage(requestMessage, "login");
			response = getUniqueChild(responseBody, JIRA_LOGIN_RESPONSE, responseBody);
			final Node responseReturn = getUniqueChild(response, JIRA_LOGIN_RETURN, responseBody);
			loginToken = extractText(responseReturn);
			getBugtrackerConnection().acknowledgeCredentials(loginToken != null);
			loggedInUsername = username;
			return loginToken != null;
		}
		catch (SOAPException ex) {
			loginToken = null;
			getBugtrackerConnection().acknowledgeCredentials(false);
			loggedInUsername = null;
			throw ex;
		}
	}

	public synchronized boolean logout() throws SOAPException, IOException {
		if (loginToken == null) {
			return false;
		}

		final SOAPMessage requestMessage = createMessage(messageFactory);
		final SOAPBodyElement request = createBody(JIRA_LOGOUT, requestMessage, soapFactory);
		addString(request, JIRA_IN0, loginToken, soapFactory);

		final SOAPBody responseBody = sendMessage(requestMessage, "logout");
		final Node responseReturn;
		try {
			final Node response = getUniqueChild(responseBody, JIRA_LOGOUT_RESPONSE, responseBody);
			responseReturn = getUniqueChild(response, JIRA_LOGOUT_RETURN, responseBody);
		}
		finally {
			connection = null;
			messageFactory = null;
			soapFactory = null;
		}
		return JIRA_TRUE.equals(extractText(responseReturn));
	}

	public synchronized String getProjectIDByKey(String key) throws SOAPException, IOException {
		final SOAPMessage requestMessage = createMessage(messageFactory);
		final SOAPBodyElement request = createBody(JIRA_GET_PROJECT_BY_KEY, requestMessage, soapFactory);
		addString(request, JIRA_IN0, loginToken, soapFactory);
		addString(request, JIRA_IN1, key, soapFactory);

		final SOAPBody responseBody = sendMessage(requestMessage, "getProjectIDByKey");
		final Node response = getUniqueChild(responseBody, JIRA_GET_PROJECT_BY_KEY_RESPONSE, responseBody);
		final Node responseReturn = getUniqueChild(response, JIRA_GET_PROJECT_BY_KEY_RETURN, responseBody);
		final Node id = getUniqueChild(responseReturn, SOAP_ID, responseBody);
		return extractText(id);
	}

	public synchronized JiraIssue getIssueByKey(String issueKey) throws IOException, SOAPException {
		final SOAPMessage requestMessage = createMessage(messageFactory);
		final SOAPBodyElement request = createBody(JIRA_GET_ISSUE, requestMessage, soapFactory);
		addString(request, JIRA_IN0, loginToken, soapFactory);
		addString(request, JIRA_IN1, issueKey, soapFactory);

		final SOAPBody responseBody = sendMessage(requestMessage, "getIssueByKey");
		final Node response = getUniqueChild(responseBody, JIRA_GET_ISSUE_RESPONSE, responseBody);
		final Node responseReturn = getUniqueChild(response, JIRA_GET_ISSUE_RETURN, responseBody);
		final JiraIssue issue = new JiraIssue();
		issue.setKey(extractText(getUniqueChild(responseReturn, SOAP_ID, responseBody)), issueKey);
		issue.setSummary(extractText(getUniqueChild(responseReturn, JiraIssue.FIELD_SUMMARY, responseBody)));
		issue.setAssignee(extractText(getUniqueChild(responseReturn, JiraIssue.FIELD_ASSIGNEE, responseBody)));
		issue.setType(JiraIssueType.parseFromId(extractText(getUniqueChild(responseReturn, JiraIssue.FIELD_TYPE, responseBody))));
		issue.setStatus(JiraIssueStatus.parseFromId(extractText(getUniqueChild(responseReturn, JiraIssue.FIELD_STATUS, responseBody))));
		issue.setResolution(JiraIssueResolution.parseFromId(extractText(getUniqueChild(responseReturn, JiraIssue.FIELD_RESOLUTION, responseBody))));
		final List<Node> fixVersionNodes = getChildren(responseReturn, JiraIssue.FIELD_FIX_VERSIONS, responseBody);
		final List<JiraVersion> fixVersions = new ArrayList<>();
		for (Node fixVersionNode : fixVersionNodes) {
			fixVersions.add(new JiraVersion(extractText(getUniqueChild(fixVersionNode, SOAP_ID, responseBody)), extractText(getUniqueChild(fixVersionNode, JIRA_NAME, responseBody))));
		}

		issue.setFixVersions(fixVersions);
		return issue;
	}

	public synchronized List<JiraVersion> getUnreleasedVersionsByProjectKeyInOrder(String projectKey) throws SOAPException, IOException {
		final SOAPMessage requestMessage = createMessage(messageFactory);
		final SOAPBodyElement request = createBody(JIRA_GET_VERSIONS, requestMessage, soapFactory);
		addString(request, JIRA_IN0, loginToken, soapFactory);
		addString(request, JIRA_IN1, projectKey, soapFactory);

		final SOAPBody responseBody = sendMessage(requestMessage, "getUnreleasedVersionsByProjectKeyInOrder");
		final Node response = getUniqueChild(responseBody, JIRA_GET_VERSIONS_RESPONSE, responseBody);
		final List<JiraVersion> versions = new ArrayList<>();
		for (Node node : getChildren(response, JIRA_GET_VERSIONS_RETURN, responseBody)) {
			final Node archived = getUniqueChild(node, JIRA_ARCHIVED, responseBody);
			if (!JIRA_FALSE.equals(extractText(archived))) {
				continue;
			}

			final Node released = getUniqueChild(node, JIRA_RELEASED, responseBody);
			if (!JIRA_FALSE.equals(extractText(released))) {
				continue;
			}

			final String id = extractText(getUniqueChild(node, SOAP_ID, responseBody));
			final String name = extractText(getUniqueChild(node, JIRA_NAME, responseBody));
			versions.add(new JiraVersion(id, name));
		}

		return versions;
	}

	public synchronized boolean resolve(String issueKey, String assignee, JiraIssueResolution resolution, JiraVersion fixVersion) throws IOException, SOAPException {
		final List<JiraVersion> fixVersions = fixVersion != null ? Collections.singletonList(fixVersion) : getFixVersionIDs(issueKey);
		final SOAPMessage requestMessage = createMessage(messageFactory);
		final SOAPBodyElement request = createBody(JIRA_PROGRESS_WORKFLOW_ACTION, requestMessage, soapFactory);
		addString(request, JIRA_IN0, loginToken, soapFactory);
		addString(request, JIRA_IN1, issueKey, soapFactory);
		addString(request, JIRA_IN2, RESOLVED_CONSTANT, soapFactory); // 5 means resolveIssue

		final SOAPElement remoteFieldValuesElement = addArray(request, JIRA_IN3, soapFactory);
		final SOAPElement assigneeRemoteField = addElement(remoteFieldValuesElement, JIRA_IN3, JIRA_REMOTE_FIELD_VALUE, soapFactory);
		addString(assigneeRemoteField, SOAP_ID, JiraIssue.FIELD_ASSIGNEE, soapFactory);
		final SOAPElement assigneeArray = addArray(assigneeRemoteField, SOAP_VALUES, soapFactory);
		addString(assigneeArray, SOAP_VALUES, assignee, soapFactory);

		final SOAPElement resolutionRemoteField = addElement(remoteFieldValuesElement, JIRA_IN3, JIRA_REMOTE_FIELD_VALUE, soapFactory);
		addString(resolutionRemoteField, SOAP_ID, JiraIssue.FIELD_RESOLUTION, soapFactory);
		final SOAPElement resolutionArray = addArray(resolutionRemoteField, SOAP_VALUES, soapFactory);
		addString(resolutionArray, SOAP_VALUES, resolution.getJiraId(), soapFactory);

		final SOAPElement fixVersionsRemoteField = addElement(remoteFieldValuesElement, JIRA_IN3, JIRA_REMOTE_FIELD_VALUE, soapFactory);
		addString(fixVersionsRemoteField, SOAP_ID, JiraIssue.FIELD_FIX_VERSIONS, soapFactory);
		final SOAPElement fixVersionsArray = addArray(fixVersionsRemoteField, SOAP_VALUES, soapFactory);
		for (JiraVersion aFixVersion : fixVersions) {
			addString(fixVersionsArray, SOAP_VALUES, aFixVersion.getId(), soapFactory);
		}

		final SOAPBody responseBody = sendMessage(requestMessage, "resolve");
		final Node response = getUniqueChild(responseBody, JIRA_PROGRESS_WORKFLOW_ACTION_RESPONSE, responseBody);
		final Node issue = getUniqueChild(response, JIRA_PROGRESS_WORKFLOW_ACTION_RETURN, responseBody);
		final Node status = getUniqueChild(issue, JiraIssue.FIELD_STATUS, responseBody);
		return RESOLVED_CONSTANT.equals(extractText(status));
	}

	// Utils ==================================================================

	private SOAPBody sendMessage(SOAPMessage request, String requestTitle) throws SOAPException, IOException {
		if (JiraPlugin.LOGGER.isDebugEnabled()) {
			JiraPlugin.LOGGER.debug("Sending request '" + requestTitle + "' to " + endpoint);

			final ByteArrayOutputStream out = new ByteArrayOutputStream();
			request.writeTo(out);
			JiraPlugin.LOGGER.debug(new String(out.toByteArray()));
		}

		final SOAPMessage response;
		try {
			response = connection.call(request, createConnectableURL(new URL(endpoint)));
			getBugtrackerConnection().acknowledgeSSLClientCertificate(true);
			return response.getSOAPBody();
		}
		catch (SSLException | SOAPException ex) {
			JiraPlugin.LOGGER.debug("Received error: " + ex.getMessage(), ex);

			checkSSLException(ex, getBugtrackerConnection());
			throw ex;
		}
	}

	private List<JiraVersion> getFixVersionIDs(String issueKey) throws SOAPException, IOException {
		final JiraIssue issue = getIssueByKey(issueKey);
		return issue != null ? issue.getFixVersions() : Collections.<JiraVersion>emptyList();
	}

	private static void checkSSLException(Exception ex, BugtrackerConnection connection) {
		for (Throwable tempEx = ex; tempEx != null; tempEx = tempEx.getCause()) {
			if (tempEx instanceof SSLException) {
				connection.acknowledgeSSLClientCertificate(false);
			}
		}
	}

	private static String extractText(Node node) {
		if (node.getNodeType() == Node.TEXT_NODE) {
			return node.getNodeValue();
		}

		final Node firstChild = node.getFirstChild();
		return firstChild != null ? firstChild.getNodeValue() : null;
	}

	private static Node getUniqueChild(Node parent, String name, SOAPBody body) throws SOAPException {
		final List<Node> children = getChildren(parent, name, body);
		if (children.isEmpty()) {
			throw new SOAPException("'" + name + "' not found in element " + parent);
		}
		if (children.size() >= 2) {
			throw new SOAPException("'" + name + "' found multiple times in element " + parent);
		}

		return children.get(0);
	}

	private static List<Node> getChildren(Node parent, String name, SOAPBody body) throws SOAPException {
		final NodeList childNodes = parent.getChildNodes();
		final List<Node> rawFoundNodes = new ArrayList<>();
		for (int index = 0; index < childNodes.getLength(); index++) {
			final Node node = childNodes.item(index);
			final String nodeName = node.getLocalName();
			if (SOAP_FAULT.equals(nodeName)) {
				throw new SOAPException(extractError(node));
			}

			if (!nodeName.equals(name)) {
				continue;
			}

			rawFoundNodes.add(node);
		}

		final List<Node> foundNodes = new ArrayList<>();
		for (Node node : rawFoundNodes) {
			if (node.getAttributes().getNamedItem(SOAP_ENC_ARRAYTYPE) != null) {
				final NodeList children = node.getChildNodes();
				for (int index = 0; index < children.getLength(); index++) {
					foundNodes.add(children.item(index));
				}
			}
			else {
				foundNodes.add(node);
			}
		}

		final List<Node> resolvedNodes = new ArrayList<>();
		for (Node node : foundNodes) {
			resolvedNodes.add(resolveHref(node, body));
		}

		return resolvedNodes;
	}

	private static Node resolveHref(Node node, SOAPBody body) throws SOAPException {
		final Node href = node.getAttributes().getNamedItem(SOAP_HREF);
		if (href == null) {
			return node;
		}

		String hrefId = extractText(href);
		if (hrefId.startsWith("#")) {
			hrefId = hrefId.substring(1);
		}

		for (final Iterator it = body.getChildElements(); it.hasNext();) {
			final Node child = (Node)it.next();
			if (!SOAP_MULTIREF.equals(child.getNodeName())) {
				continue;
			}

			final Node idAttribute = child.getAttributes().getNamedItem(SOAP_ID);
			if (idAttribute == null) {
				continue;
			}

			if (extractText(idAttribute).equals(hrefId)) {
				return child;
			}
		}

		throw new SOAPException("multiref with ID '" + hrefId + "' not found in body.");
	}

	private static void addString(SOAPElement parent, String key, String value, SOAPFactory soapFactory) throws SOAPException {
		parent.addChildElement(soapFactory.createName(key)).addTextNode(value).setAttribute(XSI_TYPE, XSD_STRING);
	}

	private static SOAPElement addElement(SOAPElement parent, String key, String type, SOAPFactory soapFactory) throws SOAPException {
		final SOAPElement element = parent.addChildElement(soapFactory.createName(key));
		element.addAttribute(soapFactory.createName(XSI_TYPE), "ns2:" + type);
		return element;
	}

	private static SOAPElement addArray(SOAPElement bodyElement, String key, SOAPFactory soapFactory) throws SOAPException {
		final SOAPElement array = bodyElement.addChildElement(soapFactory.createName(key));
		array.addAttribute(soapFactory.createName(XSI_TYPE), SOAP_ENC_ARRAY);
		array.addAttribute(soapFactory.createName("xmlns:ns2"), "http://beans.soap.rpc.jira.atlassian.com");
		return array;
	}

	private static SOAPBodyElement createBody(String command, SOAPMessage request, SOAPFactory soapFactory) throws SOAPException {
		final Name bodyName = soapFactory.createName(command, "ns1", "http://soap.rpc.jira.atlassian.com");
		return request.getSOAPBody().addBodyElement(bodyName);
	}

	private static SOAPMessage createMessage(MessageFactory messageFactory) throws SOAPException {
		final SOAPMessage request = messageFactory.createMessage();
		final MimeHeaders mimeHeaders = request.getMimeHeaders();
		mimeHeaders.addHeader("SOAPAction", "");
		request.getSOAPHeader().detachNode();
		request.getSOAPBody().addNamespaceDeclaration("xsi", "http://www.w3.org/2001/XMLSchema-instance");
		return request;
	}

	private static String extractError(Node node) {
		String error = extractRawError(node);
		if (error == null) {
			return "Unknown error";
		}

		error = error.trim();

		final String REMOTE_EXCEPTION = "com.atlassian.jira.rpc.exception.RemoteException:";
		if (error.startsWith(REMOTE_EXCEPTION)) {
			error = error.substring(REMOTE_EXCEPTION.length());
		}

		return error.trim();
	}

	private static String extractRawError(Node node) {
		if (node.getNodeType() == Node.TEXT_NODE) {
			return extractText(node);
		}

		final NodeList children = node.getChildNodes();
		for (int index = 0; index < children.getLength(); index++) {
			final Node child = children.item(index);
			if (child.getLocalName().equals(SOAP_FAULTSTRING)) {
				return extractText(child);
			}
		}

		return extractText(node);
	}
}