package com.syntevo.plugin.jira.commit.workflow;

import java.util.List;

import org.eclipse.swt.*;
import org.eclipse.swt.events.*;
import org.eclipse.swt.layout.*;
import org.eclipse.swt.widgets.*;
import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.*;
import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.jira.transport.*;

/**
 * @author syntevo GmbH
 */
final class JiraResolveIssueDialog extends AbstractOkCancelDialog {

	// Static =================================================================

	@NotNull
	public static JiraResolveIssueDialog createInstance(List<JiraVersion> unreleasedVersions, JiraResolvablePacket packet, GuiSpacings spacings) {
		final JiraIssue issue = packet.getIssue();
		String summary = issue.getSummary();
		if (summary.length() > 50) {
			summary = summary.substring(0, 47) + "...";
		}

		return new JiraResolveIssueDialog("Do you want to resolve issue " + issue.getKey() + ": " + summary + "?", unreleasedVersions, issue.getFixVersions(), spacings);
	}

	// Fields =================================================================

	private final String message;
	private final List<JiraVersion> unreleasedVersions;
	private final List<JiraVersion> fixVersions;
	private final GuiSpacings spacings;

	private Button resolveButton;
	private Combo fixVersionComboBox;
	private Button dontResolveButton;

	private JiraVersion fixVersion;

	// Setup ==================================================================

	private JiraResolveIssueDialog(String message, List<JiraVersion> unreleasedVersions, List<JiraVersion> fixVersions, GuiSpacings spacings) {
		this.message = message;
		this.unreleasedVersions = unreleasedVersions;
		this.fixVersions = fixVersions;
		this.spacings = spacings;
	}

	// Implemented ============================================================

	@NotNull
	@Override
	public String getTitle() {
		return "Resolve JIRA Issue";
	}

	@NotNull
	@Override
	public Control createComponent(@NotNull Composite parent) {
		final Composite panel = new Composite(parent, SWT.NONE);

		final GridLayout layout = new GridLayout(1, false);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.verticalSpacing = spacings.getYSeparator();
		panel.setLayout(layout);

		final Control messageLabel = UiUtils.createLabel(message, panel);
		messageLabel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));

		final Control radioButtonPanel = createRadioButtonPanel(panel);
		radioButtonPanel.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false, 1, 1));

		final String[] versionNames = new String[unreleasedVersions.size()];
		int indexToSelect = -1;
		for (int i = 0; i < versionNames.length; i++) {
			final JiraVersion version = unreleasedVersions.get(i);
			versionNames[i] = version.getName();
			if (indexToSelect < 0 && fixVersions.contains(version)) {
				indexToSelect = i;
			}
		}
		fixVersionComboBox.setItems(versionNames);
		if (indexToSelect >= 0) {
			fixVersionComboBox.select(indexToSelect);
		}

		final SelectionAdapter selectionListener = new SelectionAdapter() {
			@Override
			public void widgetSelected(SelectionEvent e) {
				updateEnabledStates();
			}
		};
		resolveButton.addSelectionListener(selectionListener);
		dontResolveButton.addSelectionListener(selectionListener);

		updateEnabledStates();

		return panel;
	}

	@Override
	public void dialogShowing() {
		super.dialogShowing();
		if (fixVersionComboBox.isEnabled()) {
			fixVersionComboBox.setFocus();
		}
		else {
			dontResolveButton.setFocus();
		}
	}

	@Override
	protected void ok() throws DialogValidationFailedException {
		if (resolveButton.getSelection()) {
			final int selectionIndex = fixVersionComboBox.getSelectionIndex();
			if (selectionIndex < 0) {
				throw new DialogValidationFailedException(fixVersionComboBox, "No version selected.", "Please select the version where the issue should be marked as fixed.");
			}

			fixVersion = unreleasedVersions.get(selectionIndex);
		}
		else {
			fixVersion = null;
		}
		super.ok();
	}

	// Accessing ==============================================================

	public JiraVersion getFixVersion() {
		return fixVersion;
	}

	// Utils ==================================================================

	private Control createRadioButtonPanel(Composite parent) {
		final Composite panel = new Composite(parent, SWT.NONE);

		final GridLayout layout = new GridLayout(2, false);
		layout.marginWidth = 0;
		layout.marginHeight = 0;
		layout.horizontalSpacing = spacings.getXRelated();
		layout.verticalSpacing = spacings.getYUnrelated();
		panel.setLayout(layout);

		dontResolveButton = UiUtils.createRadioButton("&Don't mark as resolved", panel);
		dontResolveButton.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, false, 2, 1));
		dontResolveButton.setSelection(true);

		resolveButton = UiUtils.createRadioButton("&Mark as resolved in version:", panel);
		resolveButton.setLayoutData(new GridData(SWT.LEFT, SWT.FILL, false, false));

		fixVersionComboBox = new Combo(panel, SWT.BORDER | SWT.READ_ONLY);
		fixVersionComboBox.setLayoutData(new GridData(SWT.FILL, SWT.FILL, true, false));

		return panel;
	}

	private void updateEnabledStates() {
		fixVersionComboBox.setEnabled(resolveButton.getSelection());
	}
}
