package com.syntevo.plugin.jira.commit.workflow;

import java.io.*;
import java.util.*;
import javax.xml.soap.*;

import org.jetbrains.annotations.*;

import com.syntevo.openapi.deprecated.gui.*;
import com.syntevo.openapi.deprecated.gui.dialog.*;
import com.syntevo.openapi.deprecated.smartsvn.command.commit.*;
import com.syntevo.openapi.deprecated.smartsvn.settings.*;
import com.syntevo.openapi.deprecated.smartsvn.thread.*;
import com.syntevo.openapi.deprecated.util.*;
import com.syntevo.plugin.jira.*;
import com.syntevo.plugin.jira.commit.*;
import com.syntevo.plugin.jira.transport.*;

/**
 * @author syntevo GmbH
 */
final class JiraFilesPhase extends FilterCommitPacketPhase {

	// Constants ==============================================================

	private static final boolean RESOLVE_ENABLED = "true".equals(System.getProperty("smartsvn.plugin.jira.show-resolveIssue-dialog", "true"));

	// Fields =================================================================

	private final SettingsServices settingsServices;
	private final JobExecutor jobExecutor;
	private final DialogDisplayer dialogDisplayer;
	private final GuiSpacings spacings;

	private List<JiraResolvePacket> selectedResolvePackets;

	// Setup ==================================================================

	public JiraFilesPhase(ICommitPacketPhase internalPhase, SettingsServices settingsServices, JobExecutor jobExecutor, GuiSpacings spacings, DialogDisplayer dialogDisplayer) {
		super(internalPhase);
		this.settingsServices = settingsServices;
		this.jobExecutor = jobExecutor;
		this.dialogDisplayer = dialogDisplayer;
		this.spacings = spacings;
	}

	// Implemented ============================================================

	@Override
	public void updateTo(final @NotNull CommitPacketPhaseState state, @NotNull final Runnable successRunnable) {
		super.updateTo(state, new Runnable() {
			@Override
			public void run() {
				selectedResolvePackets = null;

				if (!RESOLVE_ENABLED) {
					successRunnable.run();
					return;
				}

				final Set<JiraResolvablePacket> allResolvablePackets = new HashSet<>();
				for (CommitPacket packet : new ArrayList<>(state.getPackets())) {
					allResolvablePackets.addAll(extractResolvablePackets(packet, settingsServices, spacings, dialogDisplayer));
				}

				if (allResolvablePackets.isEmpty()) {
					successRunnable.run();
					return;
				}

				processResolvablePackets(allResolvablePackets, jobExecutor, successRunnable, dialogDisplayer);
			}
		});
	}

	// Accessing ==============================================================

	@Nullable
	public List<JiraResolvePacket> getResolvePackets() {
		UiUtils.assertThreadAccess(); // We don't sync on selectedResolvePackets, so we must touch that only in the GUI thread.
		return selectedResolvePackets != null ? Collections.unmodifiableList(selectedResolvePackets) : null;
	}

	// Utils ==================================================================

	private void processResolvablePackets(Set<JiraResolvablePacket> allResolvablePackets, JobExecutor jobExecutor, @NotNull final Runnable successRunnable, final DialogDisplayer dialogDisplayer) {
		JiraPlugin.LOGGER.debug("Filtering resolvable packets");

		final List<JiraResolvablePacket> resolvablePacketsInOrder = new ArrayList<>(allResolvablePackets);
		Collections.sort(resolvablePacketsInOrder, new Comparator<JiraResolvablePacket>() {
			@Override
			public int compare(JiraResolvablePacket o1, JiraResolvablePacket o2) {
				return o1.getIssueKey().compareTo(o2.getIssueKey());
			}
		});

		jobExecutor.executeAsync("JIRA", new IJob() {
			@Override
			public IJobSuccessRunnable execute(@NotNull ProgressViewer progressViewer) {
				int progress = 0;

				final List<JiraResolvablePacket> resolvablePackets = new ArrayList<>();
				for (JiraResolvablePacket packet : resolvablePacketsInOrder) {
					progressViewer.setMessage("Querying " + packet.getConnection().getUrl());
					progressViewer.setProgress(progress++, resolvablePacketsInOrder.size());

					final JiraSoapClient client = new JiraSoapClient(packet.getConnection());
					try {
						if (!client.login()) {
							JiraPlugin.LOGGER.warn("JIRA login failed, can't check " + packet);
							continue;
						}

						try {
							final JiraIssue issue = client.getIssueByKey(packet.getIssueKey());
							if (!isResolvable(issue)) {
								JiraPlugin.LOGGER.debug(" ... " + packet + " not resolvable (status=" + issue.getStatus() + ")");
								continue;
							}

							final List<JiraVersion> unreleasedVersions = client.getUnreleasedVersionsByProjectKeyInOrder(packet.getProjectKey());
							packet.setIssue(issue);
							packet.setUnreleasedVersions(unreleasedVersions);

							resolvablePackets.add(packet);
							JiraPlugin.LOGGER.debug(" ... " + packet + " is resolvable.");
						}
						finally {
							client.logout();
						}
					}
					catch (IOException | SOAPException ex) {
						JiraPlugin.LOGGER.error(ex.getMessage(), ex);
						dialogDisplayer.showErrorDialogSync("JIRA", ex.getMessage(), null);
					}
				}

				selectedResolvePackets = chooseResolvePackets(resolvablePackets);
				return new IJobSuccessRunnable() {
					@Override
					public void runInEdt(@NotNull DialogDisplayer dialogDisplayer) {
						if (selectedResolvePackets != null) {
							successRunnable.run();
						}
					}
				};
			}
		});
	}

	@Nullable
	private List<JiraResolvePacket> chooseResolvePackets(List<JiraResolvablePacket> resolvablePackets) {
		final List<JiraResolvePacket> selectedResolvePackets = new ArrayList<>();
		for (final JiraResolvablePacket packet : resolvablePackets) {
			final List<JiraVersion> unreleasedVersions = packet.getUnreleasedVersions();
			if (unreleasedVersions == null || unreleasedVersions.isEmpty()) {
				JiraPlugin.LOGGER.debug(" ... " + packet + " has no unreleased versions");
				continue;
			}

			final JiraResolveIssueDialog resolveIssueDialog = dialogDisplayer.showSync(new IDialogFactory<JiraResolveIssueDialog>() {
				@NotNull
				@Override
				public JiraResolveIssueDialog createDialog() {
					return JiraResolveIssueDialog.createInstance(unreleasedVersions, packet, spacings);
				}
			});
			if (resolveIssueDialog == null) {
				JiraPlugin.LOGGER.debug(" ... " + packet + " cancelled");
				return null;
			}

			final JiraVersion fixVersion = resolveIssueDialog.getFixVersion();
			if (fixVersion == null) {
				JiraPlugin.LOGGER.debug(" ... " + packet + " not selected");
				continue;
			}

			JiraPlugin.LOGGER.debug(" ... " + packet + " selected");
			selectedResolvePackets.add(new JiraResolvePacket(packet.getConnection(), packet.getIssueKey(), fixVersion));
		}
		return selectedResolvePackets;
	}

	private static boolean isResolvable(JiraIssue issue) {
		final JiraIssueStatus status = issue.getStatus();
		return status == JiraIssueStatus.IN_PROGRESS || status == JiraIssueStatus.OPEN || status == JiraIssueStatus.REOPENED;
	}

	private static Collection<JiraResolvablePacket> extractResolvablePackets(CommitPacket packet, SettingsServices settingsServices, GuiSpacings spacings, DialogDisplayer dialogDisplayer) {
		JiraPlugin.LOGGER.debug("Extracting resolvable packets for " + packet.getCommitRoot());

		final CommitBugtraqProperties bugtraqProperties = packet.getBugtraqProperties();
		if (bugtraqProperties == null) {
			JiraPlugin.LOGGER.debug(" ... no bugtraq-properties");
			return Collections.emptyList();
		}

		final JiraUrl url = JiraUrl.parseFromBugtraqUrls(bugtraqProperties.getURL());
		if (url == null) {
			JiraPlugin.LOGGER.debug(" ... no bugtraq URL");
			return Collections.emptyList();
		}

		JiraPlugin.LOGGER.debug(" ... URL is " + url);

		final String message = packet.getMessage();
		final String projectKey = url.getProjectKey();
		JiraPlugin.LOGGER.debug(" ... projectKey is " + projectKey);

		final List<JiraResolvablePacket> resolvePackets = new ArrayList<>();
		final JiraUIConnection connection = new JiraUIConnection(url.getBaseUrl(), settingsServices, true, spacings, dialogDisplayer);

		JiraPlugin.LOGGER.debug("Parsing message '" + message + "'");
		if (message != null) {
			for (int index = message.indexOf(projectKey); index != -1; index = message.indexOf(projectKey, index + 1)) {
				final String issueID = extractIssueID(message, projectKey, index);
				if (issueID == null) {
					index++;
					continue;
				}

				JiraPlugin.LOGGER.debug(" ... found issue " + issueID);
				resolvePackets.add(new JiraResolvablePacket(issueID, projectKey, connection));
				index += issueID.length();
			}
		}

		return resolvePackets;
	}

	private static boolean isIssueIDCharacter(char ch) {
		return Character.isLetter(ch) || Character.isDigit(ch) || ch == '-';
	}

	@Nullable
	private static String extractIssueID(String message, String projectKey, int startIndex) {
		if (startIndex > 0 && isIssueIDCharacter(message.charAt(startIndex - 1))) {
			return null;
		}

		final int dashIndex = startIndex + projectKey.length();
		if (message.length() <= dashIndex || message.charAt(dashIndex) != '-') {
			return null;
		}

		int index = dashIndex + 1;

		while (index < message.length() && Character.isDigit(message.charAt(index))) {
			index++;
		}

		if (index < message.length() && isIssueIDCharacter(message.charAt(index))) {
			return null;
		}

		return message.substring(startIndex, index);
	}
}
