package com.syntevo.plugin.committemplate;

import java.io.*;
import java.util.*;

import org.apache.subversion.javahl.*;
import org.apache.subversion.javahl.types.*;

import com.syntevo.openapi.deprecated.smartsvn.command.commit.*;

/**
 * @author syntevo GmbH
 */
final class CommitTemplateCommitWorkflow extends FilterCommitWorkflow {

	// Constants ==============================================================

	private static final String COMMIT_TEMPLATE_PROPERTY_NAME = System.getProperty("commitTemplate.svnPropertyName", "tsvn:logtemplate");

	// Fields =================================================================

	private final CommitWorkflowServices services;

	// Setup ==================================================================

	public CommitTemplateCommitWorkflow(ICommitWorkflow baseWorkflow, CommitWorkflowServices services) {
		super(baseWorkflow);
		this.services = services;
	}

	// Implemented ============================================================

	@Override
	public Set<CommitPacket> prepareForPacketPhases(Set<CommitPacket> packets) {
		for (CommitPacket packet : packets) {
			if (packet.getMessage() == null) {
				packet.setMessage(getTemplateMessage(packet.getCommitRoot().getFile()));
			}
		}
		return super.prepareForPacketPhases(packets);
	}

	// Utils ==================================================================

	private String getTemplateMessage(File root) {
		final byte[] propertyData;
		final String propertyValue;
		try {
			final ISVNClient svnClient = this.services.getClients().createClient();
			try {
				propertyData = svnClient.propertyGet(root.getAbsolutePath(),
				                                     COMMIT_TEMPLATE_PROPERTY_NAME,
				                                     Revision.WORKING, Revision.WORKING);
			} catch (SubversionException ex) {
				CommitTemplatePlugin.LOGGER.error("Failed to load commit message template by reading the property " + COMMIT_TEMPLATE_PROPERTY_NAME + " from " + root, ex);
				return null;
			} finally {
				svnClient.dispose();
			}
		} catch (SubversionException ex) {
			CommitTemplatePlugin.LOGGER.error("Failed to instantiate client connection", ex);
			return null;
		}

		if (propertyData == null || propertyData.length == 0) {
			return null;
		}

		try {
			propertyValue = new String(propertyData, "ASCII");
		} catch (UnsupportedEncodingException ex) {
			CommitTemplatePlugin.LOGGER.warn("Failed to decode property " + COMMIT_TEMPLATE_PROPERTY_NAME + " value set on " + root, ex);
			return null;
		}
		return propertyValue;
	}
}
