/*******************************************************************************
 * Copyright (c) 2000, 2011 IBM Corporation and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 *
 * Contributors:
 *     IBM Corporation - initial API and implementation
 *     Syntevo GmbH    - major redesign
 *******************************************************************************/
package com.syntevo.q.swt;

import org.eclipse.swt.*;
import org.eclipse.swt.graphics.*;
import org.eclipse.swt.widgets.*;

final class QTabFolderRenderer {

	// Constants ==============================================================

	private static final int BUTTON_SIZE = 18;

	private static final int BUTTON_BORDER = SWT.COLOR_WIDGET_DARK_SHADOW;
	private static final int BUTTON_FILL = SWT.COLOR_LIST_BACKGROUND;

	private static final int ITEM_TOP_MARGIN = 2;
	private static final int ITEM_BOTTOM_MARGIN = 2;
	private static final int ITEM_LEFT_MARGIN = 4;
	private static final int ITEM_RIGHT_MARGIN = 4;
	public static final int INTERNAL_SPACING = 4;
	private static final int FLAGS = SWT.DRAW_TRANSPARENT | SWT.DRAW_MNEMONIC;
	private static final String ELLIPSIS = "..."; //$NON-NLS-1$

	//Part constants
	public static final int PART_BODY = -1;
	public static final int PART_HEADER = -2;
	public static final int PART_BORDER = -3;
	public static final int PART_CHEVRON_BUTTON = -7;
	public static final int PART_CLOSE_BUTTON = -8;

	public static final int MINIMUM_SIZE = 1 << 24; //TODO: Should this be a state?
	public static final int MIN_CHARS = 20;

	// Fields =================================================================

	private final QTabFolder tabFolder;

	// Setup ==================================================================

	public QTabFolderRenderer(QTabFolder tabFolder) {
		if (tabFolder == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
		}
		this.tabFolder = tabFolder;
	}

	// Accessing ==============================================================

	public int getMinTabWidth(QTabFolderItem item, GC gc) {
		return getTabSize(item, gc, true).x;
	}

	public int getTabWidth(QTabFolderItem item, GC gc) {
		return getTabSize(item, gc, false).x;
	}

	public Point getChevronButtonSize() {
		return new Point(3 * BUTTON_SIZE / 2, BUTTON_SIZE);
	}

	public Point getCloseButtonSize() {
		return new Point(BUTTON_SIZE, BUTTON_SIZE);
	}

	public int getHeaderHeight() {
		final int fixedTabHeight = tabFolder.getFixedTabHeight();
		if (fixedTabHeight != SWT.DEFAULT) {
			return fixedTabHeight + 1;
		}

		final GC gc = new GC(tabFolder);
		final int itemCount = tabFolder.getItemCount();
		int height = 0;
		if (itemCount == 0) {
			height = gc.textExtent("Default", FLAGS).y + ITEM_TOP_MARGIN + ITEM_BOTTOM_MARGIN; //$NON-NLS-1$
		}
		else {
			for (int i = 0; i < itemCount; i++) {
				final QTabFolderItem item = tabFolder.getItem(i);
				final Point tabSize = getTabSize(item, gc, false);
				height = Math.max(height, tabSize.y);
			}
		}
		gc.dispose();
		return height;
	}

	public Rectangle computeBodyTrim(int x, int y, int width, int height) {
		final int borderSize = tabFolder.isBorderVisible() ? 1 : 0;
		final int tabHeight = tabFolder.getTabHeight();
		final int style = tabFolder.getStyle();
		int highlight_header = (style & SWT.FLAT) != 0 ? 1 : 3;
		if (tabFolder.getFixedTabHeight() == 0 && (style & SWT.FLAT) != 0 && (style & SWT.BORDER) == 0) {
			highlight_header = 0;
		}
		x -= borderSize;
		width += 2 * borderSize;
		y -= highlight_header + tabHeight;
		height += borderSize + tabHeight + highlight_header;
		return new Rectangle(x, y, width, height);
	}

	public Rectangle computeBorderTrim() {
		final int borderSize = tabFolder.isBorderVisible() ? 1 : 0;
		return new Rectangle(-borderSize, 0, 2 * borderSize, borderSize);
	}

	public Rectangle computeHeaderTrim(int x, int y, int width, int height) {
		return new Rectangle(x, y, width, height);
	}

	public Rectangle computeTabTrim(int width, int height) {
		width += ITEM_LEFT_MARGIN + ITEM_RIGHT_MARGIN;
		height += ITEM_TOP_MARGIN + ITEM_BOTTOM_MARGIN;
		return new Rectangle(-ITEM_LEFT_MARGIN, -ITEM_TOP_MARGIN, width, height);
	}

	public void drawTab(QTabFolderItem item, GC gc) {
		if (item.isZeroSize()) {
			return;
		}

		if ((item.getState() & SWT.SELECTED) != 0) {
			drawSelectedTab(item, gc);
		}
		else {
			drawUnselectedTab(item, gc);
		}
	}

	public void drawBody(Point size, GC gc) {
		if (size == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
			return;
		}

		if (gc == null) {
			SWT.error(SWT.ERROR_NULL_ARGUMENT);
			return;
		}

		final int tabHeight = tabFolder.getTabHeight();

		final int borderLeft = tabFolder.isBorderVisible() ? 1 : 0;

		final int style = tabFolder.getStyle();
		final int highlight_header = (style & SWT.FLAT) != 0 ? 1 : 3;

		// fill in body
		final int width = size.x - borderLeft - borderLeft;
		final int height = size.y - borderLeft - tabHeight - highlight_header;
		// Draw client area
		if ((tabFolder.getStyle() & SWT.NO_BACKGROUND) != 0) {
			gc.setBackground(tabFolder.getBackground());
			gc.fillRectangle(borderLeft, tabHeight + highlight_header, width, height);
		}

		// draw 1 pixel border left, below and right to the content
		if (borderLeft > 0) {
			gc.setForeground(tabFolder.getBorderColor());
			final int x1 = borderLeft - 1;
			final int x2 = size.x - borderLeft;
			final int y2 = size.y - borderLeft;
			gc.drawLine(x1, tabHeight, x1, y2); // left
			gc.drawLine(x2, tabHeight, x2, y2); // right
			gc.drawLine(x1, y2, x2, y2); // bottom
		}
	}

	@SuppressWarnings({"UnnecessaryCodeBlock"})
	public void drawChevron(int chevronImageState, Rectangle chevronRect, GC gc) {
		if (chevronRect.width == 0 || chevronRect.height == 0) {
			return;
		}

		// draw chevron (10x7)
		final Display display = tabFolder.getDisplay();
		final Point dpi = display.getDPI();
		final int fontHeight = 72 * 10 / dpi.y;
		final FontData fd = tabFolder.getFont().getFontData()[0];
		fd.setHeight(fontHeight);
		final Font f = new Font(display, fd);
		final int fHeight = f.getFontData()[0].getHeight() * dpi.y / 72;
		final int indent = Math.max(2, (chevronRect.height - fHeight - 4) / 2);
		final int x = chevronRect.x + 2;
		final int y = chevronRect.y + indent;
		final int count;
		final int itemCount = tabFolder.getItemCount();
		int showCount = 0;
		while (showCount < tabFolder.getPriority().length && tabFolder.getItem(tabFolder.getPriority()[showCount]).isShowing()) {
			showCount++;
		}
		count = itemCount - showCount;
		final String chevronString = count > 99 ? "99+" : String.valueOf(count); //$NON-NLS-1$
		switch (chevronImageState & (SWT.HOT | SWT.SELECTED)) {
		case SWT.NONE: {
			final Color chevronBorder = tabFolder.getForeground();
			gc.setForeground(chevronBorder);
			gc.setFont(f);
			gc.drawLine(x, y, x + 2, y + 2);
			gc.drawLine(x + 2, y + 2, x, y + 4);
			gc.drawLine(x + 1, y, x + 3, y + 2);
			gc.drawLine(x + 3, y + 2, x + 1, y + 4);
			gc.drawLine(x + 4, y, x + 6, y + 2);
			gc.drawLine(x + 6, y + 2, x + 5, y + 4);
			gc.drawLine(x + 5, y, x + 7, y + 2);
			gc.drawLine(x + 7, y + 2, x + 4, y + 4);
			gc.drawString(chevronString, x + 7, y + 3, true);
			break;
		}
		case SWT.HOT: {
			gc.setForeground(display.getSystemColor(BUTTON_BORDER));
			gc.setBackground(display.getSystemColor(BUTTON_FILL));
			gc.setFont(f);
			gc.fillRoundRectangle(chevronRect.x, chevronRect.y, chevronRect.width, chevronRect.height, 6, 6);
			gc.drawRoundRectangle(chevronRect.x, chevronRect.y, chevronRect.width - 1, chevronRect.height - 1, 6, 6);
			gc.drawLine(x, y, x + 2, y + 2);
			gc.drawLine(x + 2, y + 2, x, y + 4);
			gc.drawLine(x + 1, y, x + 3, y + 2);
			gc.drawLine(x + 3, y + 2, x + 1, y + 4);
			gc.drawLine(x + 4, y, x + 6, y + 2);
			gc.drawLine(x + 6, y + 2, x + 5, y + 4);
			gc.drawLine(x + 5, y, x + 7, y + 2);
			gc.drawLine(x + 7, y + 2, x + 4, y + 4);
			gc.drawString(chevronString, x + 7, y + 3, true);
			break;
		}
		case SWT.SELECTED: {
			gc.setForeground(display.getSystemColor(BUTTON_BORDER));
			gc.setBackground(display.getSystemColor(BUTTON_FILL));
			gc.setFont(f);
			gc.fillRoundRectangle(chevronRect.x, chevronRect.y, chevronRect.width, chevronRect.height, 6, 6);
			gc.drawRoundRectangle(chevronRect.x, chevronRect.y, chevronRect.width - 1, chevronRect.height - 1, 6, 6);
			gc.drawLine(x + 1, y + 1, x + 3, y + 3);
			gc.drawLine(x + 3, y + 3, x + 1, y + 5);
			gc.drawLine(x + 2, y + 1, x + 4, y + 3);
			gc.drawLine(x + 4, y + 3, x + 2, y + 5);
			gc.drawLine(x + 5, y + 1, x + 7, y + 3);
			gc.drawLine(x + 7, y + 3, x + 6, y + 5);
			gc.drawLine(x + 6, y + 1, x + 8, y + 3);
			gc.drawLine(x + 8, y + 3, x + 5, y + 5);
			gc.drawString(chevronString, x + 8, y + 4, true);
			break;
		}
		}
		f.dispose();
	}

	public void drawHeader(GC gc) {
		final Point size = tabFolder.getSize();
		final int tabHeight = tabFolder.getTabHeight();
		final int style = tabFolder.getStyle();

		final int borderLeft = tabFolder.isBorderVisible() ? 1 : 0;

		final int highlight_header = (style & SWT.FLAT) != 0 ? 1 : 3;

		final int x = Math.max(0, borderLeft - 1);
		final int width = size.x - borderLeft - borderLeft + 1;
		final int height = tabHeight - 1;
		// Draw Tab Header
		// Fill in background
		drawBackground(gc, false);

		// Draw border line (#)
		// +---------+#################################
		// |         |                                #
		// |         |                                #
		if (borderLeft > 0) {
			gc.setForeground(tabFolder.getBorderColor());

			final int[] shape = new int[8];
			shape[0] = x;
			shape[1] = height + highlight_header + 1;
			shape[2] = x;
			shape[3] = 0;
			shape[4] = x + width;
			shape[5] = 0;
			shape[6] = x + width;
			shape[7] = height + highlight_header + 1;
			gc.drawPolyline(shape);
		}
	}

	// Utils ==================================================================

	private Point getTabSize(QTabFolderItem item, GC gc, boolean minimum) {
		if (item.isDisposed()) {
			return new Point(0, 0);
		}

		int width = 0;
		int height = 0;

		String text;
		if (minimum) {
			text = item.getText();
			if (text != null && text.length() > MIN_CHARS) {
				final int end = MIN_CHARS - ELLIPSIS.length();
				text = text.substring(0, end);
				text += ELLIPSIS;
			}
		}
		else {
			text = item.getText();
		}

		if (text != null) {
			if (width > 0) {
				width += INTERNAL_SPACING;
			}
			final Font font = tabFolder.getFont();
			if (font == null) {
				final Point size = gc.textExtent(text, FLAGS);
				width += size.x;
				height = Math.max(height, size.y);
			}
			else {
				final Font gcFont = gc.getFont();
				gc.setFont(font);
				final Point size = gc.textExtent(text, FLAGS);
				width += size.x;
				height = Math.max(height, size.y);
				gc.setFont(gcFont);
			}
		}

		if (tabFolder.isShowClose() || item.isShowClose()) {
			if (tabFolder.isShowUnselectedClose() || tabFolder.isSelected(item)) {
				if (width > 0) {
					width += INTERNAL_SPACING;
				}
				width += getCloseButtonSize().x;
			}
		}

		final Rectangle trim = computeTabTrim(width, height);
		return new Point(trim.width, trim.height);
	}

	private void drawBackground(GC gc, boolean selected) {
		final Point size = tabFolder.getSize();
		int width = size.x;
		final int height = tabFolder.getTabHeight() + ((tabFolder.getStyle() & SWT.FLAT) != 0 ? 1 : 3);
		int x = 0;

		final int borderLeft = tabFolder.isBorderVisible() ? 1 : 0;
		final int borderTop = 0;

		if (borderLeft > 0) {
			x += 1;
			width -= 2;
		}
		drawBackground(gc, x, borderTop, width, height, getTabFolderBackground(selected));
	}

	private Color getTabFolderBackground(boolean selected) {
		return selected ? tabFolder.getSelectionBackground() : tabFolder.getBackground();
	}

	private void drawBackground(GC gc, int x, int y, int width, int height, Color background) {
		if ((tabFolder.getStyle() & SWT.NO_BACKGROUND) != 0 || !background.equals(tabFolder.getBackground())) {
			gc.setBackground(background);
			gc.fillRectangle(x, y, width, height);
		}
	}

	private void drawBorder(GC gc, int[] shape) {
		gc.setForeground(tabFolder.getBorderColor());
		gc.drawPolyline(shape);
	}

	private void drawClose(GC gc, Rectangle closeRect, int closeImageState, boolean selected) {
		if (closeRect.width == 0 || closeRect.height == 0) {
			return;
		}

		final Display display = tabFolder.getDisplay();

		// draw X 9x9
		final int x = closeRect.x + Math.max(1, (closeRect.width - 9) / 2);
		final int y = closeRect.y + Math.max(1, (closeRect.height - 9) / 2) + 1;

		closeImageState &= SWT.HOT | SWT.SELECTED | SWT.BACKGROUND;
		if (closeImageState == SWT.BACKGROUND) {
			drawBackground(gc, false);
		}
		else {
			final Color crossColor;
			if (closeImageState != SWT.NONE) {
				// SU-4249
				// antialiasing must only be enabled for the oval, not for the cross because
				// otherwise the background color of other tabs will be corrupted
				final int prevAntialias = gc.getAntialias();
				gc.setAntialias(SWT.ON);
				try {
					gc.setBackground(display.getSystemColor(SWT.COLOR_WIDGET_NORMAL_SHADOW));
					gc.fillOval(x, y, 11, 11);
					gc.setBackground(display.getSystemColor(SWT.COLOR_WIDGET_LIGHT_SHADOW));
					gc.fillOval(x, y, 10, 10);
				}
				finally {
					gc.setAntialias(prevAntialias);
				}

				crossColor = display.getSystemColor(BUTTON_BORDER);
			}
			else {
				final Color background = getTabFolderBackground(selected);
				final Color darkColor = display.getSystemColor(BUTTON_BORDER);
				final Color lightColor = display.getSystemColor(SWT.COLOR_WIDGET_LIGHT_SHADOW);
				final int darkDiff = colorDiff(darkColor, background);
				final int lightDiff = colorDiff(lightColor, background);
				crossColor = darkDiff > lightDiff ? darkColor : lightColor;
			}

			gc.setForeground(crossColor);
			drawCross(gc, x, y);
		}
	}

	private void drawCross(GC gc, int x, int y) {
		final int leftTop = 3;
		final int rightBottom = 7;
		gc.drawLine(x + leftTop, y + leftTop, x + rightBottom, y + rightBottom);
		gc.drawLine(x + leftTop, y + rightBottom, x + rightBottom, y + leftTop);
	}

	/*
	 * Draw the unselected border for the receiver on the left.
	 *
	 * @param gc
	 */
	private void drawLeftUnselectedBorder(GC gc, Rectangle bounds) {
		final int x = bounds.x;
		final int y = bounds.y;
		final int height = bounds.height;

		drawBorder(gc, new int[] {
				x, y + height,
				x, y
		});
	}

	/*
	 * Draw the unselected border for the receiver on the right.
	 *
	 * @param gc
	 */
	private void drawRightUnselectedBorder(GC gc, Rectangle bounds) {
		final int x = bounds.x;
		final int y = bounds.y;
		final int width = bounds.width;
		final int height = bounds.height;

		final int startX = x + width - 1;

		drawBorder(gc, new int[] {
				startX, y,
				startX, y + height
		});
	}

	private void drawSelectedTab(QTabFolderItem item, GC gc) {
		final Rectangle bounds = item.getBounds();
		final int x = bounds.x;
		final int y = bounds.y;
		final int height = bounds.height;
		final int width = bounds.width;

		final Point size = tabFolder.getSize();

		final int rightEdge = Math.min(x + width, tabFolder.getRightItemEdge());

		// background
		final int highlight_header = (tabFolder.getStyle() & SWT.FLAT) != 0 ? 1 : 3;
		gc.setBackground(tabFolder.getSelectionBackground());
		gc.fillRectangle(1, tabFolder.getTabHeight() + 1, size.x - 2, highlight_header - 1);

		if (!item.isShowing()) {
			final int x1 = 0;
			final int y1 = y + height;
			final int x2 = size.x - 1;
			gc.setForeground(tabFolder.getBorderColor());
			gc.drawLine(x1, y1, x2, y1);
			return;
		}

		// draw selected tab background and outline
		final Rectangle clipping = gc.getClipping();
		bounds.height++;
		if (clipping.intersects(bounds)) {
			// fill in tab background
			// 0        x         rightEdge-1
			//          +---------+  y
			//          +---------+  y + height  + 1
			drawBackground(gc, x, y + 1, width, height, tabFolder.getSelectionBackground());
		}

		// draw outline
		// 0        x         rightEdge-1        size.x
		//          +---------+                    y
		// ---------+         +------------------- y + height
		final int[] shape = new int[12];
		shape[0] = 0;
		shape[1] = y + height;
		shape[2] = x;
		shape[3] = y + height;
		shape[4] = x;
		shape[5] = y;
		shape[6] = rightEdge - 1;
		shape[7] = y;
		shape[8] = rightEdge - 1;
		shape[9] = y + height;
		shape[10] = size.x;
		shape[11] = y + height;

		gc.setForeground(tabFolder.getBorderColor());
		gc.drawPolyline(shape);

		if (!clipping.intersects(bounds)) {
			return;
		}

		// foreground
		final Rectangle trim = computeTabTrim(0, 0);
		final int xDraw = x - trim.x;

		// draw Text
		int textWidth = rightEdge - xDraw - (trim.width + trim.x);
		final Rectangle closeRect = item.getCloseRect();
		if (closeRect.width > 0) {
			textWidth -= closeRect.width + INTERNAL_SPACING;
		}
		if (textWidth > 0) {
			final Font gcFont = gc.getFont();
			gc.setFont(tabFolder.getFont());

			if (item.getShortenedText() == null || item.getShortenedTextWidth() != textWidth) {
				item.setShortenedText(shortenText(gc, item.getText(), textWidth));
				item.setShortenedTextWidth(textWidth);
			}
			final Point extent = gc.textExtent(item.getShortenedText(), FLAGS);
			int textY = y + (height - extent.y) / 2;
			textY += 1;

			gc.setForeground(tabFolder.getSelectionForeground());
			gc.drawText(item.getShortenedText(), xDraw, textY, FLAGS);
			gc.setFont(gcFont);
		}
		if (tabFolder.isShowClose() || item.isShowClose()) {
			drawClose(gc, closeRect, item.getCloseImageState(), true);
		}
	}

	private void drawUnselectedTab(QTabFolderItem item, GC gc) {
		// Do not draw partial items
		if (!item.isShowing()) {
			return;
		}

		final Rectangle bounds = item.getBounds();
		final Rectangle clipping = gc.getClipping();
		if (!clipping.intersects(bounds)) {
			return;
		}

		final int x = bounds.x;
		final int y = bounds.y;
		final int height = bounds.height;
		final int width = bounds.width;

		// background
		final int index = tabFolder.indexOf(item);
		if (index > 0 && index < tabFolder.getSelectedIndex()) {
			drawLeftUnselectedBorder(gc, bounds);
		}
		// If it is the last one then draw a line
		if (index > tabFolder.getSelectedIndex()) {
			drawRightUnselectedBorder(gc, bounds);
		}

		// foreground
		final Rectangle trim = computeTabTrim(0, 0);
		final int xDraw = x - trim.x;
		// draw Text
		int textWidth = x + width - xDraw - (trim.width + trim.x);
		if (tabFolder.isShowUnselectedClose() && (tabFolder.isShowClose() || item.isShowClose())) {
			textWidth -= item.getCloseRect().width + INTERNAL_SPACING;
		}
		if (textWidth > 0) {
			final Font gcFont = gc.getFont();
			gc.setFont(tabFolder.getFont());
			if (item.getShortenedText() == null || item.getShortenedTextWidth() != textWidth) {
				item.setShortenedText(shortenText(gc, item.getText(), textWidth));
				item.setShortenedTextWidth(textWidth);
			}
			final Point extent = gc.textExtent(item.getShortenedText(), FLAGS);
			int textY = y + (height - extent.y) / 2;
			textY += 1;
			gc.setForeground(tabFolder.getForeground());
			gc.drawText(item.getShortenedText(), xDraw, textY, FLAGS);
			gc.setFont(gcFont);
		}
		// draw close
		if (tabFolder.isShowUnselectedClose() && (tabFolder.isShowClose() || item.isShowClose())) {
			drawClose(gc, item.getCloseRect(), item.getCloseImageState(), false);
		}
	}

	private String shortenText(GC gc, String text, int width) {
		if (gc.textExtent(text, FLAGS).x <= width) {
			return text;
		}

		final int ellipseWidth = gc.textExtent(ELLIPSIS, FLAGS).x;
		final int length = text.length();
		final TextLayout layout = new TextLayout(tabFolder.getDisplay());
		layout.setText(text);
		int end = layout.getPreviousOffset(length, SWT.MOVEMENT_CLUSTER);
		while (end > 0) {
			text = text.substring(0, end);
			final int l = gc.textExtent(text, FLAGS).x;
			if (l + ellipseWidth <= width) {
				break;
			}
			end = layout.getPreviousOffset(end, SWT.MOVEMENT_CLUSTER);
		}
		layout.dispose();
		return end == 0 ? text.substring(0, 1) : text + ELLIPSIS;
	}

	private static int colorDiff(Color c1, Color c2) {
		final int dR = Math.abs(c1.getRed() - c2.getRed());
		final int dG = Math.abs(c1.getGreen() - c2.getGreen());
		final int dB = Math.abs(c1.getBlue() - c2.getBlue());
		return dR * dR + dG + dG + dB * dB;
	}
}
