/** @file
    @brief    Check sets of conglomerations

  $Id: ExpectationConglomeration.h 1491 2010-01-02 22:21:45Z ewald-arnold $

 ***************************************************************************/

/**************************************************************************

   begin                : Sat Dec 21 2002
   copyright            : (C) 2002-2010 by Ewald Arnold
   email                : mockpp at ewald-arnold dot de

   This program is free software; you can redistribute it and/or modify
   it under the terms of the GNU Lesser General Public License as
   published by the Free Software Foundation; either version 2 of the License,
   or (at your option) any later version.

   This program is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
   GNU General Public License for more details.

   You should have received a copy of the GNU Lesser General Public License
   along with this program; if not, write to the Free Software
   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

 **/

#ifndef MOCKPP_EXPECTATIONCONGLOMERATION_H
#define MOCKPP_EXPECTATIONCONGLOMERATION_H

#include <mockpp/mockpp.h> // always first

#include MOCKPP_ALGORITHM_H
#include MOCKPP_SET_H
#include MOCKPP_VECTOR_H

#include <mockpp/AbstractExpectationCollection.h>
#include <mockpp/util/AssertMo.h>
#include <mockpp/ExpectationBoundary.h>


MOCKPP_NS_START


/** A class to verify conglomerations of expectations: each expected object must occur at
  * least once.
  * \ingroup grp_basic_mo
  */
template <class T>
class ExpectationConglomeration : public AbstractExpectationCollection<T>
{

  public:

  /** Constructs the expectation.
    * @param name    human readable description about the expectation
    * @param parent  parent verifiable
    */
    ExpectationConglomeration(const String &name, VerifiableList *parent = 0)
      : AbstractExpectationCollection<T>(name, parent),
        expectNothing(false),
        haveActualValue(false)
    {
    }


  /** Adds another actual value to the list.
    * @param actualItem  new value
    */
    void addActual(const T &actualItem)
    {
      actualItems.insert(actualItem);
      haveActualValue = true;

      if (this->shouldCheckImmediately())
        checkImmediateValue(actualItem);
    }


  /** Adds a sequence of actual values to the set.
    * @param items    start iterator
    * @param end      terminating iterator (note: one element "behind" as always with STL)
    */
    template <class I>
    void addActual(I items, I end)
    {
      for ( /* -- */; items != end; ++items)
        addActual(*items);
    }


  /** Adds another expectation value to the list.
    * @param expectedItem  new value
    * @return reference to itself for chaining
    */
    ExpectationConglomeration& addExpected(const T &expectedItem)
    {
      expectedSingleItems.push_back(expectedItem);
      expectNothing = false;
      this->setHasExpectations();
      return *this;
    }


  /** Adds another expectation boundary to the list.
    * @param lower   new lower value
    * @param upper   new upper value
    * @return reference to itself for chaining
    */
    ExpectationConglomeration& addExpectedBoundary(const T &lower, const T &upper)
    {
      expectedBoundaryItems.push_back(Boundary (lower, upper));
      expectNothing = false;
      this->setHasExpectations();
      return *this;
    }


  /** Adds a sequence of expectation values to the set.
    * @param items    start iterator
    * @param end      terminating iterator (note: one element "behind" as always with STL)
    * @return reference to itself for chaining
    */
    template <class I>
    ExpectationConglomeration& addExpected(I items, I end)
    {
      for ( /* -- */; items != end; ++items)
        addExpected(*items);
      return *this;
    }


  /** Resets the internal state to reflect that there is no actual value set.
    */
    virtual void reset()
    {
      this->clearFailOnVerify();
      clearActual();
      clearExpectation();
      expectNothing = false;
    }


  /**
    * Resets the internal state to reflect that there is no actual value set.
    */
    virtual void clearActual()
    {
      haveActualValue = false;
      actualItems.clear();
    }


  /**
    * Verify that the expected values equal the expected ones.
    * Note: in a set there is always only one element of a given value and the order
    * of the elements is not relevant.
    * If it fails, an AssertionFailedError is thrown
    */
    virtual void verify()
    {
      if( expectNothing)
      {
        String fmt = mockpp_i18n(MOCKPP_PCHAR("%1 expected no value."));
        fmt << getVerifiableName();
        MOCKPP_ASSERT_FALSE_MESSAGE(fmt, haveActualValue );
      }
      else
      {
        if (!this->hasExpectations() )
          return;

        String fmt = mockpp_i18n(MOCKPP_PCHAR("%1 expected a value."));
        fmt << this->getVerifiableName();
        MOCKPP_ASSERT_TRUE_MESSAGE(fmt, haveActualValue );

        fmt = mockpp_i18n(MOCKPP_PCHAR("%1 did not receive the expected item set."));
        fmt << this->getVerifiableName();
        typename MOCKPP_STL::set<T>::const_iterator it;
        for (it = actualItems.begin(); it != actualItems.end(); ++it)
          MOCKPP_ASSERT_TRUE_MESSAGE(fmt, contains(*it));
      }
    }


  /**
    * Tell the object to expect nothing to happen to it, perhaps because the test is exercising
    * the handling of an error. The Expectation will fail if any actual values are set.
    *
    * Note that this is not the same as not setting any expectations, in which case verify()
    * will do nothing.
    */
    virtual void setExpectNothing()
    {
      expectNothing = true;
      clearExpectation();
      this->setHasExpectations();
    }


  /**
    * Checks if the actual value is contained in the conglomeration.
    * @param val  the value to search
    * @return  true: the value was found
    */
    bool contains(const T &val) const
    {
      if (MOCKPP_STL::find(expectedSingleItems.begin(), expectedSingleItems.end(), val) != expectedSingleItems.end())
         return true;

      typename MOCKPP_STL::vector<Boundary>::const_iterator it;
      for (it = expectedBoundaryItems.begin(); it != expectedBoundaryItems.end(); ++it)
        if ( (*it).contains(val))
          return true;

      return false;
    }


    AbstractExpectationCollection<T>::getVerifiableName;

  protected:

  /**
    * Clears the expectation list.
    */
    virtual void clearExpectation()
    {
      this->clearHasExpectations();
      expectedBoundaryItems.clear();
      expectedSingleItems.clear();
    }


  /**
    * Checks if the actual value matches the expectation.
    * Note: in a set there is always only one element of a given value and the order
    * of the elements is not relevant.
    * If it fails, an AssertionFailedError is thrown
    */
    virtual void checkImmediateValue(const T &actualItem) const
    {
      String fmt = mockpp_i18n(MOCKPP_PCHAR("%1 did not receive an expected item.\nUnexpected: %2"));
      fmt << getVerifiableName() << actualItem;

      MOCKPP_ASSERT_TRUE_MESSAGE(fmt, contains(actualItem));
    }


  private:

    class Boundary
    {
      public:

        Boundary (const T &low, const T &up)
         : lower(low)
         , upper(up)
        {
        }

        bool contains(const T &val) const
        {
          return val <= upper && val >= lower;
        }

      private:

       T lower;
       T upper;
    };

    MOCKPP_STL::set<T>           actualItems;
    MOCKPP_STL::vector<T>        expectedSingleItems;
    MOCKPP_STL::vector<Boundary> expectedBoundaryItems;
    bool                  expectNothing;
    bool                  haveActualValue;
};



MOCKPP_NS_END


#endif // MOCKPP_EXPECTATIONCONGLOMERATION_H

