// vm.s
//
// The kForth Virtual Machine
//
// Copyright (c) 1998--2011 Krishna Myneni, Creative Consulting for
//   Research and Education
//
// This software is provided under the terms of the General Public License.
//
// Usage from C++
//
//       extern "C" int vm (byte* ip);
//       ecode = vm(ip);
//
// Originally written for the A386 assembler
// Revisions:
//	3-21-98
//	8-25-1998 ported to GNU assembler under Linux
//	9-8-1998  additional functions added: c@, c!, f=, f<>
//	9-10-1998 added: and, or, not, xor
//	9-11-1998 added: = , <>, <, >, <=, >=
//	9-15-1998 added: +!, fsqrt, fsin, fcos, -rot, sp@, rp@
//	9-17-1998 added: a@
//	9-27-1998 added: mod, 2*, 2/
//	10-1-1998 added: pick
//	10-4-1998 fixed signed integer division and mod; added /MOD
//	10-6-1998 added ?dup
//	10-14-1998 added count
//	10-16-1998 fixed L_div error
//	10-19-1998 added 0<, 0=, 0>, true, false
//	10-20-1998 added 2+, 2-
//	02-09-1999 added execute
//	03-01-1999 added open, lseek, close, read, write
//	03-02-1999 added ioctl
//	03-03-1999 added usleep	
//	03-07-1999 added fill, cmove
//	03-09-1999 interchanged meaning of execute and call
//		   to be consistent with ANS Forth
//	03-27-1999 added +loop, unloop
//	03-29-1999 added roll
//	03-31-1999 added cmove>, key
//	05-05-1999 fixed +loop
//	05-06-1999 added fround
//	05-15-1999 added floor
//	05-26-1999 added fatan2, lshift, rshift
//	05-27-1999 added u<, quit, base
//	06-02-1999 added */, */mod
//	06-09-1999 call CPP functions from vm
//	07-18-1999 added find
//	09-06-1999 added pTIB, word, tick
//	09-12-1999 added system
//	10-04-1999 added create, variable, fvariable as intrinsic words
//	10-06-1999 added constant, fconstant as intrinsic words
//	10-07-1999 added chdir
//      10-08-1999 added erase, brackettick
//	10-09-1999 added time&date, ms, question, bl
//      10-10-1999 added char, forget, cold
//      10-20-1999 added >file, console
//	10-28-1999 added key?
//	12-24-1999 added hooks for f0=, f0<, f0>, u>, s>d, d>f, f>d, 
//	              um*, um/mod, m*, m+, m/, m*/
//	12-25-1999 added cells, cell+, dfloats, dfloat+
//	12-27-1999 added bye
//	01-08-2000 fixed f0<, increased vm loop efficiency
//	01-13-1999 added ?allot, fixed f0=
//	01-22-2000 modified + to remove ordering sensitivity for address arithmetic
//	01-23-2000 added 0<>, [char], .r, u.r, changed opcodes for 
//	             relational operators
//	01-24-2000 added CPP_literal, CPP_cquote, CPP_squote, CPP_dotquote
//	02-04-2000 implemented m*, um*
//	02-26-2000 fixed fm/mod, added sm/rem, um/mod
//	03-02-2000 modified QUIT to clear the return stacks, added CPP_do
//	03-05-2000 added CPP_begin, CPP_while, CPP_repeat, CPP_until, CPP_again,
//	             CPP_leave, CPP_if, CPP_else, CPP_then, CPP_lparen
//	05-17-2000 added CPP_does
//	05-18-2000 fix L_plusloop for negative increments
//	06-04-2000 fix L_roll to roll the typestack as well
//	06-11-2000 added CPP_case, CPP_endcase, CPP_of, and CPP_endof
//	06-15-2000 added CPP_querydo, CPP_abortquote
//	09-05-2000 handled overflow condition on um/mod; added code for m/;
//		     added m+	;  hooks for CPP_lbracket, CPP_rbracket, CPP_ddot
//	11-30-2000 added L_dabs, L_dnegate, L_udmstar, L_mstarslash
//	01-26-2001 changed jumptable address for usleep to C_usec
//	04-01-2001 changed jmpl to jmp for L_fzerolt
//	04-22-2001 added L_dplus, L_dminus
//	04-24-2001 added L_twopush, L_twopop, L_tworfetch
//      05-07-2001 added L_definition
//	05-08-2001 improved vm execution efficiency by about 10%
//	05-13-2001 added CPP_dotparen, L_dlt, L_dzeroeq, L_deq
//	05-20-2001 added CPP_sharp, CPP_bracketsharp, CPP_sharpbracket, 
//	             CPP_sharps, CPP_sign, CPP_hold; fixed problem with L_dnegate
//	07-12-2001 fixed type-stack alignment problem with L_utmslash
//	08-26-2001 updated L_fval to avoid routing fp through NDP, changed all
//	             typestack settings of OP_FVAL to OP_IVAL
//	09-02-2001 added L_tobody
//      12-08-2001 added CPP_evaluate
//      07-29-2002 fixed L_ret to set GlobalRp and GlobalRtp to bottom upon
//                   return from vm.
//	07-31-2002 fixed L_cmovefrom for zero count argument; added
//		     CPP_backslash; fixed bug in L_fmslashmod for divisor < 0
//	08-01-2002 added C_msfetch, C_search, C_compare
//      09-11-2002 modified vm and L_ret to make vm re-entrant
//	09-29-2002 restored L_usleep since it provides a different function 
//	             than L_ms, added CPP_immediate
//	04-11-2003 renamed L_ftos to L_froundtos
//	04-15-2003 added L_ftrunc and L_ftrunctos
//	04-18-2003 fixed L_ftrunctos typestack pointer update
//	06-15-2003 recoded L_rot, L_swap, and L_over for increased efficiency
//	01-31-2004 extended JumpTable past 256 entries, 
//                   added CPP_include, CPP_source, CPP_refill, and
//                   CPP_nondeferred, CPP_state, CPP_colon, CPP_semicolon
//	02-09-2004 added CPP_allocate and CPP_free
//	03-18-2004 added L_fsincos, C_fsinh, C_fcosh, C_ftanh, C_fasinh,
//                   C_facosh, C_fatanh
//      03-21-2004 implemented L_fsincos
//      03-30-2004 modified L_ret to remove unneeded instructions
//      04-08-2004 added CPP_postpone and CPP_compilecomma
//      04-09-2004 added CPP_semicolon
//      04-18-2004 completed recoding with macros for hybrid threading.
//	04-26-2004 exported L_udmstar and L_utmslash for use by ForthVM.cpp
//      06-20-2004 modified L_call to prevent loss of return error code
//      09-05-2004 added C_forth_signal, C_raise, C_setitimer, C_getitimer
//      09-10-2004 moved CPP_spstore and CPP_rpstore to opcodes below 256
//      09-28-2005 changed some comment markers to // to prevent GNU as from quirky errors
//	02-10-2006 The EBP register is now used to hold GlobalIp for the jump-threaded portion
//                   of the VM. This results in 10--20% speedup for typical apps.
//                   Any additions to code in vm.s which needs to make use of the EBP
//		     register should save and restore this register using "pushl %ebp" at 
//	             the beginning and "popl %ebp" before returning (ret) or using NEXT.
//	             Also, calls from vm.s to external words (C/C++) that reference
//		     GlobalIp should update GlobalIp prior to the call:	"movl %ebp, GlobalIp"
//	04-01-2006 fixed bug in L_mslash
//      04-11-2006 added stubs for L_dult, L_utsslashmod, L_stsslashrem, L_udmstar, L_cputest,
//                   L_dsstar, L_dmax, and L_dmin 
//      04-19-2006 added C_syscall
//	04-21-2006 recoded relational operators L_eq, L_neq, L_ult, L_ugt so that they don't 
//                   require conditional branching! Added macro EQ
//      04-22-2006 finished recoding dyadic single relational operators L_lt, L_gt, L_le, L_ge,
//                   L_zeroeq, L_zerone, L_zerolt, L_zerogt. Added macros REL_DYADIC and REL_ZERO
//                   and removed macro EQ.
//      04-23-2006 added macro LOGIC_DYADIC and used it for recoding _AND, _OR, and _XOR macros.
//      04-24-2006 recoded L_dequal, L_dlt, and L_dzeroeq to remove conditional branching.
//      04-27-2006 made macro STOD and fixed L_mplus by using STOD and DPLUS macros; coded L_dult,
//		     recoded L_dlt to fix problem caused by using DMINUS; simplified _NOT macro
//		     and used it in L_fne; recoded L_feq and L_flt to remove conditional branches.
//	04-29-2006 coded L_dmin and L_dmax; added following macros: DLT FDUP FSWAP FDROP 
//	04-30-2006 added macros _ABS and TNEG, and finished coding L_dsstar
//      05-02-2006 added macro FREL_DYADIC and used it to recode dyadic fp relational operators.
//	05-30-2006 rename L_cmove to L_move, and coded correct L_cmove
//      10-17-2006 recoded L_fzeroeq and L_fzerolt to remove conditional branching. Added
//                   L_fzerogt.
//      11-06-2006 fixed L_utmslash (now passes divtest.4th) and added division by zero
//                   and division overflow checks in this word ;  added proper checks and handling
//                   for division overflow and div by zero in L_umslashmod ;  added division by
//                   zero checks in L_fmslashmod and L_smslashrem
//      02-19-2007 division by zero error checks added for L_mod and L_slashmod
//      07-01-2007 implemented UTS/MOD and STS/REM
//	10-05-2007 replaced memcpy with memmove in L_move	
//      03-15-2008 implemented L_calladdr
//      09-19-2009 implemented L_blank
//      09-26-2009 implemented L_within, L_starslash
//      09-28-2009 temporarily make L_dnegate and L_dplus words which ret instead of NEXT
//      11-22-2009 commented out code for L_tick, which has been replaced by CPP_tick
//      11-26-2009 recoded the macro FDROP; implemented L_dtwostar and L_dtwodiv
//      05-01-2010 recoded +LOOP for Forth-94 compatibility
//      12-22-2010 implemented L_f2dup
//  2011-02-05  km  changed _PUSH and _POP to PUSH_R and POP_R , and
//                    L_push and L_pop to L_push_r and L_pop_r, and
//                    L_twopush and L_twopop to L_twopush_r and L_twopop_r
.include "vm-common.s"

	.comm GlobalTp,4,4
	.comm GlobalRtp,4,4
	.comm BottomOfTypeStack,4,4
	.comm BottomOfReturnTypeStack,4,4


.macro SWAP
	pushl %ebp
        LDSP
        addl $WSIZE, %ebx
	movl %ebx, %ebp
	movl (%ebp), %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	movl %ecx, (%ebx)
	movl %edx, (%ebp)
        movl GlobalTp, %ebx
        incl %ebx
	movl %ebx, %ebp
	movb (%ebp), %al
	incl %ebx
	movb (%ebx), %cl
	movb %al, (%ebx)
	movb %cl, (%ebp)
	popl %ebp
        xor %eax, %eax
.endm

.macro OVER
        LDSP
        movl 2*WSIZE(%ebx), %eax
        movl %eax, (%ebx)
	DEC_DSP
        movl GlobalTp, %ebx
        movb 2(%ebx), %al
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
.endm

.macro FDUP
	LDSP
	movl %ebx, %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ecx, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	DEC_DSP
	movl GlobalTp, %ebx
	incl %ebx
	movw (%ebx), %ax
	subl $2, %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
.endm
	
.macro FDROP
	movl $2*WSIZE, %eax
	addl %eax, GlobalSp
        addl $2, GlobalTp
	xor %eax, %eax
.endm

.macro FSWAP
	LDSP
	movl $WSIZE, %ecx
	addl %ecx, %ebx
	movl (%ebx), %edx
	addl %ecx, %ebx
	movl (%ebx), %eax
	addl %ecx, %ebx
	xchgl %edx, (%ebx)
	addl %ecx, %ebx
	xchgl %eax, (%ebx)
	subl %ecx, %ebx
	subl %ecx, %ebx
	movl %eax, (%ebx)
	subl %ecx, %ebx
	movl %edx, (%ebx)
	movl GlobalTp, %ebx
	incl %ebx
	movw (%ebx), %ax
	addl $2, %ebx
	xchgw %ax, (%ebx)
	subl $2, %ebx
	movw %ax, (%ebx)
	xor %eax, %eax
.endm
		
.macro FOVER
	LDSP
	movl %ebx, %ecx
	addl $3*WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ecx, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	DEC_DSP
	movl GlobalTp, %ebx
	movl %ebx, %ecx
	addl $3, %ebx
	movw (%ebx), %ax
	movl %ecx, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax	
.endm
	
.macro PUSH_R
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx	
        movl %ebx, GlobalSp
	movl (%ebx), %ecx
	movl GlobalRp, %ebx
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalRp
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	movb (%ebx), %al
	movl GlobalRtp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
        xor %eax, %eax
.endm

	
.macro POP_R
	movl $WSIZE, %eax
	movl GlobalRp, %ebx
	addl %eax, %ebx
	movl %ebx, GlobalRp
	movl (%ebx), %ecx
	LDSP
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movl %ebx, GlobalRtp
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
.endm
	
// Dyadic Logic operators 
	
.macro LOGIC_DYADIC op
	LDSP
	movl $WSIZE, %ecx
	addl %ecx, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	addl %ecx, %ebx
	\op (%ebx), %eax
	movl %eax, (%ebx)
	movl GlobalTp, %eax
	incl %eax
	movl %eax, GlobalTp
	movb $OP_IVAL, 1(%eax)	
	xorl %eax, %eax 
.endm
	
.macro _AND
	LOGIC_DYADIC andl
.endm
	
.macro _OR
	LOGIC_DYADIC orl
.endm
	
.macro _XOR
	LOGIC_DYADIC xorl
.endm
	

// use algorithm from DNW's vm-osxppc.s
.macro _ABS	
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	xorl %eax, %eax
	cmpl %eax, %ecx
	setl %al
	negl %eax
	movl %eax, %edx
	xorl %ecx, %edx
	subl %eax, %edx
	movl %edx, (%ebx)
	xorl %eax, %eax
.endm
	
// Dyadic relational operators (single length numbers) 
	
.macro REL_DYADIC setx
	LDSP
	movl $WSIZE, %ecx
	addl %ecx, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	addl %ecx, %ebx
	cmpl %eax, (%ebx)
	movl $0, %eax
	\setx %al
	negl %eax
	movl %eax, (%ebx)
	movl GlobalTp, %eax
	incl %eax
	movl %eax, GlobalTp
	movb $OP_IVAL, 1(%eax)
	xorl %eax, %eax
.endm

// Relational operators for zero (single length numbers)
	
.macro REL_ZERO setx
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	cmpl $0, %eax
	movl $0, %eax
	\setx %al
	negl %eax
	movl %eax, (%ebx)
	movl GlobalTp, %eax
	movb $OP_IVAL, 1(%eax)
	xorl %eax, %eax
.endm

.macro FREL_DYADIC logic arg set
	LDSP
	movl $WSIZE, %ecx
	addl %ecx, %ebx
	fldl (%ebx)
	addl %ecx, %ebx
	addl %ecx, %ebx
	movl %ebx, GlobalSp
	fcompl (%ebx)
	fnstsw %ax
	andb $65, %ah
	\logic \arg, %ah
	movl $0, %eax
	\set %al
	negl %eax
	addl %ecx, %ebx
	movl %eax, (%ebx)
	movl GlobalTp, %eax
	addl $3, %eax
	movl %eax, GlobalTp
	movb $OP_IVAL, 1(%eax)
	xorl %eax, %eax
.endm
				
.macro STOD
	LDSP
	movl $WSIZE, %ecx
	movl WSIZE(%ebx), %eax
	cdq
	movl %edx, (%ebx)
	subl %ecx, %ebx
	movl %ebx, GlobalSp
	STD_IVAL
	xorl %eax, %eax
.endm


	# b = (d1.hi < d2.hi) OR ((d1.hi = d2.hi) AND (d1.lo u< d2.lo))
.macro DLT
	LDSP
	movl $WSIZE, %ecx
	xorl %edx, %edx
	addl %ecx, %ebx
	movl (%ebx), %eax
	cmpl %eax, 2*WSIZE(%ebx)
	sete %dl
	setl %dh
	addl %ecx, %ebx
	movl (%ebx), %eax
	addl %ecx, %ebx
	movl %ebx, GlobalSp
	addl %ecx, %ebx
	cmpl %eax, (%ebx)
	setb %al
	andb %al, %dl
	orb  %dh, %dl
	xorl %eax, %eax
	movb %dl, %al
	negl %eax
	movl %eax, (%ebx)
	movl GlobalTp, %eax
	addl $4, %eax
	movb $OP_IVAL, (%eax)
	decl %eax
	movl %eax, GlobalTp	
	xorl %eax, %eax
.endm
			
.macro DNEGATE
	LDSP
	addl $WSIZE, %ebx
	movl %ebx, %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	notl %eax
	clc
	addl $1, %eax
	movl %eax, (%ebx)
	movl %ecx, %ebx
	movl (%ebx), %eax
	notl %eax
	adcl $0, %eax
	movl %eax, (%ebx)
	xor %eax, %eax	
.endm
	
.macro DPLUS
	LDSP
	addl $2*WSIZE, %ebx
	movl (%ebx), %eax
	clc
	addl 2*WSIZE(%ebx), %eax
	movl %eax, 2*WSIZE(%ebx)
	movl WSIZE(%ebx), %eax
	adcl -WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
.endm
	
.macro DMINUS
	LDSP
	addl $2*WSIZE, %ebx
	movl 2*WSIZE(%ebx), %eax
	clc
	subl (%ebx), %eax
	movl %eax, 2*WSIZE(%ebx)
	movl WSIZE(%ebx), %eax
	sbbl -WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
.endm
	
.macro STARSLASH
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        movl WSIZE(%ebx), %eax
        imull (%ebx)
	idivl -WSIZE(%ebx)
	movl %eax, WSIZE(%ebx)
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
.endm
	

.macro TNEG
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl (%ebx), %eax
	notl %eax
	notl %ecx
	notl %edx
	clc
	addl $1, %eax
	adcl $0, %ecx
	adcl $0, %edx
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, %ebx
	movl %ecx, (%ebx)
	subl %eax, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax	
.endm

// VIRTUAL MACHINE 
						
.global vm
	.type	vm,@function
vm:	
        pushl %ebp
        pushl %ebx
        pushl %ecx
        pushl %edx
	pushl GlobalIp
	pushl vmEntryRp
        movl %esp, %ebp
        movl 28(%ebp), %ebp     # load the Forth instruction pointer
        movl %ebp, GlobalIp
	movl GlobalRp, %eax
	movl %eax, vmEntryRp
	xor %eax, %eax
next:
        movb (%ebp), %al         # get the opcode
	movl JumpTable(,%eax,4), %ebx	# machine code address of word
	xor %eax, %eax          # clear error code
	call *%ebx		# call the word
	movl GlobalIp, %ebp
	incl %ebp		 # increment the Forth instruction ptr
	movl %ebp, GlobalIp
	cmpb $0, %al		 # check for error
	jz next        
exitloop:
        cmpl $OP_RET, %eax         # return from vm?
        jnz vmexit
        xor %eax, %eax            # clear the error
vmexit:
	pop vmEntryRp
	pop GlobalIp
	pop %edx
        pop %ecx
        pop %ebx
        pop %ebp
        ret

L_ret:
	movl vmEntryRp, %eax		# Return Stack Ptr on entry to VM
	movl GlobalRp, %ecx
	cmpl %eax, %ecx
	jl ret1
        movl $OP_RET, %eax             # exhausted the return stack so exit vm
        ret
ret1:
	addl $WSIZE, %ecx
        movl %ecx, GlobalRp
        incl GlobalRtp	
	movl GlobalRtp, %ebx
	movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jz ret2
        movl $E_RET_STK_CORRUPT, %eax   # indicate return stack corrupted
        jmp retexit
ret2:   movl (%ecx), %eax
	movl %eax, GlobalIp		# reset the instruction ptr
        xor %eax, %eax
retexit:
        ret

L_tobody:
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx	# code address
	incl %ecx		# the data address is offset by one
	movl (%ecx), %ecx
	movl %ecx, (%ebx)
	ret
#
# For precision delays, use MS instead of USLEEP
# Use USLEEP when task can be put to sleep and reawakened by OS
#
L_usleep:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	call usleep
	popl %eax
	xor %eax, %eax
	ret
L_ms:
	LDSP
	movl WSIZE(%ebx), %eax
	imull $1000, %eax
	movl %eax, WSIZE(%ebx)
	call C_usec
	ret

L_fill:
	SWAP
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz fill2
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	jmp fillexit
fill2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memset
	addl $12, %esp
	xor %eax, %eax
fillexit:	
	ret
L_erase:
	LDSP
	movl $0, (%ebx)
	DEC_DSP
	DEC_DTSP
	call L_fill
	ret
L_blank:
	LDSP
	movl $32, (%ebx)
	DEC_DSP
	DEC_DTSP
	call L_fill
	ret	
L_move:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	SWAP
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz move2
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
move2:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz move3
	popl %ebx
	popl %ebx
	movl $E_NOT_ADDR, %eax
	ret
move3:	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	pushl %ebx
	call memmove
	addl $12, %esp
	xor %eax, %eax				
	ret
L_cmove:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %ecx		# nbytes in ecx
	cmpl $0, %ecx
	jnz  cmove1
	addl $2*WSIZE, %ebx
	movl %ebx, GlobalSp
	addl $3, GlobalTp
	xorl %eax, %eax
	NEXT		
cmove1:	incl GlobalTp
	addl %eax, %ebx
	movl (%ebx), %edx		# dest addr in edx
	movl %ebx, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmove2
	movl $E_NOT_ADDR, %eax
	ret
cmove2:	movl GlobalSp, %ebx
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax		# src addr in eax
	movl %ebx, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %bl
	cmpb $OP_ADDR, %bl
	jz cmove3
	movl $E_NOT_ADDR, %eax
	ret
cmove3:	movl %eax, %ebx			# src addr in ebx
cmoveloop: movb (%ebx), %al
	movb %al, (%edx)
	incl %ebx
	incl %edx
	loop cmoveloop
	xor %eax, %eax				
	NEXT				
L_cmovefrom:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx
	movl (%ebx), %ecx	# load count register
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom2
	movl $E_NOT_ADDR, %eax						
	ret
cmovefrom2:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	decl %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr in %edx
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jz cmovefrom3
	movl $E_NOT_ADDR, %eax
	ret
cmovefrom3:
	movl GlobalSp, %ebx
	movl (%ebx), %ebx
	movl %ecx, %eax
	cmpl $0, %eax
	jnz cmovefrom4
	ret
cmovefrom4:	
	decl %eax
	addl %eax, %ebx		# src addr in %ebx
cmovefromloop:	
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop cmovefromloop	
	xor %eax, %eax
	ret

L_slashstring:
	LDSP
	DROP
	movl (%ebx), %ecx
	addl $WSIZE, %ebx
	subl %ecx, (%ebx)
	addl $WSIZE, %ebx
	addl %ecx, (%ebx)
	NEXT

L_call:	
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	xor %eax, %eax
	incl GlobalTp
	movl GlobalSp, %ebx
	jmpl *(%ebx)
	ret
L_push_r:
	PUSH_R
        NEXT
L_pop_r:
	POP_R
	NEXT

L_twopush_r:
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ebx, GlobalSp
	movl GlobalRp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalRp
	movl GlobalTp, %ebx
	incl %ebx
	movw (%ebx), %ax
	incl %ebx
	movl %ebx, GlobalTp
	movl GlobalRtp, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
	xor %eax, %eax
	NEXT

L_twopop_r:
	movl GlobalRp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ebx, GlobalRp
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movw (%ebx), %ax
	incl %ebx
	movl %ebx, GlobalRtp
	movl GlobalTp, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax				
	NEXT
L_puship:
        movl %ebp, %eax
        movl GlobalRp, %ebx
        mov %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, %al
        movb %al, (%ebx)
	decl GlobalRtp
        xor %eax, %eax
        NEXT
L_execute:	
        movl %ebp, %ecx
        movl GlobalRp, %ebx
        movl %ecx, (%ebx)
	movl $WSIZE, %eax 
        subl %eax, %ebx 
	movl %ebx, GlobalRp
        movl GlobalRtp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
        movl %ebx, GlobalRtp
	movl GlobalSp, %ebx
        addl %eax, %ebx
	movl %ebx, GlobalSp
        movl (%ebx), %eax
	decl %eax
	movl %eax, %ebp
	incl GlobalTp
        xor %eax, %eax
        NEXT
L_definition:
        movl %ebp, %ebx
	movl $WSIZE, %eax
	incl %ebx
	movl (%ebx), %ecx # address to execute
	addl $3, %ebx
	movl %ebx, %edx
	movl GlobalRp, %ebx
	movl %edx, (%ebx)
	subl %eax, %ebx
	movl %ebx, GlobalRp
	movl GlobalRtp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalRtp
	decl %ecx
	movl %ecx, %ebp
        xor %eax, %eax	
	NEXT
L_rfetch:
        movl GlobalRp, %ebx
        addl $WSIZE, %ebx
        movl (%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        movl GlobalTp, %ebx
        movb %al, (%ebx)
        decl GlobalTp
        xor %eax, %eax
	NEXT
L_tworfetch:
	movl GlobalRp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalRtp, %ebx
	incl %ebx
	movw (%ebx), %ax
	incl %ebx
	movl GlobalTp, %ebx
	decl %ebx
	movw %ax, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax				
	NEXT
L_rpfetch:
	LDSP
	movl GlobalRp, %eax
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT
L_spfetch:
	movl GlobalSp, %eax
	movl %eax, %ebx
	addl $WSIZE, %eax
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_ADDR, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax 
	NEXT
L_i:
        movl GlobalRtp, %ebx
        movb 3(%ebx), %al
        movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
        movl GlobalRp, %ebx
        movl 3*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, %ebx 
	movl %ebx, GlobalSp
        xor %eax, %eax
        NEXT
L_j:
        movl GlobalRtp, %ebx
        movb 6(%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
        movl GlobalRp, %ebx
        movl 6*WSIZE(%ebx), %eax
        movl GlobalSp, %ebx
        movl %eax, (%ebx)
	movl $WSIZE, %eax
        subl %eax, %ebx
	movl %ebx, GlobalSp
        xor %eax, %eax
        NEXT

L_loop:
        movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
        movl GlobalRp, %ebx	
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
        movl (%ebx), %eax
        incl %eax
	cmpl %ecx, %eax	
        jz L_unloop
loop1:	
        movl %eax, (%ebx)	# set loop counter to next value
	movl %edx, %ebp		# set instruction ptr to start of loop
        xorl %eax, %eax
        NEXT

L_unloop:
	UNLOOP
	xorl %eax, %eax
        NEXT

loopbad:
        movl $E_RET_STK_CORRUPT, %eax
        ret

L_plusloop:
	pushl %ebp
	movl GlobalRtp, %ebx
        incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz loopbad
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
	addl %eax, %ebx
	movl (%ebx), %ebp	# get loop increment 
	movl %ebx, GlobalSp
	incl GlobalTp		
        movl GlobalRp, %ebx
	addl %eax, %ebx		# get ip and save in edx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %ecx	# get terminal count in ecx
	addl %eax, %ebx
	movl (%ebx), %eax	# get current loop index
	addl %ebp, %eax         # new loop index
	cmpl $0, %ebp           
	jl plusloop1            # loop inc < 0?

     # positive loop increment
	cmpl %ecx, %eax
	jl plusloop2            # is new loop index < ecx?
	addl %ebp, %ecx
	cmpl %ecx, %eax
	jge plusloop2            # is new index >= ecx + inc?
	popl %ebp
	xorl %eax, %eax
	UNLOOP
	NEXT

plusloop1:       # negative loop increment
	decl %ecx
	cmpl %ecx, %eax
	jg plusloop2           # is new loop index > ecx-1?
	addl %ebp, %ecx
	cmpl %ecx, %eax
	jle plusloop2           # is new index <= ecx + inc - 1?
	popl %ebp
	xorl %eax, %eax
	UNLOOP
	NEXT

plusloop2:
	popl %ebp
	movl %eax, (%ebx)
	movl %edx, %ebp
	xorl %eax, %eax
	NEXT

L_count:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz counterror
	movb $OP_IVAL, (%ebx)
	decl GlobalTp
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	xor %eax, %eax
	movb (%ebx), %al
	movl GlobalSp, %ebx
	incl WSIZE(%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
	ret
counterror:
	movl $E_NOT_ADDR, %eax
	ret

L_ival:
	LDSP
        movl %ebp, %ecx
        incl %ecx
        movl (%ecx), %eax
	addl $WSIZE-1, %ecx
	movl %ecx, %ebp
	movl %eax, (%ebx)
	DEC_DSP
	STD_IVAL
	xorl %eax, %eax
	NEXT

L_addr:
	LDSP
	movl %ebp, %ecx
	incl %ecx
	movl (%ecx), %eax
	addl $WSIZE-1, %ecx
	movl %ecx, %ebp
	movl %eax, (%ebx)
	DEC_DSP
	STD_ADDR
	xorl %eax, %eax
	NEXT

L_fval:
        movl %ebp, %ebx
        incl %ebx
        movl GlobalSp, %ecx
        subl $WSIZE, %ecx
        movl (%ebx), %eax
	movl %eax, (%ecx)
	movl WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ecx)
	subl $WSIZE, %ecx
	movl %ecx, GlobalSp
	addl $2*WSIZE-1, %ebx
	movl %ebx, %ebp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT

L_and:
	_AND
	NEXT
L_or:
	_OR
	NEXT
L_not:
	LDSP
	_NOT
	NEXT
L_xor:
	_XOR
	NEXT

L_eq:
	REL_DYADIC sete
	NEXT

L_ne:
	REL_DYADIC setne
	NEXT

L_ult:
	REL_DYADIC setb
	NEXT

L_ugt:
	REL_DYADIC seta 
	NEXT

L_lt:
	REL_DYADIC setl
	NEXT

L_gt:
	REL_DYADIC setg
	NEXT

L_le:
	REL_DYADIC setle
	NEXT

L_ge:
	REL_DYADIC setge
	NEXT

L_zeroeq:
	REL_ZERO setz
	NEXT

L_zerone:
	REL_ZERO setnz
	NEXT

L_zerolt:
	REL_ZERO setl
	NEXT

L_zerogt:
	REL_ZERO setg
	NEXT

L_within:
        LDSP                       # stack: a b c 
        movl 2*WSIZE(%ebx), %ecx   # ecx = b
	movl WSIZE(%ebx), %eax     # eax = c
	subl %ecx, %eax            # eax = c - b
	addl $WSIZE, %ebx     
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %edx     # edx = a
        subl %ecx, %edx            # edx = a - b
	cmpl %eax, %edx
	movl $0, %eax
	setb %al
	negl %eax
	movl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	addl $3, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	xorl %eax, %eax
        NEXT

L_deq:
	movl GlobalTp, %ebx
	addl $4, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	subl %edx, %eax
	addl $WSIZE, %ebx
	movl (%ebx), %edx
	subl %ecx, %edx
	orl %edx, %eax
	cmpl $0, %eax
	movl $0, %eax
	setz %al
	negl %eax
	movl %eax, (%ebx)
	xorl %eax, %eax
	NEXT

L_dzeroeq:
	movl GlobalTp, %ebx
	addl $2, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	orl (%ebx), %eax
	cmpl $0, %eax
	movl $0, %eax
	setz %al
	negl %eax
	movl %eax, (%ebx)
	xorl %eax, %eax
	NEXT

L_dlt:	
	DLT
	NEXT

L_dult:	# b = (d1.hi u< d2.hi) OR ((d1.hi = d2.hi) AND (d1.lo u< d2.lo)) 
	LDSP
	movl $WSIZE, %ecx
	xorl %edx, %edx
	addl %ecx, %ebx
	movl (%ebx), %eax
	cmpl %eax, 2*WSIZE(%ebx)
	sete %dl
	setb %dh
	addl %ecx, %ebx
	movl (%ebx), %eax
	addl %ecx, %ebx
	movl %ebx, GlobalSp
	addl %ecx, %ebx
	cmpl %eax, (%ebx)
	setb %al
	andb %al, %dl
	orb  %dh, %dl
	xorl %eax, %eax
	movb %dl, %al
	negl %eax
	movl %eax, (%ebx)
	movl GlobalTp, %eax
	addl $4, %eax
	movb $OP_IVAL, (%eax)
	decl %eax
	movl %eax, GlobalTp
	xorl %eax, %eax
	NEXT
	
L_querydup:
	LDSP
	movl WSIZE(%ebx), %eax
	cmpl $0, %eax
	je L_querydupexit
	movl %eax, (%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	decl GlobalTp
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	xor %eax, %eax
L_querydupexit:
	ret


L_swap:
	SWAP
        NEXT

L_over:
	OVER
        NEXT

L_rot:
	pushl %ebp
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, %ebp
	addl %eax, %ebx
	addl %eax, %ebx
	movl (%ebx), %ecx
	movl (%ebp), %edx
	movl %ecx, (%ebp)
	addl %eax, %ebp
	movl (%ebp), %ecx
	movl %edx, (%ebp)
	movl %ecx, (%ebx)
        movl GlobalTp, %ebx
        incl %ebx
	movl %ebx, %ebp
	movw (%ebx), %cx
	addl $2, %ebx
	movb (%ebx), %al
	movb %al, (%ebp)
	incl %ebp
	movw %cx, (%ebp)
	xor %eax, %eax
	popl %ebp
	NEXT

L_minusrot:
	LDSP
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	movl -2*WSIZE(%ebx), %eax
	movl %eax, WSIZE(%ebx)
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	movb %al, (%ebx)
	incl %ebx
	movw 1(%ebx), %ax
	movw %ax, (%ebx)
	movb -1(%ebx), %al
	movb %al, 2(%ebx)
	xor %eax, %eax
	NEXT

L_nip:
        SWAP
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
	xor %eax, %eax
        NEXT

L_tuck:
        SWAP
        OVER
        NEXT

L_pick:
	LDSP
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax
	movl GlobalSp, %ebx
	movl %eax, (%ebx)
	movl WSIZE(%ebx), %eax
	addl $2, %eax
	movl GlobalTp, %ebx
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, 1(%ebx)
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
	NEXT

L_roll:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalSp, %ebx 
	movl (%ebx), %eax
	incl %eax
	pushl %eax
	pushl %eax
	pushl %eax
	pushl %ebx
	imul $WSIZE, %eax
	addl %eax, %ebx		# addr of item to roll
	movl (%ebx), %eax
	popl %ebx
	movl %eax, (%ebx)
	popl %eax		# number of cells to copy
	movl %eax, %ecx
	imul $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, %edx		# dest addr
	subl $WSIZE, %ebx	# src addr
rollloop:
	movl (%ebx), %eax
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	movl %eax, (%ebx)
	sub $WSIZE, %ebx
	xchgl %ebx, %edx
	loop rollloop

	popl %eax		# now we have to roll the typestack
	mov GlobalTp, %ebx	
	addl %eax, %ebx
	movb (%ebx), %al
	movl GlobalTp, %ebx
	movb %al, (%ebx)
	popl %eax
	movl %eax, %ecx
	addl %eax, %ebx
	movl %ebx, %edx
	decl %ebx
rolltloop:
	movb (%ebx), %al
	decl %ebx
	xchgl %ebx, %edx
	movb %al, (%ebx)
	decl %ebx
	xchgl %ebx, %edx
	loop rolltloop
	xor %eax, %eax
	ret

L_depth:
	LDSP
	movl BottomOfStack, %eax
	subl %ebx, %eax
	movl $WSIZE, (%ebx)
	movl $0, %edx
	idivl (%ebx)
	movl %eax, (%ebx)
	movl $WSIZE, %eax
	subl %eax, GlobalSp
	STD_IVAL
	xorl %eax, %eax
        ret

L_2drop:
	FDROP
        NEXT

L_f2drop:
	FDROP
	FDROP
	NEXT

L_f2dup:
        FOVER
	FOVER
	NEXT

L_2dup:
	FDUP
        NEXT

L_2swap:
	FSWAP	
        NEXT

L_2over:
	FOVER
        NEXT

L_2rot:
	LDSP
	addl $WSIZE, %ebx
	movl %ebx, %ecx
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	xchgl %edx, (%ebx)
	addl $WSIZE, %ebx
	xchgl %eax, (%ebx)
	addl $WSIZE, %ebx
	xchgl %edx, (%ebx)
	addl $WSIZE, %ebx
	xchgl %eax, (%ebx)
	movl %ecx, %ebx
	movl %edx, (%ebx)
	addl $WSIZE, %ebx
	movl %eax, (%ebx)
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, %ecx
	movw (%ebx), %ax
	addl $2, %ebx
	xchgw %ax, (%ebx)
	addl $2, %ebx
	xchgw %ax, (%ebx)
	movl %ecx, %ebx
	movw %ax, (%ebx)
	xor %eax, %eax
        NEXT

L_question:
	call L_fetch
	cmpl $0, %eax
	jnz questionexit
	call CPP_dot
questionexit:	
	ret	

L_fetch:
	movl GlobalSp, %edx
	movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_IVAL, (%ebx)	
	addl $WSIZE, %edx
        movl (%edx), %ebx	
        movl (%ebx), %eax
	movl %eax, (%edx)
	xor %eax, %eax
	ret
fetcherror:
        movl $E_NOT_ADDR, %eax
        ret

L_store:
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movl $WSIZE, %eax
	movl GlobalSp, %ebx
        addl %eax, %ebx
        movl (%ebx), %ecx	# address to store to in ecx
	addl %eax, %ebx
	movl (%ebx), %edx	# value to store in edx
	movl %ebx, GlobalSp
	movl %edx, (%ecx)
	movl GlobalTp, %ebx
	addl $2, %ebx
	movl %ebx, GlobalTp
	xor %eax, %eax
	NEXT

L_afetch:
	movl GlobalSp, %edx
	movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_ADDR, (%ebx)
	addl $WSIZE, %edx
	movl (%edx), %ebx
	movl (%ebx), %eax
	movl %eax, (%edx)
	xor %eax, %eax
	NEXT

L_cfetch:
	movl GlobalTp, %ebx
	incl %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, (%ebx)
	xor %eax, %eax
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	movb (%ecx), %al
	movl %eax, (%ebx)
	xor %eax, %eax
        NEXT

L_cstore:
	movl GlobalTp, %edx
	incl %edx
	movb (%edx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx	# address to store
	addl $WSIZE, %ebx
	movl (%ebx), %eax	# value to store
	movb %al, (%ecx)
	movl %ebx, GlobalSp
	incl %edx
	movl %edx, GlobalTp
	xor %eax, %eax
	NEXT	

L_wfetch:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movb $OP_IVAL, 1(%ebx)
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %ebx
	movw (%ebx), %ax
	cwde
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	xor %eax, %eax
        NEXT

L_wstore:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	movl GlobalTp, %ebx
	movb (%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	pushl %eax
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	pop %ebx
	movw %ax, (%ebx)
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	incl GlobalTp
	xor %eax, %eax
        NEXT

L_sffetch:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movb $OP_IVAL, (%ebx)
        decl %ebx
        movb $OP_IVAL, (%ebx)
        decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %ebx
        flds (%ebx)
	movl $WSIZE, %eax
        subl %eax, GlobalSp
        movl GlobalSp, %ebx
        fstpl (%ebx)
        subl %eax, GlobalSp
	xor %eax, %eax
        NEXT

L_sfstore:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalTp, %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
        movl GlobalSp, %ebx
        addl $WSIZE, %ebx
        fldl (%ebx)              # load the f number into NDP
        subl $WSIZE, %ebx
        movl (%ebx), %ebx          # load the dest address
        fstps (%ebx)             # store as single precision float
	movl $WSIZE, %eax
	sall $1, %eax
        addl %eax, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
        NEXT

L_dffetch:	
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp 
	movl GlobalSp, %ebx
	movl %ebx, %edx
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	movl (%ecx), %eax
	movl %eax, (%edx)
	addl $WSIZE, %ecx
	movl (%ecx), %eax
	movl %eax, (%ebx)
	subl $WSIZE, %edx
	movl %edx, GlobalSp
	xor %eax, %eax
	NEXT

L_dfstore:
        movl GlobalTp, %ebx
	incl %ebx
        movb (%ebx), %al
        cmpb $OP_ADDR, %al
        jnz fetcherror
	addl $2, %ebx
	movl %ebx, GlobalTp
	movl GlobalSp, %ebx
	movl $WSIZE, %edx
	addl %edx, %ebx
	movl %ebx, %eax
	movl (%ebx), %ebx  # address to store
	addl %edx, %eax
	movl (%eax), %ecx
	movl %ecx, (%ebx)
	addl %edx, %eax
	addl %edx, %ebx
	movl (%eax), %ecx
	movl %ecx, (%ebx)
	movl %eax, GlobalSp
	xor %eax, %eax
	NEXT

L_abs:
	_ABS
        NEXT

L_max:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jl max1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp maxexit
max1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
maxexit:
	incl GlobalTp
	xor %eax, %eax
        ret

L_min:
	movl $WSIZE, %eax
	addl %eax, GlobalSp
	movl GlobalSp, %ebx
	movl (%ebx), %eax
	movl WSIZE(%ebx), %ebx
	cmpl %eax, %ebx
	jg min1
	movl %ebx, %eax
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
	jmp minexit
min1:
	movl GlobalSp, %ebx
	movl %eax, WSIZE(%ebx)
minexit:
	incl GlobalTp
	xor %eax, %eax
        ret

L_dmax:
	FOVER
	FOVER
	DLT
	incl GlobalTp
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	movl %ebx, GlobalSp
	cmpl $0, %eax
	jne dmin1
	FDROP
	xorl %eax, %eax
	NEXT

L_dmin:
	FOVER
	FOVER
	DLT
	incl GlobalTp
	movl $WSIZE, %ecx
	movl GlobalSp, %ebx
	addl %ecx, %ebx
	movl (%ebx), %eax
	movl %ebx, GlobalSp
	cmpl $0, %eax
	je dmin1
	FDROP
	xorl %eax, %eax
	NEXT
dmin1:
	FSWAP
	FDROP
	xorl %eax, %eax
	NEXT

#  L_dtwostar and L_dtwodiv are valid for two's-complement systems
L_dtwostar:
        LDSP
        addl $WSIZE, %ebx
        movl WSIZE(%ebx), %eax
        movl %eax, %ecx
        sall $1, %eax
        movl %eax, WSIZE(%ebx)
        shrl $31, %ecx
        movl (%ebx), %eax
        sall $1, %eax
        orl  %ecx, %eax
        movl %eax, (%ebx)
        xorl %eax, %eax
        NEXT

L_dtwodiv:
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %eax
        movl %eax, %ecx
        sarl $1, %eax
        movl %eax, (%ebx)
        shll $31, %ecx
        movl WSIZE(%ebx), %eax
        shrl $1, %eax
        orl %ecx, %eax
        movl %eax, WSIZE(%ebx)
        xorl %eax, %eax
        NEXT

L_add:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax
	addl %eax, WSIZE(%ebx)
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	movw (%ebx), %ax
	andb %ah, %al		# and the two types to preserve addr type
	incl %ebx
	movb %al, (%ebx)
        xor %eax, %eax
        NEXT

L_div:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        movl (%ebx), %eax
        cmpl $0, %eax
        jnz div1
        movl $E_DIV_ZERO, %eax
        jmp divexit
div1:	
	addl $WSIZE, %ebx
        movl (%ebx), %eax
	cdq
        idivl -WSIZE(%ebx)
        movl %eax, (%ebx)
	xor %eax, %eax
divexit:
        ret

L_mod:
	call L_div
	cmpl $0, %eax
	jnz  divexit
	movl %edx, (%ebx)
	NEXT

L_slashmod:
	call L_div
	cmpl $0, %eax
	jnz  divexit
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	SWAP
	NEXT

L_starslash:
	STARSLASH	
	NEXT

L_starslashmod:
	STARSLASH
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	SWAP
	ret

L_plusstore:
	movl GlobalTp, %ebx
	movb 1(%ebx), %al
	cmpb $OP_ADDR, %al
	jnz fetcherror
	movl GlobalSp, %ebx
	push %ebx
	push %ebx
	push %ebx
	movl WSIZE(%ebx), %ebx
	movl (%ebx), %eax
	pop %ebx
	movl 2*WSIZE(%ebx), %ebx
	addl %ebx, %eax
	pop %ebx
	movl WSIZE(%ebx), %ebx
	movl %eax, (%ebx)
	pop %ebx
	movl $WSIZE, %eax
	sall $1, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	incl GlobalTp
	xor %eax, %eax
	NEXT

L_dabs:
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	movl %ecx, %eax
	cmpl $0, %eax
	jl dabs_go
	xor %eax, %eax
	ret
dabs_go:
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	clc
	subl $1, %eax
	notl %eax
	movl %eax, (%ebx)
	movl %ecx, %eax
	sbbl $0, %eax
	notl %eax
	movl %eax, -WSIZE(%ebx)
	xor %eax, %eax
	ret

L_dnegate:
	DNEGATE
#	NEXT	
	ret

L_dplus:
	DPLUS
#	NEXT
        ret

L_dminus:
	DMINUS
	ret

L_umstar:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	mull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax				
	ret

L_dsstar:
	# multiply signed double and signed to give triple length product
	LDSP
	movl $WSIZE, %ecx
	addl %ecx, %ebx
	movl (%ebx), %edx
	cmpl $0, %edx
	setl %al
	addl %ecx, %ebx
	movl (%ebx), %edx
	cmpl $0, %edx
	setl %ah
	xorb %ah, %al      # sign of result
	andl $1, %eax
	pushl %eax
	_ABS
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	call L_dabs
	movl GlobalSp, %ebx
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_udmstar
	popl %eax
	cmpl $0, %eax
	jne dsstar1
	NEXT
dsstar1:
	TNEG
	NEXT

L_umslashmod:
# Divide unsigned double length by unsigned single length to
# give unsigned single quotient and remainder. A "Divide overflow"
# error results if the quotient doesn't fit into a single word.
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	STSP
	movl (%ebx), %ecx
	cmpl $0, %ecx
	jnz  umslashmod0
	movl $E_DIV_ZERO, %eax
	ret
umslashmod0:	
	addl %eax, %ebx
	movl $0, %edx
	movl (%ebx), %eax
	divl %ecx
	cmpl $0, %eax
	jne umslashmod_ovflow
	movl (%ebx), %edx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	divl %ecx
	jmp umslashmod_exit
umslashmod_ovflow:
	movl $E_DIV_OVERFLOW, %eax
	ret
umslashmod_exit:	
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax		
	ret

L_mstar:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %ecx
	addl %eax, %ebx
	movl %ecx, %eax
	imull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %edx, (%ebx)
	xor %eax, %eax		
	ret

L_mplus:
	STOD
	DPLUS
	NEXT

L_mslash:
        LDSP
	movl $WSIZE, %eax
        incl GlobalTp
	addl %eax, %ebx
        movl (%ebx), %ecx
	incl GlobalTp
	addl %eax, %ebx
	STSP
        cmpl $0, %ecx
        je mslash1
        movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
        idivl %ecx
        movl %eax, (%ebx)
	xor %eax, %eax		
	ret
mslash1:			
        movl $E_DIV_ZERO, %eax
        ret

L_udmstar:
# Multiply unsigned double and unsigned single to give 
# the triple length product.
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	mull %ecx
	movl %edx, -WSIZE(%ebx)
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl %ecx, %eax
	mull (%ebx)
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl (%ebx), %eax
	subl $WSIZE, %ebx
	clc
	addl %edx, %eax
	movl %eax, WSIZE(%ebx)
	movl (%ebx), %eax
	adcl $0, %eax
	movl %eax, (%ebx)
	xor %eax, %eax 		
	ret

L_utsslashmod:
# Divide unsigned triple length by unsigned single length to
# give an unsigned triple quotient and single remainder.
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx		# divisor in ecx
	cmpl $0, %ecx
	jnz  utsslashmod0
	movl $E_DIV_ZERO, %eax	
	ret
utsslashmod0:	
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut3
	movl $0, %edx
	divl %ecx			# ut3/u
	call utmslash1
	movl GlobalSp, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	addl $WSIZE, %ebx	
	movl -17*WSIZE(%ebx), %eax       # r7
	movl %eax, (%ebx)
	subl $3*WSIZE, %ebx
	movl -5*WSIZE(%ebx), %eax        # q3
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	decl GlobalTp
	xorl %eax, %eax	
	ret

L_tabs:
# Triple length absolute value (needed by L_stsslashrem, STS/REM)
        LDSP
        addl $WSIZE, %ebx
        movl (%ebx), %ecx
        movl %ecx, %eax
        cmpl $0, %eax
        jl tabs1
        xor %eax, %eax
        ret
tabs1:
        addl $2*WSIZE, %ebx
        movl (%ebx), %eax
        clc
        subl $1, %eax
        notl %eax
        movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl (%ebx), %eax
	sbbl $0, %eax
	notl %eax
	movl %eax, (%ebx)
        movl %ecx, %eax
        sbbl $0, %eax
        notl %eax
        movl %eax, -WSIZE(%ebx)
        xor %eax, %eax
        ret

L_stsslashrem:
# Divide signed triple length by signed single length to give a
# signed triple quotient and single remainder, according to the
# rule for symmetric division.
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx		# divisor in ecx
	cmpl $0, %ecx
	jnz  stsslashrem0
	movl $E_DIV_ZERO, %eax	
stsslashrem0:
	movl WSIZE(%ebx), %eax		# t3
	pushl %eax
	cmpl $0, %eax
	movl $0, %eax
	setl %al
	negl %eax
	movl %eax, %edx
	cmpl $0, %ecx
	movl $0, %eax
	setl %al
	negl %eax
	xorl %eax, %edx			# sign of quotient
	pushl %edx
	movl %ebx, GlobalSp
	call L_tabs
	subl $WSIZE, GlobalSp
	_ABS
	call L_utsslashmod
	popl %edx
	cmpl $0, %edx
	jz stsslashrem1
	TNEG
stsslashrem1:	
	popl %eax
	cmpl $0, %eax
	jz stsslashrem2
	movl GlobalSp, %ebx
	addl $4*WSIZE, %ebx
	negl (%ebx)	
stsslashrem2:
	xorl %eax, %eax
	ret

L_utmslash:
# Divide unsigned triple length by unsigned single to give 
# unsigned double quotient. A "Divide Overflow" error results
# if the quotient doesn't fit into a double word.
	LDSP
	addl $WSIZE, %ebx
	movl (%ebx), %ecx		# divisor in ecx
	cmpl $0, %ecx
	jnz  utmslash0
	movl $E_DIV_ZERO, %eax
	ret
utmslash0:	
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut3
	movl $0, %edx
	divl %ecx			# ut3/u
	cmpl $0, %eax
	jz   utmslash1
	movl $E_DIV_OVERFLOW, %eax
	ret
utmslash1:	
	pushl %ebx			# keep local stack ptr
	movl GlobalSp, %ebx
	movl %eax, -4*WSIZE(%ebx)	# q3
	movl %edx, -5*WSIZE(%ebx)	# r3
	popl %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut2
	movl $0, %edx
	divl %ecx			# ut2/u
	pushl %ebx
	movl GlobalSp, %ebx
	movl %eax, -2*WSIZE(%ebx)	# q2
	movl %edx, -3*WSIZE(%ebx)	# r2
	popl %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax		# ut1
	movl $0, %edx
	divl %ecx			# ut1/u
	pushl %ebx
	movl GlobalSp, %ebx
	movl %eax, (%ebx)		# q1
	movl %edx, -WSIZE(%ebx)		# r1
	movl -5*WSIZE(%ebx), %edx	# r3 << 32
	movl $0, %eax
	divl %ecx			# (r3 << 32)/u
	movl %eax, -6*WSIZE(%ebx)	# q4
	movl %edx, -7*WSIZE(%ebx)	# r4
	movl -3*WSIZE(%ebx), %edx	# r2 << 32
	movl $0, %eax
	divl %ecx			# (r2 << 32)/u
	movl %eax, -8*WSIZE(%ebx)	# q5
	movl %edx, -9*WSIZE(%ebx)	# r5
	movl -7*WSIZE(%ebx), %edx	# r4 << 32
	movl $0, %eax
	divl %ecx			# (r4 << 32)/u
	movl %eax, -10*WSIZE(%ebx)	# q6
	movl %edx, -11*WSIZE(%ebx)	# r6
	movl $0, %edx
	movl -WSIZE(%ebx), %eax		# r1
	addl -9*WSIZE(%ebx), %eax	# r1 + r5
	jnc   utmslash2
	incl %edx
utmslash2:			
	addl -11*WSIZE(%ebx), %eax	# r1 + r5 + r6
	jnc  utmslash3
	incl %edx
utmslash3:
	divl %ecx
	movl %eax, -12*WSIZE(%ebx)	# q7
	movl %edx, -13*WSIZE(%ebx)	# r7
	movl $0, %edx
	addl -10*WSIZE(%ebx), %eax	# q7 + q6
	jnc  utmslash4
	incl %edx
utmslash4:	
	addl -8*WSIZE(%ebx), %eax	# q7 + q6 + q5
	jnc  utmslash5
	incl %edx
utmslash5:	
	addl (%ebx), %eax		# q7 + q6 + q5 + q1
	jnc utmslash6
	incl %edx
utmslash6:
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	pushl %ebx
	movl GlobalSp, %ebx
	movl -2*WSIZE(%ebx), %eax	# q2
	addl -6*WSIZE(%ebx), %eax	# q2 + q4
	addl %edx, %eax
	popl %ebx
	movl %eax, (%ebx)
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	addl $2, GlobalTp
	xorl %eax, %eax
	ret

L_mstarslash:
	LDSP
	addl $WSIZE, %ebx
	addl $WSIZE, %ebx
	movl (%ebx), %eax
	addl $WSIZE, %ebx
	xorl (%ebx), %eax
	shrl $31, %eax
	pushl %eax	# keep sign of result -- negative is nonzero
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	_ABS
	movl GlobalSp, %ebx
	addl $WSIZE, %ebx
	movl %ebx, GlobalSp
	incl GlobalTp
	call L_dabs
	movl GlobalSp, %ebx
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_udmstar
	movl GlobalSp, %ebx
	subl $WSIZE, %ebx
	movl %ebx, GlobalSp
	decl GlobalTp
	call L_utmslash	
	popl %eax
	cmpl $0, %eax
	jnz mstarslash_neg
	xor %eax, %eax
	ret
mstarslash_neg:
	DNEGATE
	xor %eax, %eax
	ret
		
L_fmslashmod:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	STSP
	movl (%ebx), %ecx
	cmpl $0, %ecx
	jnz  fmslashmod0
	movl $E_DIV_ZERO, %eax
	ret
fmslashmod0:
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	idivl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	cmpl $0, %ecx
	jg fmslashmod2
	cmpl $0, %edx
	jg fmslashmod3
	xor %eax, %eax
	ret
fmslashmod2:		
	cmpl $0, %edx
	jge fmslashmodexit
fmslashmod3:	
	decl %eax		# floor the result
	movl %eax, (%ebx)
	addl $WSIZE, %ebx
	addl %ecx, (%ebx)
fmslashmodexit:
	xor %eax, %eax
	ret

L_smslashrem:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	STSP
	movl (%ebx), %ecx
	cmpl $0, %ecx
	jnz smslashrem0
	movl $E_DIV_ZERO, %eax
	ret
smslashrem0:	
	addl %eax, %ebx
	movl (%ebx), %edx
	addl %eax, %ebx
	movl (%ebx), %eax
	idivl %ecx
	movl %edx, (%ebx)
	subl $WSIZE, %ebx
	movl %eax, (%ebx)
	incl GlobalTp
	xor %eax, %eax		
	ret
L_stod:
	STOD
	NEXT
L_stof:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        incl GlobalTp
        movl GlobalSp, %ebx
        fildl (%ebx)
        movl GlobalTp, %ebx
        movb $OP_IVAL, (%ebx)
        decl %ebx
        movb $OP_IVAL, (%ebx)
	decl GlobalTp
	decl GlobalTp
        movl GlobalSp, %ebx
	movl $WSIZE, %eax
        subl %eax, %ebx
        fstpl (%ebx)
	sall $1, %eax
        subl %eax, GlobalSp
	xor %eax, %eax
        NEXT

L_dtof:
        LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %eax
	xchgl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
        fildq (%ebx)
        fstpl (%ebx)
	xor %eax, %eax	
	NEXT	

L_froundtos:
	movl $WSIZE, %eax
        addl %eax, GlobalSp
        movl GlobalSp, %ebx
        fldl (%ebx)
        addl %eax, %ebx
        fistpl (%ebx)
        incl GlobalTp
        movl GlobalTp, %ebx
        incl %ebx
        movb $OP_IVAL, (%ebx)
	xor %eax, %eax
        NEXT

L_ftrunctos:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	fldl (%ebx)
	fnstcw (%ebx)
	movl (%ebx), %ecx	# save NDP control word		
	movl %ecx, %edx	
	movb $12, %dh
	movl %edx, (%ebx)
	fldcw (%ebx)
	addl %eax, %ebx	
	fistpl (%ebx)
	subl %eax, %ebx
	movl %ecx, (%ebx)
	fldcw (%ebx)		# restore NDP control word
	incl GlobalTp
	movl GlobalTp, %ebx
	incl %ebx
	movb $OP_IVAL, (%ebx)
	xor %eax, %eax	
	NEXT	
L_ftod:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	fldl (%ebx)
	subl %eax, %ebx
	fnstcw (%ebx)
	movl (%ebx), %ecx	# save NDP control word	
	movl %ecx, %edx
	movb $12, %dh		
	movl %edx, (%ebx)
	fldcw (%ebx)
	addl %eax, %ebx	
	fistpq (%ebx)
	subl %eax, %ebx
	movl %ecx, (%ebx)
	fldcw (%ebx)		# restore NDP control word
	addl %eax, %ebx 
	movl (%ebx), %eax
	xchgl WSIZE(%ebx), %eax
	movl %eax, (%ebx)
	xor %eax, %eax	
	NEXT
	
L_degtorad:
        LDSP
	fldl FCONST_180
        addl $WSIZE, %ebx
        fldl (%ebx)
        fdivp %st, %st(1)
        fldpi
        fmulp %st, %st(1)
        fstpl (%ebx)
        NEXT

L_radtodeg:
        LDSP
        addl $WSIZE, %ebx
        fldl (%ebx)
        fldpi
	fxch
        fdivp %st, %st(1)
        fldl FCONST_180
        fmulp %st, %st(1)
        fstpl (%ebx)
        NEXT
L_fne:
	FREL_DYADIC xorb $64 setnz
	NEXT
L_feq:
	FREL_DYADIC andb $64 setnz
	NEXT
L_flt:
	FREL_DYADIC andb $65 setz
	NEXT
L_fgt:
	FREL_DYADIC andb $1 setnz
	NEXT	
L_fle:
	FREL_DYADIC xorb $1 setnz
	NEXT
L_fge:
	FREL_DYADIC andb $65 setnz
	NEXT
L_fzeroeq:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl (%ebx), %ecx
	movl %ebx, GlobalSp
	addl %eax, %ebx
	movl (%ebx), %eax
	shll $1, %eax
	orl %ecx, %eax
	movl $0, %eax
	setz %al
	negl %eax
	movl %eax, (%ebx)
frelzero:
	movl GlobalTp, %ebx
	incl %ebx
	movl %ebx, GlobalTp
	incl %ebx
	movb $OP_IVAL, (%ebx)
	xorl %eax, %eax
	NEXT
L_fzerolt:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	fldl (%ebx)
	addl %eax, %ebx
	fldz
	fcompp	
	fnstsw %ax
	andb $69, %ah
	movl $0, %eax
	setz %al
	negl %eax
	movl %eax, (%ebx)
	jmp frelzero
L_fzerogt:
	LDSP
	movl $WSIZE, %eax
	addl %eax, %ebx
	movl %ebx, GlobalSp
	fldz
	fldl (%ebx)
	addl %eax, %ebx
	fucompp	
	fnstsw %ax
	sahf 
	movl $0, %eax
	seta %al
	negl %eax
	movl %eax, (%ebx)
	jmp frelzero				

L_floor:
	LDSP
	addl $WSIZE, %ebx
	movl WSIZE(%ebx), %eax
	pushl %eax
	movl (%ebx), %eax
	pushl %eax
	call floor
	addl $8, %esp
	fstpl (%ebx)
	xor %eax, %eax		
	NEXT

L_fround:
	LDSP
	addl $WSIZE, %ebx
	fldl (%ebx)
	frndint
	fstpl (%ebx)
	NEXT

L_ftrunc:
	LDSP
	addl $WSIZE, %ebx
	fldl (%ebx)
	fnstcw NDPcw            # save NDP control word
        movl NDPcw, %ecx
	movb $12, %ch
	movl %ecx, (%ebx)
	fldcw (%ebx)
	frndint
        fldcw NDPcw             # restore NDP control word
	fstpl (%ebx)
	NEXT

L_fatan2:
	LDSP
	addl $2*WSIZE, %ebx
	fldl WSIZE(%ebx)
	fldl -WSIZE(%ebx)
	fpatan
	fstpl WSIZE(%ebx)
	STSP
	incl GlobalTp
	incl GlobalTp
	NEXT

L_fsincos:
	LDSP
	fldl WSIZE(%ebx)
	fsincos
	fstpl -WSIZE(%ebx)
	fstpl WSIZE(%ebx)
	subl $2*WSIZE, %ebx
	movl %ebx, GlobalSp
	movl GlobalTp, %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movb $OP_IVAL, (%ebx)
	decl %ebx
	movl %ebx, GlobalTp	
	NEXT

