\ naacb.4th
\
\ Neutral Atom-Atom Collision Broadening of Atomic Spectral Lines -- Lorentz Model
\
\ Copyright (c) 2006--2009 Krishna Myneni
\ Provided under the GNU General Public License
\
\ Measurement of spectral line width provides a way to determine the
\ temperature and pressure of a remote source, such as the surface of a star.
\ This program computes the spectrum of light emitted by an atom undergoing
\ collisions in a vapor of inert buffer gas. A simple model of random phase
\ interruptions of a wave train, as proposed by H. A. Lorentz, is used.
\ By increasing the frequency of collisions, i.e. the density of the buffer gas,
\ the width of the spectral line may be seen to increase.
\
\ Here we assume a sodium (Na) atom emitting light on its 3p-3s transition
\ (3s ^2S_{1/2}--3p ^2P_{3/2}). The experimentally determined relation
\ between perturber density and line width, with Krypton buffer gas,
\ on this transition is [1][2]
\
\   Width (MHz) = 9.80 MHz + 7.98e-16(MHz/cm^-3)*Density(cm^-3) 
\
\ The width is the full width at half maximum (FWHM). In
\ the low pressure limit of no collisions, the spectral line has a
\ Lorentzian shape, with a width determined only by the decay time
\ of the atom's excited state.
\
\
\ NOTES:
\
\  0. Doppler broadening is not included in the calculation of the spectrum
\     in order to demonstrate the collision broadening component of the spectral
\     width. For a realistic spectrum, the Doppler profile must be computed
\     for the given temperature and mass of the atom, and this profile convolved
\     with the collision-broadened spectrum.
\ 
\  1. The emitted field is modeled as
\
\         E(t) = exp{-t/(2*tau)}*exp{i*phi(t, tau_c)}
\
\     where tau is the lifetime of the upper state, and phi(t, tau_c) is the phase
\     which jumps to random values in the interval [-pi, pi] at random time intervals.
\     The time intervals between phase jumps are sampled from a Poisson distribution,
\     with a mean interval of tau_c, the average time between collisions.
\     Between phase jumps, phi(t) is constant, meaning that we have shifted the carrier
\     (optical) frequency to zero.
\
\  2. Using an appropriate collision cross section, the Lorentz model provides a
\     good estimate of the experimentally observed line width for a given density,
\     over a wide range of densities. However, use of random phase interruptions
\     of a wave train to describe the effect of atomic collisions on a radiating atom
\     ignores the continuous nature of the collision process. The phase
\     change depends in a smooth way on the force between the atoms during
\     the collision, and leads to interesting phenomena such as a small shift in the
\     peak frequency, and secondary (satellite) peaks in the spectrum. Refined
\     theories which take into account the interatomic forces and treat the
\     collision quantum mechanically are described in Ref [1].
\
\  3. The numerical calculation works well from zero density (Pressure = 0) to
\     an upper density of about 1e19 cm^-3. For densities greater than this
\     value, the mean collision time, tau_c, becomes so short that the array size
\     does not span enough time to follow the decaying field over one lifetime.
\
\     
\ References:
\ 
\ [1] N. Allard and J. Kielkopf, Reviews of Modern Physics, vol 54, p. 1103 (1982).
\
\ [2] J.E. Sansonetti, W.C. Martin, and S.L. Young, Handbook of Basic Atomic
\     Spectroscopic Data(version 1.1.2):
\     http://physics.nist.gov/PhysRefData/Handbook/Tables/sodiumtable3.htm
\
\ Revisions:
\    2009-01-15  km;  broadening parameter was off by factor of two due to
\                                misinterpretation of "half-width"; broadening cross-section 
\                                has been correspondingly changed;
\                                Revised for latest versions of FSL modules.

include ans-words
include asm-x86
include fft-x86
include fsl/fsl-util
include fsl/complex
include fsl/horner
include fsl/extras/noise

2 FLOATS constant complex
pi 2e f* fconstant 2pi
1.3806503e-23  fconstant kB           \ Boltzmann constant in J/K
133.28e        fconstant Pa/Torr      \ conversion from Torr to Pascals (N/m^2)
1.66053873e-27 fconstant kg/amu       \ conversion from atomic mass unit to kg

\ Parameters describing the atom and its environment

fvariable m_A           \ mass of atom A (perturbed atom) in atomic mass units
fvariable m_B 	\ mass of atom B (buffer gas) in a.m.u.
fvariable tau           \ lifetime of upper state in seconds
fvariable Temp          \ vapor temperature in K
fvariable Pressure      \ vapor pressure in Torr
fvariable sigma         \ collision cross section in square Angstroms

\ Set defaults
22.989767e    m_A  f!  \ Sodium, Na; see Ref. [2]
84e                 m_B  f!  \ Krypton, Kr;
16.2e-9       tau  f!  \ Lifetime of Na upper level (3p ^2P_{3/2}); see Ref. [2]
300e          Temp f!  \ room temperature
1e            Pressure f!  \ Torr
430e          sigma f! \ square Angstroms  


\ Derived Parameters (computed from above parameters)
fvariable n           \ perturber atom density in cm^-3
fvariable vel         \ atom velocity in cm/s
fvariable tau_c       \ average time between collisions in seconds


ms@ idum !            \ seed the random number generator

: random-phase ( -- f | return a random phase value between [-pi, pi])
    ran0 2pi f* pi f- ;

: random-interval ( -- f | return a random time interval with a Poisson distribution )
    1e ran0 F- FLN tau_c F@ F* FNEGATE ;

4096  constant NP
NP 2/ constant NP2
NP complex array E{
NP float   array spec{

fvariable phi0
fvariable dt
variable nsteps
fvariable min_freq	\ minumum of the spectrum frequency
fvariable freq_step	\ frequency step for the spectrum


: calc ( -- | calculate the lineshape with the current parameters )
    \ compute atom density in cm^-3 using n = p/kT
    Pressure f@ Pa/Torr f* kB Temp f@ f* f/ 1e-6 f* n f!
    cr ." # Perturber Density (cm^-3) = " n f@ f.  
    \ compute velocity of atom in cm/s using v = sqrt(3kT/m)
    3e kB f* Temp f@ f* m_a f@ kg/amu f* f/ fsqrt 100e f* vel f!  
    \ compute collision rate in s^-1 using 1/tau_c = n*v*sigma
    n f@ vel f@ f* sigma f@ 1e-16 f* f*
    1e fswap f/ tau_c f!
    cr ." # Mean collision time (sec) = " tau_c f@ f.
    tau_c f@ tau f@ fmin 10e f/ dt f! \ step size for computing the emitted field

    \ Compute the spectrum, averaged from many sources

    NP 0 DO 0e spec{ I } f! LOOP
    
    200 0 DO
	
	tau_c f@ tau f@ 1000e f* f> IF NP ELSE 0 THEN nsteps !
	
	NP 0 DO
	    I nsteps @ = IF
		random-phase phi0 F!     \ new phase
		random-interval dt f@ f/ fround>s
		1 max nsteps +!          \ index of next phase change
		\ nsteps ?
	    THEN
	    phi0 f@ FSINCOS fswap                 \ exp{i*phi0}
	    I s>f dt f@ f* 2e f/ tau f@ f/ fnegate fexp \ exp{-t/tau}
	    z*f E{ I } z!
	LOOP
	NP E{ fft
	\ Accumulate the averaged spectrum
	NP 0 DO  E{ I } z@ |z|^2 spec{ I } dup >r F@ F+ r> F!  LOOP 
    LOOP

    \ Reorder the spectrum for increasing frequency

    NP2 0 DO spec{ I } f@ spec{ NP2 I + } dup >r f@ fswap r> f! spec{ I } f! LOOP  

    \ Compute frequency scale
    
    dt f@ 2e f* -1e fswap f/     min_freq  f!
    dt f@ NP s>f f* 1e fswap f/  freq_step f!

    \ Normalize the spectrum to unit area

    0e NP 0 DO spec{ I } f@ f+ LOOP freq_step f@ f*
    NP 0 DO spec{ I } f@ fover f/ spec{ I } f! LOOP fdrop

    \ Output the spectrum
    
    NP 0 DO
	cr freq_step f@ I s>f f* min_freq f@ f+ 1e6 f/ f.  \ frequency in MHz
	spec{ I } f@ f.
    LOOP

;
