/*
 * Compile with:
 *
 * gcc -fPIC -ffunction-sections -fdata-sections -m64 -Os -pipe -g3 frida-gumjs-example.c -o frida-gumjs-example -L. -lfrida-gumjs -ldl -lm -latomic -lrt -lresolv -lpthread -pthread -fuse-ld=gold -Wl,--icf=all,--gc-sections,-z,noexecstack -static-libgcc
 *
 * Visit https://frida.re to learn more about Frida.
 */

#include "frida-gumjs.h"

#include <fcntl.h>
#include <string.h>
#include <unistd.h>

static void on_message (GumScript * script, const gchar * message, GBytes * data, gpointer user_data);

int
main (int argc,
      char * argv[])
{
  GumScriptBackend * backend;
  GCancellable * cancellable = NULL;
  GError * error = NULL;
  GumScript * script;
  GMainContext * context;

  gum_init_embedded ();

  backend = gum_script_backend_obtain_qjs ();

  script = gum_script_backend_create_sync (backend, "example",
      "Interceptor.attach(Module.getExportByName(null, 'open'), {\n"
      "  onEnter(args) {\n"
      "    console.log(`[*] open(\"${args[0].readUtf8String()}\")`);\n"
      "  }\n"
      "});\n"
      "Interceptor.attach(Module.getExportByName(null, 'close'), {\n"
      "  onEnter(args) {\n"
      "    console.log(`[*] close(${args[0].toInt32()})`);\n"
      "  }\n"
      "});",
      cancellable, &error);
  g_assert (error == NULL);

  gum_script_set_message_handler (script, on_message, NULL, NULL);

  gum_script_load_sync (script, cancellable);

  close (open ("/etc/hosts", O_RDONLY));
  close (open ("/etc/fstab", O_RDONLY));

  context = g_main_context_get_thread_default ();
  while (g_main_context_pending (context))
    g_main_context_iteration (context, FALSE);

  gum_script_unload_sync (script, cancellable);

  g_object_unref (script);

  gum_deinit_embedded ();

  return 0;
}

static void
on_message (GumScript * script,
            const gchar * message,
            GBytes * data,
            gpointer user_data)
{
  JsonParser * parser;
  JsonObject * root;
  const gchar * type;

  parser = json_parser_new ();
  json_parser_load_from_data (parser, message, -1, NULL);
  root = json_node_get_object (json_parser_get_root (parser));

  type = json_object_get_string_member (root, "type");
  if (strcmp (type, "log") == 0)
  {
    const gchar * log_message;

    log_message = json_object_get_string_member (root, "payload");
    g_print ("%s\n", log_message);
  }
  else
  {
    g_print ("on_message: %s\n", message);
  }

  g_object_unref (parser);
}
