/*
 * Copyright (C) 1997 Red Hat Software, Inc.
 *	Cristian Gafton <gafton@redhat.com>
 *
 * Modify:  2013/01/12  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/01/09  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/05/14  lme@FreeBSD.org
 *
 * Modify:  2013/03/19  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2013/01/06  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2006/24/10  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2004/08/09  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2003/11/19  Tommy Scheunemann <net@arrishq.org>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

#include <stdio.h>
#include <errno.h>
#include <unistd.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <dirent.h>
#include <stdlib.h>
#include <string.h>
#include <ctype.h>
#include <pwd.h>
#include <strings.h>

#include "wmconfig.h"

/*
 * We read all the packages here, do overrides, then build
 * root_group from what's left
 */
struct package *packages = NULL;
int num_packages = 0;

static int max_packages = 0;

int init_packages(void)
{
    num_packages = 0;
    max_packages = MAX_PACKAGES;

    packages = (struct package *)calloc(MAX_PACKAGES, sizeof(struct package));
    if (packages == (struct package *)NULL) {
	fprintf(stderr, gettext ("Out of memory in %d@%s!\n"), __LINE__, __FUNCTION__);
	exit(-2);
    }
    return 0;
}

/* seek for a package */
static struct package *find_package(const char *package)
{
    int	i;

    for (i = 0; i < num_packages; i++) {
	if (strcmp(packages[i].__package, package) == 0) {
	    return packages + i;
	}
    }
    return NULL;
}

/* get a pointer to a new package */
static struct package *new_package(void)
{
    if (num_packages == max_packages) {
	/* we need more room for packages ... */
	packages = (struct package *) realloc (packages, (max_packages+MAX_PACKAGES)*sizeof(struct package));
	if ((packages == NULL) || (packages == (struct package *) NULL)) {
	    free(packages);
	    fprintf(stderr, gettext ("%d@%s: out of memory!\n"), __LINE__, __FUNCTION__);
	    return NULL;
	}
	max_packages += MAX_PACKAGES;
    }
    memset(packages + num_packages, 0, sizeof(struct package));
    return packages + num_packages++;
}

/*
 * Updates one package with the tag and value
 */
int update_package(const char *package, const char *tag, const char *value)
{
    struct package *pkg = NULL;
    char **adr = NULL;

    /* safety measures comes first */
    if (!package || !tag || !value) {
	return -1;
    }

    pkg = find_package(package);
    if (pkg == (struct package *)NULL) {
	pkg = new_package();
	if (pkg == (struct package *)NULL) {
	    /* arghh... horror */
	    fprintf(stderr, gettext ("%d@%s: out of memory!\n"), __LINE__, __FUNCTION__);
	    return -1;
	}
	/* do a little initialization... */
	pkg->__package = x_strdup(package);
    }

    /*
     * This very ugly thing should be replaced ASAP with something
     * more intelligent...
     * --just not yet
     */
    if (strcasecmp(tag, "name") == 0) {
	adr = &pkg->name;
    } else if (strcasecmp(tag, "icon") == 0) {
	adr = &pkg->icon;
    } else if (strcasecmp(tag, "description") == 0) {
	adr = &pkg->description;
    } else if (strcasecmp(tag, "mini-icon") == 0) {
	adr = &pkg->mini_icon;
    } else if (strcasecmp(tag, "exec") == 0) {
	adr = &pkg->exec;
    } else if (strcasecmp(tag, "Xresource") == 0) {
	adr = &pkg->Xresource;
    } else if (strcasecmp(tag, "group") == 0) {
	adr = &pkg->group;
    } else if (strcasecmp(tag, "copy") == 0) {
	adr = &pkg->copy;
    } else if (strcasecmp(tag, "mimetype") == 0) {
	adr = &pkg->mimetype;
    } else if (strcasecmp(tag, "terminal") == 0) {
	adr = &pkg->terminal;
    } else if (strcasecmp(tag, "restart") == 0) {
	adr = &pkg->restart;
    }

    if (!adr) {
	return -1;
    }
    *adr = x_strdup(value);

    return 0;
}


/*
 * for each package that has a copy field set, and some of
 * its structure members are not set, copy from other packages
 */

#define DO_ASSIGN(pkg_a,pkg_b,elem)	\
if (pkg_a->elem == NULL) {	\
    if (pkg_b->elem != NULL)	\
	changed++;		\
    pkg_a->elem = pkg_b->elem;	\
}

void complete_packages(void)
{
    int i;
    int changed = 1;
    int rec_level = 0; /* recursion level */

    while (changed && (rec_level < MAX_RECLEVEL)) {
	changed = 0;
	for (i = 0; i < num_packages; i++) {
	    struct package *pkg, *o_pkg;

	    pkg = packages + i;
	    if (pkg->copy == NULL) {
		continue;
	    }
	    o_pkg = find_package(pkg->copy);
	    if (o_pkg == (struct package *)NULL) {
		fprintf(stderr, gettext ("Warning: could not find package \"%s\", needed by \"%s\"\n"), pkg->copy, pkg->__package);
		continue;
	    }
	    DO_ASSIGN(pkg, o_pkg, name);
	    DO_ASSIGN(pkg, o_pkg, icon);
	    DO_ASSIGN(pkg, o_pkg, mini_icon);
	    DO_ASSIGN(pkg, o_pkg, description);
	    DO_ASSIGN(pkg, o_pkg, Xresource);
	    DO_ASSIGN(pkg, o_pkg, exec);
	    DO_ASSIGN(pkg, o_pkg, group);
	    DO_ASSIGN(pkg, o_pkg, mimetype);
	    DO_ASSIGN(pkg, o_pkg, terminal);
	    DO_ASSIGN(pkg, o_pkg, restart);
	}
	rec_level++;
    }
    return;
}

static char *w_getenv(const char *envname) {
    char *envvalue;
    if (! (envvalue = getenv (envname) )) {
	return 0;
    }
    return strdup(envvalue);
}

/* Mark packages, which are deemed not to exist, to be skipped. */

static int package_exists(struct package *pkg)
{
    char *buf, *cmd, *p, *b, *c;
    char *PATH = w_getenv("PATH");
    int len;

    if (!pkg->exec) {
	return 0; /* doesn't exist */
    }
    cmd = strdup(pkg->exec);

    /* extract the command name */
    for (p = cmd; *p != '\0' && *p != '\t' && *p != ' '; ++p);
    *p = '\0';

    /* seek command as if full path */
    if (access(cmd, X_OK) == 0) {
	free(cmd);
	return 1; /* found */
    }

    /* allocate some scratch memory */
    len = strlen(PATH) + strlen(cmd) + 3; /* always enough */
    buf = (char*)malloc(len);
    if (buf == NULL) /* ran out of memory... report, and return */
    {
	fprintf(stderr, gettext ("Out of memory in %d@%s! (not critical)\n"),
	__LINE__, __FUNCTION__);
	free(cmd);
	return 1; /* assume the package exists */
    }

    /* seek command in PATH */
    p = PATH;
    while (*p != '\0') {

	/* copy a PATH string to buf */
	for (b = buf; (b-buf)<len && *p != ':' && *p != '\0'; ++p, ++b) {
	    *b = *p;
	}

	if (b-buf >= len) {
	    continue; /* this shouldn't ever happen. */
	}

	/* add slash, if necessary */
	if (b != buf && *(b-1) != '/' && cmd[0] != '/') {
	    *(b++) = '/';
	}

	/* catenate command */
	for (c = cmd; (b-buf)<len && *c != '\0'; ++c, ++b)
	*b = *c;
	*b = '\0';

	if (b-buf >= len) {
	    continue; /* this shouldn't ever happen. */
	}

	/* check existence */
	if (access(buf, X_OK) == 0) {
	    free(cmd);
	    free(buf);
	    return 1; /* found! */
	}

	/* next PATH string, if any */
	if (*p != '\0') {
	    ++p;
	}
    }

    /* free allocated stuff */
    free(cmd);
    free(buf);
    free(PATH);
    return 0; /* doesn't exist */
}

void unselect_nonexistent_packages()
{
    int i;
    for (i = 0; i < num_packages; i++) {
	if (!package_exists(&packages[i])) {
	    if (packages[i].group) {
		free(packages[i].group);
	    }
	    packages[i].group = strdup("-");
	}
    }
}
