/*
 * Copyright (C) 1997-2000 Red Hat, Inc.
 *	Cristian Gafton <gafton@redhat.com>
 *
 * Modify:  2015/11/26  Tommy Scheunemann <net@arrishq.net>
 *
 * Modify:  2014/01/07  Tommy Scheunemann <net@arrishq.net>
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */

/*
 * Functions needed for producing a AfterStep directory hierarchy
 * configuration
 *
 * In this case the rootmenu is the top level directory
 */

#include <stdio.h>
#include <stdlib.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <fcntl.h>
#include <unistd.h>
#include <dirent.h>
#include <errno.h>
#include <pwd.h>
#include <strings.h>
#include <string.h>

#include "wmconfig.h"
#include "prototypes.h"
#include "package.h"

#if (defined(__unix__) || defined(unix)) && !defined(USG)
#include <limits.h>
#include <sys/param.h>
#endif

#define OUTPUTDIR "GNUstep/Library/AfterStep/wmconfig"

extern const char *root_menu_name;
extern const char *output_directory;
extern unsigned int flags;

static void create_file(const char *file, const char *action)
{
    int fd;
    FILE *f;

    if (file == NULL || action == NULL) {
	return;
    }

    fd = open(file, O_CREAT | O_EXCL | O_RDWR, 0600);
    f = fdopen (fd, "wx");
    if (f == (FILE *) NULL) {
	fprintf(stderr, gettext("Could not create file %s\n"), file);
	perror("Error code");
	return;
    }
    fprintf(f, "%s\n", action);
    fclose(f);
}

static void create_file_newstyle(const char *file, const struct package *item,const char *use_term)
{
    int fd;
    FILE *f;

    if (file == NULL || item->exec == NULL) {
	return;
    }
	
    if (!item->restart) {
	fd = open(file, O_CREAT | O_EXCL | O_RDWR, 0600);
	f = fdopen (fd, "wx");
	if (f == (FILE *) NULL) {
	    fprintf(stderr, gettext("Could not create file %s\n"), file);
	    perror("Error code");
	    return;
	}
	if (item->terminal) {
	    fprintf(f, "Exec \"%s\" exec %s -e %s\n", item->name, use_term, item->exec);
	} else {
	    fprintf(f, "Exec \"%s\" exec %s\n", item->name, item->exec);
	}
	if ((item->mini_icon != NULL) && (!is_set(NO_MINI_ICONS))) {
	    fprintf(f, "MiniPixmap \"%s\"\n", item->mini_icon);
	}
	fclose(f);
    }
}

static void make_dir(struct group *root, int level, const char *dir_name,const char *use_term)
{
    struct item *item;
    char c_tmp[PATH_MAX];

    if (root == (struct group *)NULL) {
	return;
    }

    item = root->items;
    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
	    struct group *tmp;

	    tmp = (struct group *)item->data;
	    snprintf(c_tmp, sizeof(c_tmp), "%s/%s", dir_name, tmp->name);
	    mkdir(c_tmp, 0700);
	} else if (item->type == ITEM_APP) {
	    struct package *app;

	    app = (struct package *)item->data;
	    if (app->name && app->exec) {
		snprintf(c_tmp, sizeof(c_tmp), "%s/%s", dir_name, app->name);
		if (!is_set(NEWSTYLE_DIRECTORIES)) {
		    create_file(c_tmp, app->exec);
		} else {
		    create_file_newstyle(c_tmp, app, use_term);
		}
	    }
	}
	item++;
    }

    /* second pass, recursive output... */
    item = root->items;
    while (item->type != 0) {
	if (item->type == ITEM_MENU) {
            struct group *tmp;

	    tmp = (struct group *)item->data;
	    snprintf(c_tmp, sizeof(c_tmp), "%s/%s", dir_name, tmp->name);
	    make_dir(item->data, level+1, c_tmp, use_term);
	}
	item++;
    }
}

static void clean_dir(const char *dir_name)
{
    struct dirent **namelist;
    int nr_files;
    int i;

    if (dir_name == NULL)
	return;

    /* read the whole directory in one pass */
    nr_files = scandir(dir_name, &namelist, NULL, alphasort);
    for (i = 0; i < nr_files; i++) {
	if ( (strcmp(namelist[i]->d_name, ".") != 0) &&
	     (strcmp(namelist[i]->d_name, "..") != 0) ) {
	    struct stat st;
	    char tmp[PATH_MAX];
	    int retval;

	    snprintf(tmp, sizeof(tmp), "%s/%s", dir_name, namelist[i]->d_name);
	    if (stat(tmp, &st) != 0) {
		fprintf(stderr, gettext("Could not stat %s\n"), tmp);
		return;
	    }
	    if (S_ISDIR(st.st_mode)) {
		clean_dir(tmp);
		retval = rmdir(tmp);
	    } else {
		retval = unlink(tmp);
	    }
	    if (retval != 0) {
		perror("Could not remove");
		return;
	    }
	}
	free(namelist[i]);
    }
    if (nr_files && namelist) {
	free(namelist);
    }
}

void output_afterstep_dir(struct group *root,const char *use_term)
{
    struct stat st;
    char root_dir[PATH_MAX];
    struct passwd *pw = NULL;

    pw = getpwuid(getuid());
    if (pw == (struct passwd *)NULL) {
	fprintf(stderr, gettext("Could not find out who you are (getpwnam failed)!\n"));
	return;
    }
    if (output_directory == NULL) {
	snprintf(root_dir, sizeof(root_dir), "%s/%s", pw->pw_dir, OUTPUTDIR);
    } else {
	snprintf(root_dir, sizeof(root_dir), "%s", output_directory);
    }

    /* First, check if the root_dir is a dir */
    if (stat(root_dir, &st) == 0) {
	/* that one exists */
	if (!S_ISDIR(st.st_mode)) {
	    fprintf(stderr, gettext("Error: %s exists but it is not a directory\n"), root_dir);
	    return;
	}
	clean_dir(root_dir);	
    } else {
	/* we must make that directory first */
	if (mkdir(root_dir, 0700) != 0) {
	    fprintf(stderr, gettext("Could not create directory '%s'\n"), root_dir);
	    return;
	}
    }
    make_dir(root, 0, root_dir, use_term);
}
