/*
 * Pidgin Toolbar Icons
 * Copyright (C) 2010 Konrad Gräfe
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02111-1301, USA.
 */

#include "toolbar_shrink.h"

#include "iconification.h"
#include "prefs.h"

#include <gtk/gtk.h>
#include <gtkconv.h>
#include <debug.h>
#include <gtkimhtmltoolbar.h>

extern PurplePlugin *plugin;

static gchar **button_text;
static gchar **prefs;

static void force_hidden(GtkWidget *widget, GdkEventVisibility *event, gpointer null) {
	gtk_widget_hide(widget);
}

static GtkWidget *gtk_container_get_child_before_widget(GtkWidget *widget) {
	GtkWidget *parent;
	GList *children, *child;

	if(!GTK_IS_WIDGET(widget)) return NULL;

	parent = gtk_widget_get_parent(widget);
	if(!GTK_IS_CONTAINER(parent)) return NULL;

	children = gtk_container_get_children(GTK_CONTAINER(parent));

	child = g_list_find(children, widget);
	if(!child || !child->prev) return NULL;

	return child->prev->data;
}


static void widget_set_iconification(GtkWidget *widget, gpointer hide) {
	gulong *handler_id;
	int i;
	GtkWidget *bbox, *button, *sep;
	int shrinkmode;

	if(GTK_IS_LABEL(widget)) {
		bbox = gtk_widget_get_parent(widget);
		button = gtk_widget_get_parent(bbox);
		sep = gtk_container_get_child_before_widget(button);

		shrinkmode = SHRINK_MODE_NONE;
		for(i = 0; i < sizeof(ToolbarButtons) && shrinkmode != -1; i++) {
			if(purple_strequal(gtk_label_get_label(GTK_LABEL(widget)), button_text[i])) {
				shrinkmode = purple_prefs_get_int(prefs[i]);
			}
		}

		if(GPOINTER_TO_BOOLEAN(hide)) {
			if(shrinkmode == SHRINK_MODE_SHRINK) {
				/* Label verstecken */
				gtk_widget_hide(widget);

				if(!g_object_get_data(G_OBJECT(widget), DATA_KEY_HANDLER)) {
					handler_id = g_malloc(sizeof(gulong));
					*handler_id = g_signal_connect(GTK_WIDGET(widget), "show", G_CALLBACK(force_hidden), hide);
					g_object_set_data(G_OBJECT(widget), DATA_KEY_HANDLER, handler_id);
				}
			} else if(shrinkmode == SHRINK_MODE_HIDE) {
				/* Button und Separator verstecken */
				gtk_widget_hide(button);
				if(!g_object_get_data(G_OBJECT(button), DATA_KEY_HANDLER)) {
					handler_id = g_malloc(sizeof(gulong));
					*handler_id = g_signal_connect(GTK_WIDGET(button), "show", G_CALLBACK(force_hidden), hide);
					g_object_set_data(G_OBJECT(button), DATA_KEY_HANDLER, handler_id);
				}

				if(sep) {
					gtk_widget_hide(sep);
					if(!g_object_get_data(G_OBJECT(sep), DATA_KEY_HANDLER)) {
						handler_id = g_malloc(sizeof(gulong));
						*handler_id = g_signal_connect(GTK_WIDGET(sep), "show", G_CALLBACK(force_hidden), hide);
						g_object_set_data(G_OBJECT(sep), DATA_KEY_HANDLER, handler_id);
					}
				}

				
			}
		} else {
			/* Label anzeigen */
			handler_id = g_object_get_data(G_OBJECT(widget), DATA_KEY_HANDLER);
			if(handler_id) {
				g_signal_handler_disconnect(G_OBJECT(widget), *handler_id);
				g_object_set_data(G_OBJECT(widget), DATA_KEY_HANDLER, NULL);
				g_free(handler_id);
			}
			gtk_widget_show_all(widget);

			/* Button anzeigen */
			handler_id = g_object_get_data(G_OBJECT(button), DATA_KEY_HANDLER);
			if(handler_id) {
				g_signal_handler_disconnect(G_OBJECT(button), *handler_id);
				g_object_set_data(G_OBJECT(button), DATA_KEY_HANDLER, NULL);
				g_free(handler_id);
			}
			gtk_widget_show_all(button);

			/* Separatoranzeigen */
			if(sep) {
				handler_id = g_object_get_data(G_OBJECT(sep), DATA_KEY_HANDLER);
				if(handler_id) {
					g_signal_handler_disconnect(G_OBJECT(sep), *handler_id);
					g_object_set_data(G_OBJECT(sep), DATA_KEY_HANDLER, NULL);
					g_free(handler_id);
				}
				gtk_widget_show_all(sep);
			}
		}
	} else if(GTK_IS_CONTAINER(widget)) {
		gtk_container_foreach(GTK_CONTAINER(widget), widget_set_iconification, hide);
	}
}


static void toolbar_set_iconification(PurpleConversation *conv, gpointer hide) {
	PidginConversation *gtkconv;
	GtkWidget *toolbar;
	GtkWidget *lean_view_box;

	if(!PIDGIN_IS_PIDGIN_CONVERSATION(conv)) return;

	gtkconv = PIDGIN_CONVERSATION(conv);

	toolbar = gtkconv->toolbar;
	if(!toolbar) return;
	
	lean_view_box = g_object_get_data(G_OBJECT(toolbar), "lean-view");
	if(!lean_view_box) return;

	gtk_container_foreach(GTK_CONTAINER(lean_view_box), widget_set_iconification, hide);
}

static gchar *get_pidgins_translation(const gchar* str) {
	return dgettext("pidgin", str);
}

void iconification_init(void) {
	GList *l;

	button_text = g_malloc(sizeof(gchar *) * sizeof(ToolbarButtons));
	prefs = g_malloc(sizeof(gchar *) * sizeof(ToolbarButtons));

	/* get Pidgins translations */
	button_text[TOOLBAR_BUTTON_FONT] = get_pidgins_translation("_Font");
	button_text[TOOLBAR_BUTTON_INSERT] = get_pidgins_translation("_Insert");
	button_text[TOOLBAR_BUTTON_SMILE] = get_pidgins_translation("_Smile!");
	button_text[TOOLBAR_BUTTON_ATTENTION] = get_pidgins_translation("_Attention!");

	prefs[TOOLBAR_BUTTON_FONT] = PLUGIN_PREFS_PREFIX "/font_button";
	prefs[TOOLBAR_BUTTON_INSERT] = PLUGIN_PREFS_PREFIX "/insert_button";
	prefs[TOOLBAR_BUTTON_SMILE] = PLUGIN_PREFS_PREFIX "/smile_button";
	prefs[TOOLBAR_BUTTON_ATTENTION] = PLUGIN_PREFS_PREFIX "/attention_button";

	purple_signal_connect(purple_conversations_get_handle(), "conversation-created", plugin, PURPLE_CALLBACK(toolbar_set_iconification), GBOOLEAN_TO_POINTER(TRUE));

	for(l = purple_get_conversations(); l != NULL; l = l->next) {
		toolbar_set_iconification(l->data, GBOOLEAN_TO_POINTER(TRUE));
	}
}

void iconification_uninit(void) {
	GList *l;

	for(l = purple_get_conversations(); l != NULL; l = l->next) {
		toolbar_set_iconification(l->data, GBOOLEAN_TO_POINTER(FALSE));
	}

	g_free(button_text);
	g_free(prefs);
}

