/* OpenVAS Manager
 * $Id$
 * Description: Module for OpenVAS Manager: the OMP library.
 *
 * Authors:
 * Matthew Mundell <matthew.mundell@greenbone.net>
 * Timo Pollmeier <timo.pollmeier@greenbone.net>
 *
 * Copyright:
 * Copyright (C) 2009-2013 Greenbone Networks GmbH
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA 02110-1301 USA.
 */

/**
 * @file  omp.c
 * @brief The OpenVAS Manager OMP library.
 *
 * This file defines an OpenVAS Management Protocol (OMP) library, for
 * implementing OpenVAS managers such as the OpenVAS Manager daemon.
 *
 * The library provides \ref process_omp_client_input.
 * This function parses a given string of OMP XML and tracks and manipulates
 * tasks in reaction to the OMP commands in the string.
 */

/**
 * @internal
 * The OMP-"Processor" is always in a state (\ref client_state_t
 * \ref client_state ) and currently looking at the opening of an OMP element
 * (\ref omp_xml_handle_start_element ), at the text of an OMP element
 * (\ref omp_xml_handle_text ) or at the closing of an OMP element
 * (\ref omp_xml_handle_end_element ).
 *
 * The state usually represents the current location of the parser within the
 * XML (OMP) tree.  There has to be one state for every OMP element.
 *
 * State transitions occur in the start and end element handler callbacks.
 *
 * Generally, the strategy is to wait until the closing of an element before
 * doing any action or sending a response.  Also, error cases are to be detected
 * in the end element handler.
 *
 * If data has to be stored, it goes to \ref command_data (_t) , which is a
 * union.
 * More specific incarnations of this union are e.g. \ref create_user_data (_t)
 * , where the data to create a new user is stored (until the end element of
 * that command is reached).
 *
 * For implementing new commands that have to store data (e.g. not
 * "\<help_extended/\>"), \ref command_data has to be freed and NULL'ed in case
 * of errors and the \ref current_state has to be reset.
 * It can then be assumed that it is NULL'ed at the start of every new
 * command element.  To implement a new start element handler, be sure to just
 * copy an existing case and keep its structure.
 *
 * Attributes are easier to implement than elements.
 * E.g.
 * @code
 * <key_value_pair key="k" value="v"/>
 * @endcode
 * is obviously easier to handle than
 * @code
 * <key><attribute name="k"/><value>v</value></key>
 * @endcode
 * .
 * For this reason the GET commands like GET_TASKS all use attributes only.
 *
 * However, for the other commands it is preferred to avoid attributes and use
 * the text of elements
 * instead, like in
 * @code
 * <key_value_pair><key>k</key><value>v</value></key_value_pair>
 * @endcode
 * .
 *
 * If new elements are built of multiple words, separate the words with an
 * underscore.
 */

#include "omp.h"
#include "manage.h"
#include "manage_sql.h"
/** @todo For access to scanner_t scanner. */
#include "otp.h"
#include "tracef.h"

#include <arpa/inet.h>
#include <assert.h>
#include <ctype.h>
#include <errno.h>
#include <fcntl.h>
#include <glib/gstdio.h>
#include <netinet/in.h>
#include <string.h>
#include <stdlib.h>
#include <sys/socket.h>
#include <sys/stat.h>
#include <sys/types.h>
#include <sys/wait.h>

#include <gnutls/x509.h>

#include <openvas/base/nvti.h>
#include <openvas/base/osp.h>
#include <openvas/base/openvas_string.h>
#include <openvas/base/openvas_file.h>
#include <openvas/base/openvas_networking.h>
#include <openvas/base/pwpolicy.h>
#include <openvas/misc/openvas_auth.h>
#include <openvas/misc/openvas_logging.h>
#include <openvas/omp/xml.h>

#ifdef S_SPLINT_S
#include "splint.h"
#endif

#undef G_LOG_DOMAIN
/**
 * @brief GLib log domain.
 */
#define G_LOG_DOMAIN "md    omp"

/**
 * @brief GLib log domain.
 */
#define APPLY_OVERRIDES_DEFAULT 0


/* Static headers. */

/** @todo Exported for manage_sql.c. */
void
buffer_results_xml (GString *, iterator_t *, task_t, int, int, int, int, int,
                    int, int, const char *, iterator_t *, int);

static void
buffer_xml_append_printf (GString*, const char*, ...);


/* Helper functions. */

/**
 * @brief Check whether a string is a UUID.
 *
 * @param[in]  uuid  Potential UUID.
 *
 * @return 1 yes, 0 no.
 */
static int
is_uuid (const char *uuid)
{
  while (*uuid) if (isxdigit (*uuid) || (*uuid == '-')) uuid++; else return 0;
  return 1;
}

/**
 * @brief Return time defined by broken down time strings.
 *
 * If any argument is NULL, use the value from the current time.
 *
 * @param[in]   hour          Hour (0 to 23).
 * @param[in]   minute        Minute (0 to 59).
 * @param[in]   day_of_month  Day of month (1 to 31).
 * @param[in]   month         Month (1 to 12).
 * @param[in]   year          Year.
 * @param[in]   timezone      Timezone.
 *
 * @return Time described by arguments on success, -2 if failed to switch to
 *         timezone, -1 on error.
 */
static time_t
time_from_strings (const char *hour, const char *minute,
                   const char *day_of_month, const char *month,
                   const char *year, const char *timezone)
{
  struct tm given_broken, *now_broken;
  time_t now, ret;
  gchar *tz;

  tz = NULL;
  if (timezone)
    {
      /* Store current TZ. */
      tz = getenv ("TZ") ? g_strdup (getenv ("TZ")) : NULL;

      if (setenv ("TZ", timezone, 1) == -1)
        {
          g_free (tz);
          return -2;
        }
      tzset ();
    }

  time (&now);
  now_broken = localtime (&now);

  given_broken.tm_sec = 0;
  given_broken.tm_min = (minute ? atoi (minute) : now_broken->tm_min);
  given_broken.tm_hour = (hour ? atoi (hour) : now_broken->tm_hour);
  given_broken.tm_mday = (day_of_month
                           ? atoi (day_of_month)
                           : now_broken->tm_mday);
  given_broken.tm_mon = (month ? (atoi (month) - 1) : now_broken->tm_mon);
  given_broken.tm_year = (year ? (atoi (year) - 1900) : now_broken->tm_year);
  given_broken.tm_isdst = now_broken->tm_isdst;

  ret = mktime (&given_broken);

  if (timezone)
    {
      /* Revert to stored TZ. */
      if (tz)
        setenv ("TZ", tz, 1);
      else
        unsetenv ("TZ");
      g_free (tz);
      tzset ();
    }

  return ret;
}

/**
 * @brief Return interval defined by time and unit strings.
 *
 * @param[in]   value   Value.
 * @param[in]   unit    Calendar unit: second, minute, hour, day, week,
 *                      month, year or decade.  "second" if NULL.
 * @param[out]  months  Months return.
 *
 * @return Interval described by arguments on success, -2 if value was NULL,
 *         -1 if value was NULL.
 */
static time_t
interval_from_strings (const char *value, const char *unit, time_t *months)
{
  if (value == NULL)
    return -1;

  if ((unit == NULL) || (strcasecmp (unit, "second") == 0))
    {
      long int val;
      val = strtol (value, NULL, 10);
      if ((val >= INT_MAX) || (val < 0))
        return -3;
      return val;
    }

  if (strcasecmp (unit, "minute") == 0)
    {
      long int val;
      val = strtol (value, NULL, 10);
      if ((val >= (INT_MAX / 60)) || (val < 0))
        return -3;
      return val * 60;
    }

  if (strcasecmp (unit, "hour") == 0)
    {
      long int val;
      val = strtol (value, NULL, 10);
      if ((val >= (INT_MAX / (60 * 60))) || (val < 0))
        return -3;
      return val * 60 * 60;
    }

  if (strcasecmp (unit, "day") == 0)
    {
      long int val;
      val = strtol (value, NULL, 10);
      if ((val >= (INT_MAX / (60 * 60 * 24))) || (val < 0))
        return -3;
      return val * 60 * 60 * 24;
    }

  if (strcasecmp (unit, "week") == 0)
    {
      long int val;
      val = strtol (value, NULL, 10);
      if ((val >= (INT_MAX / (60 * 60 * 24 * 7))) || (val < 0))
        return -3;
      return val * 60 * 60 * 24 * 7;
    }

  if (months)
    {
      if (strcasecmp (unit, "month") == 0)
        {
          *months = atoi (value);
          if ((*months >= INT_MAX) || (*months < 0))
            return -3;
          return 0;
        }

      if (strcasecmp (unit, "year") == 0)
        {
          *months = atoi (value);
          if ((*months >= (INT_MAX / 12)) || (*months < 0))
            return -3;
          *months = *months * 12;
          return 0;
        }

      if (strcasecmp (unit, "decade") == 0)
        {
          *months = atoi (value);
          if ((*months >= (INT_MAX / (12 * 10))) || (*months < 0))
            return -3;
          *months = *months * 12 * 10;
          return 0;
        }
    }

  return -2;
}

/**
 * @brief Find an attribute in a parser callback list of attributes.
 *
 * @param[in]   attribute_names   List of names.
 * @param[in]   attribute_values  List of values.
 * @param[in]   attribute_name    Name of sought attribute.
 * @param[out]  attribute_value   Attribute value return.
 *
 * @return 1 if found, else 0.
 */
int
find_attribute (const gchar **attribute_names,
                const gchar **attribute_values,
                const char *attribute_name,
                const gchar **attribute_value)
{
  while (*attribute_names && *attribute_values)
    if (strcmp (*attribute_names, attribute_name))
      attribute_names++, attribute_values++;
    else
      {
        *attribute_value = *attribute_values;
        return 1;
      }
  return 0;
}

/**
 * @brief Find an attribute in a parser callback list of attributes and append
 * @brief it to a string using openvas_append_string.
 *
 * @param[in]   attribute_names   List of names.
 * @param[in]   attribute_values  List of values.
 * @param[in]   attribute_name    Name of sought attribute.
 * @param[out]  string            String to append attribute value to, if
 *                                found.
 *
 * @return 1 if found and appended, else 0.
 */
int
append_attribute (const gchar **attribute_names,
                  const gchar **attribute_values,
                  const char *attribute_name,
                  gchar **string)
{
  const gchar* attribute;
  if (find_attribute (attribute_names, attribute_values, attribute_name,
                      &attribute))
    {
      openvas_append_string (string, attribute);
      return 1;
    }
  return 0;
}

/**
 * @brief A simple key/value-pair.
 */
typedef struct
{
  gchar *key;                   ///< The key.
  gchar *value;                 ///< The value.
} auth_conf_setting_t;

/**
 * @brief Forms a authentication configuration key/value-pair from xml element.
 *
 * Parameters fit glibs xml_start_element callback data.
 *
 * Expected input is generated from an element like:
 * @code <auth_conf_setting key="key" value="value"> @endcode
 *
 * @param[in]  element_name      Name of current element.
 * @param[in]  attribute_names   Attribute names.
 * @param[in]  attribute_values  Attribute names.
 *
 * @return 0 if input is not generated from a auth_conf_setting xml element as
 *         expected, pointer to freshly allocated auth_conf_setting_t that
 *         owns it key and value string otherwise.
 */
static auth_conf_setting_t *
auth_conf_setting_from_xml (const gchar * element_name,
                            const gchar ** attribute_names,
                            const gchar ** attribute_values)
{
  auth_conf_setting_t *kvp = NULL;

  if (strcasecmp (element_name, "auth_conf_setting") == 0)
    {
      const gchar *key_attr;
      const gchar *val_attr;
      if (find_attribute (attribute_names, attribute_values, "key", &key_attr)
          && find_attribute (attribute_names, attribute_values, "value",
                             &val_attr))
        {
          kvp = g_malloc0 (sizeof (auth_conf_setting_t));
          kvp->key = g_strdup (key_attr);
          kvp->value = g_strdup (val_attr);
        }
    }

  return kvp;
}

/**
 * @brief Init for a GET handler.
 *
 * @param[in]  command       OMP command name.
 * @param[in]  get           GET data.
 * @param[in]  setting_name  Type name for setting.
 * @param[out] first         First result, from filter.
 *
 * @return 0 success, -1 error.
 */
int
init_get (gchar *command, get_data_t * get, const gchar *setting_name,
          int *first)
{
  gchar *filter, *replacement;

  if (user_may (command) == 0)
    return 99;

  /* Get any replacement out of get->filter, before it changes.  Used to add
   * task_id to the filter for GET_REPORTS. */

  if (get->filter_replace && strlen (get->filter_replace) && get->filter)
    replacement = filter_term_value (get->filter, get->filter_replace);
  else
    replacement = NULL;

  /* Switch to the default filter from the setting, if required. */

  if (get->filt_id && strcmp (get->filt_id, "-2") == 0)
    {
      char *user_filter = setting_filter (setting_name);

      if (user_filter && strlen (user_filter))
        {
          get->filt_id = user_filter;
          get->filter = filter_term (user_filter);
        }
      else
        get->filt_id = g_strdup ("0");
    }

  /* Get the actual filter string. */

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      filter = filter_term (get->filt_id);
      if (filter == NULL)
        {
          char *user_filter;

          /* Probably the user deleted the filter, switch to default. */

          g_free (get->filt_id);

          user_filter = setting_filter (setting_name);
          if (user_filter && strlen (user_filter))
            {
              get->filt_id = user_filter;
              get->filter = filter_term (user_filter);
              filter = filter_term (get->filt_id);
            }
          else
            get->filt_id = g_strdup ("0");
        }
    }
  else
    filter = NULL;

  if (replacement)
    {
      const gchar *term;

      /* Replace the term in filter.  Used to add task_id to the filter
       * for GET_REPORTS. */

      term = filter ? filter : get->filter;

      if (term)
        {
          gchar *new_filter, *clean;

          clean = manage_clean_filter_remove (term, get->filter_replace);
          new_filter = g_strdup_printf
                        ("%s=%s %s",
                         get->filter_replace,
                         replacement,
                         clean);
          g_free (clean);
          if (get->filter)
            {
              g_free (get->filter);
              get->filter = new_filter;
            }
          else
            {
              g_free (filter);
              filter = new_filter;
            }
          get->filter_replacement = g_strdup (new_filter);
        }

      g_free (replacement);
    }

  /* Get the value of "first" from the filter string.
   *
   * This is used by get_next when the result set is empty, to determine if
   * the query should be rerun with first 1.
   */
  manage_filter_controls (filter ? filter : get->filter, first, NULL, NULL,
                          NULL);

  g_free (filter);

  return 0;
}


/* Status codes. */

/* HTTP status codes used:
 *
 *     200 OK
 *     201 Created
 *     202 Accepted
 *     400 Bad request
 *     401 Must auth
 *     404 Missing
 */

/**
 * @brief Response code for a syntax error.
 */
#define STATUS_ERROR_SYNTAX            "400"

/**
 * @brief Response code when authorisation is required.
 */
#define STATUS_ERROR_MUST_AUTH         "401"

/**
 * @brief Response code when authorisation is required.
 */
#define STATUS_ERROR_MUST_AUTH_TEXT    "Authenticate first"

/**
 * @brief Response code for forbidden access.
 */
#define STATUS_ERROR_ACCESS            "403"

/**
 * @brief Response code text for forbidden access.
 */
#define STATUS_ERROR_ACCESS_TEXT       "Access to resource forbidden"

/**
 * @brief Response code for a missing resource.
 */
#define STATUS_ERROR_MISSING           "404"

/**
 * @brief Response code text for a missing resource.
 */
#define STATUS_ERROR_MISSING_TEXT      "Resource missing"

/**
 * @brief Response code for a busy resource.
 */
#define STATUS_ERROR_BUSY              "409"

/**
 * @brief Response code text for a busy resource.
 */
#define STATUS_ERROR_BUSY_TEXT         "Resource busy"

/**
 * @brief Response code when authorisation failed.
 */
#define STATUS_ERROR_AUTH_FAILED       "400"

/**
 * @brief Response code text when authorisation failed.
 */
#define STATUS_ERROR_AUTH_FAILED_TEXT  "Authentication failed"

/**
 * @brief Response code on success.
 */
#define STATUS_OK                      "200"

/**
 * @brief Response code text on success.
 */
#define STATUS_OK_TEXT                 "OK"

/**
 * @brief Response code on success, when a resource is created.
 */
#define STATUS_OK_CREATED              "201"

/**
 * @brief Response code on success, when a resource is created.
 */
#define STATUS_OK_CREATED_TEXT         "OK, resource created"

/**
 * @brief Response code on success, when the operation will finish later.
 */
#define STATUS_OK_REQUESTED            "202"

/**
 * @brief Response code text on success, when the operation will finish later.
 */
#define STATUS_OK_REQUESTED_TEXT       "OK, request submitted"

/**
 * @brief Response code for an internal error.
 */
#define STATUS_INTERNAL_ERROR          "500"

/**
 * @brief Response code text for an internal error.
 */
#define STATUS_INTERNAL_ERROR_TEXT     "Internal error"

/**
 * @brief Response code when a service is unavailable.
 */
#define STATUS_SERVICE_UNAVAILABLE     "503"

/**
 * @brief Response code text when a service is unavailable.
 */
#define STATUS_SERVICE_UNAVAILABLE_TEXT "Service unavailable"

/**
 * @brief Response code when a service is down.
 */
#define STATUS_SERVICE_DOWN            "503"

/**
 * @brief Response code text when a service is down.
 */
#define STATUS_SERVICE_DOWN_TEXT       "Service temporarily down"


/* OMP parser. */

/**
 * @brief A handle on an OMP parser.
 */
typedef struct
{
  int (*client_writer) (const char*, void*);  ///< Writes to the client.
  void* client_writer_data;       ///< Argument to client_writer.
  int importing;                  ///< Whether the current op is importing.
  int read_over;                  ///< Read over any child elements.
  int parent_state;               ///< Parent state when reading over.
  gchar **disabled_commands;      ///< Disabled commands.
} omp_parser_t;

static int
process_omp (omp_parser_t *, const gchar *, gchar **);

/**
 * @brief Create an OMP parser.
 *
 * @param[in]  write_to_client       Function to write to client.
 * @param[in]  write_to_client_data  Argument to \p write_to_client.
 * @param[in]  disable               Commands to disable.  Freed by
 *                                   omp_parser_free.
 *
 * @return An OMP parser.
 */
omp_parser_t *
omp_parser_new (int (*write_to_client) (const char*, void*), void* write_to_client_data,
                gchar **disable)
{
  omp_parser_t *omp_parser = (omp_parser_t*) g_malloc0 (sizeof (omp_parser_t));
  omp_parser->client_writer = write_to_client;
  omp_parser->client_writer_data = write_to_client_data;
  omp_parser->read_over = 0;
  omp_parser->disabled_commands = disable;
  return omp_parser;
}

/**
 * @brief Free an OMP parser.
 *
 * @param[in]  omp_parser  OMP parser.
 *
 * @return An OMP parser.
 */
void
omp_parser_free (omp_parser_t *omp_parser)
{
  g_strfreev (omp_parser->disabled_commands);
  g_free (omp_parser);
}

/**
 * @brief Check if command has been disabled.
 *
 * @param[in]  omp_parser  Parser.
 * @param[in]  name        Command name.
 *
 * @return 1 disabled, 0 enabled.
 */
static int
command_disabled (omp_parser_t *omp_parser, const gchar *name)
{
  gchar **disabled;
  disabled = omp_parser->disabled_commands;
  if (disabled)
    while (*disabled)
      {
        if (strcasecmp (*disabled, name) == 0)
          return 1;
        disabled++;
      }
  return 0;
}


/* Command data passed between parser callbacks. */

/**
 * @brief Create a new preference.
 *
 * @param[in]  name      Name of preference.
 * @param[in]  type      Type of preference.
 * @param[in]  value     Value of preference.
 * @param[in]  nvt_name  Name of NVT of preference.
 * @param[in]  nvt_oid   OID of NVT of preference.
 * @param[in]  alts      Array of gchar's.  Alternative values for type radio.
 *
 * @return Newly allocated preference.
 */
static gpointer
preference_new (char *name, char *type, char *value, char *nvt_name,
                char *nvt_oid, array_t *alts)
{
  preference_t *preference;

  preference = (preference_t*) g_malloc0 (sizeof (preference_t));
  preference->name = name;
  preference->type = type;
  preference->value = value;
  preference->nvt_name = nvt_name;
  preference->nvt_oid = nvt_oid;
  preference->alts = alts;

  return preference;
}

/**
 * @brief Create a new NVT selector.
 *
 * @param[in]  name           Name of NVT selector.
 * @param[in]  type           Type of NVT selector.
 * @param[in]  include        Include/exclude flag.
 * @param[in]  family_or_nvt  Family or NVT.
 *
 * @return Newly allocated NVT selector.
 */
static gpointer
nvt_selector_new (char *name, char *type, int include, char *family_or_nvt)
{
  nvt_selector_t *selector;

  selector = (nvt_selector_t*) g_malloc0 (sizeof (nvt_selector_t));
  selector->name = name;
  selector->type = type;
  selector->include = include;
  selector->family_or_nvt = family_or_nvt;

  return selector;
}

/**
 * @brief Command data for the create_agent command.
 */
typedef struct
{
  char *comment;                  ///< Comment.
  char *copy;                     ///< UUID of resource to copy.
  char *howto_install;            ///< Install HOWTO.
  char *howto_use;                ///< Usage HOWTO.
  char *installer;                ///< Installer content.
  char *installer_filename;       ///< Installer filename.
  char *installer_signature;      ///< Installer signature.
  char *name;                     ///< Agent name.
} create_agent_data_t;

/**
 * @brief Free members of a create_agent_data_t and set them to NULL.
 */
static void
create_agent_data_reset (create_agent_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->howto_install);
  free (data->howto_use);
  free (data->installer);
  free (data->installer_filename);
  free (data->installer_signature);
  free (data->name);

  memset (data, 0, sizeof (create_agent_data_t));
}

/**
 * @brief Command data for the import part of the create_config command.
 */
typedef struct
{
  int import;                        ///< The import element was present.
  char *comment;                     ///< Comment.
  char *name;                        ///< Config name.
  array_t *nvt_selectors;            ///< Array of nvt_selector_t's.
  char *nvt_selector_name;           ///< In NVT_SELECTORS name of selector.
  char *nvt_selector_type;           ///< In NVT_SELECTORS type of selector.
  char *nvt_selector_include;        ///< In NVT_SELECTORS include/exclude flag.
  char *nvt_selector_family_or_nvt;  ///< In NVT_SELECTORS family/NVT flag.
  array_t *preferences;              ///< Array of preference_t's.
  array_t *preference_alts;          ///< Array of gchar's in PREFERENCES.
  char *preference_alt;              ///< Single radio alternative in PREFERENCE.
  char *preference_name;             ///< Name in PREFERENCE.
  char *preference_nvt_name;         ///< NVT name in PREFERENCE.
  char *preference_nvt_oid;          ///< NVT OID in PREFERENCE.
  char *preference_type;             ///< Type in PREFERENCE.
  char *preference_value;            ///< Value in PREFERENCE.
} import_config_data_t;

/**
 * @brief Command data for the create_config command.
 */
typedef struct
{
  char *comment;                     ///< Comment.
  char *scanner;                     ///< Scanner to create config from.
  char *copy;                        ///< Config to copy.
  import_config_data_t import;       ///< Config to import.
  char *name;                        ///< Name.
} create_config_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_config_data_reset (create_config_data_t *data)
{
  import_config_data_t *import = (import_config_data_t*) &data->import;

  g_free (data->comment);
  g_free (data->copy);
  g_free (data->scanner);

  g_free (import->comment);
  g_free (import->name);
  array_free (import->nvt_selectors);
  g_free (import->nvt_selector_name);
  g_free (import->nvt_selector_type);
  g_free (import->nvt_selector_family_or_nvt);

  if (import->preferences)
    {
      guint index = import->preferences->len;
      while (index--)
        {
          const preference_t *preference;
          preference = (preference_t*) g_ptr_array_index (import->preferences,
                                                          index);
          if (preference)
            array_free (preference->alts);
        }
      array_free (import->preferences);
    }

  g_free (import->preference_alt);
  g_free (import->preference_name);
  g_free (import->preference_nvt_name);
  g_free (import->preference_nvt_oid);
  g_free (import->preference_type);
  g_free (import->preference_value);

  g_free (data->name);

  memset (data, 0, sizeof (create_config_data_t));
}

/**
 * @brief Command data for the create_alert command.
 *
 * The pointers in the *_data arrays point to memory that contains two
 * strings concatentated, with a single \\0 between them.  The first string
 * is the name of the extra data (for example "To Address"), the second is
 * the value the the data (for example "alice@example.org").
 */
typedef struct
{
  char *comment;             ///< Comment.
  char *copy;                ///< UUID of alert to copy.
  char *condition;           ///< Condition for alert, e.g. "Always".
  array_t *condition_data;   ///< Array of pointers.  Extra data for condition.
  char *event;               ///< Event that will cause alert.
  array_t *event_data;       ///< Array of pointers.  Extra data for event.
  char *filter_id;           ///< UUID of filter.
  char *method;              ///< Method of alert, e.g. "Email".
  array_t *method_data;      ///< Array of pointer.  Extra data for method.
  char *name;                ///< Name of alert.
  char *part_data;           ///< Second part of data during *_data: value.
  char *part_name;           ///< First part of data during *_data: name.
} create_alert_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_alert_data_reset (create_alert_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->condition);
  array_free (data->condition_data);
  free (data->event);
  array_free (data->event_data);
  free (data->filter_id);
  free (data->method);
  array_free (data->method_data);
  free (data->name);
  free (data->part_data);
  free (data->part_name);

  memset (data, 0, sizeof (create_alert_data_t));
}

/**
 * @brief Command data for the create_filter command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *copy;                    ///< UUID of resource to copy.
  char *make_name_unique;        ///< Boolean.  Whether to make name unique.
  char *name;                    ///< Name of new filter.
  char *term;                    ///< Filter term.
  char *type;                    ///< Type of new filter.
} create_filter_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_filter_data_reset (create_filter_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->make_name_unique);
  free (data->name);
  free (data->term);
  free (data->type);

  memset (data, 0, sizeof (create_filter_data_t));
}

/**
 * @brief Command data for the create_group command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *copy;                    ///< UUID of resource to copy.
  char *name;                    ///< Name of new group.
  char *users;                   ///< Users belonging to new group.
} create_group_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_group_data_reset (create_group_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->name);
  free (data->users);

  memset (data, 0, sizeof (create_group_data_t));
}

/**
 * @brief Command data for the create_lsc_credential command.
 */
typedef struct
{
  char *comment;           ///< Comment.
  char *copy;              ///< UUID of resource to copy.
  int key;                 ///< Whether the command included a key element.
  char *key_phrase;        ///< Passphrase for key.
  char *key_private;       ///< Private key from key.
  char *login;             ///< Login name.
  char *name;              ///< LSC credential name.
  char *password;          ///< Password associated with login name.
} create_lsc_credential_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_lsc_credential_data_reset (create_lsc_credential_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->key_phrase);
  free (data->key_private);
  free (data->login);
  free (data->name);
  free (data->password);

  memset (data, 0, sizeof (create_lsc_credential_data_t));
}

/**
 * @brief Command data for the create_note command.
 */
typedef struct
{
  char *active;       ///< Whether the note is active.
  char *copy;         ///< UUID of resource to copy.
  char *hosts;        ///< Hosts to which to limit override.
  char *nvt_oid;      ///< NVT to which to limit override.
  char *port;         ///< Port to which to limit override.
  char *result_id;    ///< ID of result to which to limit override.
  char *severity;     ///< Severity score to which to limit note.
  char *task_id;      ///< ID of task to which to limit override.
  char *text;         ///< Text of override.
  char *threat;       ///< Threat to which to limit override.
} create_note_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_note_data_reset (create_note_data_t *data)
{
  free (data->active);
  free (data->copy);
  free (data->hosts);
  free (data->nvt_oid);
  free (data->port);
  free (data->result_id);
  free (data->severity);
  free (data->task_id);
  free (data->text);
  free (data->threat);

  memset (data, 0, sizeof (create_note_data_t));
}

/**
 * @brief Command data for the create_override command.
 */
typedef struct
{
  char *active;       ///< Whether the override is active.
  char *copy;         ///< UUID of resource to copy.
  char *hosts;        ///< Hosts to which to limit override.
  char *new_threat;   ///< New threat value of overridden results.
  char *new_severity; ///< New severity score of overridden results.
  char *nvt_oid;      ///< NVT to which to limit override.
  char *port;         ///< Port to which to limit override.
  char *result_id;    ///< ID of result to which to limit override.
  char *severity;     ///< Severity score of results to override.
  char *task_id;      ///< ID of task to which to limit override.
  char *text;         ///< Text of override.
  char *threat;       ///< Threat to which to limit override.
} create_override_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_override_data_reset (create_override_data_t *data)
{
  free (data->active);
  free (data->copy);
  free (data->hosts);
  free (data->new_threat);
  free (data->new_severity);
  free (data->nvt_oid);
  free (data->port);
  free (data->result_id);
  free (data->task_id);
  free (data->text);
  free (data->threat);
  free (data->severity);

  memset (data, 0, sizeof (create_override_data_t));
}

/**
 * @brief Command data for the create_permission command.
 */
typedef struct
{
  char *comment;         ///< Comment.
  char *copy;            ///< UUID of resource to copy.
  char *name;            ///< Permission name.
  char *resource_type;   ///< Resource type, for special permissions.
  char *resource_id;     ///< Resource permission applies to.
  char *subject_type;    ///< Subject type permission applies to.
  char *subject_id;      ///< Subject permission applies to.
} create_permission_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_permission_data_reset (create_permission_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->name);
  free (data->resource_type);
  free (data->resource_id);
  free (data->subject_type);
  free (data->subject_id);

  memset (data, 0, sizeof (create_permission_data_t));
}

/**
 * @brief A port range.
 */
struct create_port_list_range
{
  char *comment;            ///< Comment.
  char *end;                ///< End.
  char *id;                 ///< UUID.
  char *start;              ///< Start.
  char *type;               ///< Type.
};

/**
 * @brief Port range type.
 */
typedef struct create_port_list_range create_port_list_range_t;

/**
 * @brief Command data for the create_port_list command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *id;                      ///< UUID.
  char *copy;                    ///< UUID of Port List to copy.
  int import;                    ///< Import flag.
  char *name;                    ///< Name of new port list.
  char *port_range;              ///< Port range for new port list.
  create_port_list_range_t *range;  ///< Current port range for import.
  array_t *ranges;               ///< Port ranges for import.
} create_port_list_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_port_list_data_reset (create_port_list_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->name);
  free (data->port_range);

  if (data->ranges)
    {
      guint index;

      index = data->ranges->len;
      while (index--)
        {
          create_port_list_range_t *range;
          range = (create_port_list_range_t*) g_ptr_array_index (data->ranges,
                                                                 index);
          if (range)
            {
              free (range->comment);
              free (range->end);
              free (range->id);
              free (range->start);
              free (range->type);
            }
        }
      array_free (data->ranges);
    }

  memset (data, 0, sizeof (create_port_list_data_t));
}

/**
 * @brief Command data for the create_port_range command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *end;                     ///< Last port.
  char *port_list_id;            ///< Port list for new port range.
  char *start;                   ///< First port.
  char *type;                    ///< Type of new port range.
} create_port_range_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_port_range_data_reset (create_port_range_data_t *data)
{
  free (data->comment);
  free (data->end);
  free (data->port_list_id);
  free (data->start);
  free (data->type);

  memset (data, 0, sizeof (create_port_range_data_t));
}

/**
 * @brief Command data for the create_report command.
 */
typedef struct
{
  char *detail_name;              ///< Name of current host detail.
  char *detail_value;             ///< Value of current host detail.
  char *detail_source_name;       ///< Name of source of current host detail.
  char *detail_source_type;       ///< Type of source of current host detail.
  char *detail_source_desc;       ///< Description of source of current detail.
  array_t *details;               ///< Host details.
  char *host_end;                 ///< End time for a host.
  char *host_end_host;            ///< Host name for end time.
  array_t *host_ends;             ///< All host ends.
  char *host_start;               ///< Start time for a host.
  char *host_start_host;          ///< Host name for start time.
  array_t *host_starts;           ///< All host starts.
  char *ip;                       ///< Current host for host details.
  char *result_description;       ///< Description of NVT for current result.
  char *result_host;              ///< Host for current result.
  char *result_nvt_oid;           ///< OID of NVT for current result.
  char *result_port;              ///< Port for current result.
  char *result_qod;               ///< QoD value of current result.
  char *result_qod_type;          ///< QoD type of current result.
  char *result_scan_nvt_version;  ///< Version of NVT used in scan.
  char *result_severity;          ///< Severity score for current result.
  char *result_threat;            ///< Message type for current result.
  array_t *results;               ///< All results.
  char *scan_end;                 ///< End time for a scan.
  char *scan_start;               ///< Start time for a scan.
  char *task_comment;             ///< Comment for container task.
  char *task_id;                  ///< ID of container task.
  char *task_name;                ///< Name for container task.
  char *type;                     ///< Type of report.
  int wrapper;                    ///< Whether there was a wrapper REPORT.
} create_report_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_report_data_reset (create_report_data_t *data)
{
  if (data->details)
    {
      guint index = data->details->len;
      while (index--)
        {
          host_detail_t *detail;
          detail = (host_detail_t*) g_ptr_array_index (data->details, index);
          if (detail)
            host_detail_free (detail);
        }
      array_free (data->details);
    }
  free (data->host_end);
  free (data->host_start);
  free (data->ip);
  free (data->result_description);
  free (data->result_host);
  free (data->result_nvt_oid);
  free (data->result_port);
  free (data->result_threat);
  if (data->results)
    {
      guint index = data->results->len;
      while (index--)
        {
          create_report_result_t *result;
          result = (create_report_result_t*) g_ptr_array_index (data->results,
                                                                index);
          if (result)
            {
              free (result->host);
              free (result->description);
              free (result->nvt_oid);
              free (result->port);
              free (result->qod);
              free (result->qod_type);
              free (result->scan_nvt_version);
              free (result->severity);
            }
        }
      array_free (data->results);
    }
  free (data->scan_end);
  free (data->scan_start);
  free (data->task_comment);
  free (data->task_id);
  free (data->task_name);
  free (data->type);

  memset (data, 0, sizeof (create_report_data_t));
}

/**
 * @brief Command data for the create_report_format command.
 */
typedef struct
{
  char *content_type;     ///< Content type.
  char *description;      ///< Description.
  char *extension;        ///< File extension.
  char *file;             ///< Current file during ...GRFR_REPORT_FORMAT_FILE.
  char *file_name;        ///< Name of current file.
  array_t *files;         ///< All files.
  char *global;           ///< Global flag.
  char *id;               ///< ID.
  int import;             ///< Boolean.  Whether to import a format.
  char *name;             ///< Name.
  char *param_value;      ///< Param value during ...GRFR_REPORT_FORMAT_PARAM.
  char *param_default;    ///< Default value for above param.
  char *param_name;       ///< Name of above param.
  char *param_option;     ///< Current option of above param.
  array_t *param_options; ///< Options for above param.
  array_t *params_options; ///< Options for all params.
  char *param_type;       ///< Type of above param.
  char *param_type_min;   ///< Min qualifier of above type.
  char *param_type_max;   ///< Max qualifier of above type.
  array_t *params;        ///< All params.
  char *signature;        ///< Signature.
  char *summary;          ///< Summary.
  char *copy;             ///< UUID of Report Format to copy.
} create_report_format_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_report_format_data_reset (create_report_format_data_t *data)
{
  free (data->content_type);
  free (data->description);
  free (data->extension);
  free (data->file);
  free (data->file_name);
  array_free (data->files);
  free (data->global);
  free (data->id);
  free (data->name);
  free (data->copy);
  free (data->param_default);
  free (data->param_name);

  if (data->params_options)
    {
      guint index = data->params_options->len;
      while (index--)
        {
          array_t *options;
          options = (array_t*) g_ptr_array_index (data->params_options, index);
          if (options)
            array_free (options);
        }
      g_ptr_array_free (data->params_options, TRUE);
    }

  free (data->param_type);
  free (data->param_type_min);
  free (data->param_type_max);
  free (data->param_value);
  array_free (data->params);
  free (data->summary);

  memset (data, 0, sizeof (create_report_format_data_t));
}

/**
 * @brief Command data for the create_role command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *copy;                    ///< UUID of resource to copy.
  char *name;                    ///< Name of new role.
  char *users;                   ///< Users belonging to new role.
} create_role_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_role_data_reset (create_role_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->name);
  free (data->users);

  memset (data, 0, sizeof (create_role_data_t));
}

/**
 * @brief Command data for the create_scanner command.
 */
typedef struct
{
  char *name;               ///< Name for new scanner.
  char *copy;               ///< UUID of scanner to copy.
  char *comment;            ///< Comment.
  char *host;               ///< Host of new scanner.
  char *port;               ///< Port of new scanner.
  char *type;               ///< Type of new scanner.
  char *ca_pub;             ///< CA Certificate of new scanner.
  char *key_pub;            ///< Certificate of new scanner.
  char *key_priv;           ///< Private key of new scanner.
} create_scanner_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_scanner_data_reset (create_scanner_data_t *data)
{
  free (data->name);
  free (data->copy);
  free (data->comment);
  free (data->host);
  free (data->port);
  free (data->type);
  free (data->ca_pub);
  free (data->key_pub);
  free (data->key_priv);

  memset (data, 0, sizeof (create_scanner_data_t));
}

/**
 * @brief Command data for the create_schedule command.
 */
typedef struct
{
  char *name;                    ///< Name for new schedule.
  char *comment;                 ///< Comment.
  char *copy;                    ///< UUID of resource to copy.
  char *first_time_day_of_month; ///< Day of month schedule must first run.
  char *first_time_hour;         ///< Hour schedule must first run.
  char *first_time_minute;       ///< Minute schedule must first run.
  char *first_time_month;        ///< Month schedule must first run.
  char *first_time_year;         ///< Year schedule must first run.
  char *period;                  ///< Period of schedule (how often it runs).
  char *period_unit;             ///< Unit of period: "hour", "day", "week", ....
  char *duration;                ///< Duration of schedule (how long it runs for).
  char *duration_unit;           ///< Unit of duration: "hour", "day", "week", ....
  char *timezone;                ///< Time zone of the schedule
} create_schedule_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_schedule_data_reset (create_schedule_data_t *data)
{
  free (data->name);
  free (data->copy);
  free (data->comment);
  free (data->first_time_day_of_month);
  free (data->first_time_hour);
  free (data->first_time_minute);
  free (data->first_time_month);
  free (data->first_time_year);
  free (data->period);
  free (data->period_unit);
  free (data->duration);
  free (data->duration_unit);
  free (data->timezone);

  memset (data, 0, sizeof (create_schedule_data_t));
}

/**
 * @brief Command data for the create_slave command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *host;                    ///< Host for new slave.
  char *copy;                    ///< UUID of slave to copy.
  char *login;                   ///< Login on slave.
  char *name;                    ///< Name of new slave.
  char *password;                ///< Password for login.
  char *port;                    ///< Port on host.
} create_slave_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_slave_data_reset (create_slave_data_t *data)
{
  free (data->comment);
  free (data->copy);
  free (data->host);
  free (data->login);
  free (data->name);
  free (data->password);
  free (data->port);

  memset (data, 0, sizeof (create_slave_data_t));
}

/**
 * @brief Command data for the create_target command.
 */
typedef struct
{
  char *alive_tests;             ///< Alive tests.
  char *comment;                 ///< Comment.
  char *exclude_hosts;           ///< Hosts to exclude from set.
  char *reverse_lookup_only;     ///< Boolean. Whether to consider only hosts that reverse lookup.
  char *reverse_lookup_unify;    ///< Boolean. Whether to unify based on reverse lookup.
  char *copy;                    ///< UUID of resource to copy.
  char *hosts;                   ///< Hosts for new target.
  char *port_list_id;            ///< Port list for new target.
  char *port_range;              ///< Port range for new target.
  char *ssh_lsc_credential_id;   ///< SSH LSC credential for new target.
  char *ssh_port;                ///< Port for SSH LSC.
  char *smb_lsc_credential_id;   ///< SMB LSC credential for new target.
  char *esxi_lsc_credential_id;  ///< ESXi LSC credential for new target.
  char *make_name_unique;        ///< Boolean.  Whether to make name unique.
  char *name;                    ///< Name of new target.
} create_target_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_target_data_reset (create_target_data_t *data)
{
  free (data->alive_tests);
  free (data->comment);
  free (data->exclude_hosts);
  free (data->reverse_lookup_only);
  free (data->reverse_lookup_unify);
  free (data->copy);
  free (data->hosts);
  free (data->port_list_id);
  free (data->port_range);
  free (data->ssh_lsc_credential_id);
  free (data->ssh_port);
  free (data->smb_lsc_credential_id);
  free (data->esxi_lsc_credential_id);
  free (data->make_name_unique);
  free (data->name);

  memset (data, 0, sizeof (create_target_data_t));
}

/**
 * @brief Command data for the create_tag command.
 */
typedef struct
{
  char *active;        ///< Whether the tag is active.
  char *resource_id;   ///< ID of the resource to which to attach the tag.
  char *resource_type; ///< Type of the resource to which to attach the tag.
  char *comment;       ///< Comment to add to the tag.
  char *name;          ///< Name of the tag.
  char *value;         ///< Value of the tag.
  char *copy;          ///< UUID of resource to copy.
} create_tag_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_tag_data_reset (create_tag_data_t *data)
{
  free (data->active);
  free (data->resource_type);
  free (data->resource_id);
  free (data->comment);
  free (data->name);
  free (data->value);
  free (data->copy);
  memset (data, 0, sizeof (create_tag_data_t));
}

/**
 * @brief Command data for the create_task command.
 */
typedef struct
{
  char *alterable;      ///< Boolean.  Whether task is alterable.
  char *config_id;      ///< ID of task config.
  char *hosts_ordering; ///< Order for scanning target hosts.
  char *scanner_id;     ///< ID of task scanner.
  array_t *alerts;      ///< IDs of alerts.
  char *copy;           ///< UUID of resource to copy.
  array_t *groups;      ///< IDs of groups.
  char *observers;      ///< Space separated names of observer users.
  name_value_t *preference;  ///< Current preference.
  array_t *preferences; ///< Preferences.
  char *schedule_id;    ///< ID of task schedule.
  char *schedule_periods; ///< Number of periods the schedule must run for.
  char *slave_id;       ///< ID of task slave.
  char *target_id;      ///< ID of task target.
  task_t task;          ///< ID of new task.
} create_task_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
create_task_data_reset (create_task_data_t *data)
{
  free (data->alterable);
  free (data->config_id);
  free (data->hosts_ordering);
  free (data->scanner_id);
  free (data->copy);
  array_free (data->alerts);
  array_free (data->groups);
  free (data->observers);
  if (data->preferences)
    {
      guint index = data->preferences->len;
      while (index--)
        {
          name_value_t *pair;
          pair = (name_value_t*) g_ptr_array_index (data->preferences, index);
          if (pair)
            {
              g_free (pair->name);
              g_free (pair->value);
            }
        }
    }
  array_free (data->preferences);
  free (data->schedule_id);
  free (data->schedule_periods);
  free (data->slave_id);
  free (data->target_id);

  memset (data, 0, sizeof (create_task_data_t));
}

/* Command data passed between parser callbacks. */

typedef struct
{
  char *copy;             ///< UUID of resource to copy.
  int sort_order;
  array_t *groups;        ///< IDs of groups.
  char *hosts;
  int hosts_allow;
  char *ifaces;
  int ifaces_allow;
  char *name;
  char *password;
  array_t *roles;
  gchar *current_source;
  array_t *sources;
} create_user_data_t;

/**
 * @brief Reset CREATE_USER data.
 *
 * @param[in]  data  Command data.
 */
static void
create_user_data_reset (create_user_data_t * data)
{
  g_free (data->copy);
  array_free (data->groups);
  g_free (data->name);
  g_free (data->password);
  g_free (data->hosts);
  g_free (data->ifaces);
  array_free (data->roles);
  if (data->sources)
    {
      array_free (data->sources);
    }
  g_free (data->current_source);
  memset (data, 0, sizeof (create_user_data_t));
}

/**
 * @brief Command data for the delete_agent command.
 */
typedef struct
{
  char *agent_id;   ///< ID of agent to delete.
  int ultimate;     ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_agent_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_agent_data_reset (delete_agent_data_t *data)
{
  free (data->agent_id);

  memset (data, 0, sizeof (delete_agent_data_t));
}

/**
 * @brief Command data for the delete_config command.
 */
typedef struct
{
  char *config_id;   ///< ID of config to delete.
  int ultimate;      ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_config_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_config_data_reset (delete_config_data_t *data)
{
  free (data->config_id);

  memset (data, 0, sizeof (delete_config_data_t));
}

/**
 * @brief Command data for the delete_alert command.
 */
typedef struct
{
  char *alert_id;   ///< ID of alert to delete.
  int ultimate;     ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_alert_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_alert_data_reset (delete_alert_data_t *data)
{
  free (data->alert_id);

  memset (data, 0, sizeof (delete_alert_data_t));
}

/**
 * @brief Command data for the delete_filter command.
 */
typedef struct
{
  char *filter_id;   ///< ID of filter to delete.
  int ultimate;      ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_filter_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_filter_data_reset (delete_filter_data_t *data)
{
  free (data->filter_id);

  memset (data, 0, sizeof (delete_filter_data_t));
}

/**
 * @brief Command data for the delete_group command.
 */
typedef struct
{
  char *group_id;   ///< ID of group to delete.
  int ultimate;      ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_group_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_group_data_reset (delete_group_data_t *data)
{
  free (data->group_id);

  memset (data, 0, sizeof (delete_group_data_t));
}

/**
 * @brief Command data for the delete_lsc_credential command.
 */
typedef struct
{
  char *lsc_credential_id;   ///< ID of LSC credential to delete.
  int ultimate;      ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_lsc_credential_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_lsc_credential_data_reset (delete_lsc_credential_data_t *data)
{
  free (data->lsc_credential_id);

  memset (data, 0, sizeof (delete_lsc_credential_data_t));
}

/**
 * @brief Command data for the delete_note command.
 */
typedef struct
{
  char *note_id;   ///< ID of note to delete.
  int ultimate;    ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_note_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_note_data_reset (delete_note_data_t *data)
{
  free (data->note_id);

  memset (data, 0, sizeof (delete_note_data_t));
}

/**
 * @brief Command data for the delete_override command.
 */
typedef struct
{
  char *override_id;   ///< ID of override to delete.
  int ultimate;        ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_override_data_t;

/**
 * @brief Command data for the delete_permission command.
 */
typedef struct
{
  char *permission_id; ///< ID of permission to delete.
  int ultimate;        ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_permission_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_permission_data_reset (delete_permission_data_t *data)
{
  free (data->permission_id);

  memset (data, 0, sizeof (delete_permission_data_t));
}

/**
 * @brief Command data for the delete_port_list command.
 */
typedef struct
{
  char *port_list_id;  ///< ID of port list to delete.
  int ultimate;        ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_port_list_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_port_list_data_reset (delete_port_list_data_t *data)
{
  free (data->port_list_id);

  memset (data, 0, sizeof (delete_port_list_data_t));
}

/**
 * @brief Command data for the delete_port_range command.
 */
typedef struct
{
  char *port_range_id;  ///< ID of port range to delete.
  int ultimate;         ///< Dummy field for generic macros.
} delete_port_range_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_port_range_data_reset (delete_port_range_data_t *data)
{
  free (data->port_range_id);

  memset (data, 0, sizeof (delete_port_range_data_t));
}

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_override_data_reset (delete_override_data_t *data)
{
  free (data->override_id);

  memset (data, 0, sizeof (delete_override_data_t));
}

/**
 * @brief Command data for the delete_report command.
 */
typedef struct
{
  char *report_id;   ///< ID of report to delete.
  int ultimate;      ///< Dummy field for generic macros.
} delete_report_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_report_data_reset (delete_report_data_t *data)
{
  free (data->report_id);

  memset (data, 0, sizeof (delete_report_data_t));
}

/**
 * @brief Command data for the delete_report_format command.
 */
typedef struct
{
  char *report_format_id;   ///< ID of report format to delete.
  int ultimate;     ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_report_format_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_report_format_data_reset (delete_report_format_data_t *data)
{
  free (data->report_format_id);

  memset (data, 0, sizeof (delete_report_format_data_t));
}

/**
 * @brief Command data for the delete_role command.
 */
typedef struct
{
  char *role_id;     ///< ID of role to delete.
  int ultimate;      ///< Dummy field for generic macros.
} delete_role_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_role_data_reset (delete_role_data_t *data)
{
  free (data->role_id);

  memset (data, 0, sizeof (delete_role_data_t));
}

/**
 * @brief Command data for the delete_schedule command.
 */
typedef struct
{
  char *schedule_id;   ///< ID of schedule to delete.
  int ultimate;        ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_schedule_data_t;

/**
 * @brief Command data for the delete_scanner command.
 */
typedef struct
{
  char *scanner_id; ///< ID of scanner to delete.
  int ultimate;     ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_scanner_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_scanner_data_reset (delete_scanner_data_t *data)
{
  g_free (data->scanner_id);

  memset (data, 0, sizeof (delete_scanner_data_t));
}

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_schedule_data_reset (delete_schedule_data_t *data)
{
  free (data->schedule_id);

  memset (data, 0, sizeof (delete_schedule_data_t));
}

/**
 * @brief Command data for the delete_slave command.
 */
typedef struct
{
  char *slave_id;   ///< ID of slave to delete.
  int ultimate;     ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_slave_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_slave_data_reset (delete_slave_data_t *data)
{
  free (data->slave_id);

  memset (data, 0, sizeof (delete_slave_data_t));
}

/**
 * @brief Command data for the delete_tag command.
 */
typedef struct
{
  char *tag_id;      ///< ID of tag to delete.
  int ultimate;      ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_tag_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_tag_data_reset (delete_tag_data_t *data)
{
  free (data->tag_id);

  memset (data, 0, sizeof (delete_tag_data_t));
}

/**
 * @brief Command data for the delete_target command.
 */
typedef struct
{
  char *target_id;   ///< ID of target to delete.
  int ultimate;      ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_target_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_target_data_reset (delete_target_data_t *data)
{
  free (data->target_id);

  memset (data, 0, sizeof (delete_target_data_t));
}

/**
 * @brief Command data for the delete_task command.
 */
typedef struct
{
  char *task_id;   ///< ID of task to delete.
  int ultimate;    ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_task_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_task_data_reset (delete_task_data_t *data)
{
  free (data->task_id);

  memset (data, 0, sizeof (delete_task_data_t));
}

/**
 * @brief Command data for the delete_user command.
 */
typedef struct
{
  char *name;      ///< Name of user to delete.
  char *user_id;   ///< ID of user to delete.
  int ultimate;    ///< Boolean.  Whether to remove entirely or to trashcan.
} delete_user_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
delete_user_data_reset (delete_user_data_t *data)
{
  free (data->name);
  free (data->user_id);

  memset (data, 0, sizeof (delete_user_data_t));
}

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_data_reset (get_data_t *data)
{
  free (data->id);
  free (data->filt_id);
  free (data->filter);
  free (data->filter_replace);
  free (data->filter_replacement);
  free (data->subtype);
  free (data->type);

  memset (data, 0, sizeof (get_data_t));
}

/**
 * @brief Reset command data.
 *
 * @param[in]  data              GET operation data.
 * @param[in]  type              Resource type.
 * @param[in]  attribute_names   XML attribute names.
 * @param[in]  attribute_values  XML attribute values.
 *
 * @param[in]  data  Command data.
 */
static void
get_data_parse_attributes (get_data_t *data, const gchar *type,
                           const gchar **attribute_names,
                           const gchar **attribute_values)
{
  gchar *name;
  const gchar *attribute;

  data->type = g_strdup (type);

  append_attribute (attribute_names, attribute_values, "filter",
                    &data->filter);

  name = g_strdup_printf ("%s_id", type);
  append_attribute (attribute_names, attribute_values, name,
                    &data->id);
  g_free (name);

  append_attribute (attribute_names, attribute_values, "filt_id",
                    &data->filt_id);

  if (find_attribute (attribute_names, attribute_values,
                      "trash", &attribute))
    data->trash = strcmp (attribute, "0");
  else
    data->trash = 0;

  if (find_attribute (attribute_names, attribute_values,
                      "details", &attribute))
    data->details = strcmp (attribute, "0");
  else
    data->details = 0;

  append_attribute (attribute_names, attribute_values, "filter_replace",
                    &data->filter_replace);
}

/**
 * @brief Command data for the get_agents command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
  char *format;          ///< Format requested: "installer", "howto_use", ....
} get_agents_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_agents_data_reset (get_agents_data_t *data)
{
  get_data_reset (&data->get);
  free (data->format);

  memset (data, 0, sizeof (get_agents_data_t));
}

/**
 * @brief Command data for the get_aggregates command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
  char *type;            ///< Resource type.
  char *subtype;         ///< Resource subtype.
  char *data_column;     ///< Column to calculate aggregate for.
  char *group_column;    ///< Column to group data by.
} get_aggregates_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_aggregates_data_reset (get_aggregates_data_t *data)
{
  get_data_reset (&data->get);
  free (data->type);
  free (data->data_column);
  free (data->group_column);

  memset (data, 0, sizeof (get_aggregates_data_t));
}


/**
 * @brief Command data for the get_configs command.
 */
typedef struct
{
  int families;          ///< Boolean.  Whether to include config families.
  int preferences;       ///< Boolean.  Whether to include config preferences.
  get_data_t get;        ///< Get args.
  int tasks;             ///< Boolean.  Whether to include tasks that use scan config.
} get_configs_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_configs_data_reset (get_configs_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_configs_data_t));
}

/**
 * @brief Command data for the get_alerts command.
 */
typedef struct
{
  get_data_t get;   ///< Get args.
  int tasks;        ///< Boolean.  Whether to include tasks that use alert.
} get_alerts_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_alerts_data_reset (get_alerts_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_alerts_data_t));
}

/**
 * @brief Command data for the get_filters command.
 */
typedef struct
{
  get_data_t get;    ///< Get args.
  int alerts;        ///< Boolean.  Whether to include alerts that use filter.
} get_filters_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_filters_data_reset (get_filters_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_filters_data_t));
}

/**
 * @brief Command data for the get_groups command.
 */
typedef struct
{
  get_data_t get;    ///< Get args.
} get_groups_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_groups_data_reset (get_groups_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_groups_data_t));
}

/**
 * @brief Command data for the get_info command.
 */
typedef struct
{
  char *type;         ///< Requested information type.
  char *name;         ///< Name of the info
  get_data_t get;     ///< Get Args.
  int details;        ///< Boolean. Weather to include full details.
} get_info_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_info_data_reset (get_info_data_t *data)
{
  free (data->type);
  free (data->name);
  get_data_reset (&data->get);

  memset (data, 0, sizeof (get_info_data_t));
}

/**
 * @brief Command data for the get_lsc_credentials command.
 */
typedef struct
{
  char *format;      ///< Format requested: "key", "deb", ....
  get_data_t get;    ///< Get Args.
  int targets;       ///< Boolean.  Whether to return targets using credential.
} get_lsc_credentials_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_lsc_credentials_data_reset (get_lsc_credentials_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_lsc_credentials_data_t));
}

/**
 * @brief Command data for the get_notes command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
  char *note_id;         ///< ID of single note to get.
  char *nvt_oid;         ///< OID of NVT to which to limit listing.
  char *task_id;         ///< ID of task to which to limit listing.
  int result;            ///< Boolean.  Whether to include associated results.
} get_notes_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_notes_data_reset (get_notes_data_t *data)
{
  free (data->note_id);
  free (data->nvt_oid);
  free (data->task_id);

  memset (data, 0, sizeof (get_notes_data_t));
}

/**
 * @brief Command data for the get_nvts command.
 */
typedef struct
{
  char *config_id;       ///< ID of config to which to limit NVT selection.
  int details;           ///< Boolean.  Whether to include full NVT details.
  char *family;          ///< Name of family to which to limit NVT selection.
  char *nvt_oid;         ///< Name of single NVT to get.
  int preference_count;  ///< Boolean.  Whether to include NVT preference count.
  int preferences;       ///< Boolean.  Whether to include NVT preferences.
  char *sort_field;      ///< Field to sort results on.
  int sort_order;        ///< Result sort order: 0 descending, else ascending.
  int timeout;           ///< Boolean.  Whether to include timeout preference.
} get_nvts_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_nvts_data_reset (get_nvts_data_t *data)
{
  free (data->config_id);
  free (data->family);
  free (data->nvt_oid);
  free (data->sort_field);

  memset (data, 0, sizeof (get_nvts_data_t));
}

/**
 * @brief Command data for the get_nvt_families command.
 */
typedef struct
{
  int sort_order;        ///< Result sort order: 0 descending, else ascending.
} get_nvt_families_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_nvt_families_data_reset (get_nvt_families_data_t *data)
{
  memset (data, 0, sizeof (get_nvt_families_data_t));
}

/**
 * @brief Command data for the get_overrides command.
 */
typedef struct
{
  get_data_t get;      ///< Get args.
  char *override_id;   ///< ID of override to get.
  char *nvt_oid;       ///< OID of NVT to which to limit listing.
  char *task_id;       ///< ID of task to which to limit listing.
  int result;          ///< Boolean.  Whether to include associated results.
} get_overrides_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_overrides_data_reset (get_overrides_data_t *data)
{
  free (data->override_id);
  free (data->nvt_oid);
  free (data->task_id);

  memset (data, 0, sizeof (get_overrides_data_t));
}

/**
 * @brief Command data for the get_permissions command.
 */
typedef struct
{
  get_data_t get;     ///< Get args.
  char *resource_id;  ///< Resource whose permissions to get.
} get_permissions_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_permissions_data_reset (get_permissions_data_t *data)
{
  free (data->resource_id);

  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_permissions_data_t));
}

/**
 * @brief Command data for the get_port_lists command.
 */
typedef struct
{
  int targets;         ///< Boolean. Include targets that use Port List or not.
  get_data_t get;      ///< Get args.
} get_port_lists_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_port_lists_data_reset (get_port_lists_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_port_lists_data_t));
}

/**
 * @brief Command data for the get_preferences command.
 */
typedef struct
{
  char *config_id;  ///< Config whose preference values to get.
  char *nvt_oid;    ///< Single NVT whose preferences to get.
  char *preference; ///< Single preference to get.
} get_preferences_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_preferences_data_reset (get_preferences_data_t *data)
{
  free (data->config_id);
  free (data->nvt_oid);
  free (data->preference);

  memset (data, 0, sizeof (get_preferences_data_t));
}

/**
 * @brief Command data for the get_reports command.
 */
typedef struct
{
  get_data_t get;        ///< Get args with result filtering.
  get_data_t report_get; ///< Get args with report filtering.
  int apply_overrides;   ///< Boolean.  Whether to apply overrides to results.
  char *delta_report_id; ///< ID of report to compare single report to.
  char *delta_states;    ///< Delta states (Changed Gone New Same) to include.
  char *format_id;       ///< ID of report format.
  char *alert_id;        ///< ID of alert.
  char *report_id;       ///< ID of single report to get.
  int first_result;      ///< Skip over results before this result number.
  int max_results;       ///< Maximum number of results return.
  int host_first_result; ///< Skip over results before this result number.
  int host_max_results;  ///< Maximum number of results return.
  char *sort_field;      ///< Field to sort results on.
  int sort_order;        ///< Result sort order: 0 descending, else ascending.
  char *levels;          ///< Letter encoded threat level filter.
  char *host_levels;     ///< Letter encoded threat level filter, for hosts.
  char *search_phrase;   ///< Search phrase result filter.
  char *host_search_phrase;  ///< Search phrase result filter.
  char *min_cvss_base;   ///< Minimum CVSS base filter.
  char *min_qod;         ///< Minimum QoD filter.
  int autofp;            ///< Boolean.  Whether to apply auto FP filter.
  int notes;             ///< Boolean.  Whether to include associated notes.
  int notes_details;     ///< Boolean.  Whether to include details of above.
  int overrides;         ///< Boolean.  Whether to include associated overrides.
  int overrides_details; ///< Boolean.  Whether to include details of above.
  int result_hosts_only; ///< Boolean.  Whether to include only resulted hosts.
  char *type;            ///< Type of report.
  char *host;            ///< Host for asset report.
  char *pos;             ///< Position of report from end.
  int ignore_pagination; ///< Boolean.  Whether to ignore pagination filters.
  char *timezone;        ///< Timezone.
} get_reports_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_reports_data_reset (get_reports_data_t *data)
{
  get_data_reset (&data->get);
  get_data_reset (&data->report_get);
  free (data->delta_report_id);
  free (data->delta_states);
  free (data->format_id);
  free (data->alert_id);
  free (data->report_id);
  free (data->sort_field);
  free (data->levels);
  free (data->host_levels);
  free (data->search_phrase);
  free (data->host_search_phrase);
  free (data->min_cvss_base);
  free (data->min_qod);
  free (data->type);
  free (data->host);
  free (data->pos);

  memset (data, 0, sizeof (get_reports_data_t));
}

/**
 * @brief Command data for the get_report_formats command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
  int alerts;   ///< Boolean.  Whether to include alerts that use Report Format
  int params;            ///< Boolean.  Whether to include params.
} get_report_formats_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_report_formats_data_reset (get_report_formats_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_report_formats_data_t));
}

/**
 * @brief Command data for the get_results command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
  int apply_overrides;   ///< Boolean.  Whether to apply overrides to results.
  int apply_overrides_set; /// < Boolean. Whether apply_overrides was set.
  int autofp;            ///< Boolean.  Whether to apply auto FP filter.
  char *task_id;         ///< Task associated with results.
  int notes;             ///< Boolean.  Whether to include associated notes.
  int notes_details;     ///< Boolean.  Whether to include details of above.
  int overrides;         ///< Boolean.  Whether to include associated overrides.
  int overrides_details; ///< Boolean.  Whether to include details of above.
} get_results_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_results_data_reset (get_results_data_t *data)
{
  get_data_reset (&data->get);
  free (data->task_id);

  memset (data, 0, sizeof (get_results_data_t));
}

/**
 * @brief Command data for the get_roles command.
 */
typedef struct
{
  get_data_t get;    ///< Get args.
} get_roles_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_roles_data_reset (get_roles_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_roles_data_t));
}

/**
 * @brief Command data for the get_schedules command.
 */
typedef struct
{
  get_data_t get;      ///< Get args.
  int tasks;           ///< Boolean.  Whether to include tasks that use this schedule.
} get_schedules_data_t;

/**
 * @brief Command data for the get_scanners command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
} get_scanners_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_scanners_data_reset (get_scanners_data_t *data)
{
  get_data_reset (&data->get);

  memset (data, 0, sizeof (get_scanners_data_t));
}

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_schedules_data_reset (get_schedules_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_schedules_data_t));
}

/**
 * @brief Command data.
 */
typedef struct
{
  char *filter;        ///< Filter term.
  int first;           ///< Skip over rows before this number.
  int max;             ///< Maximum number of rows returned.
  char *sort_field;    ///< Field to sort results on.
  int sort_order;      ///< Result sort order: 0 descending, else ascending.
  char *setting_id;    ///< UUID of single setting to get.
} get_settings_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_settings_data_reset (get_settings_data_t *data)
{
  free (data->filter);
  free (data->setting_id);
  free (data->sort_field);

  memset (data, 0, sizeof (get_settings_data_t));
}

/**
 * @brief Command data for the get_slaves command.
 */
typedef struct
{
  get_data_t get;      ///< Get args.
  int tasks;           ///< Boolean.  Whether to include tasks that use this slave.
} get_slaves_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_slaves_data_reset (get_slaves_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_slaves_data_t));
}

/**
 * @brief Command data for the get_system_reports command.
 */
typedef struct
{
  int brief;        ///< Boolean.  Whether respond in brief.
  char *name;       ///< Name of single report to get.
  char *duration;   ///< Duration into the past to report on.
  char *slave_id;   ///< Slave that reports apply to, 0 for local Manager.
} get_system_reports_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_system_reports_data_reset (get_system_reports_data_t *data)
{
  free (data->name);
  free (data->duration);
  free (data->slave_id);

  memset (data, 0, sizeof (get_system_reports_data_t));
}

/**
 * @brief Command data for the get_tags command.
 */
typedef struct
{
  get_data_t get;    ///< Get args.
  int names_only;    ///< Boolean. Whether to get only distinct names.
} get_tags_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_tags_data_reset (get_tags_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_tags_data_t));
}

/**
 * @brief Command data for the get_targets command.
 */
typedef struct
{
  get_data_t get;    ///< Get args.
  int tasks;         ///< Boolean.  Whether to include tasks that use target.
} get_targets_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_targets_data_reset (get_targets_data_t *data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_targets_data_t));
}

/**
 * @brief Command data for the get_users command.
 */
typedef struct
{
  get_data_t get;    ///< Get args.
} get_users_data_t;

/**
 * @brief Reset GET_USERS data.
 */
static void
get_users_data_reset (get_users_data_t * data)
{
  get_data_reset (&data->get);
  memset (data, 0, sizeof (get_users_data_t));
}

/**
 * @brief Command data for the modify_config command.
 */
typedef struct
{
  char *comment;                       ///< New comment for config.
  char *config_id;                     ///< ID of config to modify.
  array_t *families_growing_empty; ///< New family selection: growing, empty.
  array_t *families_growing_all;   ///< New family selection: growing, all NVTs.
  array_t *families_static_all;    ///< New family selection: static, all NVTs.
  int family_selection_family_all;     ///< All flag in FAMILY_SELECTION/FAMILY.
  char *family_selection_family_all_text; ///< Text version of above.
  int family_selection_family_growing; ///< FAMILY_SELECTION/FAMILY growing flag.
  char *family_selection_family_growing_text; ///< Text version of above.
  char *family_selection_family_name;  ///< FAMILY_SELECTION/FAMILY family name.
  int family_selection_growing;        ///< Whether families in selection grow.
  char *family_selection_growing_text; ///< Text version of above.
  char *name;                          ///< New name for config.
  array_t *nvt_selection;              ///< OID array. New NVT set for config.
  char *nvt_selection_family;          ///< Family of NVT selection.
  char *nvt_selection_nvt_oid;         ///< OID during NVT_selection/NVT.
  char *preference_name;               ///< Config preference to modify.
  char *preference_nvt_oid;            ///< OID of NVT of preference.
  char *preference_value;              ///< New value for preference.
} modify_config_data_t;

/**
 * @brief Command data for the get_tasks command.
 */
typedef struct
{
  get_data_t get;        ///< Get args.
} get_tasks_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
get_tasks_data_reset (get_tasks_data_t *data)
{
  get_data_reset (&data->get);

  memset (data, 0, sizeof (get_tasks_data_t));
}

/**
 * @brief Command data for the help command.
 */
typedef struct
{
  char *format;       ///< Format.
  char *type;         ///< Type of help.
} help_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
help_data_reset (help_data_t *data)
{
  free (data->format);
  free (data->type);

  memset (data, 0, sizeof (help_data_t));
}

/**
 * @brief Command data for the modify_agent command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of agent.
  char *agent_id;                ///< agent UUID.
} modify_agent_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_agent_data_reset (modify_agent_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->agent_id);

  memset (data, 0, sizeof (modify_agent_data_t));
}

/**
 * @brief Command data for the modify_alert command.
 */
typedef struct
{
  char *alert_id;                ///< alert UUID.
  char *name;                    ///< Name of alert.
  char *comment;                 ///< Comment.
  char *event;                   ///< Event that will cause alert.
  array_t *event_data;           ///< Array of pointers. Extra data for event.
  char *filter_id;               ///< UUID of filter.
  char *condition;               ///< Condition for alert, e.g. "Always".
  array_t *condition_data;       ///< Array of pointers.  Extra data for condition.
  char *method;                  ///< Method of alert, e.g. "Email".
  array_t *method_data;          ///< Array of pointer.  Extra data for method.
  char *part_data;               ///< Second part of data during *_data: value.
  char *part_name;               ///< First part of data during *_data: name.
} modify_alert_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_alert_data_reset (modify_alert_data_t *data)
{
  free (data->alert_id);
  free (data->name);
  free (data->comment);
  free (data->filter_id);
  free (data->event);
  array_free (data->event_data);
  free (data->condition);
  array_free (data->condition_data);
  free (data->method);
  array_free (data->method_data);

  memset (data, 0, sizeof (modify_alert_data_t));
}

/**
 * @brief Authentication method settings.
 */
typedef struct
{
  gchar *group_name;            ///< Name of the current group
  GSList *settings;             ///< List of auth_conf_setting_t.
} auth_group_t;

/**
 * @brief Command data for the modify_auth command.
 */
typedef struct
{
  GSList *groups;               ///< List of auth_group_t
  GSList *curr_group_settings;  ///< Settings of currently parsed group.
} modify_auth_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
void
modify_auth_data_reset (modify_auth_data_t * data)
{
  GSList *item = data->groups;
  GSList *subitem = NULL;
  while (item)
    {
      auth_group_t *group = (auth_group_t *) item->data;
      g_free (group->group_name);
      /* Free settings. */
      subitem = group->settings;
      while (subitem)
        {
          auth_conf_setting_t *kvp = (auth_conf_setting_t *) subitem->data;
          g_free (kvp->key);
          g_free (kvp->value);
          g_free (kvp);
          subitem = g_slist_next (subitem);
        }
      item = g_slist_next (item);
    }
  g_slist_free (data->groups);

  if (data->curr_group_settings)
    {
      item = data->curr_group_settings;
      while (item)
        {
          /* Free settings. */
          auth_conf_setting_t *kvp = (auth_conf_setting_t *) item->data;
          g_free (kvp->key);
          g_free (kvp->value);
          g_free (kvp);
          item = g_slist_next (item);
        }
      g_slist_free (data->curr_group_settings);
    }
  memset (data, 0, sizeof (modify_auth_data_t));
}

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_config_data_reset (modify_config_data_t *data)
{
  free (data->comment);
  free (data->config_id);
  array_free (data->families_growing_empty);
  array_free (data->families_growing_all);
  array_free (data->families_static_all);
  free (data->family_selection_family_all_text);
  free (data->family_selection_family_growing_text);
  free (data->family_selection_family_name);
  free (data->family_selection_growing_text);
  free (data->name);
  array_free (data->nvt_selection);
  free (data->nvt_selection_family);
  free (data->nvt_selection_nvt_oid);
  free (data->preference_name);
  free (data->preference_nvt_oid);
  free (data->preference_value);

  memset (data, 0, sizeof (modify_config_data_t));
}

/**
 * @brief Command data for the modify_filter command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of filter.
  char *filter_id;               ///< Filter UUID.
  char *term;                    ///< Term for filter.
  char *type;                    ///< Type of filter.
} modify_filter_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_filter_data_reset (modify_filter_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->filter_id);
  free (data->term);
  free (data->type);

  memset (data, 0, sizeof (modify_filter_data_t));
}

/**
 * @brief Command data for the modify_group command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of group.
  char *group_id;                ///< Group UUID.
  char *users;                   ///< Users for group.
} modify_group_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_group_data_reset (modify_group_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->group_id);
  free (data->users);

  memset (data, 0, sizeof (modify_group_data_t));
}

/**
 * @brief Command data for the modify_lsc_credential command.
 */
typedef struct
{
  char *lsc_credential_id;    ///< ID of credential to modify.
  char *name;                 ///< Name.
  char *comment;              ///< Comment.
  char *login;                ///< Login.
  char *password;             ///< Password.
} modify_lsc_credential_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_lsc_credential_data_reset (modify_lsc_credential_data_t *data)
{
  free (data->lsc_credential_id);
  free (data->name);
  free (data->comment);
  free (data->login);
  free (data->password);

  memset (data, 0, sizeof (modify_lsc_credential_data_t));
}

/**
 * @brief Command data for the modify_permission command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of permission.
  char *permission_id;           ///< Permission UUID.
  char *resource_id;             ///< Resource.
  char *resource_type;           ///< Resource type, for Super permissions.
  char *subject_type;            ///< Subject type.
  char *subject_id;              ///< Subject UUID.
} modify_permission_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_permission_data_reset (modify_permission_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->resource_id);
  free (data->resource_type);
  free (data->permission_id);
  free (data->subject_type);
  free (data->subject_id);

  memset (data, 0, sizeof (modify_permission_data_t));
}

/**
 * @brief Command data for the modify_port_list command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of Port List.
  char *port_list_id;            ///< UUID of Port List.
} modify_port_list_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_port_list_data_reset (modify_port_list_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->port_list_id);

  memset (data, 0, sizeof (modify_port_list_data_t));
}

/**
 * @brief Command data for the modify_report command.
 */
typedef struct
{
  char *comment;       ///< Comment.
  char *report_id;     ///< ID of report to modify.
} modify_report_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_report_data_reset (modify_report_data_t *data)
{
  free (data->comment);
  free (data->report_id);

  memset (data, 0, sizeof (modify_report_data_t));
}

/**
 * @brief Command data for the modify_report_format command.
 */
typedef struct
{
  char *active;               ///< Boolean.  Whether report format is active.
  char *name;                 ///< Name.
  char *param_name;           ///< Param name.
  char *param_value;          ///< Param value.
  char *report_format_id;     ///< ID of report format to modify.
  char *summary;              ///< Summary.
} modify_report_format_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_report_format_data_reset (modify_report_format_data_t *data)
{
  free (data->active);
  free (data->name);
  free (data->param_name);
  free (data->param_value);
  free (data->report_format_id);
  free (data->summary);

  memset (data, 0, sizeof (modify_report_format_data_t));
}

/**
 * @brief Command data for the modify_role command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of role.
  char *role_id;                 ///< Role UUID.
  char *users;                   ///< Users for role.
} modify_role_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_role_data_reset (modify_role_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->role_id);
  free (data->users);

  memset (data, 0, sizeof (modify_role_data_t));
}

/**
 * @brief Command data for the modify_scanner command.
 */
typedef struct
{
  char *comment;            ///< Comment.
  char *name;               ///< Name of scanner.
  char *host;               ///< Host of scanner.
  char *port;               ///< Port of scanner.
  char *type;               ///< Type of scanner.
  char *scanner_id;         ///< scanner UUID.
  char *ca_pub;             ///< CA Certificate of scanner.
  char *key_pub;            ///< Certificate of scanner.
  char *key_priv;           ///< Private key of scanner.
} modify_scanner_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_scanner_data_reset (modify_scanner_data_t *data)
{
  g_free (data->comment);
  g_free (data->name);
  g_free (data->host);
  g_free (data->port);
  g_free (data->type);
  g_free (data->scanner_id);
  free (data->ca_pub);
  free (data->key_pub);
  free (data->key_priv);

  memset (data, 0, sizeof (modify_scanner_data_t));
}

/**
 * @brief Command data for the modify_schedule command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of schedule.
  char *schedule_id;             ///< Schedule UUID.
  char *first_time_day_of_month; ///< Day of month schedule must first run.
  char *first_time_hour;         ///< Hour schedule must first run.
  char *first_time_minute;       ///< Minute schedule must first run.
  char *first_time_month;        ///< Month schedule must first run.
  char *first_time_year;         ///< Year schedule must first run.
  char *period;                  ///< Period of schedule (how often it runs).
  char *period_unit;             ///< Unit of period: "hour", "day", "week", ....
  char *duration;                ///< Duration of schedule (how long it runs for).
  char *duration_unit;           ///< Unit of duration: "hour", "day", "week", ....
  char *timezone;                ///< Timezone.
} modify_schedule_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_schedule_data_reset (modify_schedule_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->schedule_id);
  free (data->first_time_day_of_month);
  free (data->first_time_hour);
  free (data->first_time_minute);
  free (data->first_time_month);
  free (data->first_time_year);
  free (data->period);
  free (data->period_unit);
  free (data->duration);
  free (data->duration_unit);
  free (data->timezone);

  memset (data, 0, sizeof (modify_schedule_data_t));
}

/**
 * @brief Command data for the modify_slave command.
 */
typedef struct
{
  char *comment;                 ///< Comment.
  char *name;                    ///< Name of slave.
  char *slave_id;                ///< Slave UUID.
  char *host;                    ///< Slave hostname.
  char *port;                    ///< Slave port.
  char *login;                   ///< Slave login.
  char *password;                ///< Slave password.
} modify_slave_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_slave_data_reset (modify_slave_data_t *data)
{
  free (data->comment);
  free (data->name);
  free (data->slave_id);
  free (data->host);
  free (data->port);
  free (data->login);
  free (data->password);

  memset (data, 0, sizeof (modify_slave_data_t));
}

/**
 * @brief Command data for the modify_tag command.
 */
typedef struct
{
  char *tag_id;         ///< UUID of the tag.
  char *active;         ///< Whether the tag is active.
  char *resource_id;    ///< ID of the resource to which to attach the tag.
  char *resource_type;  ///< Type of the resource to which to attach the tag.
  char *comment;        ///< Comment to add to the tag.
  char *name;           ///< Name of the tag.
  char *value;          ///< Value of the tag.
  int  resource_count;  ///< Number of attach tags.
} modify_tag_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_tag_data_reset (modify_tag_data_t *data)
{
  free (data->tag_id);
  free (data->active);
  free (data->resource_type);
  free (data->resource_id);
  free (data->comment);
  free (data->name);
  free (data->value);

  memset (data, 0, sizeof (modify_tag_data_t));
}

/**
 * @brief Command data for the modify_setting command.
 */
typedef struct
{
  char *name;           ///< Name.
  char *setting_id;     ///< Setting.
  char *value;          ///< Value.
} modify_setting_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_setting_data_reset (modify_setting_data_t *data)
{
  free (data->name);
  free (data->setting_id);
  free (data->value);

  memset (data, 0, sizeof (modify_setting_data_t));
}

/**
 * @brief Command data for the modify_target command.
 */
typedef struct
{
  char *alive_tests;             ///< Alive tests.
  char *comment;                 ///< Comment.
  char *exclude_hosts;           ///< Hosts to exclude from set.
  char *reverse_lookup_only;     ///< Boolean. Whether to consider only hosts that reverse lookup.
  char *reverse_lookup_unify;    ///< Boolean. Whether to unify based on reverse lookup.
  char *hosts;                   ///< Hosts for target.
  char *name;                    ///< Name of target.
  char *port_list_id;            ///< Port list for target.
  char *ssh_lsc_credential_id;   ///< SSH LSC credential for target.
  char *ssh_port;                ///< Port for SSH LSC.
  char *smb_lsc_credential_id;   ///< SMB LSC credential for target.
  char *esxi_lsc_credential_id;  ///< ESXi LSC credential for target.
  char *target_id;               ///< Target UUID.
} modify_target_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_target_data_reset (modify_target_data_t *data)
{
  free (data->alive_tests);
  free (data->exclude_hosts);
  free (data->reverse_lookup_only);
  free (data->reverse_lookup_unify);
  free (data->comment);
  free (data->hosts);
  free (data->name);
  free (data->port_list_id);
  free (data->ssh_lsc_credential_id);
  free (data->ssh_port);
  free (data->smb_lsc_credential_id);
  free (data->target_id);

  memset (data, 0, sizeof (modify_target_data_t));
}

/**
 * @brief Command data for the modify_task command.
 */
typedef struct
{
  char *action;        ///< What to do to file: "update" or "remove".
  char *alterable;     ///< Boolean. Whether the task is alterable.
  char *comment;       ///< Comment.
  char *hosts_ordering; ///< Order for scanning of target hosts.
  char *scanner_id;    ///< ID of new scanner for task.
  char *config_id;     ///< ID of new config for task.
  array_t *alerts;     ///< IDs of new alerts for task.
  char *file;          ///< File to attach to task.
  char *file_name;     ///< Name of file to attach to task.
  array_t *groups;     ///< IDs of new groups for task.
  char *name;          ///< New name for task.
  char *observers;     ///< Space separated list of observer user names.
  name_value_t *preference;  ///< Current preference.
  array_t *preferences;   ///< Preferences.
  char *schedule_id;   ///< ID of new schedule for task.
  char *schedule_periods; ///< Number of periods the schedule must run for.
  char *slave_id;      ///< ID of new slave for task.
  char *target_id;     ///< ID of new target for task.
  char *task_id;       ///< ID of task to modify.
} modify_task_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_task_data_reset (modify_task_data_t *data)
{
  free (data->action);
  free (data->alterable);
  array_free (data->alerts);
  array_free (data->groups);
  free (data->comment);
  free (data->hosts_ordering);
  free (data->scanner_id);
  free (data->config_id);
  free (data->file);
  free (data->file_name);
  free (data->name);
  free (data->observers);
  if (data->preferences)
    {
      guint index = data->preferences->len;
      while (index--)
        {
          name_value_t *pair;
          pair = (name_value_t*) g_ptr_array_index (data->preferences, index);
          if (pair)
            {
              g_free (pair->name);
              g_free (pair->value);
            }
        }
    }
  array_free (data->preferences);
  free (data->schedule_id);
  free (data->schedule_periods);
  free (data->slave_id);
  free (data->target_id);
  free (data->task_id);

  memset (data, 0, sizeof (modify_task_data_t));
}

/**
 * @brief Command data for the modify_note command.
 */
typedef struct
{
  char *active;       ///< Whether the note is active.
  char *hosts;        ///< Hosts to which to limit override.
  char *note_id;      ///< ID of note to modify.
  char *nvt_oid;      ///< NVT to which to limit override.
  char *port;         ///< Port to which to limit override.
  char *result_id;    ///< ID of result to which to limit override.
  char *severity;     ///< Severity score to which to limit note.
  char *task_id;      ///< ID of task to which to limit override.
  char *text;         ///< Text of override.
  char *threat;       ///< Threat to which to limit override.
} modify_note_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_note_data_reset (modify_note_data_t *data)
{
  free (data->active);
  free (data->hosts);
  free (data->note_id);
  free (data->nvt_oid);
  free (data->port);
  free (data->result_id);
  free (data->severity);
  free (data->task_id);
  free (data->text);
  free (data->threat);

  memset (data, 0, sizeof (modify_note_data_t));
}

/**
 * @brief Command data for the modify_override command.
 */
typedef struct
{
  char *active;       ///< Whether the override is active.
  char *hosts;        ///< Hosts to which to limit override.
  char *new_severity; ///< New severity score of overridden results.
  char *new_threat;   ///< New threat value of overridden results.
  char *nvt_oid;      ///< NVT to which to limit override.
  char *override_id;  ///< ID of override to modify.
  char *port;         ///< Port to which to limit override.
  char *result_id;    ///< ID of result to which to limit override.
  char *severity;     ///< Severity score of results to override.
  char *task_id;      ///< ID of task to which to limit override.
  char *text;         ///< Text of override.
  char *threat;       ///< Threat to which to limit override.
} modify_override_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
modify_override_data_reset (modify_override_data_t *data)
{
  free (data->active);
  free (data->hosts);
  free (data->new_severity);
  free (data->new_threat);
  free (data->nvt_oid);
  free (data->override_id);
  free (data->port);
  free (data->result_id);
  free (data->severity);
  free (data->task_id);
  free (data->text);
  free (data->threat);

  memset (data, 0, sizeof (modify_override_data_t));
}

/**
 * @brief Command data for the modify_user command.
 */
typedef struct
{
  array_t *groups;        ///< IDs of groups.
  int sort_order;
  gchar *hosts;
  int hosts_allow;
  char *ifaces;
  int ifaces_allow;
  gboolean modify_password;
  gchar *name;
  gchar *new_name;
  gchar *password;
  array_t *roles;         ///< IDs of roles.
  array_t *sources;
  gchar *current_source;
  gchar *user_id;
} modify_user_data_t;

/**
 * @brief Reset MODIFY_USER data.
 */
static void
modify_user_data_reset (modify_user_data_t * data)
{
  array_free (data->groups);
  g_free (data->name);
  g_free (data->new_name);
  g_free (data->user_id);
  g_free (data->password);
  g_free (data->hosts);
  g_free (data->ifaces);
  array_free (data->roles);
  if (data->sources)
    {
      array_free (data->sources);
    }
  g_free (data->current_source);
  memset (data, 0, sizeof (modify_user_data_t));
}

/**
 * @brief Command data for the restore command.
 */
typedef struct
{
  char *id;   ///< ID of resource to restore.
} restore_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
restore_data_reset (restore_data_t *data)
{
  free (data->id);

  memset (data, 0, sizeof (restore_data_t));
}

/**
 * @brief Command data for the resume_task command.
 */
typedef struct
{
  char *task_id;   ///< ID of task to resume.
} resume_task_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
resume_task_data_reset (resume_task_data_t *data)
{
  free (data->task_id);

  memset (data, 0, sizeof (resume_task_data_t));
}

/**
 * @brief Command data for the start_task command.
 */
typedef struct
{
  char *task_id;   ///< ID of task to start.
} start_task_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
start_task_data_reset (start_task_data_t *data)
{
  free (data->task_id);

  memset (data, 0, sizeof (start_task_data_t));
}

/**
 * @brief Command data for the stop_task command.
 */
typedef struct
{
  char *task_id;   ///< ID of task to stop.
} stop_task_data_t;

/**
 * @brief Free members of a stop_task_data_t and set them to NULL.
 */
/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
stop_task_data_reset (stop_task_data_t *data)
{
  free (data->task_id);

  memset (data, 0, sizeof (stop_task_data_t));
}

/**
 * @brief Command data for the test_alert command.
 */
typedef struct
{
  char *alert_id;   ///< ID of alert to test.
} test_alert_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
test_alert_data_reset (test_alert_data_t *data)
{
  free (data->alert_id);

  memset (data, 0, sizeof (test_alert_data_t));
}

/**
 * @brief Command data for the verify_agent command.
 */
typedef struct
{
  char *agent_id;   ///< ID of agent to verify.
} verify_agent_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
verify_agent_data_reset (verify_agent_data_t *data)
{
  free (data->agent_id);

  memset (data, 0, sizeof (verify_agent_data_t));
}

/**
 * @brief Command data for the verify_report_format command.
 */
typedef struct
{
  char *report_format_id;   ///< ID of report format to verify.
} verify_report_format_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
verify_report_format_data_reset (verify_report_format_data_t *data)
{
  free (data->report_format_id);

  memset (data, 0, sizeof (verify_report_format_data_t));
}

/**
 * @brief Command data for the verify_scanner command.
 */
typedef struct
{
  char *scanner_id;   ///< ID of scanner to verify.
} verify_scanner_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
verify_scanner_data_reset (verify_scanner_data_t *data)
{
  g_free (data->scanner_id);

  memset (data, 0, sizeof (verify_scanner_data_t));
}

/**
 * @brief Command data for the wizard command.
 */
typedef struct
{
  char *mode;          ///< Mode to run the wizard in.
  char *name;          ///< Name of the wizard.
  name_value_t *param; ///< Current param.
  array_t *params;     ///< Parameters.
  char *read_only;     ///< Read only flag.
} run_wizard_data_t;

/**
 * @brief Reset command data.
 *
 * @param[in]  data  Command data.
 */
static void
run_wizard_data_reset (run_wizard_data_t *data)
{
  free (data->mode);
  free (data->name);
  free (data->read_only);
  if (data->params)
    {
      guint index = data->params->len;
      while (index--)
        {
          name_value_t *pair;
          pair = (name_value_t*) g_ptr_array_index (data->params, index);
          if (pair)
            {
              g_free (pair->name);
              g_free (pair->value);
            }
        }
    }
  array_free (data->params);

  memset (data, 0, sizeof (run_wizard_data_t));
}

/**
 * @brief Command data, as passed between OMP parser callbacks.
 */
typedef union
{
  create_agent_data_t create_agent;                   ///< create_agent
  create_config_data_t create_config;                 ///< create_config
  create_alert_data_t create_alert;                   ///< create_alert
  create_filter_data_t create_filter;                 ///< create_filter
  create_group_data_t create_group;                   ///< create_group
  create_lsc_credential_data_t create_lsc_credential; ///< create_lsc_credential
  create_note_data_t create_note;                     ///< create_note
  create_override_data_t create_override;             ///< create_override
  create_permission_data_t create_permission;         ///< create_permission
  create_port_list_data_t create_port_list;           ///< create_port_list
  create_port_range_data_t create_port_range;         ///< create_port_range
  create_report_data_t create_report;                 ///< create_report
  create_report_format_data_t create_report_format;   ///< create_report_format
  create_role_data_t create_role;                     ///< create_role
  create_scanner_data_t create_scanner;               ///< create_scanner
  create_schedule_data_t create_schedule;             ///< create_schedule
  create_slave_data_t create_slave;                   ///< create_slave
  create_tag_data_t create_tag;                       ///< create_tag
  create_target_data_t create_target;                 ///< create_target
  create_task_data_t create_task;                     ///< create_task
  create_user_data_t create_user;                     ///< create_user
  delete_agent_data_t delete_agent;                   ///< delete_agent
  delete_config_data_t delete_config;                 ///< delete_config
  delete_alert_data_t delete_alert;                   ///< delete_alert
  delete_filter_data_t delete_filter;                 ///< delete_filter
  delete_group_data_t delete_group;                   ///< delete_group
  delete_lsc_credential_data_t delete_lsc_credential; ///< delete_lsc_credential
  delete_note_data_t delete_note;                     ///< delete_note
  delete_override_data_t delete_override;             ///< delete_override
  delete_permission_data_t delete_permission;         ///< delete_permission
  delete_port_list_data_t delete_port_list;           ///< delete_port_list
  delete_port_range_data_t delete_port_range;         ///< delete_port_range
  delete_report_data_t delete_report;                 ///< delete_report
  delete_report_format_data_t delete_report_format;   ///< delete_report_format
  delete_role_data_t delete_role;                     ///< delete_role
  delete_scanner_data_t delete_scanner;               ///< delete_scanner
  delete_schedule_data_t delete_schedule;             ///< delete_schedule
  delete_slave_data_t delete_slave;                   ///< delete_slave
  delete_tag_data_t delete_tag;                       ///< delete_tag
  delete_target_data_t delete_target;                 ///< delete_target
  delete_task_data_t delete_task;                     ///< delete_task
  delete_user_data_t delete_user;                     ///< delete_user
  get_agents_data_t get_agents;                       ///< get_agents
  get_aggregates_data_t get_aggregates;               ///< get_aggregates
  get_configs_data_t get_configs;                     ///< get_configs
  get_alerts_data_t get_alerts;                       ///< get_alerts
  get_filters_data_t get_filters;                     ///< get_filters
  get_groups_data_t get_groups;                       ///< get_groups
  get_info_data_t get_info;                           ///< get_info
  get_lsc_credentials_data_t get_lsc_credentials;     ///< get_lsc_credentials
  get_notes_data_t get_notes;                         ///< get_notes
  get_nvts_data_t get_nvts;                           ///< get_nvts
  get_nvt_families_data_t get_nvt_families;           ///< get_nvt_families
  get_overrides_data_t get_overrides;                 ///< get_overrides
  get_permissions_data_t get_permissions;             ///< get_permissions
  get_port_lists_data_t get_port_lists;               ///< get_port_lists
  get_preferences_data_t get_preferences;             ///< get_preferences
  get_reports_data_t get_reports;                     ///< get_reports
  get_report_formats_data_t get_report_formats;       ///< get_report_formats
  get_results_data_t get_results;                     ///< get_results
  get_roles_data_t get_roles;                         ///< get_roles
  get_schedules_data_t get_schedules;                 ///< get_schedules
  get_scanners_data_t get_scanners;                   ///< get_scanners
  get_settings_data_t get_settings;                   ///< get_settings
  get_slaves_data_t get_slaves;                       ///< get_slaves
  get_system_reports_data_t get_system_reports;       ///< get_system_reports
  get_tags_data_t get_tags;                           ///< get_tags
  get_targets_data_t get_targets;                     ///< get_targets
  get_tasks_data_t get_tasks;                         ///< get_tasks
  get_users_data_t get_users;                         ///< get_users
  help_data_t help;                                   ///< help
  modify_agent_data_t modify_agent;                   ///< modify_agent
  modify_alert_data_t modify_alert;                   ///< modify_alert
  modify_auth_data_t modify_auth;                     ///< modify_auth
  modify_config_data_t modify_config;                 ///< modify_config
  modify_filter_data_t modify_filter;                 ///< modify_filter
  modify_group_data_t modify_group;                   ///< modify_group
  modify_lsc_credential_data_t modify_lsc_credential; ///< modify_lsc_credential
  modify_permission_data_t modify_permission;         ///< modify_permission
  modify_port_list_data_t modify_port_list;           ///< modify_port_list
  modify_report_data_t modify_report;                 ///< modify_report
  modify_report_format_data_t modify_report_format;   ///< modify_report_format
  modify_role_data_t modify_role;                     ///< modify_role
  modify_scanner_data_t modify_scanner;               ///< modify_scanner
  modify_schedule_data_t modify_schedule;             ///< modify_schedule
  modify_setting_data_t modify_setting;               ///< modify_setting
  modify_slave_data_t modify_slave;                   ///< modify_slave
  modify_tag_data_t modify_tag;                       ///< modify_tag
  modify_target_data_t modify_target;                 ///< modify_target
  modify_task_data_t modify_task;                     ///< modify_task
  modify_user_data_t modify_user;                     ///< modify_user
  restore_data_t restore;                             ///< restore
  resume_task_data_t resume_task;                     ///< resume_task
  start_task_data_t start_task;                       ///< start_task
  stop_task_data_t stop_task;                         ///< stop_task
  test_alert_data_t test_alert;                       ///< test_alert
  verify_agent_data_t verify_agent;                   ///< verify_agent
  verify_report_format_data_t verify_report_format;   ///< verify_report_format
  verify_scanner_data_t verify_scanner;               ///< verify_scanner
  run_wizard_data_t wizard;                           ///< run_wizard
} command_data_t;

/**
 * @brief Initialise command data.
 */
static void
command_data_init (command_data_t *data)
{
  memset (data, 0, sizeof (command_data_t));
}


/* Global variables. */

/**
 * @brief Parser callback data.
 */
command_data_t command_data;

/**
 * @brief Parser callback data for CREATE_AGENT.
 */
create_agent_data_t *create_agent_data
 = (create_agent_data_t*) &(command_data.create_agent);

/**
 * @brief Parser callback data for CREATE_CONFIG.
 */
create_config_data_t *create_config_data
 = (create_config_data_t*) &(command_data.create_config);

/**
 * @brief Parser callback data for CREATE_ALERT.
 */
create_alert_data_t *create_alert_data
 = (create_alert_data_t*) &(command_data.create_alert);

/**
 * @brief Parser callback data for CREATE_FILTER.
 */
create_filter_data_t *create_filter_data
 = (create_filter_data_t*) &(command_data.create_filter);

/**
 * @brief Parser callback data for CREATE_GROUP.
 */
create_group_data_t *create_group_data
 = (create_group_data_t*) &(command_data.create_group);

/**
 * @brief Parser callback data for CREATE_LSC_CREDENTIAL.
 */
create_lsc_credential_data_t *create_lsc_credential_data
 = (create_lsc_credential_data_t*) &(command_data.create_lsc_credential);

/**
 * @brief Parser callback data for CREATE_NOTE.
 */
create_note_data_t *create_note_data
 = (create_note_data_t*) &(command_data.create_note);

/**
 * @brief Parser callback data for CREATE_OVERRIDE.
 */
create_override_data_t *create_override_data
 = (create_override_data_t*) &(command_data.create_override);

/**
 * @brief Parser callback data for CREATE_PERMISSION.
 */
create_permission_data_t *create_permission_data
 = (create_permission_data_t*) &(command_data.create_permission);

/**
 * @brief Parser callback data for CREATE_PORT_LIST.
 */
create_port_list_data_t *create_port_list_data
 = (create_port_list_data_t*) &(command_data.create_port_list);

/**
 * @brief Parser callback data for CREATE_PORT_RANGE.
 */
create_port_range_data_t *create_port_range_data
 = (create_port_range_data_t*) &(command_data.create_port_range);

/**
 * @brief Parser callback data for CREATE_ROLE.
 */
create_role_data_t *create_role_data
 = (create_role_data_t*) &(command_data.create_role);

/**
 * @brief Parser callback data for CREATE_REPORT.
 */
create_report_data_t *create_report_data
 = (create_report_data_t*) &(command_data.create_report);

/**
 * @brief Parser callback data for CREATE_REPORT_FORMAT.
 */
create_report_format_data_t *create_report_format_data
 = (create_report_format_data_t*) &(command_data.create_report_format);

/**
 * @brief Parser callback data for CREATE_SCANNER.
 */
create_scanner_data_t *create_scanner_data
 = (create_scanner_data_t*) &(command_data.create_scanner);

/**
 * @brief Parser callback data for CREATE_SCHEDULE.
 */
create_schedule_data_t *create_schedule_data
 = (create_schedule_data_t*) &(command_data.create_schedule);

/**
 * @brief Parser callback data for CREATE_SLAVE.
 */
create_slave_data_t *create_slave_data
 = (create_slave_data_t*) &(command_data.create_slave);

/**
 * @brief Parser callback data for CREATE_TAG.
 */
create_tag_data_t *create_tag_data
 = (create_tag_data_t*) &(command_data.create_tag);

/**
 * @brief Parser callback data for CREATE_TARGET.
 */
create_target_data_t *create_target_data
 = (create_target_data_t*) &(command_data.create_target);

/**
 * @brief Parser callback data for CREATE_TASK.
 */
create_task_data_t *create_task_data
 = (create_task_data_t*) &(command_data.create_task);

/**
 * @brief Parser callback data for CREATE_USER.
 */
create_user_data_t *create_user_data
 = &(command_data.create_user);

/**
 * @brief Parser callback data for DELETE_AGENT.
 */
delete_agent_data_t *delete_agent_data
 = (delete_agent_data_t*) &(command_data.delete_agent);

/**
 * @brief Parser callback data for DELETE_CONFIG.
 */
delete_config_data_t *delete_config_data
 = (delete_config_data_t*) &(command_data.delete_config);

/**
 * @brief Parser callback data for DELETE_ALERT.
 */
delete_alert_data_t *delete_alert_data
 = (delete_alert_data_t*) &(command_data.delete_alert);

/**
 * @brief Parser callback data for DELETE_FILTER.
 */
delete_filter_data_t *delete_filter_data
 = (delete_filter_data_t*) &(command_data.delete_filter);

/**
 * @brief Parser callback data for DELETE_GROUP.
 */
delete_group_data_t *delete_group_data
 = (delete_group_data_t*) &(command_data.delete_group);

/**
 * @brief Parser callback data for DELETE_LSC_CREDENTIAL.
 */
delete_lsc_credential_data_t *delete_lsc_credential_data
 = (delete_lsc_credential_data_t*) &(command_data.delete_lsc_credential);

/**
 * @brief Parser callback data for DELETE_NOTE.
 */
delete_note_data_t *delete_note_data
 = (delete_note_data_t*) &(command_data.delete_note);

/**
 * @brief Parser callback data for DELETE_OVERRIDE.
 */
delete_override_data_t *delete_override_data
 = (delete_override_data_t*) &(command_data.delete_override);

/**
 * @brief Parser callback data for DELETE_PERMISSION.
 */
delete_permission_data_t *delete_permission_data
 = (delete_permission_data_t*) &(command_data.delete_permission);

/**
 * @brief Parser callback data for DELETE_PORT_LIST.
 */
delete_port_list_data_t *delete_port_list_data
 = (delete_port_list_data_t*) &(command_data.delete_port_list);

/**
 * @brief Parser callback data for DELETE_PORT_RANGE.
 */
delete_port_range_data_t *delete_port_range_data
 = (delete_port_range_data_t*) &(command_data.delete_port_range);

/**
 * @brief Parser callback data for DELETE_REPORT.
 */
delete_report_data_t *delete_report_data
 = (delete_report_data_t*) &(command_data.delete_report);

/**
 * @brief Parser callback data for DELETE_REPORT_FORMAT.
 */
delete_report_format_data_t *delete_report_format_data
 = (delete_report_format_data_t*) &(command_data.delete_report_format);

/**
 * @brief Parser callback data for DELETE_ROLE.
 */
delete_role_data_t *delete_role_data
 = (delete_role_data_t*) &(command_data.delete_role);

/**
 * @brief Parser callback data for DELETE_SCANNER.
 */
delete_scanner_data_t *delete_scanner_data
 = (delete_scanner_data_t*) &(command_data.delete_scanner);

/**
 * @brief Parser callback data for DELETE_SCHEDULE.
 */
delete_schedule_data_t *delete_schedule_data
 = (delete_schedule_data_t*) &(command_data.delete_schedule);

/**
 * @brief Parser callback data for DELETE_SLAVE.
 */
delete_slave_data_t *delete_slave_data
 = (delete_slave_data_t*) &(command_data.delete_slave);

/**
 * @brief Parser callback data for DELETE_TAG.
 */
delete_tag_data_t *delete_tag_data
 = (delete_tag_data_t*) &(command_data.delete_tag);

/**
 * @brief Parser callback data for DELETE_TARGET.
 */
delete_target_data_t *delete_target_data
 = (delete_target_data_t*) &(command_data.delete_target);

/**
 * @brief Parser callback data for DELETE_TASK.
 */
delete_task_data_t *delete_task_data
 = (delete_task_data_t*) &(command_data.delete_task);

/**
 * @brief Parser callback data for DELETE_USER.
 */
delete_user_data_t *delete_user_data
 = (delete_user_data_t*) &(command_data.delete_user);

/**
 * @brief Parser callback data for GET_AGENTS.
 */
get_agents_data_t *get_agents_data
 = &(command_data.get_agents);

/**
 * @brief Parser callback data for GET_AGGREGATES.
 */
get_aggregates_data_t *get_aggregates_data
 = &(command_data.get_aggregates);

/**
 * @brief Parser callback data for GET_CONFIGS.
 */
get_configs_data_t *get_configs_data
 = &(command_data.get_configs);

/**
 * @brief Parser callback data for GET_ALERTS.
 */
get_alerts_data_t *get_alerts_data
 = &(command_data.get_alerts);

/**
 * @brief Parser callback data for GET_FILTERS.
 */
get_filters_data_t *get_filters_data
 = &(command_data.get_filters);

/**
 * @brief Parser callback data for GET_GROUPS.
 */
get_groups_data_t *get_groups_data
 = &(command_data.get_groups);

/**
 * @brief Parser callback data for GET_INFO.
 */
get_info_data_t *get_info_data
 = &(command_data.get_info);

/**
 * @brief Parser callback data for GET_LSC_CREDENTIALS.
 */
get_lsc_credentials_data_t *get_lsc_credentials_data
 = &(command_data.get_lsc_credentials);

/**
 * @brief Parser callback data for GET_NOTES.
 */
get_notes_data_t *get_notes_data
 = &(command_data.get_notes);

/**
 * @brief Parser callback data for GET_NVTS.
 */
get_nvts_data_t *get_nvts_data
 = &(command_data.get_nvts);

/**
 * @brief Parser callback data for GET_NVT_FAMILIES.
 */
get_nvt_families_data_t *get_nvt_families_data
 = &(command_data.get_nvt_families);

/**
 * @brief Parser callback data for GET_OVERRIDES.
 */
get_overrides_data_t *get_overrides_data
 = &(command_data.get_overrides);

/**
 * @brief Parser callback data for GET_PERMISSIONS.
 */
get_permissions_data_t *get_permissions_data
 = &(command_data.get_permissions);

/**
 * @brief Parser callback data for GET_PORT_LISTS.
 */
get_port_lists_data_t *get_port_lists_data
 = &(command_data.get_port_lists);

/**
 * @brief Parser callback data for GET_PREFERENCES.
 */
get_preferences_data_t *get_preferences_data
 = &(command_data.get_preferences);

/**
 * @brief Parser callback data for GET_REPORTS.
 */
get_reports_data_t *get_reports_data
 = &(command_data.get_reports);

/**
 * @brief Parser callback data for GET_REPORT_FORMATS.
 */
get_report_formats_data_t *get_report_formats_data
 = &(command_data.get_report_formats);

/**
 * @brief Parser callback data for GET_RESULTS.
 */
get_results_data_t *get_results_data
 = &(command_data.get_results);

/**
 * @brief Parser callback data for GET_ROLES.
 */
get_roles_data_t *get_roles_data
 = &(command_data.get_roles);

/**
 * @brief Parser callback data for GET_scannerS.
 */
get_scanners_data_t *get_scanners_data
 = &(command_data.get_scanners);

/**
 * @brief Parser callback data for GET_SCHEDULES.
 */
get_schedules_data_t *get_schedules_data
 = &(command_data.get_schedules);

/**
 * @brief Parser callback data for GET_SETTINGS.
 */
get_settings_data_t *get_settings_data
 = &(command_data.get_settings);

/**
 * @brief Parser callback data for GET_SLAVES.
 */
get_slaves_data_t *get_slaves_data
 = &(command_data.get_slaves);

/**
 * @brief Parser callback data for GET_SYSTEM_REPORTS.
 */
get_system_reports_data_t *get_system_reports_data
 = &(command_data.get_system_reports);

/**
 * @brief Parser callback data for GET_TAGS.
 */
get_tags_data_t *get_tags_data
 = &(command_data.get_tags);

/**
 * @brief Parser callback data for GET_TARGETS.
 */
get_targets_data_t *get_targets_data
 = &(command_data.get_targets);

/**
 * @brief Parser callback data for GET_TASKS.
 */
get_tasks_data_t *get_tasks_data
 = &(command_data.get_tasks);

/**
 * @brief Parser callback data for GET_USERS.
 */
get_users_data_t *get_users_data
 = &(command_data.get_users);

/**
 * @brief Parser callback data for HELP.
 */
help_data_t *help_data
 = &(command_data.help);

/**
 * @brief Parser callback data for CREATE_CONFIG (import).
 */
import_config_data_t *import_config_data
 = (import_config_data_t*) &(command_data.create_config.import);

/**
 * @brief Parser callback data for MODIFY_CONFIG.
 */
modify_config_data_t *modify_config_data
 = &(command_data.modify_config);

/**
 * @brief Parser callback data for MODIFY_AGENT.
 */
modify_agent_data_t *modify_agent_data
 = &(command_data.modify_agent);

/**
 * @brief Parser callback data for MODIFY_ALERT.
 */
modify_alert_data_t *modify_alert_data
 = &(command_data.modify_alert);

/**
 * @brief Parser callback data for MODIFY_AUTH.
 */
modify_auth_data_t *modify_auth_data
 = &(command_data.modify_auth);

/**
 * @brief Parser callback data for MODIFY_FILTER.
 */
modify_filter_data_t *modify_filter_data
 = &(command_data.modify_filter);

/**
 * @brief Parser callback data for MODIFY_GROUP.
 */
modify_group_data_t *modify_group_data
 = &(command_data.modify_group);

/**
 * @brief Parser callback data for MODIFY_LSC_CREDENTIAL.
 */
modify_lsc_credential_data_t *modify_lsc_credential_data
 = &(command_data.modify_lsc_credential);

/**
 * @brief Parser callback data for MODIFY_NOTE.
 */
modify_note_data_t *modify_note_data
 = (modify_note_data_t*) &(command_data.create_note);

/**
 * @brief Parser callback data for MODIFY_OVERRIDE.
 */
modify_override_data_t *modify_override_data
 = (modify_override_data_t*) &(command_data.create_override);

/**
 * @brief Parser callback data for MODIFY_PERMISSION.
 */
modify_permission_data_t *modify_permission_data
 = &(command_data.modify_permission);

/**
 * @brief Parser callback data for MODIFY_PORT_LIST.
 */
modify_port_list_data_t *modify_port_list_data
 = &(command_data.modify_port_list);

/**
 * @brief Parser callback data for MODIFY_REPORT.
 */
modify_report_data_t *modify_report_data
 = &(command_data.modify_report);

/**
 * @brief Parser callback data for MODIFY_REPORT_FORMAT.
 */
modify_report_format_data_t *modify_report_format_data
 = &(command_data.modify_report_format);

/**
 * @brief Parser callback data for MODIFY_ROLE.
 */
modify_role_data_t *modify_role_data
 = &(command_data.modify_role);

/**
 * @brief Parser callback data for MODIFY_SCANNER.
 */
modify_scanner_data_t *modify_scanner_data
 = &(command_data.modify_scanner);

/**
 * @brief Parser callback data for MODIFY_SCHEDULE.
 */
modify_schedule_data_t *modify_schedule_data
 = &(command_data.modify_schedule);

/**
 * @brief Parser callback data for MODIFY_SETTING.
 */
modify_setting_data_t *modify_setting_data
 = &(command_data.modify_setting);

/**
 * @brief Parser callback data for MODIFY_SLAVE.
 */
modify_slave_data_t *modify_slave_data
 = &(command_data.modify_slave);

/**
 * @brief Parser callback data for MODIFY_TAG.
 */
modify_tag_data_t *modify_tag_data
 = (modify_tag_data_t*) &(command_data.modify_tag);

/**
 * @brief Parser callback data for MODIFY_TARGET.
 */
modify_target_data_t *modify_target_data
 = &(command_data.modify_target);

/**
 * @brief Parser callback data for MODIFY_TASK.
 */
modify_task_data_t *modify_task_data
 = &(command_data.modify_task);

/**
 * @brief Parser callback data for MODIFY_USER.
 */
modify_user_data_t *modify_user_data = &(command_data.modify_user);

/**
 * @brief Parser callback data for RESTORE.
 */
restore_data_t *restore_data
 = (restore_data_t*) &(command_data.restore);

/**
 * @brief Parser callback data for RESUME_TASK.
 */
resume_task_data_t *resume_task_data
 = (resume_task_data_t*) &(command_data.resume_task);

/**
 * @brief Parser callback data for START_TASK.
 */
start_task_data_t *start_task_data
 = (start_task_data_t*) &(command_data.start_task);

/**
 * @brief Parser callback data for STOP_TASK.
 */
stop_task_data_t *stop_task_data
 = (stop_task_data_t*) &(command_data.stop_task);

/**
 * @brief Parser callback data for TEST_ALERT.
 */
test_alert_data_t *test_alert_data
 = (test_alert_data_t*) &(command_data.test_alert);

/**
 * @brief Parser callback data for VERIFY_AGENT.
 */
verify_agent_data_t *verify_agent_data
 = (verify_agent_data_t*) &(command_data.verify_agent);

/**
 * @brief Parser callback data for VERIFY_REPORT_FORMAT.
 */
verify_report_format_data_t *verify_report_format_data
 = (verify_report_format_data_t*) &(command_data.verify_report_format);

/**
 * @brief Parser callback data for VERIFY_SCANNER.
 */
verify_scanner_data_t *verify_scanner_data
 = (verify_scanner_data_t*) &(command_data.verify_scanner);

/**
 * @brief Parser callback data for WIZARD.
 */
run_wizard_data_t *run_wizard_data
 = (run_wizard_data_t*) &(command_data.wizard);

/**
 * @brief Hack for returning forked process status from the callbacks.
 */
int current_error;

/**
 * @brief Hack for returning fork status to caller.
 */
int forked;

/**
 * @brief Buffer of output to the client.
 */
char to_client[TO_CLIENT_BUFFER_SIZE];

/**
 * @brief The start of the data in the \ref to_client buffer.
 */
buffer_size_t to_client_start = 0;
/**
 * @brief The end of the data in the \ref to_client buffer.
 */
buffer_size_t to_client_end = 0;

/**
 * @brief Client input parsing context.
 */
static /*@null@*/ /*@only@*/ GMarkupParseContext*
xml_context = NULL;

/**
 * @brief Client input parser.
 */
static GMarkupParser xml_parser;

/**
 * @brief The nvt synchronization script for this daemon.
 */
static const gchar *sync_script = SBINDIR "/openvas-nvt-sync";

/**
 * @brief The scap synchronization script for this daemon.
 */
static const gchar *scap_script = SBINDIR "/openvas-scapdata-sync";

/**
 * @brief The CERT synchronization script for this daemon.
 */
static const gchar *cert_script = SBINDIR "/openvas-certdata-sync";


/* Client state. */

/**
 * @brief Possible states of the client.
 */
typedef enum
{
  CLIENT_TOP,
  CLIENT_AUTHENTIC,

  CLIENT_AUTHENTICATE,
  CLIENT_AUTHENTICATE_CREDENTIALS,
  CLIENT_AUTHENTICATE_CREDENTIALS_PASSWORD,
  CLIENT_AUTHENTICATE_CREDENTIALS_USERNAME,
  CLIENT_AUTHENTIC_COMMANDS,
  CLIENT_COMMANDS,
  CLIENT_CREATE_AGENT,
  CLIENT_CREATE_AGENT_COMMENT,
  CLIENT_CREATE_AGENT_COPY,
  CLIENT_CREATE_AGENT_HOWTO_INSTALL,
  CLIENT_CREATE_AGENT_HOWTO_USE,
  CLIENT_CREATE_AGENT_INSTALLER,
  CLIENT_CREATE_AGENT_INSTALLER_FILENAME,
  CLIENT_CREATE_AGENT_INSTALLER_SIGNATURE,
  CLIENT_CREATE_AGENT_NAME,
  CLIENT_CREATE_ALERT,
  CLIENT_CREATE_ALERT_COMMENT,
  CLIENT_CREATE_ALERT_CONDITION,
  CLIENT_CREATE_ALERT_CONDITION_DATA,
  CLIENT_CREATE_ALERT_CONDITION_DATA_NAME,
  CLIENT_CREATE_ALERT_COPY,
  CLIENT_CREATE_ALERT_EVENT,
  CLIENT_CREATE_ALERT_EVENT_DATA,
  CLIENT_CREATE_ALERT_EVENT_DATA_NAME,
  CLIENT_CREATE_ALERT_FILTER,
  CLIENT_CREATE_ALERT_METHOD,
  CLIENT_CREATE_ALERT_METHOD_DATA,
  CLIENT_CREATE_ALERT_METHOD_DATA_NAME,
  CLIENT_CREATE_ALERT_NAME,
  CLIENT_CREATE_CONFIG,
  CLIENT_CREATE_CONFIG_COMMENT,
  CLIENT_CREATE_CONFIG_COPY,
  CLIENT_CREATE_CONFIG_SCANNER,
  CLIENT_CREATE_CONFIG_NAME,
  /* get_configs_response (GCR) is used for config export.  CLIENT_C_C is
   * for CLIENT_CREATE_CONFIG. */
  CLIENT_C_C_GCR,
  CLIENT_C_C_GCR_CONFIG,
  CLIENT_C_C_GCR_CONFIG_COMMENT,
  CLIENT_C_C_GCR_CONFIG_NAME,
  CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS,
  CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR,
  CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_FAMILY_OR_NVT,
  CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_INCLUDE,
  CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_NAME,
  CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_TYPE,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_ALT,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NAME,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NVT,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NVT_NAME,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_TYPE,
  CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_VALUE,
  CLIENT_CREATE_FILTER,
  CLIENT_CREATE_FILTER_COMMENT,
  CLIENT_CREATE_FILTER_COPY,
  CLIENT_CREATE_FILTER_NAME,
  CLIENT_CREATE_FILTER_NAME_MAKE_UNIQUE,
  CLIENT_CREATE_FILTER_TERM,
  CLIENT_CREATE_FILTER_TYPE,
  CLIENT_CREATE_GROUP,
  CLIENT_CREATE_GROUP_COMMENT,
  CLIENT_CREATE_GROUP_COPY,
  CLIENT_CREATE_GROUP_NAME,
  CLIENT_CREATE_GROUP_USERS,
  CLIENT_CREATE_LSC_CREDENTIAL,
  CLIENT_CREATE_LSC_CREDENTIAL_COMMENT,
  CLIENT_CREATE_LSC_CREDENTIAL_COPY,
  CLIENT_CREATE_LSC_CREDENTIAL_KEY,
  CLIENT_CREATE_LSC_CREDENTIAL_KEY_PHRASE,
  CLIENT_CREATE_LSC_CREDENTIAL_KEY_PRIVATE,
  CLIENT_CREATE_LSC_CREDENTIAL_LOGIN,
  CLIENT_CREATE_LSC_CREDENTIAL_NAME,
  CLIENT_CREATE_LSC_CREDENTIAL_PASSWORD,
  CLIENT_CREATE_NOTE,
  CLIENT_CREATE_NOTE_ACTIVE,
  CLIENT_CREATE_NOTE_COPY,
  CLIENT_CREATE_NOTE_HOSTS,
  CLIENT_CREATE_NOTE_NVT,
  CLIENT_CREATE_NOTE_PORT,
  CLIENT_CREATE_NOTE_RESULT,
  CLIENT_CREATE_NOTE_SEVERITY,
  CLIENT_CREATE_NOTE_TASK,
  CLIENT_CREATE_NOTE_TEXT,
  CLIENT_CREATE_NOTE_THREAT,
  CLIENT_CREATE_OVERRIDE,
  CLIENT_CREATE_OVERRIDE_ACTIVE,
  CLIENT_CREATE_OVERRIDE_COPY,
  CLIENT_CREATE_OVERRIDE_HOSTS,
  CLIENT_CREATE_OVERRIDE_NEW_SEVERITY,
  CLIENT_CREATE_OVERRIDE_NEW_THREAT,
  CLIENT_CREATE_OVERRIDE_NVT,
  CLIENT_CREATE_OVERRIDE_PORT,
  CLIENT_CREATE_OVERRIDE_RESULT,
  CLIENT_CREATE_OVERRIDE_SEVERITY,
  CLIENT_CREATE_OVERRIDE_TASK,
  CLIENT_CREATE_OVERRIDE_TEXT,
  CLIENT_CREATE_OVERRIDE_THREAT,
  CLIENT_CREATE_PERMISSION,
  CLIENT_CREATE_PERMISSION_COMMENT,
  CLIENT_CREATE_PERMISSION_COPY,
  CLIENT_CREATE_PERMISSION_NAME,
  CLIENT_CREATE_PERMISSION_RESOURCE,
  CLIENT_CREATE_PERMISSION_RESOURCE_TYPE,
  CLIENT_CREATE_PERMISSION_SUBJECT,
  CLIENT_CREATE_PERMISSION_SUBJECT_TYPE,
  CLIENT_CREATE_PORT_LIST,
  CLIENT_CREATE_PORT_LIST_COMMENT,
  CLIENT_CREATE_PORT_LIST_COPY,
  CLIENT_CREATE_PORT_LIST_NAME,
  CLIENT_CREATE_PORT_LIST_PORT_RANGE,
  /* get_port_lists (GPL) is used for port lists export.  CLIENT_CPL is
   * for CLIENT_CREATE_PORT_LIST. */
  CLIENT_CPL_GPLR,
  CLIENT_CPL_GPLR_PORT_LIST,
  CLIENT_CPL_GPLR_PORT_LIST_COMMENT,
  CLIENT_CPL_GPLR_PORT_LIST_IN_USE,
  CLIENT_CPL_GPLR_PORT_LIST_NAME,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGE,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_COMMENT,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_END,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_START,
  CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_TYPE,
  CLIENT_CPL_GPLR_PORT_LIST_TARGETS,
  CLIENT_CREATE_PORT_RANGE,
  CLIENT_CREATE_PORT_RANGE_COMMENT,
  CLIENT_CREATE_PORT_RANGE_END,
  CLIENT_CREATE_PORT_RANGE_PORT_LIST,
  CLIENT_CREATE_PORT_RANGE_START,
  CLIENT_CREATE_PORT_RANGE_TYPE,
  /* CREATE_REPORT_FORMAT. */
  CLIENT_CREATE_REPORT_FORMAT,
  CLIENT_CREATE_REPORT_FORMAT_COPY,
  CLIENT_CRF_GRFR,
  CLIENT_CRF_GRFR_REPORT_FORMAT,
  CLIENT_CRF_GRFR_REPORT_FORMAT_CONTENT_TYPE,
  CLIENT_CRF_GRFR_REPORT_FORMAT_DESCRIPTION,
  CLIENT_CRF_GRFR_REPORT_FORMAT_EXTENSION,
  CLIENT_CRF_GRFR_REPORT_FORMAT_FILE,
  CLIENT_CRF_GRFR_REPORT_FORMAT_GLOBAL,
  CLIENT_CRF_GRFR_REPORT_FORMAT_NAME,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_DEFAULT,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_NAME,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS_OPTION,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE_MAX,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE_MIN,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_VALUE,
  CLIENT_CRF_GRFR_REPORT_FORMAT_PREDEFINED,
  CLIENT_CRF_GRFR_REPORT_FORMAT_SIGNATURE,
  CLIENT_CRF_GRFR_REPORT_FORMAT_SUMMARY,
  CLIENT_CRF_GRFR_REPORT_FORMAT_TRUST,
  /* CREATE_REPORT. */
  CLIENT_CREATE_REPORT,
  CLIENT_CREATE_REPORT_REPORT,
  CLIENT_CREATE_REPORT_RR,
  CLIENT_CREATE_REPORT_RR_FILTERS,
  /* RR_H is for RR_HOST because it clashes with entities like HOST_START. */
  CLIENT_CREATE_REPORT_RR_H,
  CLIENT_CREATE_REPORT_RR_HOSTS,
  CLIENT_CREATE_REPORT_RR_HOST_COUNT,
  CLIENT_CREATE_REPORT_RR_HOST_END,
  CLIENT_CREATE_REPORT_RR_HOST_END_HOST,
  CLIENT_CREATE_REPORT_RR_HOST_START,
  CLIENT_CREATE_REPORT_RR_HOST_START_HOST,
  CLIENT_CREATE_REPORT_RR_H_DETAIL,
  CLIENT_CREATE_REPORT_RR_H_DETAIL_NAME,
  CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE,
  CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_DESC,
  CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_NAME,
  CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_TYPE,
  CLIENT_CREATE_REPORT_RR_H_DETAIL_VALUE,
  CLIENT_CREATE_REPORT_RR_H_END,
  CLIENT_CREATE_REPORT_RR_H_IP,
  CLIENT_CREATE_REPORT_RR_H_START,
  CLIENT_CREATE_REPORT_RR_PORTS,
  CLIENT_CREATE_REPORT_RR_REPORT_FORMAT,
  CLIENT_CREATE_REPORT_RR_RESULTS,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_COMMENT,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_CREATION_TIME,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_DESCRIPTION,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_DETECTION,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_HOST,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_MODIFICATION_TIME,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NAME,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NOTES,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_BID,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CERT,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CERT_CERT_REF,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CVE,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CVSS_BASE,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_FAMILY,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_NAME,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_XREF,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_OWNER,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_ORIGINAL_SEVERITY,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_ORIGINAL_THREAT,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_OVERRIDES,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_PORT,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD_TYPE,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD_VALUE,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_SCAN_NVT_VERSION,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_SEVERITY,
  CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_THREAT,
  CLIENT_CREATE_REPORT_RR_RESULT_COUNT,
  CLIENT_CREATE_REPORT_RR_SCAN_END,
  CLIENT_CREATE_REPORT_RR_SCAN_RUN_STATUS,
  CLIENT_CREATE_REPORT_RR_SCAN_START,
  CLIENT_CREATE_REPORT_RR_SORT,
  CLIENT_CREATE_REPORT_RR_TASK,
  CLIENT_CREATE_REPORT_TASK,
  CLIENT_CREATE_REPORT_TASK_COMMENT,
  CLIENT_CREATE_REPORT_TASK_NAME,
  CLIENT_CREATE_ROLE,
  CLIENT_CREATE_ROLE_COMMENT,
  CLIENT_CREATE_ROLE_COPY,
  CLIENT_CREATE_ROLE_NAME,
  CLIENT_CREATE_ROLE_USERS,
  CLIENT_CREATE_SCANNER,
  CLIENT_CREATE_SCANNER_COMMENT,
  CLIENT_CREATE_SCANNER_COPY,
  CLIENT_CREATE_SCANNER_NAME,
  CLIENT_CREATE_SCANNER_HOST,
  CLIENT_CREATE_SCANNER_PORT,
  CLIENT_CREATE_SCANNER_TYPE,
  CLIENT_CREATE_SCANNER_CA_PUB,
  CLIENT_CREATE_SCANNER_KEY_PUB,
  CLIENT_CREATE_SCANNER_KEY_PRIV,
  CLIENT_CREATE_SCHEDULE,
  CLIENT_CREATE_SCHEDULE_COMMENT,
  CLIENT_CREATE_SCHEDULE_COPY,
  CLIENT_CREATE_SCHEDULE_DURATION,
  CLIENT_CREATE_SCHEDULE_DURATION_UNIT,
  CLIENT_CREATE_SCHEDULE_FIRST_TIME,
  CLIENT_CREATE_SCHEDULE_FIRST_TIME_DAY_OF_MONTH,
  CLIENT_CREATE_SCHEDULE_FIRST_TIME_HOUR,
  CLIENT_CREATE_SCHEDULE_FIRST_TIME_MINUTE,
  CLIENT_CREATE_SCHEDULE_FIRST_TIME_MONTH,
  CLIENT_CREATE_SCHEDULE_FIRST_TIME_YEAR,
  CLIENT_CREATE_SCHEDULE_NAME,
  CLIENT_CREATE_SCHEDULE_PERIOD,
  CLIENT_CREATE_SCHEDULE_PERIOD_UNIT,
  CLIENT_CREATE_SCHEDULE_TIMEZONE,
  CLIENT_CREATE_SLAVE,
  CLIENT_CREATE_SLAVE_COMMENT,
  CLIENT_CREATE_SLAVE_COPY,
  CLIENT_CREATE_SLAVE_HOST,
  CLIENT_CREATE_SLAVE_LOGIN,
  CLIENT_CREATE_SLAVE_NAME,
  CLIENT_CREATE_SLAVE_PASSWORD,
  CLIENT_CREATE_SLAVE_PORT,
  CLIENT_CREATE_TAG,
  CLIENT_CREATE_TAG_ACTIVE,
  CLIENT_CREATE_TAG_COMMENT,
  CLIENT_CREATE_TAG_COPY,
  CLIENT_CREATE_TAG_NAME,
  CLIENT_CREATE_TAG_RESOURCE,
  CLIENT_CREATE_TAG_RESOURCE_TYPE,
  CLIENT_CREATE_TAG_VALUE,
  CLIENT_CREATE_TARGET,
  CLIENT_CREATE_TARGET_ALIVE_TESTS,
  CLIENT_CREATE_TARGET_EXCLUDE_HOSTS,
  CLIENT_CREATE_TARGET_REVERSE_LOOKUP_ONLY,
  CLIENT_CREATE_TARGET_REVERSE_LOOKUP_UNIFY,
  CLIENT_CREATE_TARGET_COMMENT,
  CLIENT_CREATE_TARGET_COPY,
  CLIENT_CREATE_TARGET_ESXI_LSC_CREDENTIAL,
  CLIENT_CREATE_TARGET_HOSTS,
  CLIENT_CREATE_TARGET_NAME,
  CLIENT_CREATE_TARGET_NAME_MAKE_UNIQUE,
  CLIENT_CREATE_TARGET_PORT_LIST,
  CLIENT_CREATE_TARGET_PORT_RANGE,
  CLIENT_CREATE_TARGET_SMB_LSC_CREDENTIAL,
  CLIENT_CREATE_TARGET_SSH_LSC_CREDENTIAL,
  CLIENT_CREATE_TARGET_SSH_LSC_CREDENTIAL_PORT,
  CLIENT_CREATE_TASK,
  CLIENT_CREATE_TASK_ALERT,
  CLIENT_CREATE_TASK_ALTERABLE,
  CLIENT_CREATE_TASK_COMMENT,
  CLIENT_CREATE_TASK_HOSTS_ORDERING,
  CLIENT_CREATE_TASK_SCANNER,
  CLIENT_CREATE_TASK_CONFIG,
  CLIENT_CREATE_TASK_COPY,
  CLIENT_CREATE_TASK_NAME,
  CLIENT_CREATE_TASK_OBSERVERS,
  CLIENT_CREATE_TASK_OBSERVERS_GROUP,
  CLIENT_CREATE_TASK_PREFERENCES,
  CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE,
  CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE_NAME,
  CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE_VALUE,
  CLIENT_CREATE_TASK_SCHEDULE,
  CLIENT_CREATE_TASK_SCHEDULE_PERIODS,
  CLIENT_CREATE_TASK_SLAVE,
  CLIENT_CREATE_TASK_TARGET,
  CLIENT_CREATE_USER,
  CLIENT_CREATE_USER_COPY,
  CLIENT_CREATE_USER_GROUPS,
  CLIENT_CREATE_USER_GROUPS_GROUP,
  CLIENT_CREATE_USER_HOSTS,
  CLIENT_CREATE_USER_IFACES,
  CLIENT_CREATE_USER_NAME,
  CLIENT_CREATE_USER_PASSWORD,
  CLIENT_CREATE_USER_ROLE,
  CLIENT_CREATE_USER_SOURCES,
  CLIENT_CREATE_USER_SOURCES_SOURCE,
  CLIENT_DELETE_AGENT,
  CLIENT_DELETE_ALERT,
  CLIENT_DELETE_CONFIG,
  CLIENT_DELETE_FILTER,
  CLIENT_DELETE_GROUP,
  CLIENT_DELETE_LSC_CREDENTIAL,
  CLIENT_DELETE_NOTE,
  CLIENT_DELETE_OVERRIDE,
  CLIENT_DELETE_PERMISSION,
  CLIENT_DELETE_PORT_LIST,
  CLIENT_DELETE_PORT_RANGE,
  CLIENT_DELETE_REPORT,
  CLIENT_DELETE_REPORT_FORMAT,
  CLIENT_DELETE_ROLE,
  CLIENT_DELETE_SCANNER,
  CLIENT_DELETE_SCHEDULE,
  CLIENT_DELETE_SLAVE,
  CLIENT_DELETE_TAG,
  CLIENT_DELETE_TARGET,
  CLIENT_DELETE_TASK,
  CLIENT_DELETE_USER,
  CLIENT_DESCRIBE_AUTH,
  CLIENT_DESCRIBE_CERT,
  CLIENT_DESCRIBE_FEED,
  CLIENT_DESCRIBE_SCAP,
  CLIENT_EMPTY_TRASHCAN,
  CLIENT_GET_AGENTS,
  CLIENT_GET_AGGREGATES,
  CLIENT_GET_ALERTS,
  CLIENT_GET_CONFIGS,
  CLIENT_GET_FILTERS,
  CLIENT_GET_GROUPS,
  CLIENT_GET_INFO,
  CLIENT_GET_LSC_CREDENTIALS,
  CLIENT_GET_NOTES,
  CLIENT_GET_NVTS,
  CLIENT_GET_NVT_FAMILIES,
  CLIENT_GET_NVT_FEED_VERSION,
  CLIENT_GET_OVERRIDES,
  CLIENT_GET_PERMISSIONS,
  CLIENT_GET_PORT_LISTS,
  CLIENT_GET_PREFERENCES,
  CLIENT_GET_REPORTS,
  CLIENT_GET_REPORT_FORMATS,
  CLIENT_GET_RESULTS,
  CLIENT_GET_ROLES,
  CLIENT_GET_SCANNERS,
  CLIENT_GET_SCHEDULES,
  CLIENT_GET_SETTINGS,
  CLIENT_GET_SLAVES,
  CLIENT_GET_SYSTEM_REPORTS,
  CLIENT_GET_TAGS,
  CLIENT_GET_TARGETS,
  CLIENT_GET_TASKS,
  CLIENT_GET_USERS,
  CLIENT_GET_VERSION,
  CLIENT_GET_VERSION_AUTHENTIC,
  CLIENT_HELP,
  CLIENT_MODIFY_AGENT,
  CLIENT_MODIFY_AGENT_COMMENT,
  CLIENT_MODIFY_AGENT_NAME,
  CLIENT_MODIFY_ALERT,
  CLIENT_MODIFY_ALERT_COMMENT,
  CLIENT_MODIFY_ALERT_CONDITION,
  CLIENT_MODIFY_ALERT_CONDITION_DATA,
  CLIENT_MODIFY_ALERT_CONDITION_DATA_NAME,
  CLIENT_MODIFY_ALERT_EVENT,
  CLIENT_MODIFY_ALERT_EVENT_DATA,
  CLIENT_MODIFY_ALERT_EVENT_DATA_NAME,
  CLIENT_MODIFY_ALERT_FILTER,
  CLIENT_MODIFY_ALERT_METHOD,
  CLIENT_MODIFY_ALERT_METHOD_DATA,
  CLIENT_MODIFY_ALERT_METHOD_DATA_NAME,
  CLIENT_MODIFY_ALERT_NAME,
  CLIENT_MODIFY_AUTH,
  CLIENT_MODIFY_AUTH_GROUP,
  CLIENT_MODIFY_AUTH_GROUP_AUTHCONFSETTING,
  CLIENT_MODIFY_CONFIG,
  CLIENT_MODIFY_CONFIG_COMMENT,
  CLIENT_MODIFY_CONFIG_FAMILY_SELECTION,
  CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY,
  CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_ALL,
  CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_GROWING,
  CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_NAME,
  CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_GROWING,
  CLIENT_MODIFY_CONFIG_NAME,
  CLIENT_MODIFY_CONFIG_NVT_SELECTION,
  CLIENT_MODIFY_CONFIG_NVT_SELECTION_FAMILY,
  CLIENT_MODIFY_CONFIG_NVT_SELECTION_NVT,
  CLIENT_MODIFY_CONFIG_PREFERENCE,
  CLIENT_MODIFY_CONFIG_PREFERENCE_NAME,
  CLIENT_MODIFY_CONFIG_PREFERENCE_NVT,
  CLIENT_MODIFY_CONFIG_PREFERENCE_VALUE,
  CLIENT_MODIFY_FILTER,
  CLIENT_MODIFY_FILTER_COMMENT,
  CLIENT_MODIFY_FILTER_NAME,
  CLIENT_MODIFY_FILTER_TERM,
  CLIENT_MODIFY_FILTER_TYPE,
  CLIENT_MODIFY_GROUP,
  CLIENT_MODIFY_GROUP_COMMENT,
  CLIENT_MODIFY_GROUP_NAME,
  CLIENT_MODIFY_GROUP_USERS,
  CLIENT_MODIFY_LSC_CREDENTIAL,
  CLIENT_MODIFY_LSC_CREDENTIAL_COMMENT,
  CLIENT_MODIFY_LSC_CREDENTIAL_LOGIN,
  CLIENT_MODIFY_LSC_CREDENTIAL_NAME,
  CLIENT_MODIFY_LSC_CREDENTIAL_PASSWORD,
  CLIENT_MODIFY_NOTE,
  CLIENT_MODIFY_NOTE_ACTIVE,
  CLIENT_MODIFY_NOTE_HOSTS,
  CLIENT_MODIFY_NOTE_PORT,
  CLIENT_MODIFY_NOTE_RESULT,
  CLIENT_MODIFY_NOTE_SEVERITY,
  CLIENT_MODIFY_NOTE_TASK,
  CLIENT_MODIFY_NOTE_TEXT,
  CLIENT_MODIFY_NOTE_THREAT,
  CLIENT_MODIFY_OVERRIDE,
  CLIENT_MODIFY_OVERRIDE_ACTIVE,
  CLIENT_MODIFY_OVERRIDE_HOSTS,
  CLIENT_MODIFY_OVERRIDE_NEW_SEVERITY,
  CLIENT_MODIFY_OVERRIDE_NEW_THREAT,
  CLIENT_MODIFY_OVERRIDE_PORT,
  CLIENT_MODIFY_OVERRIDE_RESULT,
  CLIENT_MODIFY_OVERRIDE_SEVERITY,
  CLIENT_MODIFY_OVERRIDE_TASK,
  CLIENT_MODIFY_OVERRIDE_TEXT,
  CLIENT_MODIFY_OVERRIDE_THREAT,
  CLIENT_MODIFY_PERMISSION,
  CLIENT_MODIFY_PERMISSION_COMMENT,
  CLIENT_MODIFY_PERMISSION_NAME,
  CLIENT_MODIFY_PERMISSION_RESOURCE,
  CLIENT_MODIFY_PERMISSION_RESOURCE_TYPE,
  CLIENT_MODIFY_PERMISSION_SUBJECT,
  CLIENT_MODIFY_PERMISSION_SUBJECT_TYPE,
  CLIENT_MODIFY_PORT_LIST,
  CLIENT_MODIFY_PORT_LIST_COMMENT,
  CLIENT_MODIFY_PORT_LIST_NAME,
  CLIENT_MODIFY_REPORT,
  CLIENT_MODIFY_REPORT_COMMENT,
  CLIENT_MODIFY_REPORT_FORMAT,
  CLIENT_MODIFY_REPORT_FORMAT_ACTIVE,
  CLIENT_MODIFY_REPORT_FORMAT_NAME,
  CLIENT_MODIFY_REPORT_FORMAT_PARAM,
  CLIENT_MODIFY_REPORT_FORMAT_PARAM_NAME,
  CLIENT_MODIFY_REPORT_FORMAT_PARAM_VALUE,
  CLIENT_MODIFY_REPORT_FORMAT_SUMMARY,
  CLIENT_MODIFY_ROLE,
  CLIENT_MODIFY_ROLE_COMMENT,
  CLIENT_MODIFY_ROLE_NAME,
  CLIENT_MODIFY_ROLE_USERS,
  CLIENT_MODIFY_SCANNER,
  CLIENT_MODIFY_SCANNER_COMMENT,
  CLIENT_MODIFY_SCANNER_NAME,
  CLIENT_MODIFY_SCANNER_HOST,
  CLIENT_MODIFY_SCANNER_PORT,
  CLIENT_MODIFY_SCANNER_TYPE,
  CLIENT_MODIFY_SCANNER_CA_PUB,
  CLIENT_MODIFY_SCANNER_KEY_PUB,
  CLIENT_MODIFY_SCANNER_KEY_PRIV,
  CLIENT_MODIFY_SCHEDULE,
  CLIENT_MODIFY_SCHEDULE_COMMENT,
  CLIENT_MODIFY_SCHEDULE_DURATION,
  CLIENT_MODIFY_SCHEDULE_DURATION_UNIT,
  CLIENT_MODIFY_SCHEDULE_FIRST_TIME,
  CLIENT_MODIFY_SCHEDULE_FIRST_TIME_DAY_OF_MONTH,
  CLIENT_MODIFY_SCHEDULE_FIRST_TIME_HOUR,
  CLIENT_MODIFY_SCHEDULE_FIRST_TIME_MINUTE,
  CLIENT_MODIFY_SCHEDULE_FIRST_TIME_MONTH,
  CLIENT_MODIFY_SCHEDULE_FIRST_TIME_YEAR,
  CLIENT_MODIFY_SCHEDULE_NAME,
  CLIENT_MODIFY_SCHEDULE_PERIOD,
  CLIENT_MODIFY_SCHEDULE_PERIOD_UNIT,
  CLIENT_MODIFY_SCHEDULE_TIMEZONE,
  CLIENT_MODIFY_SETTING,
  CLIENT_MODIFY_SETTING_NAME,
  CLIENT_MODIFY_SETTING_VALUE,
  CLIENT_MODIFY_SLAVE,
  CLIENT_MODIFY_SLAVE_COMMENT,
  CLIENT_MODIFY_SLAVE_HOST,
  CLIENT_MODIFY_SLAVE_LOGIN,
  CLIENT_MODIFY_SLAVE_NAME,
  CLIENT_MODIFY_SLAVE_PASSWORD,
  CLIENT_MODIFY_SLAVE_PORT,
  CLIENT_MODIFY_TAG,
  CLIENT_MODIFY_TAG_ACTIVE,
  CLIENT_MODIFY_TAG_COMMENT,
  CLIENT_MODIFY_TAG_NAME,
  CLIENT_MODIFY_TAG_RESOURCE,
  CLIENT_MODIFY_TAG_RESOURCE_TYPE,
  CLIENT_MODIFY_TAG_VALUE,
  CLIENT_MODIFY_TARGET,
  CLIENT_MODIFY_TARGET_ALIVE_TESTS,
  CLIENT_MODIFY_TARGET_COMMENT,
  CLIENT_MODIFY_TARGET_ESXI_LSC_CREDENTIAL,
  CLIENT_MODIFY_TARGET_HOSTS,
  CLIENT_MODIFY_TARGET_EXCLUDE_HOSTS,
  CLIENT_MODIFY_TARGET_REVERSE_LOOKUP_ONLY,
  CLIENT_MODIFY_TARGET_REVERSE_LOOKUP_UNIFY,
  CLIENT_MODIFY_TARGET_NAME,
  CLIENT_MODIFY_TARGET_PORT_LIST,
  CLIENT_MODIFY_TARGET_SMB_LSC_CREDENTIAL,
  CLIENT_MODIFY_TARGET_SSH_LSC_CREDENTIAL,
  CLIENT_MODIFY_TARGET_SSH_LSC_CREDENTIAL_PORT,
  CLIENT_MODIFY_TASK,
  CLIENT_MODIFY_TASK_ALERT,
  CLIENT_MODIFY_TASK_ALTERABLE,
  CLIENT_MODIFY_TASK_COMMENT,
  CLIENT_MODIFY_TASK_CONFIG,
  CLIENT_MODIFY_TASK_FILE,
  CLIENT_MODIFY_TASK_NAME,
  CLIENT_MODIFY_TASK_OBSERVERS,
  CLIENT_MODIFY_TASK_OBSERVERS_GROUP,
  CLIENT_MODIFY_TASK_PREFERENCES,
  CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE,
  CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE_NAME,
  CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE_VALUE,
  CLIENT_MODIFY_TASK_SCHEDULE,
  CLIENT_MODIFY_TASK_SCHEDULE_PERIODS,
  CLIENT_MODIFY_TASK_SLAVE,
  CLIENT_MODIFY_TASK_TARGET,
  CLIENT_MODIFY_TASK_HOSTS_ORDERING,
  CLIENT_MODIFY_TASK_SCANNER,
  CLIENT_MODIFY_USER,
  CLIENT_MODIFY_USER_GROUPS,
  CLIENT_MODIFY_USER_GROUPS_GROUP,
  CLIENT_MODIFY_USER_HOSTS,
  CLIENT_MODIFY_USER_IFACES,
  CLIENT_MODIFY_USER_NAME,
  CLIENT_MODIFY_USER_NEW_NAME,
  CLIENT_MODIFY_USER_PASSWORD,
  CLIENT_MODIFY_USER_ROLE,
  CLIENT_MODIFY_USER_SOURCES,
  CLIENT_MODIFY_USER_SOURCES_SOURCE,
  CLIENT_RESTORE,
  CLIENT_RESUME_TASK,
  CLIENT_RUN_WIZARD,
  CLIENT_RUN_WIZARD_MODE,
  CLIENT_RUN_WIZARD_NAME,
  CLIENT_RUN_WIZARD_PARAMS,
  CLIENT_RUN_WIZARD_PARAMS_PARAM,
  CLIENT_RUN_WIZARD_PARAMS_PARAM_NAME,
  CLIENT_RUN_WIZARD_PARAMS_PARAM_VALUE,
  CLIENT_START_TASK,
  CLIENT_STOP_TASK,
  CLIENT_SYNC_CERT,
  CLIENT_SYNC_FEED,
  CLIENT_SYNC_SCAP,
  CLIENT_TEST_ALERT,
  CLIENT_VERIFY_AGENT,
  CLIENT_VERIFY_REPORT_FORMAT,
  CLIENT_VERIFY_SCANNER,
} client_state_t;

/**
 * @brief The state of the client.
 */
static client_state_t client_state = CLIENT_TOP;

/**
 * @brief Set the client state.
 */
static void
set_client_state (client_state_t state)
{
  client_state = state;
  tracef ("   client state set: %i\n", client_state);
}


/* Communication. */

/**
 * @brief Send a response message to the client.
 *
 * @param[in]  msg                       The message, a string.
 * @param[in]  user_send_to_client       Function to send to client.
 * @param[in]  user_send_to_client_data  Argument to \p user_send_to_client.
 *
 * @return TRUE if send to client failed, else FALSE.
 */
static gboolean
send_to_client (const char* msg,
                int (*user_send_to_client) (const char*, void*),
                void* user_send_to_client_data)
{
  if (user_send_to_client && msg)
    return user_send_to_client (msg, user_send_to_client_data);
  return FALSE;
}

/**
 * @brief Send an XML element error response message to the client.
 *
 * @param[in]  command  Command name.
 * @param[in]  element  Element name.
 * @param[in]  write_to_client       Function to write to client.
 * @param[in]  write_to_client_data  Argument to \p write_to_client.
 *
 * @return TRUE if out of space in to_client, else FALSE.
 */
static gboolean
send_element_error_to_client (const char* command, const char* element,
                              int (*write_to_client) (const char*, void*),
                              void* write_to_client_data)
{
  gchar *msg;
  gboolean ret;

  /** @todo Set gerror so parsing terminates. */
  msg = g_strdup_printf ("<%s_response status=\""
                         STATUS_ERROR_SYNTAX
                         "\" status_text=\"Bogus element: %s\"/>",
                         command,
                         element);
  ret = send_to_client (msg, write_to_client, write_to_client_data);
  g_free (msg);
  return ret;
}

/**
 * @brief Send an XML find error response message to the client.
 *
 * @param[in]  command      Command name.
 * @param[in]  type         Resource type.
 * @param[in]  id           Resource ID.
 * @param[in]  omp_parser   OMP Parser.
 *
 * @return TRUE if out of space in to_client, else FALSE.
 */
static gboolean
send_find_error_to_client (const char* command, const char* type,
                           const char* id, omp_parser_t *omp_parser)
{
  gchar *msg;
  gboolean ret;

  msg = g_strdup_printf ("<%s_response status=\""
                         STATUS_ERROR_MISSING
                         "\" status_text=\"Failed to find %s '%s'\"/>",
                         command, type, id);
  ret = send_to_client (msg, omp_parser->client_writer,
                        omp_parser->client_writer_data);
  g_free (msg);
  return ret;
}

/**
 * @brief Set an out of space parse error on a GError.
 *
 * @param [out]  error  The error.
 */
static void
error_send_to_client (GError** error)
{
  tracef ("   send_to_client out of space in to_client\n");
  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
               "Manager out of space for reply to client.");
}

/**
 * @brief Set an internal error on a GError.
 *
 * @param [out]  error  The error.
 */
static void
internal_error_send_to_client (GError** error)
{
  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
               "Internal Error.");
}


/* XML parser handlers. */

/**
 * @brief Expand to XML for a STATUS_ERROR_SYNTAX response.
 *
 * @param  tag   Name of the command generating the response.
 * @param  text  Text for the status_text attribute of the response.
 */
#define XML_ERROR_SYNTAX(tag, text)                      \
 "<" tag "_response"                                     \
 " status=\"" STATUS_ERROR_SYNTAX "\""                   \
 " status_text=\"" text "\"/>"

/**
 * @brief Expand to XML for a STATUS_ERROR_SYNTAX response.
 *
 * This is a variant of the \ref XML_ERROR_SYNTAX macro to allow for a
 * runtime defined syntax_text attribute value.
 *
 * @param  tag   Name of the command generating the response.
 * @param text   Value for the status_text attribute of the response.
 *               The function takes care of proper quoting.
 *
 * @return A malloced XML string.  The caller must use g_free to
 *         release it.
 */
static char *
make_xml_error_syntax (const char *tag, const char *text)
{
  char *textbuf;
  char *ret;

  textbuf = g_markup_escape_text (text, -1);
  ret = g_strdup_printf ("<%s_response status=\"" STATUS_ERROR_SYNTAX "\""
                         " status_text=\"%s\"/>", tag, textbuf);
  g_free (textbuf);
  return ret;
}


/**
 * @brief Expand to XML for a STATUS_ERROR_ACCESS response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_ERROR_ACCESS(tag)                            \
 "<" tag "_response"                                     \
 " status=\"" STATUS_ERROR_ACCESS "\""                   \
 " status_text=\"" STATUS_ERROR_ACCESS_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_SERVICE_UNAVAILABLE response.
 *
 * @param  tag   Name of the command generating the response.
 */
#define XML_ERROR_UNAVAILABLE(tag)                        \
 "<" tag "_response"                                      \
 " status=\"" STATUS_SERVICE_UNAVAILABLE "\""             \
 " status_text=\"" STATUS_SERVICE_UNAVAILABLE_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_ERROR_MISSING response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_ERROR_MISSING(tag)                           \
 "<" tag "_response"                                     \
 " status=\"" STATUS_ERROR_MISSING "\""                  \
 " status_text=\"" STATUS_ERROR_MISSING_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_ERROR_AUTH_FAILED response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_ERROR_AUTH_FAILED(tag)                       \
 "<" tag "_response"                                     \
 " status=\"" STATUS_ERROR_AUTH_FAILED "\""              \
 " status_text=\"" STATUS_ERROR_AUTH_FAILED_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_ERROR_BUSY response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_ERROR_BUSY(tag)                              \
 "<" tag "_response"                                     \
 " status=\"" STATUS_ERROR_BUSY "\""                     \
 " status_text=\"" STATUS_ERROR_BUSY_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_OK response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_OK(tag)                                      \
 "<" tag "_response"                                     \
 " status=\"" STATUS_OK "\""                             \
 " status_text=\"" STATUS_OK_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_OK_CREATED response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_OK_CREATED(tag)                              \
 "<" tag "_response"                                     \
 " status=\"" STATUS_OK_CREATED "\""                     \
 " status_text=\"" STATUS_OK_CREATED_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_OK_CREATED response with %s for ID.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_OK_CREATED_ID(tag)                           \
 "<" tag "_response"                                     \
 " status=\"" STATUS_OK_CREATED "\""                     \
 " status_text=\"" STATUS_OK_CREATED_TEXT "\""           \
 " id=\"%s\"/>"

/**
 * @brief Expand to XML for a STATUS_OK_REQUESTED response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_OK_REQUESTED(tag)                            \
 "<" tag "_response"                                     \
 " status=\"" STATUS_OK_REQUESTED "\""                   \
 " status_text=\"" STATUS_OK_REQUESTED_TEXT "\"/>"

/**
 * @brief Expand to XML for a STATUS_INTERNAL_ERROR response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define XML_INTERNAL_ERROR(tag)                          \
 "<" tag "_response"                                     \
 " status=\"" STATUS_INTERNAL_ERROR "\""                 \
 " status_text=\"" STATUS_INTERNAL_ERROR_TEXT "\"/>"

/**
 * @brief Sends XML for a STATUS_SERVICE_DOWN response.
 *
 * @param  tag  Name of the command generating the response.
 */
#define SEND_XML_SERVICE_DOWN(tag)                                            \
  do {                                                                        \
    char *str;                                                                \
    if (scanner_current_loading && scanner_total_loading)                     \
      str = g_strdup_printf ("<%s_response status='%s' "                      \
                             "status_text='Scanner loading nvts (%d/%d)'/>",  \
                             tag, STATUS_SERVICE_DOWN,                        \
                             scanner_current_loading, scanner_total_loading); \
    else                                                                      \
      str = g_strdup_printf ("<%s_response status='%s' status_text='%s'/>",   \
                             tag, STATUS_SERVICE_DOWN,                        \
                             STATUS_SERVICE_DOWN_TEXT);                       \
    SEND_TO_CLIENT_OR_FAIL(str);                                              \
    g_free (str);                                                             \
  } while (0);

/** @cond STATIC */

/**
 * @brief Send start of GET response.
 *
 * @param[in]  type                  Type.
 * @param[in]  write_to_client       Function that sends to clients.
 * @param[in]  write_to_client_data  Data for write_to_client.
 */
int
send_get_start (const char *type, int (*write_to_client) (const char*, void*),
                void* write_to_client_data)
{
  gchar *msg;

  if (strcmp (type, "info"))
    msg = g_markup_printf_escaped ("<get_%ss_response"
                                   " status=\"" STATUS_OK "\""
                                   " status_text=\"" STATUS_OK_TEXT "\">",
                                   type);
  else
    msg = g_markup_printf_escaped ("<get_%s_response"
                                   " status=\"" STATUS_OK "\""
                                   " status_text=\"" STATUS_OK_TEXT "\">",
                                   type);


  if (send_to_client (msg, write_to_client, write_to_client_data))
    {
      g_free (msg);
      return 1;
    }
  g_free (msg);
  return 0;
}

/**
 * @brief Send common part of GET response for a single resource.
 *
 * @param[in]  type                  Type.
 * @param[in]  get                   GET data.
 * @param[in]  iterator              Iterator.
 * @param[in]  write_to_client       Function that sends to clients.
 * @param[in]  write_to_client_data  Data for write_to_client.
 * @param[in]  writable              Whether the resource is writable.
 * @param[in]  in_use                Whether the resource is in use.
 *
 * @return 0 success, 1 send error.
 */
int
send_get_common (const char *type, get_data_t *get, iterator_t *iterator,
                 int (*write_to_client) (const char *, void*),
                 void* write_to_client_data, int writable, int in_use)
{
  GString *buffer;
  const char *tag_type;
  iterator_t tags;

  buffer = g_string_new ("");

  buffer_xml_append_printf (buffer,
                            "<%s id=\"%s\">"
                            "<owner><name>%s</name></owner>"
                            "<name>%s</name>"
                            "<comment>%s</comment>"
                            "<creation_time>%s</creation_time>"
                            "<modification_time>%s</modification_time>"
                            "<writable>%i</writable>"
                            "<in_use>%i</in_use>"
                            "<permissions>",
                            type,
                            get_iterator_uuid (iterator)
                            ? get_iterator_uuid (iterator)
                            : "",
                            get_iterator_owner_name (iterator)
                            ? get_iterator_owner_name (iterator)
                            : "",
                            get_iterator_name (iterator)
                            ? get_iterator_name (iterator)
                            : "",
                            get_iterator_comment (iterator)
                            ? get_iterator_comment (iterator)
                            : "",
                            get_iterator_creation_time (iterator)
                            ? get_iterator_creation_time (iterator)
                            : "",
                            get_iterator_modification_time (iterator)
                            ? get_iterator_modification_time (iterator)
                            : "",
                            writable,
                            in_use);

  if (/* The user is the owner. */
      (current_credentials.username
       && get_iterator_owner_name (iterator)
       && (strcmp (get_iterator_owner_name (iterator),
                   current_credentials.username)
           == 0))
      /* Or the user is effectively the owner. */
      || user_has_super (current_credentials.uuid,
                         get_iterator_owner (iterator))
      /* Or the user has Admin rights and the resource is a permission... */
      || (current_credentials.uuid
          && ((strcmp (type, "permission") == 0)
              && get_iterator_uuid (iterator)
              /* ... but not the special Admin permission. */
              && strcmp (get_iterator_uuid (iterator),
                         PERMISSION_UUID_ADMIN_EVERYTHING))
          && user_can_everything (current_credentials.uuid)))
    {
      buffer_xml_append_printf (buffer,
                                "<permission>"
                                "<name>Everything</name>"
                                "</permission>"
                                "</permissions>");
    }
  else if (current_credentials.uuid
           && ((strcmp (type, "user") == 0)
               || (strcmp (type, "role") == 0)
               || (strcmp (type, "group") == 0))
           && (get_iterator_owner (iterator) == 0)
           && user_can_everything (current_credentials.uuid))
    {
      if ((strcmp (type, "user") == 0)
          && user_can_super_everyone (get_iterator_uuid (iterator))
          && strcmp (get_iterator_uuid (iterator), current_credentials.uuid))
        {
          /* Resource is the Super Admin. */
          buffer_xml_append_printf (buffer,
                                    "<permission><name>get_users</name></permission>"
                                    "</permissions>");
        }
      else
        /* The user has admin rights and it's a global user/role/group.
         *
         * These are left over from before users/roles/groups had owners. */
        buffer_xml_append_printf (buffer,
                                  "<permission>"
                                  "<name>Everything</name>"
                                  "</permission>"
                                  "</permissions>");
    }
  else
    {
      iterator_t perms;
      get_data_t perms_get;

      memset (&perms_get, '\0', sizeof (perms_get));
      perms_get.filter = g_strdup_printf ("resource_uuid=%s"
                                          " owner=any"
                                          " permission=any",
                                          get_iterator_uuid (iterator));
      init_permission_iterator (&perms, &perms_get);
      g_free (perms_get.filter);
      while (next (&perms))
        buffer_xml_append_printf (buffer,
                                  "<permission><name>%s</name></permission>",
                                  get_iterator_name (&perms));
      cleanup_iterator (&perms);

      buffer_xml_append_printf (buffer, "</permissions>");
    }

  tag_type = get->subtype ? get->subtype : get->type;

  if (get->details || get->id)
    {
      buffer_xml_append_printf (buffer,
                                "<user_tags>"
                                "<count>%i</count>",
                                resource_tag_count (tag_type,
                                                    get_iterator_resource
                                                      (iterator),
                                                    1));

      init_resource_tag_iterator (&tags, tag_type,
                                  get_iterator_resource (iterator),
                                  1, NULL, 1);

      while (next (&tags))
        {
          buffer_xml_append_printf (buffer,
                                    "<tag id=\"%s\">"
                                    "<name>%s</name>"
                                    "<value>%s</value>"
                                    "<comment>%s</comment>"
                                    "</tag>",
                                    resource_tag_iterator_uuid (&tags),
                                    resource_tag_iterator_name (&tags),
                                    resource_tag_iterator_value (&tags),
                                    resource_tag_iterator_comment (&tags));
        }

      cleanup_iterator (&tags);

      buffer_xml_append_printf (buffer,
                                "</user_tags>");
    }
  else
    {
      buffer_xml_append_printf (buffer,
                                "<user_tags>"
                                "<count>%i</count>"
                                "</user_tags>",
                                resource_tag_count (tag_type,
                                                    get_iterator_resource
                                                      (iterator),
                                                    1));
    }

  if (send_to_client (buffer->str, write_to_client, write_to_client_data))
    {
      g_string_free (buffer, TRUE);
      return 1;
    }
  g_string_free (buffer, TRUE);
  return 0;
}

/**
 * @brief Write data of a GET command filter to a string buffer as XML.
 *
 * @param[in] msg          The string buffer to write to.
 * @param[in] type         The filtered type.
 * @param[in] get          GET data.
 * @param[in] filter_term  Filter term.
 */
int
buffer_get_filter_xml (GString *msg, const char* type,
                       get_data_t *get, const char* filter_term)
{
  keyword_t **point;
  array_t *split;
  filter_t filter;

  buffer_xml_append_printf (msg,
                            "<filters id=\"%s\">"
                            "<term>%s</term>",
                            get->filt_id ? get->filt_id : "",
                            filter_term);

  if (get->filt_id
      && strcmp (get->filt_id, "")
      && (find_filter (get->filt_id, &filter) == 0)
      && filter != 0)
    buffer_xml_append_printf (msg,
                              "<name>%s</name>",
                              filter_name (filter));

  buffer_xml_append_printf (msg,
                            "<keywords>");

  split = split_filter (filter_term);
  point = (keyword_t**) split->pdata;
  while (*point)
    {
      keyword_t *keyword;
      keyword = *point;
      buffer_xml_append_printf (msg,
                                "<keyword>"
                                "<column>%s</column>"
                                "<relation>%s</relation>"
                                "<value>%s%s%s</value>"
                                "</keyword>",
                                keyword->column ? keyword->column : "",
                                keyword_relation_symbol (keyword->relation),
                                keyword->quoted ? "\"" : "",
                                keyword->string ? keyword->string : "",
                                keyword->quoted ? "\"" : "");
      point++;
    }
  filter_free (split);

  buffer_xml_append_printf (msg,
                            "</keywords>"
                            "</filters>");
  return 0;
}

/**
 * @brief Send end of GET response.
 *
 * @param[in]  type                  Type.
 * @param[in]  get                   GET data.
 * @param[in]  count                 Page count.
 * @param[in]  filtered              Filtered count.
 * @param[in]  full                  Full count.
 * @param[in]  write_to_client       Function that sends to clients.
 * @param[in]  write_to_client_data  Data for write_to_client.
 */
int
send_get_end (const char *type, get_data_t *get, int count, int filtered,
              int full, int (*write_to_client) (const char *, void*),
              void* write_to_client_data)
{
  gchar *sort_field, *filter;
  int first, max, sort_order;
  GString *type_many, *msg;

  if (get->filt_id && strcmp (get->filt_id, "0"))
    {
      if (get->filter_replacement)
        filter = g_strdup (get->filter_replacement);
      else
        filter = filter_term (get->filt_id);
      if (filter == NULL)
        return 2;
    }
  else
    filter = NULL;

  manage_filter_controls (filter ? filter : get->filter,
                          &first, &max, &sort_field, &sort_order);

  if (filter || get->filter)
    {
      gchar *new_filter;
      new_filter = manage_clean_filter (filter ? filter : get->filter);
      g_free (filter);
      if (((strcmp (type, "task") == 0)
           || (strcmp (type, "report") == 0))
          && (filter_term_value (new_filter, "apply_overrides") == NULL))
        {
          filter = new_filter;
          new_filter = g_strdup_printf ("apply_overrides=%i %s",
                                        APPLY_OVERRIDES_DEFAULT,
                                        filter);
          g_free (filter);
        }
      filter = new_filter;
    }
  else
    filter = manage_clean_filter ("");

  type_many = g_string_new (type);

  if (strcmp (type, "info") != 0)
    g_string_append (type_many, "s");

  msg = g_string_new ("");

  buffer_get_filter_xml (msg, type, get, filter);

  buffer_xml_append_printf (msg,
                            "<sort>"
                            "<field>%s<order>%s</order></field>"
                            "</sort>"
                            "<%s start=\"%i\" max=\"%i\"/>"
                            "<%s_count>"
                            "%i"
                            "<filtered>%i</filtered>"
                            "<page>%i</page>"
                            "</%s_count>"
                            "</get_%s_response>",
                            sort_field,
                            sort_order ? "ascending" : "descending",
                            type_many->str,
                            first,
                            max,
                            type,
                            full,
                            filtered,
                            count,
                            type,
                            type_many->str);
  g_string_free (type_many, TRUE);
  g_free (sort_field);
  g_free (filter);

  if (send_to_client (msg->str, write_to_client, write_to_client_data))
    {
      g_string_free (msg, TRUE);
      return 1;
    }
  g_string_free (msg, TRUE);
  return 0;
}

/**
 * @brief Send start of GET response to client, returning on fail.
 *
 * @param[in]  type  Type of resource.
 * @param[in]  get   GET data.
 */
#define SEND_GET_START(type)                                                 \
  do                                                                         \
    {                                                                        \
      if (send_get_start (type, omp_parser->client_writer,                   \
                          omp_parser->client_writer_data))                   \
        {                                                                    \
          error_send_to_client (error);                                      \
          return;                                                            \
        }                                                                    \
    }                                                                        \
  while (0)

/**
 * @brief Send common part of GET response to client, returning on fail.
 *
 * @param[in]  type      Type of resource.
 * @param[in]  get       GET data.
 * @param[in]  iterator  Iterator.
 */
#define SEND_GET_COMMON(type, get, iterator)                                   \
  do                                                                           \
    {                                                                          \
      if (send_get_common (G_STRINGIFY (type), get, iterator,                  \
                           omp_parser->client_writer,                          \
                           omp_parser->client_writer_data,                     \
                           (get)->trash                                        \
                            ? trash_ ## type ## _writable                      \
                               (get_iterator_resource                          \
                                 (iterator))                                   \
                            : type ## _writable                                \
                               (get_iterator_resource                          \
                                 (iterator)),                                  \
                           (get)->trash                                        \
                            ? trash_ ## type ## _in_use                        \
                               (get_iterator_resource                          \
                                 (iterator))                                   \
                            : type ## _in_use                                  \
                               (get_iterator_resource                          \
                                 (iterator))))                                 \
        {                                                                      \
          error_send_to_client (error);                                        \
          return;                                                              \
        }                                                                      \
    }                                                                          \
  while (0)

/**
 * @brief Send end of GET response to client, returning on fail.
 *
 * @param[in]  type  Type of resource.
 * @param[in]  get   GET data.
 */
#define SEND_GET_END(type, get, count, filtered)                             \
  do                                                                         \
    {                                                                        \
      if (send_get_end (type, get, count, filtered,                          \
                        resource_count (type, get),                          \
                        omp_parser->client_writer,                           \
                        omp_parser->client_writer_data))                     \
        {                                                                    \
          error_send_to_client (error);                                      \
          return;                                                            \
        }                                                                    \
    }                                                                        \
  while (0)

/**
 * @brief Send response message to client, returning on fail.
 *
 * Queue a message in \ref to_client with \ref send_to_client.  On failure
 * call \ref error_send_to_client on a GError* called "error" and do a return.
 *
 * @param[in]   msg    The message, a string.
 */
#define SEND_TO_CLIENT_OR_FAIL(msg)                                          \
  do                                                                         \
    {                                                                        \
      if (send_to_client (msg, omp_parser->client_writer,                    \
                          omp_parser->client_writer_data))                   \
        {                                                                    \
          error_send_to_client (error);                                      \
          return;                                                            \
        }                                                                    \
    }                                                                        \
  while (0)

/**
 * @brief Send response message to client, returning on fail.
 *
 * Queue a message in \ref to_client with \ref send_to_client.  On failure
 * call \ref error_send_to_client on a GError* called "error" and do a return.
 *
 * @param[in]   format    Format string for message.
 * @param[in]   args      Arguments for format string.
 */
#define SENDF_TO_CLIENT_OR_FAIL(format, args...)                             \
  do                                                                         \
    {                                                                        \
      gchar* msg = g_markup_printf_escaped (format , ## args);               \
      if (send_to_client (msg, omp_parser->client_writer,                    \
                          omp_parser->client_writer_data))                   \
        {                                                                    \
          g_free (msg);                                                      \
          error_send_to_client (error);                                      \
          return;                                                            \
        }                                                                    \
      g_free (msg);                                                          \
    }                                                                        \
  while (0)

/** @endcond */


/**
 * @brief Insert else clause for omp_xml_handle_start_element.
 *
 * @param[in]  op  Operation.
 */
#define ELSE_ERROR(op)                                          \
  else if (omp_parser->importing)                               \
    {                                                           \
      if (omp_parser->read_over == 0)                           \
        {                                                       \
          omp_parser->read_over = 1;                            \
          omp_parser->parent_state = client_state;              \
        }                                                       \
    }                                                           \
  else                                                          \
    {                                                           \
      if (send_element_error_to_client (op, element_name,       \
                                        write_to_client,        \
                                        write_to_client_data))  \
        {                                                       \
          error_send_to_client (error);                         \
          return;                                               \
        }                                                       \
      set_client_state (CLIENT_AUTHENTIC);                      \
      g_set_error (error,                                       \
                   G_MARKUP_ERROR,                              \
                   G_MARKUP_ERROR_UNKNOWN_ELEMENT,              \
                   "Error");                                    \
    }                                                           \
  break

/**
 * @brief Insert else clause for omp_xml_handle_start_element in create_task.
 *
 */
#define ELSE_ERROR_CREATE_TASK()                                     \
  else if (omp_parser->importing)                                    \
    {                                                                \
      if (omp_parser->read_over == 0)                                \
        {                                                            \
          omp_parser->read_over = 1;                                 \
          omp_parser->parent_state = client_state;                   \
        }                                                            \
    }                                                                \
  else                                                               \
    {                                                                \
      request_delete_task (&create_task_data->task);                 \
      if (send_element_error_to_client ("create_task", element_name, \
                                        write_to_client,             \
                                        write_to_client_data))       \
        {                                                            \
          error_send_to_client (error);                              \
          return;                                                    \
        }                                                            \
      set_client_state (CLIENT_AUTHENTIC);                           \
      g_set_error (error,                                            \
                   G_MARKUP_ERROR,                                   \
                   G_MARKUP_ERROR_UNKNOWN_ELEMENT,                   \
                   "Error");                                         \
    }                                                                \
  break

/**
 * @brief Creates a log event entry for a resource action.
 *
 * @param[in]   event       Event type.
 * @param[in]   resource    Resource name.
 * @param[in]   id          Resource id.
 * @param[in]   action      Action done.
 */
static void
log_event (const char *event, const char *resource, const char *id,
           const char *action)
{
  gchar* domain = g_strdup_printf ("event %s", event);
  if (id)
    g_log (domain, G_LOG_LEVEL_MESSAGE,
           "%s %s has been %s by %s", resource, id, action,
           current_credentials.username);
  else
    g_log (domain, G_LOG_LEVEL_MESSAGE,
           "%s has been %s by %s", resource, action,
           current_credentials.username);

  g_free (domain);
}

/**
 * @brief Creates a log event failure entry for a resource action.
 *
 * @param[in]   event       Event type.
 * @param[in]   resource    Resource name.
 * @param[in]   id          Resource id.
 * @param[in]   action      Action done.
 */
static void
log_event_fail (const char *event, const char *resource, const char *id,
                const char *action)
{
  gchar* domain = g_strdup_printf ("event %s", event);
  if (id)
    g_log (domain, G_LOG_LEVEL_MESSAGE,
           "%s %s could not be %s by %s", resource, id, action,
           current_credentials.username);
  else
    g_log (domain, G_LOG_LEVEL_MESSAGE,
           "%s could not be %s by %s", resource, action,
           current_credentials.username);

  g_free (domain);
}

/** @todo Free globals when tags open, in case of duplicate tags. */
/**
 * @brief Handle the start of an OMP XML element.
 *
 * React to the start of an XML element according to the current value
 * of \ref client_state, usually adjusting \ref client_state to indicate
 * the change (with \ref set_client_state).  Call \ref send_to_client to
 * queue any responses for the client.
 *
 * Set error parameter on encountering an error.
 *
 * @param[in]  context           Parser context.
 * @param[in]  element_name      XML element name.
 * @param[in]  attribute_names   XML attribute names.
 * @param[in]  attribute_values  XML attribute values.
 * @param[in]  user_data         OMP parser.
 * @param[in]  error             Error parameter.
 */
static void
omp_xml_handle_start_element (/*@unused@*/ GMarkupParseContext* context,
                              const gchar *element_name,
                              const gchar **attribute_names,
                              const gchar **attribute_values,
                              gpointer user_data,
                              GError **error)
{
  omp_parser_t *omp_parser = (omp_parser_t*) user_data;
  int (*write_to_client) (const char *, void*)
    = (int (*) (const char *, void*)) omp_parser->client_writer;
  void* write_to_client_data = (void*) omp_parser->client_writer_data;

  tracef ("   XML  start: %s (%i)\n", element_name, client_state);

  if (omp_parser->read_over)
    omp_parser->read_over++;
  else switch (client_state)
    {
      case CLIENT_TOP:
        if (strcasecmp ("GET_VERSION", element_name) == 0)
          {
            set_client_state (CLIENT_GET_VERSION);
            break;
          }
        /*@fallthrough@*/
      case CLIENT_COMMANDS:
        if (strcasecmp ("AUTHENTICATE", element_name) == 0)
          {
            set_client_state (CLIENT_AUTHENTICATE);
          }
        else if (strcasecmp ("COMMANDS", element_name) == 0)
          {
            SENDF_TO_CLIENT_OR_FAIL
             ("<commands_response"
              " status=\"" STATUS_OK "\" status_text=\"" STATUS_OK_TEXT "\">");
            set_client_state (CLIENT_COMMANDS);
          }
        else
          {
            /** @todo If a real OMP command, return STATUS_ERROR_MUST_AUTH. */
            if (send_to_client
                 (XML_ERROR_SYNTAX ("omp",
                                    "First command must be AUTHENTICATE,"
                                    " COMMANDS or GET_VERSION"),
                  write_to_client,
                  write_to_client_data))
              {
                error_send_to_client (error);
                return;
              }
            if (client_state == CLIENT_COMMANDS)
              send_to_client ("</commands_response>",
                              write_to_client,
                              write_to_client_data);
            g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Must authenticate first.");
          }
        break;

      case CLIENT_AUTHENTIC:
      case CLIENT_AUTHENTIC_COMMANDS:
        if (command_disabled (omp_parser, element_name))
          {
            SEND_TO_CLIENT_OR_FAIL (XML_ERROR_UNAVAILABLE ("omp"));
            g_set_error (error,
                         G_MARKUP_ERROR,
                         G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Command Unavailable");
          }
        else if (strcasecmp ("AUTHENTICATE", element_name) == 0)
          {
            if (save_tasks ()) abort ();
            free_tasks ();
            free_credentials (&current_credentials);
            set_client_state (CLIENT_AUTHENTICATE);
          }
        else if (strcasecmp ("COMMANDS", element_name) == 0)
          {
            SEND_TO_CLIENT_OR_FAIL
             ("<commands_response"
              " status=\"" STATUS_OK "\" status_text=\"" STATUS_OK_TEXT "\">");
            set_client_state (CLIENT_AUTHENTIC_COMMANDS);
          }
        else if (strcasecmp ("CREATE_AGENT", element_name) == 0)
          {
            openvas_append_string (&create_agent_data->comment, "");
            openvas_append_string (&create_agent_data->installer, "");
            openvas_append_string (&create_agent_data->installer_filename, "");
            openvas_append_string (&create_agent_data->installer_signature, "");
            openvas_append_string (&create_agent_data->howto_install, "");
            openvas_append_string (&create_agent_data->howto_use, "");
            set_client_state (CLIENT_CREATE_AGENT);
          }
        else if (strcasecmp ("CREATE_CONFIG", element_name) == 0)
          {
            openvas_append_string (&create_config_data->comment, "");
            openvas_append_string (&create_config_data->name, "");
            set_client_state (CLIENT_CREATE_CONFIG);
          }
        else if (strcasecmp ("CREATE_ALERT", element_name) == 0)
          {
            create_alert_data->condition_data = make_array ();
            create_alert_data->event_data = make_array ();
            create_alert_data->method_data = make_array ();

            openvas_append_string (&create_alert_data->part_data, "");
            openvas_append_string (&create_alert_data->part_name, "");
            openvas_append_string (&create_alert_data->comment, "");
            openvas_append_string (&create_alert_data->name, "");
            openvas_append_string (&create_alert_data->condition, "");
            openvas_append_string (&create_alert_data->method, "");
            openvas_append_string (&create_alert_data->event, "");

            set_client_state (CLIENT_CREATE_ALERT);
          }
        else if (strcasecmp ("CREATE_FILTER", element_name) == 0)
          {
            openvas_append_string (&create_filter_data->comment, "");
            openvas_append_string (&create_filter_data->term, "");
            set_client_state (CLIENT_CREATE_FILTER);
          }
        else if (strcasecmp ("CREATE_GROUP", element_name) == 0)
          {
            openvas_append_string (&create_group_data->users, "");
            set_client_state (CLIENT_CREATE_GROUP);
          }
        else if (strcasecmp ("CREATE_ROLE", element_name) == 0)
          {
            openvas_append_string (&create_role_data->users, "");
            set_client_state (CLIENT_CREATE_ROLE);
          }
        else if (strcasecmp ("CREATE_LSC_CREDENTIAL", element_name) == 0)
          {
            openvas_append_string (&create_lsc_credential_data->comment, "");
            openvas_append_string (&create_lsc_credential_data->login, "");
            openvas_append_string (&create_lsc_credential_data->name, "");
            set_client_state (CLIENT_CREATE_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("CREATE_NOTE", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE);
        else if (strcasecmp ("CREATE_OVERRIDE", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE);
        else if (strcasecmp ("CREATE_PORT_LIST", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_LIST);
        else if (strcasecmp ("CREATE_PORT_RANGE", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_RANGE);
        else if (strcasecmp ("CREATE_PERMISSION", element_name) == 0)
          {
            openvas_append_string (&create_permission_data->comment, "");
            set_client_state (CLIENT_CREATE_PERMISSION);
          }
        else if (strcasecmp ("CREATE_REPORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT);
        else if (strcasecmp ("CREATE_REPORT_FORMAT", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_FORMAT);
        else if (strcasecmp ("CREATE_SCANNER", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER);
        else if (strcasecmp ("CREATE_SLAVE", element_name) == 0)
          {
            openvas_append_string (&create_slave_data->comment, "");
            openvas_append_string (&create_slave_data->password, "");
            set_client_state (CLIENT_CREATE_SLAVE);
          }
        else if (strcasecmp ("CREATE_SCHEDULE", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE);
        else if (strcasecmp ("CREATE_TAG", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_TAG);
          }
        else if (strcasecmp ("CREATE_TARGET", element_name) == 0)
          {
            openvas_append_string (&create_target_data->comment, "");
            set_client_state (CLIENT_CREATE_TARGET);
          }
        else if (strcasecmp ("CREATE_TASK", element_name) == 0)
          {
            create_task_data->task = make_task (NULL, NULL);
            create_task_data->alerts = make_array ();
            create_task_data->groups = make_array ();
            set_client_state (CLIENT_CREATE_TASK);
          }
        else if (strcasecmp ("CREATE_USER", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_USER);
            create_user_data->groups = make_array ();
            create_user_data->roles = make_array ();
            create_user_data->hosts_allow = 0;
            create_user_data->ifaces_allow = 0;
          }
        else if (strcasecmp ("DELETE_AGENT", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values,
                              "agent_id", &delete_agent_data->agent_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_agent_data->ultimate = strcmp (attribute, "0");
            else
              delete_agent_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_AGENT);
          }
        else if (strcasecmp ("DELETE_CONFIG", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values,
                              "config_id", &delete_config_data->config_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_config_data->ultimate = strcmp (attribute, "0");
            else
              delete_config_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_CONFIG);
          }
        else if (strcasecmp ("DELETE_ALERT", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values,
                              "alert_id",
                              &delete_alert_data->alert_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_alert_data->ultimate = strcmp (attribute, "0");
            else
              delete_alert_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_ALERT);
          }
        else if (strcasecmp ("DELETE_FILTER", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "filter_id",
                              &delete_filter_data->filter_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_filter_data->ultimate = strcmp (attribute, "0");
            else
              delete_filter_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_FILTER);
          }
        else if (strcasecmp ("DELETE_GROUP", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "group_id",
                              &delete_group_data->group_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_group_data->ultimate = strcmp (attribute, "0");
            else
              delete_group_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_GROUP);
          }
        else if (strcasecmp ("DELETE_LSC_CREDENTIAL", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values,
                              "lsc_credential_id",
                              &delete_lsc_credential_data->lsc_credential_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_lsc_credential_data->ultimate
               = strcmp (attribute, "0");
            else
              delete_lsc_credential_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("DELETE_NOTE", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "note_id",
                              &delete_note_data->note_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_note_data->ultimate = strcmp (attribute, "0");
            else
              delete_note_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_NOTE);
          }
        else if (strcasecmp ("DELETE_OVERRIDE", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "override_id",
                              &delete_override_data->override_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_override_data->ultimate = strcmp (attribute, "0");
            else
              delete_override_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_OVERRIDE);
          }
        else if (strcasecmp ("DELETE_PERMISSION", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values,
                              "permission_id",
                              &delete_permission_data->permission_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_permission_data->ultimate = strcmp (attribute, "0");
            else
              delete_permission_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_PERMISSION);
          }
        else if (strcasecmp ("DELETE_PORT_LIST", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "port_list_id",
                              &delete_port_list_data->port_list_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_port_list_data->ultimate = strcmp (attribute, "0");
            else
              delete_port_list_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_PORT_LIST);
          }
        else if (strcasecmp ("DELETE_PORT_RANGE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "port_range_id",
                              &delete_port_range_data->port_range_id);
            set_client_state (CLIENT_DELETE_PORT_RANGE);
          }
        else if (strcasecmp ("DELETE_REPORT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "report_id",
                              &delete_report_data->report_id);
            set_client_state (CLIENT_DELETE_REPORT);
          }
        else if (strcasecmp ("DELETE_REPORT_FORMAT", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "report_format_id",
                              &delete_report_format_data->report_format_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_report_format_data->ultimate = strcmp (attribute,
                                                            "0");
            else
              delete_report_format_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_REPORT_FORMAT);
          }
        else if (strcasecmp ("DELETE_ROLE", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "role_id",
                              &delete_role_data->role_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_role_data->ultimate = strcmp (attribute, "0");
            else
              delete_role_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_ROLE);
          }
        else if (strcasecmp ("DELETE_SCANNER", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values,
                              "scanner_id", &delete_scanner_data->scanner_id);
            if (find_attribute (attribute_names, attribute_values, "ultimate",
                                &attribute))
              delete_scanner_data->ultimate = strcmp (attribute, "0");
            else
              delete_scanner_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_SCANNER);
          }
        else if (strcasecmp ("DELETE_SCHEDULE", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "schedule_id",
                              &delete_schedule_data->schedule_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_schedule_data->ultimate = strcmp (attribute, "0");
            else
              delete_schedule_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_SCHEDULE);
          }
        else if (strcasecmp ("DELETE_SLAVE", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "slave_id",
                              &delete_slave_data->slave_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_slave_data->ultimate = strcmp (attribute, "0");
            else
              delete_slave_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_SLAVE);
          }
        else if (strcasecmp ("DELETE_TAG", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "tag_id",
                              &delete_tag_data->tag_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_tag_data->ultimate = strcmp (attribute, "0");
            else
              delete_tag_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_TAG);
          }
        else if (strcasecmp ("DELETE_TARGET", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "target_id",
                              &delete_target_data->target_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_target_data->ultimate = strcmp (attribute, "0");
            else
              delete_target_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_TARGET);
          }
        else if (strcasecmp ("DELETE_TASK", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "task_id",
                              &delete_task_data->task_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_task_data->ultimate = strcmp (attribute, "0");
            else
              delete_task_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_TASK);
          }
        else if (strcasecmp ("DELETE_USER", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "name",
                              &delete_user_data->name);
            append_attribute (attribute_names, attribute_values, "user_id",
                              &delete_user_data->user_id);
            if (find_attribute (attribute_names, attribute_values,
                                "ultimate", &attribute))
              delete_user_data->ultimate = strcmp (attribute, "0");
            else
              delete_user_data->ultimate = 0;
            set_client_state (CLIENT_DELETE_USER);
          }
        else if (strcasecmp ("DESCRIBE_AUTH", element_name) == 0)
          set_client_state (CLIENT_DESCRIBE_AUTH);
        else if (strcasecmp ("DESCRIBE_CERT", element_name) == 0)
          set_client_state (CLIENT_DESCRIBE_CERT);
        else if (strcasecmp ("DESCRIBE_FEED", element_name) == 0)
          set_client_state (CLIENT_DESCRIBE_FEED);
        else if (strcasecmp ("DESCRIBE_SCAP", element_name) == 0)
          set_client_state (CLIENT_DESCRIBE_SCAP);
        else if (strcasecmp ("EMPTY_TRASHCAN", element_name) == 0)
          set_client_state (CLIENT_EMPTY_TRASHCAN);
        else if (strcasecmp ("GET_AGENTS", element_name) == 0)
          {
            get_data_parse_attributes (&get_agents_data->get, "agent",
                                       attribute_names,
                                       attribute_values);
            append_attribute (attribute_names, attribute_values, "format",
                              &get_agents_data->format);
            set_client_state (CLIENT_GET_AGENTS);
          }
        else if (strcasecmp ("GET_AGGREGATES", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "type",
                              &get_aggregates_data->type);

            if (get_aggregates_data->type
                && strcasecmp (get_aggregates_data->type, "info") == 0)
            {
              append_attribute (attribute_names, attribute_values, "info_type",
                                &get_aggregates_data->subtype);
            }
            append_attribute (attribute_names, attribute_values, "data_column",
                              &get_aggregates_data->data_column);

            append_attribute (attribute_names, attribute_values, "group_column",
                              &get_aggregates_data->group_column);

            get_data_parse_attributes (&get_aggregates_data->get,
                                       get_aggregates_data->type
                                        ? get_aggregates_data->type
                                        : "",
                                       attribute_names,
                                       attribute_values);
            set_client_state (CLIENT_GET_AGGREGATES);
          }
        else if (strcasecmp ("GET_CONFIGS", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_configs_data->get,
                                       "config",
                                       attribute_names,
                                       attribute_values);

            if (find_attribute (attribute_names, attribute_values,
                                "tasks", &attribute))
              get_configs_data->tasks = strcmp (attribute, "0");
            else
              get_configs_data->tasks = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "families", &attribute))
              get_configs_data->families = strcmp (attribute, "0");
            else
              get_configs_data->families = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "preferences", &attribute))
              get_configs_data->preferences = strcmp (attribute, "0");
            else
              get_configs_data->preferences = 0;

            set_client_state (CLIENT_GET_CONFIGS);
          }
        else if (strcasecmp ("GET_ALERTS", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_alerts_data->get,
                                       "alert",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "tasks", &attribute))
              get_alerts_data->tasks = strcmp (attribute, "0");
            else
              get_alerts_data->tasks = 0;

            set_client_state (CLIENT_GET_ALERTS);
          }
        else if (strcasecmp ("GET_FILTERS", element_name) == 0)
          {
            const gchar* attribute;
            get_data_parse_attributes (&get_filters_data->get, "filter",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "alerts", &attribute))
              get_filters_data->alerts = strcmp (attribute, "0");
            else
              get_filters_data->alerts = 0;
            set_client_state (CLIENT_GET_FILTERS);
          }
        else if (strcasecmp ("GET_GROUPS", element_name) == 0)
          {
            get_data_parse_attributes (&get_groups_data->get, "group",
                                       attribute_names,
                                       attribute_values);
            set_client_state (CLIENT_GET_GROUPS);
          }
        else if (strcasecmp ("GET_LSC_CREDENTIALS", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_lsc_credentials_data->get,
                                       "lsc_credential",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "targets", &attribute))
              get_lsc_credentials_data->targets = strcmp (attribute, "0");
            else
              get_lsc_credentials_data->targets = 0;
            append_attribute (attribute_names, attribute_values, "format",
                              &get_lsc_credentials_data->format);
            set_client_state (CLIENT_GET_LSC_CREDENTIALS);
          }
        else if (strcasecmp ("GET_NOTES", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_notes_data->get, "note",
                                       attribute_names,
                                       attribute_values);

            append_attribute (attribute_names, attribute_values, "note_id",
                              &get_notes_data->note_id);

            append_attribute (attribute_names, attribute_values, "nvt_oid",
                              &get_notes_data->nvt_oid);

            append_attribute (attribute_names, attribute_values, "task_id",
                              &get_notes_data->task_id);

            if (find_attribute (attribute_names, attribute_values,
                                "result", &attribute))
              get_notes_data->result = strcmp (attribute, "0");
            else
              get_notes_data->result = 0;

            set_client_state (CLIENT_GET_NOTES);
          }
        else if (strcasecmp ("GET_NVT_FEED_VERSION", element_name) == 0)
            set_client_state (CLIENT_GET_NVT_FEED_VERSION);
        else if (strcasecmp ("GET_NVTS", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "nvt_oid",
                              &get_nvts_data->nvt_oid);
            append_attribute (attribute_names, attribute_values, "config_id",
                              &get_nvts_data->config_id);
            if (find_attribute (attribute_names, attribute_values,
                                "details", &attribute))
              get_nvts_data->details = strcmp (attribute, "0");
            else
              get_nvts_data->details = 0;
            append_attribute (attribute_names, attribute_values, "family",
                              &get_nvts_data->family);
            if (find_attribute (attribute_names, attribute_values,
                                "preferences", &attribute))
              get_nvts_data->preferences = strcmp (attribute, "0");
            else
              get_nvts_data->preferences = 0;
            if (find_attribute (attribute_names, attribute_values,
                                "preference_count", &attribute))
              get_nvts_data->preference_count = strcmp (attribute, "0");
            else
              get_nvts_data->preference_count = 0;
            if (find_attribute (attribute_names, attribute_values,
                                "timeout", &attribute))
              get_nvts_data->timeout = strcmp (attribute, "0");
            else
              get_nvts_data->timeout = 0;
            append_attribute (attribute_names, attribute_values, "sort_field",
                              &get_nvts_data->sort_field);
            if (find_attribute (attribute_names, attribute_values,
                                "sort_order", &attribute))
              get_nvts_data->sort_order = strcmp (attribute,
                                                         "descending");
            else
              get_nvts_data->sort_order = 1;
            set_client_state (CLIENT_GET_NVTS);
          }
        else if (strcasecmp ("GET_NVT_FAMILIES", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values,
                                "sort_order", &attribute))
              get_nvt_families_data->sort_order = strcmp (attribute,
                                                          "descending");
            else
              get_nvt_families_data->sort_order = 1;
            set_client_state (CLIENT_GET_NVT_FAMILIES);
          }
        else if (strcasecmp ("GET_OVERRIDES", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_overrides_data->get, "override",
                                       attribute_names,
                                       attribute_values);

            append_attribute (attribute_names, attribute_values, "override_id",
                              &get_overrides_data->override_id);

            append_attribute (attribute_names, attribute_values, "nvt_oid",
                              &get_overrides_data->nvt_oid);

            append_attribute (attribute_names, attribute_values, "task_id",
                              &get_overrides_data->task_id);

            if (find_attribute (attribute_names, attribute_values,
                                "result", &attribute))
              get_overrides_data->result = strcmp (attribute, "0");
            else
              get_overrides_data->result = 0;

            set_client_state (CLIENT_GET_OVERRIDES);
          }
        else if (strcasecmp ("GET_PORT_LISTS", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_port_lists_data->get,
                                       "port_list",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "targets", &attribute))
              get_port_lists_data->targets = strcmp (attribute, "0");
            else
              get_port_lists_data->targets = 0;
            set_client_state (CLIENT_GET_PORT_LISTS);
          }
        else if (strcasecmp ("GET_PERMISSIONS", element_name) == 0)
          {
            get_data_parse_attributes (&get_permissions_data->get, "permission",
                                       attribute_names,
                                       attribute_values);
            append_attribute (attribute_names, attribute_values, "resource_id",
                              &get_permissions_data->resource_id);
            set_client_state (CLIENT_GET_PERMISSIONS);
          }
        else if (strcasecmp ("GET_PREFERENCES", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "nvt_oid",
                              &get_preferences_data->nvt_oid);
            append_attribute (attribute_names, attribute_values, "config_id",
                              &get_preferences_data->config_id);
            append_attribute (attribute_names, attribute_values, "preference",
                              &get_preferences_data->preference);
            set_client_state (CLIENT_GET_PREFERENCES);
          }
        else if (strcasecmp ("GET_REPORTS", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_reports_data->get, "report",
                                       attribute_names,
                                       attribute_values);

            get_data_parse_attributes (&get_reports_data->report_get, "report",
                                       attribute_names,
                                       attribute_values);

            /* Special case details with default 1, for backward
             * compatibility. */
            if (find_attribute (attribute_names, attribute_values,
                                "details", &attribute))
              get_reports_data->report_get.details = strcmp (attribute, "0");
            else
              get_reports_data->report_get.details = 1;
            get_reports_data->get.details
             = get_reports_data->report_get.details;

            g_free (get_reports_data->report_get.filt_id);
            get_reports_data->report_get.filt_id = NULL;
            append_attribute (attribute_names, attribute_values,
                              "report_filt_id",
                              &get_reports_data->report_get.filt_id);

            g_free (get_reports_data->report_get.filter);
            get_reports_data->report_get.filter = NULL;
            append_attribute (attribute_names, attribute_values,
                              "report_filter",
                              &get_reports_data->report_get.filter);

            append_attribute (attribute_names, attribute_values, "report_id",
                              &get_reports_data->report_id);

            append_attribute (attribute_names, attribute_values,
                              "delta_report_id",
                              &get_reports_data->delta_report_id);

            append_attribute (attribute_names, attribute_values, "alert_id",
                              &get_reports_data->alert_id);

            append_attribute (attribute_names, attribute_values, "format_id",
                              &get_reports_data->format_id);

            if (find_attribute (attribute_names, attribute_values,
                                "first_result", &attribute))
              /* Subtract 1 to switch from 1 to 0 indexing. */
              get_reports_data->first_result = atoi (attribute) - 1;
            else
              get_reports_data->first_result = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "host_first_result", &attribute))
              /* Subtract 1 to switch from 1 to 0 indexing. */
              get_reports_data->host_first_result = atoi (attribute) - 1;
            else
              get_reports_data->host_first_result = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "max_results", &attribute))
              get_reports_data->max_results = atoi (attribute);
            else
              get_reports_data->max_results = -1;

            if (find_attribute (attribute_names, attribute_values,
                                "host_max_results", &attribute))
              get_reports_data->host_max_results = atoi (attribute);
            else
              get_reports_data->host_max_results = -1;

            append_attribute (attribute_names, attribute_values, "sort_field",
                              &get_reports_data->sort_field);

            if (find_attribute (attribute_names, attribute_values,
                                "sort_order", &attribute))
              get_reports_data->sort_order = strcmp (attribute, "descending");
            else
              {
                if (get_reports_data->sort_field == NULL
                    || (strcmp (get_reports_data->sort_field, "type") == 0))
                  /* Normally it makes more sense to order type descending. */
                  get_reports_data->sort_order = 0;
                else
                  get_reports_data->sort_order = 1;
              }

            append_attribute (attribute_names, attribute_values, "levels",
                              &get_reports_data->levels);

            append_attribute (attribute_names, attribute_values, "host_levels",
                              &get_reports_data->host_levels);

            append_attribute (attribute_names, attribute_values,
                              "host_search_phrase",
                              &get_reports_data->host_search_phrase);

            append_attribute (attribute_names, attribute_values, "delta_states",
                              &get_reports_data->delta_states);

            append_attribute (attribute_names, attribute_values,
                              "search_phrase",
                              &get_reports_data->search_phrase);

            if (find_attribute (attribute_names, attribute_values,
                                "autofp", &attribute))
              get_reports_data->autofp = strcmp (attribute, "0");
            else
              get_reports_data->autofp = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "notes", &attribute))
              get_reports_data->notes = strcmp (attribute, "0");
            else
              get_reports_data->notes = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "notes_details", &attribute))
              get_reports_data->notes_details = strcmp (attribute, "0");
            else
              get_reports_data->notes_details = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "overrides", &attribute))
              get_reports_data->overrides = strcmp (attribute, "0");
            else
              get_reports_data->overrides = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "overrides_details", &attribute))
              get_reports_data->overrides_details = strcmp (attribute, "0");
            else
              get_reports_data->overrides_details = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "apply_overrides", &attribute))
              get_reports_data->apply_overrides = strcmp (attribute, "0");
            else
              get_reports_data->apply_overrides = 0;

            append_attribute (attribute_names, attribute_values,
                              "min_cvss_base",
                              &get_reports_data->min_cvss_base);

            append_attribute (attribute_names, attribute_values,
                              "min_qod",
                              &get_reports_data->min_qod);

            if (find_attribute (attribute_names, attribute_values,
                                "result_hosts_only", &attribute))
              get_reports_data->result_hosts_only = strcmp (attribute, "0");
            else
              get_reports_data->result_hosts_only = 1;

            if (find_attribute (attribute_names, attribute_values,
                                "type", &attribute))
              openvas_append_string (&get_reports_data->type, attribute);
            else
              get_reports_data->type = g_strdup ("scan");

            append_attribute (attribute_names,
                              attribute_values,
                              "host",
                              &get_reports_data->host);

            append_attribute (attribute_names,
                              attribute_values,
                              "pos",
                              &get_reports_data->pos);

            if (find_attribute (attribute_names, attribute_values,
                                "ignore_pagination", &attribute))
              get_reports_data->ignore_pagination = atoi (attribute);
            else
              get_reports_data->ignore_pagination = 0;

            append_attribute (attribute_names, attribute_values,
                              "timezone",
                              &get_reports_data->timezone);

            set_client_state (CLIENT_GET_REPORTS);
          }
        else if (strcasecmp ("GET_REPORT_FORMATS", element_name) == 0)
          {
            const gchar* attribute;

            get_data_parse_attributes (&get_report_formats_data->get,
                                       "report_format",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "alerts", &attribute))
              get_report_formats_data->alerts = strcmp (attribute, "0");
            else
              get_report_formats_data->alerts = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "params", &attribute))
              get_report_formats_data->params = strcmp (attribute, "0");
            else
              get_report_formats_data->params = 0;

            set_client_state (CLIENT_GET_REPORT_FORMATS);
          }
        else if (strcasecmp ("GET_RESULTS", element_name) == 0)
          {
            const gchar* attribute;
            get_data_parse_attributes (&get_results_data->get,
                                       "result",
                                       attribute_names,
                                       attribute_values);

            append_attribute (attribute_names, attribute_values, "task_id",
                              &get_results_data->task_id);

            if (find_attribute (attribute_names, attribute_values,
                                "notes", &attribute))
              get_results_data->notes = strcmp (attribute, "0");
            else
              get_results_data->notes = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "notes_details", &attribute))
              get_results_data->notes_details = strcmp (attribute, "0");
            else
              get_results_data->notes_details = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "overrides", &attribute))
              get_results_data->overrides = strcmp (attribute, "0");
            else
              get_results_data->overrides = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "overrides_details", &attribute))
              get_results_data->overrides_details = strcmp (attribute, "0");
            else
              get_results_data->overrides_details = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "apply_overrides", &attribute))
              {
                get_results_data->apply_overrides = strcmp (attribute, "0");
                get_results_data->apply_overrides_set = 1;
              }
            else
              {
                get_results_data->apply_overrides = 0;
                get_results_data->apply_overrides_set = 0;
              }

            if (find_attribute (attribute_names, attribute_values,
                                "autofp", &attribute))
              get_results_data->autofp = strcmp (attribute, "0");
            else
              get_results_data->autofp = 0;

            set_client_state (CLIENT_GET_RESULTS);
          }
        else if (strcasecmp ("GET_ROLES", element_name) == 0)
          {
            get_data_parse_attributes (&get_roles_data->get, "role",
                                       attribute_names,
                                       attribute_values);
            set_client_state (CLIENT_GET_ROLES);
          }
        else if (strcasecmp ("GET_SCANNERS", element_name) == 0)
          {
            get_data_parse_attributes (&get_scanners_data->get, "scanner",
                                       attribute_names, attribute_values);
            set_client_state (CLIENT_GET_SCANNERS);
          }
        else if (strcasecmp ("GET_SCHEDULES", element_name) == 0)
          {
            const gchar *attribute;
            get_data_parse_attributes (&get_schedules_data->get, "schedule",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "tasks", &attribute))
              get_schedules_data->tasks = strcmp (attribute, "0");
            else
              get_schedules_data->tasks = 0;
            set_client_state (CLIENT_GET_SCHEDULES);
          }
        else if (strcasecmp ("GET_SETTINGS", element_name) == 0)
          {
            const gchar* attribute;

            append_attribute (attribute_names, attribute_values, "setting_id",
                              &get_settings_data->setting_id);

            append_attribute (attribute_names, attribute_values, "filter",
                              &get_settings_data->filter);

            if (find_attribute (attribute_names, attribute_values,
                                "first", &attribute))
              /* Subtract 1 to switch from 1 to 0 indexing. */
              get_settings_data->first = atoi (attribute) - 1;
            else
              get_settings_data->first = 0;
            if (get_settings_data->first < 0)
              get_settings_data->first = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "max", &attribute))
              get_settings_data->max = atoi (attribute);
            else
              get_settings_data->max = -1;
            if (get_settings_data->max < 1)
              get_settings_data->max = -1;

            append_attribute (attribute_names, attribute_values, "sort_field",
                              &get_settings_data->sort_field);

            if (find_attribute (attribute_names, attribute_values,
                                "sort_order", &attribute))
              get_settings_data->sort_order = strcmp (attribute, "descending");
            else
              get_settings_data->sort_order = 1;

            set_client_state (CLIENT_GET_SETTINGS);
          }
        else if (strcasecmp ("GET_SLAVES", element_name) == 0)
          {
            const gchar* attribute;
            get_data_parse_attributes (&get_slaves_data->get, "slave",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "tasks", &attribute))
              get_slaves_data->tasks = strcmp (attribute, "0");
            else
              get_slaves_data->tasks = 0;
            set_client_state (CLIENT_GET_SLAVES);
          }
        else if (strcasecmp ("GET_TAGS", element_name) == 0)
          {
            const gchar* attribute;
            get_data_parse_attributes (&get_tags_data->get, "tag",
                                       attribute_names,
                                       attribute_values);

            if (find_attribute (attribute_names, attribute_values,
                                "names_only", &attribute))
              get_tags_data->names_only = strcmp (attribute, "0");
            else
              get_tags_data->names_only = 0;

            set_client_state (CLIENT_GET_TAGS);
          }
        else if (strcasecmp ("GET_SYSTEM_REPORTS", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "name",
                              &get_system_reports_data->name);
            append_attribute (attribute_names, attribute_values, "duration",
                              &get_system_reports_data->duration);
            append_attribute (attribute_names, attribute_values, "slave_id",
                              &get_system_reports_data->slave_id);
            if (find_attribute (attribute_names, attribute_values,
                                "brief", &attribute))
              get_system_reports_data->brief = strcmp (attribute, "0");
            else
              get_system_reports_data->brief = 0;
            set_client_state (CLIENT_GET_SYSTEM_REPORTS);
          }
        else if (strcasecmp ("GET_TARGETS", element_name) == 0)
          {
            const gchar *attribute;
            get_data_parse_attributes (&get_targets_data->get, "target",
                                       attribute_names,
                                       attribute_values);
            if (find_attribute (attribute_names, attribute_values,
                                "tasks", &attribute))
              get_targets_data->tasks = strcmp (attribute, "0");
            else
              get_targets_data->tasks = 0;
            set_client_state (CLIENT_GET_TARGETS);
          }
        else if (strcasecmp ("GET_TASKS", element_name) == 0)
          {
            get_data_parse_attributes (&get_tasks_data->get, "task",
                                       attribute_names,
                                       attribute_values);

            set_client_state (CLIENT_GET_TASKS);
          }
        else if (strcasecmp ("GET_USERS", element_name) == 0)
          {
            get_data_parse_attributes (&get_users_data->get, "user",
                                       attribute_names,
                                       attribute_values);
            set_client_state (CLIENT_GET_USERS);
          }
        else if (strcasecmp ("GET_INFO", element_name) == 0)
          {
            const gchar* attribute;
            const gchar* typebuf;
            get_data_parse_attributes (&get_info_data->get, "info",
                                       attribute_names,
                                       attribute_values);
            append_attribute (attribute_names, attribute_values, "name",
                              &get_info_data->name);
            if (find_attribute (attribute_names, attribute_values,
                                "details", &attribute))
              get_info_data->details = strcmp (attribute, "0");
            else
              get_info_data->details = 0;

            if (find_attribute (attribute_names, attribute_values,
                                "type", &typebuf))
              get_info_data->type = g_ascii_strdown (typebuf, -1);
            set_client_state (CLIENT_GET_INFO);
          }
        else if (strcasecmp ("GET_VERSION", element_name) == 0)
          set_client_state (CLIENT_GET_VERSION_AUTHENTIC);
        else if (strcasecmp ("HELP", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "format",
                              &help_data->format);
            append_attribute (attribute_names, attribute_values, "type",
                              &help_data->type);
            set_client_state (CLIENT_HELP);
          }
        else if (strcasecmp ("MODIFY_AGENT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "agent_id",
                              &modify_agent_data->agent_id);
            set_client_state (CLIENT_MODIFY_AGENT);
          }
        else if (strcasecmp ("MODIFY_ALERT", element_name) == 0)
          {
            modify_alert_data->event_data = make_array ();
            openvas_append_string (&modify_alert_data->event, "");
            modify_alert_data->condition_data = make_array ();
            openvas_append_string (&modify_alert_data->condition, "");
            modify_alert_data->method_data = make_array ();
            openvas_append_string (&modify_alert_data->method, "");

            append_attribute (attribute_names, attribute_values, "alert_id",
                              &modify_alert_data->alert_id);
            set_client_state (CLIENT_MODIFY_ALERT);
          }
        else if (strcasecmp ("MODIFY_AUTH", element_name) == 0)
          set_client_state (CLIENT_MODIFY_AUTH);
        else if (strcasecmp ("MODIFY_CONFIG", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "config_id",
                              &modify_config_data->config_id);
            set_client_state (CLIENT_MODIFY_CONFIG);
          }
        else if (strcasecmp ("MODIFY_FILTER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "filter_id",
                              &modify_filter_data->filter_id);
            set_client_state (CLIENT_MODIFY_FILTER);
          }
        else if (strcasecmp ("MODIFY_GROUP", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "group_id",
                              &modify_group_data->group_id);
            set_client_state (CLIENT_MODIFY_GROUP);
          }
        else if (strcasecmp ("MODIFY_PORT_LIST", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values,
                              "port_list_id",
                              &modify_port_list_data->port_list_id);
            set_client_state (CLIENT_MODIFY_PORT_LIST);
          }
        else if (strcasecmp ("MODIFY_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values,
                              "lsc_credential_id",
                              &modify_lsc_credential_data->lsc_credential_id);
            set_client_state (CLIENT_MODIFY_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("MODIFY_NOTE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "note_id",
                              &modify_note_data->note_id);
            set_client_state (CLIENT_MODIFY_NOTE);
          }
        else if (strcasecmp ("MODIFY_OVERRIDE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "override_id",
                              &modify_override_data->override_id);
            set_client_state (CLIENT_MODIFY_OVERRIDE);
          }
        else if (strcasecmp ("MODIFY_PERMISSION", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values,
                              "permission_id",
                              &modify_permission_data->permission_id);
            set_client_state (CLIENT_MODIFY_PERMISSION);
          }
        else if (strcasecmp ("MODIFY_REPORT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "report_id",
                              &modify_report_data->report_id);
            set_client_state (CLIENT_MODIFY_REPORT);
          }
        else if (strcasecmp ("MODIFY_REPORT_FORMAT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values,
                              "report_format_id",
                              &modify_report_format_data->report_format_id);
            set_client_state (CLIENT_MODIFY_REPORT_FORMAT);
          }
        else if (strcasecmp ("MODIFY_ROLE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "role_id",
                              &modify_role_data->role_id);
            set_client_state (CLIENT_MODIFY_ROLE);
          }
        else if (strcasecmp ("MODIFY_SCANNER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "scanner_id",
                              &modify_scanner_data->scanner_id);
            set_client_state (CLIENT_MODIFY_SCANNER);
          }
        else if (strcasecmp ("MODIFY_SCHEDULE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "schedule_id",
                              &modify_schedule_data->schedule_id);
            set_client_state (CLIENT_MODIFY_SCHEDULE);
          }
        else if (strcasecmp ("MODIFY_SETTING", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values,
                              "setting_id",
                              &modify_setting_data->setting_id);
            set_client_state (CLIENT_MODIFY_SETTING);
          }
        else if (strcasecmp ("MODIFY_SLAVE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "slave_id",
                              &modify_slave_data->slave_id);
            set_client_state (CLIENT_MODIFY_SLAVE);
          }
        else if (strcasecmp ("MODIFY_TAG", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "tag_id",
                              &modify_tag_data->tag_id);
            set_client_state (CLIENT_MODIFY_TAG);
          }
        else if (strcasecmp ("MODIFY_TARGET", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "target_id",
                              &modify_target_data->target_id);
            set_client_state (CLIENT_MODIFY_TARGET);
          }
        else if (strcasecmp ("MODIFY_TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "task_id",
                              &modify_task_data->task_id);
            modify_task_data->alerts = make_array ();
            modify_task_data->groups = make_array ();
            set_client_state (CLIENT_MODIFY_TASK);
          }
        else if (strcasecmp ("MODIFY_USER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "user_id",
                              &modify_user_data->user_id);
            set_client_state (CLIENT_MODIFY_USER);
          }
        else if (strcasecmp ("RESTORE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &restore_data->id);
            set_client_state (CLIENT_RESTORE);
          }
        else if (strcasecmp ("RESUME_TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "task_id",
                              &resume_task_data->task_id);
            set_client_state (CLIENT_RESUME_TASK);
          }
        else if (strcasecmp ("RUN_WIZARD", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "name",
                              &run_wizard_data->name);
            append_attribute (attribute_names, attribute_values, "read_only",
                              &run_wizard_data->read_only);
            set_client_state (CLIENT_RUN_WIZARD);
          }
        else if (strcasecmp ("START_TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "task_id",
                              &start_task_data->task_id);
            set_client_state (CLIENT_START_TASK);
          }
        else if (strcasecmp ("STOP_TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "task_id",
                              &stop_task_data->task_id);
            set_client_state (CLIENT_STOP_TASK);
          }
        else if (strcasecmp ("SYNC_CERT", element_name) == 0)
          set_client_state (CLIENT_SYNC_CERT);
        else if (strcasecmp ("SYNC_FEED", element_name) == 0)
          set_client_state (CLIENT_SYNC_FEED);
        else if (strcasecmp ("SYNC_SCAP", element_name) == 0)
          set_client_state (CLIENT_SYNC_SCAP);
        else if (strcasecmp ("TEST_ALERT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values,
                              "alert_id",
                              &test_alert_data->alert_id);
            set_client_state (CLIENT_TEST_ALERT);
          }
        else if (strcasecmp ("VERIFY_AGENT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "agent_id",
                              &verify_agent_data->agent_id);
            set_client_state (CLIENT_VERIFY_AGENT);
          }
        else if (strcasecmp ("VERIFY_REPORT_FORMAT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "report_format_id",
                              &verify_report_format_data->report_format_id);
            set_client_state (CLIENT_VERIFY_REPORT_FORMAT);
          }
        else if (strcasecmp ("VERIFY_SCANNER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "scanner_id",
                              &verify_scanner_data->scanner_id);
            set_client_state (CLIENT_VERIFY_SCANNER);
          }
        else
          {
            if (send_to_client (XML_ERROR_SYNTAX ("omp", "Bogus command name"),
                                write_to_client,
                                write_to_client_data))
              {
                error_send_to_client (error);
                return;
              }
            g_set_error (error,
                         G_MARKUP_ERROR,
                         G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Error");
          }
        break;

      case CLIENT_AUTHENTICATE:
        if (strcasecmp ("CREDENTIALS", element_name) == 0)
          {
            /* Init, so it's the empty string when the entity is empty. */
            append_to_credentials_password (&current_credentials, "", 0);
            set_client_state (CLIENT_AUTHENTICATE_CREDENTIALS);
          }
        ELSE_ERROR ("authenticate");

      case CLIENT_AUTHENTICATE_CREDENTIALS:
        if (strcasecmp ("USERNAME", element_name) == 0)
          set_client_state (CLIENT_AUTHENTICATE_CREDENTIALS_USERNAME);
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          set_client_state (CLIENT_AUTHENTICATE_CREDENTIALS_PASSWORD);
        ELSE_ERROR ("authenticate");

      case CLIENT_CREATE_SCANNER:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_COPY);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_NAME);
        else if (strcasecmp ("HOST", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_HOST);
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_PORT);
        else if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_TYPE);
        else if (strcasecmp ("CA_PUB", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_CA_PUB);
        else if (strcasecmp ("KEY_PUB", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_KEY_PUB);
        else if (strcasecmp ("KEY_PRIV", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCANNER_KEY_PRIV);
        ELSE_ERROR ("create_scanner");

      case CLIENT_CREATE_SCHEDULE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_COPY);
        else if (strcasecmp ("DURATION", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_DURATION);
        else if (strcasecmp ("FIRST_TIME", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_FIRST_TIME);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_NAME);
        else if (strcasecmp ("PERIOD", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_PERIOD);
        else if (strcasecmp ("TIMEZONE", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_TIMEZONE);
        ELSE_ERROR ("create_schedule");

      case CLIENT_CREATE_SCHEDULE_FIRST_TIME:
        if (strcasecmp ("DAY_OF_MONTH", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_FIRST_TIME_DAY_OF_MONTH);
        else if (strcasecmp ("HOUR", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_FIRST_TIME_HOUR);
        else if (strcasecmp ("MINUTE", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_FIRST_TIME_MINUTE);
        else if (strcasecmp ("MONTH", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_FIRST_TIME_MONTH);
        else if (strcasecmp ("YEAR", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_FIRST_TIME_YEAR);
        ELSE_ERROR ("create_schedule");

      case CLIENT_CREATE_SCHEDULE_DURATION:
        if (strcasecmp ("UNIT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_DURATION_UNIT);
        ELSE_ERROR ("create_schedule");

      case CLIENT_CREATE_SCHEDULE_PERIOD:
        if (strcasecmp ("UNIT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SCHEDULE_PERIOD_UNIT);
        ELSE_ERROR ("create_schedule");

      case CLIENT_MODIFY_AGENT:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_agent_data->comment, "");
            set_client_state (CLIENT_MODIFY_AGENT_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_agent_data->name, "");
            set_client_state (CLIENT_MODIFY_AGENT_NAME);
          }
        ELSE_ERROR ("modify_agent");

      case CLIENT_MODIFY_ALERT:
        if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_alert_data->name, "");
            set_client_state (CLIENT_MODIFY_ALERT_NAME);
          }
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_alert_data->comment, "");
            set_client_state (CLIENT_MODIFY_ALERT_COMMENT);
          }
        else if (strcasecmp ("EVENT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_EVENT);
        else if (strcasecmp ("FILTER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_alert_data->filter_id);
            set_client_state (CLIENT_MODIFY_ALERT_FILTER);
          }
        else if (strcasecmp ("CONDITION", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_CONDITION);
        else if (strcasecmp ("METHOD", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_METHOD);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_ALERT_EVENT:
        if (strcasecmp ("DATA", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_EVENT_DATA);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_ALERT_EVENT_DATA:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_EVENT_DATA_NAME);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_ALERT_CONDITION:
        if (strcasecmp ("DATA", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_CONDITION_DATA);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_ALERT_CONDITION_DATA:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_CONDITION_DATA_NAME);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_ALERT_METHOD:
        if (strcasecmp ("DATA", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_METHOD_DATA);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_ALERT_METHOD_DATA:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_ALERT_METHOD_DATA_NAME);
        ELSE_ERROR ("modify_alert");

      case CLIENT_MODIFY_AUTH:
        if (strcasecmp ("GROUP", element_name) == 0)
          {
            const gchar* attribute;
            auth_group_t *new_group;

            new_group = g_malloc0 (sizeof (auth_group_t));
            if (find_attribute (attribute_names, attribute_values, "name",
                                &attribute))
              new_group->group_name = g_strdup (attribute);
            modify_auth_data->groups =
              g_slist_prepend (modify_auth_data->groups, new_group);
            set_client_state (CLIENT_MODIFY_AUTH_GROUP);
          }
        ELSE_ERROR ("modify_auth");

      case CLIENT_MODIFY_AUTH_GROUP:
        if (strcasecmp ("AUTH_CONF_SETTING", element_name) == 0)
          {
            set_client_state (CLIENT_MODIFY_AUTH_GROUP_AUTHCONFSETTING);
            auth_conf_setting_t *setting =
              auth_conf_setting_from_xml (element_name,
                                          attribute_names,
                                          attribute_values);
            modify_auth_data->curr_group_settings =
              g_slist_prepend (modify_auth_data->curr_group_settings, setting);
          }
        ELSE_ERROR ("modify_auth");

      case CLIENT_MODIFY_AUTH_GROUP_AUTHCONFSETTING:
        {
          /* AUTH_CONF_SETTING should not have any children. */
          if (send_element_error_to_client ("auth_conf_setting", element_name,
                                            write_to_client,
                                            write_to_client_data))
            {
              error_send_to_client (error);
              return;
            }
          modify_auth_data_reset (modify_auth_data);
          set_client_state (CLIENT_AUTHENTIC);
          g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                       "Error");
          break;
        }

      case CLIENT_MODIFY_CONFIG:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_free_string_var (&modify_config_data->comment);
            openvas_append_string (&modify_config_data->comment, "");
            set_client_state (CLIENT_MODIFY_CONFIG_COMMENT);
          }
        else if (strcasecmp ("FAMILY_SELECTION", element_name) == 0)
          {
            modify_config_data->families_growing_all = make_array ();
            modify_config_data->families_static_all = make_array ();
            modify_config_data->families_growing_empty = make_array ();
            /* For GROWING entity, in case missing. */
            modify_config_data->family_selection_growing = 0;
            set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_CONFIG_NAME);
        else if (strcasecmp ("NVT_SELECTION", element_name) == 0)
          {
            modify_config_data->nvt_selection = make_array ();
            set_client_state (CLIENT_MODIFY_CONFIG_NVT_SELECTION);
          }
        else if (strcasecmp ("PREFERENCE", element_name) == 0)
          {
            openvas_free_string_var (&modify_config_data->preference_name);
            openvas_free_string_var (&modify_config_data->preference_nvt_oid);
            openvas_free_string_var (&modify_config_data->preference_value);
            set_client_state (CLIENT_MODIFY_CONFIG_PREFERENCE);
          }
        ELSE_ERROR ("modify_config");

      case CLIENT_MODIFY_CONFIG_NVT_SELECTION:
        if (strcasecmp ("FAMILY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_CONFIG_NVT_SELECTION_FAMILY);
        else if (strcasecmp ("NVT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "oid",
                              &modify_config_data->nvt_selection_nvt_oid);
            set_client_state (CLIENT_MODIFY_CONFIG_NVT_SELECTION_NVT);
          }
        ELSE_ERROR ("modify_config");

      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION:
        if (strcasecmp ("FAMILY", element_name) == 0)
          {
            /* For ALL entity, in case missing. */
            modify_config_data->family_selection_family_all = 0;
            /* For GROWING entity, in case missing. */
            modify_config_data->family_selection_family_growing = 0;
            set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY);
          }
        else if (strcasecmp ("GROWING", element_name) == 0)
          set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_GROWING);
        ELSE_ERROR ("modify_config");

      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY:
        if (strcasecmp ("ALL", element_name) == 0)
          set_client_state
           (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_ALL);
        else if (strcasecmp ("GROWING", element_name) == 0)
          set_client_state
           (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_GROWING);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_NAME);
        ELSE_ERROR ("modify_config");

      case CLIENT_MODIFY_CONFIG_PREFERENCE:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_CONFIG_PREFERENCE_NAME);
        else if (strcasecmp ("NVT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "oid",
                              &modify_config_data->preference_nvt_oid);
            set_client_state (CLIENT_MODIFY_CONFIG_PREFERENCE_NVT);
          }
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_CONFIG_PREFERENCE_VALUE);
        ELSE_ERROR ("modify_config");

      case CLIENT_MODIFY_FILTER:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_filter_data->comment, "");
            set_client_state (CLIENT_MODIFY_FILTER_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_filter_data->name, "");
            set_client_state (CLIENT_MODIFY_FILTER_NAME);
          }
        else if (strcasecmp ("TERM", element_name) == 0)
          {
            openvas_append_string (&modify_filter_data->term, "");
            set_client_state (CLIENT_MODIFY_FILTER_TERM);
          }
        else if (strcasecmp ("TYPE", element_name) == 0)
          {
            openvas_append_string (&modify_filter_data->type, "");
            set_client_state (CLIENT_MODIFY_FILTER_TYPE);
          }
        ELSE_ERROR ("modify_filter");

      case CLIENT_MODIFY_GROUP:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_group_data->comment, "");
            set_client_state (CLIENT_MODIFY_GROUP_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_group_data->name, "");
            set_client_state (CLIENT_MODIFY_GROUP_NAME);
          }
        else if (strcasecmp ("USERS", element_name) == 0)
          {
            openvas_append_string (&modify_group_data->users, "");
            set_client_state (CLIENT_MODIFY_GROUP_USERS);
          }
        ELSE_ERROR ("modify_group");

      case CLIENT_MODIFY_PERMISSION:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_PERMISSION_COMMENT);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_PERMISSION_NAME);
        else if (strcasecmp ("RESOURCE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_permission_data->resource_id);
            set_client_state (CLIENT_MODIFY_PERMISSION_RESOURCE);
          }
        else if (strcasecmp ("SUBJECT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_permission_data->subject_id);
            set_client_state (CLIENT_MODIFY_PERMISSION_SUBJECT);
          }
        ELSE_ERROR ("modify_permission");

      case CLIENT_MODIFY_PERMISSION_RESOURCE:
        if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_PERMISSION_RESOURCE_TYPE);
        ELSE_ERROR ("modify_permission");

      case CLIENT_MODIFY_PERMISSION_SUBJECT:
        if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_PERMISSION_SUBJECT_TYPE);
        ELSE_ERROR ("modify_permission");

      case CLIENT_MODIFY_PORT_LIST:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_PORT_LIST_NAME);
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_free_string_var (&modify_port_list_data->comment);
            openvas_append_string (&modify_port_list_data->comment, "");
            set_client_state (CLIENT_MODIFY_PORT_LIST_COMMENT);
          }
        ELSE_ERROR ("modify_port_list");

      case CLIENT_MODIFY_LSC_CREDENTIAL:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_LSC_CREDENTIAL_NAME);
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_free_string_var (&modify_lsc_credential_data->comment);
            openvas_append_string (&modify_lsc_credential_data->comment, "");
            set_client_state (CLIENT_MODIFY_LSC_CREDENTIAL_COMMENT);
          }
        else if (strcasecmp ("LOGIN", element_name) == 0)
          set_client_state (CLIENT_MODIFY_LSC_CREDENTIAL_LOGIN);
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          {
            openvas_free_string_var (&modify_lsc_credential_data->password);
            openvas_append_string (&modify_lsc_credential_data->password, "");
            set_client_state (CLIENT_MODIFY_LSC_CREDENTIAL_PASSWORD);
          }
        ELSE_ERROR ("modify_lsc_credential");

      case CLIENT_MODIFY_REPORT:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_COMMENT);
        ELSE_ERROR ("modify_report");

      case CLIENT_MODIFY_REPORT_FORMAT:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_FORMAT_ACTIVE);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_FORMAT_NAME);
        else if (strcasecmp ("SUMMARY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_FORMAT_SUMMARY);
        else if (strcasecmp ("PARAM", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_FORMAT_PARAM);
        ELSE_ERROR ("modify_report_format");

      case CLIENT_MODIFY_REPORT_FORMAT_PARAM:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_FORMAT_PARAM_NAME);
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_REPORT_FORMAT_PARAM_VALUE);
        ELSE_ERROR ("modify_report_format");

      case CLIENT_MODIFY_ROLE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_role_data->comment, "");
            set_client_state (CLIENT_MODIFY_ROLE_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_role_data->name, "");
            set_client_state (CLIENT_MODIFY_ROLE_NAME);
          }
        else if (strcasecmp ("USERS", element_name) == 0)
          {
            openvas_append_string (&modify_role_data->users, "");
            set_client_state (CLIENT_MODIFY_ROLE_USERS);
          }
        ELSE_ERROR ("modify_role");

      case CLIENT_MODIFY_SCANNER:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->comment, "");
            set_client_state (CLIENT_MODIFY_SCANNER_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->name, "");
            set_client_state (CLIENT_MODIFY_SCANNER_NAME);
          }
        else if (strcasecmp ("HOST", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->host, "");
            set_client_state (CLIENT_MODIFY_SCANNER_HOST);
          }
        else if (strcasecmp ("PORT", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->port, "");
            set_client_state (CLIENT_MODIFY_SCANNER_PORT);
          }
        else if (strcasecmp ("TYPE", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->type, "");
            set_client_state (CLIENT_MODIFY_SCANNER_TYPE);
          }
        else if (strcasecmp ("CA_PUB", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->ca_pub, "");
            set_client_state (CLIENT_MODIFY_SCANNER_CA_PUB);
          }
        else if (strcasecmp ("KEY_PUB", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->key_pub, "");
            set_client_state (CLIENT_MODIFY_SCANNER_KEY_PUB);
          }
        else if (strcasecmp ("KEY_PRIV", element_name) == 0)
          {
            openvas_append_string (&modify_scanner_data->key_priv, "");
            set_client_state (CLIENT_MODIFY_SCANNER_KEY_PRIV);
          }
        ELSE_ERROR ("modify_scanner");

      case CLIENT_MODIFY_SCHEDULE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_schedule_data->comment, "");
            set_client_state (CLIENT_MODIFY_SCHEDULE_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_schedule_data->name, "");
            set_client_state (CLIENT_MODIFY_SCHEDULE_NAME);
          }
        else if (strcasecmp ("DURATION", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_DURATION);
        else if (strcasecmp ("FIRST_TIME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_FIRST_TIME);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_NAME);
        else if (strcasecmp ("PERIOD", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_PERIOD);
        else if (strcasecmp ("TIMEZONE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_TIMEZONE);
        ELSE_ERROR ("modify_schedule");

      case CLIENT_MODIFY_SCHEDULE_FIRST_TIME:
        if (strcasecmp ("DAY_OF_MONTH", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_FIRST_TIME_DAY_OF_MONTH);
        else if (strcasecmp ("HOUR", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_FIRST_TIME_HOUR);
        else if (strcasecmp ("MINUTE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_FIRST_TIME_MINUTE);
        else if (strcasecmp ("MONTH", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_FIRST_TIME_MONTH);
        else if (strcasecmp ("YEAR", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_FIRST_TIME_YEAR);
        ELSE_ERROR ("modify_schedule");

      case CLIENT_MODIFY_SCHEDULE_DURATION:
        if (strcasecmp ("UNIT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_DURATION_UNIT);
        ELSE_ERROR ("modify_schedule");

      case CLIENT_MODIFY_SCHEDULE_PERIOD:
        if (strcasecmp ("UNIT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SCHEDULE_PERIOD_UNIT);
        ELSE_ERROR ("modify_schedule");

      case CLIENT_MODIFY_SETTING:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_SETTING_NAME);
        else if (strcasecmp ("VALUE", element_name) == 0)
          {
            openvas_append_string (&modify_setting_data->value, "");
            set_client_state (CLIENT_MODIFY_SETTING_VALUE);
          }
        ELSE_ERROR ("modify_setting");

      case CLIENT_MODIFY_SLAVE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_slave_data->comment, "");
            set_client_state (CLIENT_MODIFY_SLAVE_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_slave_data->name, "");
            set_client_state (CLIENT_MODIFY_SLAVE_NAME);
          }
        else if (strcasecmp ("HOST", element_name) == 0)
          {
            openvas_append_string (&modify_slave_data->host, "");
            set_client_state (CLIENT_MODIFY_SLAVE_HOST);
          }
        else if (strcasecmp ("PORT", element_name) == 0)
          {
            openvas_append_string (&modify_slave_data->port, "");
            set_client_state (CLIENT_MODIFY_SLAVE_PORT);
          }
        else if (strcasecmp ("LOGIN", element_name) == 0)
          {
            openvas_append_string (&modify_slave_data->login, "");
            set_client_state (CLIENT_MODIFY_SLAVE_LOGIN);
          }
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          {
            openvas_append_string (&modify_slave_data->password, "");
            set_client_state (CLIENT_MODIFY_SLAVE_PASSWORD);
          }
        ELSE_ERROR ("modify_slave");

      case CLIENT_MODIFY_TAG:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          {
            openvas_append_string (&modify_tag_data->active, "");
            set_client_state (CLIENT_MODIFY_TAG_ACTIVE);
          }
        else if (strcasecmp ("RESOURCE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_tag_data->resource_id);
            set_client_state (CLIENT_MODIFY_TAG_RESOURCE);
          }
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_tag_data->comment, "");
            set_client_state (CLIENT_MODIFY_TAG_COMMENT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_tag_data->name, "");
            set_client_state (CLIENT_MODIFY_TAG_NAME);
          }
        else if (strcasecmp ("VALUE", element_name) == 0)
          {
            openvas_append_string (&modify_tag_data->value, "");
            set_client_state (CLIENT_MODIFY_TAG_VALUE);
          }
        ELSE_ERROR ("modify_tag");

      case CLIENT_MODIFY_TAG_RESOURCE:
        if (strcasecmp ("TYPE", element_name) == 0)
          {
            openvas_append_string (&modify_tag_data->resource_type, "");
            set_client_state (CLIENT_MODIFY_TAG_RESOURCE_TYPE);
          }
        ELSE_ERROR ("modify_tag");

      case CLIENT_MODIFY_TARGET:
        if (strcasecmp ("EXCLUDE_HOSTS", element_name) == 0)
          {
            openvas_append_string (&modify_target_data->exclude_hosts, "");
            set_client_state (CLIENT_MODIFY_TARGET_EXCLUDE_HOSTS);
          }
        else if (strcasecmp ("REVERSE_LOOKUP_ONLY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TARGET_REVERSE_LOOKUP_ONLY);
        else if (strcasecmp ("REVERSE_LOOKUP_UNIFY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TARGET_REVERSE_LOOKUP_UNIFY);
        else if (strcasecmp ("ALIVE_TESTS", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TARGET_ALIVE_TESTS);
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_target_data->comment, "");
            set_client_state (CLIENT_MODIFY_TARGET_COMMENT);
          }
        else if (strcasecmp ("ESXI_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_target_data->esxi_lsc_credential_id);
            set_client_state (CLIENT_MODIFY_TARGET_ESXI_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("HOSTS", element_name) == 0)
          {
            openvas_append_string (&modify_target_data->hosts, "");
            set_client_state (CLIENT_MODIFY_TARGET_HOSTS);
          }
        else if (strcasecmp ("PORT_LIST", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_target_data->port_list_id);
            set_client_state (CLIENT_MODIFY_TARGET_PORT_LIST);
          }
        else if (strcasecmp ("SSH_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_target_data->ssh_lsc_credential_id);
            set_client_state (CLIENT_MODIFY_TARGET_SSH_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("SMB_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_target_data->smb_lsc_credential_id);
            set_client_state (CLIENT_MODIFY_TARGET_SMB_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&modify_target_data->name, "");
            set_client_state (CLIENT_MODIFY_TARGET_NAME);
          }
        ELSE_ERROR ("modify_target");

      case CLIENT_MODIFY_TARGET_SSH_LSC_CREDENTIAL:
        if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TARGET_SSH_LSC_CREDENTIAL_PORT);
        ELSE_ERROR ("modify_target");

      case CLIENT_MODIFY_TASK:
        if (strcasecmp ("ALTERABLE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TASK_ALTERABLE);
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&modify_task_data->comment, "");
            set_client_state (CLIENT_MODIFY_TASK_COMMENT);
          }
        else if (strcasecmp ("HOSTS_ORDERING", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TASK_HOSTS_ORDERING);
        else if (strcasecmp ("SCANNER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_task_data->scanner_id);
            set_client_state (CLIENT_MODIFY_TASK_SCANNER);
          }
        else if (strcasecmp ("ALERT", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (modify_task_data->alerts, g_strdup (attribute));
            set_client_state (CLIENT_MODIFY_TASK_ALERT);
          }
        else if (strcasecmp ("CONFIG", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_task_data->config_id);
            set_client_state (CLIENT_MODIFY_TASK_CONFIG);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TASK_NAME);
        else if (strcasecmp ("OBSERVERS", element_name) == 0)
          {
            openvas_append_string (&modify_task_data->observers, "");
            set_client_state (CLIENT_MODIFY_TASK_OBSERVERS);
          }
        else if (strcasecmp ("PREFERENCES", element_name) == 0)
          {
            modify_task_data->preferences = make_array ();
            set_client_state (CLIENT_MODIFY_TASK_PREFERENCES);
          }
        else if (strcasecmp ("SCHEDULE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_task_data->schedule_id);
            set_client_state (CLIENT_MODIFY_TASK_SCHEDULE);
          }
        else if (strcasecmp ("SCHEDULE_PERIODS", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TASK_SCHEDULE_PERIODS);
        else if (strcasecmp ("SLAVE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_task_data->slave_id);
            set_client_state (CLIENT_MODIFY_TASK_SLAVE);
          }
        else if (strcasecmp ("TARGET", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_task_data->target_id);
            set_client_state (CLIENT_MODIFY_TASK_TARGET);
          }
        else if (strcasecmp ("FILE", element_name) == 0)
          {
            const gchar* attribute;
            append_attribute (attribute_names, attribute_values, "name",
                              &modify_task_data->file_name);
            if (find_attribute (attribute_names, attribute_values,
                                "action", &attribute))
              openvas_append_string (&modify_task_data->action, attribute);
            else
              openvas_append_string (&modify_task_data->action, "update");
            set_client_state (CLIENT_MODIFY_TASK_FILE);
          }
        ELSE_ERROR ("modify_task");

      case CLIENT_MODIFY_TASK_OBSERVERS:
        if (strcasecmp ("GROUP", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (modify_task_data->groups, g_strdup (attribute));
            set_client_state (CLIENT_MODIFY_TASK_OBSERVERS_GROUP);
          }
        ELSE_ERROR ("modify_task");

      case CLIENT_MODIFY_TASK_PREFERENCES:
        if (strcasecmp ("PREFERENCE", element_name) == 0)
          {
            assert (modify_task_data->preference == NULL);
            modify_task_data->preference = g_malloc (sizeof (name_value_t));
            modify_task_data->preference->name = NULL;
            modify_task_data->preference->value = NULL;
            set_client_state (CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE);
          }
        ELSE_ERROR ("modify_task");

      case CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE:
        if (strcasecmp ("SCANNER_NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE_NAME);
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE_VALUE);
        ELSE_ERROR ("modify_task");

      case CLIENT_MODIFY_USER:
        if (strcasecmp ("GROUPS", element_name) == 0)
          {
            if (modify_user_data->groups)
              array_free (modify_user_data->groups);
            modify_user_data->groups = make_array ();
            set_client_state (CLIENT_MODIFY_USER_GROUPS);
          }
        else if (strcasecmp ("HOSTS", element_name) == 0)
          {
            const gchar *attribute;
            if (find_attribute
                (attribute_names, attribute_values, "allow", &attribute))
              modify_user_data->hosts_allow = strcmp (attribute, "0");
            else
              modify_user_data->hosts_allow = 1;
            /* Init, so that modify_user clears hosts if HOSTS is empty. */
            openvas_append_string (&modify_user_data->hosts, "");
            set_client_state (CLIENT_MODIFY_USER_HOSTS);
          }
        else if (strcasecmp ("IFACES", element_name) == 0)
          {
            const gchar *attribute;
            if (find_attribute
                (attribute_names, attribute_values, "allow", &attribute))
              modify_user_data->ifaces_allow = strcmp (attribute, "0");
            else
              modify_user_data->ifaces_allow = 1;
            openvas_append_string (&modify_user_data->ifaces, "");
            set_client_state (CLIENT_MODIFY_USER_IFACES);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_USER_NAME);
        else if (strcasecmp ("NEW_NAME", element_name) == 0)
          set_client_state (CLIENT_MODIFY_USER_NEW_NAME);
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          {
            const gchar *attribute;
            if (find_attribute
                (attribute_names, attribute_values, "modify", &attribute))
              modify_user_data->modify_password = strcmp (attribute, "0");
            else
              modify_user_data->modify_password = 1;
            set_client_state (CLIENT_MODIFY_USER_PASSWORD);
          }
        else if (strcasecmp ("ROLE", element_name) == 0)
          {
            const gchar* attribute;
            /* Init array here, so it's NULL if there are no ROLEs. */
            if (modify_user_data->roles == NULL)
              {
                array_free (modify_user_data->roles);
                modify_user_data->roles = make_array ();
              }
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (modify_user_data->roles, g_strdup (attribute));
            set_client_state (CLIENT_MODIFY_USER_ROLE);
          }
        else if (strcasecmp ("SOURCES", element_name) == 0)
          {
            modify_user_data->sources = make_array ();
            set_client_state (CLIENT_MODIFY_USER_SOURCES);
          }
        else
          {
            if (send_element_error_to_client ("modify_user", element_name,
                                              write_to_client,
                                              write_to_client_data))
              {
                error_send_to_client (error);
                return;
              }
            set_client_state (CLIENT_AUTHENTIC);
            g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Error");
          }
        break;

      case CLIENT_MODIFY_USER_GROUPS:
        if (strcasecmp ("GROUP", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (modify_user_data->groups, g_strdup (attribute));
            set_client_state (CLIENT_MODIFY_USER_GROUPS_GROUP);
          }
        ELSE_ERROR ("modify_user");

      case CLIENT_MODIFY_USER_SOURCES:
        if (strcasecmp ("SOURCE", element_name) == 0)
         {
           set_client_state (CLIENT_MODIFY_USER_SOURCES_SOURCE);
         }
        else
          {
            if (send_element_error_to_client ("modify_user_sources",
                                              element_name,
                                              write_to_client,
                                              write_to_client_data))
              {
                error_send_to_client (error);
                return;
              }
            set_client_state (CLIENT_AUTHENTIC);
            g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Error");
          }
        break;

      case CLIENT_CREATE_AGENT:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_COPY);
        else if (strcasecmp ("HOWTO_INSTALL", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_HOWTO_INSTALL);
        else if (strcasecmp ("HOWTO_USE", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_HOWTO_USE);
        else if (strcasecmp ("INSTALLER", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_INSTALLER);
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_agent_data->name, "");
            set_client_state (CLIENT_CREATE_AGENT_NAME);
          }
        ELSE_ERROR ("create_agent");
      case CLIENT_CREATE_AGENT_INSTALLER:
        if (strcasecmp ("FILENAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_INSTALLER_FILENAME);
        else if (strcasecmp ("SIGNATURE", element_name) == 0)
          set_client_state (CLIENT_CREATE_AGENT_INSTALLER_SIGNATURE);
        ELSE_ERROR ("create_agent");

      case CLIENT_CREATE_CONFIG:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_CONFIG_COMMENT);
        else if (strcasecmp ("SCANNER", element_name) == 0)
          set_client_state (CLIENT_CREATE_CONFIG_SCANNER);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_CONFIG_COPY);
        else if (strcasecmp ("GET_CONFIGS_RESPONSE", element_name) == 0)
          {
            omp_parser->importing = 1;
            import_config_data->import = 1;
            set_client_state (CLIENT_C_C_GCR);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_CONFIG_NAME);
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR:
        if (strcasecmp ("CONFIG", element_name) == 0)
          {
            /* Reset here in case there was a previous config element. */
            create_config_data_reset (create_config_data);
            import_config_data->import = 1;
            set_client_state (CLIENT_C_C_GCR_CONFIG);
          }
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR_CONFIG:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_C_C_GCR_CONFIG_COMMENT);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_C_C_GCR_CONFIG_NAME);
        else if (strcasecmp ("NVT_SELECTORS", element_name) == 0)
          {
            /* Reset array, in case there was a previous nvt_selectors element. */
            array_reset (&import_config_data->nvt_selectors);
            set_client_state (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS);
          }
        else if (strcasecmp ("PREFERENCES", element_name) == 0)
          {
            /* Reset array, in case there was a previous preferences element. */
            array_reset (&import_config_data->preferences);
            set_client_state (CLIENT_C_C_GCR_CONFIG_PREFERENCES);
          }
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS:
        if (strcasecmp ("NVT_SELECTOR", element_name) == 0)
          set_client_state (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR);
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR:
        if (strcasecmp ("INCLUDE", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_INCLUDE);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_NAME);
        else if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_TYPE);
        else if (strcasecmp ("FAMILY_OR_NVT", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR_FAMILY_OR_NVT);
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR_CONFIG_PREFERENCES:
        if (strcasecmp ("PREFERENCE", element_name) == 0)
          {
            array_reset (&import_config_data->preference_alts);
            set_client_state (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE);
          }
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE:
        if (strcasecmp ("ALT", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_ALT);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NAME);
        else if (strcasecmp ("NVT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "oid",
                              &import_config_data->preference_nvt_oid);
            set_client_state
             (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NVT);
          }
        else if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_TYPE);
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_VALUE);
        ELSE_ERROR ("create_config");

      case CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NVT:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state
           (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NVT_NAME);
        ELSE_ERROR ("create_config");

      case CLIENT_CREATE_ALERT:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_COPY);
        else if (strcasecmp ("CONDITION", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_CONDITION);
        else if (strcasecmp ("EVENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_EVENT);
        else if (strcasecmp ("FILTER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_alert_data->filter_id);
            set_client_state (CLIENT_CREATE_ALERT_FILTER);
          }
        else if (strcasecmp ("METHOD", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_METHOD);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_NAME);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_ALERT_CONDITION:
        if (strcasecmp ("DATA", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_CONDITION_DATA);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_ALERT_CONDITION_DATA:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_CONDITION_DATA_NAME);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_ALERT_EVENT:
        if (strcasecmp ("DATA", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_EVENT_DATA);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_ALERT_EVENT_DATA:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_EVENT_DATA_NAME);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_ALERT_METHOD:
        if (strcasecmp ("DATA", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_METHOD_DATA);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_ALERT_METHOD_DATA:
        if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_ALERT_METHOD_DATA_NAME);
        ELSE_ERROR ("create_alert");

      case CLIENT_CREATE_FILTER:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_FILTER_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_FILTER_COPY);
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_filter_data->name, "");
            set_client_state (CLIENT_CREATE_FILTER_NAME);
          }
        else if (strcasecmp ("TERM", element_name) == 0)
          set_client_state (CLIENT_CREATE_FILTER_TERM);
        else if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_CREATE_FILTER_TYPE);
        ELSE_ERROR ("create_filter");

      case CLIENT_CREATE_FILTER_NAME:
        if (strcasecmp ("MAKE_UNIQUE", element_name) == 0)
          set_client_state (CLIENT_CREATE_FILTER_NAME_MAKE_UNIQUE);
        ELSE_ERROR ("create_filter");

      case CLIENT_CREATE_GROUP:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_GROUP_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_GROUP_COPY);
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_group_data->name, "");
            set_client_state (CLIENT_CREATE_GROUP_NAME);
          }
        else if (strcasecmp ("USERS", element_name) == 0)
          set_client_state (CLIENT_CREATE_GROUP_USERS);
        ELSE_ERROR ("create_group");

      case CLIENT_CREATE_LSC_CREDENTIAL:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_COMMENT);
        else if (strcasecmp ("KEY", element_name) == 0)
          {
            create_lsc_credential_data->key = 1;
            set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_KEY);
          }
        else if (strcasecmp ("LOGIN", element_name) == 0)
          set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_LOGIN);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_COPY);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_NAME);
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          {
            openvas_append_string (&create_lsc_credential_data->password, "");
            set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_PASSWORD);
          }
        ELSE_ERROR ("create_lsc_credential");

      case CLIENT_CREATE_LSC_CREDENTIAL_KEY:
        if (strcasecmp ("PHRASE", element_name) == 0)
          {
            openvas_append_string (&create_lsc_credential_data->key_phrase, "");
            set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_KEY_PHRASE);
          }
        else if (strcasecmp ("PRIVATE", element_name) == 0)
          set_client_state (CLIENT_CREATE_LSC_CREDENTIAL_KEY_PRIVATE);
        ELSE_ERROR ("create_lsc_credential");

      case CLIENT_CREATE_NOTE:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_ACTIVE);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_COPY);
        else if (strcasecmp ("HOSTS", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_HOSTS);
        else if (strcasecmp ("NVT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "oid",
                              &create_note_data->nvt_oid);
            set_client_state (CLIENT_CREATE_NOTE_NVT);
          }
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_PORT);
        else if (strcasecmp ("RESULT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_note_data->result_id);
            if (create_note_data->result_id
                && create_note_data->result_id[0] == '\0')
              {
                g_free (create_note_data->result_id);
                create_note_data->result_id = NULL;
              }
            set_client_state (CLIENT_CREATE_NOTE_RESULT);
          }
        else if (strcasecmp ("SEVERITY", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_SEVERITY);
        else if (strcasecmp ("TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_note_data->task_id);
            if (create_note_data->task_id
                && create_note_data->task_id[0] == '\0')
              {
                g_free (create_note_data->task_id);
                create_note_data->task_id = NULL;
              }
            set_client_state (CLIENT_CREATE_NOTE_TASK);
          }
        else if (strcasecmp ("TEXT", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_TEXT);
        else if (strcasecmp ("THREAT", element_name) == 0)
          set_client_state (CLIENT_CREATE_NOTE_THREAT);
        ELSE_ERROR ("create_note");

      case CLIENT_CREATE_PERMISSION:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_PERMISSION_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_PERMISSION_COPY);
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_permission_data->name, "");
            set_client_state (CLIENT_CREATE_PERMISSION_NAME);
          }
        else if (strcasecmp ("RESOURCE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_permission_data->resource_id);
            set_client_state (CLIENT_CREATE_PERMISSION_RESOURCE);
          }
        else if (strcasecmp ("SUBJECT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_permission_data->subject_id);
            set_client_state (CLIENT_CREATE_PERMISSION_SUBJECT);
          }
        ELSE_ERROR ("create_permission");

      case CLIENT_CREATE_PERMISSION_RESOURCE:
        if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_CREATE_PERMISSION_RESOURCE_TYPE);
        ELSE_ERROR ("create_permission");

      case CLIENT_CREATE_PERMISSION_SUBJECT:
        if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_CREATE_PERMISSION_SUBJECT_TYPE);
        ELSE_ERROR ("create_permission");

      case CLIENT_CREATE_PORT_LIST:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_LIST_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_LIST_COPY);
        else if (strcasecmp ("GET_PORT_LISTS_RESPONSE", element_name) == 0)
          {
            omp_parser->importing = 1;
            create_port_list_data->import = 1;
            set_client_state (CLIENT_CPL_GPLR);
          }
        else if (strcasecmp ("PORT_RANGE", element_name) == 0)
          {
            openvas_append_string (&create_port_list_data->port_range, "");
            set_client_state (CLIENT_CREATE_PORT_LIST_PORT_RANGE);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_LIST_NAME);
        ELSE_ERROR ("create_port_list");

      case CLIENT_CPL_GPLR:
        if (strcasecmp ("PORT_LIST", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_port_list_data->id);
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST);
          }
        ELSE_ERROR ("create_port_list");

      case CLIENT_CPL_GPLR_PORT_LIST:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CPL_GPLR_PORT_LIST_COMMENT);
        else if (strcasecmp ("IN_USE", element_name) == 0)
          set_client_state (CLIENT_CPL_GPLR_PORT_LIST_IN_USE);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CPL_GPLR_PORT_LIST_NAME);
        else if (strcasecmp ("PORT_RANGE", element_name) == 0)
          set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGE);
        else if (strcasecmp ("PORT_RANGES", element_name) == 0)
          {
            create_port_list_data->ranges = make_array ();
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES);
          }
        else if (strcasecmp ("TARGETS", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_TARGETS);
          }
        ELSE_ERROR ("create_port_list");

      case CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES:
        if (strcasecmp ("PORT_RANGE", element_name) == 0)
          {
            assert (create_port_list_data->range == NULL);
            create_port_list_data->range
             = g_malloc0 (sizeof (create_port_list_range_t));
            append_attribute (attribute_names, attribute_values, "id",
                              &(create_port_list_data->range->id));
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE);
          }
        ELSE_ERROR ("create_port_list");

      case CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&create_port_list_data->range->comment, "");
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_COMMENT);
          }
        else if (strcasecmp ("END", element_name) == 0)
          {
            openvas_append_string (&create_port_list_data->range->end, "");
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_END);
          }
        else if (strcasecmp ("START", element_name) == 0)
          {
            openvas_append_string (&create_port_list_data->range->start, "");
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_START);
          }
        else if (strcasecmp ("TYPE", element_name) == 0)
          {
            openvas_append_string (&create_port_list_data->range->type, "");
            set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE_TYPE);
          }
        ELSE_ERROR ("create_port_list");

      case CLIENT_CREATE_PORT_RANGE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_RANGE_COMMENT);
        else if (strcasecmp ("END", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_RANGE_END);
        else if (strcasecmp ("PORT_LIST", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_port_range_data->port_list_id);
            set_client_state (CLIENT_CREATE_PORT_RANGE_PORT_LIST);
          }
        else if (strcasecmp ("START", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_RANGE_START);
        else if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_CREATE_PORT_RANGE_TYPE);
        ELSE_ERROR ("create_port_range");

      case CLIENT_CREATE_ROLE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_ROLE_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_ROLE_COPY);
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_role_data->name, "");
            set_client_state (CLIENT_CREATE_ROLE_NAME);
          }
        else if (strcasecmp ("USERS", element_name) == 0)
          set_client_state (CLIENT_CREATE_ROLE_USERS);
        ELSE_ERROR ("create_role");

      case CLIENT_CREATE_REPORT:
        if (strcasecmp ("REPORT", element_name) == 0)
          {
            const gchar* attribute;

            omp_parser->importing = 1;

            append_attribute (attribute_names, attribute_values,
                              "type", &create_report_data->type);

            if (find_attribute (attribute_names, attribute_values, "format_id",
                                &attribute))
              {
                /* Assume this is the wrapper REPORT. */
                create_report_data->wrapper = 1;
                set_client_state (CLIENT_CREATE_REPORT_REPORT);
              }
            else
              {
                /* Assume the report is immediately inside the CREATE_REPORT. */
                create_report_data->wrapper = 0;
                create_report_data->details = make_array ();
                create_report_data->host_ends = make_array ();
                create_report_data->host_starts = make_array ();
                create_report_data->results = make_array ();
                set_client_state (CLIENT_CREATE_REPORT_RR);
              }
          }
        else if (strcasecmp ("TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_report_data->task_id);
            set_client_state (CLIENT_CREATE_REPORT_TASK);
          }
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_REPORT:
        if (strcasecmp ("REPORT", element_name) == 0)
          {
            create_report_data->details = make_array ();
            create_report_data->host_ends = make_array ();
            create_report_data->host_starts = make_array ();
            create_report_data->results = make_array ();
            set_client_state (CLIENT_CREATE_REPORT_RR);
          }
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR:
        if (strcasecmp ("FILTERS", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_FILTERS);
          }
        else if (strcasecmp ("HOST", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H);
          }
        else if (strcasecmp ("HOST_COUNT", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_HOST_COUNT);
          }
        else if (strcasecmp ("HOST_END", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_HOST_END);
        else if (strcasecmp ("HOST_START", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_HOST_START);
        else if (strcasecmp ("HOSTS", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_HOSTS);
          }
        else if (strcasecmp ("PORTS", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_PORTS);
          }
        else if (strcasecmp ("REPORT_FORMAT", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_REPORT_FORMAT);
          }
        else if (strcasecmp ("RESULTS", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS);
        else if (strcasecmp ("RESULT_COUNT", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_RESULT_COUNT);
          }
        else if (strcasecmp ("SCAN_RUN_STATUS", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state
             (CLIENT_CREATE_REPORT_RR_SCAN_RUN_STATUS);
          }
        else if (strcasecmp ("SCAN_END", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_SCAN_END);
          }
        else if (strcasecmp ("SCAN_START", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_SCAN_START);
          }
        else if (strcasecmp ("SORT", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_SORT);
          }
        else if (strcasecmp ("TASK", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_TASK);
          }
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_HOST_END:
        if (strcasecmp ("HOST", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_HOST_END_HOST);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_HOST_START:
        if (strcasecmp ("HOST", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_HOST_START_HOST);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_H:
        if (strcasecmp ("IP", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_IP);
          }
        else if (strcasecmp ("DETAIL", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL);
          }
        else if (strcasecmp ("END", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_END);
          }
        else if (strcasecmp ("START", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_START);
          }
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_H_DETAIL:
        if (strcasecmp ("NAME", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_NAME);
          }
        else if (strcasecmp ("VALUE", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_VALUE);
          }
        else if (strcasecmp ("SOURCE", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE);
          }
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE:
        if (strcasecmp ("DESCRIPTION", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_DESC);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_NAME);
          }
        else if (strcasecmp ("TYPE", element_name) == 0)
          {
            set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_TYPE);
          }
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_RESULTS:
        if (strcasecmp ("RESULT", element_name) == 0)
          set_client_state
           (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_RESULTS_RESULT:
        if (strcasecmp ("COMMENT", element_name) == 0)
          {
            set_client_state
              (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_COMMENT);
            omp_parser->read_over = 1;
          }
        else if (strcasecmp ("CREATION_TIME", element_name) == 0)
          {
            set_client_state
              (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_CREATION_TIME);
            omp_parser->read_over = 1;
          }
        else if (strcasecmp ("DESCRIPTION", element_name) == 0)
          set_client_state
           (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_DESCRIPTION);
        else if (strcasecmp ("DETECTION", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_DETECTION);
          }
        else if (strcasecmp ("HOST", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_HOST);
        else if (strcasecmp ("MODIFICATION_TIME", element_name) == 0)
          {
            set_client_state
              (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_MODIFICATION_TIME);
            omp_parser->read_over = 1;
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            set_client_state
              (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NAME);
            omp_parser->read_over = 1;
          }
        else if (strcasecmp ("NOTES", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NOTES);
          }
        else if (strcasecmp ("NVT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "oid",
                              &create_report_data->result_nvt_oid);
            set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT);
          }
        else if (strcasecmp ("ORIGINAL_SEVERITY", element_name) == 0)
          set_client_state
           (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_ORIGINAL_SEVERITY);
        else if (strcasecmp ("ORIGINAL_THREAT", element_name) == 0)
          set_client_state
           (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_ORIGINAL_THREAT);
        else if (strcasecmp ("OVERRIDES", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_OVERRIDES);
          }
        else if (strcasecmp ("OWNER", element_name) == 0)
          {
            set_client_state
              (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_OWNER);
            omp_parser->read_over = 1;
          }
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_PORT);
        else if (strcasecmp ("QOD", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD);
        else if (strcasecmp ("SCAN_NVT_VERSION", element_name) == 0)
          set_client_state
           (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_SCAN_NVT_VERSION);
        else if (strcasecmp ("SEVERITY", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_SEVERITY);
        else if (strcasecmp ("THREAT", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_THREAT);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT:
        if (strcasecmp ("BID", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_BID);
        else if (strcasecmp ("CVE", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CVE);
        else if (strcasecmp ("CVSS_BASE", element_name) == 0)
          set_client_state
           (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CVSS_BASE);
        else if (strcasecmp ("FAMILY", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_FAMILY);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_NAME);
        else if (strcasecmp ("XREF", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_XREF);
        else if (strcasecmp ("CERT", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CERT);
        ELSE_ERROR ("create_report");

      case (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CERT):
        if (strcasecmp ("CERT_REF", element_name) == 0)
          set_client_state
              (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CERT_CERT_REF);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD:
        if (strcasecmp ("TYPE", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD_TYPE);
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD_VALUE);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_TASK:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_TASK_COMMENT);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_TASK_NAME);
        ELSE_ERROR ("create_report");

      case CLIENT_CREATE_REPORT_FORMAT:
        if (strcasecmp ("GET_REPORT_FORMATS_RESPONSE", element_name) == 0)
          {
            omp_parser->importing = 1;
            create_report_format_data->import = 1;
            set_client_state (CLIENT_CRF_GRFR);
          }
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_REPORT_FORMAT_COPY);
        ELSE_ERROR ("create_report_format");

      case CLIENT_CRF_GRFR:
        if (strcasecmp ("REPORT_FORMAT", element_name) == 0)
          {
            create_report_format_data->files = make_array ();
            create_report_format_data->params = make_array ();
            create_report_format_data->params_options = make_array ();
            append_attribute (attribute_names, attribute_values, "id",
                              &create_report_format_data->id);
            set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT);
          }
        ELSE_ERROR ("create_report_format");

      case CLIENT_CRF_GRFR_REPORT_FORMAT:
        if (strcasecmp ("CONTENT_TYPE", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_CONTENT_TYPE);
        else if (strcasecmp ("DESCRIPTION", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_DESCRIPTION);
        else if (strcasecmp ("EXTENSION", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_EXTENSION);
        else if (strcasecmp ("GLOBAL", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_GLOBAL);
        else if (strcasecmp ("FILE", element_name) == 0)
          {
            assert (create_report_format_data->file == NULL);
            assert (create_report_format_data->file_name == NULL);
            openvas_append_string (&create_report_format_data->file, "");
            append_attribute (attribute_names, attribute_values, "name",
                              &create_report_format_data->file_name);
            set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_FILE);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_NAME);
        else if (strcasecmp ("PARAM", element_name) == 0)
          {
            assert (create_report_format_data->param_name == NULL);
            assert (create_report_format_data->param_type == NULL);
            assert (create_report_format_data->param_value == NULL);
            openvas_append_string (&create_report_format_data->param_name, "");
            openvas_append_string (&create_report_format_data->param_value, "");
            create_report_format_data->param_options = make_array ();
            set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM);
          }
        else if (strcasecmp ("PREDEFINED", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PREDEFINED);
        else if (strcasecmp ("SIGNATURE", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_SIGNATURE);
        else if (strcasecmp ("SUMMARY", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_SUMMARY);
        else if (strcasecmp ("TRUST", element_name) == 0)
          {
            omp_parser->read_over = 1;
            set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_TRUST);
          }
        ELSE_ERROR ("create_report_format");

      case CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM:
        if (strcasecmp ("DEFAULT", element_name) == 0)
          {
            openvas_append_string (&create_report_format_data->param_default,
                                   "");
            set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_DEFAULT);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_NAME);
        else if (strcasecmp ("OPTIONS", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS);
        else if (strcasecmp ("TYPE", element_name) == 0)
          {
            openvas_append_string (&create_report_format_data->param_type, "");
            set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE);
          }
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_VALUE);
        ELSE_ERROR ("create_report_format");

      case CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS:
        if (strcasecmp ("OPTION", element_name) == 0)
          {
            openvas_append_string (&create_report_format_data->param_option,
                                   "");
            set_client_state
             (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS_OPTION);
          }
        ELSE_ERROR ("create_report_format");

      case CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE:
        if (strcasecmp ("MAX", element_name) == 0)
          {
            set_client_state
             (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE_MAX);
          }
        else if (strcasecmp ("MIN", element_name) == 0)
          {
            set_client_state
             (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE_MIN);
          }
        ELSE_ERROR ("create_report_format");

      case CLIENT_CREATE_OVERRIDE:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_ACTIVE);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_COPY);
        else if (strcasecmp ("HOSTS", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_HOSTS);
        else if (strcasecmp ("NEW_SEVERITY", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_NEW_SEVERITY);
        else if (strcasecmp ("NEW_THREAT", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_NEW_THREAT);
        else if (strcasecmp ("NVT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "oid",
                              &create_override_data->nvt_oid);
            set_client_state (CLIENT_CREATE_OVERRIDE_NVT);
          }
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_PORT);
        else if (strcasecmp ("RESULT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_override_data->result_id);
            if (create_override_data->result_id
                && create_override_data->result_id[0] == '\0')
              {
                g_free (create_override_data->result_id);
                create_override_data->result_id = NULL;
              }
            set_client_state (CLIENT_CREATE_OVERRIDE_RESULT);
          }
        else if (strcasecmp ("SEVERITY", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_SEVERITY);
        else if (strcasecmp ("TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_override_data->task_id);
            if (create_override_data->task_id
                && create_override_data->task_id[0] == '\0')
              {
                g_free (create_override_data->task_id);
                create_override_data->task_id = NULL;
              }
            set_client_state (CLIENT_CREATE_OVERRIDE_TASK);
          }
        else if (strcasecmp ("TEXT", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_TEXT);
        else if (strcasecmp ("THREAT", element_name) == 0)
          set_client_state (CLIENT_CREATE_OVERRIDE_THREAT);
        ELSE_ERROR ("create_override");

      case CLIENT_CREATE_SLAVE:
        if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_COPY);
        else if (strcasecmp ("HOST", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_HOST);
        else if (strcasecmp ("LOGIN", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_LOGIN);
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_NAME);
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_PASSWORD);
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_SLAVE_PORT);
        ELSE_ERROR ("create_slave");

      case CLIENT_CREATE_TAG:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          {
            openvas_append_string (&create_tag_data->active, "");
            set_client_state (CLIENT_CREATE_TAG_ACTIVE);
          }
        else if (strcasecmp ("RESOURCE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_tag_data->resource_id);
            set_client_state (CLIENT_CREATE_TAG_RESOURCE);
          }
        else if (strcasecmp ("COMMENT", element_name) == 0)
          {
            openvas_append_string (&create_tag_data->comment, "");
            set_client_state (CLIENT_CREATE_TAG_COMMENT);
          }
        else if (strcasecmp ("COPY", element_name) == 0)
          {
            openvas_append_string (&create_tag_data->copy, "");
            set_client_state (CLIENT_CREATE_TAG_COPY);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_tag_data->name, "");
            set_client_state (CLIENT_CREATE_TAG_NAME);
          }
        else if (strcasecmp ("VALUE", element_name) == 0)
          {
            openvas_append_string (&create_tag_data->value, "");
            set_client_state (CLIENT_CREATE_TAG_VALUE);
          }
        ELSE_ERROR ("create_tag");

      case CLIENT_CREATE_TAG_RESOURCE:
        if (strcasecmp ("TYPE", element_name) == 0)
          {
            openvas_append_string (&create_tag_data->resource_type, "");
            set_client_state (CLIENT_CREATE_TAG_RESOURCE_TYPE);
          }
        ELSE_ERROR ("create_tag");

      case CLIENT_CREATE_TARGET:
        if (strcasecmp ("EXCLUDE_HOSTS", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_EXCLUDE_HOSTS);
        else if (strcasecmp ("REVERSE_LOOKUP_ONLY", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_REVERSE_LOOKUP_ONLY);
        else if (strcasecmp ("REVERSE_LOOKUP_UNIFY", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_REVERSE_LOOKUP_UNIFY);
        else if (strcasecmp ("ALIVE_TESTS", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_ALIVE_TESTS);
        else if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_COMMENT);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_COPY);
        else if (strcasecmp ("ESXI_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_target_data->esxi_lsc_credential_id);
            set_client_state (CLIENT_CREATE_TARGET_ESXI_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("HOSTS", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_HOSTS);
        else if (strcasecmp ("PORT_LIST", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_target_data->port_list_id);
            set_client_state (CLIENT_CREATE_TARGET_PORT_LIST);
          }
        else if (strcasecmp ("PORT_RANGE", element_name) == 0)
          {
            openvas_append_string (&create_target_data->port_range, "");
            set_client_state (CLIENT_CREATE_TARGET_PORT_RANGE);
          }
        else if (strcasecmp ("SSH_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_target_data->ssh_lsc_credential_id);
            set_client_state (CLIENT_CREATE_TARGET_SSH_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("SMB_LSC_CREDENTIAL", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_target_data->smb_lsc_credential_id);
            set_client_state (CLIENT_CREATE_TARGET_SMB_LSC_CREDENTIAL);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            openvas_append_string (&create_target_data->name, "");
            set_client_state (CLIENT_CREATE_TARGET_NAME);
          }
        ELSE_ERROR ("create_target");

      case CLIENT_CREATE_TARGET_NAME:
        if (strcasecmp ("MAKE_UNIQUE", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_NAME_MAKE_UNIQUE);
        ELSE_ERROR ("create_target");

      case CLIENT_CREATE_TARGET_SSH_LSC_CREDENTIAL:
        if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_CREATE_TARGET_SSH_LSC_CREDENTIAL_PORT);
        ELSE_ERROR ("create_target");

      case CLIENT_CREATE_TASK:
        if (strcasecmp ("ALTERABLE", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_ALTERABLE);
        else if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_COPY);
        else if (strcasecmp ("PREFERENCES", element_name) == 0)
          {
            create_task_data->preferences = make_array ();
            set_client_state (CLIENT_CREATE_TASK_PREFERENCES);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_NAME);
        else if (strcasecmp ("COMMENT", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_COMMENT);
        else if (strcasecmp ("HOSTS_ORDERING", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_HOSTS_ORDERING);
        else if (strcasecmp ("SCANNER", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_task_data->scanner_id);
            set_client_state (CLIENT_CREATE_TASK_SCANNER);
          }
        else if (strcasecmp ("CONFIG", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_task_data->config_id);
            set_client_state (CLIENT_CREATE_TASK_CONFIG);
          }
        else if (strcasecmp ("ALERT", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (create_task_data->alerts, g_strdup (attribute));
            set_client_state (CLIENT_CREATE_TASK_ALERT);
          }
        else if (strcasecmp ("OBSERVERS", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_OBSERVERS);
        else if (strcasecmp ("SCHEDULE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_task_data->schedule_id);
            set_client_state (CLIENT_CREATE_TASK_SCHEDULE);
          }
        else if (strcasecmp ("SCHEDULE_PERIODS", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_SCHEDULE_PERIODS);
        else if (strcasecmp ("SLAVE", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_task_data->slave_id);
            set_client_state (CLIENT_CREATE_TASK_SLAVE);
          }
        else if (strcasecmp ("TARGET", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &create_task_data->target_id);
            set_client_state (CLIENT_CREATE_TASK_TARGET);
          }
        ELSE_ERROR_CREATE_TASK ();

      case CLIENT_CREATE_TASK_OBSERVERS:
        if (strcasecmp ("GROUP", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (create_task_data->groups, g_strdup (attribute));
            set_client_state (CLIENT_CREATE_TASK_OBSERVERS_GROUP);
          }
        ELSE_ERROR_CREATE_TASK ();

      case CLIENT_CREATE_TASK_PREFERENCES:
        if (strcasecmp ("PREFERENCE", element_name) == 0)
          {
            assert (create_task_data->preference == NULL);
            create_task_data->preference = g_malloc (sizeof (name_value_t));
            create_task_data->preference->name = NULL;
            create_task_data->preference->value = NULL;
            set_client_state (CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE);
          }
        ELSE_ERROR_CREATE_TASK ();

      case CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE:
        if (strcasecmp ("SCANNER_NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE_NAME);
        else if (strcasecmp ("VALUE", element_name) == 0)
          set_client_state (CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE_VALUE);
        ELSE_ERROR_CREATE_TASK ();

      case CLIENT_CREATE_USER:
        if (strcasecmp ("COPY", element_name) == 0)
          set_client_state (CLIENT_CREATE_USER_COPY);
        else if (strcasecmp ("GROUPS", element_name) == 0)
          set_client_state (CLIENT_CREATE_USER_GROUPS);
        else if (strcasecmp ("HOSTS", element_name) == 0)
          {
            const gchar *attribute;
            if (find_attribute
                (attribute_names, attribute_values, "allow", &attribute))
              create_user_data->hosts_allow = strcmp (attribute, "0");
            else
              create_user_data->hosts_allow = 1;
            set_client_state (CLIENT_CREATE_USER_HOSTS);
          }
        else if (strcasecmp ("IFACES", element_name) == 0)
          {
            const gchar *attribute;
            if (find_attribute
                (attribute_names, attribute_values, "allow", &attribute))
              create_user_data->ifaces_allow = strcmp (attribute, "0");
            else
              create_user_data->ifaces_allow = 1;
            set_client_state (CLIENT_CREATE_USER_IFACES);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          set_client_state (CLIENT_CREATE_USER_NAME);
        else if (strcasecmp ("PASSWORD", element_name) == 0)
          set_client_state (CLIENT_CREATE_USER_PASSWORD);
        else if (strcasecmp ("ROLE", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (create_user_data->roles, g_strdup (attribute));
            set_client_state (CLIENT_CREATE_USER_ROLE);
          }
        else if (strcasecmp ("SOURCES", element_name) == 0)
          {
            create_user_data->sources = make_array ();
            set_client_state (CLIENT_CREATE_USER_SOURCES);
          }
        else
          {
            if (send_element_error_to_client ("create_user", element_name,
                                              write_to_client,
                                              write_to_client_data))
              {
                error_send_to_client (error);
                return;
              }
            set_client_state (CLIENT_AUTHENTIC);
            g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Error");
          }
        break;

      case CLIENT_CREATE_USER_GROUPS:
        if (strcasecmp ("GROUP", element_name) == 0)
          {
            const gchar* attribute;
            if (find_attribute (attribute_names, attribute_values, "id",
                                &attribute))
              array_add (create_user_data->groups, g_strdup (attribute));
            set_client_state (CLIENT_CREATE_USER_GROUPS_GROUP);
          }
        ELSE_ERROR ("create_user");

      case CLIENT_CREATE_USER_SOURCES:
        if (strcasecmp ("SOURCE", element_name) == 0)
          set_client_state (CLIENT_CREATE_USER_SOURCES_SOURCE);
        else
          {
            if (send_element_error_to_client ("create_user", element_name,
                                              write_to_client,
                                              write_to_client_data))
              {
                error_send_to_client (error);
                return;
              }
            set_client_state (CLIENT_AUTHENTIC);
            g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                         "Error");
          }
        break;

      case CLIENT_MODIFY_NOTE:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_NOTE_ACTIVE);
        else if (strcasecmp ("HOSTS", element_name) == 0)
          set_client_state (CLIENT_MODIFY_NOTE_HOSTS);
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_NOTE_PORT);
        else if (strcasecmp ("RESULT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_note_data->result_id);
            if (modify_note_data->result_id
                && modify_note_data->result_id[0] == '\0')
              {
                g_free (modify_note_data->result_id);
                modify_note_data->result_id = NULL;
              }
            set_client_state (CLIENT_MODIFY_NOTE_RESULT);
          }
        else if (strcasecmp ("SEVERITY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_NOTE_SEVERITY);
        else if (strcasecmp ("TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_note_data->task_id);
            if (modify_note_data->task_id
                && modify_note_data->task_id[0] == '\0')
              {
                g_free (modify_note_data->task_id);
                modify_note_data->task_id = NULL;
              }
            set_client_state (CLIENT_MODIFY_NOTE_TASK);
          }
        else if (strcasecmp ("TEXT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_NOTE_TEXT);
        else if (strcasecmp ("THREAT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_NOTE_THREAT);
        ELSE_ERROR ("modify_note");

      case CLIENT_MODIFY_OVERRIDE:
        if (strcasecmp ("ACTIVE", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_ACTIVE);
        else if (strcasecmp ("HOSTS", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_HOSTS);
        else if (strcasecmp ("NEW_SEVERITY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_NEW_SEVERITY);
        else if (strcasecmp ("NEW_THREAT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_NEW_THREAT);
        else if (strcasecmp ("PORT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_PORT);
        else if (strcasecmp ("RESULT", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_override_data->result_id);
            if (modify_override_data->result_id
                && modify_override_data->result_id[0] == '\0')
              {
                g_free (modify_override_data->result_id);
                modify_override_data->result_id = NULL;
              }
            set_client_state (CLIENT_MODIFY_OVERRIDE_RESULT);
          }
        else if (strcasecmp ("SEVERITY", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_SEVERITY);
        else if (strcasecmp ("TASK", element_name) == 0)
          {
            append_attribute (attribute_names, attribute_values, "id",
                              &modify_override_data->task_id);
            if (modify_override_data->task_id
                && modify_override_data->task_id[0] == '\0')
              {
                g_free (modify_override_data->task_id);
                modify_override_data->task_id = NULL;
              }
            set_client_state (CLIENT_MODIFY_OVERRIDE_TASK);
          }
        else if (strcasecmp ("TEXT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_TEXT);
        else if (strcasecmp ("THREAT", element_name) == 0)
          set_client_state (CLIENT_MODIFY_OVERRIDE_THREAT);
        ELSE_ERROR ("modify_override");

      case CLIENT_RUN_WIZARD:
        if (strcasecmp ("MODE", element_name) == 0)
          {
            set_client_state (CLIENT_RUN_WIZARD_MODE);
          }
        else if (strcasecmp ("NAME", element_name) == 0)
          {
            set_client_state (CLIENT_RUN_WIZARD_NAME);
          }
        else if (strcasecmp ("PARAMS", element_name) == 0)
          {
            run_wizard_data->params = make_array ();
            set_client_state (CLIENT_RUN_WIZARD_PARAMS);
          }
        ELSE_ERROR ("run_wizard");

      case CLIENT_RUN_WIZARD_PARAMS:
        if (strcasecmp ("PARAM", element_name) == 0)
          {
            assert (run_wizard_data->param == NULL);
            run_wizard_data->param = g_malloc (sizeof (name_value_t));
            run_wizard_data->param->name = NULL;
            run_wizard_data->param->value = NULL;
            set_client_state (CLIENT_RUN_WIZARD_PARAMS_PARAM);
          }
        ELSE_ERROR ("run_wizard");

      case CLIENT_RUN_WIZARD_PARAMS_PARAM:
        if (strcasecmp ("NAME", element_name) == 0)
          {
            set_client_state (CLIENT_RUN_WIZARD_PARAMS_PARAM_NAME);
          }
        else if (strcasecmp ("VALUE", element_name) == 0)
          {
            set_client_state (CLIENT_RUN_WIZARD_PARAMS_PARAM_VALUE);
          }
        ELSE_ERROR ("run_wizard");

      default:
        /* Send a generic response. */
        if (send_element_error_to_client ("omp", element_name,
                                          write_to_client,
                                          write_to_client_data))
          {
            error_send_to_client (error);
            return;
          }
        set_client_state (CLIENT_AUTHENTIC);
        g_set_error (error,
                     G_MARKUP_ERROR,
                     G_MARKUP_ERROR_UNKNOWN_ELEMENT,
                     "Error");
        break;
    }

  return;
}

/**
 * @brief Send XML for an NVT.
 *
 * The caller must send the closing NVT tag.
 *
 * @param[in]  nvts        The NVT.
 * @param[in]  details     If true, detailed XML, else simple XML.
 * @param[in]  preferences If true, included preferences.
 * @param[in]  pref_count  Preference count.  Used if details is true.
 * @param[in]  timeout     Timeout.  Used if details is true.
 * @param[in]  config      Config, used if preferences is true.
 * @param[in]  write_to_client       Function to write to client.
 * @param[in]  write_to_client_data  Argument to \p write_to_client.
 *
 * @return TRUE if out of space in to_client buffer, else FALSE.
 */
static gboolean
send_nvt (iterator_t *nvts, int details, int preferences, int pref_count,
          const char *timeout, config_t config,
          int (*write_to_client) (const char *, void*),
          void* write_to_client_data)
{
  gchar *msg;

  msg = get_nvti_xml (nvts, details, pref_count, preferences, timeout, config,
                      0);
  if (send_to_client (msg, write_to_client, write_to_client_data))
    {
      g_free (msg);
      return TRUE;
    }
  g_free (msg);
  return FALSE;
}

/**
 * @brief Send XML for the reports of a task.
 *
 * @param[in]  task             The task.
 * @param[in]  apply_overrides  Whether to apply overrides.
 * @param[in]  write_to_client       Function to write to client.
 * @param[in]  write_to_client_data  Argument to \p write_to_client.
 *
 * @return 0 success, -4 out of space in to_client,
 *         -5 failed to get report counts, -6 failed to get timestamp.
 */
static int
send_reports (task_t task, int apply_overrides, int min_qod,
              int (*write_to_client) (const char*, void*),
              void* write_to_client_data)
{
  iterator_t iterator;
  report_t index;

  if (send_to_client ("<reports>", write_to_client, write_to_client_data))
    return -4;

  init_report_iterator_task (&iterator, task);
  while (next_report (&iterator, &index))
    {
      gchar *uuid, *timestamp, *msg;
      int debugs, false_positives, holes, infos, logs, warnings, run_status;
      double severity;
      char *scan_end;

      uuid = report_uuid (index);

      if (report_counts (uuid, &debugs, &holes, &infos, &logs, &warnings,
                         &false_positives, &severity, apply_overrides,
                         0, min_qod))
        {
          free (uuid);
          return -5;
        }

      if (report_timestamp (uuid, &timestamp))
        {
          free (uuid);
          return -6;
        }

      tracef ("     %s\n", uuid);

      report_scan_run_status (index, &run_status);
      scan_end = scan_end_time (index);
      msg = g_strdup_printf ("<report"
                             " id=\"%s\">"
                             "<timestamp>%s</timestamp>"
                             "<scan_end>%s</scan_end>"
                             "<scan_run_status>%s</scan_run_status>"
                             "<result_count>"
                             "<debug>%i</debug>"
                             "<hole>%i</hole>"
                             "<info>%i</info>"
                             "<log>%i</log>"
                             "<warning>%i</warning>"
                             "<false_positive>%i</false_positive>"
                             "</result_count>"
                             "<severity>%1.1f</severity>"
                             "</report>",
                             uuid,
                             timestamp,
                             scan_end,
                             run_status_name
                              (run_status ? run_status
                                          : TASK_STATUS_INTERNAL_ERROR),
                             debugs,
                             holes,
                             infos,
                             logs,
                             warnings,
                             false_positives,
                             severity);
      free (scan_end);
      g_free (timestamp);
      if (send_to_client (msg, write_to_client, write_to_client_data))
        {
          g_free (msg);
          free (uuid);
          return -4;
        }
      g_free (msg);
      free (uuid);
    }
  cleanup_iterator (&iterator);

  if (send_to_client ("</reports>", write_to_client, write_to_client_data))
    return -4;

  return 0;
}

/**
 * @brief Convert \n's to real newline's.
 *
 * @param[in]  text  The text in which to insert newlines.
 *
 * @return A newly allocated version of text.
 */
static gchar*
convert_to_newlines (const char *text)
{
  char *nptr, *new;

  new = g_malloc (strlen (text) + 1);
  nptr = new;
  while (*text)
    if (*text == '\\')
      {
         /* Convert "\\n" to '\n' */
         if (*(text+1) == 'n')
           {
             text += 2;
             *nptr++ = '\n';
           }
         /* Skip "\\r" */
         else if (*(text+1) == 'r')
           text += 2;
         else
           *nptr++ = *text++;
      }
    else
      *nptr++ = *text++;
  *nptr = '\0';

  return new;
}

/**
 * @brief Format XML into a buffer.
 *
 * @param[in]  buffer  Buffer.
 * @param[in]  format  Format string for XML.
 * @param[in]  ...     Arguments for format string.
 */
static void
buffer_xml_append_printf (GString *buffer, const char *format, ...)
{
  va_list args;
  gchar *msg;
  va_start (args, format);
  msg = g_markup_vprintf_escaped (format, args);
  va_end (args);
  g_string_append (buffer, msg);
  g_free (msg);
}

/**
 * @brief Get substring of UTF8 string.
 *
 * @param[in]  str        String
 * @param[in]  start_pos  Start.
 * @param[in]  end_pos    End.
 *
 * @return Substring.
 */
static gchar *
utf8_substring (const gchar *str, glong start_pos, glong end_pos)
{
  gchar *start, *end, *out;

  /* TODO This is a copy of g_utf8_substring from glib 2.38.2.  Once our glib
   * minimum goes past 2.30 we can just use g_utf8_substring. */

  start = g_utf8_offset_to_pointer (str, start_pos);
  end = g_utf8_offset_to_pointer (start, end_pos - start_pos);

  out = g_malloc (end - start + 1);
  memcpy (out, start, end - start);
  out[end - start] = 0;

  return out;
}

/**
 * @brief Buffer XML for some notes.
 *
 * @param[in]  buffer                 Buffer into which to buffer notes.
 * @param[in]  notes                  Notes iterator.
 * @param[in]  include_notes_details  Whether to include details of notes.
 * @param[in]  include_result         Whether to include associated result.
 * @param[out] count                  Number of notes.
 */
static void
buffer_notes_xml (GString *buffer, iterator_t *notes, int include_notes_details,
                  int include_result, int *count)
{
  while (next (notes))
    {
      char *uuid_task, *uuid_result;

      if (count)
        (*count)++;

      if (note_iterator_task (notes))
        task_uuid (note_iterator_task (notes),
                   &uuid_task);
      else
        uuid_task = NULL;

      if (note_iterator_result (notes))
        result_uuid (note_iterator_result (notes),
                     &uuid_result);
      else
        uuid_result = NULL;

      buffer_xml_append_printf (buffer,
                                "<note id=\"%s\">"
                                "<permissions>",
                                get_iterator_uuid (notes));

      if (current_credentials.username
          && get_iterator_owner_name (notes)
          && (strcmp (get_iterator_owner_name (notes),
                      current_credentials.username)
              == 0))
        buffer_xml_append_printf (buffer,
                                  "<permission><name>Everything</name></permission>"
                                  "</permissions>");
      else
        {
          iterator_t perms;
          get_data_t perms_get;

          memset (&perms_get, '\0', sizeof (perms_get));
          perms_get.filter = g_strdup_printf ("resource_uuid=%s"
                                              " owner=any"
                                              " permission=any",
                                              get_iterator_uuid (notes));
          init_permission_iterator (&perms, &perms_get);
          g_free (perms_get.filter);
          while (next (&perms))
            buffer_xml_append_printf (buffer,
                                      "<permission><name>%s</name></permission>",
                                      get_iterator_name (&perms));
          cleanup_iterator (&perms);

          buffer_xml_append_printf (buffer, "</permissions>");
        }

      if (include_notes_details == 0)
        {
          const char *text = note_iterator_text (notes);
          gchar *excerpt = utf8_substring (text, 0, 60);
          /* This must match send_get_common. */
          buffer_xml_append_printf (buffer,
                                    "<owner><name>%s</name></owner>"
                                    "<nvt oid=\"%s\">"
                                    "<name>%s</name>"
                                    "</nvt>"
                                    "<creation_time>%s</creation_time>"
                                    "<modification_time>%s</modification_time>"
                                    "<writable>1</writable>"
                                    "<in_use>0</in_use>"
                                    "<active>%i</active>"
                                    "<text excerpt=\"%i\">%s</text>"
                                    "<orphan>%i</orphan>"
                                    "<user_tags>"
                                    "<count>%i</count>"
                                    "</user_tags>"
                                    "</note>",
                                    get_iterator_owner_name (notes)
                                     ? get_iterator_owner_name (notes)
                                     : "",
                                    note_iterator_nvt_oid (notes),
                                    note_iterator_nvt_name (notes),
                                    get_iterator_creation_time (notes),
                                    get_iterator_modification_time (notes),
                                    note_iterator_active (notes),
                                    strlen (excerpt) < strlen (text),
                                    excerpt,
                                    ((note_iterator_task (notes)
                                      && (uuid_task == NULL))
                                     || (note_iterator_result (notes)
                                         && (uuid_result == NULL))),
                                    resource_tag_count ("note",
                                                        get_iterator_resource
                                                          (notes),
                                                        1));
          g_free (excerpt);
        }
      else
        {
          char *name_task;
          int trash_task;
          time_t end_time;
          iterator_t tags;

          if (uuid_task)
            {
              name_task = task_name (note_iterator_task (notes));
              trash_task = task_in_trash (note_iterator_task (notes));
            }
          else
            {
              name_task = NULL;
              trash_task = 0;
            }

          end_time = note_iterator_end_time (notes);

          /* This must match send_get_common. */
          buffer_xml_append_printf
           (buffer,
            "<owner><name>%s</name></owner>"
            "<nvt oid=\"%s\"><name>%s</name></nvt>"
            "<creation_time>%s</creation_time>"
            "<modification_time>%s</modification_time>"
            "<writable>1</writable>"
            "<in_use>0</in_use>"
            "<active>%i</active>"
            "<end_time>%s</end_time>"
            "<text>%s</text>"
            "<hosts>%s</hosts>"
            "<port>%s</port>"
            "<severity>%s</severity>"
            "<threat>%s</threat>"
            "<task id=\"%s\"><name>%s</name><trash>%i</trash></task>"
            "<orphan>%i</orphan>",
            get_iterator_owner_name (notes)
             ? get_iterator_owner_name (notes)
             : "",
            note_iterator_nvt_oid (notes),
            note_iterator_nvt_name (notes),
            get_iterator_creation_time (notes),
            get_iterator_modification_time (notes),
            note_iterator_active (notes),
            end_time > 1 ? iso_time (&end_time) : "",
            note_iterator_text (notes),
            note_iterator_hosts (notes)
             ? note_iterator_hosts (notes) : "",
            note_iterator_port (notes)
             ? note_iterator_port (notes) : "",
            note_iterator_severity (notes)
             ? note_iterator_severity (notes) : "",
            note_iterator_threat (notes)
             ? note_iterator_threat (notes) : "",
            uuid_task ? uuid_task : "",
            name_task ? name_task : "",
            trash_task,
            ((note_iterator_task (notes) && (uuid_task == NULL))
             || (note_iterator_result (notes) && (uuid_result == NULL))));

          free (name_task);

          if (include_result && note_iterator_result (notes))
            {
              iterator_t results;

              init_result_iterator (&results, 0,
                                    note_iterator_result (notes),
                                    0, 1, 1, NULL, NULL, 1, NULL, 0, NULL,
                                    NULL, 0);
              while (next (&results))
                buffer_results_xml (buffer,
                                    &results,
                                    0,
                                    0,  /* Notes. */
                                    0,  /* Note details. */
                                    0,  /* Overrides. */
                                    0,  /* Override details. */
                                    0,  /* Tags. */
                                    0,  /* Tag details. */
                                    0,  /* Result details. */
                                    NULL,
                                    NULL,
                                    0);
              cleanup_iterator (&results);
            }
          else
            buffer_xml_append_printf (buffer,
                                      "<result id=\"%s\"/>",
                                      uuid_result ? uuid_result : "");

          buffer_xml_append_printf (buffer,
                                    "<user_tags>"
                                    "<count>%i</count>",
                                    resource_tag_count ("note",
                                                        get_iterator_resource
                                                          (notes),
                                                        1));

          init_resource_tag_iterator (&tags, "note",
                                      get_iterator_resource (notes),
                                      1, NULL, 1);

          while (next (&tags))
            {
              buffer_xml_append_printf (buffer,
                                        "<tag id=\"%s\">"
                                        "<name>%s</name>"
                                        "<value>%s</value>"
                                        "<comment>%s</comment>"
                                        "</tag>",
                                        resource_tag_iterator_uuid (&tags),
                                        resource_tag_iterator_name (&tags),
                                        resource_tag_iterator_value (&tags),
                                        resource_tag_iterator_comment (&tags));
            }

          cleanup_iterator (&tags);

          buffer_xml_append_printf (buffer,
                                    "</user_tags>"
                                    "</note>");
        }
      free (uuid_task);
      free (uuid_result);
    }
}

/**
 * @brief Buffer XML for some overrides.
 *
 * @param[in]  buffer                     Buffer into which to buffer overrides.
 * @param[in]  overrides                  Overrides iterator.
 * @param[in]  include_overrides_details  Whether to include details of overrides.
 * @param[in]  include_result             Whether to include associated result.
 * @param[out] count                      Number of overrides.
 */
static void
buffer_overrides_xml (GString *buffer, iterator_t *overrides,
                      int include_overrides_details, int include_result,
                      int *count)
{
  while (next (overrides))
    {
      char *uuid_task, *uuid_result;

      if (count)
        (*count)++;

      if (override_iterator_task (overrides))
        task_uuid (override_iterator_task (overrides),
                   &uuid_task);
      else
        uuid_task = NULL;

      if (override_iterator_result (overrides))
        result_uuid (override_iterator_result (overrides),
                     &uuid_result);
      else
        uuid_result = NULL;

      buffer_xml_append_printf (buffer,
                                "<override id=\"%s\">"
                                "<permissions>",
                                get_iterator_uuid (overrides));

      if (current_credentials.username
          && get_iterator_owner_name (overrides)
          && (strcmp (get_iterator_owner_name (overrides),
                      current_credentials.username)
              == 0))
        buffer_xml_append_printf (buffer,
                                  "<permission><name>Everything</name></permission>"
                                  "</permissions>");
      else
        {
          iterator_t perms;
          get_data_t perms_get;

          memset (&perms_get, '\0', sizeof (perms_get));
          perms_get.filter = g_strdup_printf ("resource_uuid=%s"
                                              " owner=any"
                                              " permission=any",
                                              get_iterator_uuid (overrides));
          init_permission_iterator (&perms, &perms_get);
          g_free (perms_get.filter);
          while (next (&perms))
            buffer_xml_append_printf (buffer,
                                      "<permission><name>%s</name></permission>",
                                      get_iterator_name (&perms));
          cleanup_iterator (&perms);

          buffer_xml_append_printf (buffer, "</permissions>");
        }

      if (include_overrides_details == 0)
        {
          const char *text = override_iterator_text (overrides);
          gchar *excerpt = utf8_substring (text, 0, 60);
          /* This must match send_get_common. */
          buffer_xml_append_printf (buffer,
                                    "<owner><name>%s</name></owner>"
                                    "<nvt oid=\"%s\">"
                                    "<name>%s</name>"
                                    "</nvt>"
                                    "<creation_time>%s</creation_time>"
                                    "<modification_time>%s</modification_time>"
                                    "<writable>1</writable>"
                                    "<in_use>0</in_use>"
                                    "<active>%i</active>"
                                    "<text excerpt=\"%i\">%s</text>"
                                    "<threat>%s</threat>"
                                    "<severity>%s</severity>"
                                    "<new_threat>%s</new_threat>"
                                    "<new_severity>%s</new_severity>"
                                    "<orphan>%i</orphan>"
                                    "<user_tags>"
                                    "<count>%i</count>"
                                    "</user_tags>"
                                    "</override>",
                                    get_iterator_owner_name (overrides)
                                     ? get_iterator_owner_name (overrides)
                                     : "",
                                    override_iterator_nvt_oid (overrides),
                                    override_iterator_nvt_name (overrides),
                                    get_iterator_creation_time (overrides),
                                    get_iterator_modification_time (overrides),
                                    override_iterator_active (overrides),
                                    strlen (excerpt) < strlen (text),
                                    excerpt,
                                    override_iterator_threat (overrides)
                                     ? override_iterator_threat (overrides)
                                     : "",
                                    override_iterator_severity (overrides)
                                     ? override_iterator_severity (overrides)
                                     : "",
                                    override_iterator_new_threat (overrides),
                                    override_iterator_new_severity (overrides),
                                    ((override_iterator_task (overrides)
                                      && (uuid_task == NULL))
                                     || (override_iterator_result (overrides)
                                         && (uuid_result == NULL))),
                                    resource_tag_count ("override",
                                                        get_iterator_resource
                                                          (overrides),
                                                        1));
          g_free (excerpt);
        }
      else
        {
          char *name_task;
          int trash_task;
          time_t end_time;
          iterator_t tags;

          if (uuid_task)
            {
              name_task = task_name (override_iterator_task (overrides));
              trash_task = task_in_trash (override_iterator_task (overrides));
            }
          else
            {
              name_task = NULL;
              trash_task = 0;
            }

          end_time = override_iterator_end_time (overrides);

          /* This must match send_get_common. */
          buffer_xml_append_printf
           (buffer,
            "<owner><name>%s</name></owner>"
            "<nvt oid=\"%s\"><name>%s</name></nvt>"
            "<creation_time>%s</creation_time>"
            "<modification_time>%s</modification_time>"
            "<writable>1</writable>"
            "<in_use>0</in_use>"
            "<active>%i</active>"
            "<end_time>%s</end_time>"
            "<text>%s</text>"
            "<hosts>%s</hosts>"
            "<port>%s</port>"
            "<threat>%s</threat>"
            "<severity>%s</severity>"
            "<new_threat>%s</new_threat>"
            "<new_severity>%s</new_severity>"
            "<task id=\"%s\"><name>%s</name><trash>%i</trash></task>"
            "<orphan>%i</orphan>",
            get_iterator_owner_name (overrides)
             ? get_iterator_owner_name (overrides)
             : "",
            override_iterator_nvt_oid (overrides),
            override_iterator_nvt_name (overrides),
            get_iterator_creation_time (overrides),
            get_iterator_modification_time (overrides),
            override_iterator_active (overrides),
            end_time > 1 ? iso_time (&end_time) : "",
            override_iterator_text (overrides),
            override_iterator_hosts (overrides)
             ? override_iterator_hosts (overrides) : "",
            override_iterator_port (overrides)
             ? override_iterator_port (overrides) : "",
            override_iterator_threat (overrides)
             ? override_iterator_threat (overrides) : "",
            override_iterator_severity (overrides)
             ? override_iterator_severity (overrides) : "",
            override_iterator_new_threat (overrides),
            override_iterator_new_severity (overrides),
            uuid_task ? uuid_task : "",
            name_task ? name_task : "",
            trash_task,
            ((override_iterator_task (overrides) && (uuid_task == NULL))
             || (override_iterator_result (overrides) && (uuid_result == NULL))));

          free (name_task);

          if (include_result && override_iterator_result (overrides))
            {
              iterator_t results;

              init_result_iterator (&results, 0,
                                    override_iterator_result (overrides),
                                    0, 1, 1, NULL, NULL, 1, NULL, 0, NULL,
                                    NULL, 0);
              while (next (&results))
                buffer_results_xml (buffer,
                                    &results,
                                    0,
                                    0,  /* Overrides. */
                                    0,  /* Override details. */
                                    0,  /* Overrides. */
                                    0,  /* Override details. */
                                    0,  /* Tags. */
                                    0,  /* Tag details. */
                                    0,  /* Result details. */
                                    NULL,
                                    NULL,
                                    0);
              cleanup_iterator (&results);
            }
          else
            buffer_xml_append_printf (buffer,
                                      "<result id=\"%s\"/>",
                                      uuid_result ? uuid_result : "");

          buffer_xml_append_printf (buffer,
                                    "<user_tags>"
                                    "<count>%i</count>",
                                    resource_tag_count ("override",
                                                        get_iterator_resource
                                                          (overrides),
                                                        1));

          init_resource_tag_iterator (&tags, "override",
                                      get_iterator_resource (overrides),
                                      1, NULL, 1);

          while (next (&tags))
            {
              buffer_xml_append_printf (buffer,
                                        "<tag id=\"%s\">"
                                        "<name>%s</name>"
                                        "<value>%s</value>"
                                        "<comment>%s</comment>"
                                        "</tag>",
                                        resource_tag_iterator_uuid (&tags),
                                        resource_tag_iterator_name (&tags),
                                        resource_tag_iterator_value (&tags),
                                        resource_tag_iterator_comment (&tags));
            }

          cleanup_iterator (&tags);

          buffer_xml_append_printf (buffer,
                                    "</user_tags>"
                                    "</override>");
        }
      free (uuid_task);
      free (uuid_result);
    }
}

/* External for manage.c. */
/**
 * @brief Buffer XML for the NVT preference of a config.
 *
 * @param[in]  buffer  Buffer.
 * @param[in]  prefs   NVT preference iterator.
 * @param[in]  config  Config.
 * @param[in]  hide_passwords  Whether to hide passwords.
 */
void
buffer_config_preference_xml (GString *buffer, iterator_t *prefs,
                              config_t config, int hide_passwords)
{
  char *real_name, *type, *value, *nvt;
  const char *default_value;
  char *oid = NULL;

  real_name = nvt_preference_iterator_real_name (prefs);
  type = nvt_preference_iterator_type (prefs);
  value = nvt_preference_iterator_config_value (prefs, config);
  nvt = nvt_preference_iterator_nvt (prefs);

  default_value = nvt_preference_iterator_value (prefs);

  if (nvt) oid = nvt_oid (nvt);

  buffer_xml_append_printf (buffer,
                            "<preference>"
                            "<nvt oid=\"%s\"><name>%s</name></nvt>"
                            "<name>%s</name>"
                            "<type>%s</type>",
                            oid ? oid : "",
                            nvt ? nvt : "",
                            real_name ? real_name : "",
                            type ? type : "");

  if (value
      && type
      && (strcmp (type, "radio") == 0))
    {
      /* Handle the other possible values. */
      char *pos = strchr (value, ';');
      if (pos) *pos = '\0';
      buffer_xml_append_printf (buffer, "<value>%s</value>", value);
      while (pos)
        {
          char *pos2 = strchr (++pos, ';');
          if (pos2) *pos2 = '\0';
          buffer_xml_append_printf (buffer, "<alt>%s</alt>", pos);
          pos = pos2;
        }
    }
  else if (value
           && type
           && hide_passwords
           && (strcmp (type, "password") == 0))
    buffer_xml_append_printf (buffer, "<value></value>");
  else
    buffer_xml_append_printf (buffer, "<value>%s</value>", value ? value : "");

  if (default_value
      && type
      && (strcmp (type, "radio") == 0))
    {
      /* Handle the other possible values. */
      char *pos = strchr (default_value, ';');
      if (pos) *pos = '\0';
      buffer_xml_append_printf (buffer, "<default>%s</default>", default_value);
    }
  else if (default_value
           && type
           && (strcmp (type, "password") == 0))
    buffer_xml_append_printf (buffer, "<default></default>");
  else
    buffer_xml_append_printf (buffer, "<default>%s</default>", default_value
                                                               ? default_value
                                                               : "");

  buffer_xml_append_printf (buffer, "</preference>");

  free (real_name);
  free (type);
  free (value);
  free (nvt);
  free (oid);
}

/**
 * @brief Compare two string with the "diff" command.
 *
 * @param[in]  one     First string.
 * @param[in]  two     Second string.
 *
 * @return Output of "diff", or NULL on error.
 */
gchar *
strdiff (const gchar *one, const gchar *two)
{
  gchar **cmd, *ret, *one_file, *two_file, *old_lc_all, *old_language;
  gint exit_status;
  gchar *standard_out = NULL;
  gchar *standard_err = NULL;
  char dir[] = "/tmp/openvasmd-strdiff-XXXXXX";
  GError *error = NULL;

  if (mkdtemp (dir) == NULL)
    return NULL;

  one_file = g_build_filename (dir, "Report 1", NULL);

  g_file_set_contents (one_file, one, strlen (one), &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      openvas_file_remove_recurse (dir);
      g_free (one_file);
      return NULL;
    }

  two_file = g_build_filename (dir, "Report 2", NULL);

  g_file_set_contents (two_file, two, strlen (two), &error);
  if (error)
    {
      g_warning ("%s", error->message);
      g_error_free (error);
      openvas_file_remove_recurse (dir);
      g_free (one_file);
      g_free (two_file);
      return NULL;
    }

  old_lc_all = getenv ("LC_ALL") ? g_strdup (getenv ("LC_ALL")) : NULL;
  if (setenv ("LC_ALL", "C", 1) == -1)
    {
      g_warning ("%s: failed to set LC_ALL\n", __FUNCTION__);
      return NULL;
    }

  old_language = getenv ("LANGUAGE") ? g_strdup (getenv ("LANGUAGE")) : NULL;
  if (setenv ("LANGUAGE", "C", 1) == -1)
    {
      g_warning ("%s: failed to set LANGUAGE\n", __FUNCTION__);
      return NULL;
    }

  cmd = (gchar **) g_malloc (5 * sizeof (gchar *));

  cmd[0] = g_strdup ("diff");
  cmd[1] = g_strdup ("-u");
  cmd[2] = g_strdup ("Report 1");
  cmd[3] = g_strdup ("Report 2");
  cmd[4] = NULL;
  g_debug ("%s: Spawning in %s: %s \"%s\" \"%s\"\n",
           __FUNCTION__, dir,
           cmd[0], cmd[1], cmd[2]);
  if ((g_spawn_sync (dir,
                     cmd,
                     NULL,                 /* Environment. */
                     G_SPAWN_SEARCH_PATH,
                     NULL,                 /* Setup func. */
                     NULL,
                     &standard_out,
                     &standard_err,
                     &exit_status,
                     NULL) == FALSE)
      || (WIFEXITED (exit_status) == 0)
      || WEXITSTATUS (exit_status))
    {
      if (WEXITSTATUS (exit_status) == 1)
        ret = standard_out;
      else
        {
          g_debug ("%s: failed to run diff: %d (WIF %i, WEX %i)",
                   __FUNCTION__,
                   exit_status,
                   WIFEXITED (exit_status),
                   WEXITSTATUS (exit_status));
          g_debug ("%s: stdout: %s\n", __FUNCTION__, standard_out);
          g_debug ("%s: stderr: %s\n", __FUNCTION__, standard_err);
          ret = NULL;
          g_free (standard_out);
        }
    }
  else
    ret = standard_out;

  if (old_lc_all && (setenv ("LC_ALL", old_lc_all, 1) == -1))
    {
      g_warning ("%s: failed to reset LC_ALL\n", __FUNCTION__);
      ret = NULL;
    }
  else if (old_language && (setenv ("LANGUAGE", old_language, 1) == -1))
    {
      g_warning ("%s: failed to reset LANGUAGE\n", __FUNCTION__);
      ret = NULL;
    }

  g_free (old_lc_all);
  g_free (old_language);
  g_free (cmd[0]);
  g_free (cmd[1]);
  g_free (cmd[2]);
  g_free (cmd[3]);
  g_free (cmd);
  g_free (standard_err);
  g_free (one_file);
  g_free (two_file);
  openvas_file_remove_recurse (dir);

  return ret;
}

/**
 * @brief Buffer XML for notes of a result.
 *
 * @param[in]  buffer                 Buffer into which to buffer results.
 * @param[in]  result                 Result.
 * @param[in]  task                   Task associated with result.
 * @param[in]  include_notes_details  Whether to include details of notes.
 */
static void
buffer_result_notes_xml (GString *buffer, result_t result, task_t task,
                         int include_notes_details)
{
  g_string_append (buffer, "<notes>");

  if (task)
    {
      get_data_t get;
      iterator_t notes;
      memset (&get, '\0', sizeof (get));
      /* Most recent first. */
      get.filter = "sort-reverse=created owner=any permission=any";
      init_note_iterator (&notes,
                          &get,
                          0,
                          result,
                          task);
      buffer_notes_xml (buffer,
                        &notes,
                        include_notes_details,
                        0,
                        NULL);
      cleanup_iterator (&notes);
    }

  g_string_append (buffer, "</notes>");
}

/**
 * @brief Buffer XML for overrides of a result.
 *
 * @param[in]  buffer                 Buffer into which to buffer results.
 * @param[in]  result                 Result.
 * @param[in]  task                   Task associated with result.
 * @param[in]  include_overrides_details  Whether to include details of overrides.
 */
static void
buffer_result_overrides_xml (GString *buffer, result_t result, task_t task,
                             int include_overrides_details)
{
  g_string_append (buffer, "<overrides>");

  if (task)
    {
      get_data_t get;
      iterator_t overrides;
      memset (&get, '\0', sizeof (get));
      /* Most recent first. */
      get.filter = "sort-reverse=created owner=any permission=any";
      init_override_iterator (&overrides,
                              &get,
                              0,
                              result,
                              task);
      buffer_overrides_xml (buffer,
                            &overrides,
                            include_overrides_details,
                            0,
                            NULL);
      cleanup_iterator (&overrides);
    }

  g_string_append (buffer, "</overrides>");
}

/**
 * @brief Add a detail block to a XML buffer.
 */
#define ADD_DETAIL(buff, dname, dvalue) do { \
                                buffer_xml_append_printf (buff,   \
                                                          "<detail>"          \
                                                          "<name>%s</name>"   \
                                                          "<value>%s</value>" \
                                                          "</detail>",        \
                                                          dname,              \
                                                          dvalue);            \
                                } while (0)

static void
results_xml_append_cert (GString *buffer, const char *oid)
{
  iterator_t cert_refs_iterator;

  buffer_xml_append_printf (buffer, "<cert>");
  if (manage_cert_loaded ())
    {
      init_nvt_cert_bund_adv_iterator (&cert_refs_iterator, oid, 0, 0);
      while (next (&cert_refs_iterator))
        {
          g_string_append_printf
           (buffer, "<cert_ref type=\"CERT-Bund\" id=\"%s\"/>",
            get_iterator_name (&cert_refs_iterator));
        }
      cleanup_iterator (&cert_refs_iterator);

      init_nvt_dfn_cert_adv_iterator (&cert_refs_iterator, oid, 0, 0);
      while (next (&cert_refs_iterator))
        {
          g_string_append_printf
           (buffer, "<cert_ref type=\"DFN-CERT\" id=\"%s\"/>",
            get_iterator_name (&cert_refs_iterator));
        }
      cleanup_iterator (&cert_refs_iterator);
    }
  else
    g_string_append_printf (buffer,
                            "<warning>database not available</warning>");
  buffer_xml_append_printf (buffer, "</cert>");
}

static void
results_xml_append_nvt (iterator_t *results, GString *buffer)
{
  const char *oid = result_iterator_nvt_oid (results);

  assert (results);
  assert (buffer);

  if (g_str_has_prefix (oid, "CVE-"))
    {
      gchar *cvss_base;
      cvss_base = cve_cvss_base (oid);
      buffer_xml_append_printf (buffer,
                                "<nvt oid=\"%s\">"
                                "<type>cve</type>"
                                "<name>%s</name>"
                                "<cvss_base>%s</cvss_base>"
                                "<cpe id='%s'/>"
                                "<cve>%s</cve>"
                                "</nvt>",
                                oid,
                                oid,
                                cvss_base,
                                result_iterator_port (results),
                                oid);
      g_free (cvss_base);
      return;
    }

  if (g_str_has_prefix (oid, "oval:"))
    {
      int ret;
      char *cves;
      get_data_t get;
      iterator_t iterator;

      memset (&get, '\0', sizeof (get));
      get.id = g_strdup (oid);
      ret = init_ovaldef_info_iterator (&iterator, &get, NULL);
      if (ret)
        assert (0);
      if (!next (&iterator))
        abort ();
      cves = ovaldef_cves (oid);
      buffer_xml_append_printf
       (buffer,
        "<nvt oid=\"%s\"><name>%s</name><family/><cvss_base>%s</cvss_base>"
        "<cve>%s</cve><bid/><tags>summary=%s</tags><xref/>",
        oid,
        ovaldef_info_iterator_title (&iterator),
        ovaldef_info_iterator_max_cvss (&iterator),
        cves ?: "",
        ovaldef_info_iterator_description (&iterator));
      g_free (cves);
      g_free (get.id);
      cleanup_iterator (&iterator);
    }
  else
    {
      const char *cvss_base = result_iterator_nvt_cvss_base (results);

      if (!cvss_base && !strcmp (oid, "0"))
        cvss_base = "0.0";

      buffer_xml_append_printf
       (buffer,
        "<nvt oid=\"%s\"><name>%s</name><family>%s</family>"
        "<cvss_base>%s</cvss_base><cve>%s</cve><bid>%s</bid>"
        "<xref>%s</xref><tags>%s</tags>",
        oid, result_iterator_nvt_name (results) ?: oid,
        result_iterator_nvt_family (results) ?: "",
        cvss_base ?: "",
        result_iterator_nvt_cve (results) ?: "",
        result_iterator_nvt_bid (results) ?: "",
        result_iterator_nvt_xref (results) ?: "",
        result_iterator_nvt_tag (results) ?: "");
    }

  results_xml_append_cert (buffer, oid);
  buffer_xml_append_printf (buffer, "</nvt>");
}

/** @todo Exported for manage_sql.c. */
/**
 * @brief Buffer XML for some results.
 *
 * @param[in]  buffer                 Buffer into which to buffer results.
 * @param[in]  results                Result iterator.
 * @param[in]  task                   Task associated with results.  Only
 *                                    needed with include_notes or
 *                                    include_overrides.
 * @param[in]  include_notes          Whether to include notes.
 * @param[in]  include_notes_details  Whether to include details of notes.
 * @param[in]  include_overrides          Whether to include overrides.
 * @param[in]  include_overrides_details  Whether to include details of overrides.
 * @param[in]  include_tags           Whether to include user tag count.
 * @param[in]  include_tags_details   Whether to include details of tags.
 * @param[in]  include_details        Whether to include details of the result.
 * @param[in]  delta_state            Delta state of result, or NULL.
 * @param[in]  delta_results          Iterator for delta result to include, or
 *                                    NULL.
 * @param[in]  changed                Whether the result is a "changed" delta.
 */
void
buffer_results_xml (GString *buffer, iterator_t *results, task_t task,
                    int include_notes, int include_notes_details,
                    int include_overrides, int include_overrides_details,
                    int include_tags, int include_tags_details,
                    int include_details,
                    const char *delta_state, iterator_t *delta_results,
                    int changed)
{
  const char *descr = result_iterator_descr (results);
  const char *name, *owner_name, *comment, *creation_time, *modification_time;
  gchar *nl_descr = descr ? convert_to_newlines (descr) : NULL;
  const char *qod = result_iterator_qod (results);
  const char *qod_type = result_iterator_qod_type (results);
  result_t result = result_iterator_result (results);
  char *uuid;
  char *detect_ref, *detect_cpe, *detect_loc, *detect_oid, *detect_name;
  task_t selected_task;

  result_uuid (result, &uuid);

  buffer_xml_append_printf (buffer, "<result id=\"%s\">", uuid);

  selected_task = task;

  name = get_iterator_name (results);
  if (name)
    buffer_xml_append_printf (buffer,
                              "<name>%s</name>",
                              name);

  owner_name = get_iterator_owner_name (results);
  if (owner_name)
    buffer_xml_append_printf (buffer,
                              "<owner><name>%s</name></owner>",
                              owner_name);

  comment = get_iterator_comment (results);
  if (comment)
    buffer_xml_append_printf (buffer,
                              "<comment>%s</comment>",
                              comment);

  creation_time = get_iterator_creation_time (results);
  if (creation_time)
    buffer_xml_append_printf (buffer,
                              "<creation_time>%s</creation_time>",
                              creation_time);

  modification_time = get_iterator_modification_time (results);
  if (modification_time)
    buffer_xml_append_printf (buffer,
                              "<modification_time>%s</modification_time>",
                              modification_time);

  if (include_details)
    {
      char *result_report_id, *result_task_id, *result_task_name;

      if (task == 0)
        selected_task = result_iterator_task (results);

      task_uuid (selected_task, &result_task_id);
      result_task_name = task_name (result_iterator_task (results));
      result_report_id = report_uuid (result_iterator_report (results));

      g_string_append_printf (buffer,
                              "<report id=\"%s\"/>"
                              "<task id=\"%s\"><name>%s</name></task>",
                              result_report_id,
                              result_task_id,
                              result_task_name);

      free (result_report_id);
      free (result_task_id);
      free (result_task_name);
    }

  if (include_tags)
    {
      buffer_xml_append_printf (buffer,
                                "<user_tags>"
                                "<count>%i</count>",
                                resource_tag_count ("result", result, 1));

      if (include_tags_details)
        {
          iterator_t tags;

          init_resource_tag_iterator (&tags, "result", result, 1, NULL, 1);

          while (next (&tags))
            {
              buffer_xml_append_printf (buffer,
                                        "<tag id=\"%s\">"
                                        "<name>%s</name>"
                                        "<value>%s</value>"
                                        "<comment>%s</comment>"
                                        "</tag>",
                                        resource_tag_iterator_uuid (&tags),
                                        resource_tag_iterator_name (&tags),
                                        resource_tag_iterator_value (&tags),
                                        resource_tag_iterator_comment (&tags));
            }

          cleanup_iterator (&tags);
        }

      buffer_xml_append_printf (buffer, "</user_tags>");
    }

  detect_ref = detect_cpe = detect_loc = detect_oid = detect_name = NULL;
  if (result_detection_reference (result, &detect_ref, &detect_cpe, &detect_loc,
                                  &detect_oid, &detect_name) == 0)
    {
      buffer_xml_append_printf (buffer,
                                "<detection>"
                                "<result id=\"%s\">"
                                "<details>",
                                detect_ref);

      ADD_DETAIL(buffer, "product", detect_cpe);
      ADD_DETAIL(buffer, "location", detect_loc);
      ADD_DETAIL(buffer, "source_oid", detect_oid);
      ADD_DETAIL(buffer, "source_name", detect_name);

      buffer_xml_append_printf (buffer,
                                "</details>"
                                "</result>"
                                "</detection>");
    }
  g_free (detect_ref);
  g_free (detect_cpe);
  g_free (detect_loc);
  g_free (detect_oid);
  g_free (detect_name);

  buffer_xml_append_printf
   (buffer, "<host>%s</host><port>%s</port>",
    result_iterator_host (results), result_iterator_port (results));
  results_xml_append_nvt (results, buffer);

  buffer_xml_append_printf
   (buffer,
    "<scan_nvt_version>%s</scan_nvt_version>"
    "<threat>%s</threat>"
    "<severity>%.1f</severity>"
    "<qod><value>%s</value><type>%s</type></qod>"
    "<description>%s</description>",
    result_iterator_scan_nvt_version (results),
    result_iterator_level (results),
    result_iterator_severity_double (results),
    qod ? qod : "",
    qod_type ? qod_type : "",
    descr ? nl_descr : "");

  if (include_overrides)
    buffer_xml_append_printf (buffer,
                              "<original_threat>%s</original_threat>"
                              "<original_severity>%s</original_severity>",
                              result_iterator_original_level (results),
                              result_iterator_original_severity (results));

  free (uuid);

  if (include_notes)
    buffer_result_notes_xml (buffer, result,
                             selected_task, include_notes_details);

  if (include_overrides)
    buffer_result_overrides_xml (buffer, result,
                                 selected_task, include_overrides_details);

  if (delta_state || delta_results)
    {
      g_string_append (buffer, "<delta>");
      if (delta_state)
        g_string_append_printf (buffer, "%s", delta_state);
      if (changed && delta_results)
        {
          gchar *diff, *delta_nl_descr;
          const char *delta_descr;
          buffer_results_xml (buffer, delta_results, selected_task,
                              include_notes, include_notes_details,
                              include_overrides, include_overrides_details,
                              include_tags, include_tags_details,
                              include_details, delta_state, NULL, 0);
          delta_descr = result_iterator_descr (delta_results);
          delta_nl_descr = delta_descr ? convert_to_newlines (delta_descr)
                                       : NULL;
          diff = strdiff (descr ? nl_descr : "",
                          delta_descr ? delta_nl_descr : "");
          g_free (delta_nl_descr);
          if (diff)
            {
              gchar **split, *diff_xml;
              /* Remove the leading filename lines. */
              split = g_strsplit ((gchar*) diff, "\n", 3);
              if (split[0] && split[1] && split[2])
                diff_xml = g_markup_escape_text (split[2], strlen (split[2]));
              else
                diff_xml = g_markup_escape_text (diff, strlen (diff));
              g_strfreev (split);
              g_string_append_printf (buffer, "<diff>%s</diff>", diff_xml);
              g_free (diff_xml);
              g_free (diff);
            }
          else
            g_string_append (buffer, "<diff>Error creating diff.</diff>");
        }

      if (delta_results)
        {
          if (include_notes)
            buffer_result_notes_xml (buffer,
                                     result_iterator_result (delta_results),
                                     selected_task,
                                     include_notes_details);

          if (include_overrides)
            buffer_result_overrides_xml (buffer,
                                         result_iterator_result (delta_results),
                                         selected_task,
                                         include_overrides_details);
        }
      g_string_append (buffer, "</delta>");
    }

  if (descr) g_free (nl_descr);

  g_string_append (buffer, "</result>");
}

#undef ADD_DETAIL

/**
 * @brief Convert ranges to manage ranges.
 *
 * @param[in]  ranges  Ranges buffered in CREATE_PORT_LIST.
 *
 * @return Array of manage ranges on success, else NULL.
 */
static array_t *
convert_to_manage_ranges (array_t *ranges)
{
  if (ranges)
    {
      guint index;
      array_t *manage_ranges;

      manage_ranges = make_array ();

      index = ranges->len;
      while (index--)
        {
          create_port_list_range_t *range;
          range = (create_port_list_range_t*) g_ptr_array_index (ranges,
                                                                 index);
          if (range)
            {
              range_t *manage_range;

              manage_range = g_malloc0 (sizeof (range_t));
              manage_range->comment = range->comment;
              manage_range->end = atoi (range->end);
              manage_range->id = range->id;
              manage_range->start = atoi (range->start);
              if (strcasecmp (range->type, "TCP") == 0)
                manage_range->type = PORT_PROTOCOL_TCP;
              else if (strcasecmp (range->type, "UDP") == 0)
                manage_range->type = PORT_PROTOCOL_UDP;
              else
                manage_range->type = PORT_PROTOCOL_OTHER;
              manage_range->exclude = 0;

              array_add (manage_ranges, manage_range);
            }
        }
      return manage_ranges;
    }
  return NULL;
}

/**
 * @brief Insert else clause for omp_xml_handle_start_element.
 *
 * @param[in]  parent   Parent element.
 * @param[in]  element  Element.
 */
#define CLOSE(parent, element)                                           \
  case parent ## _ ## element:                                           \
    assert (strcasecmp (G_STRINGIFY (element), element_name) == 0);      \
    set_client_state (parent);                                           \
    break

/**
 * @brief Insert else clause for omp_xml_handle_start_element.
 *
 * Stop the parser from reading over elements at the same time.
 *
 * @param[in]  parent   Parent element.
 * @param[in]  element  Element.
 */
#define CLOSE_READ_OVER(parent, element)                                 \
  case parent ## _ ## element:                                           \
    assert (strcasecmp (G_STRINGIFY (element), element_name) == 0);      \
    omp_parser->read_over = 0;                                           \
    set_client_state (parent);                                           \
    break

/**
 * @brief Iterate a GET iterator.
 *
 * If the user requested to start at an offset from the first result, but
 * the result set was empty, then reset the iterator to start from the
 * first result.
 *
 * @param[in]  resources  Resource iterator.
 * @param[in]  get        GET command data.
 * @param[out] first      First.  Number of first item to get.
 * @param[out] count      Count.
 * @param[in]  init       Init function, to reset the iterator.
 *
 * @return What to do next: 0 continue, 1 end, -1 fail.
 */
static int
get_next (iterator_t *resources, get_data_t *get, int *first, int *count,
          int (*init) (iterator_t*, const get_data_t *))
{
  if (next (resources) == FALSE)
   {
     gchar *new_filter;

     if (get->filt_id && strcmp (get->filt_id, "0"))
       /* If filtering by a named filter, then just end, because changing
        * the filter term would probably surprise the user. */
       return 1;

     if (*first == 0)
       return 1;

     if (*first == 1 || *count > 0)
       /* Some results were found or first was 1, so stop iterating. */
       return 1;

     /* Reset the iterator with first 1, and start again. */
     cleanup_iterator (resources);
     new_filter = g_strdup_printf ("first=1 %s", get->filter);
     g_free (get->filter);
     get->filter = new_filter;
     if (init (resources, get))
       return -1;
     *count = 0;
     *first = 1;
     if (next (resources) == FALSE)
       return 1;
   }
 return 0;
}


/**
 * @brief Insert DELETE case for omp_xml_handle_end_element.
 *
 * @param[in]  upper    Resource type in uppercase.
 * @param[in]  type     Resource type.
 * @param[in]  capital  Resource type capitalised.
 */
#define CASE_DELETE(upper, type, capital)                                   \
  case CLIENT_DELETE_ ## upper :                                            \
    assert (strcasecmp ("DELETE_" G_STRINGIFY (upper), element_name) == 0); \
    if (delete_ ## type ## _data-> type ## _id)                             \
      switch (delete_ ## type (delete_ ## type ## _data-> type ## _id,      \
                             delete_ ## type ## _data->ultimate))           \
        {                                                                   \
          case 0:                                                           \
            SEND_TO_CLIENT_OR_FAIL (XML_OK ("delete_" G_STRINGIFY (type))); \
            log_event (G_STRINGIFY(type), capital,                          \
                       delete_ ## type ## _data-> type ## _id, "deleted");  \
            break;                                                          \
          case 1:                                                           \
            SEND_TO_CLIENT_OR_FAIL                                          \
             (XML_ERROR_SYNTAX ("delete_" G_STRINGIFY (type),               \
                                capital " is in use"));                     \
            log_event_fail (G_STRINGIFY(type), capital,                     \
                            delete_ ## type ## _data-> type ## _id,         \
                            "deleted");                                     \
            break;                                                          \
          case 2:                                                           \
            if (send_find_error_to_client                                   \
                 ("delete_" G_STRINGIFY (type),                             \
                  G_STRINGIFY (type),                                       \
                  delete_ ## type ## _data-> type ## _id,                   \
                  omp_parser))                                              \
              {                                                             \
                error_send_to_client (error);                               \
                return;                                                     \
              }                                                             \
            log_event_fail (G_STRINGIFY(type), capital,                     \
                            delete_ ## type ## _data-> type ## _id,         \
                            "deleted");                                     \
            break;                                                          \
          case 3:                                                           \
            SEND_TO_CLIENT_OR_FAIL                                          \
             (XML_ERROR_SYNTAX ("delete_" G_STRINGIFY (type),               \
                                "Attempt to delete a predefined"            \
                                " " G_STRINGIFY (type)));                   \
            break;                                                          \
          case 99:                                                          \
            SEND_TO_CLIENT_OR_FAIL                                          \
             (XML_ERROR_SYNTAX ("delete_" G_STRINGIFY (type),               \
                                "Permission denied"));                      \
            log_event_fail (G_STRINGIFY(type), capital,                     \
                            delete_ ## type ## _data-> type ## _id,         \
                            "deleted");                                     \
            break;                                                          \
          default:                                                          \
            SEND_TO_CLIENT_OR_FAIL                                          \
             (XML_INTERNAL_ERROR ("delete_" G_STRINGIFY (type)));           \
            log_event_fail (G_STRINGIFY(type), capital,                     \
                            delete_ ## type ## _data-> type ## _id,         \
                            "deleted");                                     \
        }                                                                   \
    else                                                                    \
      SEND_TO_CLIENT_OR_FAIL                                                \
       (XML_ERROR_SYNTAX ("delete_" G_STRINGIFY (type),                     \
                          "DELETE_" G_STRINGIFY (upper) " requires a "      \
                          G_STRINGIFY (type) "_id attribute"));             \
    delete_ ## type ## _data_reset (delete_ ## type ## _data);              \
    set_client_state (CLIENT_AUTHENTIC);                                    \
    break

/**
 * @brief Call init_get for a GET end handler.
 *
 * @param[in]  type     Resource type.
 * @param[in]  capital  Resource type, capitalised.
 */
#define INIT_GET(type, capital)                                          \
  count = 0;                                                             \
  ret = init_get ("get_" G_STRINGIFY (type) "s",                         \
                  &get_ ## type ## s_data->get,                          \
                  G_STRINGIFY (capital) "s",                             \
                  &first);                                               \
  if (ret)                                                               \
    {                                                                    \
      switch (ret)                                                       \
        {                                                                \
          case 99:                                                       \
            SEND_TO_CLIENT_OR_FAIL                                       \
             (XML_ERROR_SYNTAX ("get_" G_STRINGIFY (type) "s",           \
                                "Permission denied"));                   \
            break;                                                       \
          default:                                                       \
            internal_error_send_to_client (error);                       \
            return;                                                      \
        }                                                                \
      get_ ## type ## s_data_reset (get_ ## type ## s_data);             \
      set_client_state (CLIENT_AUTHENTIC);                               \
      return;                                                            \
    }

/**
 * @brief Get list of ovaldi definitions files using the values in ovaldefs
 *        table in scap.db
 *
 * @return String of |-concatenated file names. Free with g_free().
 */
char *
get_ovaldi_files ()
{
  iterator_t iterator;
  char *result = NULL;

  init_iterator (&iterator,
                 sql_is_sqlite3 ()
                  ? "SELECT DISTINCT xml_file FROM ovaldefs;"
                  // FIX Until we have SCAP in Postgres.
                  : "SELECT * FROM meta WHERE 1 = 0;");
  while (next (&iterator))
    {
      char *tmp;
      const char *fname = iterator_string (&iterator, 0);

      if (!fname)
        continue;
      tmp = g_strconcat (fname, result ? "|" : "", result, NULL);
      g_free (result);
      result = tmp;
    }
  cleanup_iterator (&iterator);
  return result;
}

/**
 * @brief Handle end of GET_SCANNERS element.
 *
 * @param[in]  omp_parser   OMP parser.
 * @param[in]  error        Error parameter.
 */
static void
handle_get_scanners (omp_parser_t *omp_parser, GError **error)
{
  iterator_t scanners;
  int ret, count, filtered, first;

  INIT_GET (scanner, Scanner);
  ret = init_scanner_iterator (&scanners, &get_scanners_data->get);
  switch (ret)
    {
      case 0:
        break;
      case 1:
        if (send_find_error_to_client
             ("get_scanners", "scanners", get_scanners_data->get.id,
              omp_parser))
          {
            error_send_to_client (error);
            break;
          }
        break;
      case 2:
        if (send_find_error_to_client
             ("get_scanners", "filter", get_scanners_data->get.filt_id,
              omp_parser))
          {
            error_send_to_client (error);
            break;
          }
        break;
      case -1:
        SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_scanners"));
        break;
    }
  if (ret)
    {
      get_scanners_data_reset (get_scanners_data);
      set_client_state (CLIENT_AUTHENTIC);
      return;
    }

  SEND_GET_START ("scanner");
  while (1)
    {
      ret = get_next (&scanners, &get_scanners_data->get, &first, &count,
                      init_scanner_iterator);
      if (ret == 1)
        break;
      if (ret == -1)
        {
          internal_error_send_to_client (error);
          break;
        }

      SEND_GET_COMMON (scanner, &get_scanners_data->get, &scanners);
      SENDF_TO_CLIENT_OR_FAIL
       ("<host>%s</host><port>%d</port><type>%d</type>"
        "<ca_pub>%s</ca_pub><key_pub>%s</key_pub>",
        scanner_iterator_host (&scanners), scanner_iterator_port (&scanners),
        scanner_iterator_type (&scanners), scanner_iterator_ca_pub (&scanners),
        scanner_iterator_key_pub (&scanners));
      count++;
      if (get_scanners_data->get.details)
        {
          iterator_t tasks;

          SEND_TO_CLIENT_OR_FAIL ("<tasks>");
          init_scanner_task_iterator (&tasks,
                                      get_iterator_resource (&scanners));
          while (next (&tasks))
            {
              SENDF_TO_CLIENT_OR_FAIL
               ("<task id=\"%s\">"
                "<name>%s</name>",
                scanner_task_iterator_uuid (&tasks),
                scanner_task_iterator_name (&tasks));

              if (schedule_task_iterator_readable (&tasks))
                SEND_TO_CLIENT_OR_FAIL ("</task>");
              else
                SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                        "</task>");
            }
          cleanup_iterator (&tasks);
          SEND_TO_CLIENT_OR_FAIL ("</tasks>");
        }
      if (scanner_iterator_type (&scanners) == SCANNER_TYPE_OSP
          && get_scanners_data->get.details)
        {
          char *s_name = NULL, *s_ver = NULL;
          char *d_name = NULL, *d_ver = NULL;
          char *p_name = NULL, *p_ver = NULL, *desc = NULL;
          GSList *params = NULL, *nodes;

          if (!osp_get_version_from_iterator
                (&scanners, &s_name, &s_ver, &d_name, &d_ver, &p_name, &p_ver)
              && !osp_get_details_from_iterator (&scanners, &desc, &params))
            {
              SENDF_TO_CLIENT_OR_FAIL
               ("<info><scanner><name>%s</name><version>%s</version>"
                "</scanner><daemon><name>%s</name><version>%s</version>"
                "</daemon><protocol><name>%s</name><version>%s"
                "</version></protocol><description>%s</description>",
                s_name, s_ver, d_name, d_ver, p_name, p_ver, desc);

              SENDF_TO_CLIENT_OR_FAIL ("<params>");
              nodes = params;
              while (nodes)
                {
                  osp_param_t *param = nodes->data;

                  SENDF_TO_CLIENT_OR_FAIL
                   ("<param><id>%s</id><default>%s</default>"
                    "<description>%s</description>"
                    "<type>osp_%s</type></param>", osp_param_id (param),
                    osp_param_default (param), osp_param_desc (param),
                    osp_param_type_str (param));

                  osp_param_free (nodes->data);
                  nodes = nodes->next;
                }
              SENDF_TO_CLIENT_OR_FAIL ("</params></info>");
            }
          else
            SENDF_TO_CLIENT_OR_FAIL
             ("<info><scanner><name/><version/></scanner>"
              "<daemon><name/><version/></daemon>"
              "<protocol><name/><version/></protocol><description/><params/>"
              "</info>");
          g_free (s_name);
          g_free (s_ver);
          g_free (d_name);
          g_free (d_ver);
          g_free (p_name);
          g_free (p_ver);
          g_free (desc);
          g_slist_free (params);
        }
      else if (get_scanners_data->get.details)
        {
          SENDF_TO_CLIENT_OR_FAIL
           ("<info><scanner><name>OpenVAS</name><version/></scanner>"
            "<daemon><name/><version/></daemon>"
            "<protocol><name/><version/></protocol><description/><params/>"
            "</info>");
        }
      SEND_TO_CLIENT_OR_FAIL ("</scanner>");
    }
  cleanup_iterator (&scanners);
  filtered = get_scanners_data->get.id
              ? 1 : scanner_count (&get_scanners_data->get);
  SEND_GET_END ("scanner", &get_scanners_data->get, count, filtered);
  get_scanners_data_reset (get_scanners_data);
  set_client_state (CLIENT_AUTHENTIC);
}

/**
 * @brief Check that a string represents a valid Certificate.
 *
 * @param[in]  cert_str     Certificate string.
 *
 * @return 0 if valid, 1 otherwise.
 */
static int
check_scanner_cert (const char *cert_str)
{
  gnutls_x509_crt_t crt;
  gnutls_datum_t data;
  int ret = 0;

  assert (cert_str);
  if (gnutls_x509_crt_init (&crt))
    return 1;
  data.size = strlen (cert_str);
  data.data = (void *) g_strdup (cert_str);
  if (gnutls_x509_crt_import (crt, &data, GNUTLS_X509_FMT_PEM))
    {
      gnutls_x509_crt_deinit (crt);
      g_free (data.data);
      return 1;
    }

  if (time (NULL) > gnutls_x509_crt_get_expiration_time (crt))
    {
      g_warning ("Certificate expiration time passed");
      ret = 1;
    }
  if (time (NULL) < gnutls_x509_crt_get_activation_time (crt))
    {
      g_warning ("Certificate activation time in the future");
      ret = 1;
    }
  g_free (data.data);
  gnutls_x509_crt_deinit (crt);
  return ret;
}

/**
 * @brief Check that a string represents a valid Private Key.
 *
 * @param[in]  cert_str     Private Key string.
 *
 * @return 0 if valid, 1 otherwise.
 */
static int
check_scanner_private (const char *key_str)
{
  gnutls_x509_privkey_t key;
  gnutls_datum_t data;

  assert (key_str);
  if (gnutls_x509_privkey_init (&key))
    return 1;
  data.size = strlen (key_str);
  data.data = (void *) g_strdup (key_str);
  if (gnutls_x509_privkey_import (key, &data, GNUTLS_X509_FMT_PEM))
    {
      gnutls_x509_privkey_deinit (key);
      g_free (data.data);
      return 1;
    }
  g_free (data.data);
  gnutls_x509_privkey_deinit (key);
  return 0;
}

/**
 * @brief Handle end of CREATE_SCANNER element.
 *
 * @param[in]  omp_parser   OMP parser.
 * @param[in]  error        Error parameter.
 */
static void
handle_create_scanner (omp_parser_t *omp_parser, GError **error)
{
  scanner_t new_scanner;

  if (create_scanner_data->copy)
    switch (copy_scanner (create_scanner_data->name,
                          create_scanner_data->comment,
                          create_scanner_data->copy, &new_scanner))
      {
        case 0:
          {
            char *uuid;
            uuid = scanner_uuid (new_scanner);
            SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_scanner"),
                                     uuid);
            log_event ("scanner", "scanner", uuid, "created");
            g_free (uuid);
            goto create_scanner_leave;
          }
        case 1:
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("create_scanner", "Scanner name exists already"));
          log_event_fail ("scanner", "Scanner", NULL, "created");
          goto create_scanner_leave;
        case 2:
          if (send_find_error_to_client ("create_scanner", "scanner",
                                         create_scanner_data->copy, omp_parser))
            {
              error_send_to_client (error);
              goto create_scanner_leave;
            }
          log_event_fail ("scanner", "Scanner", NULL, "created");
          goto create_scanner_leave;
        case 99:
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("create_scanner", "Permission denied"));
          log_event_fail ("scanner", "Scanner", NULL, "created");
          goto create_scanner_leave;
        case -1:
        default:
          SEND_TO_CLIENT_OR_FAIL
           (XML_INTERNAL_ERROR ("create_scanner"));
          log_event_fail ("scanner", "Scanner", NULL, "created");
          goto create_scanner_leave;
      }

  if (!create_scanner_data->name || !create_scanner_data->host
      || !create_scanner_data->port || !create_scanner_data->type
      || !create_scanner_data->ca_pub || !create_scanner_data->key_pub
      || !create_scanner_data->key_priv)
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("create_scanner", "Missing entity"));
      goto create_scanner_leave;
    }

  if (check_scanner_cert (create_scanner_data->ca_pub))
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("create_scanner", "Erroneous CA Certificate."));
      goto create_scanner_leave;
    }
  if (check_scanner_cert (create_scanner_data->key_pub))
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("create_scanner", "Erroneous Certificate."));
      goto create_scanner_leave;
    }
  if (check_scanner_private (create_scanner_data->key_priv))
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("create_scanner", "Erroneous Private Key."));
      goto create_scanner_leave;
    }
  switch (create_scanner
           (create_scanner_data->name, create_scanner_data->comment,
            create_scanner_data->host, create_scanner_data->port,
            create_scanner_data->type, &new_scanner,
            create_scanner_data->ca_pub, create_scanner_data->key_pub,
            create_scanner_data->key_priv))
    {
      case 0:
        {
          char *uuid = scanner_uuid (new_scanner);
          SENDF_TO_CLIENT_OR_FAIL
           (XML_OK_CREATED_ID ("create_scanner"), uuid);
          log_event ("scanner", "Scanner", uuid, "created");
          g_free (uuid);
          break;
        }
      case 1:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("create_scanner", "Scanner exists already"));
        log_event_fail ("scanner", "Scanner", NULL, "created");
        break;
      case 2:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("create_scanner", "Invalid entity value"));
        log_event_fail ("scanner", "Scanner", NULL, "created");
        break;
      case 99:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("create_scanner", "Permission denied"));
        log_event_fail ("scanner", "Scanner", NULL, "created");
        break;
      case -1:
        SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_scanner"));
        log_event_fail ("scanner", "Scanner", NULL, "created");
        break;
      default:
        assert (0);
        SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_scanner"));
        log_event_fail ("scanner", "Scanner", NULL, "created");
        break;
    }

create_scanner_leave:
  create_scanner_data_reset (create_scanner_data);
  set_client_state (CLIENT_AUTHENTIC);
}

/**
 * @brief Handle end of MODIFY_SCANNER element.
 *
 * @param[in]  omp_parser   OMP parser.
 * @param[in]  error        Error parameter.
 */
static void
handle_modify_scanner (omp_parser_t *omp_parser, GError **error)
{
  if (modify_scanner_data->ca_pub && *modify_scanner_data->ca_pub
      && check_scanner_cert (modify_scanner_data->ca_pub))
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("modify_scanner", "Erroneous CA Certificate."));
      goto modify_scanner_leave;
    }
  if (modify_scanner_data->key_pub && *modify_scanner_data->key_pub
      && check_scanner_cert (modify_scanner_data->key_pub))
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("modify_scanner", "Erroneous Certificate."));
      goto modify_scanner_leave;
    }
  if (modify_scanner_data->key_priv && *modify_scanner_data->key_priv
      && check_scanner_private (modify_scanner_data->key_priv))
    {
      SEND_TO_CLIENT_OR_FAIL
       (XML_ERROR_SYNTAX ("modify_scanner", "Erroneous Private Key."));
      goto modify_scanner_leave;
    }
  switch (modify_scanner
           (modify_scanner_data->scanner_id, modify_scanner_data->name,
            modify_scanner_data->comment, modify_scanner_data->host,
            modify_scanner_data->port, modify_scanner_data->type,
            modify_scanner_data->ca_pub, modify_scanner_data->key_pub,
            modify_scanner_data->key_priv))
    {
      case 0:
        SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_scanner"));
        log_event ("scanner", "Scanner", modify_scanner_data->scanner_id,
                   "modified");
        break;
      case 1:
        if (send_find_error_to_client ("modify_scanner", "scanner",
                                       modify_scanner_data->scanner_id,
                                       omp_parser))
          {
            error_send_to_client (error);
            return;
          }
        log_event_fail ("scanner", "Scanner", modify_scanner_data->scanner_id,
                        "modified");
        break;
      case 2:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("modify_scanner",
                            "scanner with new name exists already"));
        log_event_fail ("scanner", "Scanner", modify_scanner_data->scanner_id,
                        "modified");
        break;
      case 3:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("modify_scanner", "Missing scanner_id"));
        log_event_fail ("scanner", "Scanner", modify_scanner_data->scanner_id,
                        "modified");
        break;
      case 4:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("modify_scanner", "Invalid value"));
        log_event_fail ("scanner", "Scanner", modify_scanner_data->scanner_id,
                        "modified");
        break;
      case 99:
        SEND_TO_CLIENT_OR_FAIL
         (XML_ERROR_SYNTAX ("modify_scanner", "Permission denied"));
        log_event_fail ("scanner", "Scanner", modify_scanner_data->scanner_id,
                        "modified");
        break;
      default:
      case -1:
        SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_scanner"));
        log_event_fail ("scanner", "Scanner", modify_scanner_data->scanner_id,
                        "modified");
        break;
    }
modify_scanner_leave:
  modify_scanner_data_reset (modify_scanner_data);
  set_client_state (CLIENT_AUTHENTIC);
}

/**
 * @brief Handle the end of an OMP XML element.
 *
 * React to the end of an XML element according to the current value
 * of \ref client_state, usually adjusting \ref client_state to indicate
 * the change (with \ref set_client_state).  Call \ref send_to_client to queue
 * any responses for the client.  Call the task utilities to adjust the
 * tasks (for example \ref start_task, \ref stop_task, \ref set_task_parameter,
 * \ref delete_task and \ref find_task ).
 *
 * Set error parameter on encountering an error.
 *
 * @param[in]  context           Parser context.
 * @param[in]  element_name      XML element name.
 * @param[in]  user_data         OMP parser.
 * @param[in]  error             Error parameter.
 */
static void
omp_xml_handle_end_element (/*@unused@*/ GMarkupParseContext* context,
                            const gchar *element_name,
                            gpointer user_data,
                            GError **error)
{
  omp_parser_t *omp_parser = (omp_parser_t*) user_data;
  int (*write_to_client) (const char *, void*)
    = (int (*) (const char *, void*)) omp_parser->client_writer;
  void* write_to_client_data = (void*) omp_parser->client_writer_data;

  tracef ("   XML    end: %s\n", element_name);

  if (omp_parser->read_over > 1)
    {
      omp_parser->read_over--;
    }
  else if ((omp_parser->read_over == 1) && omp_parser->parent_state)
    {
      client_state = omp_parser->parent_state;
      omp_parser->parent_state = 0;
      omp_parser->read_over = 0;
    }
  else switch (client_state)
    {
      case CLIENT_TOP:
        assert (0);
        break;

      case CLIENT_AUTHENTICATE:
        switch (authenticate (&current_credentials))
          {
            case 0:   /* Authentication succeeded. */
              if (load_tasks ())
                {
                  g_warning ("%s: failed to load tasks\n", __FUNCTION__);
                  g_set_error (error, G_MARKUP_ERROR, G_MARKUP_ERROR_PARSE,
                               "Manager failed to load tasks.");
                  free_credentials (&current_credentials);
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("authenticate"));
                  set_client_state (CLIENT_TOP);
                }
              else
                {
                  const char *timezone, *severity;
                  char *pw_warning;

                  timezone = (current_credentials.timezone
                              && strlen (current_credentials.timezone))
                               ? current_credentials.timezone
                               : "UTC";

                  if (setenv ("TZ", timezone, 1) == -1)
                    {
                      free_credentials (&current_credentials);
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_INTERNAL_ERROR ("authenticate"));
                      set_client_state (CLIENT_TOP);
                      break;
                    }
                  tzset ();

                  severity = setting_severity ();
                  pw_warning = openvas_validate_password (
                                 current_credentials.password,
                                 current_credentials.username);

                  if (pw_warning)
                    SENDF_TO_CLIENT_OR_FAIL
                    ("<authenticate_response"
                      " status=\"" STATUS_OK "\""
                      " status_text=\"" STATUS_OK_TEXT "\">"
                      "<role>%s</role>"
                      "<timezone>%s</timezone>"
                      "<severity>%s</severity>"
                      "<password_warning>%s</password_warning>"
                      "</authenticate_response>",
                      current_credentials.role
                        ? current_credentials.role
                        : "",
                      timezone,
                      severity,
                      pw_warning ? pw_warning : "");
                  else
                    SENDF_TO_CLIENT_OR_FAIL
                    ("<authenticate_response"
                      " status=\"" STATUS_OK "\""
                      " status_text=\"" STATUS_OK_TEXT "\">"
                      "<role>%s</role>"
                      "<timezone>%s</timezone>"
                      "<severity>%s</severity>"
                      "</authenticate_response>",
                      current_credentials.role
                        ? current_credentials.role
                        : "",
                      timezone,
                      severity);

                  set_client_state (CLIENT_AUTHENTIC);
                }
              break;
            case 1:   /* Authentication failed. */
              free_credentials (&current_credentials);
              SEND_TO_CLIENT_OR_FAIL (XML_ERROR_AUTH_FAILED ("authenticate"));
              set_client_state (CLIENT_TOP);
              break;
            case 99:   /* Authentication failed. */
              free_credentials (&current_credentials);
              SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX ("authenticate",
                                                        "Permission denied"));
              set_client_state (CLIENT_TOP);
              break;
            case -1:  /* Error while authenticating. */
            default:
              free_credentials (&current_credentials);
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("authenticate"));
              set_client_state (CLIENT_TOP);
              break;
          }
        break;

      case CLIENT_AUTHENTICATE_CREDENTIALS:
        assert (strcasecmp ("CREDENTIALS", element_name) == 0);
        set_client_state (CLIENT_AUTHENTICATE);
        break;

      case CLIENT_AUTHENTICATE_CREDENTIALS_USERNAME:
        assert (strcasecmp ("USERNAME", element_name) == 0);
        set_client_state (CLIENT_AUTHENTICATE_CREDENTIALS);
        break;

      case CLIENT_AUTHENTICATE_CREDENTIALS_PASSWORD:
        assert (strcasecmp ("PASSWORD", element_name) == 0);
        set_client_state (CLIENT_AUTHENTICATE_CREDENTIALS);
        break;

      case CLIENT_AUTHENTIC:
      case CLIENT_COMMANDS:
      case CLIENT_AUTHENTIC_COMMANDS:
        assert (strcasecmp ("COMMANDS", element_name) == 0);
        SENDF_TO_CLIENT_OR_FAIL ("</commands_response>");
        break;

      CASE_DELETE (AGENT, agent, "Agent");
      CASE_DELETE (ALERT, alert, "Alert");
      CASE_DELETE (CONFIG, config, "Config");
      CASE_DELETE (FILTER, filter, "Filter");
      CASE_DELETE (GROUP, group, "Group");
      CASE_DELETE (LSC_CREDENTIAL, lsc_credential, "LSC Credential");
      CASE_DELETE (NOTE, note, "Note");
      CASE_DELETE (OVERRIDE, override, "Override");
      CASE_DELETE (PERMISSION, permission, "Permission");
      CASE_DELETE (PORT_LIST, port_list, "Port list");
      CASE_DELETE (PORT_RANGE, port_range, "Port range");
      CASE_DELETE (REPORT, report, "Report");
      CASE_DELETE (REPORT_FORMAT, report_format, "Report format");
      CASE_DELETE (ROLE, role, "Role");
      CASE_DELETE (SCANNER, scanner, "Scanner");
      CASE_DELETE (SCHEDULE, schedule, "Schedule");
      CASE_DELETE (SLAVE, slave, "Slave");
      CASE_DELETE (TAG, tag, "Tag");
      CASE_DELETE (TARGET, target, "Target");

      case CLIENT_DELETE_TASK:
        if (delete_task_data->task_id)
          {
            switch (request_delete_task_uuid (delete_task_data->task_id,
                                              delete_task_data->ultimate))
              {
                case 0:    /* Deleted. */
                  SEND_TO_CLIENT_OR_FAIL (XML_OK ("delete_task"));
                  log_event ("task", "Task", delete_task_data->task_id,
                             "deleted");
                  break;
                case 1:    /* Delete requested. */
                  SEND_TO_CLIENT_OR_FAIL (XML_OK_REQUESTED ("delete_task"));
                  log_event ("task", "Task", delete_task_data->task_id,
                             "requested for delete");
                  break;
                case 2:    /* Hidden task. */
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("delete_task",
                                      "Attempt to delete a hidden task"));
                  log_event_fail ("task", "Task", delete_task_data->task_id,
                                  "deleted");
                  break;
                case 3:  /* Failed to find task. */
                  if (send_find_error_to_client
                       ("delete_task", "task", delete_task_data->task_id,
                        omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("delete_task",
                                      "Permission denied"));
                  log_event_fail ("task", "Task", delete_task_data->task_id,
                                  "deleted");
                  break;
                default:   /* Programming error. */
                  assert (0);
                case -1:
                  /* to_scanner is full. */
                  /** @todo Or some other error occurred. */
                  /** @todo Consider reverting parsing for retry. */
                  /** @todo process_omp_client_input must return -2. */
                  tracef ("delete_task failed\n");
                  abort ();
                  break;
                case -5:
                  SEND_XML_SERVICE_DOWN ("delete_task");
                  log_event_fail ("task", "Task",
                                  delete_task_data->task_id,
                                  "deleted");
                  break;
              }
          }
        else
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("delete_task",
                              "DELETE_TASK requires a task_id attribute"));
        delete_task_data_reset (delete_task_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_DELETE_USER:
        assert (strcasecmp ("DELETE_USER", element_name) == 0);
        if (delete_user_data->user_id || delete_user_data->name)
          switch (delete_user (delete_user_data->user_id,
                               delete_user_data->name,
                               delete_user_data->ultimate,
                               1))
            {
              case 0:
                SEND_TO_CLIENT_OR_FAIL (XML_OK ("delete_user"));
                log_event ("user", "User", delete_user_data->user_id,
                           "deleted");
                break;
              case 2:
                if (send_find_error_to_client ("delete_user",
                                               "user",
                                               delete_user_data->user_id
                                                ? delete_user_data->user_id
                                                : delete_user_data->name,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("user", "User", delete_user_data->user_id,
                                "deleted");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("delete_user",
                                    "Attempt to delete a predefined"
                                    " user"));
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("delete_user",
                                    "User has an active task"));
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("delete_user",
                                    "Attempt to delete current user"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("delete_user",
                                    "Permission denied"));
                log_event_fail ("user", "User", delete_user_data->user_id,
                                "deleted");
                break;
              default:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("delete_user"));
                log_event_fail ("user", "User", delete_user_data->user_id,
                                "deleted");
            }
        else
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("delete_user",
                              "DELETE_USER requires a user_id attribute"));
        delete_user_data_reset (delete_user_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_DESCRIBE_AUTH:
        {
          gchar *config_file, *content, *resp;

          assert (current_credentials.username);

          if (user_may ("describe_auth") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("describe_auth",
                                  "Permission denied"));
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          /* Get base 64 encoded content of the auth configuration file. */
          config_file = g_build_filename (OPENVAS_STATE_DIR, "openvasmd",
                                          "auth.conf", NULL);
          content = keyfile_to_auth_conf_settings_xml (config_file);
          g_free (config_file);
          if (content == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL (XML_ERROR_MISSING ("describe_auth"));
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          resp = g_strdup_printf ("<describe_auth_response"
                                  " status=\"" STATUS_OK "\""
                                  " status_text=\"" STATUS_OK_TEXT "\">"
                                  "%s"
                                  "</describe_auth_response>",
                                  content);
          g_free (content);

          SEND_TO_CLIENT_OR_FAIL (resp);
          g_free (resp);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_DESCRIBE_FEED:
        {
          gchar *feed_description = NULL;
          gchar *feed_identification = NULL;
          gchar *feed_version = NULL;

          assert (current_credentials.username);

          if (user_may ("describe_feed") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("describe_feed",
                                  "Permission denied"));
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (openvas_get_sync_script_description (sync_script, &feed_description)
              && openvas_get_sync_script_identification (sync_script,
                                                         &feed_identification,
                                                         NVT_FEED)
              && openvas_get_sync_script_feed_version (sync_script,
                                                       &feed_version))
            {
              gchar *user, *timestamp;
              int syncing;
              gchar **ident = g_strsplit (feed_identification, "|", 6);
              gchar *selftest_result = NULL;

              syncing = openvas_current_sync (sync_script, &timestamp, &user);
              if (syncing < 0 || ident[0] == NULL || ident[1] == NULL
                  || ident[2] == NULL || ident[3] == NULL)
                {
                  g_strfreev (ident);
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("describe_feed"));
                }
              else
                {
                  SENDF_TO_CLIENT_OR_FAIL
                   ("<describe_feed_response"
                    " status=\"" STATUS_OK "\""
                    " status_text=\"" STATUS_OK_TEXT "\">"
                    "<feed>"
                    "<name>%s</name>"
                    "<version>%s</version>"
                    "<description>%s</description>",
                    ident[3], feed_version,
                    feed_description);
                  g_strfreev (ident);
                  if (openvas_sync_script_perform_selftest
                      (sync_script, &selftest_result) == FALSE)
                    {
                      SENDF_TO_CLIENT_OR_FAIL ("<sync_not_available>"
                                               "<error>%s</error>"
                                               "</sync_not_available>",
                                               selftest_result ? selftest_result :
                                               "");
                      g_free (selftest_result);
                    }

                  if (syncing > 0)
                    {
                      SENDF_TO_CLIENT_OR_FAIL ("<currently_syncing>"
                                               "<timestamp>%s</timestamp>"
                                               "<user>%s</user>"
                                               "</currently_syncing>",
                                               timestamp ? timestamp : "",
                                               user ? user : "");
                      g_free (timestamp);
                      g_free (user);
                    }
                  SEND_TO_CLIENT_OR_FAIL ("</feed>"
                                          "</describe_feed_response>");
                }

              g_free (feed_identification);
              g_free (feed_version);
            }
          else
            {
              SEND_TO_CLIENT_OR_FAIL ("<describe_feed_response"
                                      " status=\"" STATUS_OK "\""
                                      " status_text=\"" STATUS_OK_TEXT "\">");
              SEND_TO_CLIENT_OR_FAIL ("<feed>");
              SEND_TO_CLIENT_OR_FAIL ("<name></name>");
              SEND_TO_CLIENT_OR_FAIL ("<description></description>");
              SEND_TO_CLIENT_OR_FAIL ("</feed>");
              SEND_TO_CLIENT_OR_FAIL ("</describe_feed_response>");
            }
          g_free (feed_description);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_DESCRIBE_SCAP:
        {
          gchar *scap_description = NULL;
          gchar *scap_identification = NULL;
          gchar *scap_version = NULL;

          assert (current_credentials.username);

          if (user_may ("describe_scap") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("describe_scap",
                                  "Permission denied"));
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (openvas_get_sync_script_description (scap_script, &scap_description)
              && openvas_get_sync_script_identification (scap_script,
                                                         &scap_identification,
                                                         SCAP_FEED)
              && openvas_get_sync_script_feed_version (scap_script,
                                                       &scap_version))
            {
              gchar *user, *timestamp;
              int syncing;
              gchar **ident = g_strsplit (scap_identification, "|", 6);
              gchar *selftest_result = NULL;

              syncing = openvas_current_sync (scap_script, &timestamp, &user);
              if (syncing < 0 || ident[0] == NULL || ident[1] == NULL
                  || ident[2] == NULL || ident[3] == NULL)
                {
                  g_strfreev (ident);
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("describe_scap"));
                }
              else
                {
                  SENDF_TO_CLIENT_OR_FAIL
                   ("<describe_scap_response"
                    " status=\"" STATUS_OK "\""
                    " status_text=\"" STATUS_OK_TEXT "\">"
                    "<scap>"
                    "<name>%s</name>"
                    "<version>%s</version>"
                    "<description>%s</description>",
                    ident[3], scap_version,
                    scap_description);
                  g_strfreev (ident);
                  if (openvas_sync_script_perform_selftest
                      (scap_script, &selftest_result) == FALSE)
                    {
                      SENDF_TO_CLIENT_OR_FAIL ("<sync_not_available>"
                                               "<error>%s</error>"
                                               "</sync_not_available>",
                                               selftest_result ? selftest_result :
                                               "");
                      g_free (selftest_result);
                    }

                  if (syncing > 0)
                    {
                      SENDF_TO_CLIENT_OR_FAIL ("<currently_syncing>"
                                               "<timestamp>%s</timestamp>"
                                               "<user>%s</user>"
                                               "</currently_syncing>",
                                               timestamp ? timestamp : "",
                                               user ? user : "");
                      g_free (timestamp);
                      g_free (user);
                    }
                  SEND_TO_CLIENT_OR_FAIL ("</scap>"
                                          "</describe_scap_response>");
                }

              g_free (scap_identification);
              g_free (scap_version);
            }
          else
            {
              SEND_TO_CLIENT_OR_FAIL ("<describe_scap_response"
                                      " status=\"" STATUS_OK "\""
                                      " status_text=\"" STATUS_OK_TEXT "\">");
              SEND_TO_CLIENT_OR_FAIL ("<scap>");
              SEND_TO_CLIENT_OR_FAIL ("<name></name>");
              SEND_TO_CLIENT_OR_FAIL ("<description></description>");
              SEND_TO_CLIENT_OR_FAIL ("</scap>");
              SEND_TO_CLIENT_OR_FAIL ("</describe_scap_response>");
            }
          g_free (scap_description);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_DESCRIBE_CERT:
        {
          gchar *cert_description = NULL;
          gchar *cert_identification = NULL;
          gchar *cert_version = NULL;

          assert (current_credentials.username);

          if (user_may ("describe_cert") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("describe_cert",
                                  "Permission denied"));
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (openvas_get_sync_script_description (cert_script,
                                                   &cert_description)
              && openvas_get_sync_script_identification (cert_script,
                                                         &cert_identification,
                                                         CERT_FEED)
              && openvas_get_sync_script_feed_version (cert_script,
                                                       &cert_version))
            {
              gchar *user, *timestamp;
              int syncing;
              gchar **ident = g_strsplit (cert_identification, "|", 6);
              gchar *selftest_result = NULL;

              syncing = openvas_current_sync (cert_script, &timestamp, &user);
              if (syncing < 0 || ident[0] == NULL || ident[1] == NULL
                  || ident[2] == NULL || ident[3] == NULL)
                {
                  g_strfreev (ident);
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("describe_cert"));
                }
              else
                {
                  SENDF_TO_CLIENT_OR_FAIL
                   ("<describe_cert_response"
                    " status=\"" STATUS_OK "\""
                    " status_text=\"" STATUS_OK_TEXT "\">"
                    "<cert>"
                    "<name>%s</name>"
                    "<version>%s</version>"
                    "<description>%s</description>",
                    ident[3], cert_version,
                    cert_description);
                  g_strfreev (ident);
                  if (openvas_sync_script_perform_selftest
                      (cert_script, &selftest_result) == FALSE)
                    {
                      SENDF_TO_CLIENT_OR_FAIL ("<sync_not_available>"
                                               "<error>%s</error>"
                                               "</sync_not_available>",
                                               selftest_result
                                                ? selftest_result
                                                : "");
                      g_free (selftest_result);
                    }

                  if (syncing > 0)
                    {
                      SENDF_TO_CLIENT_OR_FAIL ("<currently_syncing>"
                                               "<timestamp>%s</timestamp>"
                                               "<user>%s</user>"
                                               "</currently_syncing>",
                                               timestamp ? timestamp : "",
                                               user ? user : "");
                      g_free (timestamp);
                      g_free (user);
                    }
                  SEND_TO_CLIENT_OR_FAIL ("</cert>"
                                          "</describe_cert_response>");
                }

              g_free (cert_identification);
              g_free (cert_version);
            }
          else
            {
              SEND_TO_CLIENT_OR_FAIL ("<describe_cert_response"
                                      " status=\"" STATUS_OK "\""
                                      " status_text=\"" STATUS_OK_TEXT "\">");
              SEND_TO_CLIENT_OR_FAIL ("<cert>");
              SEND_TO_CLIENT_OR_FAIL ("<name></name>");
              SEND_TO_CLIENT_OR_FAIL ("<description></description>");
              SEND_TO_CLIENT_OR_FAIL ("</cert>");
              SEND_TO_CLIENT_OR_FAIL ("</describe_cert_response>");
            }
          g_free (cert_description);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_AGENTS:
        {
          int format;

          assert (strcasecmp ("GET_AGENTS", element_name) == 0);

          if (get_agents_data->format)
            {
              if (strlen (get_agents_data->format))
                {
                  if (strcasecmp (get_agents_data->format, "installer") == 0)
                    format = 1;
                  else if (strcasecmp (get_agents_data->format,
                                       "howto_install")
                           == 0)
                    format = 2;
                  else if (strcasecmp (get_agents_data->format, "howto_use")
                           == 0)
                    format = 3;
                  else
                    format = -1;
                }
              else
                format = 0;
            }
          else if (get_agents_data->get.details == 1) /* For exporting */
            format = 1;
          else
            format = 0;

          if (format == -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_agents",
                                "GET_AGENTS format attribute should"
                                " be 'installer', 'howto_install' or 'howto_use'."));
          else
            {
              iterator_t agents;
              int ret, count, filtered, first;

              INIT_GET (agent, Agent);

              ret = init_agent_iterator (&agents,
                                         &get_agents_data->get);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client ("get_agents",
                                                       "agents",
                                                       get_agents_data->get.id,
                                                       omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_agents", "filter",
                              get_agents_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_agents"));
                        break;
                    }
                  get_agents_data_reset (get_agents_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("agent");
              while (1)
                {
                  ret = get_next (&agents, &get_agents_data->get, &first,
                                  &count, init_agent_iterator);
                  if (ret == 1)
                    break;
                  if (ret == -1)
                    {
                      internal_error_send_to_client (error);
                      return;
                    }

                  SEND_GET_COMMON (agent, &get_agents_data->get,
                                   &agents);
                  switch (format)
                    {
                      case 1: /* installer */
                        {
                          time_t trust_time;
                          trust_time = agent_iterator_trust_time (&agents);

                          SENDF_TO_CLIENT_OR_FAIL
                           ("<package format=\"installer\">"
                            "<filename>%s</filename>"
                            "%s"
                            "</package>"
                            "<installer>"
                            "<trust>%s<time>%s</time></trust>"
                            "</installer>"
                            "</agent>",
                            agent_iterator_installer_filename (&agents),
                            agent_iterator_installer_64 (&agents),
                            agent_iterator_trust (&agents),
                            iso_time (&trust_time));
                        }
                        break;
                      case 2: /* howto_install */
                        SENDF_TO_CLIENT_OR_FAIL
                         ("<package format=\"howto_install\">%s</package>"
                          "</agent>",
                          agent_iterator_howto_install (&agents));
                        break;
                      case 3: /* howto_use */
                        SENDF_TO_CLIENT_OR_FAIL
                         ("<package format=\"howto_use\">%s</package>"
                          "</agent>",
                          agent_iterator_howto_use (&agents));
                        break;
                      default:
                        {
                          time_t trust_time;

                          trust_time = agent_iterator_trust_time (&agents);

                          SENDF_TO_CLIENT_OR_FAIL
                           ("<installer>"
                            "<trust>%s<time>%s</time></trust>"
                            "</installer>"
                            "</agent>",
                            agent_iterator_trust (&agents),
                            iso_time (&trust_time));
                        }
                        break;
                    }
                  count++;
                }
              cleanup_iterator (&agents);
              filtered = get_agents_data->get.id
                          ? 1
                          : agent_count (&get_agents_data->get);
              SEND_GET_END ("agent", &get_agents_data->get, count, filtered);
            }
          get_agents_data_reset (get_agents_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_ALERTS:
        {
          iterator_t alerts;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_ALERTS", element_name) == 0);

          INIT_GET (alert, Alert);

          ret = init_alert_iterator (&alerts, &get_alerts_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_alerts", "alert",
                                                   get_alerts_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_alerts", "alert", get_alerts_data->get.filt_id,
                          omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_alerts"));
                    break;
                }
              get_alerts_data_reset (get_alerts_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("alert");
          while (1)
            {
              iterator_t data;
              char *filter_uuid;
              int notice, message;
              const char *method;

              ret = get_next (&alerts, &get_alerts_data->get, &first,
                              &count, init_alert_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }
              SEND_GET_COMMON (alert, &get_alerts_data->get,
                               &alerts);

              /* Filter. */

              filter_uuid = alert_iterator_filter_uuid (&alerts);
              if (filter_uuid)
                {
                  SENDF_TO_CLIENT_OR_FAIL ("<filter id=\"%s\">"
                                           "<name>%s</name>"
                                           "<trash>%i</trash>",
                                           filter_uuid,
                                           alert_iterator_filter_name (&alerts),
                                           alert_iterator_filter_trash
                                            (&alerts));
                  if (alert_iterator_filter_readable (&alerts))
                    SEND_TO_CLIENT_OR_FAIL ("</filter>");
                  else
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                            "</filter>");
                }

              /* Condition. */

              SENDF_TO_CLIENT_OR_FAIL ("<condition>%s",
                                       alert_condition_name
                                        (alert_iterator_condition
                                          (&alerts)));
              init_alert_data_iterator (&data, get_iterator_resource (&alerts),
                                        get_alerts_data->get.trash,
                                        "condition");
              while (next (&data))
                SENDF_TO_CLIENT_OR_FAIL ("<data>"
                                         "<name>%s</name>"
                                         "%s"
                                         "</data>",
                                         alert_data_iterator_name (&data),
                                         alert_data_iterator_data (&data));
              cleanup_iterator (&data);

              SEND_TO_CLIENT_OR_FAIL ("</condition>");

              /* Event. */

              SENDF_TO_CLIENT_OR_FAIL ("<event>%s",
                                       event_name (alert_iterator_event
                                        (&alerts)));
              init_alert_data_iterator (&data, get_iterator_resource (&alerts),
                                        get_alerts_data->get.trash, "event");
              while (next (&data))
                SENDF_TO_CLIENT_OR_FAIL ("<data>"
                                         "<name>%s</name>"
                                         "%s"
                                         "</data>",
                                         alert_data_iterator_name (&data),
                                         alert_data_iterator_data (&data));
              cleanup_iterator (&data);
              SEND_TO_CLIENT_OR_FAIL ("</event>");

              /* Method. */

              method = alert_method_name (alert_iterator_method (&alerts));
              SENDF_TO_CLIENT_OR_FAIL ("<method>%s", method);
              init_alert_data_iterator (&data, get_iterator_resource (&alerts),
                                        get_alerts_data->get.trash, "method");
              notice = -1;
              message = 0;
              while (next (&data))
                {
                  const char *name;
                  name = alert_data_iterator_name (&data);
                  if (strcmp (name, "notice") == 0)
                    notice = atoi (alert_data_iterator_data (&data));
                  else if (strcmp (method, "Email") == 0
                           && strcmp (name, "message") == 0)
                    {
                      if (strlen (alert_data_iterator_data (&data)) == 0)
                        continue;
                      message = 1;
                    }
                  SENDF_TO_CLIENT_OR_FAIL ("<data>"
                                           "<name>%s</name>"
                                           "%s"
                                           "</data>",
                                           name,
                                           alert_data_iterator_data (&data));
                }
              /* If there is no email message data, send the default. */
              if (strcmp (method, "Email") == 0
                  && message == 0
                  && (notice == 0 || notice == 2))
                SENDF_TO_CLIENT_OR_FAIL ("<data>"
                                         "<name>message</name>"
                                         "%s"
                                         "</data>",
                                         notice == 0
                                          ? ALERT_MESSAGE_INCLUDE
                                          : ALERT_MESSAGE_ATTACH);
              cleanup_iterator (&data);
              SEND_TO_CLIENT_OR_FAIL ("</method>");

              if (get_alerts_data->tasks)
                {
                  iterator_t tasks;

                  SEND_TO_CLIENT_OR_FAIL ("<tasks>");
                  init_alert_task_iterator (&tasks,
                                            get_iterator_resource (&alerts), 0);
                  while (next (&tasks))
                    {
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<task id=\"%s\">"
                        "<name>%s</name>",
                        alert_task_iterator_uuid (&tasks),
                        alert_task_iterator_name (&tasks));

                      if (alert_task_iterator_readable (&tasks))
                        SEND_TO_CLIENT_OR_FAIL ("</task>");
                      else
                        SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                "</task>");
                    }
                  cleanup_iterator (&tasks);
                  SEND_TO_CLIENT_OR_FAIL ("</tasks>");
                }

              SEND_TO_CLIENT_OR_FAIL ("</alert>");
              count++;
            }
          cleanup_iterator (&alerts);
          filtered = get_alerts_data->get.id
                      ? 1
                      : alert_count (&get_alerts_data->get);
          SEND_GET_END ("alert", &get_alerts_data->get, count, filtered);

          get_alerts_data_reset (get_alerts_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_CONFIGS:
        {
          iterator_t configs;
          int ret, filtered, first, count;

          assert (strcasecmp ("GET_CONFIGS", element_name) == 0);

          INIT_GET (config, Config);

          ret = init_config_iterator (&configs, &get_configs_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_configs", "config",
                                                   get_configs_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_configs", "config",
                          get_configs_data->get.filt_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_configs"));
                    break;
                }
              get_configs_data_reset (get_configs_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("config");
          while (1)
            {
              int config_nvts_growing, config_families_growing, config_type;
              const char *selector;
              config_t config;

              ret = get_next (&configs, &get_configs_data->get, &first,
                              &count, init_config_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }
              SEND_GET_COMMON (config, &get_configs_data->get,
                               &configs);

              /** @todo This should really be an nvt_selector_t. */
              selector = config_iterator_nvt_selector (&configs);
              config = get_iterator_resource (&configs);
              config_nvts_growing = config_iterator_nvts_growing (&configs);
              config_type = config_iterator_type (&configs);
              config_families_growing = config_iterator_families_growing
                                         (&configs);

              SENDF_TO_CLIENT_OR_FAIL ("<family_count>"
                                       "%i<growing>%i</growing>"
                                       "</family_count>"
                                       /* The number of NVT's selected
                                        * by the selector. */
                                       "<nvt_count>"
                                       "%i<growing>%i</growing>"
                                       "</nvt_count>"
                                       "<type>%i</type>",
                                       config_iterator_family_count (&configs),
                                       config_families_growing,
                                       config_iterator_nvt_count (&configs),
                                       config_nvts_growing,
                                       config_type);

              if (config_type == 0 && (get_configs_data->families
                                       || get_configs_data->get.details))
                {
                  iterator_t families;
                  int max_nvt_count = 0, known_nvt_count = 0;

                  SENDF_TO_CLIENT_OR_FAIL ("<families>");
                  init_family_iterator (&families,
                                        config_families_growing,
                                        selector,
                                        1);
                  while (next (&families))
                    {
                      int family_growing, family_max;
                      int family_selected_count;
                      const char *family;

                      family = family_iterator_name (&families);
                      if (family)
                        {
                          family_growing = nvt_selector_family_growing
                                            (selector,
                                             family,
                                             config_families_growing);
                          family_max = family_nvt_count (family);
                          family_selected_count
                            = nvt_selector_nvt_count (selector,
                                                      family,
                                                      family_growing);
                          known_nvt_count += family_selected_count;
                        }
                      else
                        {
                          /* The family can be NULL if an RC adds an
                           * NVT to a config and the NVT is missing
                           * from the NVT cache. */
                          family_growing = 0;
                          family_max = -1;
                          family_selected_count = nvt_selector_nvt_count
                                                   (selector, NULL, 0);
                        }

                      SENDF_TO_CLIENT_OR_FAIL
                       ("<family>"
                        "<name>%s</name>"
                        /* The number of selected NVT's. */
                        "<nvt_count>%i</nvt_count>"
                        /* The total number of NVT's in the family. */
                        "<max_nvt_count>%i</max_nvt_count>"
                        "<growing>%i</growing>"
                        "</family>",
                        family ? family : "",
                        family_selected_count,
                        family_max,
                        family_growing);
                      if (family_max > 0)
                        max_nvt_count += family_max;
                    }
                  cleanup_iterator (&families);
                  SENDF_TO_CLIENT_OR_FAIL
                   ("</families>"
                    /* The total number of NVT's in all the
                     * families for selector selects at least one
                     * NVT. */
                    "<max_nvt_count>%i</max_nvt_count>"
                    /* Total number of selected known NVT's. */
                    "<known_nvt_count>"
                    "%i"
                    "</known_nvt_count>",
                    max_nvt_count,
                    known_nvt_count);
                }

              if (config_type > 0)
                {
                  iterator_t prefs;
                  config_t config = get_iterator_resource (&configs);

                  assert (config);

                  SEND_TO_CLIENT_OR_FAIL ("<preferences>");

                  init_preference_iterator (&prefs, config);
                  while (next (&prefs))
                    {
                      const char *name, *value, *type, *def;
                      char *ovaldi_files = NULL;

                      name = preference_iterator_name (&prefs);
                      value = preference_iterator_value (&prefs);
                      def = preference_iterator_default (&prefs);
                      if (!strcmp (name, "definitions_file"))
                        ovaldi_files = get_ovaldi_files ();
                      type = preference_iterator_type (&prefs);
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<preference><nvt oid=\"\"><name/></nvt>"
                        "<name>%s</name><type>osp_%s</type>"
                        "<value>%s</value><default>%s</default></preference>",
                        name, type, value ?: "", ovaldi_files ?: def);
                      g_free (ovaldi_files);
                    }
                  cleanup_iterator (&prefs);

                  SEND_TO_CLIENT_OR_FAIL ("</preferences>");
                }
              else if (get_configs_data->preferences
                       || get_configs_data->get.details)
                {
                  iterator_t prefs;
                  config_t config = get_iterator_resource (&configs);

                  assert (config);

                  SEND_TO_CLIENT_OR_FAIL ("<preferences>");

                  /* Send NVT timeout preferences where a timeout has been
                   * specified. */
                  init_config_timeout_iterator (&prefs, config);
                  while (next (&prefs))
                    {
                      const char *timeout;

                      timeout = config_timeout_iterator_value (&prefs);

                      if (timeout && strlen (timeout))
                        SENDF_TO_CLIENT_OR_FAIL
                         ("<preference>"
                          "<nvt oid=\"%s\">"
                          "<name>%s</name>"
                          "</nvt>"
                          "<name>Timeout</name>"
                          "<type>entry</type>"
                          "<value>%s</value>"
                          "</preference>",
                          config_timeout_iterator_oid (&prefs),
                          config_timeout_iterator_nvt_name (&prefs),
                          timeout);
                    }
                  cleanup_iterator (&prefs);

                  init_nvt_preference_iterator (&prefs, NULL);
                  while (next (&prefs))
                    {
                      GString *buffer = g_string_new ("");
                      buffer_config_preference_xml (buffer, &prefs, config, 1);
                      SEND_TO_CLIENT_OR_FAIL (buffer->str);
                      g_string_free (buffer, TRUE);
                    }
                  cleanup_iterator (&prefs);

                  SEND_TO_CLIENT_OR_FAIL ("</preferences>");
                }

              if (config_type == 0 && get_configs_data->get.details)
                {
                  iterator_t selectors;

                  SEND_TO_CLIENT_OR_FAIL ("<nvt_selectors>");

                  init_nvt_selector_iterator (&selectors,
                                              NULL,
                                              config,
                                              NVT_SELECTOR_TYPE_ANY);
                  while (next (&selectors))
                    {
                      int type = nvt_selector_iterator_type (&selectors);
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<nvt_selector>"
                        "<name>%s</name>"
                        "<include>%i</include>"
                        "<type>%i</type>"
                        "<family_or_nvt>%s</family_or_nvt>"
                        "</nvt_selector>",
                        nvt_selector_iterator_name (&selectors),
                        nvt_selector_iterator_include (&selectors),
                        type,
                        (type == NVT_SELECTOR_TYPE_ALL
                          ? ""
                          : nvt_selector_iterator_nvt (&selectors)));
                    }
                  cleanup_iterator (&selectors);

                  SEND_TO_CLIENT_OR_FAIL ("</nvt_selectors>");
                }

              if (get_configs_data->tasks)
                {
                  iterator_t tasks;

                  SEND_TO_CLIENT_OR_FAIL ("<tasks>");
                  init_config_task_iterator
                   (&tasks, get_iterator_resource (&configs), 0);
                  while (next (&tasks))
                    {
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<task id=\"%s\">"
                        "<name>%s</name>",
                        config_task_iterator_uuid (&tasks),
                        config_task_iterator_name (&tasks));
                      if (config_task_iterator_readable (&tasks))
                        SEND_TO_CLIENT_OR_FAIL ("</task>");
                      else
                        SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                "</task>");
                    }
                  cleanup_iterator (&tasks);
                  SEND_TO_CLIENT_OR_FAIL ("</tasks>");
                }

              SEND_TO_CLIENT_OR_FAIL ("</config>");
              count++;
            }
          cleanup_iterator (&configs);
          filtered = get_configs_data->get.id
                      ? 1
                      : config_count (&get_configs_data->get);
          SEND_GET_END ("config", &get_configs_data->get, count, filtered);

          get_configs_data_reset (get_configs_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_FILTERS:
        {
          iterator_t filters;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_FILTERS", element_name) == 0);

          INIT_GET (filter, Filter);

          ret = init_filter_iterator (&filters, &get_filters_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_filters", "filter",
                                                   get_filters_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_filters", "filter",
                          get_filters_data->get.filt_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_filters"));
                    break;
                }
              get_filters_data_reset (get_filters_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("filter");
          while (1)
            {
              ret = get_next (&filters, &get_filters_data->get, &first, &count,
                              init_filter_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (filter, &get_filters_data->get, &filters);

              SENDF_TO_CLIENT_OR_FAIL ("<type>%s</type>"
                                       "<term>%s</term>",
                                       filter_iterator_type (&filters),
                                       filter_iterator_term (&filters));

              if (get_filters_data->alerts)
                {
                  iterator_t alerts;

                  SEND_TO_CLIENT_OR_FAIL ("<alerts>");
                  init_filter_alert_iterator (&alerts,
                                              get_iterator_resource
                                               (&filters));
                  while (next (&alerts))
                    {
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<alert id=\"%s\">"
                        "<name>%s</name>",
                        filter_alert_iterator_uuid (&alerts),
                        filter_alert_iterator_name (&alerts));
                      if (filter_alert_iterator_readable (&alerts))
                        SEND_TO_CLIENT_OR_FAIL ("</alert>");
                      else
                        SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                "</alert>");
                    }
                  cleanup_iterator (&alerts);
                  SEND_TO_CLIENT_OR_FAIL ("</alerts>");
                }

              SEND_TO_CLIENT_OR_FAIL ("</filter>");

              count++;
            }
          cleanup_iterator (&filters);
          filtered = get_filters_data->get.id
                      ? 1
                      : filter_count (&get_filters_data->get);
          SEND_GET_END ("filter", &get_filters_data->get, count, filtered);

          get_filters_data_reset (get_filters_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_GROUPS:
        {
          iterator_t groups;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_GROUPS", element_name) == 0);

          INIT_GET (group, Group);

          ret = init_group_iterator (&groups, &get_groups_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_groups", "group",
                                                   get_groups_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_groups", "group", get_groups_data->get.filt_id,
                          omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_groups"));
                    break;
                }
              get_groups_data_reset (get_groups_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("group");
          while (1)
            {
              gchar *users;

              ret = get_next (&groups, &get_groups_data->get, &first, &count,
                              init_group_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (group, &get_groups_data->get, &groups);

              users = group_users (get_iterator_resource (&groups));
              SENDF_TO_CLIENT_OR_FAIL ("<users>%s</users>", users ? users : "");
              g_free (users);

              SEND_TO_CLIENT_OR_FAIL ("</group>");

              count++;
            }
          cleanup_iterator (&groups);
          filtered = get_groups_data->get.id
                      ? 1
                      : group_count (&get_groups_data->get);
          SEND_GET_END ("group", &get_groups_data->get, count, filtered);

          get_groups_data_reset (get_groups_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_INFO:
        {
          iterator_t info;
          int count, first, filtered, ret;
          int (*init_info_iterator) (iterator_t*, get_data_t *, const char *);
          int (*info_count) (const get_data_t *get);
          const char *update_time;
          get_data_t *get;

          if (user_may ("get_info") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_info",
                                  "Permission denied"));
              get_info_data_reset (get_info_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (manage_scap_loaded () == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_info",
                                  "GET_INFO requires the SCAP database."));
              get_info_data_reset (get_info_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }
          if (manage_cert_loaded () == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_info",
                                  "GET_INFO requires the CERT database."));
              get_info_data_reset (get_info_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (get_info_data->name && get_info_data->get.id)
            {
              SEND_TO_CLIENT_OR_FAIL
                (XML_ERROR_SYNTAX ("get_info",
                                   "Only one of name and the id attribute"
                                   " may be given."));
              get_info_data_reset (get_info_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }
          if (get_info_data->type == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL
                (XML_ERROR_SYNTAX ("get_info",
                                   "No type specified."));
              get_info_data_reset (get_info_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          get = &get_info_data->get;
          if (get->filt_id && strcmp (get->filt_id, "-2") == 0)
            {
              char *user_filter;
              gchar *name;

              if (strcmp (get_info_data->type, "cpe") == 0)
                name = g_strdup ("CPE");
              else if (strcmp (get_info_data->type, "cve") == 0)
                name = g_strdup ("CVE");
              else if (strcmp (get_info_data->type, "ovaldef") == 0)
                name = g_strdup ("OVAL");
              else if (strcmp (get_info_data->type, "cert_bund_adv") == 0)
                name = g_strdup ("CERT-Bund");
              else if (strcmp (get_info_data->type, "dfn_cert_adv") == 0)
                name = g_strdup ("DFN-CERT");
              else if (strcmp (get_info_data->type, "nvt") == 0)
                name = g_strdup ("NVT");
              else if (strcmp (get_info_data->type, "allinfo") == 0)
                name = g_strdup ("All SecInfo");
              else
                {
                  if (send_find_error_to_client ("get_info", "type",
                                                 get_info_data->type,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  get_info_data_reset (get_info_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              user_filter = setting_filter (name);
              g_free (name);

              if (user_filter && strlen (user_filter))
                {
                  get->filt_id = user_filter;
                  get->filter = filter_term (user_filter);
                }
              else
                get->filt_id = g_strdup("0");
            }

          /* Set type specific functions */
          if (g_strcmp0 ("cpe", get_info_data->type) == 0)
            {
              init_info_iterator = init_cpe_info_iterator;
              info_count = cpe_info_count;
              get_info_data->get.subtype = g_strdup ("cpe");
            }
          else if (g_strcmp0 ("cve", get_info_data->type) == 0)
            {
              init_info_iterator = init_cve_info_iterator;
              info_count = cve_info_count;
              get_info_data->get.subtype = g_strdup ("cve");
            }
          else if ((g_strcmp0 ("nvt", get_info_data->type) == 0)
                   && (get_info_data->name == NULL)
                   && (get_info_data->get.id == NULL))
            {
              init_info_iterator = init_nvt_info_iterator;
              info_count = nvt_info_count;
              get_info_data->get.subtype = g_strdup ("nvt");
            }
          else if (g_strcmp0 ("nvt", get_info_data->type) == 0)
            {
              gchar *result;

              get_info_data->get.subtype = g_strdup ("nvt");

              manage_read_info (get_info_data->type, get_info_data->get.id,
                                get_info_data->name, &result);
              if (result)
                {
                  SEND_GET_START ("info");
                  SEND_TO_CLIENT_OR_FAIL ("<info>");
                  SEND_TO_CLIENT_OR_FAIL (result);
                  SEND_TO_CLIENT_OR_FAIL ("</info>");
                  SEND_TO_CLIENT_OR_FAIL ("<details>1</details>");
                  SEND_GET_END ("info", &get_info_data->get, 1, 1);
                  g_free (result);
                  get_info_data_reset (get_info_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }
              else
                {
                  if (send_find_error_to_client ("get_info",
                                                 get_info_data->name
                                                  ? "name"
                                                  : "ID",
                                                 get_info_data->name
                                                  ? get_info_data->name
                                                  : get_info_data->get.id,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  get_info_data_reset (get_info_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }
            }
          else if (g_strcmp0 ("ovaldef", get_info_data->type) == 0)
            {
              init_info_iterator = init_ovaldef_info_iterator;
              info_count = ovaldef_info_count;
              get_info_data->get.subtype = g_strdup ("ovaldef");
            }
          else if (g_strcmp0 ("cert_bund_adv", get_info_data->type) == 0)
            {
              init_info_iterator = init_cert_bund_adv_info_iterator;
              info_count = cert_bund_adv_info_count;
              get_info_data->get.subtype = g_strdup ("cert_bund_adv");
            }
          else if (g_strcmp0 ("dfn_cert_adv", get_info_data->type) == 0)
            {
              init_info_iterator = init_dfn_cert_adv_info_iterator;
              info_count = dfn_cert_adv_info_count;
              get_info_data->get.subtype = g_strdup ("dfn_cert_adv");
            }
          else if (g_strcmp0 ("allinfo", get_info_data->type) == 0)
            {
              init_info_iterator = init_all_info_iterator;
              info_count = all_info_count;
              get_info_data->get.subtype = g_strdup ("allinfo");
            }
          else
            {
              if (send_find_error_to_client ("get_info", "type",
                                             get_info_data->type, omp_parser))
                {
                  error_send_to_client (error);
                }
              return;
            }

          ret = init_info_iterator (&info, &get_info_data->get, get_info_data->name);
          if (ret)
            {
              switch (ret)
                {
                case 1:
                  if (send_find_error_to_client ("get_info", "type",
                                                 get_info_data->type,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case 2:
                  if (send_find_error_to_client
                      ("get_info", "filter", get_info_data->get.filt_id,
                       omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case -1:
                  SEND_TO_CLIENT_OR_FAIL
                    (XML_INTERNAL_ERROR ("get_info"));
                  break;
                }
              get_info_data_reset (get_info_data);
              set_client_state (CLIENT_AUTHENTIC);
              return;
            }

          count = 0;
          manage_filter_controls (get_info_data->get.filter, &first, NULL, NULL, NULL);
          SEND_GET_START ("info");
          update_time = manage_scap_update_time ();
          while (next (&info))
            {
              GString *result;

              /* Info's are currently always read only */
              if (send_get_common ("info", &get_info_data->get, &info,
                               write_to_client, write_to_client_data, 0, 0))
                {
                  error_send_to_client (error);
                  return;
                }

              SENDF_TO_CLIENT_OR_FAIL ("<update_time>%s</update_time>",
                                       update_time);

              result = g_string_new ("");

              /* Information depending on type */

              if (g_strcmp0 ("cpe", get_info_data->type) == 0)
                {
                  const char *title;

                  xml_string_append (result, "<cpe>");
                  title = cpe_info_iterator_title (&info);
                  if (title)
                    xml_string_append (result,
                                       "<title>%s</title>",
                                       cpe_info_iterator_title (&info));
                  xml_string_append (result,
                                     "<nvd_id>%s</nvd_id>"
                                     "<max_cvss>%s</max_cvss>"
                                     "<cve_refs>%s</cve_refs>"
                                     "<status>%s</status>",
                                     cpe_info_iterator_nvd_id (&info),
                                     cpe_info_iterator_max_cvss (&info),
                                     cpe_info_iterator_cve_refs (&info),
                                     cpe_info_iterator_status (&info) ?
                                     cpe_info_iterator_status (&info) : "");

                  if (get_info_data->details == 1)
                    {
                      iterator_t cves;
                      g_string_append (result, "<cves>");
                      init_cpe_cve_iterator (&cves, get_iterator_name (&info), 0, NULL);
                      while (next (&cves))
                        xml_string_append (result,
                                           "<cve>"
                                           "<entry"
                                           " xmlns:cpe-lang=\"http://cpe.mitre.org/language/2.0\""
                                           " xmlns:vuln=\"http://scap.nist.gov/schema/vulnerability/0.4\""
                                           " xmlns:xsi=\"http://www.w3.org/2001/XMLSchema-instance\""
                                           " xmlns:patch=\"http://scap.nist.gov/schema/patch/0.1\""
                                           " xmlns:scap-core=\"http://scap.nist.gov/schema/scap-core/0.1\""
                                           " xmlns:cvss=\"http://scap.nist.gov/schema/cvss-v2/0.2\""
                                           " xmlns=\"http://scap.nist.gov/schema/feed/vulnerability/2.0\""
                                           " id=\"%s\">"
                                           "<vuln:cvss>"
                                           "<cvss:base_metrics>"
                                           "<cvss:score>%s</cvss:score>"
                                           "</cvss:base_metrics>"
                                           "</vuln:cvss>"
                                           "</entry>"
                                           "</cve>",
                                           cve_iterator_name (&cves),
                                           cve_iterator_cvss (&cves));
                      cleanup_iterator (&cves);
                      g_string_append (result, "</cves>");
                    }
                }
              else if (g_strcmp0 ("cve", get_info_data->type) == 0)
                {
                  xml_string_append (result,
                                     "<cve>"
                                     "<cvss>%s</cvss>"
                                     "<vector>%s</vector>"
                                     "<complexity>%s</complexity>"
                                     "<authentication>%s</authentication>"
                                     "<confidentiality_impact>%s</confidentiality_impact>"
                                     "<integrity_impact>%s</integrity_impact>"
                                     "<availability_impact>%s</availability_impact>"
                                     "<description>%s</description>"
                                     "<products>%s</products>",
                                     cve_info_iterator_cvss (&info),
                                     cve_info_iterator_vector (&info),
                                     cve_info_iterator_complexity (&info),
                                     cve_info_iterator_authentication (&info),
                                     cve_info_iterator_confidentiality_impact (&info),
                                     cve_info_iterator_integrity_impact (&info),
                                     cve_info_iterator_availability_impact (&info),
                                     cve_info_iterator_description (&info),
                                     cve_info_iterator_products (&info));
                  if (get_info_data->details == 1)
                    {
                      iterator_t nvts;
                      iterator_t cert_advs;
                      init_cve_nvt_iterator (&nvts,  get_iterator_name (&info), 1, NULL);
                      g_string_append (result, "<nvts>");
                      while (next (&nvts))
                        xml_string_append (result,
                                           "<nvt oid=\"%s\">"
                                           "<name>%s</name>"
                                           "</nvt>",
                                           nvt_iterator_oid (&nvts),
                                           nvt_iterator_name (&nvts));
                      g_string_append (result, "</nvts>");
                      cleanup_iterator (&nvts);

                      g_string_append (result, "<cert>");
                      if (manage_cert_loaded())
                        {
                          init_cve_cert_bund_adv_iterator (&cert_advs,
                                                          get_iterator_name
                                                            (&info),
                                                          1, NULL);
                          while (next (&cert_advs))
                            {
                              xml_string_append
                                (result,
                                 "<cert_ref type=\"CERT-Bund\">"
                                 "<name>%s</name>"
                                 "<title>%s</title>"
                                 "</cert_ref>",
                                 get_iterator_name (&cert_advs),
                                 cert_bund_adv_info_iterator_title
                                  (&cert_advs));
                          };
                          cleanup_iterator (&cert_advs);

                          init_cve_dfn_cert_adv_iterator (&cert_advs,
                                                          get_iterator_name
                                                            (&info),
                                                          1, NULL);
                          while (next (&cert_advs))
                            {
                              xml_string_append (result,
                                                "<cert_ref type=\"DFN-CERT\">"
                                                "<name>%s</name>"
                                                "<title>%s</title>"
                                                "</cert_ref>",
                                                get_iterator_name (&cert_advs),
                                                dfn_cert_adv_info_iterator_title
                                                  (&cert_advs));
                          };
                          cleanup_iterator (&cert_advs);
                        }
                      else
                        {
                          g_string_append(result, "<warning>"
                                                  "database not available"
                                                  "</warning>");
                        }
                      g_string_append (result, "</cert>");
                    }
                }
              else if (g_strcmp0 ("ovaldef", get_info_data->type) == 0)
                {
                  const char *description;
                  xml_string_append (result,
                                     "<ovaldef>"
                                     "<version>%s</version>"
                                     "<deprecated>%s</deprecated>"
                                     "<status>%s</status>"
                                     "<class>%s</class>"
                                     "<title>%s</title>"
                                     "<max_cvss>%s</max_cvss>"
                                     "<cve_refs>%s</cve_refs>"
                                     "<file>%s</file>",
                                     ovaldef_info_iterator_version (&info),
                                     ovaldef_info_iterator_deprecated (&info),
                                     ovaldef_info_iterator_status (&info),
                                     ovaldef_info_iterator_class (&info),
                                     ovaldef_info_iterator_title (&info),
                                     ovaldef_info_iterator_max_cvss (&info),
                                     ovaldef_info_iterator_cve_refs (&info),
                                     ovaldef_info_iterator_file (&info));
                  description = ovaldef_info_iterator_description (&info);
                  if (get_info_data->details == 1)
                    xml_string_append (result,
                                       "<description>%s</description>",
                                       description);
                }
              else if (g_strcmp0 ("cert_bund_adv", get_info_data->type) == 0)
                xml_string_append (result,
                                   "<cert_bund_adv>"
                                   "<title>%s</title>"
                                   "<summary>%s</summary>"
                                   "<max_cvss>%s</max_cvss>"
                                   "<cve_refs>%s</cve_refs>",
                                   cert_bund_adv_info_iterator_title (&info),
                                   cert_bund_adv_info_iterator_summary (&info),
                                   cert_bund_adv_info_iterator_max_cvss(&info),
                                   cert_bund_adv_info_iterator_cve_refs (&info));
              else if (g_strcmp0 ("dfn_cert_adv", get_info_data->type) == 0)
                xml_string_append (result,
                                   "<dfn_cert_adv>"
                                   "<title>%s</title>"
                                   "<summary>%s</summary>"
                                   "<max_cvss>%s</max_cvss>"
                                   "<cve_refs>%s</cve_refs>",
                                   dfn_cert_adv_info_iterator_title (&info),
                                   dfn_cert_adv_info_iterator_summary (&info),
                                   dfn_cert_adv_info_iterator_max_cvss(&info),
                                   dfn_cert_adv_info_iterator_cve_refs (&info));
              else if (g_strcmp0 ("nvt", get_info_data->type) == 0)
                {
                  if (send_nvt (&info, 1, 1, -1, NULL, 0, write_to_client,
                                write_to_client_data))
                    {
                      cleanup_iterator (&info);
                      error_send_to_client (error);
                      return;
                    }
                }
              else if (g_strcmp0 ("allinfo", get_info_data->type) == 0)
                {
                  const char *extra = all_info_iterator_extra (&info);
                  xml_string_append (result,
                                     "<allinfo>"
                                     "<type>%s</type>"
                                     "<extra>%s</extra>"
                                     "<severity>%s</severity>",
                                     all_info_iterator_type (&info),
                                     extra ? extra : "",
                                     all_info_iterator_severity (&info));
                }

              /* Append raw data if full details are requested */

              if (get_info_data->details == 1)
                {
                  gchar *raw_data = NULL;
                  gchar *nonconst_id = g_strdup(get_iterator_uuid (&info));
                  gchar *nonconst_name = g_strdup(get_iterator_name (&info));
                  manage_read_info (get_info_data->type, nonconst_id,
                                    nonconst_name, &raw_data);
                  g_string_append_printf (result, "<raw_data>%s</raw_data>",
                                          raw_data);
                  g_free(nonconst_id);
                  g_free(nonconst_name);
                  g_free(raw_data);
                }

              g_string_append_printf (result, "</%s></info>", get_info_data->type);
              SEND_TO_CLIENT_OR_FAIL (result->str);
              count++;
              g_string_free (result, TRUE);
            }
          cleanup_iterator (&info);

          if (get_info_data->details == 1)
            SEND_TO_CLIENT_OR_FAIL ("<details>1</details>");

          filtered = get_info_data->get.id
                      ? 1
                      : (get_info_data->name
                          ? info_name_count (get_info_data->type,
                                             get_info_data->name)
                          : info_count (&get_info_data->get));

          if (strcmp (get_info_data->type, "allinfo"))
            SEND_GET_END ("info", &get_info_data->get, count, filtered);
          else
            send_get_end ("info", &get_info_data->get, count, filtered,
                          total_info_count(&get_info_data->get, 0),
                          write_to_client, write_to_client_data);

          get_info_data_reset (get_info_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_LSC_CREDENTIALS:
        {
          iterator_t credentials;
          int count, filtered, ret, first;
          int format;
          char *data_format;

          assert (strcasecmp ("GET_LSC_CREDENTIALS", element_name) == 0);

          data_format = get_lsc_credentials_data->format;
          if (data_format)
            {
              if (strlen (data_format))
                {
                  if (strcasecmp (data_format, "key") == 0)
                    format = 1;
                  else if (strcasecmp (data_format, "rpm") == 0)
                    format = 2;
                  else if (strcasecmp (data_format, "deb") == 0)
                    format = 3;
                  else if (strcasecmp (data_format, "exe") == 0)
                    format = 4;
                  else
                    format = -1;
                }
              else
                format = 0;
            }
          else
            format = 0;

          if (format == -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_lsc_credentials",
                                "GET_LSC_CREDENTIALS format attribute should"
                                " be 'key', 'rpm', 'deb' or 'exe'."));

          INIT_GET (lsc_credential, Credential);

          ret = init_lsc_credential_iterator (&credentials,
                                              &get_lsc_credentials_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_lsc_credentials",
                                                   "lsc_credential",
                                                   get_lsc_credentials_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_lsc_credentials", "lsc_credential",
                          get_lsc_credentials_data->get.filt_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_lsc_credentials"));
                    break;
                }
              get_lsc_credentials_data_reset (get_lsc_credentials_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START("lsc_credential");
          while (1)
            {
              const char* private_key;

              ret = get_next (&credentials, &get_lsc_credentials_data->get,
                              &first, &count, init_lsc_credential_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (lsc_credential, &get_lsc_credentials_data->get,
                               &credentials);
              private_key = lsc_credential_iterator_private_key (&credentials);
              SENDF_TO_CLIENT_OR_FAIL
               ("<login>%s</login>"
                "<type>%s</type>",
                lsc_credential_iterator_login (&credentials),
                private_key ? "gen" : "pass");

              switch (format)
                {
                  char *package;

                  case 1: /* key */
                    {
                      char *pub;
                      const char *pass;

                      pass = lsc_credential_iterator_password (&credentials);
                      pub = openvas_ssh_public_from_private (private_key, pass);
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<public_key>%s</public_key>", pub ?: "");
                      g_free (pub);
                      break;
                    }
                  case 2: /* rpm */
                    package = lsc_credential_iterator_rpm (&credentials);
                    SENDF_TO_CLIENT_OR_FAIL
                     ("<package format=\"rpm\">%s</package>", package ?: "");
                    g_free (package);
                    break;
                  case 3: /* deb */
                    package = lsc_credential_iterator_deb (&credentials);
                    SENDF_TO_CLIENT_OR_FAIL
                     ("<package format=\"deb\">%s</package>", package ?: "");
                    g_free (package);
                    break;
                  case 4: /* exe */
                    package = lsc_credential_iterator_exe (&credentials);
                    SENDF_TO_CLIENT_OR_FAIL
                     ("<package format=\"exe\">%s</package>", package ?: "");
                    g_free (package);
                    break;
                }

              if (get_lsc_credentials_data->targets)
                {
                  iterator_t targets;

                  SENDF_TO_CLIENT_OR_FAIL ("<targets>");
                  init_lsc_credential_target_iterator
                   (&targets, get_iterator_resource (&credentials), 0);
                  while (next (&targets))
                    {
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<target id=\"%s\">"
                        "<name>%s</name>",
                        lsc_credential_target_iterator_uuid (&targets),
                        lsc_credential_target_iterator_name (&targets));
                      if (lsc_credential_target_iterator_readable (&targets))
                        SEND_TO_CLIENT_OR_FAIL ("</target>");
                      else
                        SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                "</target>");
                    }
                  cleanup_iterator (&targets);

                  SEND_TO_CLIENT_OR_FAIL ("</targets>");
                }

              SEND_TO_CLIENT_OR_FAIL ("</lsc_credential>");
              count++;
            }

          cleanup_iterator (&credentials);
          filtered = get_lsc_credentials_data->get.id
                      ? 1
                      : lsc_credential_count (&get_lsc_credentials_data->get);
          SEND_GET_END ("lsc_credential", &get_lsc_credentials_data->get,
                        count, filtered);
          get_lsc_credentials_data_reset (get_lsc_credentials_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_NOTES:
        {
          nvt_t nvt = 0;
          task_t task = 0;

          assert (strcasecmp ("GET_NOTES", element_name) == 0);

          if (get_notes_data->note_id && get_notes_data->nvt_oid)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_notes",
                                "Only one of NVT and the note_id attribute"
                                " may be given"));
          else if (get_notes_data->note_id && get_notes_data->task_id)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_notes",
                                "Only one of the note_id and task_id"
                                " attributes may be given"));
          else if (get_notes_data->task_id
                   && find_task (get_notes_data->task_id, &task))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_notes"));
          else if (get_notes_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("get_notes",
                                             "task", get_notes_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (get_notes_data->nvt_oid
                   && find_nvt (get_notes_data->nvt_oid, &nvt))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_notes"));
          else if (get_notes_data->nvt_oid && nvt == 0)
            {
              if (send_find_error_to_client ("get_notes", "NVT",
                                             get_notes_data->nvt_oid,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else
            {
              iterator_t notes;
              GString *buffer;
              int count, filtered, ret, first;

              INIT_GET (note, Note);

              ret = init_note_iterator (&notes, &get_notes_data->get, nvt, 0,
                                        task);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client ("get_notes", "note",
                                                       get_notes_data->get.id,
                                                       omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_notes", "filter",
                              get_notes_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_notes"));
                        break;
                    }
                  get_notes_data_reset (get_notes_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("note");

              buffer = g_string_new ("");

              // TODO: Do the iteration with get_next so it checks "first".
              buffer_notes_xml (buffer, &notes, get_notes_data->get.details,
                                get_notes_data->result, &count);

              SEND_TO_CLIENT_OR_FAIL (buffer->str);
              g_string_free (buffer, TRUE);

              cleanup_iterator (&notes);
              filtered = get_notes_data->get.id
                          ? 1
                          : note_count (&get_notes_data->get, nvt, 0, task);
              SEND_GET_END ("note", &get_notes_data->get, count, filtered);
            }
          get_notes_data_reset (get_notes_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_NVT_FEED_VERSION:
        {
          char *feed_version;

          if (user_may ("get_nvt_feed_version") == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_nvt_feed_version",
                                "Permission denied"));
          else if ((feed_version = nvts_feed_version ()))
            {
              SEND_TO_CLIENT_OR_FAIL ("<get_nvt_feed_version_response"
                                      " status=\"" STATUS_OK "\""
                                      " status_text=\"" STATUS_OK_TEXT "\">"
                                      "<version>");
              SEND_TO_CLIENT_OR_FAIL (feed_version);
              free (feed_version);
              SEND_TO_CLIENT_OR_FAIL ("</version>"
                                      "</get_nvt_feed_version_response>");
            }
          else
            SEND_XML_SERVICE_DOWN ("get_nvt_feed_version");
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_NVTS:
        {
          char *feed_version;

          if (user_may ("get_nvts") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_nvts",
                                  "Permission denied"));
              get_nvts_data_reset (get_nvts_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          feed_version = nvts_feed_version ();
          if (feed_version)
            {
              config_t config = (config_t) 0;
              nvt_t nvt = 0;

              free (feed_version);

              if (get_nvts_data->nvt_oid && get_nvts_data->family)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_nvts",
                                    "Too many parameters at once"));
              else if ((get_nvts_data->details == 0)
                       && get_nvts_data->preference_count)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_nvts",
                                    "GET_NVTS preference_count attribute"
                                    " requires the details attribute"));
              else if ((get_nvts_data->details == 0)
                       && get_nvts_data->preferences)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_nvts",
                                    "GET_NVTS preferences attribute"
                                    " requires the details attribute"));
              else if (((get_nvts_data->details == 0)
                        || (get_nvts_data->config_id == NULL))
                       && get_nvts_data->timeout)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_nvts",
                                    "GET_NVTS timeout attribute"
                                    " requires the details and config_id"
                                    " attributes"));
              else if (get_nvts_data->nvt_oid
                       && find_nvt (get_nvts_data->nvt_oid, &nvt))
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("get_nvts"));
              else if (get_nvts_data->nvt_oid && nvt == 0)
                {
                  if (send_find_error_to_client ("get_nvts", "NVT",
                                                 get_nvts_data->nvt_oid,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                }
              else if (get_nvts_data->config_id
                       && find_config_with_permission (get_nvts_data->config_id,
                                                       &config,
                                                       NULL))
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("get_nvts"));
              else if (get_nvts_data->config_id && (config == 0))
                {
                  if (send_find_error_to_client
                       ("get_nvts", "config", get_nvts_data->config_id,
                        omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                }
              else
                {
                  iterator_t nvts;

                  SENDF_TO_CLIENT_OR_FAIL
                   ("<get_nvts_response"
                    " status=\"" STATUS_OK "\""
                    " status_text=\"" STATUS_OK_TEXT "\">");

                  init_nvt_iterator (&nvts,
                                     nvt,
                                     get_nvts_data->nvt_oid
                                      /* Presume the NVT is in the config (if
                                       * a config was given). */
                                      ? 0
                                      : config,
                                     get_nvts_data->family,
                                     NULL,
                                     get_nvts_data->sort_order,
                                     get_nvts_data->sort_field);
                  if (get_nvts_data->details)
                    while (next (&nvts))
                      {
                        int pref_count = -1;
                        char *timeout = NULL;

                        if (get_nvts_data->timeout)
                          timeout = config_nvt_timeout (config,
                                                        nvt_iterator_oid (&nvts));

                        if (get_nvts_data->preferences && (timeout == NULL))
                          timeout = config_nvt_timeout
                                     (config,
                                      nvt_iterator_oid (&nvts));

                        if (get_nvts_data->preference_count)
                          {
                            const char *nvt_name = nvt_iterator_name (&nvts);
                            pref_count = nvt_preference_count (nvt_name);
                          }
                        if (send_nvt (&nvts, 1, get_nvts_data->preferences,
                                      pref_count, timeout, config,
                                      write_to_client, write_to_client_data))
                          {
                            free (timeout);
                            cleanup_iterator (&nvts);
                            error_send_to_client (error);
                            return;
                          }
                        free (timeout);

                        SEND_TO_CLIENT_OR_FAIL ("</nvt>");
                      }
                  else
                    while (next (&nvts))
                      {
                        if (send_nvt (&nvts, 0, 0, -1, NULL, 0, write_to_client,
                                      write_to_client_data))
                          {
                            cleanup_iterator (&nvts);
                            error_send_to_client (error);
                            return;
                          }
                        SEND_TO_CLIENT_OR_FAIL ("</nvt>");
                      }
                  cleanup_iterator (&nvts);

                  SEND_TO_CLIENT_OR_FAIL ("</get_nvts_response>");
                }
            }
          else
            SEND_XML_SERVICE_DOWN ("get_nvts");
        }
        get_nvts_data_reset (get_nvts_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_GET_NVT_FAMILIES:
        {
          iterator_t families;

          if (user_may ("get_nvt_families") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_nvt_families",
                                  "Permission denied"));
              get_nvt_families_data_reset (get_nvt_families_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_TO_CLIENT_OR_FAIL ("<get_nvt_families_response"
                                  " status=\"" STATUS_OK "\""
                                  " status_text=\"" STATUS_OK_TEXT "\">"
                                  "<families>");

          init_family_iterator (&families,
                                1,
                                NULL,
                                get_nvt_families_data->sort_order);
          while (next (&families))
            {
              int family_max;
              const char *family;

              family = family_iterator_name (&families);
              if (family)
                family_max = family_nvt_count (family);
              else
                family_max = -1;

              SENDF_TO_CLIENT_OR_FAIL
               ("<family>"
                "<name>%s</name>"
                /* The total number of NVT's in the family. */
                "<max_nvt_count>%i</max_nvt_count>"
                "</family>",
                family ? family : "",
                family_max);
            }
          cleanup_iterator (&families);

          SEND_TO_CLIENT_OR_FAIL ("</families>"
                                  "</get_nvt_families_response>");
        }
        get_nvt_families_data_reset (get_nvt_families_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_GET_OVERRIDES:
        {
          nvt_t nvt = 0;
          task_t task = 0;

          assert (strcasecmp ("GET_OVERRIDES", element_name) == 0);

          if (get_overrides_data->override_id && get_overrides_data->nvt_oid)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_overrides",
                                "Only one of NVT and the override_id attribute"
                                " may be given"));
          else if (get_overrides_data->override_id
                   && get_overrides_data->task_id)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_overrides",
                                "Only one of the override_id and task_id"
                                " attributes may be given"));
          else if (get_overrides_data->task_id
                   && find_task (get_overrides_data->task_id, &task))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_overrides"));
          else if (get_overrides_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("get_overrides", "task",
                                             get_overrides_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (get_overrides_data->nvt_oid
                   && find_nvt (get_overrides_data->nvt_oid, &nvt))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_overrides"));
          else if (get_overrides_data->nvt_oid && nvt == 0)
            {
              if (send_find_error_to_client ("get_overrides",
                                             "NVT", get_overrides_data->nvt_oid,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else
            {
              iterator_t overrides;
              GString *buffer;
              int count, filtered, ret, first;

              INIT_GET (override, Override);

              ret = init_override_iterator (&overrides,
                                            &get_overrides_data->get, nvt, 0,
                                            task);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client
                             ("get_overrides", "override",
                              get_overrides_data->get.id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_overrides", "filter",
                              get_overrides_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_overrides"));
                        break;
                    }
                  get_overrides_data_reset (get_overrides_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("override");

              buffer = g_string_new ("");

              // TODO: Do the iteration with get_next so it checks "first".
              buffer_overrides_xml (buffer, &overrides,
                                    get_overrides_data->get.details,
                                    get_overrides_data->result, &count);

              SEND_TO_CLIENT_OR_FAIL (buffer->str);
              g_string_free (buffer, TRUE);

              cleanup_iterator (&overrides);
              filtered = get_overrides_data->get.id
                          ? 1
                          : override_count (&get_overrides_data->get, nvt, 0,
                                            task);
              SEND_GET_END ("override", &get_overrides_data->get, count,
                            filtered);
            }
          get_overrides_data_reset (get_overrides_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_PERMISSIONS:
        {
          iterator_t permissions;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_PERMISSIONS", element_name) == 0);

          INIT_GET (permission, Permission);

          ret = init_permission_iterator (&permissions,
                                          &get_permissions_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_permissions",
                                                   "permission",
                                                   get_permissions_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_permissions", "filter",
                          get_permissions_data->get.filt_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_permissions"));
                    break;
                }
              get_permissions_data_reset (get_permissions_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("permission");
          while (1)
            {
              const char *resource_type;

              ret = get_next (&permissions, &get_permissions_data->get, &first,
                              &count, init_permission_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (permission, &get_permissions_data->get, &permissions);

              resource_type = permission_iterator_resource_type (&permissions);
              SENDF_TO_CLIENT_OR_FAIL
               ("<resource id=\"%s\">"
                "<name>%s</name>"
                "<type>%s</type>"
                "<trash>%i</trash>"
                "<deleted>%i</deleted>"
                "</resource>"
                "<subject id=\"%s\">"
                "<name>%s</name>"
                "<type>%s</type>"
                "<trash>%i</trash>"
                "</subject>",
                permission_iterator_resource_uuid (&permissions),
                resource_type && strcmp (resource_type, "")
                 ? permission_iterator_resource_name (&permissions)
                 : "",
                permission_iterator_resource_type (&permissions),
                permission_iterator_resource_in_trash (&permissions),
                permission_iterator_resource_orphan (&permissions),
                permission_iterator_subject_uuid (&permissions),
                permission_iterator_subject_name (&permissions),
                permission_iterator_subject_type (&permissions),
                permission_iterator_subject_in_trash (&permissions));

              SEND_TO_CLIENT_OR_FAIL ("</permission>");
              count++;
            }
          cleanup_iterator (&permissions);
          filtered = get_permissions_data->get.id
                      ? 1
                      : permission_count (&get_permissions_data->get);
          SEND_GET_END ("permission", &get_permissions_data->get, count, filtered);

          get_permissions_data_reset (get_permissions_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_PORT_LISTS:
        {
          iterator_t port_lists;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_PORT_LISTS", element_name) == 0);

          INIT_GET (port_list, Port List);

          ret = init_port_list_iterator (&port_lists,
                                         &get_port_lists_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_port_lists",
                                                   "port_list",
                                                   get_port_lists_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_port_lists", "port_list",
                          get_port_lists_data->get.filt_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_port_lists"));
                    break;
                }
              get_port_lists_data_reset (get_port_lists_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("port_list");
          while (1)
            {
              ret = get_next (&port_lists, &get_port_lists_data->get, &first,
                              &count, init_port_list_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (port_list, &get_port_lists_data->get,
                               &port_lists);

              SENDF_TO_CLIENT_OR_FAIL ("<port_count>"
                                       "<all>%i</all>"
                                       "<tcp>%i</tcp>"
                                       "<udp>%i</udp>"
                                       "</port_count>",
                                       port_list_iterator_count_all
                                        (&port_lists),
                                       port_list_iterator_count_tcp
                                        (&port_lists),
                                       port_list_iterator_count_udp
                                        (&port_lists));

              if (get_port_lists_data->get.details)
                {
                  iterator_t ranges;

                  SEND_TO_CLIENT_OR_FAIL ("<port_ranges>");

                  init_port_range_iterator (&ranges,
                                            get_iterator_resource (&port_lists),
                                            0, 1, NULL);
                  while (next (&ranges))
                    SENDF_TO_CLIENT_OR_FAIL
                     ("<port_range id=\"%s\">"
                      "<start>%s</start>"
                      "<end>%s</end>"
                      "<type>%s</type>"
                      "<comment>%s</comment>"
                      "</port_range>",
                      port_range_iterator_uuid (&ranges),
                      port_range_iterator_start (&ranges),
                      port_range_iterator_end (&ranges)
                       ? port_range_iterator_end (&ranges)
                       : port_range_iterator_start (&ranges),
                      port_range_iterator_type (&ranges),
                      port_range_iterator_comment (&ranges));
                  cleanup_iterator (&ranges);

                  SENDF_TO_CLIENT_OR_FAIL ("</port_ranges>");
                }

              if (get_port_lists_data->targets)
                {
                  iterator_t targets;

                  SEND_TO_CLIENT_OR_FAIL ("<targets>");

                  init_port_list_target_iterator (&targets,
                                                  get_iterator_resource
                                                   (&port_lists), 0);
                  while (next (&targets))
                    {
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<target id=\"%s\">"
                        "<name>%s</name>",
                        port_list_target_iterator_uuid (&targets),
                        port_list_target_iterator_name (&targets));
                      if (port_list_target_iterator_readable (&targets))
                        SEND_TO_CLIENT_OR_FAIL ("</target>");
                      else
                        SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                "</target>");
                    }

                  cleanup_iterator (&targets);

                  SEND_TO_CLIENT_OR_FAIL ("</targets>");
                }

              SEND_TO_CLIENT_OR_FAIL ("</port_list>");

              count++;
            }

          cleanup_iterator (&port_lists);
          filtered = get_port_lists_data->get.id
                      ? 1
                      : port_list_count (&get_port_lists_data->get);
          SEND_GET_END ("port_list", &get_port_lists_data->get, count,
                        filtered);

          get_port_lists_data_reset (get_port_lists_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_PREFERENCES:
        {
          iterator_t prefs;
          nvt_t nvt = 0;
          config_t config = 0;

          if (user_may ("get_preferences") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_preferences",
                                  "Permission denied"));
              get_preferences_data_reset (get_preferences_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (get_preferences_data->nvt_oid
              && find_nvt (get_preferences_data->nvt_oid, &nvt))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_preferences"));
          else if (get_preferences_data->nvt_oid && nvt == 0)
            {
              if (send_find_error_to_client ("get_preferences", "NVT",
                                             get_preferences_data->nvt_oid,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (get_preferences_data->config_id
                   && find_config (get_preferences_data->config_id, &config))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_preferences"));
          else if (get_preferences_data->config_id && config == 0)
            {
              if (send_find_error_to_client ("get_preferences", "config",
                                             get_preferences_data->config_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else
            {
              char *nvt_name = manage_nvt_name (nvt);
              SEND_TO_CLIENT_OR_FAIL ("<get_preferences_response"
                                      " status=\"" STATUS_OK "\""
                                      " status_text=\"" STATUS_OK_TEXT "\">");
              init_nvt_preference_iterator (&prefs, nvt_name);
              free (nvt_name);
              if (get_preferences_data->preference)
                while (next (&prefs))
                  {
                    char *name = strstr (nvt_preference_iterator_name (&prefs), "]:");
                    if (name
                        && (strcmp (name + 2,
                                    get_preferences_data->preference)
                            == 0))
                      {
                        GString *buffer = g_string_new ("");
                        buffer_config_preference_xml (buffer, &prefs, config, 1);
                        SEND_TO_CLIENT_OR_FAIL (buffer->str);
                        g_string_free (buffer, TRUE);
                        break;
                      }
                  }
              else
                while (next (&prefs))
                  {
                    GString *buffer = g_string_new ("");
                    buffer_config_preference_xml (buffer, &prefs, config, 1);
                    SEND_TO_CLIENT_OR_FAIL (buffer->str);
                    g_string_free (buffer, TRUE);
                  }

              cleanup_iterator (&prefs);
              SEND_TO_CLIENT_OR_FAIL ("</get_preferences_response>");
            }
          get_preferences_data_reset (get_preferences_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_REPORTS:
        assert (strcasecmp ("GET_REPORTS", element_name) == 0);
        if (current_credentials.username == NULL)
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_reports"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        report_t request_report = 0, delta_report = 0, report;
        report_format_t report_format;
        float min_cvss_base;
        iterator_t reports;
        int count, filtered, ret, first;

        if (user_may ("get_reports") == 0)
          {
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_reports",
                                "Permission denied"));
            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        /** @todo Some checks only required when type is "scan". */

        if (strcmp (get_reports_data->type, "scan")
            && strcmp (get_reports_data->type, "assets")
            && strcmp (get_reports_data->type, "prognostic"))
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_reports",
                                "GET_REPORTS type must be scan, assets or"
                                " prognostic"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (strcmp (get_reports_data->type, "prognostic") == 0
            && manage_scap_loaded () == 0)
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_reports",
                                "GET_REPORTS with type prognostic requires the"
                                " SCAP database"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if ((strcmp (get_reports_data->type, "scan") == 0)
            && get_reports_data->report_id
            && find_report_with_permission (get_reports_data->report_id,
                                            &request_report,
                                            NULL))
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_reports"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (get_reports_data->delta_report_id
            && strcmp (get_reports_data->delta_report_id, "0")
            && find_report_with_permission (get_reports_data->delta_report_id,
                                            &delta_report,
                                            NULL))
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_reports"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (get_reports_data->format_id == NULL)
          get_reports_data->format_id
           = g_strdup ("a994b278-1f62-11e1-96ac-406186ea4fc5");

        if (find_report_format_with_permission (get_reports_data->format_id,
                                                &report_format,
                                                "get_report_formats"))
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_reports"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (report_format == 0)
          {
            if (send_find_error_to_client ("get_reports", "report format",
                                           get_reports_data->format_id,
                                           omp_parser))
              {
                error_send_to_client (error);
                return;
              }
            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if ((strcmp (get_reports_data->type, "scan") == 0)
            && get_reports_data->report_id
            && request_report == 0)
          {
            if (send_find_error_to_client ("get_reports", "report",
                                           get_reports_data->report_id,
                                           omp_parser))
              {
                error_send_to_client (error);
                return;
              }
            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if ((strcmp (get_reports_data->type, "scan") == 0)
            && get_reports_data->delta_report_id
            && strcmp (get_reports_data->delta_report_id, "0")
            && delta_report == 0)
          {
            if (send_find_error_to_client ("get_reports", "report",
                                           get_reports_data->delta_report_id,
                                           omp_parser))
              {
                error_send_to_client (error);
                return;
              }
            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (((strcmp (get_reports_data->type, "scan") == 0)
             || (strcmp (get_reports_data->type, "prognostic") == 0))
            && get_reports_data->min_cvss_base
            && strlen (get_reports_data->min_cvss_base)
            && (sscanf (get_reports_data->min_cvss_base,
                        "%f",
                        &min_cvss_base)
                != 1))
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_reports",
                                "GET_REPORTS min_cvss_base must be a float"
                                " or the empty string"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (report_format_active (report_format) == 0)
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_reports",
                                "GET_REPORTS report format must be active"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if ((report_format_predefined (report_format) == 0)
            && (report_format_trust (report_format) > 1))
          {
            get_reports_data_reset (get_reports_data);
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_reports",
                                "GET_REPORTS report format must be predefined"
                                " or trusted"));
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (strcmp (get_reports_data->type, "assets") == 0)
          {
            gchar *extension, *content_type;
            int ret, pos;
            get_data_t * get;

            /* An asset report. */

            get = &get_reports_data->get;
            if (get->filt_id && strcmp (get->filt_id, "-2") == 0)
              {
                g_free (get->filt_id);
                get->filt_id = g_strdup ("0");
              }

            if (get_reports_data->alert_id == NULL)
              SEND_TO_CLIENT_OR_FAIL
               ("<get_reports_response"
                " status=\"" STATUS_OK "\""
                " status_text=\"" STATUS_OK_TEXT "\">");

            content_type = report_format_content_type (report_format);
            extension = report_format_extension (report_format);

            SENDF_TO_CLIENT_OR_FAIL
             ("<report"
              " type=\"assets\""
              " format_id=\"%s\""
              " extension=\"%s\""
              " content_type=\"%s\">",
              get_reports_data->format_id,
              extension,
              content_type);

            g_free (extension);
            g_free (content_type);

            pos = get_reports_data->pos ? atoi (get_reports_data->pos) : 1;
            ret = manage_send_report (0,
                                      0,
                                      report_format,
                                      &get_reports_data->get,
                                      get_reports_data->sort_order,
                                      get_reports_data->sort_field,
                                      get_reports_data->result_hosts_only,
                                      NULL,
                                      NULL,
                                      get_reports_data->levels,
                                      get_reports_data->delta_states,
                                      get_reports_data->apply_overrides,
                                      get_reports_data->search_phrase,
                                      get_reports_data->autofp,
                                      get_reports_data->notes,
                                      get_reports_data->notes_details,
                                      get_reports_data->overrides,
                                      get_reports_data->overrides_details,
                                      get_reports_data->first_result,
                                      get_reports_data->max_results,
                                      get_reports_data->ignore_pagination,
                                      /* Special case the XML reports, bah. */
                                      strcmp
                                       (get_reports_data->format_id,
                                        "a994b278-1f62-11e1-96ac-406186ea4fc5")
                                      && strcmp
                                          (get_reports_data->format_id,
                                           "5057e5cc-b825-11e4"
                                           "-9d0e-28d24461215b"),
                                      send_to_client,
                                      write_to_client,
                                      write_to_client_data,
                                      get_reports_data->alert_id,
                                      "assets",
                                      get_reports_data->host,
                                      pos,
                                      NULL, NULL, 0, 0, NULL,
                                      get_reports_data->timezone);

            if (ret)
              {
                internal_error_send_to_client (error);
                get_reports_data_reset (get_reports_data);
                set_client_state (CLIENT_AUTHENTIC);
                return;
              }

            SEND_TO_CLIENT_OR_FAIL ("</report>"
                                    "</get_reports_response>");

            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (strcmp (get_reports_data->type, "prognostic") == 0)
          {
            gchar *extension, *content_type;
            int ret, pos;
            get_data_t * get;

            /* A prognostic report. */

            get = &get_reports_data->get;
            if (get->filt_id && strcmp (get->filt_id, "-2") == 0)
              {
                g_free (get->filt_id);
                get->filt_id = g_strdup ("0");
              }

            if (get_reports_data->alert_id == NULL)
              SEND_TO_CLIENT_OR_FAIL
               ("<get_reports_response"
                " status=\"" STATUS_OK "\""
                " status_text=\"" STATUS_OK_TEXT "\">");

            content_type = report_format_content_type (report_format);
            extension = report_format_extension (report_format);

            SENDF_TO_CLIENT_OR_FAIL
             ("<report"
              " type=\"prognostic\""
              " format_id=\"%s\""
              " extension=\"%s\""
              " content_type=\"%s\">",
              get_reports_data->format_id,
              extension,
              content_type);

            g_free (extension);
            g_free (content_type);

            pos = get_reports_data->pos ? atoi (get_reports_data->pos) : 1;
            ret = manage_send_report (0,
                                      0,
                                      report_format,
                                      &get_reports_data->get,
                                      get_reports_data->sort_order,
                                      get_reports_data->sort_field,
                                      get_reports_data->result_hosts_only,
                                      get_reports_data->min_cvss_base,
                                      get_reports_data->min_qod,
                                      get_reports_data->levels,
                                      get_reports_data->delta_states,
                                      get_reports_data->apply_overrides,
                                      get_reports_data->search_phrase,
                                      get_reports_data->autofp,
                                      get_reports_data->notes,
                                      get_reports_data->notes_details,
                                      get_reports_data->overrides,
                                      get_reports_data->overrides_details,
                                      get_reports_data->first_result,
                                      get_reports_data->max_results,
                                      get_reports_data->ignore_pagination,
                                      /* Special case the XML report, bah. */
                                      strcmp
                                       (get_reports_data->format_id,
                                        "a994b278-1f62-11e1-96ac-406186ea4fc5")
                                      && strcmp
                                          (get_reports_data->format_id,
                                           "5057e5cc-b825-11e4"
                                           "-9d0e-28d24461215b"),
                                      send_to_client,
                                      write_to_client,
                                      write_to_client_data,
                                      get_reports_data->alert_id,
                                      "prognostic",
                                      get_reports_data->host,
                                      pos,
                                      get_reports_data->host_search_phrase,
                                      get_reports_data->host_levels,
                                      get_reports_data->host_first_result,
                                      get_reports_data->host_max_results,
                                      NULL,
                                      get_reports_data->timezone);

            if (ret)
              {
                internal_error_send_to_client (error);
                get_reports_data_reset (get_reports_data);
                set_client_state (CLIENT_AUTHENTIC);
                return;
              }

            SEND_TO_CLIENT_OR_FAIL ("</report>"
                                    "</get_reports_response>");

            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        /* The usual scan report. */

        if (get_reports_data->get.id)
          {
            /* Showing requested report, use Results Filter setting. */
            INIT_GET (report, Result);
          }
        else
          {
            /* Showing multiple reports.  Use Report Filter setting.  Expand
             * INIT_GET here to pass the get_reports_data->report_get to
             * init_get. */
            ret = init_get ("get_reports",
                            &get_reports_data->report_get,
                            "Reports",
                            &first);
            if (ret)
              {
                switch (ret)
                  {
                    case 99:
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_ERROR_SYNTAX ("get_reports",
                                          "Permission denied"));
                      break;
                    default:
                      internal_error_send_to_client (error);
                      return;
                  }
                get_reports_data_reset (get_reports_data);
                set_client_state (CLIENT_AUTHENTIC);
                break;
              }
          }

        if ((get_reports_data->report_get.id == NULL)
            || (strlen (get_reports_data->report_get.id) == 0))
          {
            gchar *overrides, *min_qod, *filter;
            get_data_t * get;

            /* For simplicity, use a fixed result filter when filtering
             * reports.  A given result filter is only applied when getting
             * a single specified report. */

            get = &get_reports_data->report_get;

            /* Get overrides value from report filter. */
            if (get->filt_id && strcmp (get->filt_id, "0"))
              {
                filter = filter_term (get->filt_id);
                if (filter == NULL)
                  assert (0);
              }
            else
              filter = NULL;
            g_free (get_reports_data->get.filter);
            overrides
             = filter_term_value (filter ? filter : get->filter,
                                  "apply_overrides");
            min_qod
             = filter_term_value (filter ? filter : get->filter,
                                  "min_qod");
            g_free (filter);

            /* Setup result filter from overrides. */
            get_reports_data->get.filter
             = g_strdup_printf ("apply_overrides=%i min_qod=%s",
                                overrides && strcmp (overrides, "0"),
                                min_qod ? min_qod
                                        : G_STRINGIFY (MIN_QOD_DEFAULT));
            g_free (overrides);
            g_free (min_qod);
          }

        ret = init_report_iterator (&reports, &get_reports_data->report_get);
        if (ret)
          {
            switch (ret)
              {
                case 1:
                  if (send_find_error_to_client ("get_reports", "report",
                                                 get_reports_data->get.id,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case 2:
                  if (send_find_error_to_client
                       ("get_reports", "filter",
                        get_reports_data->get.filt_id, omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case -1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("get_reports"));
                  break;
              }
            get_reports_data_reset (get_reports_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (get_reports_data->alert_id == NULL)
          SEND_GET_START ("report");
        while (next_report (&reports, &report))
          {
            gchar *extension, *content_type;
            int ret;
            GString *prefix;

            prefix = g_string_new ("");
            content_type = report_format_content_type (report_format);
            extension = report_format_extension (report_format);

            if (get_reports_data->alert_id == NULL)
              g_string_append_printf (prefix,
                                      "<report"
                                      " type=\"scan\""
                                      " id=\"%s\""
                                      " format_id=\"%s\""
                                      " extension=\"%s\""
                                      " content_type=\"%s\">",
                                      report_iterator_uuid (&reports),
                                      get_reports_data->format_id,
                                      extension,
                                      content_type);

            g_free (extension);
            g_free (content_type);

            if (get_reports_data->alert_id == NULL)
              {
                task_t task;

                /* Send the standard elements.  Should match send_get_common. */
                buffer_xml_append_printf
                  (prefix,
                   "<owner><name>%s</name></owner>"
                   "<name>%s</name>"
                   "<comment>%s</comment>"
                   "<creation_time>%s</creation_time>"
                   "<modification_time>"
                   "%s"
                   "</modification_time>"
                   "<writable>0</writable>"
                   "<in_use>0</in_use>",
                   get_iterator_owner_name (&reports)
                    ? get_iterator_owner_name (&reports)
                    : "",
                   get_iterator_name (&reports)
                    ? get_iterator_name (&reports)
                    : "",
                   get_iterator_comment (&reports)
                    ? get_iterator_comment (&reports)
                    : "",
                   get_iterator_creation_time (&reports)
                    ? get_iterator_creation_time (&reports)
                    : "",
                   get_iterator_modification_time (&reports)
                    ? get_iterator_modification_time (&reports)
                    : "");
                /* Send short task and report format info */
                report_task (report, &task);
                if (task)
                  {
                    gchar *report_task_uuid, *report_task_name;
                    task_uuid (task, &report_task_uuid);
                    report_task_name = task_name (task);

                    buffer_xml_append_printf
                      (prefix,
                       "<task id=\"%s\">"
                       "<name>%s</name>"
                       "</task>",
                       report_task_uuid,
                       report_task_name);

                    g_free (report_task_uuid);
                    g_free (report_task_name);
                  }

                 if (get_reports_data->format_id)
                   {
                     gchar *format_name = NULL;
                     format_name = report_format_name (report_format);

                     buffer_xml_append_printf
                       (prefix,
                        "<report_format id=\"%s\">"
                        "<name>%s</name>"
                        "</report_format>",
                        get_reports_data->format_id,
                        format_name ? format_name : "");
                     // g_free (report_format_name);
                   }

              }
            /* If there's just one report then cleanup the iterator early.  This
             * closes the iterator transaction, allowing manage_schedule to lock
             * the db during generation of large reports. */
            if (request_report)
              cleanup_iterator (&reports);

            ret = manage_send_report (report,
                                      delta_report,
                                      report_format,
                                      &get_reports_data->get,
                                      get_reports_data->sort_order,
                                      get_reports_data->sort_field,
                                      get_reports_data->result_hosts_only,
                                      get_reports_data->min_cvss_base,
                                      get_reports_data->min_qod,
                                      get_reports_data->levels,
                                      get_reports_data->delta_states,
                                      get_reports_data->apply_overrides,
                                      get_reports_data->search_phrase,
                                      get_reports_data->autofp,
                                      get_reports_data->notes,
                                      get_reports_data->notes_details,
                                      get_reports_data->overrides,
                                      get_reports_data->overrides_details,
                                      get_reports_data->first_result,
                                      get_reports_data->max_results,
                                      get_reports_data->ignore_pagination,
                                      /* Special case the XML report, bah. */
                                      strcmp
                                       (get_reports_data->format_id,
                                        "a994b278-1f62-11e1-96ac-406186ea4fc5")
                                      && strcmp
                                          (get_reports_data->format_id,
                                           "5057e5cc-b825-11e4"
                                           "-9d0e-28d24461215b"),
                                      send_to_client,
                                      write_to_client,
                                      write_to_client_data,
                                      get_reports_data->alert_id,
                                      get_reports_data->type,
                                      NULL, 0, NULL, NULL, 0, 0, prefix->str,
                                      get_reports_data->timezone);
            g_string_free (prefix, TRUE);
            if (ret)
              {
                if (get_reports_data->alert_id)
                  switch (ret)
                    {
                      case 0:
                        break;
                      case 1:
                        if (send_find_error_to_client
                             ("get_reports", "alert",
                              get_reports_data->alert_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        /* Close the connection with the client, as part of the
                         * response may have been sent before the error
                         * occurred. */
                        internal_error_send_to_client (error);
                        if (request_report == 0)
                          cleanup_iterator (&reports);
                        get_reports_data_reset (get_reports_data);
                        set_client_state (CLIENT_AUTHENTIC);
                        return;
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_reports", "filter",
                              get_reports_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        /* This error always occurs before anything is sent
                         * to the client, so the connection can stay up. */
                        if (request_report == 0)
                          cleanup_iterator (&reports);
                        get_reports_data_reset (get_reports_data);
                        set_client_state (CLIENT_AUTHENTIC);
                        return;
                        break;
                      case -2:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_ERROR_SYNTAX ("get_reports",
                                            "Failed to find report format for"
                                            " alert"));
                        if (request_report == 0)
                          cleanup_iterator (&reports);
                        get_reports_data_reset (get_reports_data);
                        set_client_state (CLIENT_AUTHENTIC);
                        return;
                        break;
                      case -3:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_reports"));
                        if (request_report == 0)
                          cleanup_iterator (&reports);
                        get_reports_data_reset (get_reports_data);
                        set_client_state (CLIENT_AUTHENTIC);
                        return;
                        break;
                      case -4:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_ERROR_SYNTAX ("get_reports",
                                            "Failed to find filter for"
                                            " alert"));
                        if (request_report == 0)
                          cleanup_iterator (&reports);
                        get_reports_data_reset (get_reports_data);
                        set_client_state (CLIENT_AUTHENTIC);
                        return;
                        break;
                      default:
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_reports"));
                        /* Close the connection with the client, as part of the
                         * response may have been sent before the error
                         * occurred. */
                        internal_error_send_to_client (error);
                        if (request_report == 0)
                          cleanup_iterator (&reports);
                        get_reports_data_reset (get_reports_data);
                        set_client_state (CLIENT_AUTHENTIC);
                        return;
                        break;
                    }
                else if (ret == 2)
                  {
                    if (send_find_error_to_client
                         ("get_reports", "filter",
                          get_reports_data->get.filt_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    /* This error always occurs before anything is sent
                     * to the client, so the connection can stay up. */
                    if (request_report == 0)
                      cleanup_iterator (&reports);
                    get_reports_data_reset (get_reports_data);
                    set_client_state (CLIENT_AUTHENTIC);
                    return;
                  }
                else
                  {
                    /* Close the connection with the client, as part of the
                     * response may have been sent before the error
                     * occurred. */
                    internal_error_send_to_client (error);
                    if (request_report == 0)
                      cleanup_iterator (&reports);
                    get_reports_data_reset (get_reports_data);
                    set_client_state (CLIENT_AUTHENTIC);
                    return;
                  }
              }
            if (get_reports_data->alert_id == NULL)
              SEND_TO_CLIENT_OR_FAIL ("</report>");

            count++;

            if (request_report)
              /* Just to be safe, because iterator has been freed. */
              break;
          }
        if (request_report == 0)
          cleanup_iterator (&reports);

        if (get_reports_data->alert_id)
          SEND_TO_CLIENT_OR_FAIL (XML_OK ("get_reports"));
        else
          {
            filtered = get_reports_data->get.id
                        ? 1
                        : report_count (&get_reports_data->report_get);
            SEND_GET_END ("report", &get_reports_data->report_get, count,
                          filtered);
          }

        get_reports_data_reset (get_reports_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_GET_REPORT_FORMATS:
        {
          assert (strcasecmp ("GET_REPORT_FORMATS", element_name) == 0);

          if (get_report_formats_data->params &&
              get_report_formats_data->get.trash)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_report_formats",
                                "GET_REPORT_FORMATS params given with trash"));
          else
            {
              iterator_t report_formats;
              int count, filtered, ret, first;

              INIT_GET (report_format, Report Format);

              ret = init_report_format_iterator (&report_formats,
                                                 &get_report_formats_data->get);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client ("get_report_formats",
                                                       "report_format",
                                                       get_report_formats_data->get.id,
                                                       omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_report_formats", "filter",
                              get_report_formats_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_report_formats"));
                        break;
                    }
                  get_report_formats_data_reset (get_report_formats_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("report_format");
              while (1)
                {
                  time_t trust_time;

                  ret = get_next (&report_formats,
                                  &get_report_formats_data->get, &first, &count,
                                  init_report_format_iterator);
                  if (ret == 1)
                    break;
                  if (ret == -1)
                    {
                      internal_error_send_to_client (error);
                      return;
                    }

                  SEND_GET_COMMON (report_format,
                                   &get_report_formats_data->get,
                                   &report_formats);

                  trust_time = report_format_iterator_trust_time
                                (&report_formats);

                  SENDF_TO_CLIENT_OR_FAIL
                   ("<extension>%s</extension>"
                    "<content_type>%s</content_type>"
                    "<summary>%s</summary>"
                    "<description>%s</description>"
                    "<global>%i</global>"
                    "<predefined>%i</predefined>",
                    report_format_iterator_extension (&report_formats),
                    report_format_iterator_content_type (&report_formats),
                    report_format_iterator_summary (&report_formats),
                    report_format_iterator_description (&report_formats),
                    get_report_formats_data->get.trash
                      ? trash_report_format_global
                         (get_iterator_resource (&report_formats))
                      : report_format_global
                         (get_iterator_resource (&report_formats)),
                    get_report_formats_data->get.trash
                      ? 0
                      : report_format_predefined
                         (get_iterator_resource (&report_formats)));

                  if (get_report_formats_data->alerts)
                    {
                      iterator_t alerts;

                      SEND_TO_CLIENT_OR_FAIL ("<alerts>");
                      init_report_format_alert_iterator (&alerts,
                                                  get_iterator_resource
                                                   (&report_formats));
                      while (next (&alerts))
                        {
                          SENDF_TO_CLIENT_OR_FAIL
                           ("<alert id=\"%s\">"
                            "<name>%s</name>",
                            report_format_alert_iterator_uuid (&alerts),
                            report_format_alert_iterator_name (&alerts));
                          if (report_format_alert_iterator_readable (&alerts))
                            SEND_TO_CLIENT_OR_FAIL ("</alert>");
                          else
                            SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                    "</alert>");
                        }
                      cleanup_iterator (&alerts);
                      SEND_TO_CLIENT_OR_FAIL ("</alerts>");
                    }

                  if (get_report_formats_data->params
                      || get_report_formats_data->get.details)
                    {
                      iterator_t params;
                      init_report_format_param_iterator
                       (&params, get_iterator_resource (&report_formats),
                        get_report_formats_data->get.trash, 1, NULL);
                      while (next (&params))
                        {
                          long long int min, max;
                          iterator_t options;

                          SENDF_TO_CLIENT_OR_FAIL
                           ("<param>"
                            "<name>%s</name>"
                            "<type>%s",
                            report_format_param_iterator_name (&params),
                            report_format_param_iterator_type_name (&params));

                          min = report_format_param_iterator_type_min (&params);
                          if (min > LLONG_MIN)
                            SENDF_TO_CLIENT_OR_FAIL ("<min>%lli</min>", min);

                          max = report_format_param_iterator_type_max (&params);
                          if (max < LLONG_MAX)
                            SENDF_TO_CLIENT_OR_FAIL ("<max>%lli</max>", max);

                          SENDF_TO_CLIENT_OR_FAIL
                           ("</type>"
                            "<value>%s</value>"
                            "<default>%s</default>",
                            report_format_param_iterator_value (&params),
                            report_format_param_iterator_fallback (&params));

                          if (report_format_param_iterator_type (&params)
                              == REPORT_FORMAT_PARAM_TYPE_SELECTION)
                            {
                              SEND_TO_CLIENT_OR_FAIL ("<options>");
                              init_param_option_iterator
                               (&options,
                                report_format_param_iterator_param
                                 (&params),
                                1,
                                NULL);
                              while (next (&options))
                                SENDF_TO_CLIENT_OR_FAIL
                                 ("<option>%s</option>",
                                  param_option_iterator_value (&options));
                              cleanup_iterator (&options);
                              SEND_TO_CLIENT_OR_FAIL ("</options>");
                            }

                          SEND_TO_CLIENT_OR_FAIL ("</param>");
                        }
                      cleanup_iterator (&params);
                    }

                  if (get_report_formats_data->get.details)
                    {
                      file_iterator_t files;
                      if (init_report_format_file_iterator
                           (&files, get_iterator_resource (&report_formats)))
                        {
                          cleanup_iterator (&report_formats);
                          error_send_to_client (error);
                          return;
                        }
                      while (next_file (&files))
                        {
                          gchar *content = file_iterator_content_64 (&files);
                          SENDF_TO_CLIENT_OR_FAIL
                           ("<file name=\"%s\">%s</file>",
                            file_iterator_name (&files),
                            content);
                          g_free (content);
                        }
                      cleanup_file_iterator (&files);

                      SENDF_TO_CLIENT_OR_FAIL
                       ("<signature>%s</signature>",
                        report_format_iterator_signature (&report_formats));
                    }

                  SENDF_TO_CLIENT_OR_FAIL
                   ("<trust>%s<time>%s</time></trust>"
                    "<active>%i</active>",
                    report_format_predefined (get_iterator_resource
                                               (&report_formats))
                     ? "yes"
                     : report_format_iterator_trust (&report_formats),
                    iso_time (&trust_time),
                    report_format_iterator_active (&report_formats));

                  SEND_TO_CLIENT_OR_FAIL ("</report_format>");
                  count++;
                }
              cleanup_iterator (&report_formats);
              filtered = get_report_formats_data->get.id
                          ? 1
                          : report_format_count (&get_report_formats_data->get);
              SEND_GET_END ("report_format", &get_report_formats_data->get,
                            count, filtered);
            }
          get_report_formats_data_reset (get_report_formats_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_RESULTS:
        {
          result_t result = 0;
          task_t task = 0;

          assert (strcasecmp ("GET_RESULTS", element_name) == 0);

          if (user_may ("get_results") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_results",
                                  "Permission denied"));
              get_results_data_reset (get_results_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (current_credentials.username == NULL)
            {
              get_results_data_reset (get_results_data);
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_results"));
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (get_results_data->notes
              && (get_results_data->task_id == NULL)
              && (get_results_data->get.details == 0))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_results",
                                "GET_RESULTS must have a task_id attribute or"
                                " details set to true if the"
                                " notes attribute is true"));
          else if ((get_results_data->overrides
                    || get_results_data->apply_overrides)
                   && (get_results_data->task_id == NULL)
                   && (get_results_data->get.details == 0))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_results",
                                "GET_RESULTS must have a task_id attribute or"
                                " details set to true if the apply_overides or "
                                " overrides attributes are true"));
          else if (get_results_data->get.id
                   && find_result_with_permission (get_results_data->get.id,
                                                   &result,
                                                   NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_results"));
          else if (get_results_data->get.id && result == 0)
            {
              if (send_find_error_to_client ("get_results", "result",
                                             get_results_data->get.id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (get_results_data->task_id
                   && find_task_with_permission (get_results_data->task_id,
                                                 &task,
                                                 NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("get_results"));
          else if (get_results_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("get_results", "task",
                                             get_results_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else
            {
              iterator_t results;
              gchar *min_qod_str;
              int max;
              int autofp, apply_overrides, min_qod, dynamic_severity;

              if (get_results_data->autofp)
                autofp = get_results_data->autofp;
              else if (filter_term_value (get_results_data->get.filter,
                                          "autofp"))
                autofp = atoi (filter_term_value (get_results_data->get.filter,
                                                  "autofp"));
              else
                autofp = 0;

              if (autofp < 0 || autofp > 2)
                SEND_TO_CLIENT_OR_FAIL
                (XML_ERROR_SYNTAX ("get_results",
                                    "autofp in GET_RESULTS must be either"
                                    " 0, 1 or 2"));

              if (get_results_data->apply_overrides_set)
                apply_overrides = get_results_data->apply_overrides;
              else if (filter_term_value (get_results_data->get.filter,
                                          "apply_overrides"))
                apply_overrides
                  = atoi (filter_term_value (get_results_data->get.filter,
                                             "apply_overrides"))
                    ? 1 : 0;
              else
                apply_overrides = 1;

              min_qod_str = filter_term_value (get_results_data->get.filter,
                                               "min_qod");
              if (min_qod_str == NULL
                  || sscanf (min_qod_str, "%d", &min_qod) != 1)
                min_qod = MIN_QOD_DEFAULT;
              g_free (min_qod_str);

              dynamic_severity = setting_dynamic_severity_int ();

              SEND_TO_CLIENT_OR_FAIL ("<get_results_response"
                                      " status=\"" STATUS_OK "\""
                                      " status_text=\"" STATUS_OK_TEXT "\">");
              INIT_GET (result, Result);


              init_result_get_iterator (&results, &get_results_data->get,
                                        autofp,
                                        apply_overrides,
                                        min_qod,
                                        dynamic_severity);

              if (next (&results))
                {
                  if (get_results_data->get.id && (task == 0))
                    {
                      char *task_id;
                      task_uuid (result_iterator_task (&results), &task_id);
                      if (find_task_with_permission (task_id, &task, NULL))
                        {
                          free (task_id);
                          internal_error_send_to_client (error);
                          cleanup_iterator (&results);
                          return;
                        }
                      free (task_id);
                    }

                  count = 0;
                  do
                    {
                      GString *buffer = g_string_new ("");
                      buffer_results_xml (buffer,
                                          &results,
                                          task,
                                          get_results_data->notes,
                                          get_results_data->notes_details,
                                          get_results_data->overrides,
                                          get_results_data->overrides_details,
                                          1,
                                          /* show tag details if selected by ID */
                                          get_results_data->get.id != NULL,
                                          get_results_data->get.details,
                                          NULL,
                                          NULL,
                                          0);
                      SEND_TO_CLIENT_OR_FAIL (buffer->str);
                      g_string_free (buffer, TRUE);
                      count ++;
                    }
                  while (next (&results));
                }
              cleanup_iterator (&results);

              manage_filter_controls (get_results_data->get.filter,
                                      &first, &max, NULL, NULL);

              filtered = get_results_data->get.id
                          ? 1 : result_count (&get_results_data->get,
                                              autofp, apply_overrides,
                                              min_qod,
                                              dynamic_severity);

              SEND_GET_END("result", &get_results_data->get, count, filtered);
            }

          get_results_data_reset (get_results_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_ROLES:
        {
          iterator_t roles;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_ROLES", element_name) == 0);

          INIT_GET (role, Role);

          ret = init_role_iterator (&roles, &get_roles_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_roles", "role",
                                                   get_roles_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_roles", "role", get_roles_data->get.filt_id,
                          omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_roles"));
                    break;
                }
              get_roles_data_reset (get_roles_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("role");
          while (1)
            {
              gchar *users;

              ret = get_next (&roles, &get_roles_data->get, &first, &count,
                              init_role_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (role, &get_roles_data->get, &roles);

              users = role_users (get_iterator_resource (&roles));
              SENDF_TO_CLIENT_OR_FAIL ("<users>%s</users>", users ? users : "");
              g_free (users);

              SEND_TO_CLIENT_OR_FAIL ("</role>");

              count++;
            }
          cleanup_iterator (&roles);
          filtered = get_roles_data->get.id
                      ? 1
                      : role_count (&get_roles_data->get);
          SEND_GET_END ("role", &get_roles_data->get, count, filtered);

          get_roles_data_reset (get_roles_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_SCANNERS:
        assert (strcasecmp ("GET_SCANNERS", element_name) == 0);
        return handle_get_scanners (omp_parser, error);

      case CLIENT_GET_SCHEDULES:
        {
          assert (strcasecmp ("GET_SCHEDULES", element_name) == 0);

          if (get_schedules_data->tasks && get_schedules_data->get.trash)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_schedules",
                                "GET_SCHEDULES tasks given with trash"));
          else
            {
              iterator_t schedules;
              int count, filtered, ret, first;

              INIT_GET (schedule, Schedule);

              ret = init_schedule_iterator (&schedules, &get_schedules_data->get);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client ("get_schedules",
                                                       "schedule",
                                                       get_schedules_data->get.id,
                                                       omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_schedules", "filter",
                              get_schedules_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_schedules"));
                        break;
                    }
                  get_schedules_data_reset (get_schedules_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("schedule");
              while (1)
                {
                  time_t first_time, next_time;
                  gchar *iso;
                  const char *timezone, *abbrev;
                  char *simple_period_unit, *simple_duration_unit;
                  int period, period_minutes, period_hours, period_days;
                  int period_weeks, period_months, duration, duration_minutes;
                  int duration_hours, duration_days, duration_weeks;
                  int simple_period, simple_duration;

                  ret = get_next (&schedules, &get_schedules_data->get, &first,
                                  &count, init_schedule_iterator);
                  if (ret == 1)
                    break;
                  if (ret == -1)
                    {
                      internal_error_send_to_client (error);
                      return;
                    }

                  SEND_GET_COMMON (schedule, &get_schedules_data->get, &schedules);

                  timezone = schedule_iterator_timezone (&schedules);
                  first_time = schedule_iterator_first_time (&schedules);
                  next_time = schedule_iterator_next_time (&schedules);

                  first_time += schedule_iterator_initial_offset (&schedules)
                                 - time_offset (timezone, first_time);
                  if (next_time)
                    next_time += schedule_iterator_initial_offset (&schedules)
                                   - time_offset (timezone, next_time);

                  /* Duplicate static string because there's an iso_time_tz below. */
                  abbrev = NULL;
                  iso = g_strdup (iso_time_tz (&first_time, timezone, &abbrev));

                  period = schedule_iterator_period (&schedules);
                  if (period)
                    {
                      period_minutes = period / 60;
                      period_hours = period_minutes / 60;
                      period_days = period_hours / 24;
                      period_weeks = period_days / 7;
                    }
                  simple_period_unit = "";
                  if (period == 0)
                    simple_period = 0;
                  else if (period_weeks && (period % (60 * 60 * 24 * 7) == 0))
                    {
                      simple_period = period_weeks;
                      simple_period_unit = "week";
                    }
                  else if (period_days && (period % (60 * 60 * 24) == 0))
                    {
                      simple_period = period_days;
                      simple_period_unit = "day";
                    }
                  else if (period_hours && (period % (60 * 60) == 0))
                    {
                      simple_period = period_hours;
                      simple_period_unit = "hour";
                    }
                  /* The granularity of the "simple" GSA interface stops at hours. */
                  else
                    simple_period = 0;

                  period_months = schedule_iterator_period_months (&schedules);
                  if (period_months && (period_months < 25))
                    {
                      simple_period = period_months;
                      simple_period_unit = "month";
                    }

                  duration = schedule_iterator_duration (&schedules);
                  if (duration)
                    {
                      duration_minutes = duration / 60;
                      duration_hours = duration_minutes / 60;
                      duration_days = duration_hours / 24;
                      duration_weeks = duration_days / 7;
                    }
                  simple_duration_unit = "";
                  if (duration == 0)
                    simple_duration = 0;
                  else if (duration_weeks
                           && (duration % (60 * 60 * 24 * 7) == 0))
                    {
                      simple_duration = duration_weeks;
                      simple_duration_unit = "week";
                    }
                  else if (duration_days
                           && (duration % (60 * 60 * 24) == 0))
                    {
                      simple_duration = duration_days;
                      simple_duration_unit = "day";
                    }
                  else if (duration_hours
                           && (duration % (60 * 60) == 0))
                    {
                      simple_duration = duration_hours;
                      simple_duration_unit = "hour";
                    }
                  /* The granularity of the "simple" GSA interface stops at hours. */
                  else
                    simple_duration = 0;

                  SENDF_TO_CLIENT_OR_FAIL
                   ("<first_time>%s</first_time>"
                    "<next_time>%s</next_time>"
                    "<period>%ld</period>"
                    "<period_months>%ld</period_months>"
                    "<simple_period>%i<unit>%s</unit></simple_period>"
                    "<duration>%ld</duration>"
                    "<simple_duration>%i<unit>%s</unit></simple_duration>"
                    "<timezone>%s</timezone>"
                    "<timezone_abbrev>%s</timezone_abbrev>",
                    iso,
                    (next_time == 0 ? "over" : iso_time_tz (&next_time, timezone, NULL)),
                    schedule_iterator_period (&schedules),
                    schedule_iterator_period_months (&schedules),
                    simple_period,
                    simple_period_unit,
                    schedule_iterator_duration (&schedules),
                    simple_duration,
                    simple_duration_unit,
                    schedule_iterator_timezone (&schedules)
                     ? schedule_iterator_timezone (&schedules)
                     : "UTC",
                    abbrev ? abbrev : "UTC");

                  g_free (iso);
                  if (get_schedules_data->tasks)
                    {
                      iterator_t tasks;

                      SEND_TO_CLIENT_OR_FAIL ("<tasks>");
                      init_schedule_task_iterator (&tasks,
                                                   get_iterator_resource
                                                    (&schedules));
                      while (next (&tasks))
                        {
                          SENDF_TO_CLIENT_OR_FAIL ("<task id=\"%s\">"
                                                   "<name>%s</name>",
                                                   schedule_task_iterator_uuid (&tasks),
                                                   schedule_task_iterator_name (&tasks));
                          if (schedule_task_iterator_readable (&tasks))
                            SEND_TO_CLIENT_OR_FAIL ("</task>");
                          else
                            SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                    "</task>");
                        }
                      cleanup_iterator (&tasks);
                      SEND_TO_CLIENT_OR_FAIL ("</tasks>");
                    }
                  SEND_TO_CLIENT_OR_FAIL ("</schedule>");
                  count++;
                }
              cleanup_iterator (&schedules);
              filtered = get_schedules_data->get.id
                          ? 1
                          : schedule_count (&get_schedules_data->get);
              SEND_GET_END ("schedule", &get_schedules_data->get, count, filtered);
            }
          get_schedules_data_reset (get_schedules_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_SETTINGS:
        {
          setting_t setting = 0;
          iterator_t settings;
          int count, filtered;

          assert (strcasecmp ("GET_SETTINGS", element_name) == 0);

          if (user_may ("get_settings") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_settings",
                                  "Permission denied"));
              get_settings_data_reset (get_settings_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          init_setting_iterator (&settings,
                                 get_settings_data->setting_id,
                                 get_settings_data->filter,
                                 get_settings_data->first,
                                 get_settings_data->max,
                                 get_settings_data->sort_order,
                                 get_settings_data->sort_field);

          SEND_TO_CLIENT_OR_FAIL ("<get_settings_response"
                                  " status=\"" STATUS_OK "\""
                                  " status_text=\"" STATUS_OK_TEXT "\">");
          SENDF_TO_CLIENT_OR_FAIL ("<filters>"
                                   "<term>%s</term>"
                                   "</filters>"
                                   "<settings start=\"%i\" max=\"%i\"/>",
                                   get_settings_data->filter
                                    ? get_settings_data->filter
                                    : "",
                                   /* Add 1 for 1 indexing. */
                                   get_settings_data->first + 1,
                                   get_settings_data->max);
          count = 0;
          while (next (&settings))
            {
              SENDF_TO_CLIENT_OR_FAIL ("<setting id=\"%s\">"
                                       "<name>%s</name>"
                                       "<comment>%s</comment>"
                                       "<value>%s</value>"
                                       "</setting>",
                                       setting_iterator_uuid (&settings),
                                       setting_iterator_name (&settings),
                                       setting_iterator_comment (&settings),
                                       setting_iterator_value (&settings));
              count++;
            }
          filtered = setting
                      ? 1
                      : setting_count (get_settings_data->filter);
          SENDF_TO_CLIENT_OR_FAIL ("<setting_count>"
                                   "<filtered>%i</filtered>"
                                   "<page>%i</page>"
                                   "</setting_count>",
                                   filtered,
                                   count);
          cleanup_iterator (&settings);
          SEND_TO_CLIENT_OR_FAIL ("</get_settings_response>");

          get_settings_data_reset (get_settings_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_SLAVES:
        {

          assert (strcasecmp ("GET_SLAVES", element_name) == 0);

          if (get_slaves_data->tasks && get_slaves_data->get.trash)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_slaves",
                                "GET_SLAVES tasks given with trash"));
          else
            {
              iterator_t slaves;
              int count, filtered, ret, first;

              INIT_GET (slave, Slave);

              ret = init_slave_iterator (&slaves, &get_slaves_data->get);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client ("get_slaves",
                                                       "slave",
                                                       get_slaves_data->get.id,
                                                       omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_slaves", "filter",
                              get_slaves_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_slaves"));
                        break;
                    }
                  get_slaves_data_reset (get_slaves_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("slave");
              while (1)
                {

                  ret = get_next (&slaves, &get_slaves_data->get, &first,
                                  &count, init_slave_iterator);
                  if (ret == 1)
                    break;
                  if (ret == -1)
                    {
                      internal_error_send_to_client (error);
                      return;
                    }

                  SEND_GET_COMMON (slave, &get_slaves_data->get, &slaves);

                  SENDF_TO_CLIENT_OR_FAIL ("<host>%s</host>"
                                           "<port>%s</port>"
                                           "<login>%s</login>",
                                           slave_iterator_host (&slaves),
                                           slave_iterator_port (&slaves),
                                           slave_iterator_login (&slaves));

                  if (get_slaves_data->tasks)
                    {
                      iterator_t tasks;

                      SEND_TO_CLIENT_OR_FAIL ("<tasks>");
                      init_slave_task_iterator (&tasks,
                                                get_iterator_resource
                                                 (&slaves));
                      while (next (&tasks))
                        {
                          SENDF_TO_CLIENT_OR_FAIL ("<task id=\"%s\">"
                                                   "<name>%s</name>",
                                                   slave_task_iterator_uuid
                                                    (&tasks),
                                                   slave_task_iterator_name
                                                    (&tasks));
                          if (slave_task_iterator_readable (&tasks))
                            SEND_TO_CLIENT_OR_FAIL ("</task>");
                          else
                            SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                    "</task>");
                        }
                      cleanup_iterator (&tasks);
                      SEND_TO_CLIENT_OR_FAIL ("</tasks>");
                    }

                  SEND_TO_CLIENT_OR_FAIL ("</slave>");
                  count++;
                }
              cleanup_iterator (&slaves);
              filtered = get_slaves_data->get.id
                          ? 1
                          : slave_count (&get_slaves_data->get);
              SEND_GET_END ("slave", &get_slaves_data->get, count, filtered);
            }
          get_slaves_data_reset (get_slaves_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_AGGREGATES:
        {
          assert (strcasecmp ("GET_AGGREGATES", element_name) == 0);

          iterator_t aggregate;
          const char *type;
          get_data_t *get;
          const char *data_column, *group_column;
          const char *data_column_type, *group_column_type;
          int ret;
          GString *xml;
          double c_sum;
          long c_count;
          gchar *sort_field, *filter;
          int first, max, sort_order;
          GString *type_many;

          type = get_aggregates_data->type;
          if (type == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL
                  (XML_ERROR_SYNTAX ("get_aggregates",
                                      "GET_AGGREGATES requires a"
                                      " 'type' attribute"));
              return;
            }

          get = &get_aggregates_data->get;
          INIT_GET (aggregate, Aggregate);
          data_column = get_aggregates_data->data_column;
          group_column = get_aggregates_data->group_column;

          if (group_column == NULL)
            group_column_type = "";
          else if (strcmp (group_column, "severity") == 0)
            group_column_type = "cvss";
          else if (strcmp (group_column, "created") == 0
                   || strcmp (group_column, "modified") == 0)
            group_column_type = "unix_time";
          else
            group_column_type = "text";

          if (data_column == NULL)
            data_column_type = "";
          else if (strcmp (data_column, "severity") == 0)
            data_column_type = "cvss";
          else if (strcmp (data_column, "created") == 0
                   || strcmp (data_column, "modified") == 0)
            data_column_type = "unix_time";
          else
            data_column_type = "decimal";

          if (data_column == NULL && group_column == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL
                  (XML_ERROR_SYNTAX ("get_aggregates",
                                      "GET_AGGREGATES requires at least one of"
                                      " the attributes 'data_column'"
                                      " and 'group_column'"));
              return;
            }

          ret = init_aggregate_iterator (&aggregate, type, get,
                                         0 /* distinct */,
                                         data_column, group_column,
                                         NULL /* extra_tables */,
                                         NULL /* extra_where */);

          switch (ret)
            {
              case 0:
                break;
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_aggregates",
                                    "Failed to find resource"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_aggregates",
                                    "Failed to find filter"));
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_aggregates",
                                    "Invalid data_column"));
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_aggregates",
                                    "Invalid group_column"));
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                    (XML_ERROR_SYNTAX ("get_aggregates",
                                        "Invalid resource type"));
                break;
              case 6:
                SEND_TO_CLIENT_OR_FAIL
                    (XML_ERROR_SYNTAX ("get_aggregates",
                                        "Trashcan not used by resource type"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_aggregates",
                                    "Permission denied"));
                break;
              default:
                assert (0);
                /*@fallthrough@*/
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("get_aggregates"));
                break;
            }

          if (ret)
            return;

          xml = g_string_new ("<get_aggregates_response"
                              "  status_text=\"" STATUS_OK_TEXT "\""
                              "  status=\"" STATUS_OK "\">");

          g_string_append (xml, "<aggregate>");
          if (type)
            g_string_append_printf (xml,
                                    "<data_type>%s</data_type>",
                                    type);
          if (data_column)
            g_string_append_printf (xml,
                                    "<data_column>%s</data_column>",
                                    data_column);
          if (group_column)
            g_string_append_printf (xml,
                                    "<group_column>%s</group_column>",
                                    group_column);

          c_sum = 0.0;
          c_count = 0L;
          while (next (&aggregate))
            {
              gchar *value_escaped;

              if (data_column || group_column == NULL)
                c_sum += aggregate_iterator_sum (&aggregate);
              c_count += aggregate_iterator_count (&aggregate);

              if (group_column)
                value_escaped
                  = g_markup_escape_text (aggregate_iterator_value (&aggregate),
                                          -1);
              else
                value_escaped = NULL;

              if (group_column && data_column)
                g_string_append_printf (xml,
                                        "<group>"
                                        "<value>%s</value>"
                                        "<count>%d</count>"
                                        "<c_count>%ld</c_count>"
                                        "<min>%g</min>"
                                        "<max>%g</max>"
                                        "<mean>%g</mean>"
                                        "<sum>%g</sum>"
                                        "<c_sum>%g</c_sum>"
                                        "</group>",
                                        value_escaped,
                                        aggregate_iterator_count (&aggregate),
                                        c_count,
                                        aggregate_iterator_min (&aggregate),
                                        aggregate_iterator_max (&aggregate),
                                        aggregate_iterator_mean (&aggregate),
                                        aggregate_iterator_sum (&aggregate),
                                        c_sum);
              else if (group_column)
                g_string_append_printf (xml,
                                        "<group>"
                                        "<value>%s</value>"
                                        "<count>%d</count>"
                                        "<c_count>%ld</c_count>"
                                        "</group>",
                                        value_escaped,
                                        aggregate_iterator_count (&aggregate),
                                        c_count);
              else
                g_string_append_printf (xml,
                                        "<overall>"
                                        "<count>%d</count>"
                                        "<c_count>%ld</c_count>"
                                        "<min>%g</min>"
                                        "<max>%g</max>"
                                        "<mean>%g</mean>"
                                        "<sum>%g</sum>"
                                        "<c_sum>%g</c_sum>"
                                        "</overall>",
                                        aggregate_iterator_count (&aggregate),
                                        c_count,
                                        aggregate_iterator_min (&aggregate),
                                        aggregate_iterator_max (&aggregate),
                                        aggregate_iterator_mean (&aggregate),
                                        aggregate_iterator_sum (&aggregate),
                                        c_sum
                                       );

              g_free (value_escaped);
            }

          g_string_append (xml, "<column_info>");

          if (group_column)
            {
              g_string_append_printf (xml,
                                      "<aggregate_column>"
                                      "<name>value</name>"
                                      "<stat>value</stat>"
                                      "<type>%s</type>"
                                      "<column>%s</column>"
                                      "<data_type>%s</data_type>"
                                      "</aggregate_column>",
                                      type,
                                      group_column,
                                      group_column_type);
            }

          g_string_append_printf (xml,
                                  "<aggregate_column>"
                                  "<name>count</name>"
                                  "<stat>count</stat>"
                                  "<type>%s</type>"
                                  "<column></column>"
                                  "<data_type>integer</data_type>"
                                  "</aggregate_column>",
                                  type);

          g_string_append_printf (xml,
                                  "<aggregate_column>"
                                  "<name>c_count</name>"
                                  "<stat>c_count</stat>"
                                  "<type>%s</type>"
                                  "<column></column>"
                                  "<data_type>integer</data_type>"
                                  "</aggregate_column>",
                                  type);

          if (data_column)
            {
              g_string_append_printf (xml,
                                      "<aggregate_column>"
                                      "<name>min</name>"
                                      "<stat>min</stat>"
                                      "<type>%s</type>"
                                      "<column>%s</column>"
                                      "<data_type>%s</data_type>"
                                      "</aggregate_column>",
                                      type,
                                      data_column,
                                      data_column_type);
              g_string_append_printf (xml,
                                      "<aggregate_column>"
                                      "<name>max</name>"
                                      "<stat>max</stat>"
                                      "<type>%s</type>"
                                      "<column>%s</column>"
                                      "<data_type>%s</data_type>"
                                      "</aggregate_column>",
                                      type,
                                      data_column,
                                      data_column_type);
              g_string_append_printf (xml,
                                      "<aggregate_column>"
                                      "<name>mean</name>"
                                      "<stat>mean</stat>"
                                      "<type>%s</type>"
                                      "<column>%s</column>"
                                      "<data_type>%s</data_type>"
                                      "</aggregate_column>",
                                      type,
                                      data_column,
                                      data_column_type);
              g_string_append_printf (xml,
                                      "<aggregate_column>"
                                      "<name>sum</name>"
                                      "<stat>sum</stat>"
                                      "<type>%s</type>"
                                      "<column>%s</column>"
                                      "<data_type>%s</data_type>"
                                      "</aggregate_column>",
                                      type,
                                      data_column,
                                      data_column_type);
              g_string_append_printf (xml,
                                      "<aggregate_column>"
                                      "<name>c_sum</name>"
                                      "<stat>c_sum</stat>"
                                      "<type>%s</type>"
                                      "<column>%s</column>"
                                      "<data_type>%s</data_type>"
                                      "</aggregate_column>",
                                      type,
                                      data_column,
                                      data_column_type);
            }

          g_string_append (xml, "</column_info>");

          if (get->filt_id && strcmp (get->filt_id, "0"))
            {
              if (get->filter_replacement)
                filter = g_strdup (get->filter_replacement);
              else
                filter = filter_term (get->filt_id);
              if (filter == NULL)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_aggregates",
                                    "Failed to find filter"));
            }
          else
            filter = NULL;

          manage_filter_controls (filter ? filter : get->filter,
                                  &first, &max, &sort_field, &sort_order);

          if (filter || get->filter)
            {
              gchar *new_filter;
              new_filter = manage_clean_filter (filter ? filter : get->filter);
              g_free (filter);
              if ((strcmp (type, "task") == 0)
                  && (filter_term_value (new_filter, "apply_overrides")
                      == NULL))
                {
                  filter = new_filter;
                  new_filter = g_strdup_printf ("apply_overrides=%i %s",
                                                APPLY_OVERRIDES_DEFAULT,
                                                filter);
                  g_free (filter);
                }
              filter = new_filter;
            }
          else
            filter = manage_clean_filter ("");

          type_many = g_string_new (type);

          if (strcmp (type, "info") != 0)
            g_string_append (type_many, "s");

          g_string_append (xml, "</aggregate>");

          buffer_get_filter_xml (xml, type, get, filter);

          g_string_append (xml, "</get_aggregates_response>");


          SEND_TO_CLIENT_OR_FAIL (xml->str);

          cleanup_iterator (&aggregate);
          g_string_free (xml, TRUE);
          get_aggregates_data_reset (get_aggregates_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_SYSTEM_REPORTS:
        {
          int ret;
          report_type_iterator_t types;

          assert (strcasecmp ("GET_SYSTEM_REPORTS", element_name) == 0);

          ret = init_system_report_type_iterator
                 (&types,
                  get_system_reports_data->name,
                  get_system_reports_data->slave_id);
          switch (ret)
            {
              case 1:
                if (send_find_error_to_client ("get_system_reports",
                                               "system report",
                                               get_system_reports_data->name,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 2:
                if (send_find_error_to_client
                     ("get_system_reports", "slave",
                      get_system_reports_data->slave_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("get_system_reports",
                                    "Permission denied"));
                break;
              default:
                assert (0);
                /*@fallthrough@*/
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("get_system_reports"));
                break;
              case 0:
              case 3:
                {
                  int report_ret;
                  char *report;
                  SEND_TO_CLIENT_OR_FAIL ("<get_system_reports_response"
                                          " status=\"" STATUS_OK "\""
                                          " status_text=\""
                                          STATUS_OK_TEXT
                                          "\">");
                  while (next_report_type (&types))
                    if (get_system_reports_data->brief
                        && (ret != 3))
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<system_report>"
                        "<name>%s</name>"
                        "<title>%s</title>"
                        "</system_report>",
                        report_type_iterator_name (&types),
                        report_type_iterator_title (&types));
                    else if ((report_ret = manage_system_report
                                            (report_type_iterator_name (&types),
                                             get_system_reports_data->duration,
                                             get_system_reports_data->slave_id,
                                             &report))
                             && (report_ret != 3))
                      {
                        cleanup_report_type_iterator (&types);
                        internal_error_send_to_client (error);
                        return;
                      }
                    else if (report)
                      {
                        SENDF_TO_CLIENT_OR_FAIL
                         ("<system_report>"
                          "<name>%s</name>"
                          "<title>%s</title>"
                          "<report format=\"%s\" duration=\"%s\">"
                          "%s"
                          "</report>"
                          "</system_report>",
                          report_type_iterator_name (&types),
                          report_type_iterator_title (&types),
                          (ret == 3 ? "txt" : "png"),
                          get_system_reports_data->duration
                           ? get_system_reports_data->duration
                           : "86400",
                          report);
                        free (report);
                      }
                  cleanup_report_type_iterator (&types);
                  SEND_TO_CLIENT_OR_FAIL ("</get_system_reports_response>");
                }
            }

          get_system_reports_data_reset (get_system_reports_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_TAGS:
        {
          iterator_t tags;
          int ret, count, first, filtered;

          assert (strcasecmp ("GET_TAGS", element_name) == 0);

          INIT_GET (tag, Tag);

          if (get_tags_data->names_only)
            ret = init_tag_name_iterator (&tags, &get_tags_data->get);
          else
            ret = init_tag_iterator (&tags, &get_tags_data->get);

          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_tags",
                                                   "tag", get_tags_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                          ("get_tags", "filter", get_tags_data->get.filt_id,
                           omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  default:
                    SEND_TO_CLIENT_OR_FAIL
                      (XML_INTERNAL_ERROR ("get_tags"));
                }
              get_tags_data_reset (get_tags_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("tag");
          while (1)
            {
              ret = get_next (&tags, &get_tags_data->get, &first, &count,
                              get_tags_data->names_only
                               ? init_tag_name_iterator
                               : init_tag_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              if (get_tags_data->names_only)
                SENDF_TO_CLIENT_OR_FAIL ("<tag>"
                                         "<name>%s</name>"
                                         "</tag>",
                                         tag_name_iterator_name (&tags));
              else
                {
                  gchar* value;

                  value = g_markup_escape_text (tag_iterator_value (&tags), -1);

                  SEND_GET_COMMON (tag, &get_tags_data->get, &tags);

                  SENDF_TO_CLIENT_OR_FAIL ("<resource id=\"%s\">"
                                           "<type>%s</type>"
                                           "<name>%s</name>"
                                           "<trash>%d</trash>",
                                           tag_iterator_resource_uuid (&tags),
                                           tag_iterator_resource_type (&tags),
                                           tag_iterator_resource_name (&tags),
                                           tag_iterator_resource_location
                                            (&tags));

                  if (tag_iterator_resource_readable (&tags) == 0)
                    SENDF_TO_CLIENT_OR_FAIL ("<permissions/>");


                  SENDF_TO_CLIENT_OR_FAIL ("</resource>"
                                           "<value>%s</value>"
                                           "<active>%d</active>"
                                           "<orphan>%d</orphan>"
                                           "</tag>",
                                           value,
                                           tag_iterator_active (&tags),
                                           tag_iterator_orphan (&tags));
                  g_free (value);
                }
              count++;
            }
          cleanup_iterator (&tags);
          filtered = get_tags_data->get.id
                      ? 1
                      : tag_count (&get_tags_data->get);
          SEND_GET_END ("tag", &get_tags_data->get, count, filtered);

          get_tags_data_reset (get_tags_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_TARGETS:
        {
          assert (strcasecmp ("GET_TARGETS", element_name) == 0);

          if (get_targets_data->tasks && get_targets_data->get.trash)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("get_target",
                                "GET_TARGETS tasks given with trash"));
          else
            {
              iterator_t targets;
              int count, filtered, ret, first;

              INIT_GET (target, Target);

              ret = init_target_iterator (&targets, &get_targets_data->get);
              if (ret)
                {
                  switch (ret)
                    {
                      case 1:
                        if (send_find_error_to_client ("get_targets",
                                                       "target",
                                                       get_targets_data->get.id,
                                                       omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case 2:
                        if (send_find_error_to_client
                             ("get_targets", "filter",
                              get_targets_data->get.filt_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        break;
                      case -1:
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("get_targets"));
                        break;
                    }
                  get_targets_data_reset (get_targets_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  break;
                }

              SEND_GET_START ("target");
              while (1)
                {
                  char *ssh_lsc_name, *ssh_lsc_uuid, *smb_lsc_name, *smb_lsc_uuid;
                  char *esxi_lsc_name, *esxi_lsc_uuid;
                  const char *port_list_uuid, *port_list_name, *ssh_port;
                  const char *hosts, *exclude_hosts, *reverse_lookup_only;
                  const char *reverse_lookup_unify;
                  lsc_credential_t ssh_credential, smb_credential;
                  lsc_credential_t esxi_credential;
                  int port_list_trash, max_hosts, port_list_available;
                  int ssh_lsc_credential_available;
                  int smb_lsc_credential_available;
                  int esxi_lsc_credential_available;

                  ret = get_next (&targets, &get_targets_data->get, &first,
                                  &count, init_target_iterator);
                  if (ret == 1)
                    break;
                  if (ret == -1)
                    {
                      internal_error_send_to_client (error);
                      return;
                    }

                  ssh_credential = target_iterator_ssh_credential (&targets);
                  smb_credential = target_iterator_smb_credential (&targets);
                  esxi_credential = target_iterator_esxi_credential (&targets);
                  ssh_lsc_credential_available = 1;
                  if (get_targets_data->get.trash
                      && target_iterator_ssh_trash (&targets))
                    {
                      ssh_lsc_name = trash_lsc_credential_name (ssh_credential);
                      ssh_lsc_uuid = trash_lsc_credential_uuid (ssh_credential);
                      ssh_lsc_credential_available
                       = trash_lsc_credential_readable (ssh_credential);
                    }
                  else if (ssh_credential)
                    {
                      lsc_credential_t found;

                      ssh_lsc_name = lsc_credential_name (ssh_credential);
                      ssh_lsc_uuid = lsc_credential_uuid (ssh_credential);
                      if (find_lsc_credential_with_permission
                           (ssh_lsc_uuid,
                            &found,
                            "get_lsc_credentials"))
                        abort ();
                      ssh_lsc_credential_available = (found > 0);
                    }
                  else
                    {
                      ssh_lsc_name = NULL;
                      ssh_lsc_uuid = NULL;
                    }
                  smb_lsc_credential_available = 1;
                  if (get_targets_data->get.trash
                      && target_iterator_smb_trash (&targets))
                    {
                      smb_lsc_name = trash_lsc_credential_name (smb_credential);
                      smb_lsc_uuid = trash_lsc_credential_uuid (smb_credential);
                      smb_lsc_credential_available
                       = trash_lsc_credential_readable (smb_credential);
                    }
                  else if (smb_credential)
                    {
                      lsc_credential_t found;

                      smb_lsc_name = lsc_credential_name (smb_credential);
                      smb_lsc_uuid = lsc_credential_uuid (smb_credential);
                      if (find_lsc_credential_with_permission
                           (smb_lsc_uuid,
                            &found,
                            "get_lsc_credentials"))
                        abort ();
                      smb_lsc_credential_available = (found > 0);
                    }
                  else
                    {
                      smb_lsc_name = NULL;
                      smb_lsc_uuid = NULL;
                    }
                  esxi_lsc_credential_available = 1;
                  if (get_targets_data->get.trash
                      && target_iterator_esxi_trash (&targets))
                    {
                      esxi_lsc_name
                       = trash_lsc_credential_name (esxi_credential);
                      esxi_lsc_uuid
                       = trash_lsc_credential_uuid (esxi_credential);
                      esxi_lsc_credential_available
                       = trash_lsc_credential_readable (esxi_credential);
                    }
                  else if (esxi_credential)
                    {
                      lsc_credential_t found;

                      esxi_lsc_name = lsc_credential_name (esxi_credential);
                      esxi_lsc_uuid = lsc_credential_uuid (esxi_credential);
                      if (find_lsc_credential_with_permission
                           (esxi_lsc_uuid,
                            &found,
                            "get_lsc_credentials"))
                        abort ();
                      esxi_lsc_credential_available = (found > 0);
                    }
                  else
                    {
                      esxi_lsc_name = NULL;
                      esxi_lsc_uuid = NULL;
                    }
                  port_list_uuid = target_iterator_port_list_uuid (&targets);
                  port_list_name = target_iterator_port_list_name (&targets);
                  port_list_trash = target_iterator_port_list_trash (&targets);
                  ssh_port = target_iterator_ssh_port (&targets);

                  port_list_available = 1;
                  if (port_list_trash)
                    port_list_available = trash_port_list_readable_uuid
                                           (port_list_uuid);
                  else if (port_list_uuid)
                    {
                      port_list_t found;
                      if (find_port_list_with_permission (port_list_uuid,
                                                          &found,
                                                          "get_port_lists"))
                        abort ();
                      port_list_available = (found > 0);
                    }

                  SEND_GET_COMMON (target, &get_targets_data->get, &targets);

                  hosts = target_iterator_hosts (&targets);
                  exclude_hosts = target_iterator_exclude_hosts (&targets);
                  max_hosts = manage_count_hosts (hosts, exclude_hosts);
                  reverse_lookup_only = target_iterator_reverse_lookup_only
                                         (&targets);
                  reverse_lookup_unify = target_iterator_reverse_lookup_unify
                                         (&targets);

                  SENDF_TO_CLIENT_OR_FAIL ("<hosts>%s</hosts>"
                                           "<exclude_hosts>%s</exclude_hosts>"
                                           "<max_hosts>%i</max_hosts>"
                                           "<port_list id=\"%s\">"
                                           "<name>%s</name>"
                                           "<trash>%i</trash>",
                                           hosts,
                                           exclude_hosts ? exclude_hosts : "",
                                           max_hosts,
                                           port_list_uuid ? port_list_uuid : "",
                                           port_list_name ? port_list_name : "",
                                           port_list_trash);

                  if (port_list_available == 0)
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>");

                  SENDF_TO_CLIENT_OR_FAIL ("</port_list>"
                                           "<ssh_lsc_credential id=\"%s\">"
                                           "<name>%s</name>"
                                           "<port>%s</port>"
                                           "<trash>%i</trash>",
                                           ssh_lsc_uuid ? ssh_lsc_uuid : "",
                                           ssh_lsc_name ? ssh_lsc_name : "",
                                           ssh_port ? ssh_port : "",
                                           (get_targets_data->get.trash
                                             && target_iterator_ssh_trash
                                                 (&targets)));

                  if (ssh_lsc_credential_available == 0)
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>");

                  SENDF_TO_CLIENT_OR_FAIL ("</ssh_lsc_credential>"
                                           "<smb_lsc_credential id=\"%s\">"
                                           "<name>%s</name>"
                                           "<trash>%i</trash>",
                                           smb_lsc_uuid ? smb_lsc_uuid : "",
                                           smb_lsc_name ? smb_lsc_name : "",
                                           (get_targets_data->get.trash
                                             && target_iterator_smb_trash
                                                 (&targets)));

                  if (smb_lsc_credential_available == 0)
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>");

                  SENDF_TO_CLIENT_OR_FAIL ("</smb_lsc_credential>"
                                           "<esxi_lsc_credential id=\"%s\">"
                                           "<name>%s</name>"
                                           "<trash>%i</trash>",
                                           esxi_lsc_uuid ? esxi_lsc_uuid : "",
                                           esxi_lsc_name ? esxi_lsc_name : "",
                                           (get_targets_data->get.trash
                                             && target_iterator_esxi_trash
                                                 (&targets)));

                  if (esxi_lsc_credential_available == 0)
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>");

                  SENDF_TO_CLIENT_OR_FAIL ("</esxi_lsc_credential>"
                                           "<reverse_lookup_only>"
                                           "%s"
                                           "</reverse_lookup_only>"
                                           "<reverse_lookup_unify>"
                                           "%s"
                                           "</reverse_lookup_unify>"
                                           "<alive_tests>%s</alive_tests>",
                                           reverse_lookup_only,
                                           reverse_lookup_unify,
                                           target_iterator_alive_tests
                                            (&targets));

                  if (get_targets_data->get.details)
                    SENDF_TO_CLIENT_OR_FAIL ("<port_range>%s</port_range>",
                                             target_port_range
                                              (get_iterator_resource
                                                (&targets)));

                  if (get_targets_data->tasks)
                    {
                      iterator_t tasks;

                      SEND_TO_CLIENT_OR_FAIL ("<tasks>");
                      init_target_task_iterator (&tasks,
                                                 get_iterator_resource
                                                  (&targets));
                      while (next (&tasks))
                        {
                          SENDF_TO_CLIENT_OR_FAIL ("<task id=\"%s\">"
                                                   "<name>%s</name>",
                                                   target_task_iterator_uuid (&tasks),
                                                   target_task_iterator_name (&tasks));
                          if (alert_task_iterator_readable (&tasks))
                            SEND_TO_CLIENT_OR_FAIL ("</task>");
                          else
                            SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                                    "</task>");
                        }
                      cleanup_iterator (&tasks);
                      SEND_TO_CLIENT_OR_FAIL ("</tasks>");
                    }

                  SEND_TO_CLIENT_OR_FAIL ("</target>");
                  count++;
                  free (ssh_lsc_name);
                  free (ssh_lsc_uuid);
                  free (smb_lsc_name);
                  free (smb_lsc_uuid);
                }
              cleanup_iterator (&targets);
              filtered = get_targets_data->get.id
                          ? 1
                          : target_count (&get_targets_data->get);
              SEND_GET_END ("target", &get_targets_data->get, count, filtered);
            }
          get_targets_data_reset (get_targets_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_TASKS:
        {
          iterator_t tasks;
          int count, filtered, ret, first, apply_overrides, min_qod;
          get_data_t * get;
          gchar *overrides, *min_qod_str, *filter, *clean_filter;

          assert (strcasecmp ("GET_TASKS", element_name) == 0);

          if (get_tasks_data->get.details && get_tasks_data->get.trash)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("get_task",
                                  "GET_TASKS details given with trash"));
              get_tasks_data_reset (get_tasks_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          INIT_GET (task, Task);

          ret = init_task_iterator (&tasks, &get_tasks_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_tasks",
                                                   "task",
                                                   get_tasks_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_tasks", "task", get_tasks_data->get.filt_id,
                          omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_tasks"));
                    break;
                }
              get_tasks_data_reset (get_tasks_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("task");

          get = &get_tasks_data->get;
          if (get->filt_id && strcmp (get->filt_id, "0"))
            {
              filter = filter_term (get->filt_id);
              if (filter == NULL)
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else
            filter = NULL;
          clean_filter = manage_clean_filter (filter ? filter : get->filter);
          overrides = filter_term_value (clean_filter, "apply_overrides");
          min_qod_str = filter_term_value (clean_filter, "min_qod");
          min_qod = min_qod_str ? atoi (min_qod_str) : QOD_DEFAULT;
          g_free (clean_filter);
          apply_overrides = overrides
                             ? strcmp (overrides, "0")
                             : APPLY_OVERRIDES_DEFAULT;
          g_free (overrides);
          g_free (min_qod_str);
          SENDF_TO_CLIENT_OR_FAIL ("<apply_overrides>%i</apply_overrides>",
                                   apply_overrides);

          min_qod_str = g_strdup_printf ("%d", min_qod);

          while (1)
            {
              task_t index;
              gchar *progress_xml;
              target_t target;
              slave_t slave;
              scanner_t scanner;
              const char *first_report_id, *last_report_id;
              char *config_name, *config_uuid;
              char *task_target_uuid, *task_target_name;
              char *task_slave_uuid, *task_slave_name;
              char *task_schedule_uuid, *task_schedule_name;
              char *task_scanner_uuid, *task_scanner_name;
              gchar *first_report, *last_report;
              gchar *second_last_report_id, *second_last_report;
              gchar *current_report;
              report_t running_report;
              schedule_t schedule;
              time_t next_time;
              char *owner, *observers;
              int target_in_trash, schedule_in_trash;
              int debugs, holes = 0, infos = 0, logs, warnings = 0;
              int holes_2, infos_2, warnings_2;
              int false_positives, task_scanner_type, slave_available;
              int schedule_available, target_available, config_available;
              int scanner_available;
              double severity = 0, severity_2;
              gchar *response;
              iterator_t alerts, groups, roles;
              gchar *in_assets, *max_checks, *max_hosts, *source_iface;

              ret = get_next (&tasks, &get_tasks_data->get, &first, &count,
                              init_task_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              index = get_iterator_resource (&tasks);
              target = task_target (index);
              slave = task_slave (index);

              SEND_GET_COMMON (task, &get_tasks_data->get, &tasks);

              target_in_trash = task_target_in_trash (index);
              if ((target == 0)
                  && (task_iterator_run_status (&tasks) == TASK_STATUS_RUNNING))
                {
                  progress_xml = g_strdup_printf
                                  ("%i",
                                   task_upload_progress (index));
                  running_report = 0;
                }
              else
                {
                  int progress;
                  gchar *host_xml;

                  running_report = task_iterator_current_report (&tasks);
                  progress = report_progress (running_report, index, &host_xml);
                  progress_xml = g_strdup_printf ("%i%s", progress, host_xml);
                  g_free (host_xml);
                }

              if (running_report)
                {
                  gchar *timestamp;
                  char *scan_end, *current_report_id;

                  current_report_id = report_uuid (running_report);

                  if (report_timestamp (current_report_id, &timestamp))
                    g_error ("%s: GET_TASKS: error getting timestamp of report,"
                             " aborting",
                             __FUNCTION__);

                  scan_end = scan_end_time_uuid (current_report_id),

                  current_report = g_strdup_printf ("<current_report>"
                                                    "<report id=\"%s\">"
                                                    "<timestamp>"
                                                    "%s"
                                                    "</timestamp>"
                                                    "<scan_end>%s</scan_end>"
                                                    "</report>"
                                                    "</current_report>",
                                                    current_report_id,
                                                    timestamp,
                                                    scan_end);
                  free (scan_end);
                  g_free (timestamp);
                }
              else
                current_report = g_strdup ("");

              first_report_id = task_iterator_first_report (&tasks);
              if (first_report_id)
                {
                  gchar *timestamp;
                  char *scan_end;

                  // TODO Could skip this count for tasks page.
                  if (report_counts (first_report_id,
                                     &debugs, &holes_2, &infos_2, &logs,
                                     &warnings_2, &false_positives,
                                     &severity_2, apply_overrides,
                                     0, min_qod))
                    g_error ("%s: GET_TASKS: error getting counts for first"
                             " report, aborting",
                             __FUNCTION__);

                  if (report_timestamp (first_report_id, &timestamp))
                    g_error ("%s: GET_TASKS: failed to get timestamp of first"
                             " report, aborting",
                             __FUNCTION__);

                  scan_end = scan_end_time_uuid (first_report_id),

                  first_report = g_strdup_printf ("<first_report>"
                                                  "<report id=\"%s\">"
                                                  "<timestamp>"
                                                  "%s"
                                                  "</timestamp>"
                                                  "<scan_end>%s</scan_end>"
                                                  "<result_count>"
                                                  "<debug>%i</debug>"
                                                  "<hole>%i</hole>"
                                                  "<info>%i</info>"
                                                  "<log>%i</log>"
                                                  "<warning>%i</warning>"
                                                  "<false_positive>"
                                                  "%i"
                                                  "</false_positive>"
                                                  "</result_count>"
                                                  "<severity>%1.1f</severity>"
                                                  "</report>"
                                                  "</first_report>",
                                                  first_report_id,
                                                  timestamp,
                                                  scan_end,
                                                  debugs,
                                                  holes_2,
                                                  infos_2,
                                                  logs,
                                                  warnings_2,
                                                  false_positives,
                                                  severity_2);
                  free (scan_end);
                  g_free (timestamp);
                }
              else
                first_report = g_strdup ("");

              second_last_report_id = task_second_last_report_id (index);
              if (second_last_report_id)
                {
                  gchar *timestamp;
                  char *scan_end;

                  /* If the first report is the second last report then skip
                   * doing the count again. */
                  if (((first_report_id == NULL)
                       || (strcmp (second_last_report_id, first_report_id)))
                      && report_counts (second_last_report_id,
                                        &debugs, &holes_2, &infos_2,
                                        &logs, &warnings_2,
                                        &false_positives, &severity_2,
                                        apply_overrides,
                                        0, min_qod))
                    g_error ("%s: GET_TASKS: error getting counts for second"
                             " report, aborting",
                             __FUNCTION__);

                  if (report_timestamp (second_last_report_id, &timestamp))
                    g_error ("%s: GET_TASKS: error getting timestamp of second"
                             " report, aborting",
                             __FUNCTION__);

                  scan_end = scan_end_time_uuid (second_last_report_id),

                  second_last_report = g_strdup_printf
                                        ("<second_last_report>"
                                         "<report id=\"%s\">"
                                         "<timestamp>%s</timestamp>"
                                         "<scan_end>%s</scan_end>"
                                         "<result_count>"
                                         "<debug>%i</debug>"
                                         "<hole>%i</hole>"
                                         "<info>%i</info>"
                                         "<log>%i</log>"
                                         "<warning>%i</warning>"
                                         "<false_positive>"
                                         "%i"
                                         "</false_positive>"
                                         "</result_count>"
                                         "<severity>%1.1f</severity>"
                                         "</report>"
                                         "</second_last_report>",
                                         second_last_report_id,
                                         timestamp,
                                         scan_end,
                                         debugs,
                                         holes_2,
                                         infos_2,
                                         logs,
                                         warnings_2,
                                         false_positives,
                                         severity_2);
                  free (scan_end);
                  g_free (timestamp);
                }
              else
                second_last_report = g_strdup ("");

              last_report_id = task_iterator_last_report (&tasks);
              if (last_report_id)
                {
                  gchar *timestamp;
                  char *scan_end;

                  /* If the last report is the first report or the second
                   * last report, then reuse the counts from before. */
                  if ((first_report_id == NULL)
                      || (second_last_report_id == NULL)
                      || (strcmp (last_report_id, first_report_id)
                          && strcmp (last_report_id,
                                     second_last_report_id)))
                    {
                      if (report_counts
                           (last_report_id,
                            &debugs, &holes, &infos, &logs,
                            &warnings, &false_positives, &severity,
                            apply_overrides,
                            0, min_qod))
                        g_error ("%s: GET_TASKS: error getting counts for last"
                                 " report, aborting",
                                 __FUNCTION__);
                    }
                  else
                    {
                      holes = holes_2;
                      infos = infos_2;
                      warnings = warnings_2;
                      severity = severity_2;
                    }

                  if (report_timestamp (last_report_id, &timestamp))
                    g_error ("%s: GET_TASKS: error getting timestamp for last"
                             " report, aborting",
                             __FUNCTION__);

                  scan_end = scan_end_time_uuid (last_report_id);

                  last_report = g_strdup_printf ("<last_report>"
                                                 "<report id=\"%s\">"
                                                 "<timestamp>%s</timestamp>"
                                                 "<scan_end>%s</scan_end>"
                                                 "<result_count>"
                                                 "<debug>%i</debug>"
                                                 "<hole>%i</hole>"
                                                 "<info>%i</info>"
                                                 "<log>%i</log>"
                                                 "<warning>%i</warning>"
                                                 "<false_positive>"
                                                 "%i"
                                                 "</false_positive>"
                                                 "</result_count>"
                                                 "<severity>%1.1f</severity>"
                                                 "</report>"
                                                 "</last_report>",
                                                 last_report_id,
                                                 timestamp,
                                                 scan_end,
                                                 debugs,
                                                 holes,
                                                 infos,
                                                 logs,
                                                 warnings,
                                                 false_positives,
                                                 severity);
                  free (scan_end);
                  g_free (timestamp);
                }
              else
                last_report = g_strdup ("");

              g_free (second_last_report_id);

              owner = task_owner_name (index);
              observers = task_observers (index);
              config_name = task_config_name (index);
              config_uuid = task_config_uuid (index);
              target_available = 1;
              if (target_in_trash)
                {
                  task_target_uuid = trash_target_uuid (target);
                  task_target_name = trash_target_name (target);
                  target_available = trash_target_readable (target);
                }
              else if (target)
                {
                  target_t found;
                  task_target_uuid = target_uuid (target);
                  task_target_name = target_name (target);
                  if (find_target_with_permission (task_target_uuid,
                                                   &found,
                                                   "get_targets"))
                    g_error ("%s: GET_TASKS: error finding task target,"
                             " aborting",
                             __FUNCTION__);
                  target_available = (found > 0);
                }
              else
                {
                  task_target_uuid = NULL;
                  task_target_name = NULL;
                }
              config_available = 1;
              if (task_config_in_trash (index))
                config_available = trash_config_readable_uuid (config_uuid);
              else if (config_uuid)
                {
                  config_t found;
                  if (find_config_with_permission (config_uuid,
                                                   &found,
                                                   "get_configs"))
                    g_error ("%s: GET_TASKS: error finding task config,"
                             " aborting",
                             __FUNCTION__);
                  config_available = (found > 0);
                }
              slave_available = 1;
              if (task_slave_in_trash (index))
                {
                  task_slave_uuid = trash_slave_uuid (slave);
                  task_slave_name = trash_slave_name (slave);
                  slave_available = trash_slave_readable (slave);
                }
              else if (slave)
                {
                  slave_t found;
                  task_slave_uuid = slave_uuid (slave);
                  task_slave_name = slave_name (slave);
                  if (find_slave_with_permission (task_slave_uuid,
                                                  &found,
                                                  "get_slaves"))
                    g_error ("%s: GET_TASKS: error finding task slave,"
                             " aborting",
                             __FUNCTION__);

                  slave_available = (found > 0);
                }
              else
                {
                  task_slave_uuid = g_strdup ("");
                  task_slave_name = g_strdup ("");
                }
              schedule_available = 1;
              schedule = task_schedule (index);
              if (schedule)
                {
                  schedule_in_trash = task_schedule_in_trash (index);
                  if (schedule_in_trash)
                    {
                      task_schedule_uuid = schedule_uuid (schedule);
                      task_schedule_name = schedule_name (schedule);
                      schedule_available = trash_schedule_readable (slave);
                    }
                  else
                    {
                      schedule_t found;
                      task_schedule_uuid = schedule_uuid (schedule);
                      task_schedule_name = schedule_name (schedule);
                      if (find_schedule_with_permission (task_schedule_uuid,
                                                         &found,
                                                         "get_schedules"))
                        g_error ("%s: GET_TASKS: error finding task schedule,"
                                 " aborting",
                                 __FUNCTION__);
                      schedule_available = (found > 0);
                    }
                }
              else
                {
                  task_schedule_uuid = (char*) g_strdup ("");
                  task_schedule_name = (char*) g_strdup ("");
                  schedule_in_trash = 0;
                }
              scanner_available = 1;
              scanner = task_scanner (index);
              // FIX trash case?
              if (scanner)
                {
                  schedule_t found;

                  task_scanner_uuid = scanner_uuid (scanner);
                  task_scanner_name = scanner_name (scanner);
                  task_scanner_type = scanner_type (scanner);

                  if (find_scanner_with_permission (task_scanner_uuid,
                                                    &found,
                                                    "get_scanners"))
                    g_error ("%s: GET_TASKS: error finding task scanner,"
                             " aborting",
                             __FUNCTION__);
                  scanner_available = (found > 0);
                }
              else
                {
                  /* Container tasks have no associated scanner. */
                  task_scanner_uuid = g_strdup ("");
                  task_scanner_name = g_strdup ("");
                  task_scanner_type = 0;
                }
              next_time = task_schedule_next_time_tz (index);
              scanner = task_iterator_scanner (&tasks);
              response = g_strdup_printf
                          ("<alterable>%i</alterable>"
                           "<config id=\"%s\">"
                           "<name>%s</name>"
                           "<type>%i</type>"
                           "<trash>%i</trash>"
                           "%s"
                           "</config>"
                           "<target id=\"%s\">"
                           "<name>%s</name>"
                           "<trash>%i</trash>"
                           "%s"
                           "</target>"
                           "<hosts_ordering>%s</hosts_ordering>"
                           "<scanner id='%s'>"
                           "<name>%s</name>"
                           "<type>%d</type>"
                           "%s"
                           "</scanner>"
                           "<slave id=\"%s\">"
                           "<name>%s</name>"
                           "<trash>%i</trash>"
                           "%s"
                           "</slave>"
                           "<status>%s</status>"
                           "<progress>%s</progress>"
                           "<report_count>"
                           "%u<finished>%u</finished>"
                           "</report_count>"
                           "<trend>%s</trend>"
                           "<schedule id=\"%s\">"
                           "<name>%s</name>"
                           "<next_time>%s</next_time>"
                           "<trash>%i</trash>"
                           "%s"
                           "</schedule>"
                           "<schedule_periods>%i</schedule_periods>"
                           "%s%s%s%s",
                           get_tasks_data->get.trash
                            ? 0
                            : task_alterable (index),
                           config_uuid ?: "",
                           config_name ?: "",
                           config_type (task_config (index)),
                           task_config_in_trash (index),
                           config_available ? "" : "<permissions/>",
                           task_target_uuid ?: "",
                           task_target_name ?: "",
                           target_in_trash,
                           target_available ? "" : "<permissions/>",
                           task_iterator_hosts_ordering (&tasks),
                           task_scanner_uuid,
                           task_scanner_name,
                           task_scanner_type,
                           scanner_available ? "" : "<permissions/>",
                           task_slave_uuid ?: "",
                           task_slave_name ?: "",
                           task_slave_in_trash (index),
                           slave_available ? "" : "<permissions/>",
                           task_iterator_run_status_name (&tasks),
                           progress_xml,
                           task_iterator_total_reports (&tasks),
                           task_iterator_finished_reports (&tasks),
                           task_iterator_trend_counts
                            (&tasks, holes, warnings, infos, severity,
                             holes_2, warnings_2, infos_2, severity_2),
                           task_schedule_uuid,
                           task_schedule_name,
                           (next_time == 0
                             ? "over"
                             : iso_time (&next_time)),
                           schedule_in_trash,
                           schedule_available ? "" : "<permissions/>",
                           task_schedule_periods (index),
                           current_report,
                           first_report,
                           last_report,
                           second_last_report);
              g_free (config_name);
              g_free (config_uuid);
              free (task_target_name);
              free (task_target_uuid);
              g_free (progress_xml);
              g_free (current_report);
              g_free (first_report);
              g_free (last_report);
              g_free (second_last_report);
              g_free (task_schedule_uuid);
              g_free (task_schedule_name);
              g_free (task_scanner_uuid);
              g_free (task_scanner_name);
              free (task_slave_uuid);
              free (task_slave_name);
              if (send_to_client (response,
                                  write_to_client,
                                  write_to_client_data))
                {
                  g_free (response);
                  cleanup_iterator (&tasks);
                  error_send_to_client (error);
                  cleanup_iterator (&tasks);
                  return;
                }
              g_free (response);

              SENDF_TO_CLIENT_OR_FAIL
               ("<observers>%s",
                ((owner == NULL)
                 || (strcmp (owner,
                             current_credentials.username)))
                  ? ""
                  : observers);
              free (owner);
              free (observers);

              init_task_group_iterator (&groups, index);
              while (next (&groups))
                SENDF_TO_CLIENT_OR_FAIL
                 ("<group id=\"%s\">"
                  "<name>%s</name>"
                  "</group>",
                  task_group_iterator_uuid (&groups),
                  task_group_iterator_name (&groups));
              cleanup_iterator (&groups);

              init_task_role_iterator (&roles, index);
              while (next (&roles))
                SENDF_TO_CLIENT_OR_FAIL
                 ("<role id=\"%s\">"
                  "<name>%s</name>"
                  "</role>",
                  task_role_iterator_uuid (&roles),
                  task_role_iterator_name (&roles));
              cleanup_iterator (&roles);

              SENDF_TO_CLIENT_OR_FAIL ("</observers>");

              init_task_alert_iterator (&alerts, index, 0);
              while (next (&alerts))
                {
                  alert_t found;

                  if (find_alert_with_permission (task_alert_iterator_uuid
                                                   (&alerts),
                                                  &found,
                                                  "get_alerts"))
                    abort ();

                  SENDF_TO_CLIENT_OR_FAIL
                   ("<alert id=\"%s\">"
                    "<name>%s</name>",
                    task_alert_iterator_uuid (&alerts),
                    task_alert_iterator_name (&alerts));

                  if (found)
                    SENDF_TO_CLIENT_OR_FAIL
                     ("</alert>");
                  else
                    SENDF_TO_CLIENT_OR_FAIL
                     ("<permissions/>"
                      "</alert>");
                }
              cleanup_iterator (&alerts);

              if (get_tasks_data->get.details)
                {
                  /* The detailed version. */

                  /** @todo Handle error cases.
                   *
                   * The errors are either SQL errors or out of space in
                   * buffer errors.  Both should probably just lead to aborts
                   * at the SQL or buffer output level.
                   */
                  send_reports (index,
                                apply_overrides,
                                min_qod,
                                write_to_client,
                                write_to_client_data);
                }

              in_assets = task_preference_value (index, "in_assets");
              max_checks = task_preference_value (index, "max_checks");
              max_hosts = task_preference_value (index, "max_hosts");
              source_iface = task_preference_value (index, "source_iface");

              SENDF_TO_CLIENT_OR_FAIL
               ("<preferences>"
                "<preference>"
                "<name>"
                "Maximum concurrently executed NVTs per host"
                "</name>"
                "<scanner_name>max_checks</scanner_name>"
                "<value>%s</value>"
                "</preference>"
                "<preference>"
                "<name>"
                "Maximum concurrently scanned hosts"
                "</name>"
                "<scanner_name>max_hosts</scanner_name>"
                "<value>%s</value>"
                "</preference>"
                "<preference>"
                "<name>"
                "Network Source Interface"
                "</name>"
                "<scanner_name>source_iface</scanner_name>"
                "<value>%s</value>"
                "</preference>"
                "<preference>"
                "<name>"
                "Add results to Asset Management"
                "</name>"
                "<scanner_name>in_assets</scanner_name>"
                "<value>%s</value>"
                "</preference>"
                "</preferences>"
                "</task>",
                max_checks ? max_checks : "4",
                max_hosts ? max_hosts : "20",
                source_iface ? source_iface : "",
                in_assets ? in_assets : "yes");

              g_free (in_assets);
              g_free (max_checks);
              g_free (max_hosts);

              count++;
            }
          cleanup_iterator (&tasks);
          filtered = get_tasks_data->get.id
                      ? 1
                      : task_count (&get_tasks_data->get);
          SEND_GET_END ("task", &get_tasks_data->get, count, filtered);
        }

        get_tasks_data_reset (get_tasks_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_GET_USERS:
        {
          iterator_t users;
          int count, filtered, ret, first;

          assert (strcasecmp ("GET_USERS", element_name) == 0);

          INIT_GET (user, User);

          ret = init_user_iterator (&users, &get_users_data->get);
          if (ret)
            {
              switch (ret)
                {
                  case 1:
                    if (send_find_error_to_client ("get_users",
                                                   "user",
                                                   get_users_data->get.id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("get_users", "user", get_users_data->get.filt_id,
                          omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("get_users"));
                    break;
                }
              get_users_data_reset (get_users_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          SEND_GET_START ("user");
          while (1)
            {
              iterator_t groups, roles;
              const char *hosts, *ifaces;
              int hosts_allow, ifaces_allow;

              ret = get_next (&users, &get_users_data->get, &first, &count,
                              init_user_iterator);
              if (ret == 1)
                break;
              if (ret == -1)
                {
                  internal_error_send_to_client (error);
                  return;
                }

              SEND_GET_COMMON (user, &get_users_data->get, &users);

              hosts = user_iterator_hosts (&users);
              hosts_allow = user_iterator_hosts_allow (&users);

              SENDF_TO_CLIENT_OR_FAIL ("<hosts allow=\"%i\">%s</hosts>"
                                       "<sources><source>%s</source></sources>",
                                       hosts_allow,
                                       hosts ? hosts : "",
                                       user_iterator_method (&users)
                                        ? user_iterator_method (&users)
                                        : "file");

              /* Interfaces Access */
              ifaces = user_iterator_ifaces (&users);
              ifaces_allow = user_iterator_ifaces_allow (&users);
              SENDF_TO_CLIENT_OR_FAIL ("<ifaces allow=\"%i\">%s</ifaces>",
                                       ifaces_allow,
                                       ifaces ? ifaces : "");

              /* User Roles */
              init_user_role_iterator (&roles,
                                       get_iterator_resource (&users));
              while (next (&roles))
                {
                  SENDF_TO_CLIENT_OR_FAIL ("<role id=\"%s\">"
                                           "<name>%s</name>",
                                           user_role_iterator_uuid (&roles),
                                           user_role_iterator_name (&roles));
                  if (user_role_iterator_readable (&roles))
                    SEND_TO_CLIENT_OR_FAIL ("</role>");
                  else
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                            "</role>");
                }
              cleanup_iterator (&roles);

              SEND_TO_CLIENT_OR_FAIL ("<groups>");
              init_user_group_iterator (&groups,
                                        get_iterator_resource (&users));
              while (next (&groups))
                {
                  SENDF_TO_CLIENT_OR_FAIL ("<group id=\"%s\">"
                                           "<name>%s</name>",
                                           user_group_iterator_uuid (&groups),
                                           user_group_iterator_name (&groups));
                  if (user_group_iterator_readable (&groups))
                    SEND_TO_CLIENT_OR_FAIL ("</group>");
                  else
                    SEND_TO_CLIENT_OR_FAIL ("<permissions/>"
                                            "</group>");
                }
              cleanup_iterator (&groups);
              SEND_TO_CLIENT_OR_FAIL ("</groups>"
                                      "</user>");
              count++;
            }
          cleanup_iterator (&users);
          filtered = get_users_data->get.id
                      ? 1
                      : user_count (&get_users_data->get);
          SEND_GET_END ("user", &get_users_data->get, count, filtered);

          get_users_data_reset (get_users_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      case CLIENT_GET_VERSION:
      case CLIENT_GET_VERSION_AUTHENTIC:
        SEND_TO_CLIENT_OR_FAIL ("<get_version_response"
                                " status=\"" STATUS_OK "\""
                                " status_text=\"" STATUS_OK_TEXT "\">"
                                "<version>" OMP_VERSION "</version>"
                                "</get_version_response>");
        if (client_state == CLIENT_GET_VERSION_AUTHENTIC)
          set_client_state (CLIENT_AUTHENTIC);
        else
          set_client_state (CLIENT_TOP);
        break;

      case CLIENT_HELP:
        if (user_may ("help") == 0)
          {
            SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX ("help",
                                                      "Permission denied"));
            help_data_reset (help_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        if (help_data->format == NULL
            || (strcmp (help_data->format, "text") == 0))
          {
            command_t *commands;
            SEND_TO_CLIENT_OR_FAIL ("<help_response"
                                    " status=\"" STATUS_OK "\""
                                    " status_text=\"" STATUS_OK_TEXT "\">\n");
            commands = omp_commands;
            while ((*commands).name)
              {
                if (command_disabled (omp_parser, (*commands).name) == 0)
                  {
                    int count;
                    SENDF_TO_CLIENT_OR_FAIL ("    %s",
                                             (*commands).name);
                    for (count = 23 - strlen ((*commands).name);
                         count > 0;
                         count--)
                      SEND_TO_CLIENT_OR_FAIL (" ");
                    SENDF_TO_CLIENT_OR_FAIL ("%s\n",
                                             (*commands).summary);
                  }
                commands++;
              }
            SEND_TO_CLIENT_OR_FAIL ("</help_response>");
          }
        else if (help_data->type && (strcmp (help_data->type, "brief") == 0))
          {
            command_t *commands;
            SEND_TO_CLIENT_OR_FAIL ("<help_response"
                                    " status=\"" STATUS_OK "\""
                                    " status_text=\"" STATUS_OK_TEXT "\">\n"
                                    "<schema"
                                    " format=\"XML\""
                                    " extension=\"xml\""
                                    " content_type=\"text/xml\">");
            commands = omp_commands;
            while ((*commands).name)
              {
                if ((command_disabled (omp_parser, (*commands).name) == 0)
                    && ((current_credentials.uuid == NULL)
                        || user_may ((*commands).name)))
                  SENDF_TO_CLIENT_OR_FAIL ("<command>"
                                           "<name>%s</name>"
                                           "<summary>%s</summary>"
                                           "</command>",
                                           (*commands).name,
                                           (*commands).summary);
                commands++;
              }
            SEND_TO_CLIENT_OR_FAIL ("</schema>"
                                    "</help_response>");
          }
        else
          {
            gchar *extension, *content_type, *output;
            gsize output_len;

            switch (manage_schema (help_data->format,
                                   &output,
                                   &output_len,
                                   &extension,
                                   &content_type))
              {
                case 0:
                  break;
                case 1:
                  assert (help_data->format);
                  if (send_find_error_to_client ("help", "schema_format",
                                                 help_data->format, omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  help_data_reset (help_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  return;
                  break;
                case 2:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("help",
                                      "Brief help is only available in XML."));
                  help_data_reset (help_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  return;
                  break;
                default:
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("help"));
                  help_data_reset (help_data);
                  set_client_state (CLIENT_AUTHENTIC);
                  return;
                  break;
              }

            SENDF_TO_CLIENT_OR_FAIL ("<help_response"
                                     " status=\"" STATUS_OK "\""
                                     " status_text=\"" STATUS_OK_TEXT "\">"
                                     "<schema"
                                     " format=\"%s\""
                                     " extension=\"%s\""
                                     " content_type=\"%s\">",
                                     help_data->format
                                      ? help_data->format
                                      : "XML",
                                     extension,
                                     content_type);
            g_free (extension);
            g_free (content_type);

            if (output && strlen (output))
              {
                /* Encode and send the output. */

                if (help_data->format
                    && strcasecmp (help_data->format, "XML"))
                  {
                    gchar *base64;

                    base64 = g_base64_encode ((guchar*) output, output_len);
                    if (send_to_client (base64,
                                        write_to_client,
                                        write_to_client_data))
                      {
                        g_free (output);
                        g_free (base64);
                        error_send_to_client (error);
                        return;
                      }
                    g_free (base64);
                  }
                else
                  {
                    /* Special case the XML schema, bah. */
                    if (send_to_client (output,
                                        write_to_client,
                                        write_to_client_data))
                      {
                        g_free (output);
                        error_send_to_client (error);
                        return;
                      }
                  }
              }
            g_free (output);
            SEND_TO_CLIENT_OR_FAIL ("</schema>"
                                    "</help_response>");
          }
        help_data_reset (help_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_CREATE_AGENT:
        {
          agent_t agent;

          assert (strcasecmp ("CREATE_AGENT", element_name) == 0);

          if (create_agent_data->copy)
            switch (copy_agent (create_agent_data->name,
                                create_agent_data->comment,
                                create_agent_data->copy,
                                &agent))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = agent_uuid (agent);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_agent"),
                                             uuid);
                    log_event ("agent", "Agent", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_agent",
                                      "Agent exists already"));
                  log_event_fail ("agent", "Agent", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_agent", "agent",
                                                 create_agent_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("agent", "Agent", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_agent",
                                      "Permission denied"));
                  log_event_fail ("agent", "Agent", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_agent"));
                  log_event_fail ("agent", "Agent", NULL, "created");
                  break;
              }
          else if (create_agent_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_agent",
                                "CREATE_AGENT requires a NAME"));
          else if (strlen (create_agent_data->name) == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_agent",
                                  "CREATE_AGENT name must be at"
                                  " least one character long"));
            }
          else if (strlen (create_agent_data->installer) == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_agent",
                                  "CREATE_AGENT installer must be at"
                                  " least one byte long"));
            }
          else switch (create_agent (create_agent_data->name,
                                     create_agent_data->comment,
                                     create_agent_data->installer,
                                     create_agent_data->installer_filename,
                                     create_agent_data->installer_signature,
                                     create_agent_data->howto_install,
                                     create_agent_data->howto_use,
                                     &agent))
            {
              case 0:
                {
                  char *uuid;
                  uuid = agent_uuid (agent);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_agent"),
                                           uuid);
                  log_event ("agent", "Agent", uuid, "created");
                  g_free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_agent",
                                    "Agent exists already"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_agent",
                                    "Name may only contain alphanumeric"
                                    " characters"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_agent",
                                    "Permission denied"));
                break;
              default:
                assert (0);
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_agent"));
                break;
            }
          create_agent_data_reset (create_agent_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_AGENT, COMMENT);
      CLOSE (CLIENT_CREATE_AGENT, COPY);
      CLOSE (CLIENT_CREATE_AGENT, HOWTO_INSTALL);
      CLOSE (CLIENT_CREATE_AGENT, HOWTO_USE);
      CLOSE (CLIENT_CREATE_AGENT, INSTALLER);
      CLOSE (CLIENT_CREATE_AGENT_INSTALLER, FILENAME);
      CLOSE (CLIENT_CREATE_AGENT_INSTALLER, SIGNATURE);
      CLOSE (CLIENT_CREATE_AGENT, NAME);

      case CLIENT_CREATE_CONFIG:
        {
          config_t new_config;

          assert (strcasecmp ("CREATE_CONFIG", element_name) == 0);
          assert (import_config_data->import
                  || (create_config_data->name != NULL));

          /* For now the import element, GET_CONFIGS_RESPONSE, overrides
           * any other elements. */

          if (import_config_data->import)
            {
              char *name;
              array_terminate (import_config_data->nvt_selectors);
              array_terminate (import_config_data->preferences);
              switch (create_config (import_config_data->name,
                                     import_config_data->comment,
                                     import_config_data->nvt_selectors,
                                     import_config_data->preferences,
                                     &new_config,
                                     &name))
                {
                  case 0:
                    {
                      gchar *uuid = config_uuid (new_config);
                      SENDF_TO_CLIENT_OR_FAIL
                       ("<create_config_response"
                        " status=\"" STATUS_OK_CREATED "\""
                        " status_text=\"" STATUS_OK_CREATED_TEXT "\""
                        " id=\"%s\">"
                        /* This is a hack for the GSA, which should really
                         * do a GET_CONFIG with the ID to get the name. */
                        "<config id=\"%s\"><name>%s</name></config>"
                        "</create_config_response>",
                        uuid,
                        uuid,
                        name);
                      log_event ("config", "Scan config", uuid, "created");
                      g_free (uuid);
                      free (name);
                      break;
                    }
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Config exists already"));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Permission denied"));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_config"));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                  case -2:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "CREATE_CONFIG import name must be at"
                                        " least one character long"));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                  case -3:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Error in NVT_SELECTORS element."));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                  case -4:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Error in PREFERENCES element."));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                }
            }
          else if (create_config_data->scanner)
            {
              char *uuid = NULL;

              switch (create_config_from_scanner
                       (create_config_data->scanner, create_config_data->name,
                        create_config_data->comment, &uuid))
                {
                  case 0:
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID
                                              ("create_config"), uuid);
                    log_event ("config", "Scan config", uuid, "created");
                    break;
                  case 1:
                    SENDF_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Failed to find scanner"));
                    break;
                  case 2:
                    SENDF_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Scanner not of type OSP"));
                    break;
                  case 3:
                    SENDF_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Config name exists already"));
                    break;
                  case 4:
                    SENDF_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Failed to get params from scanner"));
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_config",
                                        "Permission denied"));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                  case -1:
                  default:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_config"));
                    log_event_fail ("config", "Scan config", NULL, "created");
                    break;
                }
              g_free (uuid);
            }
          else if (strlen (create_config_data->name) == 0
                   && (create_config_data->copy == NULL
                       || strlen (create_config_data->copy) == 0))
            {
              log_event_fail ("config", "Scan config", NULL, "created");
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_config",
                                  "CREATE_CONFIG name and base config to copy"
                                  " must be at least one character long"));
            }
          else if (create_config_data->copy == NULL)
            {
              log_event_fail ("config", "Scan config", NULL, "created");
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_config",
                                  "CREATE_CONFIG requires a COPY element"));
            }
          else switch (copy_config (create_config_data->name,
                                    create_config_data->comment,
                                    create_config_data->copy,
                                    &new_config))
            {
              case 0:
                {
                  char *uuid = config_uuid (new_config);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_config"),
                                           uuid);
                  log_event ("config", "Scan config", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_config",
                                    "Config exists already"));
                log_event_fail ("config", "Scan config", NULL, "created");
                break;
              case 2:
                if (send_find_error_to_client ("create_config", "config",
                                               create_config_data->copy,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("config", "Config", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_config",
                                    "Permission denied"));
                log_event_fail ("config", "Scan config", NULL, "created");
                break;
              case -1:
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_config"));
                log_event_fail ("config", "Scan config", NULL, "created");
                break;
            }
          create_config_data_reset (create_config_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_CONFIG, COMMENT);
      CLOSE (CLIENT_CREATE_CONFIG, COPY);
      CLOSE (CLIENT_CREATE_CONFIG, SCANNER);
      CLOSE (CLIENT_CREATE_CONFIG, NAME);

      case CLIENT_C_C_GCR:
        assert (strcasecmp ("GET_CONFIGS_RESPONSE", element_name) == 0);
        set_client_state (CLIENT_CREATE_CONFIG);
        break;
      CLOSE (CLIENT_C_C_GCR, CONFIG);
      CLOSE (CLIENT_C_C_GCR_CONFIG, COMMENT);
      CLOSE (CLIENT_C_C_GCR_CONFIG, NAME);
      CLOSE (CLIENT_C_C_GCR_CONFIG, NVT_SELECTORS);
      case CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR:
        {
          int include;

          assert (strcasecmp ("NVT_SELECTOR", element_name) == 0);

          if (import_config_data->nvt_selector_include
              && strcmp (import_config_data->nvt_selector_include, "0") == 0)
            include = 0;
          else
            include = 1;

          array_add (import_config_data->nvt_selectors,
                     nvt_selector_new
                      (import_config_data->nvt_selector_name,
                       import_config_data->nvt_selector_type,
                       include,
                       import_config_data->nvt_selector_family_or_nvt));

          import_config_data->nvt_selector_name = NULL;
          import_config_data->nvt_selector_type = NULL;
          free (import_config_data->nvt_selector_include);
          import_config_data->nvt_selector_include = NULL;
          import_config_data->nvt_selector_family_or_nvt = NULL;

          set_client_state (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS);
          break;
        }
      CLOSE (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR, INCLUDE);
      CLOSE (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR, NAME);
      CLOSE (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR, TYPE);
      CLOSE (CLIENT_C_C_GCR_CONFIG_NVT_SELECTORS_NVT_SELECTOR, FAMILY_OR_NVT);
      CLOSE (CLIENT_C_C_GCR_CONFIG, PREFERENCES);
      case CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE:
        assert (strcasecmp ("PREFERENCE", element_name) == 0);
        array_terminate (import_config_data->preference_alts);
        array_add (import_config_data->preferences,
                   preference_new (import_config_data->preference_name,
                                   import_config_data->preference_type,
                                   import_config_data->preference_value,
                                   import_config_data->preference_nvt_name,
                                   import_config_data->preference_nvt_oid,
                                   import_config_data->preference_alts));
        import_config_data->preference_name = NULL;
        import_config_data->preference_type = NULL;
        import_config_data->preference_value = NULL;
        import_config_data->preference_nvt_name = NULL;
        import_config_data->preference_nvt_oid = NULL;
        import_config_data->preference_alts = NULL;
        set_client_state (CLIENT_C_C_GCR_CONFIG_PREFERENCES);
        break;
      case CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_ALT:
        assert (strcasecmp ("ALT", element_name) == 0);
        array_add (import_config_data->preference_alts,
                   import_config_data->preference_alt);
        import_config_data->preference_alt = NULL;
        set_client_state (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE);
        break;
      CLOSE (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE, NAME);
      CLOSE (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE, NVT);
      CLOSE (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE_NVT, NAME);
      CLOSE (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE, TYPE);
      CLOSE (CLIENT_C_C_GCR_CONFIG_PREFERENCES_PREFERENCE, VALUE);

      case CLIENT_CREATE_ALERT:
        {
          event_t event;
          alert_condition_t condition;
          alert_method_t method;
          alert_t new_alert;

          assert (strcasecmp ("CREATE_ALERT", element_name) == 0);
          assert (create_alert_data->name != NULL);
          assert (create_alert_data->condition != NULL);
          assert (create_alert_data->method != NULL);
          assert (create_alert_data->event != NULL);

          array_terminate (create_alert_data->condition_data);
          array_terminate (create_alert_data->event_data);
          array_terminate (create_alert_data->method_data);

          if (create_alert_data->copy)
            switch (copy_alert (create_alert_data->name,
                                create_alert_data->comment,
                                create_alert_data->copy,
                                &new_alert))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = alert_uuid (new_alert);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_alert"),
                                             uuid);
                    log_event ("alert", "Alert", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_alert",
                                      "Alert exists already"));
                  log_event_fail ("alert", "Alert", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_alert", "alert",
                                                 create_alert_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("alert", "Alert", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_alert",
                                      "Permission denied"));
                  log_event_fail ("alert", "Alert", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_alert"));
                  log_event_fail ("alert", "Alert", NULL, "created");
                  break;
              }
          else if (strlen (create_alert_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "CREATE_ALERT requires NAME element which"
                                " is at least one character long"));
          else if (strlen (create_alert_data->condition) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "CREATE_ALERT requires a value in a"
                                " CONDITION element"));
          else if (strlen (create_alert_data->event) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "CREATE_ALERT requires a value in an"
                                " EVENT element"));
          else if (strlen (create_alert_data->method) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "CREATE_ALERT requires a value in a"
                                " METHOD element"));
          else if ((condition = alert_condition_from_name
                                 (create_alert_data->condition))
                   == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "Failed to recognise condition name"));
          else if ((event = event_from_name (create_alert_data->event))
                   == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "Failed to recognise event name"));
          else if ((method = alert_method_from_name
                              (create_alert_data->method))
                   == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_alert",
                                "Failed to recognise method name"));
          else
            {
              switch (create_alert (create_alert_data->name,
                                    create_alert_data->comment,
                                    create_alert_data->filter_id,
                                    event,
                                    create_alert_data->event_data,
                                    condition,
                                    create_alert_data->condition_data,
                                    method,
                                    create_alert_data->method_data,
                                    &new_alert))
                {
                  case 0:
                    {
                      char *uuid;
                      uuid = alert_uuid (new_alert);
                      SENDF_TO_CLIENT_OR_FAIL
                       (XML_OK_CREATED_ID ("create_alert"), uuid);
                      log_event ("alert", "Alert", uuid, "created");
                      free (uuid);
                      break;
                    }
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Alert exists already"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 2:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Validation of email address failed"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 3:
                    if (send_find_error_to_client ("create_alert", "filter",
                                                   create_alert_data->filter_id,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 4:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Filter type must be result if"
                                        " specified"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 5:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Invalid or unexpected condition data"
                                        " name"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 6:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Syntax error in condition data"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 7:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Email subject too long"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 8:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Email message too long"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_alert",
                                        "Permission denied"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                  default:
                    assert (0);
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_alert"));
                    log_event_fail ("alert", "Alert", NULL, "created");
                    break;
                }
            }
          create_alert_data_reset (create_alert_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_ALERT, COMMENT);
      CLOSE (CLIENT_CREATE_ALERT, COPY);
      CLOSE (CLIENT_CREATE_ALERT, CONDITION);
      CLOSE (CLIENT_CREATE_ALERT, EVENT);
      CLOSE (CLIENT_CREATE_ALERT, FILTER);
      CLOSE (CLIENT_CREATE_ALERT, METHOD);
      CLOSE (CLIENT_CREATE_ALERT, NAME);

      case CLIENT_CREATE_ALERT_CONDITION_DATA:
        {
          gchar *string;

          assert (strcasecmp ("DATA", element_name) == 0);
          assert (create_alert_data->condition_data);
          assert (create_alert_data->part_data);
          assert (create_alert_data->part_name);

          string = g_strconcat (create_alert_data->part_name,
                                "0",
                                create_alert_data->part_data,
                                NULL);
          string[strlen (create_alert_data->part_name)] = '\0';
          array_add (create_alert_data->condition_data, string);

          openvas_free_string_var (&create_alert_data->part_data);
          openvas_free_string_var (&create_alert_data->part_name);
          openvas_append_string (&create_alert_data->part_data, "");
          openvas_append_string (&create_alert_data->part_name, "");
          set_client_state (CLIENT_CREATE_ALERT_CONDITION);
          break;
        }
      case CLIENT_CREATE_ALERT_CONDITION_DATA_NAME:
        assert (strcasecmp ("NAME", element_name) == 0);
        set_client_state (CLIENT_CREATE_ALERT_CONDITION_DATA);
        break;

      case CLIENT_CREATE_ALERT_EVENT_DATA:
        {
          gchar *string;

          assert (strcasecmp ("DATA", element_name) == 0);
          assert (create_alert_data->event_data);
          assert (create_alert_data->part_data);
          assert (create_alert_data->part_name);

          string = g_strconcat (create_alert_data->part_name,
                                "0",
                                create_alert_data->part_data,
                                NULL);
          string[strlen (create_alert_data->part_name)] = '\0';
          array_add (create_alert_data->event_data, string);

          openvas_free_string_var (&create_alert_data->part_data);
          openvas_free_string_var (&create_alert_data->part_name);
          openvas_append_string (&create_alert_data->part_data, "");
          openvas_append_string (&create_alert_data->part_name, "");
          set_client_state (CLIENT_CREATE_ALERT_EVENT);
          break;
        }
      CLOSE (CLIENT_CREATE_ALERT_EVENT_DATA, NAME);

      case CLIENT_CREATE_ALERT_METHOD_DATA:
        {
          gchar *string;

          assert (strcasecmp ("DATA", element_name) == 0);
          assert (create_alert_data->method_data);
          assert (create_alert_data->part_data);
          assert (create_alert_data->part_name);

          string = g_strconcat (create_alert_data->part_name,
                                "0",
                                create_alert_data->part_data,
                                NULL);
          string[strlen (create_alert_data->part_name)] = '\0';
          array_add (create_alert_data->method_data, string);

          openvas_free_string_var (&create_alert_data->part_data);
          openvas_free_string_var (&create_alert_data->part_name);
          openvas_append_string (&create_alert_data->part_data, "");
          openvas_append_string (&create_alert_data->part_name, "");
          set_client_state (CLIENT_CREATE_ALERT_METHOD);
          break;
        }
      CLOSE (CLIENT_CREATE_ALERT_METHOD_DATA, NAME);

      case CLIENT_CREATE_FILTER:
        {
          filter_t new_filter;

          assert (strcasecmp ("CREATE_FILTER", element_name) == 0);
          assert (create_filter_data->term != NULL);

          if (create_filter_data->copy)
            /* TODO make_unique (same for targets). */
            switch (copy_filter (create_filter_data->name,
                                 create_filter_data->comment,
                                 create_filter_data->copy,
                                 &new_filter))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = filter_uuid (new_filter);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_filter"),
                                             uuid);
                    log_event ("filter", "Filter", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_filter",
                                      "Filter exists already"));
                  log_event_fail ("filter", "Filter", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_filter", "filter",
                                                 create_filter_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("filter", "Filter", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_filter",
                                      "Permission denied"));
                  log_event_fail ("filter", "Filter", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_filter"));
                  log_event_fail ("filter", "Filter", NULL, "created");
                  break;
              }
          else if (create_filter_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_filter",
                                "CREATE_FILTER requires a NAME"));
          else if (strlen (create_filter_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_filter",
                                "CREATE_FILTER name must be at"
                                " least one character long"));
          else switch (create_filter
                        (create_filter_data->name,
                         create_filter_data->comment,
                         create_filter_data->type,
                         create_filter_data->term,
                         create_filter_data->make_name_unique
                          && strcmp (create_filter_data->make_name_unique, "0"),
                         &new_filter))
            {
              case 0:
                {
                  char *uuid = filter_uuid (new_filter);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_filter"),
                                           uuid);
                  log_event ("filter", "Filter", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_filter",
                                    "Filter exists already"));
                log_event_fail ("filter", "Filter", NULL, "created");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_filter",
                                    "Type must be a valid OMP type"));
                log_event_fail ("filter", "Filter", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_filter",
                                    "Permission denied"));
                log_event_fail ("filter", "Filter", NULL, "created");
                break;
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_filter"));
                log_event_fail ("filter", "Filter", NULL, "created");
                break;
            }

          create_filter_data_reset (create_filter_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_FILTER, COMMENT);
      CLOSE (CLIENT_CREATE_FILTER, COPY);
      CLOSE (CLIENT_CREATE_FILTER, NAME);
      CLOSE (CLIENT_CREATE_FILTER, TERM);
      CLOSE (CLIENT_CREATE_FILTER, TYPE);

      CLOSE (CLIENT_CREATE_FILTER_NAME, MAKE_UNIQUE);

      case CLIENT_CREATE_GROUP:
        {
          group_t new_group;

          assert (strcasecmp ("CREATE_GROUP", element_name) == 0);
          assert (create_group_data->users != NULL);

          if (create_group_data->copy)
            switch (copy_group (create_group_data->name,
                                create_group_data->comment,
                                create_group_data->copy,
                                &new_group))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = group_uuid (new_group);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_group"),
                                             uuid);
                    log_event ("group", "Group", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_group",
                                      "Group exists already"));
                  log_event_fail ("group", "Group", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_group", "group",
                                                 create_group_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("group", "Group", NULL, "created");
                  break;
                case 4:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_group",
                                      "Syntax error in group name"));
                  log_event_fail ("group", "Group", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_group",
                                      "Permission denied"));
                  log_event_fail ("group", "Group", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_group"));
                  log_event_fail ("group", "Group", NULL, "created");
                  break;
              }
          else if (create_group_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_group",
                                "CREATE_GROUP requires a NAME"));
          else if (strlen (create_group_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_group",
                                "CREATE_GROUP name must be at"
                                " least one character long"));
          else switch (create_group
                        (create_group_data->name,
                         create_group_data->comment,
                         create_group_data->users,
                         &new_group))
            {
              case 0:
                {
                  char *uuid = group_uuid (new_group);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_group"),
                                           uuid);
                  log_event ("group", "Group", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_group",
                                    "Group exists already"));
                log_event_fail ("group", "Group", NULL, "created");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_group",
                                    "Failed to find user"));
                log_event_fail ("group", "Group", NULL, "created");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_group",
                                    "Error in user name"));
                log_event_fail ("group", "Group", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_group",
                                    "Permission denied"));
                log_event_fail ("group", "Group", NULL, "created");
                break;
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_group"));
                log_event_fail ("group", "Group", NULL, "created");
                break;
            }

          create_group_data_reset (create_group_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_GROUP, COMMENT);
      CLOSE (CLIENT_CREATE_GROUP, COPY);
      CLOSE (CLIENT_CREATE_GROUP, NAME);
      CLOSE (CLIENT_CREATE_GROUP, USERS);

      case CLIENT_CREATE_LSC_CREDENTIAL:
        {
          lsc_credential_t new_lsc_credential;

          assert (strcasecmp ("CREATE_LSC_CREDENTIAL", element_name) == 0);
          assert (create_lsc_credential_data->name != NULL);
          assert (create_lsc_credential_data->login != NULL);

          if (create_lsc_credential_data->copy)
            switch (copy_lsc_credential (create_lsc_credential_data->name,
                                         create_lsc_credential_data->comment,
                                         create_lsc_credential_data->copy,
                                         &new_lsc_credential))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = lsc_credential_uuid (new_lsc_credential);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_lsc_credential"),
                                             uuid);
                    log_event ("lsc_credential", "LSC Credential", uuid,
                               "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_lsc_credential",
                                      "Credential exists already"));
                  log_event_fail ("lsc_credential", "LSC Credential", NULL,
                                  "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_lsc_credential",
                                                 "lsc_credential",
                                                 create_lsc_credential_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("lsc_credential", "LSC Credential", NULL,
                                  "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_lsc_credential",
                                      "Permission denied"));
                  log_event_fail ("lsc_credential", "LSC Credential", NULL,
                                  "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_lsc_credential"));
                  log_event_fail ("lsc_credential", "LSC Credential", NULL,
                                  "created");
                  break;
              }
          else if (strlen (create_lsc_credential_data->name) == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_lsc_credential",
                                  "CREATE_LSC_CREDENTIAL name must be at"
                                  " least one character long"));
            }
          else if (strlen (create_lsc_credential_data->login) == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_lsc_credential",
                                  "CREATE_LSC_CREDENTIAL login must be at"
                                  " least one character long"));
            }
          else if (create_lsc_credential_data->key
                   && create_lsc_credential_data->key_private == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_lsc_credential",
                                  "CREATE_LSC_CREDENTIAL KEY requires a PRIVATE"
                                  " key"));
            }
          else switch (create_lsc_credential
                        (create_lsc_credential_data->name,
                         create_lsc_credential_data->comment,
                         create_lsc_credential_data->login,
                         create_lsc_credential_data->key_private
                          ? create_lsc_credential_data->key_phrase
                          : create_lsc_credential_data->password,
                         create_lsc_credential_data->key_private,
                         &new_lsc_credential))
            {
              case 0:
                {
                  char *uuid = lsc_credential_uuid (new_lsc_credential);
                  SENDF_TO_CLIENT_OR_FAIL
                   (XML_OK_CREATED_ID ("create_lsc_credential"), uuid);
                  log_event ("lsc_credential", "LSC Credential", uuid,
                             "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_lsc_credential",
                                    "LSC Credential exists already"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_lsc_credential",
                                    "Login may only contain alphanumeric"
                                    " characters if autogenerating"
                                    " credential"));
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_lsc_credential",
                                    "Erroneous private key or associated"
                                    " passphrase"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_lsc_credential",
                                    "Permission denied"));
                break;
              default:
                assert (0);
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_lsc_credential"));
                break;
            }
          create_lsc_credential_data_reset (create_lsc_credential_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL, COMMENT);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL, COPY);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL, KEY);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL_KEY, PHRASE);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL_KEY, PRIVATE);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL, LOGIN);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL, NAME);
      CLOSE (CLIENT_CREATE_LSC_CREDENTIAL, PASSWORD);

      case CLIENT_CREATE_NOTE:
        {
          task_t task = 0;
          result_t result = 0;
          note_t new_note;
          int max;

          assert (strcasecmp ("CREATE_NOTE", element_name) == 0);

          if (create_note_data->copy)
            switch (copy_note (create_note_data->copy, &new_note))
              {
                case 0:
                  {
                    char *uuid;
                    note_uuid (new_note, &uuid);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_note"),
                                             uuid);
                    log_event ("note", "Note", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_note",
                                      "Note exists already"));
                  log_event_fail ("note", "Note", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_note", "note",
                                                 create_note_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("note", "Note", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_note",
                                      "Permission denied"));
                  log_event_fail ("note", "Note", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_note"));
                  log_event_fail ("note", "Note", NULL, "created");
                  break;
              }
          else if (create_note_data->nvt_oid == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_note",
                                "CREATE_NOTE requires an NVT entity"));
          else if (create_note_data->text == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_note",
                                "CREATE_NOTE requires a TEXT entity"));
          else if (create_note_data->hosts
                   && ((max = manage_count_hosts (create_note_data->hosts, NULL))
                       == -1))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_note",
                                "Error in host specification"));
          else if (create_note_data->hosts && (max > manage_max_hosts ()))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_note",
                                "Host specification exceeds maximum number of"
                                " hosts"));
          else if (create_note_data->task_id
                   && find_task_with_permission (create_note_data->task_id,
                                                 &task,
                                                 NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_note"));
          else if (create_note_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("create_note", "task",
                                             create_note_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (create_note_data->result_id
                   && find_result_with_permission (create_note_data->result_id,
                                                   &result,
                                                   NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_note"));
          else if (create_note_data->result_id && result == 0)
            {
              if (send_find_error_to_client ("create_note", "result",
                                             create_note_data->result_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else switch (create_note (create_note_data->active,
                                    create_note_data->nvt_oid,
                                    create_note_data->text,
                                    create_note_data->hosts,
                                    create_note_data->port,
                                    create_note_data->severity,
                                    create_note_data->threat,
                                    task,
                                    result,
                                    &new_note))
            {
              case 0:
                {
                  char *uuid;
                  note_uuid (new_note, &uuid);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_note"),
                                           uuid);
                  log_event ("note", "Note", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                if (send_find_error_to_client ("create_note", "nvt",
                                               create_note_data->nvt_oid,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_note",
                                    "Error in port specification"));
                log_event_fail ("note", "Note", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_note",
                                    "Permission denied"));
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_note"));
                break;
              default:
                assert (0);
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_note"));
                break;
            }
          create_note_data_reset (create_note_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_NOTE, ACTIVE);
      CLOSE (CLIENT_CREATE_NOTE, COPY);
      CLOSE (CLIENT_CREATE_NOTE, HOSTS);
      CLOSE (CLIENT_CREATE_NOTE, NVT);
      CLOSE (CLIENT_CREATE_NOTE, PORT);
      CLOSE (CLIENT_CREATE_NOTE, SEVERITY);
      CLOSE (CLIENT_CREATE_NOTE, RESULT);
      CLOSE (CLIENT_CREATE_NOTE, TASK);
      CLOSE (CLIENT_CREATE_NOTE, TEXT);
      CLOSE (CLIENT_CREATE_NOTE, THREAT);

      case CLIENT_CREATE_OVERRIDE:
        {
          task_t task = 0;
          result_t result = 0;
          override_t new_override;
          int max;

          assert (strcasecmp ("CREATE_OVERRIDE", element_name) == 0);

          if (create_override_data->copy)
            switch (copy_override (create_override_data->copy, &new_override))
              {
                case 0:
                  {
                    char *uuid;
                    override_uuid (new_override, &uuid);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_override"),
                                             uuid);
                    log_event ("override", "Override", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_override",
                                      "Override exists already"));
                  log_event_fail ("override", "Override", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_override", "override",
                                                 create_override_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("override", "Override", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_override",
                                      "Permission denied"));
                  log_event_fail ("override", "Override", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_override"));
                  log_event_fail ("override", "Override", NULL, "created");
                  break;
              }
          else if (create_override_data->nvt_oid == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_override",
                                "CREATE_OVERRIDE requires an NVT entity"));
          else if (create_override_data->text == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_override",
                                "CREATE_OVERRIDE requires a TEXT entity"));
          else if (create_override_data->hosts
                   && ((max = manage_count_hosts (create_override_data->hosts,
                                                  NULL))
                       == -1))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_override",
                                "Error in host specification"));
          else if (create_override_data->hosts && (max > manage_max_hosts ()))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_override",
                                "Host specification exceeds maximum number"
                                " of hosts"));
          else if (create_override_data->new_threat == NULL
                   && create_override_data->new_severity == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_override",
                                "CREATE_OVERRIDE requires a NEW_THREAT"
                                " or NEW_SEVERITY entity"));
          else if (create_override_data->task_id
              && find_task_with_permission (create_override_data->task_id,
                                            &task,
                                            NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_override"));
          else if (create_override_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("create_override", "task",
                                             create_override_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (create_override_data->result_id
                   && find_result_with_permission (create_override_data->result_id,
                                                   &result,
                                                   NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_override"));
          else if (create_override_data->result_id && result == 0)
            {
              if (send_find_error_to_client ("create_override", "result",
                                             create_override_data->result_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else switch (create_override (create_override_data->active,
                                        create_override_data->nvt_oid,
                                        create_override_data->text,
                                        create_override_data->hosts,
                                        create_override_data->port,
                                        create_override_data->threat,
                                        create_override_data->new_threat,
                                        create_override_data->severity,
                                        create_override_data->new_severity,
                                        task,
                                        result,
                                        &new_override))
            {
              case 0:
                {
                  char *uuid;
                  override_uuid (new_override, &uuid);
                  SENDF_TO_CLIENT_OR_FAIL
                   (XML_OK_CREATED_ID ("create_override"), uuid);
                  log_event ("override", "Override", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                if (send_find_error_to_client ("create_override", "nvt",
                                               create_override_data->nvt_oid,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_override",
                                    "Error in port specification"));
                log_event_fail ("override", "Override", NULL, "created");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_override",
                                    "Error in new_severity specification"));
                log_event_fail ("override", "Override", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_override",
                                    "Permission denied"));
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_override"));
                break;
              default:
                assert (0);
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_override"));
                break;
            }
          create_override_data_reset (create_override_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_OVERRIDE, ACTIVE);
      CLOSE (CLIENT_CREATE_OVERRIDE, COPY);
      CLOSE (CLIENT_CREATE_OVERRIDE, HOSTS);
      CLOSE (CLIENT_CREATE_OVERRIDE, NEW_SEVERITY);
      CLOSE (CLIENT_CREATE_OVERRIDE, NEW_THREAT);
      CLOSE (CLIENT_CREATE_OVERRIDE, NVT);
      CLOSE (CLIENT_CREATE_OVERRIDE, PORT);
      CLOSE (CLIENT_CREATE_OVERRIDE, SEVERITY);
      CLOSE (CLIENT_CREATE_OVERRIDE, RESULT);
      CLOSE (CLIENT_CREATE_OVERRIDE, TASK);
      CLOSE (CLIENT_CREATE_OVERRIDE, TEXT);
      CLOSE (CLIENT_CREATE_OVERRIDE, THREAT);

      case CLIENT_CREATE_PERMISSION:
        {
          permission_t new_permission;

          assert (strcasecmp ("CREATE_PERMISSION", element_name) == 0);

          if (create_permission_data->copy)
            switch (copy_permission (create_permission_data->comment,
                                     create_permission_data->copy,
                                     &new_permission))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = permission_uuid (new_permission);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_permission"),
                                             uuid);
                    log_event ("permission", "Permission", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_permission",
                                      "Permission exists already"));
                  log_event_fail ("permission", "Permission", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_permission",
                                                 "permission",
                                                 create_permission_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("permission", "Permission", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_permission",
                                      "Permission denied"));
                  log_event_fail ("permission", "Permission", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_permission"));
                  log_event_fail ("permission", "Permission", NULL, "created");
                  break;
              }
          else if (create_permission_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_permission",
                                "CREATE_PERMISSION requires a NAME"));
          else if (strlen (create_permission_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_permission",
                                "CREATE_PERMISSION name must be at"
                                " least one character long"));
          else switch (create_permission
                        (create_permission_data->name,
                         create_permission_data->comment,
                         create_permission_data->resource_type,
                         create_permission_data->resource_id,
                         create_permission_data->subject_type,
                         create_permission_data->subject_id,
                         &new_permission))
            {
              case 0:
                {
                  char *uuid = permission_uuid (new_permission);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID
                                            ("create_permission"),
                                           uuid);
                  log_event ("permission", "Permission", uuid, "created");
                  free (uuid);
                  break;
                }
              case 2:
                if (send_find_error_to_client
                     ("create_permission", "subject",
                      create_permission_data->subject_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 3:
                if (send_find_error_to_client
                     ("create_permission", "resource",
                      create_permission_data->resource_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_permission",
                                    "Error in RESOURCE"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 6:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_permission",
                                    "Error in SUBJECT"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 7:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_permission",
                                    "Error in NAME"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 8:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_permission",
                                    "Attempt to create permission on"
                                    " permission"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 9:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_permission",
                                    "Permission does not accept a resource"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_permission",
                                    "Permission denied"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
              case -1:
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_permission"));
                log_event_fail ("permission", "Permission", NULL, "created");
                break;
            }

          create_permission_data_reset (create_permission_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_PERMISSION, COMMENT);
      CLOSE (CLIENT_CREATE_PERMISSION, COPY);
      CLOSE (CLIENT_CREATE_PERMISSION, NAME);
      CLOSE (CLIENT_CREATE_PERMISSION, RESOURCE);
      CLOSE (CLIENT_CREATE_PERMISSION_RESOURCE, TYPE);
      CLOSE (CLIENT_CREATE_PERMISSION, SUBJECT);
      CLOSE (CLIENT_CREATE_PERMISSION_SUBJECT, TYPE);

      case CLIENT_CREATE_PORT_LIST:
        {
          port_list_t new_port_list;
          array_t *manage_ranges;

          assert (strcasecmp ("CREATE_PORT_LIST", element_name) == 0);

          manage_ranges = NULL;

          /* The import element, GET_PORT_LISTS_RESPONSE, overrides any other
           * elements. */

          if (create_port_list_data->import)
            {
              array_terminate (create_port_list_data->ranges);

              if (create_port_list_data->name == NULL)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "CREATE_PORT_LIST"
                                    " GET_PORT_LISTS_RESPONSE requires a"
                                    " NAME element"));
              else if (strlen (create_port_list_data->name) == 0)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "CREATE_PORT_LIST"
                                    " GET_PORT_LISTS_RESPONSE NAME must be"
                                    " at least one character long"));
              else if (create_port_list_data->id == NULL)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "CREATE_PORT_LIST"
                                    " GET_PORT_LISTS_RESPONSE requires an"
                                    " ID attribute"));
              else if (strlen (create_port_list_data->id) == 0)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "CREATE_PORT_LIST"
                                    " GET_PORT_LISTS_RESPONSE ID must be"
                                    " at least one character long"));
              else if (!is_uuid (create_port_list_data->id))
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "CREATE_PORT_LIST"
                                    " GET_PORT_LISTS_RESPONSE ID must be"
                                    " a UUID"));
              else if ((manage_ranges = convert_to_manage_ranges
                                         (create_port_list_data->ranges))
                       == NULL)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "Error in GET_PORT_LISTS_RESPONSE ranges"));
              else switch (create_port_list
                            (create_port_list_data->id,
                             create_port_list_data->name,
                             create_port_list_data->comment,
                             NULL,
                             manage_ranges,
                             &new_port_list))
                {
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_port_list",
                                        "Port list exists already"));
                    log_event_fail ("port_list", "Port List", NULL, "created");
                    break;
                  case 2:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_port_list",
                                        "Port list exists already, in"
                                        " trashcan"));
                    log_event_fail ("port_list", "Port List", NULL, "created");
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_port_list",
                                        "Permission denied"));
                    log_event_fail ("port_list", "Port List", NULL, "created");
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_port_list"));
                    log_event_fail ("port_list", "Port List", NULL, "created");
                    break;
                  default:
                    {
                      char *uuid = port_list_uuid (new_port_list);
                      SENDF_TO_CLIENT_OR_FAIL
                       (XML_OK_CREATED_ID ("create_port_list"),
                        uuid);
                      log_event ("port_list", "Port List", uuid, "created");
                      free (uuid);
                      break;
                    }
                }
              /* Range fields are freed by the reset function below. */
              array_free (manage_ranges);
            }
          else if (create_port_list_data->copy)
            switch (copy_port_list (create_port_list_data->name,
                                    create_port_list_data->comment,
                                    create_port_list_data->copy,
                                    &new_port_list))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = port_list_uuid (new_port_list);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID
                                             ("create_port_list"),
                                             uuid);
                    log_event ("port_list", "Port List", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_port_list",
                                      "Port List exists already"));
                  log_event_fail ("port_list", "Port List", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_port_list",
                                                 "port_list",
                                                 create_port_list_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("port_list", "Port List", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_port_list",
                                      "Permission denied"));
                  log_event_fail ("port_list", "Port List", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_port_list"));
                  log_event_fail ("port_list", "Port List", NULL, "created");
                  break;
              }
          else if (create_port_list_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_port_list",
                                "CREATE_PORT_LIST requires a NAME"));
          else if (strlen (create_port_list_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_port_list",
                                "CREATE_PORT_LIST name must be at"
                                " least one character long"));
          else switch (create_port_list
                        (NULL,
                         create_port_list_data->name,
                         create_port_list_data->comment,
                         create_port_list_data->port_range,
                         NULL,
                         &new_port_list))
            {
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "Port list exists already"));
                log_event_fail ("port_list", "Port List", NULL, "created");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "Error in port range"));
                log_event_fail ("port_list", "Port List", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_list",
                                    "Permission denied"));
                log_event_fail ("port_list", "Port List", NULL, "created");
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_port_list"));
                log_event_fail ("port_list", "Port List", NULL, "created");
                break;
              default:
                {
                  char *uuid = port_list_uuid (new_port_list);
                  SENDF_TO_CLIENT_OR_FAIL
                   (XML_OK_CREATED_ID ("create_port_list"), uuid);
                  log_event ("port_list", "Port List", uuid, "created");
                  free (uuid);
                  break;
                }
            }

          create_port_list_data_reset (create_port_list_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_PORT_LIST, COMMENT);
      CLOSE (CLIENT_CREATE_PORT_LIST, COPY);
      case CLIENT_CPL_GPLR:
        assert (strcasecmp ("GET_PORT_LISTS_RESPONSE", element_name) == 0);
        set_client_state (CLIENT_CREATE_PORT_LIST);
        break;
      CLOSE (CLIENT_CREATE_PORT_LIST, NAME);
      CLOSE (CLIENT_CREATE_PORT_LIST, PORT_RANGE);

      CLOSE (CLIENT_CPL_GPLR, PORT_LIST);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST, COMMENT);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST, IN_USE);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST, NAME);
      CLOSE_READ_OVER (CLIENT_CPL_GPLR_PORT_LIST, TARGETS);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST, PORT_RANGE);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST, PORT_RANGES);

      case CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE:
        {
          assert (strcasecmp ("PORT_RANGE", element_name) == 0);
          assert (create_port_list_data->ranges);

          array_add (create_port_list_data->ranges,
                     create_port_list_data->range);
          create_port_list_data->range = NULL;
          set_client_state (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES);
          break;
        }
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE, COMMENT);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE, END);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE, START);
      CLOSE (CLIENT_CPL_GPLR_PORT_LIST_PORT_RANGES_PORT_RANGE, TYPE);

      case CLIENT_CREATE_PORT_RANGE:
        {
          port_range_t new_port_range;

          assert (strcasecmp ("CREATE_PORT_RANGE", element_name) == 0);

          if (create_port_range_data->start == NULL
              || create_port_range_data->end == NULL
              || create_port_range_data->port_list_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_port_range",
                                "CREATE_PORT_RANGE requires a START, END and"
                                " PORT_LIST ID"));
          else switch (create_port_range
                        (create_port_range_data->port_list_id,
                         create_port_range_data->type,
                         create_port_range_data->start,
                         create_port_range_data->end,
                         create_port_range_data->comment,
                         &new_port_range))
            {
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_range",
                                    "Port range START must be a number"
                                    " 1-65535"));
                log_event_fail ("port_range", "Port Range", NULL, "created");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_range",
                                    "Port range END must be a number"
                                    " 1-65535"));
                log_event_fail ("port_range", "Port Range", NULL, "created");
                break;
              case 3:
                if (send_find_error_to_client
                     ("create_port_range", "port_range",
                      create_port_range_data->port_list_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("port_range", "Port Range", NULL, "created");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_range",
                                    "Port range TYPE must be TCP or UDP"));
                log_event_fail ("port_range", "Port Range", NULL, "created");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_range",
                                    "Port list is in use"));
                break;
              case 6:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_range",
                                    "New range overlaps an existing"
                                    " range"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_port_range",
                                    "Permission denied"));
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_port_range"));
                log_event_fail ("port_range", "Port Range", NULL, "created");
                break;
              default:
                {
                  char *uuid;
                  uuid = port_range_uuid (new_port_range);
                  SENDF_TO_CLIENT_OR_FAIL
                   (XML_OK_CREATED_ID ("create_port_range"), uuid);
                  log_event ("port_range", "Port range", uuid, "created");
                  free (uuid);
                  break;
                }
            }

          create_port_range_data_reset (create_port_range_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_PORT_RANGE, COMMENT);
      CLOSE (CLIENT_CREATE_PORT_RANGE, END);
      CLOSE (CLIENT_CREATE_PORT_RANGE, START);
      CLOSE (CLIENT_CREATE_PORT_RANGE, TYPE);
      CLOSE (CLIENT_CREATE_PORT_RANGE, PORT_LIST);

      case CLIENT_CREATE_REPORT:
        {
          char *uuid;

          assert (strcasecmp ("CREATE_REPORT", element_name) == 0);

          array_terminate (create_report_data->results);
          array_terminate (create_report_data->host_ends);
          array_terminate (create_report_data->host_starts);
          array_terminate (create_report_data->details);

          if (create_report_data->results == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_report",
                                "CREATE_REPORT requires a REPORT element"));
          else if (create_report_data->type
                   && strcmp (create_report_data->type, "scan"))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_report",
                                "CREATE_REPORT type must be 'scan'"));
          else switch (create_report
                        (create_report_data->results,
                         create_report_data->task_id,
                         create_report_data->task_name,
                         create_report_data->task_comment,
                         create_report_data->scan_start,
                         create_report_data->scan_end,
                         create_report_data->host_starts,
                         create_report_data->host_ends,
                         create_report_data->details,
                         &uuid))
            {
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report",
                                    "Permission denied"));
                log_event_fail ("report", "Report", NULL, "created");
                break;
              case -1:
              case -2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_report"));
                log_event_fail ("report", "Report", NULL, "created");
                break;
              case -3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report",
                                    "CREATE_REPORT TASK_NAME is required"));
                log_event_fail ("report", "Report", NULL, "created");
                break;
              case -4:
                log_event_fail ("report", "Report", NULL, "created");
                if (send_find_error_to_client
                     ("create_report", "task",
                      create_report_data->task_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case -5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report",
                                    "CREATE_REPORT TASK must be a container"));
                log_event_fail ("report", "Report", NULL, "created");
                break;
              default:
                {
                  SENDF_TO_CLIENT_OR_FAIL
                   (XML_OK_CREATED_ID ("create_report"),
                    uuid);
                  log_event ("report", "Report", uuid, "created");
                  free (uuid);
                  break;
                }
            }

          omp_parser->importing = 0;
          create_report_data_reset (create_report_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_REPORT, REPORT);
      case CLIENT_CREATE_REPORT_RR:
        assert (strcasecmp ("REPORT", element_name) == 0);
        if (create_report_data->wrapper)
          set_client_state (CLIENT_CREATE_REPORT_REPORT);
        else
          set_client_state (CLIENT_CREATE_REPORT);
        break;
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, FILTERS);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, HOST_COUNT);
      case CLIENT_CREATE_REPORT_RR_HOST_END:
        assert (strcasecmp ("HOST_END", element_name) == 0);

        if (create_report_data->host_end_host)
          {
            create_report_result_t *result;

            assert (create_report_data->host_ends);
            assert (create_report_data->host_end_host);

            result = g_malloc (sizeof (create_report_result_t));
            result->description = create_report_data->host_end;
            result->host = create_report_data->host_end_host;

            array_add (create_report_data->host_ends, result);

            create_report_data->host_end = NULL;
            create_report_data->host_end_host = NULL;
          }
        else
          openvas_free_string_var (&create_report_data->host_end);

        set_client_state (CLIENT_CREATE_REPORT_RR);
        break;
      case CLIENT_CREATE_REPORT_RR_HOST_START:
        assert (strcasecmp ("HOST_START", element_name) == 0);

        if (create_report_data->host_start_host)
          {
            create_report_result_t *result;

            assert (create_report_data->host_starts);
            assert (create_report_data->host_start);
            assert (create_report_data->host_start_host);

            result = g_malloc (sizeof (create_report_result_t));
            result->description = create_report_data->host_start;
            result->host = create_report_data->host_start_host;

            array_add (create_report_data->host_starts, result);

            create_report_data->host_start = NULL;
            create_report_data->host_start_host = NULL;
          }
        else
          openvas_free_string_var (&create_report_data->host_start);

        set_client_state (CLIENT_CREATE_REPORT_RR);
        break;
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, HOSTS);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, PORTS);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, REPORT_FORMAT);
      CLOSE (CLIENT_CREATE_REPORT_RR, RESULTS);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, SCAN_RUN_STATUS);
      CLOSE (CLIENT_CREATE_REPORT_RR, SCAN_END);
      CLOSE (CLIENT_CREATE_REPORT_RR, SCAN_START);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, SORT);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, TASK);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR, RESULT_COUNT);

      CLOSE (CLIENT_CREATE_REPORT_RR_HOST_END, HOST);
      CLOSE (CLIENT_CREATE_REPORT_RR_HOST_START, HOST);

      case CLIENT_CREATE_REPORT_RR_H:
        {
          openvas_free_string_var (&create_report_data->ip);
          set_client_state (CLIENT_CREATE_REPORT_RR);
          break;
        }

      CLOSE (CLIENT_CREATE_REPORT_RR_H, IP);
      CLOSE (CLIENT_CREATE_REPORT_RR_H, START);
      CLOSE (CLIENT_CREATE_REPORT_RR_H, END);

      case CLIENT_CREATE_REPORT_RR_H_DETAIL:
        {
          assert (strcasecmp ("DETAIL", element_name) == 0);
          assert (create_report_data->details);

          if (create_report_data->ip)
            {
              host_detail_t *detail;

              detail = g_malloc (sizeof (host_detail_t));
              detail->ip = g_strdup (create_report_data->ip);
              detail->name = create_report_data->detail_name;
              detail->source_desc = create_report_data->detail_source_desc;
              detail->source_name = create_report_data->detail_source_name;
              detail->source_type = create_report_data->detail_source_type;
              detail->value = create_report_data->detail_value;

              array_add (create_report_data->details, detail);

              create_report_data->detail_name = NULL;
              create_report_data->detail_source_desc = NULL;
              create_report_data->detail_source_name = NULL;
              create_report_data->detail_source_type = NULL;
              create_report_data->detail_value = NULL;
            }

          set_client_state (CLIENT_CREATE_REPORT_RR_H);
          break;
        }

      CLOSE (CLIENT_CREATE_REPORT_RR_H_DETAIL, NAME);
      CLOSE (CLIENT_CREATE_REPORT_RR_H_DETAIL, VALUE);
      CLOSE (CLIENT_CREATE_REPORT_RR_H_DETAIL, SOURCE);

      CLOSE (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE, TYPE);
      CLOSE (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE, NAME);
      case CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE_DESC:
        assert (strcasecmp ("DESCRIPTION", element_name) == 0);
        set_client_state (CLIENT_CREATE_REPORT_RR_H_DETAIL_SOURCE);
        break;

      case CLIENT_CREATE_REPORT_RR_RESULTS_RESULT:
        {
          create_report_result_t *result;

          assert (strcasecmp ("RESULT", element_name) == 0);
          assert (create_report_data->results);

          if (create_report_data->result_scan_nvt_version == NULL)
            create_report_data->result_scan_nvt_version = strdup ("");

          if (create_report_data->result_severity == NULL)
            {
              if (create_report_data->result_threat == NULL)
                create_report_data->result_severity = strdup ("");
              else if (strcasecmp (create_report_data->result_threat,
                              "High") == 0)
                create_report_data->result_severity = strdup ("10.0");
              else if (strcasecmp (create_report_data->result_threat,
                                   "Medium") == 0)
                create_report_data->result_severity = strdup ("5.0");
              else if (strcasecmp (create_report_data->result_threat,
                                   "Low")  == 0)
                create_report_data->result_severity = strdup ("2.0");
              else if (strcasecmp (create_report_data->result_threat,
                                   "Log")  == 0)
                create_report_data->result_severity = strdup ("0.0");
              else if (strcasecmp (create_report_data->result_threat,
                                   "False Positive")  == 0)
                create_report_data->result_severity = strdup ("-1.0");
              else
                create_report_data->result_severity = strdup ("");
            }

          result = g_malloc (sizeof (create_report_result_t));
          result->description = create_report_data->result_description;
          result->host = create_report_data->result_host;
          result->nvt_oid = create_report_data->result_nvt_oid;
          result->scan_nvt_version
            = create_report_data->result_scan_nvt_version;
          result->port = create_report_data->result_port;
          result->qod = create_report_data->result_qod;
          result->qod_type = create_report_data->result_qod_type;
          result->severity = create_report_data->result_severity;
          result->threat = create_report_data->result_threat;

          array_add (create_report_data->results, result);

          create_report_data->result_description = NULL;
          create_report_data->result_host = NULL;
          create_report_data->result_nvt_oid = NULL;
          create_report_data->result_port = NULL;
          create_report_data->result_qod = NULL;
          create_report_data->result_qod_type = NULL;
          create_report_data->result_scan_nvt_version = NULL;
          create_report_data->result_severity = NULL;
          create_report_data->result_threat = NULL;

          set_client_state (CLIENT_CREATE_REPORT_RR_RESULTS);
          break;
        }
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, COMMENT);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, CREATION_TIME);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, DESCRIPTION);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, DETECTION);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, HOST);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT,
                       MODIFICATION_TIME);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, NAME);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, NOTES);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, NVT);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, ORIGINAL_SEVERITY);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, ORIGINAL_THREAT);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, OVERRIDES);
      CLOSE_READ_OVER (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, OWNER);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, PORT);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, QOD);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD, TYPE);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_QOD, VALUE);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, SCAN_NVT_VERSION);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, SEVERITY);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT, THREAT);

      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, BID);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, CVE);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, CVSS_BASE);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, FAMILY);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, NAME);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, XREF);
      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT, CERT);

      CLOSE (CLIENT_CREATE_REPORT_RR_RESULTS_RESULT_NVT_CERT, CERT_REF);

      CLOSE (CLIENT_CREATE_REPORT, TASK);
      CLOSE (CLIENT_CREATE_REPORT_TASK, COMMENT);
      CLOSE (CLIENT_CREATE_REPORT_TASK, NAME);

      case CLIENT_CREATE_REPORT_FORMAT:
        {
          report_format_t new_report_format;

          assert (strcasecmp ("CREATE_REPORT_FORMAT", element_name) == 0);

          /* For now the import element, GET_REPORT_FORMATS_RESPONSE, overrides
           * any other elements. */

          if (create_report_format_data->copy)
            {
              switch (copy_report_format (create_report_format_data->name,
                                          create_report_format_data->copy,
                                          &new_report_format))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = report_format_uuid (new_report_format);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID
                                             ("create_report_format"),
                                             uuid);
                    log_event ("report_format", "Report Format", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_report_format",
                                      "Report Format exists already"));
                  log_event_fail ("report_format", "Report Format", NULL,
                                  "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_report_format",
                                                 "report_format",
                                                 create_report_format_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("report_format", "Report Format", NULL,
                                  "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_report_format",
                                      "Permission denied"));
                  log_event_fail ("report_format", "Report Format", NULL,
                                  "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_report_format"));
                  log_event_fail ("report_format", "Report Format", NULL,
                                  "created");
                  break;
              }
            }
          else if (create_report_format_data->import)
            {
              array_terminate (create_report_format_data->files);
              array_terminate (create_report_format_data->params);
              array_terminate (create_report_format_data->params_options);

              if (create_report_format_data->name == NULL)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report_format",
                                    "CREATE_REPORT_FORMAT"
                                    " GET_REPORT_FORMATS_RESPONSE requires a"
                                    " NAME element"));
              else if (strlen (create_report_format_data->name) == 0)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report_format",
                                    "CREATE_REPORT_FORMAT"
                                    " GET_REPORT_FORMATS_RESPONSE NAME must be"
                                    " at least one character long"));
              else if (create_report_format_data->id == NULL)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report_format",
                                    "CREATE_REPORT_FORMAT"
                                    " GET_REPORT_FORMATS_RESPONSE requires an"
                                    " ID attribute"));
              else if (strlen (create_report_format_data->id) == 0)
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report_format",
                                    "CREATE_REPORT_FORMAT"
                                    " GET_REPORT_FORMATS_RESPONSE ID must be"
                                    " at least one character long"));
              else if (!is_uuid (create_report_format_data->id))
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_report_format",
                                    "CREATE_REPORT_FORMAT"
                                    " GET_REPORT_FORMATS_RESPONSE ID must be"
                                    " a UUID"));
              else switch (create_report_format
                            (create_report_format_data->id,
                             create_report_format_data->name,
                             create_report_format_data->content_type,
                             create_report_format_data->extension,
                             create_report_format_data->summary,
                             create_report_format_data->description,
                             create_report_format_data->global
                               && strcmp (create_report_format_data->global,
                                          "0"),
                             create_report_format_data->files,
                             create_report_format_data->params,
                             create_report_format_data->params_options,
                             create_report_format_data->signature,
                             &new_report_format))
                {
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_report_format"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Report format exists already"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 2:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Every FILE must have a name"
                                        " attribute"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 3:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Parameter value validation failed"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 4:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Parameter default validation failed"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 5:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "CREATE_REPORT_FORMAT PARAM requires a"
                                        " DEFAULT element"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 6:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "CREATE_REPORT_FORMAT PARAM MIN or MAX"
                                        " out of range"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 7:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "CREATE_REPORT_FORMAT PARAM requires a"
                                        " TYPE element"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 8:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Duplicate PARAM name"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 9:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Bogus PARAM type"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_report_format",
                                        "Permission denied"));
                    log_event_fail ("report_format", "Report Format", NULL,
                                    "created");
                    break;
                  default:
                    {
                      char *uuid = report_format_uuid (new_report_format);
                      SENDF_TO_CLIENT_OR_FAIL
                       (XML_OK_CREATED_ID ("create_report_format"),
                        uuid);
                      log_event ("report_format", "Report Format", uuid, "created");
                      free (uuid);
                      break;
                    }
                }
            }
          else
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_report_format",
                                "CREATE_REPORT_FORMAT requires a"
                                " GET_REPORT_FORMATS element"));

          create_report_format_data_reset (create_report_format_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_REPORT_FORMAT, COPY);
      case CLIENT_CRF_GRFR:
        assert (strcasecmp ("GET_REPORT_FORMATS_RESPONSE", element_name) == 0);
        set_client_state (CLIENT_CREATE_REPORT_FORMAT);
        break;
      CLOSE (CLIENT_CRF_GRFR, REPORT_FORMAT);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, CONTENT_TYPE);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, DESCRIPTION);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, EXTENSION);
      case CLIENT_CRF_GRFR_REPORT_FORMAT_FILE:
        {
          gchar *string;

          assert (strcasecmp ("FILE", element_name) == 0);
          assert (create_report_format_data->files);
          assert (create_report_format_data->file);
          assert (create_report_format_data->file_name);

          string = g_strconcat (create_report_format_data->file_name,
                                "0",
                                create_report_format_data->file,
                                NULL);
          string[strlen (create_report_format_data->file_name)] = '\0';
          array_add (create_report_format_data->files, string);
          openvas_free_string_var (&create_report_format_data->file);
          openvas_free_string_var (&create_report_format_data->file_name);
          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT);
          break;
        }
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, GLOBAL);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, NAME);
      case CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM:
        {
          create_report_format_param_t *param;

          assert (strcasecmp ("PARAM", element_name) == 0);
          assert (create_report_format_data->params);
          assert (create_report_format_data->param_name);
          assert (create_report_format_data->param_value);

          param = g_malloc (sizeof (*param));
          param->fallback
           = create_report_format_data->param_default
              ? g_strdup (create_report_format_data->param_default)
              : NULL;
          param->name = g_strdup (create_report_format_data->param_name);
          param->type
           = create_report_format_data->param_type
              ? g_strdup (create_report_format_data->param_type)
              : NULL;
          param->type_max
           = create_report_format_data->param_type_max
              ? g_strdup (create_report_format_data->param_type_max)
              : NULL;
          param->type_min
           = create_report_format_data->param_type_min
              ? g_strdup (create_report_format_data->param_type_min)
              : NULL;
          param->value = g_strdup (create_report_format_data->param_value);

          array_add (create_report_format_data->params, param);
          openvas_free_string_var (&create_report_format_data->param_default);
          openvas_free_string_var (&create_report_format_data->param_name);
          openvas_free_string_var (&create_report_format_data->param_type);
          openvas_free_string_var (&create_report_format_data->param_type_max);
          openvas_free_string_var (&create_report_format_data->param_type_min);
          openvas_free_string_var (&create_report_format_data->param_value);

          array_terminate (create_report_format_data->param_options);
          array_add (create_report_format_data->params_options,
                     create_report_format_data->param_options);

          set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT);
          break;
        }
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM, DEFAULT);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM, NAME);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM, TYPE);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM, OPTIONS);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM, VALUE);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, PREDEFINED);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, SIGNATURE);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT, SUMMARY);
      CLOSE_READ_OVER (CLIENT_CRF_GRFR_REPORT_FORMAT, TRUST);

      case CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS_OPTION:
        assert (strcasecmp ("OPTION", element_name) == 0);
        array_add (create_report_format_data->param_options,
                   create_report_format_data->param_option);
        create_report_format_data->param_option = NULL;
        set_client_state (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_OPTIONS);
        break;

      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE, MAX);
      CLOSE (CLIENT_CRF_GRFR_REPORT_FORMAT_PARAM_TYPE, MIN);

      case CLIENT_CREATE_ROLE:
        {
          role_t new_role;

          assert (strcasecmp ("CREATE_ROLE", element_name) == 0);
          assert (create_role_data->users != NULL);

          if (create_role_data->copy)
            switch (copy_role (create_role_data->name,
                                create_role_data->comment,
                                create_role_data->copy,
                                &new_role))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = role_uuid (new_role);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_role"),
                                             uuid);
                    log_event ("role", "Role", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_role",
                                      "Role exists already"));
                  log_event_fail ("role", "Role", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_role", "role",
                                                 create_role_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("role", "Role", NULL, "created");
                  break;
                case 4:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_role",
                                      "Syntax error in role name"));
                  log_event_fail ("role", "Role", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_role",
                                      "Permission denied"));
                  log_event_fail ("role", "Role", NULL, "created");
                  break;
                case -1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_role"));
                  log_event_fail ("role", "Role", NULL, "created");
                  break;
              }
          else if (create_role_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_role",
                                "CREATE_ROLE requires a NAME"));
          else if (strlen (create_role_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_role",
                                "CREATE_ROLE name must be at"
                                " least one character long"));
          else switch (create_role
                        (create_role_data->name,
                         create_role_data->comment,
                         create_role_data->users,
                         &new_role))
            {
              case 0:
                {
                  char *uuid = role_uuid (new_role);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_role"),
                                           uuid);
                  log_event ("role", "Role", NULL, "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_role",
                                    "Role exists already"));
                log_event_fail ("role", "Role", NULL, "created");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_role",
                                    "Failed to find user"));
                log_event_fail ("role", "Role", NULL, "created");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_role",
                                    "Error in user name"));
                log_event_fail ("group", "Group", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_role",
                                    "Permission denied"));
                log_event_fail ("role", "Role", NULL, "created");
                break;
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_role"));
                log_event_fail ("role", "Role", NULL, "created");
                break;
            }

          create_role_data_reset (create_role_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_ROLE, COMMENT);
      CLOSE (CLIENT_CREATE_ROLE, COPY);
      CLOSE (CLIENT_CREATE_ROLE, NAME);
      CLOSE (CLIENT_CREATE_ROLE, USERS);

      case CLIENT_CREATE_SCANNER:
        assert (strcasecmp ("CREATE_SCANNER", element_name) == 0);
        return handle_create_scanner (omp_parser, error);
      CLOSE (CLIENT_CREATE_SCANNER, COMMENT);
      CLOSE (CLIENT_CREATE_SCANNER, COPY);
      CLOSE (CLIENT_CREATE_SCANNER, NAME);
      CLOSE (CLIENT_CREATE_SCANNER, HOST);
      CLOSE (CLIENT_CREATE_SCANNER, PORT);
      CLOSE (CLIENT_CREATE_SCANNER, TYPE);
      CLOSE (CLIENT_CREATE_SCANNER, CA_PUB);
      CLOSE (CLIENT_CREATE_SCANNER, KEY_PUB);
      CLOSE (CLIENT_CREATE_SCANNER, KEY_PRIV);

      case CLIENT_CREATE_SCHEDULE:
        {
          time_t first_time, period, period_months, duration;
          schedule_t new_schedule;

          period_months = 0;

          assert (strcasecmp ("CREATE_SCHEDULE", element_name) == 0);

          if (create_schedule_data->copy)
            switch (copy_schedule (create_schedule_data->name,
                                   create_schedule_data->comment,
                                   create_schedule_data->copy,
                                   &new_schedule))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = schedule_uuid (new_schedule);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_schedule"),
                                             uuid);
                    log_event ("schedule", "Schedule", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_schedule",
                                      "Schedule exists already"));
                  log_event_fail ("schedule", "Schedule", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_schedule", "schedule",
                                                 create_schedule_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("schedule", "Schedule", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_schedule",
                                      "Permission denied"));
                  log_event_fail ("schedule", "Schedule", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_schedule"));
                  log_event_fail ("schedule", "Schedule", NULL, "created");
                  break;
              }
          else if (create_schedule_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "CREATE_SCHEDULE requires a NAME entity"));
          else if ((first_time = time_from_strings
                                  (create_schedule_data->first_time_hour,
                                   create_schedule_data->first_time_minute,
                                   create_schedule_data->first_time_day_of_month,
                                   create_schedule_data->first_time_month,
                                   create_schedule_data->first_time_year,
                                   create_schedule_data->timezone))
                   == -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "Failed to create time from FIRST_TIME"
                                " elements"));
          else if ((period = interval_from_strings
                              (create_schedule_data->period,
                               create_schedule_data->period_unit,
                               &period_months))
                   == -3)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "PERIOD out of range"));
          else if (period < -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "Failed to create interval from PERIOD"));
          else if ((duration = interval_from_strings
                                (create_schedule_data->duration,
                                 create_schedule_data->duration_unit,
                                 NULL))
                   == -3)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "DURATION out of range"));
          else if (duration < -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "Failed to create interval from DURATION"));
#if 0
          /* The actual time of a period in months can vary, so it's extremely
           * hard to do this check.  The schedule will still work fine if the
           * duration is longer than the period. */
          else if (period_months
                   && (duration > (period_months * 60 * 60 * 24 * 28)))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "Duration too long for number of months"));
#endif
          else if (period && (duration > period))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_schedule",
                                "Duration is longer than period"));
          else switch (create_schedule (create_schedule_data->name,
                                        create_schedule_data->comment,
                                        first_time,
                                        period == -1 ? 0 : period,
                                        period_months,
                                        duration == -1 ? 0 : duration,
                                        create_schedule_data->timezone,
                                        &new_schedule))
            {
              case 0:
                {
                  char *uuid = schedule_uuid (new_schedule);
                  SENDF_TO_CLIENT_OR_FAIL
                   (XML_OK_CREATED_ID ("create_schedule"), uuid);
                  log_event ("schedule", "Schedule", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_schedule",
                                    "Schedule exists already"));
                log_event_fail ("schedule", "Schedule", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_schedule",
                                    "Permission denied"));
                log_event_fail ("schedule", "Schedule", NULL, "created");
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_schedule"));
                log_event_fail ("schedule", "Schedule", NULL, "created");
                break;
              default:
                assert (0);
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("create_schedule"));
                log_event_fail ("schedule", "Schedule", NULL, "created");
                break;
            }
          create_schedule_data_reset (create_schedule_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_SCHEDULE, COMMENT);
      CLOSE (CLIENT_CREATE_SCHEDULE, COPY);
      CLOSE (CLIENT_CREATE_SCHEDULE, DURATION);
      CLOSE (CLIENT_CREATE_SCHEDULE, FIRST_TIME);
      CLOSE (CLIENT_CREATE_SCHEDULE, NAME);
      CLOSE (CLIENT_CREATE_SCHEDULE, PERIOD);
      CLOSE (CLIENT_CREATE_SCHEDULE, TIMEZONE);

      CLOSE (CLIENT_CREATE_SCHEDULE_FIRST_TIME, DAY_OF_MONTH);
      CLOSE (CLIENT_CREATE_SCHEDULE_FIRST_TIME, HOUR);
      CLOSE (CLIENT_CREATE_SCHEDULE_FIRST_TIME, MINUTE);
      CLOSE (CLIENT_CREATE_SCHEDULE_FIRST_TIME, MONTH);
      CLOSE (CLIENT_CREATE_SCHEDULE_FIRST_TIME, YEAR);

      CLOSE (CLIENT_CREATE_SCHEDULE_DURATION, UNIT);

      CLOSE (CLIENT_CREATE_SCHEDULE_PERIOD, UNIT);

      case CLIENT_CREATE_SLAVE:
        {
          slave_t new_slave;

          assert (strcasecmp ("CREATE_SLAVE", element_name) == 0);

          if (create_slave_data->copy)
            switch (copy_slave (create_slave_data->name,
                                create_slave_data->comment,
                                create_slave_data->copy,
                                &new_slave))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = slave_uuid (new_slave);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_slave"),
                                             uuid);
                    log_event ("slave", "Slave", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_slave",
                                      "Slave exists already"));
                  log_event_fail ("slave", "Slave", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_slave", "slave",
                                                 create_slave_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("slave", "Slave", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_slave",
                                      "Permission denied"));
                  log_event_fail ("slave", "Slave", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_slave"));
                  log_event_fail ("slave", "Slave", NULL, "created");
                  break;
              }
          else if (create_slave_data->host == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE requires a HOST"));
          else if (strlen (create_slave_data->host) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE HOST must be at"
                                " least one character long"));
          else if (create_slave_data->login == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE requires a LOGIN"));
          else if (strlen (create_slave_data->login) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE LOGIN must be at"
                                " least one character long"));
          else if (create_slave_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE requires a NAME"));
          else if (strlen (create_slave_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE NAME must be at"
                                " least one character long"));
          else if (create_slave_data->port == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE requires a PORT"));
          else if (strlen (create_slave_data->port) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_slave",
                                "CREATE_SLAVE PORT must be at"
                                " least one character long"));
          /* Create slave from host string. */
          else switch (create_slave
                        (create_slave_data->name,
                         create_slave_data->comment,
                         create_slave_data->host,
                         create_slave_data->port,
                         create_slave_data->login,
                         create_slave_data->password,
                         &new_slave))
            {
              case 0:
                {
                  char *uuid = slave_uuid (new_slave);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_slave"),
                                           uuid);
                  log_event ("slave", "Slave", uuid, "created");
                  free (uuid);
                  break;
                }
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_slave",
                                    "Slave exists already"));
                log_event_fail ("slave", "Slave", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_slave",
                                    "Permission denied"));
                log_event_fail ("slave", "Slave", NULL, "created");
                break;
              default:
                assert (0);
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_slave"));
                log_event_fail ("slave", "Slave", NULL, "created");
                break;
            }

          create_slave_data_reset (create_slave_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_SLAVE, COMMENT);
      CLOSE (CLIENT_CREATE_SLAVE, COPY);
      CLOSE (CLIENT_CREATE_SLAVE, HOST);
      CLOSE (CLIENT_CREATE_SLAVE, LOGIN);
      CLOSE (CLIENT_CREATE_SLAVE, NAME);
      CLOSE (CLIENT_CREATE_SLAVE, PASSWORD);
      CLOSE (CLIENT_CREATE_SLAVE, PORT);

      case CLIENT_CREATE_TAG:
        {
          tag_t new_tag;

          assert (strcasecmp ("CREATE_TAG", element_name) == 0);

          if (create_tag_data->copy)
            switch (copy_tag (create_tag_data->name,
                              create_tag_data->comment,
                              create_tag_data->copy,
                              &new_tag))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = tag_uuid (new_tag);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_tag"),
                                             uuid);
                    log_event ("tag", "Tag", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_tag",
                                      "Tag exists already"));
                  log_event_fail ("tag", "Tag", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_tag", "tag",
                                                 create_tag_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("tag", "Tag", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_tag",
                                      "Permission denied"));
                  log_event_fail ("tag", "Tag", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_tag"));
                  log_event_fail ("tag", "Tag", NULL, "created");
                  break;
              }
          else if (create_tag_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "CREATE_TAG requires"
                                " a NAME element"));
          else if (strlen (create_tag_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "CREATE_TAG name must be"
                                " at least one character long"));
          else if (create_tag_data->resource_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "CREATE_TAG requires"
                                " a RESOURCE element with id attribute"));
          else if (create_tag_data->resource_type == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "RESOURCE in CREATE_TAG requires"
                                " a TYPE element"));
          else if (valid_db_resource_type (create_tag_data->resource_type) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "TYPE in CREATE_TAG/RESOURCE must be"
                                " a valid resource type."));
          else if (strcasecmp (create_tag_data->resource_type, "tag") == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "TYPE type in CREATE_TAG/RESOURCE must not"
                                " be 'tag'."));
          else if (create_tag_data->resource_id
                   && strlen (create_tag_data->resource_id) > 0
                   && resource_id_exists (create_tag_data->resource_type,
                                          create_tag_data->resource_id) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_tag",
                                "RESOURCE id must refer to an existing"
                                " resource or be empty."));
          else
            {
              switch (create_tag (create_tag_data->name,
                                  create_tag_data->comment,
                                  create_tag_data->value,
                                  create_tag_data->resource_type,
                                  create_tag_data->resource_id,
                                  create_tag_data->active,
                                  &new_tag))
                {
                  case 0:
                    {
                      char *uuid;
                      uuid = tag_uuid (new_tag);
                      SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_tag"),
                                               uuid);
                      log_event ("tag", "Tag", uuid, "created");
                      free (uuid);
                      break;
                    }
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_tag",
                                        "Permission denied"));
                    log_event_fail ("tag", "Tag", NULL, "created");
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_tag"));
                    log_event_fail ("tag", "Tag", NULL, "created");
                    break;
                }
            }
          g_debug ("trying reset");
          create_tag_data_reset (create_tag_data);
          g_debug ("trying set client state");
          set_client_state (CLIENT_AUTHENTIC);

          break;
        }

      CLOSE (CLIENT_CREATE_TAG, ACTIVE);
      CLOSE (CLIENT_CREATE_TAG, RESOURCE);
      CLOSE (CLIENT_CREATE_TAG, COPY);
      CLOSE (CLIENT_CREATE_TAG, COMMENT);
      CLOSE (CLIENT_CREATE_TAG, NAME);
      CLOSE (CLIENT_CREATE_TAG, VALUE);

      CLOSE (CLIENT_CREATE_TAG_RESOURCE, TYPE);

      case CLIENT_CREATE_TARGET:
        {
          lsc_credential_t ssh_lsc_credential = 0, smb_lsc_credential = 0;
          lsc_credential_t esxi_lsc_credential = 0;
          target_t new_target;

          assert (strcasecmp ("CREATE_TARGET", element_name) == 0);

          if (create_target_data->copy)
            switch (copy_target (create_target_data->name,
                                 create_target_data->comment,
                                 create_target_data->copy,
                                 &new_target))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = target_uuid (new_target);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_target"),
                                             uuid);
                    log_event ("target", "Target", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_target",
                                      "Target exists already"));
                  log_event_fail ("target", "Target", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_target", "target",
                                                 create_target_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("target", "Target", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_target",
                                      "Permission denied"));
                  log_event_fail ("target", "Target", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_target"));
                  log_event_fail ("target", "Target", NULL, "created");
                  break;
              }
          else if (create_target_data->name == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_target",
                                "CREATE_TARGET requires a NAME"));
          else if (strlen (create_target_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_target",
                                "CREATE_TARGET name must be at"
                                " least one character long"));
          else if (create_target_data->hosts == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_target",
                                " CREATE_TARGET requires a host"));
          else if (strlen (create_target_data->hosts) == 0)
            /** @todo Legitimate to pass an empty hosts element? */
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("create_target",
                                "CREATE_TARGET hosts must be at least one"
                                " character long"));
          else if (create_target_data->ssh_lsc_credential_id
                   && find_lsc_credential_with_permission
                       (create_target_data->ssh_lsc_credential_id,
                        &ssh_lsc_credential,
                        "get_lsc_credentials"))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_target"));
          else if (create_target_data->ssh_lsc_credential_id
                   && ssh_lsc_credential == 0)
            {
              if (send_find_error_to_client
                   ("create_target", "LSC credential",
                    create_target_data->ssh_lsc_credential_id, omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (create_target_data->smb_lsc_credential_id
                   && find_lsc_credential_with_permission
                       (create_target_data->smb_lsc_credential_id,
                        &smb_lsc_credential,
                        "get_lsc_credentials"))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_target"));
          else if (create_target_data->smb_lsc_credential_id
                   && smb_lsc_credential == 0)
            {
              if (send_find_error_to_client
                   ("create_target", "LSC credential",
                    create_target_data->smb_lsc_credential_id, omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (create_target_data->esxi_lsc_credential_id
                   && find_lsc_credential_with_permission
                       (create_target_data->esxi_lsc_credential_id,
                        &esxi_lsc_credential,
                        "get_lsc_credentials"))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_target"));
          else if (create_target_data->esxi_lsc_credential_id
                   && esxi_lsc_credential == 0)
            {
              if (send_find_error_to_client
                   ("create_target", "LSC credential",
                    create_target_data->esxi_lsc_credential_id, omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          /* Create target from host string. */
          else switch (create_target
                        (create_target_data->name,
                         create_target_data->hosts,
                         create_target_data->exclude_hosts,
                         create_target_data->comment,
                         create_target_data->port_list_id,
                         create_target_data->port_range,
                         ssh_lsc_credential,
                         create_target_data->ssh_port,
                         smb_lsc_credential,
                         esxi_lsc_credential,
                         create_target_data->reverse_lookup_only,
                         create_target_data->reverse_lookup_unify,
                         create_target_data->alive_tests,
                         (create_target_data->make_name_unique
                          && strcmp (create_target_data->make_name_unique, "0"))
                           ? 1 : 0,
                         &new_target))
            {
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Target exists already"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Error in host specification"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Host specification exceeds maximum number"
                                    " of hosts"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Error in port range"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Error in SSH port"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case 6:
                log_event_fail ("target", "Target", NULL, "created");
                if (send_find_error_to_client
                     ("create_target", "port_list",
                      create_target_data->port_list_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 7:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Error in alive test"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("create_target",
                                    "Permission denied"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_target"));
                log_event_fail ("target", "Target", NULL, "created");
                break;
              default:
                {
                  char *uuid = target_uuid (new_target);
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_target"),
                                           uuid);
                  log_event ("target", "Target", uuid, "created");
                  free (uuid);
                  break;
                }
            }

          create_target_data_reset (create_target_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_TARGET, COMMENT);
      CLOSE (CLIENT_CREATE_TARGET, ESXI_LSC_CREDENTIAL);
      CLOSE (CLIENT_CREATE_TARGET, EXCLUDE_HOSTS);
      CLOSE (CLIENT_CREATE_TARGET, REVERSE_LOOKUP_ONLY);
      CLOSE (CLIENT_CREATE_TARGET, REVERSE_LOOKUP_UNIFY);
      CLOSE (CLIENT_CREATE_TARGET, ALIVE_TESTS);
      CLOSE (CLIENT_CREATE_TARGET, COPY);
      CLOSE (CLIENT_CREATE_TARGET, HOSTS);
      CLOSE (CLIENT_CREATE_TARGET, NAME);
      CLOSE (CLIENT_CREATE_TARGET, PORT_LIST);
      CLOSE (CLIENT_CREATE_TARGET, PORT_RANGE);
      CLOSE (CLIENT_CREATE_TARGET, SSH_LSC_CREDENTIAL);
      CLOSE (CLIENT_CREATE_TARGET, SMB_LSC_CREDENTIAL);

      CLOSE (CLIENT_CREATE_TARGET_NAME, MAKE_UNIQUE);

      CLOSE (CLIENT_CREATE_TARGET_SSH_LSC_CREDENTIAL, PORT);

      case CLIENT_CREATE_TASK:
        {
          config_t config = 0;
          target_t target = 0;
          scanner_t scanner = 0;
          slave_t slave = 0;
          char *tsk_uuid = NULL, *name;
          guint index;

          /* @todo Buffer the entire task creation and pass everything to a
           *       libmanage function, so that libmanage can do the locking
           *       properly instead of exposing the task_t.  Probably easier
           *       after removing the option to create a task from an RC
           *       file. */

          assert (strcasecmp ("CREATE_TASK", element_name) == 0);
          assert (create_task_data->task != (task_t) 0);

          /* The task already exists in the database at this point, so on
           * failure be sure to call request_delete_task to remove the
           * task. */
          /** @todo Any fail cases of the CLIENT_CREATE_TASK_* states must do
           *        so too. */

          if (create_task_data->copy)
            {
              int ret;
              gchar *name, *comment;
              task_t new_task;

              name = task_name (create_task_data->task);
              comment = task_comment (create_task_data->task);
              ret = copy_task (name,
                               comment,
                               create_task_data->copy,
                               (create_task_data->alterable
                                && strcmp (create_task_data->alterable, "0"))
                                ? 1
                                : 0,
                               &new_task);
              g_free (name);
              g_free (comment);
              /* Remove the task that was created while parsing elements. */
              request_delete_task (&create_task_data->task);
              switch (ret)
                {
                  case 0:
                    {
                      char *uuid;
                      task_uuid (new_task, &uuid);
                      SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_task"),
                                               uuid);
                      log_event ("task", "Task", uuid, "created");
                      free (uuid);
                      break;
                    }
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_task",
                                        "Task exists already"));
                    log_event_fail ("task", "Task", NULL, "created");
                    break;
                  case 2:
                    if (send_find_error_to_client ("create_task", "task",
                                                   create_task_data->copy,
                                                   omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    log_event_fail ("task", "Task", NULL, "created");
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("create_task",
                                        "Permission denied"));
                    log_event_fail ("task", "Task", NULL, "created");
                    break;
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("create_task"));
                    log_event_fail ("task", "Task", NULL, "created");
                    break;
                }
              create_task_data_reset (create_task_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (create_task_data->scanner_id == NULL)
            create_task_data->scanner_id = g_strdup (SCANNER_UUID_DEFAULT);
          if (strcmp (create_task_data->scanner_id, SCANNER_UUID_DEFAULT)
              && create_task_data->slave_id)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_task",
                                  "Slave used with non-default Scanner."));
              goto create_task_fail;
            }

          /* Check permissions. */

          if (user_may ("create_task") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX ("create_task",
                                                        "Permission denied"));
              goto create_task_fail;
            }

          /* Get the task ID. */

          if (task_uuid (create_task_data->task, &tsk_uuid))
            {
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
              goto create_task_fail;
            }

          /* Check for the right combination of target and config. */

          if (create_task_data->config_id == NULL
              || create_task_data->target_id == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_task",
                                  "CREATE_TASK requires a config"
                                  " a scanner, and a target"));
              goto create_task_fail;
            }

          /* Set any alert. */

          assert (create_task_data->alerts);
          index = create_task_data->alerts->len;
          while (index--)
            {
              alert_t alert;
              gchar *alert_id;

              alert_id = (gchar*) g_ptr_array_index (create_task_data->alerts,
                                                     index);
              if (strcmp (alert_id, "0") == 0)
                continue;
              if (find_alert_with_permission (alert_id, &alert, "get_alerts"))
                {
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
                  goto create_task_fail;
                }
              if (alert == 0)
                {
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_task",
                                      "CREATE_TASK alert must exist"));
                  goto create_task_fail;
                }
              add_task_alert (create_task_data->task, alert);
            }

          /* Set alterable state. */

          if (create_task_data->alterable
              && strcmp (create_task_data->alterable, "0"))
            set_task_alterable (create_task_data->task, 1);

          /* Set any schedule. */

          if (create_task_data->schedule_id)
            {
              schedule_t schedule;
              int periods;

              periods = create_task_data->schedule_periods
                         ? atoi (create_task_data->schedule_periods)
                         : 0;
              if (find_schedule_with_permission (create_task_data->schedule_id,
                                                 &schedule,
                                                 "get_schedules"))
                {
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
                  goto create_task_fail;
                }
              if (schedule == 0)
                {
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_task",
                                      "CREATE_TASK schedule must exist"));
                  goto create_task_fail;
                }
              /** @todo
               *
               * This is a contention hole.  Some other process could remove
               * the schedule at this point.  The variable "schedule" would
               * still refer to the removed schedule.
               *
               * This happens all over the place.  Anywhere that a libmanage
               * client gets a reference to a resource, in fact.
               *
               * Possibly libmanage should lock the db whenever it hands out a
               * reference, and the client should call something to release
               * the lock when it's done.
               *
               * In many cases, like this one, the client could pass the UUID
               * directly to libmanage, instead of getting the reference.  In
               * this case the client would then need something like
               * set_task_schedule_uuid.
               */
              set_task_schedule (create_task_data->task, schedule, periods);
            }

          /* Set any observers. */

          if (create_task_data->observers)
            {
              int fail;
              fail = set_task_observers (create_task_data->task,
                                         create_task_data->observers);
              switch (fail)
                {
                  case 0:
                    break;
                  case 1:
                  case 2:
                    SEND_TO_CLIENT_OR_FAIL
                      (XML_ERROR_SYNTAX ("create_task",
                                         "User name error in observers"));
                    goto create_task_fail;
                  case -1:
                  default:
                    SEND_TO_CLIENT_OR_FAIL
                      (XML_INTERNAL_ERROR ("create_task"));
                    goto create_task_fail;
                }
            }

          /* Set any observer groups. */

          if (create_task_data->groups->len)
            {
              int fail;
              gchar *fail_group_id;

              switch ((fail = set_task_groups (create_task_data->task,
                                               create_task_data->groups,
                                               &fail_group_id)))
                {
                  case 0:
                    break;
                  case 1:
                    if (send_find_error_to_client
                         ("create_task", "group", fail_group_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    log_event_fail ("task", "Task", NULL, "created");
                    goto create_task_fail;
                  case -1:
                  default:
                    SEND_TO_CLIENT_OR_FAIL
                      (XML_INTERNAL_ERROR ("create_task"));
                    log_event_fail ("task", "Task", NULL, "created");
                    goto create_task_fail;
                }
            }

          /* Check for name. */

          name = task_name (create_task_data->task);
          if (name == NULL)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_task",
                                  "CREATE_TASK requires a name attribute"));
              goto create_task_fail;
            }

          if (find_config_with_permission (create_task_data->config_id,
                                           &config,
                                           "get_configs"))
            {
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
              goto create_task_fail;
            }
          if (config == 0)
            {
              if (send_find_error_to_client ("create_task", "config",
                                             create_task_data->config_id,
                                             omp_parser))
                error_send_to_client (error);
              goto create_task_fail;
            }
          if (find_target_with_permission (create_task_data->target_id,
                                           &target,
                                           "get_targets"))
            {
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
              goto create_task_fail;
            }
          if (target == 0)
            {
              if (send_find_error_to_client ("create_task", "target",
                                             create_task_data->target_id,
                                             omp_parser))
                error_send_to_client (error);
              goto create_task_fail;
            }
          if (find_scanner_with_permission (create_task_data->scanner_id,
                                            &scanner,
                                            "get_scanners"))
            {
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
              goto create_task_fail;
            }
          if (create_task_data->scanner_id && scanner == 0)
            {
              if (send_find_error_to_client ("create_task", "scanner",
                                             create_task_data->scanner_id,
                                             omp_parser))
                error_send_to_client (error);
              goto create_task_fail;
            }
          if (!create_task_check_config_scanner (config, scanner))
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("create_task",
                                  "Scanner and config mismatched types."));
              goto create_task_fail;
            }
          if (create_task_data->slave_id
              && find_slave_with_permission (create_task_data->slave_id,
                                             &slave,
                                             "get_slaves"))
            {
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_task"));
              goto create_task_fail;
            }
          if (create_task_data->slave_id && slave == 0)
            {
              if (send_find_error_to_client ("create_task", "slave",
                                             create_task_data->slave_id,
                                             omp_parser))
                error_send_to_client (error);
              goto create_task_fail;
            }

          set_task_config (create_task_data->task, config);
          set_task_slave (create_task_data->task, slave);
          set_task_target (create_task_data->task, target);
          set_task_scanner (create_task_data->task, scanner);
          set_task_hosts_ordering (create_task_data->task,
                                   create_task_data->hosts_ordering);
          if (create_task_data->preferences)
            set_task_preferences (create_task_data->task,
                                  create_task_data->preferences);

          /* Send success response. */

          SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_task"),
                                   tsk_uuid);
          make_task_complete (tsk_uuid);
          log_event ("task", "Task", tsk_uuid, "created");
          g_free (tsk_uuid);
          create_task_data_reset (create_task_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;

create_task_fail:
          request_delete_task (&create_task_data->task);
          g_free (tsk_uuid);
          create_task_data_reset (create_task_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_CREATE_TASK, ALTERABLE);
      CLOSE (CLIENT_CREATE_TASK, COMMENT);
      CLOSE (CLIENT_CREATE_TASK, HOSTS_ORDERING);
      CLOSE (CLIENT_CREATE_TASK, SCANNER);
      CLOSE (CLIENT_CREATE_TASK, CONFIG);
      CLOSE (CLIENT_CREATE_TASK, COPY);
      CLOSE (CLIENT_CREATE_TASK, ALERT);
      CLOSE (CLIENT_CREATE_TASK, NAME);
      CLOSE (CLIENT_CREATE_TASK, OBSERVERS);
      CLOSE (CLIENT_CREATE_TASK, PREFERENCES);
      CLOSE (CLIENT_CREATE_TASK, TARGET);
      CLOSE (CLIENT_CREATE_TASK, SCHEDULE);
      CLOSE (CLIENT_CREATE_TASK, SCHEDULE_PERIODS);
      CLOSE (CLIENT_CREATE_TASK, SLAVE);

      CLOSE (CLIENT_CREATE_TASK_OBSERVERS, GROUP);

      case CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE:
        assert (strcasecmp ("PREFERENCE", element_name) == 0);
        array_add (create_task_data->preferences,
                   create_task_data->preference);
        create_task_data->preference = NULL;
        set_client_state (CLIENT_CREATE_TASK_PREFERENCES);
        break;
      case CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE_NAME:
        assert (strcasecmp ("SCANNER_NAME", element_name) == 0);
        set_client_state (CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE);
        break;
      CLOSE (CLIENT_CREATE_TASK_PREFERENCES_PREFERENCE, VALUE);

      case CLIENT_CREATE_USER:
        {
          gchar *errdesc;
          gchar *fail_group_id, *fail_role_id;
          user_t new_user;

          assert (strcasecmp ("CREATE_USER", element_name) == 0);

          errdesc = NULL;
          if (create_user_data->copy)
            switch (copy_user (create_user_data->name,
                               NULL,
                               create_user_data->copy,
                               &new_user))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = user_uuid (new_user);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_user"),
                                             uuid);
                    log_event ("user", "User", uuid, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_user",
                                      "User exists already"));
                    log_event_fail ("user", "User", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client ("create_user", "user",
                                                 create_user_data->copy,
                                                 omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                    log_event_fail ("user", "User", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_user",
                                      "Permission denied"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case -1:
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("create_user"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
              }
          else if (create_user_data->name == NULL
              || strlen (create_user_data->name) == 0)
            SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                    ("create_user",
                                     "CREATE_USER requires a name"));
          else
            switch (create_user
                     (create_user_data->name,
                      create_user_data->password ? create_user_data->password : "",
                      create_user_data->hosts,
                      create_user_data->hosts_allow,
                      create_user_data->ifaces,
                      create_user_data->ifaces_allow,
                      create_user_data->sources,
                      create_user_data->groups,
                      &fail_group_id,
                      create_user_data->roles,
                      &fail_role_id,
                      &errdesc,
                      &new_user,
                      1))
              {
                case 0:
                  {
                    char *uuid;
                    uuid = user_uuid (new_user);
                    SENDF_TO_CLIENT_OR_FAIL (XML_OK_CREATED_ID ("create_user"),
                                             uuid);
                    log_event ("user", "User", create_user_data->name, "created");
                    free (uuid);
                    break;
                  }
                case 1:
                  if (send_find_error_to_client
                       ("create_user", "group", fail_group_id, omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case 2:
                  if (send_find_error_to_client
                       ("create_user", "role", fail_role_id, omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case 3:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_user",
                                      "Error in host specification"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("create_user",
                                      "Permission denied"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case -2:
                  SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                          ("create_user", "User already exists"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case -3:
                  SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                          ("create_user", "Error in SOURCE"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
                case -1:
                  if (errdesc)
                    {
                      char *buf = make_xml_error_syntax ("create_user", errdesc);
                      SEND_TO_CLIENT_OR_FAIL (buf);
                      g_free (buf);
                      break;
                    }
                  /* Fall through.  */
                default:
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("create_user"));
                  log_event_fail ("user", "User", NULL, "created");
                  break;
              }
          create_user_data_reset (create_user_data);
          set_client_state (CLIENT_AUTHENTIC);
          g_free (errdesc);
          break;
        }
      CLOSE (CLIENT_CREATE_USER, COPY);
      CLOSE (CLIENT_CREATE_USER, GROUPS);
      CLOSE (CLIENT_CREATE_USER_GROUPS, GROUP);
      CLOSE (CLIENT_CREATE_USER, HOSTS);
      CLOSE (CLIENT_CREATE_USER, IFACES);
      CLOSE (CLIENT_CREATE_USER, NAME);
      CLOSE (CLIENT_CREATE_USER, PASSWORD);
      CLOSE (CLIENT_CREATE_USER, ROLE);
      case CLIENT_CREATE_USER_SOURCES:
        assert (strcasecmp ("SOURCES", element_name) == 0);
        array_terminate (create_user_data->sources);
        set_client_state (CLIENT_CREATE_USER);
        break;
      case CLIENT_CREATE_USER_SOURCES_SOURCE:
        assert (strcasecmp ("SOURCE", element_name) == 0);
        if (create_user_data->current_source)
          array_add (create_user_data->sources,
                     g_strdup (create_user_data->current_source));
        g_free (create_user_data->current_source);
        create_user_data->current_source = NULL;
        set_client_state (CLIENT_CREATE_USER_SOURCES);
        break;

      case CLIENT_EMPTY_TRASHCAN:
        switch (manage_empty_trashcan ())
          {
            case 0:
              SEND_TO_CLIENT_OR_FAIL (XML_OK ("empty_trashcan"));
              log_event ("trashcan", "Trashcan", NULL, "emptied");
              break;
            case 99:
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("empty_trashcan",
                                  "Permission denied"));
              break;
            default:  /* Programming error. */
              assert (0);
            case -1:
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("empty_trashcan"));
              break;
          }
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_MODIFY_AGENT:
        {
          assert (strcasecmp ("MODIFY_AGENT", element_name) == 0);

          switch (modify_agent
                   (modify_agent_data->agent_id,
                    modify_agent_data->name,
                    modify_agent_data->comment))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_agent"));
                log_event ("agent", "Agent", modify_agent_data->agent_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_agent", "agent",
                                               modify_agent_data->agent_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("agent", "Agent", modify_agent_data->agent_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_agent",
                                    "agent with new name exists already"));
                log_event_fail ("agent", "Agent", modify_agent_data->agent_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_agent",
                                    "MODIFY_agent requires a agent_id"));
                log_event_fail ("agent", "Agent", modify_agent_data->agent_id,
                                "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_agent",
                                    "Permission denied"));
                log_event_fail ("agent", "Agent", modify_agent_data->agent_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_agent"));
                log_event_fail ("agent", "Agent", modify_agent_data->agent_id,
                                "modified");
                break;
            }

          modify_agent_data_reset (modify_agent_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_AGENT, COMMENT);
      CLOSE (CLIENT_MODIFY_AGENT, NAME);

      case CLIENT_MODIFY_ALERT:
        {
          event_t event;
          alert_condition_t condition;
          alert_method_t method;

          assert (strcasecmp ("MODIFY_ALERT", element_name) == 0);

          event = EVENT_ERROR;
          condition = ALERT_CONDITION_ERROR;
          method  = ALERT_METHOD_ERROR;

          array_terminate (modify_alert_data->event_data);
          array_terminate (modify_alert_data->condition_data);
          array_terminate (modify_alert_data->method_data);

          if (strlen (modify_alert_data->event)
              && (event = event_from_name (modify_alert_data->event)) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_alert",
                                "Failed to recognise event name"));
          else if (strlen (modify_alert_data->condition) &&
                   (condition = alert_condition_from_name
                                 (modify_alert_data->condition))
                   == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_alert",
                                "Failed to recognise condition name"));
          else if (strlen (modify_alert_data->method) &&
                   (method = alert_method_from_name
                                 (modify_alert_data->method))
                   == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_alert",
                                "Failed to recognise method name"));
          else switch (modify_alert
                        (modify_alert_data->alert_id,
                         modify_alert_data->name,
                         modify_alert_data->comment,
                         modify_alert_data->filter_id,
                         event,
                         modify_alert_data->event_data,
                         condition,
                         modify_alert_data->condition_data,
                         method,
                         modify_alert_data->method_data))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_alert"));
                log_event ("alert", "Alert", modify_alert_data->alert_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_alert", "alert",
                                               modify_alert_data->alert_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "alert with new name exists already"));
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "MODIFY_alert requires an alert_id"));
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              case 4:
                if (send_find_error_to_client ("modify_alert", "filter",
                                               modify_alert_data->filter_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "Filter type must be result if"
                                    " specified"));
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              case 6:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "Validation of email address failed"));
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              case 7:
                SEND_TO_CLIENT_OR_FAIL
                  (XML_ERROR_SYNTAX ("modify_alert",
                                    "Invalid or unexpected condition data"
                                    " name"));
                log_event_fail ("alert", "Alert", NULL, "created");
                break;
              case 8:
                SEND_TO_CLIENT_OR_FAIL
                  (XML_ERROR_SYNTAX ("modify_alert",
                                    "Syntax error in condition data"));
                log_event_fail ("alert", "Alert", NULL, "created");
                break;
              case 9:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "Email subject too long"));
                log_event_fail ("alert", "Alert", NULL, "created");
                break;
              case 10:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "Email message too long"));
                log_event_fail ("alert", "Alert", NULL, "created");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_alert",
                                    "Permission denied"));
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_alert"));
                log_event_fail ("alert", "Alert", modify_alert_data->alert_id,
                                "modified");
                break;
            }

          modify_alert_data_reset (modify_alert_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_ALERT, COMMENT);
      CLOSE (CLIENT_MODIFY_ALERT, NAME);
      CLOSE (CLIENT_MODIFY_ALERT, FILTER);
      CLOSE (CLIENT_MODIFY_ALERT, EVENT);
      CLOSE (CLIENT_MODIFY_ALERT, CONDITION);
      CLOSE (CLIENT_MODIFY_ALERT, METHOD);

      case CLIENT_MODIFY_ALERT_EVENT_DATA:
        {
          gchar *string;

          assert (strcasecmp ("DATA", element_name) == 0);
          assert (modify_alert_data->event_data);
          assert (modify_alert_data->part_data);
          assert (modify_alert_data->part_name);

          string = g_strconcat (modify_alert_data->part_name,
                                "0",
                                modify_alert_data->part_data,
                                NULL);
          string[strlen (modify_alert_data->part_name)] = '\0';
          array_add (modify_alert_data->event_data, string);

          openvas_free_string_var (&modify_alert_data->part_data);
          openvas_free_string_var (&modify_alert_data->part_name);
          openvas_append_string (&modify_alert_data->part_data, "");
          openvas_append_string (&modify_alert_data->part_name, "");
          set_client_state (CLIENT_MODIFY_ALERT_EVENT);
          break;
        }
      CLOSE (CLIENT_MODIFY_ALERT_EVENT_DATA, NAME);

      case CLIENT_MODIFY_ALERT_CONDITION_DATA:
        {
          gchar *string;

          assert (strcasecmp ("DATA", element_name) == 0);
          assert (modify_alert_data->condition_data);
          assert (modify_alert_data->part_data);
          assert (modify_alert_data->part_name);

          string = g_strconcat (modify_alert_data->part_name,
                                "0",
                                modify_alert_data->part_data,
                                NULL);
          string[strlen (modify_alert_data->part_name)] = '\0';
          array_add (modify_alert_data->condition_data, string);

          openvas_free_string_var (&modify_alert_data->part_data);
          openvas_free_string_var (&modify_alert_data->part_name);
          openvas_append_string (&modify_alert_data->part_data, "");
          openvas_append_string (&modify_alert_data->part_name, "");
          set_client_state (CLIENT_MODIFY_ALERT_CONDITION);
          break;
        }
      CLOSE (CLIENT_MODIFY_ALERT_CONDITION_DATA, NAME);

      case CLIENT_MODIFY_ALERT_METHOD_DATA:
        {
          gchar *string;

          assert (strcasecmp ("DATA", element_name) == 0);
          assert (modify_alert_data->method_data);
          assert (modify_alert_data->part_data);
          assert (modify_alert_data->part_name);

          string = g_strconcat (modify_alert_data->part_name,
                                "0",
                                modify_alert_data->part_data,
                                NULL);
          string[strlen (modify_alert_data->part_name)] = '\0';
          array_add (modify_alert_data->method_data, string);

          openvas_free_string_var (&modify_alert_data->part_data);
          openvas_free_string_var (&modify_alert_data->part_name);
          openvas_append_string (&modify_alert_data->part_data, "");
          openvas_append_string (&modify_alert_data->part_name, "");
          set_client_state (CLIENT_MODIFY_ALERT_METHOD);
          break;
        }
      CLOSE (CLIENT_MODIFY_ALERT_METHOD_DATA, NAME);

      case CLIENT_MODIFY_AUTH:
        {
          GKeyFile *key_file;

          assert (strcasecmp ("MODIFY_AUTH", element_name) == 0);

          if (user_may ("modify_auth") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_auth",
                                  "Permission denied"));
              modify_auth_data_reset (modify_auth_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          key_file = g_key_file_new ();

          /* Lets output the data for now. */
          GSList *item = modify_auth_data->groups;
          while (item)
            {
              auth_group_t *auth_group;
              gchar *group;
              GSList *setting;

              auth_group = (auth_group_t *) item->data;
              group = auth_group->group_name;
              if (group == NULL)
                {
                  SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                           ("modify_auth",
                                            "GROUP requires a name attribute"));
                  g_key_file_free (key_file);
                  set_client_state (CLIENT_AUTHENTIC);
                  modify_auth_data_reset (modify_auth_data);
                  break;
                }
              setting = auth_group->settings;
              while (setting)
                {
                  auth_conf_setting_t *kvp =
                    (auth_conf_setting_t *) setting->data;
                  g_key_file_set_value (key_file, group, kvp->key, kvp->value);
                  setting = g_slist_next (setting);
                }
              item = g_slist_next (item);
            }

          /** @todo Implement sighup in and send to openvas-manager in order
            *       for the changed config to take effect. */
            // FIX
          switch (openvas_auth_write_config (key_file))
            {
            case 0:
              SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_auth"));
              break;
            case 1:
              SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                      ("modify_auth",
                                       "Valid authdn required"));
              break;
            default:
            case -1:
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_auth"));
              break;
            }

          g_key_file_free (key_file);
          modify_auth_data_reset (modify_auth_data);
          set_client_state (CLIENT_AUTHENTIC);

          break;
        }

      case CLIENT_MODIFY_AUTH_GROUP:
        {
          assert (strcasecmp ("GROUP", element_name) == 0);

          /* Add settings to group. */
          if (modify_auth_data->curr_group_settings)
            {
              auth_group_t *new_group;
              assert (modify_auth_data->groups);
              new_group = modify_auth_data->groups->data;
              assert (new_group);
              new_group->settings = modify_auth_data->curr_group_settings;
            }

          modify_auth_data->curr_group_settings = NULL;
          set_client_state (CLIENT_MODIFY_AUTH);
          break;
        }

      case CLIENT_MODIFY_AUTH_GROUP_AUTHCONFSETTING:
        {
          set_client_state (CLIENT_MODIFY_AUTH_GROUP);
          break;
        }

      case CLIENT_MODIFY_CONFIG:
        {
          config_t config;

          if (user_may ("modify_config") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_config",
                                  "Permission denied"));
              modify_config_data_reset (modify_config_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (modify_config_data->config_id == NULL
              || strlen (modify_config_data->config_id) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_config",
                                "MODIFY_CONFIG requires a config_id"
                                " attribute"));
          else if ((modify_config_data->nvt_selection_family
                    /* This array implies FAMILY_SELECTION. */
                    && modify_config_data->families_static_all)
                   || ((modify_config_data->nvt_selection_family
                        || modify_config_data->families_static_all)
                       && (modify_config_data->preference_name
                           || modify_config_data->preference_value
                           || modify_config_data->preference_nvt_oid)))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_config",
                                "MODIFY_CONFIG requires either a PREFERENCE or"
                                " an NVT_SELECTION or a FAMILY_SELECTION"));
          else if (find_config (modify_config_data->config_id, &config))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_config"));
          else if (config == 0)
            {
              if (send_find_error_to_client ("modify_config", "config",
                                             modify_config_data->config_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (modify_config_data->nvt_selection_family)
            {
              switch (manage_set_config_nvts
                       (config,
                        modify_config_data->nvt_selection_family,
                        modify_config_data->nvt_selection))
                {
                  case 0:
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_config"));
                    log_event ("config", "Scan config",
                               modify_config_data->config_id, "modified");
                    break;
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_config", "Config is in use"));
                    log_event_fail ("config", "Scan Config",
                                    modify_config_data->config_id, "modified");
                    break;
#if 0
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_config",
                                        "MODIFY_CONFIG PREFERENCE requires at"
                                        " least one of the VALUE and NVT"
                                        " elements"));
                    break;
#endif
                  default:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("modify_config"));
                    log_event_fail ("config", "Scan Config",
                                    modify_config_data->config_id, "modified");
                    break;
                }
            }
          else if (modify_config_data->families_static_all)
            {
              /* There was a FAMILY_SELECTION. */

              switch (manage_set_config_families
                       (config,
                        modify_config_data->families_growing_all,
                        modify_config_data->families_static_all,
                        modify_config_data->families_growing_empty,
                        modify_config_data->family_selection_growing))
                {
                  case 0:
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_config"));
                    log_event ("config", "Scan config",
                               modify_config_data->config_id, "modified");
                    break;
                  case 1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_config", "Config is in use"));
                    log_event_fail ("config", "Scan Config",
                                    modify_config_data->config_id, "modified");
                    break;
#if 0
                  case -1:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_config",
                                        "MODIFY_CONFIG PREFERENCE requires at"
                                        " least one of the VALUE and NVT"
                                        " elements"));
                    break;
#endif
                  default:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_INTERNAL_ERROR ("modify_config"));
                    log_event_fail ("config", "Scan Config",
                                    modify_config_data->config_id, "modified");
                    break;
                }
            }
          else if (modify_config_data->name && modify_config_data->comment)
            switch (manage_set_config_name_comment (config,
                                                    modify_config_data->name,
                                                    modify_config_data->comment))
              {
                case 0:
                  SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_config"));
                  break;
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("modify_config",
                                      "MODIFY_CONFIG name must be unique"));
                  break;
                case -1:
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_config"));
                  break;
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("modify_config"));
                  break;
              }
          else if (modify_config_data->name)
            switch (manage_set_config_name (config, modify_config_data->name))
              {
                case 0:
                  SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_config"));
                  break;
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("modify_config",
                                      "MODIFY_CONFIG name must be unique"));
                  break;
                case -1:
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_config"));
                  break;
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("modify_config"));
                  break;
              }
          else if (modify_config_data->comment)
            switch (manage_set_config_comment (config,
                                               modify_config_data->comment))
              {
                case 0:
                  SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_config"));
                  break;
                case -1:
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_config"));
                  break;
                default:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("modify_config"));
                  break;
              }
          else if (modify_config_data->preference_name == NULL
                   || strlen (modify_config_data->preference_name) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_config",
                                "MODIFY_CONFIG PREFERENCE requires a NAME"
                                " element"));
          else switch (manage_set_config_preference
                        (config,
                         modify_config_data->preference_nvt_oid,
                         modify_config_data->preference_name,
                         modify_config_data->preference_value))
            {
              case 0:
                SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_config"));
                break;
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_config", "Config is in use"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_config", "Empty radio value"));
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_config"));
                break;
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_config"));
                break;
            }
        }
        modify_config_data_reset (modify_config_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;
      CLOSE (CLIENT_MODIFY_CONFIG, COMMENT);
      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION:
        assert (strcasecmp ("FAMILY_SELECTION", element_name) == 0);
        assert (modify_config_data->families_growing_all);
        assert (modify_config_data->families_static_all);
        assert (modify_config_data->families_growing_empty);
        array_terminate (modify_config_data->families_growing_all);
        array_terminate (modify_config_data->families_static_all);
        array_terminate (modify_config_data->families_growing_empty);
        set_client_state (CLIENT_MODIFY_CONFIG);
        break;
      CLOSE (CLIENT_MODIFY_CONFIG, NAME);
      case CLIENT_MODIFY_CONFIG_NVT_SELECTION:
        assert (strcasecmp ("NVT_SELECTION", element_name) == 0);
        assert (modify_config_data->nvt_selection);
        array_terminate (modify_config_data->nvt_selection);
        set_client_state (CLIENT_MODIFY_CONFIG);
        break;
      CLOSE (CLIENT_MODIFY_CONFIG, PREFERENCE);

      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY:
        assert (strcasecmp ("FAMILY", element_name) == 0);
        if (modify_config_data->family_selection_family_name)
          {
            if (modify_config_data->family_selection_family_growing)
              {
                if (modify_config_data->family_selection_family_all)
                  /* Growing 1 and select all 1. */
                  array_add (modify_config_data->families_growing_all,
                             modify_config_data->family_selection_family_name);
                else
                  /* Growing 1 and select all 0. */
                  array_add (modify_config_data->families_growing_empty,
                             modify_config_data->family_selection_family_name);
              }
            else
              {
                if (modify_config_data->family_selection_family_all)
                  /* Growing 0 and select all 1. */
                  array_add (modify_config_data->families_static_all,
                             modify_config_data->family_selection_family_name);
                /* Else growing 0 and select all 0. */
              }
          }
        modify_config_data->family_selection_family_name = NULL;
        set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION);
        break;
      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_GROWING:
        assert (strcasecmp ("GROWING", element_name) == 0);
        if (modify_config_data->family_selection_growing_text)
          {
            modify_config_data->family_selection_growing
             = atoi (modify_config_data->family_selection_growing_text);
            openvas_free_string_var
             (&modify_config_data->family_selection_growing_text);
          }
        else
          modify_config_data->family_selection_growing = 0;
        set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION);
        break;

      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_ALL:
        assert (strcasecmp ("ALL", element_name) == 0);
        if (modify_config_data->family_selection_family_all_text)
          {
            modify_config_data->family_selection_family_all
             = atoi (modify_config_data->family_selection_family_all_text);
            openvas_free_string_var
             (&modify_config_data->family_selection_family_all_text);
          }
        else
          modify_config_data->family_selection_family_all = 0;
        set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY);
        break;
      CLOSE (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY, NAME);
      case CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY_GROWING:
        assert (strcasecmp ("GROWING", element_name) == 0);
        if (modify_config_data->family_selection_family_growing_text)
          {
            modify_config_data->family_selection_family_growing
             = atoi (modify_config_data->family_selection_family_growing_text);
            openvas_free_string_var
             (&modify_config_data->family_selection_family_growing_text);
          }
        else
          modify_config_data->family_selection_family_growing = 0;
        set_client_state (CLIENT_MODIFY_CONFIG_FAMILY_SELECTION_FAMILY);
        break;

      CLOSE (CLIENT_MODIFY_CONFIG_NVT_SELECTION, FAMILY);
      case CLIENT_MODIFY_CONFIG_NVT_SELECTION_NVT:
        assert (strcasecmp ("NVT", element_name) == 0);
        if (modify_config_data->nvt_selection_nvt_oid)
          array_add (modify_config_data->nvt_selection,
                     modify_config_data->nvt_selection_nvt_oid);
        modify_config_data->nvt_selection_nvt_oid = NULL;
        set_client_state (CLIENT_MODIFY_CONFIG_NVT_SELECTION);
        break;

      CLOSE (CLIENT_MODIFY_CONFIG_PREFERENCE, NAME);
      CLOSE (CLIENT_MODIFY_CONFIG_PREFERENCE, NVT);
      case CLIENT_MODIFY_CONFIG_PREFERENCE_VALUE:
        assert (strcasecmp ("VALUE", element_name) == 0);
        /* Init, so it's the empty string when the value is empty. */
        openvas_append_string (&modify_config_data->preference_value, "");
        set_client_state (CLIENT_MODIFY_CONFIG_PREFERENCE);
        break;

      case CLIENT_MODIFY_FILTER:
        {
          assert (strcasecmp ("MODIFY_FILTER", element_name) == 0);

          switch (modify_filter
                   (modify_filter_data->filter_id,
                    modify_filter_data->name,
                    modify_filter_data->comment,
                    modify_filter_data->term,
                    modify_filter_data->type))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_filter"));
                log_event ("filter", "Filter", modify_filter_data->filter_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_filter", "filter",
                                               modify_filter_data->filter_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_filter",
                                    "Filter with new name exists already"));
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_filter",
                                    "Error in type name"));
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_filter",
                                    "MODIFY_FILTER requires a filter_id"));
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_filter",
                                    "Filter is used by an alert so type must be"
                                    " 'result' if specified"));
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_filter",
                                    "Permission denied"));
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_filter"));
                log_event_fail ("filter", "Filter",
                                modify_filter_data->filter_id, "modified");
                break;
            }

          modify_filter_data_reset (modify_filter_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_FILTER, COMMENT);
      CLOSE (CLIENT_MODIFY_FILTER, NAME);
      CLOSE (CLIENT_MODIFY_FILTER, TYPE);
      CLOSE (CLIENT_MODIFY_FILTER, TERM);

      case CLIENT_MODIFY_GROUP:
        {
          assert (strcasecmp ("MODIFY_GROUP", element_name) == 0);

          switch (modify_group
                   (modify_group_data->group_id,
                    modify_group_data->name,
                    modify_group_data->comment,
                    modify_group_data->users))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_group"));
                log_event ("group", "Group", modify_group_data->group_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_group", "group",
                                               modify_group_data->group_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_group",
                                    "Failed to find user"));
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_group",
                                    "MODIFY_GROUP requires a group_id"
                                    " attribute"));
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_group",
                                    "Error in user name"));
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_group",
                                    "Group with new name exists already"));
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_group",
                                    "Permission denied"));
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_group"));
                log_event_fail ("group", "Group",
                                modify_group_data->group_id, "modified");
                break;
            }

          modify_group_data_reset (modify_group_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_GROUP, COMMENT);
      CLOSE (CLIENT_MODIFY_GROUP, NAME);
      CLOSE (CLIENT_MODIFY_GROUP, USERS);

      case CLIENT_MODIFY_LSC_CREDENTIAL:
        {
          assert (strcasecmp ("MODIFY_LSC_CREDENTIAL", element_name) == 0);

          switch (modify_lsc_credential
                   (modify_lsc_credential_data->lsc_credential_id,
                    modify_lsc_credential_data->name,
                    modify_lsc_credential_data->comment,
                    modify_lsc_credential_data->login,
                    modify_lsc_credential_data->password))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_lsc_credential"));
                log_event ("lsc_credential", "LSC Credential",
                           modify_lsc_credential_data->lsc_credential_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client
                     ("modify_lsc_credential", "lsc_credential",
                      modify_lsc_credential_data->lsc_credential_id,
                      omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("lsc_credential", "LSC Credential",
                                modify_lsc_credential_data->lsc_credential_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_lsc_credential",
                                    "lsc_credential with new name"
                                    " exists already"));
                log_event_fail ("lsc_credential", "LSC Credential",
                                modify_lsc_credential_data->lsc_credential_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_lsc_credential",
                                    "MODIFY_lsc_credential requires a"
                                    " lsc_credential_id"));
                log_event_fail ("lsc_credential", "LSC Credential",
                                modify_lsc_credential_data->lsc_credential_id,
                                "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_lsc_credential",
                                    "Attempt to change login or password of"
                                    " packaged LSC credential"));
                log_event_fail ("lsc_credential", "LSC Credential",
                                modify_lsc_credential_data->lsc_credential_id,
                                "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_lsc_credential",
                                    "Permission denied"));
                log_event_fail ("lsc_credential", "LSC Credential",
                                modify_lsc_credential_data->lsc_credential_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_lsc_credential"));
                log_event_fail ("lsc_credential", "LSC Credential",
                                modify_lsc_credential_data->lsc_credential_id,
                                "modified");
                break;
            }

          modify_lsc_credential_data_reset (modify_lsc_credential_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
        modify_lsc_credential_data_reset (modify_lsc_credential_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;
      CLOSE (CLIENT_MODIFY_LSC_CREDENTIAL, NAME);
      CLOSE (CLIENT_MODIFY_LSC_CREDENTIAL, COMMENT);
      CLOSE (CLIENT_MODIFY_LSC_CREDENTIAL, LOGIN);
      CLOSE (CLIENT_MODIFY_LSC_CREDENTIAL, PASSWORD);

      case CLIENT_MODIFY_NOTE:
        {
          task_t task = 0;
          result_t result = 0;
          note_t note = 0;

          assert (strcasecmp ("MODIFY_NOTE", element_name) == 0);

          if (user_may ("modify_note") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_note",
                                  "Permission denied"));
              modify_note_data_reset (modify_note_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (modify_note_data->note_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_note",
                                "MODIFY_NOTE requires a note_id attribute"));
          else if (modify_note_data->text == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_note",
                                "MODIFY_NOTE requires a TEXT entity"));
          // FIX move find_* down into manage_sql.c modify_note
          else if (find_note_with_permission (modify_note_data->note_id, &note,
                                              "modify_note"))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_note"));
          else if (note == 0)
            {
              if (send_find_error_to_client ("modify_note", "note",
                                             modify_note_data->note_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (modify_note_data->task_id
                   && find_task_with_permission (modify_note_data->task_id,
                                                 &task,
                                                 NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_note"));
          else if (modify_note_data->task_id
                   && task == 0
                   && find_trash_task_with_permission (modify_note_data
                                                         ->task_id,
                                                       &task,
                                                       NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_note"));
          else if (modify_note_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("modify_note", "task",
                                             modify_note_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (modify_note_data->result_id
                   && find_result_with_permission (modify_note_data->result_id,
                                                   &result,
                                                   NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_note"));
          else if (modify_note_data->result_id && result == 0)
            {
              if (send_find_error_to_client ("modify_note", "result",
                                             modify_note_data->result_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else switch (modify_note (note,
                                    modify_note_data->active,
                                    modify_note_data->text,
                                    modify_note_data->hosts,
                                    modify_note_data->port,
                                    modify_note_data->severity,
                                    modify_note_data->threat,
                                    task,
                                    result))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_note"));
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_note"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_note",
                                    "Error in port specification"));
                log_event_fail ("note", "Note", modify_note_data->note_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_note",
                                    "Error in severity specification"));
                log_event_fail ("note", "Note", modify_note_data->note_id,
                                "modified");
                break;
              default:
                assert (0);
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_note"));
                break;
            }
          modify_note_data_reset (modify_note_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_NOTE, ACTIVE);
      CLOSE (CLIENT_MODIFY_NOTE, HOSTS);
      CLOSE (CLIENT_MODIFY_NOTE, PORT);
      CLOSE (CLIENT_MODIFY_NOTE, RESULT);
      CLOSE (CLIENT_MODIFY_NOTE, SEVERITY);
      CLOSE (CLIENT_MODIFY_NOTE, TASK);
      CLOSE (CLIENT_MODIFY_NOTE, TEXT);
      CLOSE (CLIENT_MODIFY_NOTE, THREAT);

      case CLIENT_MODIFY_OVERRIDE:
        {
          task_t task = 0;
          result_t result = 0;
          override_t override = 0;

          assert (strcasecmp ("MODIFY_OVERRIDE", element_name) == 0);

          if (user_may ("modify_override") == 0)
            {
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_override",
                                  "Permission denied"));
              modify_override_data_reset (modify_override_data);
              set_client_state (CLIENT_AUTHENTIC);
              break;
            }

          if (modify_override_data->override_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_override",
                                "MODIFY_OVERRIDE requires a override_id attribute"));
          else if (modify_override_data->text == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_override",
                                "MODIFY_OVERRIDE requires a TEXT entity"));
          else if (find_override_with_permission
                    (modify_override_data->override_id, &override,
                     "modify_override"))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_override"));
          else if (override == 0)
            {
              if (send_find_error_to_client ("modify_override", "override",
                                             modify_override_data->override_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (modify_override_data->task_id
                   && find_task_with_permission (modify_override_data->task_id,
                                                 &task,
                                                 NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_override"));
          else if (modify_override_data->task_id
                   && task == 0
                   && find_trash_task_with_permission (modify_override_data
                                                         ->task_id,
                                                       &task,
                                                       NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_override"));
          else if (modify_override_data->task_id && task == 0)
            {
              if (send_find_error_to_client ("modify_override", "task",
                                             modify_override_data->task_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else if (modify_override_data->result_id
                   && find_result_with_permission (modify_override_data->result_id,
                                                   &result,
                                                   NULL))
            SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_override"));
          else if (modify_override_data->result_id && result == 0)
            {
              if (send_find_error_to_client ("modify_override", "result",
                                             modify_override_data->result_id,
                                             omp_parser))
                {
                  error_send_to_client (error);
                  return;
                }
            }
          else switch (modify_override (override,
                                        modify_override_data->active,
                                        modify_override_data->text,
                                        modify_override_data->hosts,
                                        modify_override_data->port,
                                        modify_override_data->threat,
                                        modify_override_data->new_threat,
                                        modify_override_data->severity,
                                        modify_override_data->new_severity,
                                        task,
                                        result))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_override"));
                break;
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_override",
                                    "ACTIVE must be an integer >= -2"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_override",
                                    "Error in port specification"));
                log_event_fail ("override", "Override",
                                modify_override_data->override_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_override",
                                    "Error in severity specification"));
                log_event_fail ("override", "Override",
                                modify_override_data->override_id,
                                "modified");
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_override"));
                break;
              default:
                assert (0);
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_override"));
                break;
            }
          modify_override_data_reset (modify_override_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_OVERRIDE, ACTIVE);
      CLOSE (CLIENT_MODIFY_OVERRIDE, HOSTS);
      CLOSE (CLIENT_MODIFY_OVERRIDE, NEW_SEVERITY);
      CLOSE (CLIENT_MODIFY_OVERRIDE, NEW_THREAT);
      CLOSE (CLIENT_MODIFY_OVERRIDE, PORT);
      CLOSE (CLIENT_MODIFY_OVERRIDE, RESULT);
      CLOSE (CLIENT_MODIFY_OVERRIDE, SEVERITY);
      CLOSE (CLIENT_MODIFY_OVERRIDE, TASK);
      CLOSE (CLIENT_MODIFY_OVERRIDE, TEXT);
      CLOSE (CLIENT_MODIFY_OVERRIDE, THREAT);

      case CLIENT_MODIFY_PERMISSION:
        {
          assert (strcasecmp ("MODIFY_PERMISSION", element_name) == 0);

          if (modify_permission_data->permission_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_permission",
                                "MODIFY_PERMISSION requires a permission_id attribute"));
          else switch (modify_permission
                        (modify_permission_data->permission_id,
                         modify_permission_data->name,
                         modify_permission_data->comment,
                         modify_permission_data->resource_id,
                         modify_permission_data->resource_type,
                         modify_permission_data->subject_type,
                         modify_permission_data->subject_id))
            {
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "Permission exists already"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 2:
                if (send_find_error_to_client
                     ("modify_permission", "subject",
                      modify_permission_data->subject_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 3:
                if (send_find_error_to_client
                     ("modify_permission", "resource",
                      modify_permission_data->resource_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "MODIFY_PERMISSION requires a PERMISSION"
                                    " ID"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "Error in RESOURCE"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 6:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "Error in SUBJECT"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 7:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "Error in NAME"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 8:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "NAME required to find resource"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case 9:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "Permission does not accept a resource"));
                log_event_fail ("permission", "Permission", NULL, "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_permission",
                                    "Permission denied"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_permission"));
                log_event_fail ("permission", "Permission",
                                modify_permission_data->permission_id,
                                "modified");
                break;
              default:
                {
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_permission"));
                  log_event ("permission", "Permission",
                             modify_permission_data->permission_id, "modified");
                  break;
                }
            }

          modify_permission_data_reset (modify_permission_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_PERMISSION, COMMENT);
      CLOSE (CLIENT_MODIFY_PERMISSION, SUBJECT);
      CLOSE (CLIENT_MODIFY_PERMISSION_SUBJECT, TYPE);
      CLOSE (CLIENT_MODIFY_PERMISSION, NAME);
      CLOSE (CLIENT_MODIFY_PERMISSION, RESOURCE);
      CLOSE (CLIENT_MODIFY_PERMISSION_RESOURCE, TYPE);

      case CLIENT_MODIFY_PORT_LIST:
        {
          assert (strcasecmp ("MODIFY_PORT_LIST", element_name) == 0);

          switch (modify_port_list
                   (modify_port_list_data->port_list_id,
                    modify_port_list_data->name,
                    modify_port_list_data->comment))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_port_list"));
                log_event ("port_list", "Port List",
                           modify_port_list_data->port_list_id, "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_port_list", "port_list",
                                               modify_port_list_data->port_list_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("port_list", "Port List",
                                modify_port_list_data->port_list_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_port_list",
                                    "Port List with new name exists already"));
                log_event_fail ("port_list", "Port List",
                                modify_port_list_data->port_list_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_port_list",
                                    "modify_port_list requires a port_list_id"));
                log_event_fail ("port_list", "Port List",
                                modify_port_list_data->port_list_id,
                                "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_port_list",
                                    "Permission denied"));
                log_event_fail ("port_list", "Port List",
                                modify_port_list_data->port_list_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_port_list"));
                log_event_fail ("port_list", "Port List",
                                modify_port_list_data->port_list_id,
                                "modified");
                break;
            }

          modify_port_list_data_reset (modify_port_list_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_PORT_LIST, COMMENT);
      CLOSE (CLIENT_MODIFY_PORT_LIST, NAME);

      case CLIENT_MODIFY_REPORT:
        {
          assert (strcasecmp ("MODIFY_REPORT", element_name) == 0);

          switch (modify_report
                   (modify_report_data->report_id,
                    modify_report_data->comment))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_report"));
                log_event ("report", "Report", modify_report_data->report_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_report", "report",
                                               modify_report_data->report_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("report", "Report",
                                modify_report_data->report_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_report",
                                    "MODIFY_report requires a report_id"));
                log_event_fail ("report", "Report",
                                modify_report_data->report_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX
                  ("modify_report",
                   "MODIFY_REPORT requires a COMMENT element"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_report",
                                    "Permission denied"));
                log_event_fail ("report", "Report",
                                modify_report_data->report_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_report"));
                log_event_fail ("report", "Report",
                                modify_report_data->report_id,
                                "modified");
                break;
            }

          modify_report_data_reset (modify_report_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_REPORT, COMMENT);

      case CLIENT_MODIFY_REPORT_FORMAT:
        {
          assert (strcasecmp ("MODIFY_REPORT_FORMAT", element_name) == 0);

          switch (modify_report_format
                   (modify_report_format_data->report_format_id,
                    modify_report_format_data->name,
                    modify_report_format_data->summary,
                    modify_report_format_data->active,
                    modify_report_format_data->param_name,
                    modify_report_format_data->param_value))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_report_format"));
                log_event ("report_format", "Report Format",
                           modify_report_format_data->report_format_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client
                     ("modify_report_format", "report_format",
                      modify_report_format_data->report_format_id,
                      omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("report_format", "Report Format",
                                modify_report_format_data->report_format_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX
                   ("modify_report_format",
                    "MODIFY_report_format requires a report_format_id"));
                log_event_fail ("report_format", "Report Format",
                                modify_report_format_data->report_format_id,
                                "modified");
                break;
              case 3:
                if (send_find_error_to_client
                     ("modify_report_format", "report format param",
                      modify_report_format_data->param_name, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("report_format", "Report Format",
                                modify_report_format_data->report_format_id,
                                "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_report_format",
                                    "Parameter validation failed"));
                log_event_fail ("report_format", "Report Format",
                                modify_report_format_data->report_format_id,
                                "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_report_format",
                                    "Permission denied"));
                log_event_fail ("report_format", "Report Format",
                                modify_report_format_data->report_format_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR
                                         ("modify_report_format"));
                log_event_fail ("report_format", "Report Format",
                                modify_report_format_data->report_format_id,
                                "modified");
                break;
            }

          modify_report_format_data_reset (modify_report_format_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_REPORT_FORMAT, ACTIVE);
      CLOSE (CLIENT_MODIFY_REPORT_FORMAT, NAME);
      CLOSE (CLIENT_MODIFY_REPORT_FORMAT, SUMMARY);
      CLOSE (CLIENT_MODIFY_REPORT_FORMAT, PARAM);
      CLOSE (CLIENT_MODIFY_REPORT_FORMAT_PARAM, NAME);
      CLOSE (CLIENT_MODIFY_REPORT_FORMAT_PARAM, VALUE);

      case CLIENT_MODIFY_ROLE:
        {
          assert (strcasecmp ("MODIFY_ROLE", element_name) == 0);

          switch (modify_role
                   (modify_role_data->role_id,
                    modify_role_data->name,
                    modify_role_data->comment,
                    modify_role_data->users))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_role"));
                log_event ("role", "Role", modify_role_data->role_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_role", "role",
                                               modify_role_data->role_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_role",
                                    "Failed to find user"));
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_role",
                                    "MODIFY_ROLE requires a role_id"
                                    " attribute"));
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_role",
                                    "Error in user name"));
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_role",
                                    "Role with new name exists already"));
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_role",
                                    "Permission denied"));
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_role"));
                log_event_fail ("role", "Role",
                                modify_role_data->role_id, "modified");
                break;
            }

          modify_role_data_reset (modify_role_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_ROLE, COMMENT);
      CLOSE (CLIENT_MODIFY_ROLE, NAME);
      CLOSE (CLIENT_MODIFY_ROLE, USERS);

      case CLIENT_MODIFY_SCANNER:
        assert (strcasecmp ("MODIFY_SCANNER", element_name) == 0);
        return handle_modify_scanner (omp_parser, error);
      CLOSE (CLIENT_MODIFY_SCANNER, TYPE);
      CLOSE (CLIENT_MODIFY_SCANNER, PORT);
      CLOSE (CLIENT_MODIFY_SCANNER, HOST);
      CLOSE (CLIENT_MODIFY_SCANNER, COMMENT);
      CLOSE (CLIENT_MODIFY_SCANNER, NAME);
      CLOSE (CLIENT_MODIFY_SCANNER, CA_PUB);
      CLOSE (CLIENT_MODIFY_SCANNER, KEY_PUB);
      CLOSE (CLIENT_MODIFY_SCANNER, KEY_PRIV);

      case CLIENT_MODIFY_SCHEDULE:
        {
          time_t first_time, period, period_months, duration;

          assert (strcasecmp ("MODIFY_SCHEDULE", element_name) == 0);

          period_months = 0;

          /* Only change schedule "first time" if given. */
          first_time = modify_schedule_data->first_time_hour
                        || modify_schedule_data->first_time_minute
                        || modify_schedule_data->first_time_day_of_month
                        || modify_schedule_data->first_time_month
                        || modify_schedule_data->first_time_year;

          if (first_time
              && ((first_time
                    = time_from_strings
                       (modify_schedule_data->first_time_hour,
                        modify_schedule_data->first_time_minute,
                        modify_schedule_data->first_time_day_of_month,
                        modify_schedule_data->first_time_month,
                        modify_schedule_data->first_time_year,
                        modify_schedule_data->timezone))
                  == -1))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "Failed to create time from FIRST_TIME"
                                " elements"));
          else if ((period = interval_from_strings
                              (modify_schedule_data->period,
                               modify_schedule_data->period_unit,
                               &period_months))
                   == -3)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "PERIOD out of range"));
          else if (period < -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "Failed to create interval from PERIOD"));
          else if ((duration = interval_from_strings
                                (modify_schedule_data->duration,
                                 modify_schedule_data->duration_unit,
                                 NULL))
                   == -3)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "DURATION out of range"));
          else if (duration < -1)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "Failed to create interval from DURATION"));
#if 0
          /* The actual time of a period in months can vary, so it's extremely
           * hard to do this check.  The schedule will still work fine if the
           * duration is longer than the period. */
          else if (period_months
                   && (duration > (period_months * 60 * 60 * 24 * 28)))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "Duration too long for number of months"));
#endif
          else if (period && (duration > period))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_schedule",
                                "Duration is longer than period"));
          else switch (modify_schedule
                        (modify_schedule_data->schedule_id,
                         modify_schedule_data->name,
                         modify_schedule_data->comment,
                         first_time,
                         period == -1 ? 0 : period,
                         period_months,
                         duration == -1 ? 0 : duration,
                         modify_schedule_data->timezone))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_schedule"));
                log_event ("schedule", "Schedule",
                           modify_schedule_data->schedule_id, "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_schedule", "schedule",
                                               modify_schedule_data->schedule_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("schedule", "Schedule",
                                modify_schedule_data->schedule_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_schedule",
                                    "Schedule with new name exists already"));
                log_event_fail ("schedule", "Schedule",
                                modify_schedule_data->schedule_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_schedule",
                                    "Error in type name"));
                log_event_fail ("schedule", "Schedule",
                                modify_schedule_data->schedule_id,
                                "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_schedule",
                                    "MODIFY_SCHEDULE requires a schedule_id"));
                log_event_fail ("schedule", "Schedule",
                                modify_schedule_data->schedule_id,
                                "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_schedule",
                                    "Permission denied"));
                log_event_fail ("schedule", "Schedule",
                                modify_schedule_data->schedule_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_schedule"));
                log_event_fail ("schedule", "Schedule",
                                modify_schedule_data->schedule_id,
                                "modified");
                break;
            }

          modify_schedule_data_reset (modify_schedule_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_SCHEDULE, COMMENT);
      CLOSE (CLIENT_MODIFY_SCHEDULE, DURATION);
      CLOSE (CLIENT_MODIFY_SCHEDULE, FIRST_TIME);
      CLOSE (CLIENT_MODIFY_SCHEDULE, NAME);
      CLOSE (CLIENT_MODIFY_SCHEDULE, PERIOD);
      CLOSE (CLIENT_MODIFY_SCHEDULE, TIMEZONE);

      CLOSE (CLIENT_MODIFY_SCHEDULE_FIRST_TIME, DAY_OF_MONTH);
      CLOSE (CLIENT_MODIFY_SCHEDULE_FIRST_TIME, HOUR);
      CLOSE (CLIENT_MODIFY_SCHEDULE_FIRST_TIME, MINUTE);
      CLOSE (CLIENT_MODIFY_SCHEDULE_FIRST_TIME, MONTH);
      CLOSE (CLIENT_MODIFY_SCHEDULE_FIRST_TIME, YEAR);

      CLOSE (CLIENT_MODIFY_SCHEDULE_DURATION, UNIT);

      CLOSE (CLIENT_MODIFY_SCHEDULE_PERIOD, UNIT);

      case CLIENT_MODIFY_SETTING:
        {
          gchar *errdesc = NULL;

          if (((modify_setting_data->name == NULL)
               && (modify_setting_data->setting_id == NULL))
              || (modify_setting_data->value == NULL))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_setting",
                                "MODIFY_SETTING requires a NAME or setting_id"
                                " and a VALUE"));
          else switch (modify_setting (modify_setting_data->setting_id,
                                       modify_setting_data->name,
                                       modify_setting_data->value,
                                       &errdesc))
            {
              case 0:
                SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_setting"));
                break;
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_setting",
                                    "Failed to find setting"));
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_setting",
                                    "Value validation failed"));
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_setting",
                                    "Permission denied"));
                break;
              case -1:
                if (errdesc)
                  {
                    char *buf = make_xml_error_syntax ("modify_setting",
                                                       errdesc);
                    SEND_TO_CLIENT_OR_FAIL (buf);
                    g_free (buf);
                    break;
                  }
                /* Fall through.  */
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_setting"));
                break;
            }
          g_free (errdesc);
        }
        modify_setting_data_reset (modify_setting_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;
      CLOSE (CLIENT_MODIFY_SETTING, NAME);
      CLOSE (CLIENT_MODIFY_SETTING, VALUE);

      case CLIENT_MODIFY_SLAVE:
        {
          assert (strcasecmp ("MODIFY_SLAVE", element_name) == 0);

          switch (modify_slave
                   (modify_slave_data->slave_id,
                    modify_slave_data->name,
                    modify_slave_data->comment,
                    modify_slave_data->host,
                    modify_slave_data->port,
                    modify_slave_data->login,
                    modify_slave_data->password))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_slave"));
                log_event ("slave", "Slave", modify_slave_data->slave_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_slave", "slave",
                                               modify_slave_data->slave_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("slave", "Slave",
                                modify_slave_data->slave_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_slave",
                                    "Slave with new name exists already"));
                log_event_fail ("slave", "Slave",
                                modify_slave_data->slave_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_slave",
                                    "MODIFY_SLAVE requires a slave_id"));
                log_event_fail ("slave", "Slave",
                                modify_slave_data->slave_id,
                                "modified");
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_slave",
                                    "Permission denied"));
                log_event_fail ("slave", "Slave",
                                modify_slave_data->slave_id,
                                "modified");
                break;
              default:
              case -1:
                SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_slave"));
                log_event_fail ("slave", "Slave",
                                modify_slave_data->slave_id,
                                "modified");
                break;
            }

          modify_slave_data_reset (modify_slave_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_SLAVE, COMMENT);
      CLOSE (CLIENT_MODIFY_SLAVE, NAME);
      CLOSE (CLIENT_MODIFY_SLAVE, HOST);
      CLOSE (CLIENT_MODIFY_SLAVE, PORT);
      CLOSE (CLIENT_MODIFY_SLAVE, LOGIN);
      CLOSE (CLIENT_MODIFY_SLAVE, PASSWORD);

      case CLIENT_MODIFY_TAG:
        {
          assert (strcasecmp ("MODIFY_TAG", element_name) == 0);

          if (modify_tag_data->tag_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_tag",
                                "MODIFY_TAG requires a tag_id attribute"));
          else if (modify_tag_data->name
                   && strcmp(modify_tag_data->name, "") == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_tag",
                                "name in MODIFY_TAG must be at least one"
                                " character long or omitted completely"));
          else if (modify_tag_data->resource_type &&
                   valid_db_resource_type (modify_tag_data->resource_type) == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_tag",
                                "TYPE in MODIFY_TAG/RESOURCE must be"
                                " a valid resource type."));
          else if (modify_tag_data->resource_type
                   && strcasecmp (modify_tag_data->resource_type, "tag") == 0)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_tag",
                                "TYPE type in MODIFY_TAG/RESOURCE must not"
                                " be 'tag'."));
          else if (modify_tag_data->resource_id
                   && strlen (modify_tag_data->resource_id) > 0
                   && (resource_id_exists (modify_tag_data->resource_type,
                                           modify_tag_data->resource_id) == 0)
                   && (trash_id_exists (modify_tag_data->resource_type,
                                        modify_tag_data->resource_id) == 0))
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_tag",
                                "RESOURCE id must refer to an"
                                " existing resource or be empty."));
          else switch (modify_tag (modify_tag_data->tag_id,
                                   modify_tag_data->name,
                                   modify_tag_data->comment,
                                   modify_tag_data->value,
                                   modify_tag_data->resource_type,
                                   modify_tag_data->resource_id,
                                   modify_tag_data->active))
            {
              case 0:
                SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_tag"));
                log_event ("tag", "Tag", modify_tag_data->tag_id,
                           "modified");
                break;
              case 1:
                if (send_find_error_to_client ("modify_tag", "tag",
                                               modify_tag_data->tag_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                log_event_fail ("tag", "Tag", modify_tag_data->tag_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_tag",
                                    "MODIFY_TAG requires a tag_id"));
                log_event_fail ("tag", "Tag", modify_tag_data->tag_id,
                                "modified");
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_tag",
                                    "Permission denied"));
                log_event_fail ("tag", "Tag", modify_tag_data->tag_id,
                                "modified");
                break;
              default:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_tag"));
                log_event_fail ("tag", "Tag", modify_tag_data->tag_id,
                                "modified");
                break;
            }

          modify_tag_data_reset (modify_tag_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }

      CLOSE (CLIENT_MODIFY_TAG, ACTIVE);
      CLOSE (CLIENT_MODIFY_TAG, RESOURCE);
      CLOSE (CLIENT_MODIFY_TAG, COMMENT);
      CLOSE (CLIENT_MODIFY_TAG, NAME);
      CLOSE (CLIENT_MODIFY_TAG, VALUE);

      CLOSE (CLIENT_MODIFY_TAG_RESOURCE, TYPE);

      case CLIENT_MODIFY_TARGET:
        {
          assert (strcasecmp ("MODIFY_TARGET", element_name) == 0);

          if (modify_target_data->target_id == NULL)
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_target",
                                "MODIFY_TARGET requires a target_id"
                                " attribute"));
          else switch (modify_target
                        (modify_target_data->target_id,
                         modify_target_data->name,
                         modify_target_data->hosts,
                         modify_target_data->exclude_hosts,
                         modify_target_data->comment,
                         modify_target_data->port_list_id,
                         modify_target_data->ssh_lsc_credential_id,
                         modify_target_data->ssh_port,
                         modify_target_data->smb_lsc_credential_id,
                         modify_target_data->esxi_lsc_credential_id,
                         modify_target_data->reverse_lookup_only,
                         modify_target_data->reverse_lookup_unify,
                         modify_target_data->alive_tests))
            {
              case 1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Target exists already"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 2:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Error in host specification"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 3:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Host specification exceeds maximum number"
                                    " of hosts"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 4:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Error in port range"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 5:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Error in SSH port"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 6:
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                if (send_find_error_to_client
                     ("modify_target", "port_list",
                      modify_target_data->port_list_id, omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 7:
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                if (send_find_error_to_client
                     ("modify_target", "LSC credential",
                      modify_target_data->ssh_lsc_credential_id,
                      omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 8:
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                if (send_find_error_to_client
                     ("modify_target", "LSC credential",
                      modify_target_data->smb_lsc_credential_id,
                      omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 9:
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                if (send_find_error_to_client
                     ("modify_target", "target", modify_target_data->target_id,
                      omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 10:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Error in alive test"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 11:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "MODIFY_TARGET name must be at"
                                    " least one character long"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 12:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "MODIFY_TARGET EXCLUDE_HOSTS requires"
                                    " a HOSTS"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 13:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "MODIFY_TARGET with a HOSTS requires an"
                                    " EXCLUDE_HOSTS"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 14:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "MODIFY_TARGET HOSTS must be at least one"
                                    "character long"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 15:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Target is in use"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case 16:
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                if (send_find_error_to_client
                     ("modify_target", "LSC credential",
                      modify_target_data->esxi_lsc_credential_id,
                      omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
                break;
              case 99:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_ERROR_SYNTAX ("modify_target",
                                    "Permission denied"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              case -1:
                SEND_TO_CLIENT_OR_FAIL
                 (XML_INTERNAL_ERROR ("modify_target"));
                log_event_fail ("target", "Target",
                                modify_target_data->target_id,
                                "modified");
                break;
              default:
                {
                  SENDF_TO_CLIENT_OR_FAIL (XML_OK ("modify_target"));
                  log_event ("target", "Target", modify_target_data->target_id,
                             "modified");
                  break;
                }
            }

          modify_target_data_reset (modify_target_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_TARGET, ESXI_LSC_CREDENTIAL);
      CLOSE (CLIENT_MODIFY_TARGET, EXCLUDE_HOSTS);
      CLOSE (CLIENT_MODIFY_TARGET, REVERSE_LOOKUP_ONLY);
      CLOSE (CLIENT_MODIFY_TARGET, REVERSE_LOOKUP_UNIFY);
      CLOSE (CLIENT_MODIFY_TARGET, ALIVE_TESTS);
      CLOSE (CLIENT_MODIFY_TARGET, COMMENT);
      CLOSE (CLIENT_MODIFY_TARGET, HOSTS);
      CLOSE (CLIENT_MODIFY_TARGET, NAME);
      CLOSE (CLIENT_MODIFY_TARGET, PORT_LIST);
      CLOSE (CLIENT_MODIFY_TARGET, SSH_LSC_CREDENTIAL);
      CLOSE (CLIENT_MODIFY_TARGET, SMB_LSC_CREDENTIAL);

      CLOSE (CLIENT_MODIFY_TARGET_SSH_LSC_CREDENTIAL, PORT);

      case CLIENT_MODIFY_TASK:
        assert (strcasecmp ("MODIFY_TASK", element_name) == 0);

        if (user_may ("modify_task") == 0)
          {
            SEND_TO_CLIENT_OR_FAIL
             (XML_ERROR_SYNTAX ("modify_task",
                                "Permission denied"));
            modify_task_data_reset (modify_task_data);
            set_client_state (CLIENT_AUTHENTIC);
            break;
          }

        /** @todo Update to match "create_task (config, target)". */
        if (modify_task_data->task_id)
          {
            task_t task = 0;
            if (find_task_with_permission (modify_task_data->task_id,
                                           &task,
                                           "modify_task"))
              SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_task"));
            else if (task == 0)
              {
                if (send_find_error_to_client ("modify_task", "task",
                                               modify_task_data->task_id,
                                               omp_parser))
                  {
                    error_send_to_client (error);
                    return;
                  }
              }
            else if (modify_task_data->action
                     && (modify_task_data->comment
                         || modify_task_data->alerts->len
                         || modify_task_data->groups->len
                         || modify_task_data->name))
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_task",
                                  "Too many parameters at once"));
            else if ((task_target (task) == 0)
                     && (modify_task_data->alerts->len
                         || modify_task_data->schedule_id
                         || modify_task_data->slave_id))
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_task",
                                  "For container tasks only name, comment and"
                                  " observers can be modified"));
            else if (!modify_task_check_slave_scanner
                       (task, modify_task_data->slave_id,
                        modify_task_data->scanner_id))
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_task",
                                  "Slave used with non-default Scanner."));
            else if (!modify_task_check_config_scanner
                       (task, modify_task_data->config_id,
                        modify_task_data->scanner_id))
              SEND_TO_CLIENT_OR_FAIL
               (XML_ERROR_SYNTAX ("modify_task", "Config and Scanner types mismatch."));
            else if (modify_task_data->action)
              {
                if (modify_task_data->file_name == NULL)
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("modify_task",
                                      "MODIFY_TASK FILE requires a name"
                                      " attribute"));
                else if (strcmp (modify_task_data->action, "update") == 0)
                  {
                    manage_task_update_file (task,
                                             modify_task_data->file_name,
                                             modify_task_data->file
                                              ? modify_task_data->file
                                              : "");
                    log_event ("task", "Task", modify_task_data->task_id,
                               "modified");
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_task"));
                  }
                else if (strcmp (modify_task_data->action, "remove") == 0)
                  {
                    manage_task_remove_file (task, modify_task_data->file_name);
                    log_event ("task", "Task", modify_task_data->task_id,
                               "modified");
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_task"));
                  }
                else
                  {
                    SEND_TO_CLIENT_OR_FAIL
                      (XML_ERROR_SYNTAX ("modify_task",
                                         "MODIFY_TASK action must be"
                                         " \"update\" or \"remove\""));
                    log_event_fail ("task", "Task",
                                    modify_task_data->task_id,
                                    "modified");
                  }
              }
            else
              {
                int fail = 0;

                /** @todo It'd probably be better to allow only one
                 * modification at a time, that is, one parameter or one of
                 * file, name and comment.  Otherwise a syntax error in a
                 * later part of the command would result in an error being
                 * returned while some part of the command actually
                 * succeeded. */

                if (modify_task_data->name)
                  {
                    fail = set_task_parameter (task,
                                               "NAME",
                                               modify_task_data->name);
                    modify_task_data->name = NULL;
                    if (fail)
                      {
                        SEND_TO_CLIENT_OR_FAIL
                          (XML_INTERNAL_ERROR ("modify_task"));
                        log_event_fail ("task", "Task",
                                        modify_task_data->task_id,
                                        "modified");
                      }
                  }

                if (fail == 0 && modify_task_data->comment)
                  {
                    fail = set_task_parameter (task,
                                               "COMMENT",
                                               modify_task_data->comment);
                    modify_task_data->comment = NULL;
                    if (fail)
                      {
                        SEND_TO_CLIENT_OR_FAIL
                          (XML_INTERNAL_ERROR ("modify_task"));
                        log_event_fail ("task", "Task",
                                        modify_task_data->task_id,
                                        "modified");
                      }
                  }

                if (fail == 0 && modify_task_data->config_id)
                  {
                    config_t config = 0;

                    if (strcmp (modify_task_data->config_id, "0") == 0)
                      {
                        /* Leave it as it is. */
                      }
                    else if ((fail = ((task_run_status (task)
                                       != TASK_STATUS_NEW)
                                      && (task_alterable (task) == 0))))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_ERROR_SYNTAX ("modify_task",
                                          "Status must be New to edit Config"));
                    else if ((fail = find_config_with_permission
                                      (modify_task_data->config_id,
                                       &config,
                                       "get_configs")))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_INTERNAL_ERROR ("modify_task"));
                    else if (config == 0)
                      {
                        if (send_find_error_to_client
                             ("modify_task", "config",
                              modify_task_data->config_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        fail = 1;
                      }
                    else
                     set_task_config (task, config);
                  }

                if (fail == 0 && modify_task_data->observers)
                  {
                    fail = set_task_observers (task,
                                               modify_task_data->observers);
                    switch (fail)
                      {
                        case 0:
                          break;
                        case 1:
                        case 2:
                          SEND_TO_CLIENT_OR_FAIL
                            (XML_ERROR_SYNTAX ("modify_task",
                                               "User name error"));
                          log_event_fail ("task", "Task",
                                          modify_task_data->task_id,
                                          "modified");
                          break;
                        case -1:
                        default:
                          SEND_TO_CLIENT_OR_FAIL
                            (XML_INTERNAL_ERROR ("modify_task"));
                          log_event_fail ("task", "Task",
                                          modify_task_data->task_id,
                                          "modified");
                      }
                  }

                if (fail == 0 && modify_task_data->alerts->len)
                  {
                    gchar *fail_alert_id;
                    switch ((fail = set_task_alerts (task,
                                                     modify_task_data->alerts,
                                                     &fail_alert_id)))
                      {
                        case 0:
                          break;
                        case 1:
                          if (send_find_error_to_client
                               ("modify_task", "alert", fail_alert_id,
                                omp_parser))
                            {
                              error_send_to_client (error);
                              return;
                            }
                          fail = 1;
                          log_event_fail ("task", "Task",
                                          modify_task_data->task_id,
                                          "modified");
                          break;
                        case -1:
                        default:
                          SEND_TO_CLIENT_OR_FAIL
                            (XML_INTERNAL_ERROR ("modify_task"));
                          log_event_fail ("task", "Task",
                                          modify_task_data->task_id,
                                          "modified");
                      }
                  }

                if (fail == 0 && modify_task_data->alterable)
                  {
                    if (task_run_status (task) != TASK_STATUS_NEW)
                      {
                        SEND_TO_CLIENT_OR_FAIL
                          (XML_ERROR_SYNTAX ("modify_task",
                                             "Task must be New to modify"
                                             " Alterable state"));
                        fail = 1;
                      }
                    else
                      set_task_alterable (task,
                                          strcmp (modify_task_data->alterable,
                                                  "0"));
                  }

                if (fail == 0 && modify_task_data->groups->len)
                  {
                    gchar *fail_group_id;
                    switch ((fail = set_task_groups (task,
                                                     modify_task_data->groups,
                                                     &fail_group_id)))
                      {
                        case 0:
                          break;
                        case 1:
                          if (send_find_error_to_client
                               ("modify_task", "group", fail_group_id,
                                omp_parser))
                            {
                              error_send_to_client (error);
                              return;
                            }
                          fail = 1;
                          log_event_fail ("task", "Task",
                                          modify_task_data->task_id,
                                          "modified");
                          break;
                        case -1:
                        default:
                          SEND_TO_CLIENT_OR_FAIL
                            (XML_INTERNAL_ERROR ("modify_task"));
                          log_event_fail ("task", "Task",
                                          modify_task_data->task_id,
                                          "modified");
                      }
                  }

                if (fail == 0 && modify_task_data->schedule_id)
                  {
                    schedule_t schedule = 0;
                    int periods;

                    periods = modify_task_data->schedule_periods
                               ? atoi (modify_task_data->schedule_periods)
                               : 0;

                    if (strcmp (modify_task_data->schedule_id, "0") == 0)
                      {
                        set_task_schedule (task, 0, 0);
                      }
                    else if ((fail = find_schedule_with_permission
                                      (modify_task_data->schedule_id,
                                       &schedule,
                                       "get_schedules")))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_INTERNAL_ERROR ("modify_task"));
                    else if (schedule == 0)
                      {
                        if (send_find_error_to_client
                             ("modify_task", "schedule",
                              modify_task_data->schedule_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        fail = 1;
                      }
                    else if (set_task_schedule (task, schedule, periods))
                      {
                        SEND_TO_CLIENT_OR_FAIL
                         (XML_INTERNAL_ERROR ("modify_task"));
                        fail = 1;
                      }
                  }
                else if (fail == 0
                         && modify_task_data->schedule_periods
                         && strlen (modify_task_data->schedule_periods))
                  set_task_schedule_periods
                   (modify_task_data->task_id,
                    atoi (modify_task_data->schedule_periods));

                if (fail == 0 && modify_task_data->slave_id)
                  {
                    slave_t slave = 0;

                    if (strcmp (modify_task_data->slave_id, "0") == 0)
                      {
                        set_task_slave (task, 0);
                      }
                    else if ((fail = find_slave_with_permission
                                      (modify_task_data->slave_id,
                                       &slave,
                                       "get_slaves")))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_INTERNAL_ERROR ("modify_task"));
                    else if (slave == 0)
                      {
                        if (send_find_error_to_client
                             ("modify_task", "slave",
                              modify_task_data->slave_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        fail = 1;
                      }
                    else
                      {
                        set_task_slave (task, slave);
                      }
                  }

                if (fail == 0 && modify_task_data->target_id)
                  {
                    target_t target = 0;

                    if (strcmp (modify_task_data->target_id, "0") == 0)
                      {
                        /* Leave it as it is. */
                      }
                    else if ((fail = (task_run_status (task)
                                      != TASK_STATUS_NEW
                                      && (task_alterable (task) == 0))))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_ERROR_SYNTAX ("modify_task",
                                          "Status must be New to edit Target"));
                    else if ((fail = find_target_with_permission
                                      (modify_task_data->target_id,
                                       &target,
                                       "get_targets")))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_INTERNAL_ERROR ("modify_task"));
                    else if (target == 0)
                      {
                        if (send_find_error_to_client
                             ("modify_task", "target",
                              modify_task_data->target_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        fail = 1;
                      }
                    else
                      set_task_target (task, target);
                  }

                if (fail == 0 && modify_task_data->scanner_id)
                  {
                    scanner_t scanner = 0;

                    if (strcmp (modify_task_data->scanner_id, "0") == 0)
                      {
                        /* Leave it as is. */
                      }
                    else if ((fail = (task_run_status (task)
                                      != TASK_STATUS_NEW
                                      && (task_alterable (task) == 0))))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_ERROR_SYNTAX
                         ("modify_task", "Status must be New to edit Scanner"));
                    else if ((fail = find_scanner_with_permission
                                      (modify_task_data->scanner_id,
                                       &scanner,
                                       "get_scanners")))
                      SEND_TO_CLIENT_OR_FAIL
                       (XML_INTERNAL_ERROR ("modify_task"));
                    else if (scanner == 0)
                      {
                        if (send_find_error_to_client
                             ("modify_task", "scanner",
                              modify_task_data->scanner_id, omp_parser))
                          {
                            error_send_to_client (error);
                            return;
                          }
                        fail = 1;
                      }
                    else
                      set_task_scanner (task, scanner);
                  }

                if (fail == 0 && modify_task_data->preferences)
                  set_task_preferences (task,
                                        modify_task_data->preferences);

                if (fail == 0 && modify_task_data->hosts_ordering)
                  set_task_hosts_ordering (task,
                                           modify_task_data->hosts_ordering);


                if (fail == 0)
                  {
                    log_event ("task", "Task", modify_task_data->task_id,
                               "modified");
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_task"));
                  }
              }
          }
        else
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("modify_task",
                              "MODIFY_TASK requires a task_id attribute"));
        modify_task_data_reset (modify_task_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;
      CLOSE (CLIENT_MODIFY_TASK, ALTERABLE);
      CLOSE (CLIENT_MODIFY_TASK, COMMENT);
      CLOSE (CLIENT_MODIFY_TASK, HOSTS_ORDERING);
      CLOSE (CLIENT_MODIFY_TASK, SCANNER);
      CLOSE (CLIENT_MODIFY_TASK, CONFIG);
      CLOSE (CLIENT_MODIFY_TASK, ALERT);
      CLOSE (CLIENT_MODIFY_TASK, NAME);
      CLOSE (CLIENT_MODIFY_TASK, OBSERVERS);
      CLOSE (CLIENT_MODIFY_TASK, PREFERENCES);
      CLOSE (CLIENT_MODIFY_TASK, SCHEDULE);
      CLOSE (CLIENT_MODIFY_TASK, SCHEDULE_PERIODS);
      CLOSE (CLIENT_MODIFY_TASK, SLAVE);
      CLOSE (CLIENT_MODIFY_TASK, TARGET);
      CLOSE (CLIENT_MODIFY_TASK, FILE);

      CLOSE (CLIENT_MODIFY_TASK_OBSERVERS, GROUP);

      case CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE:
        assert (strcasecmp ("PREFERENCE", element_name) == 0);
        array_add (modify_task_data->preferences,
                   modify_task_data->preference);
        modify_task_data->preference = NULL;
        set_client_state (CLIENT_MODIFY_TASK_PREFERENCES);
        break;
      case CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE_NAME:
        assert (strcasecmp ("SCANNER_NAME", element_name) == 0);
        set_client_state (CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE);
        break;
      CLOSE (CLIENT_MODIFY_TASK_PREFERENCES_PREFERENCE, VALUE);

      case CLIENT_MODIFY_USER:
        {
          assert (strcasecmp ("MODIFY_USER", element_name) == 0);

          if ((modify_user_data->name == NULL
               && modify_user_data->user_id == NULL)
              || (modify_user_data->name
                  && (strlen (modify_user_data->name) == 0))
              || (modify_user_data->user_id
                  && (strlen (modify_user_data->user_id) == 0)))
            SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                    ("modify_user",
                                     "MODIFY_USER requires NAME or user_id"));
          else
            {
              gchar *fail_group_id, *fail_role_id, *errdesc;

              errdesc = NULL;

              switch (modify_user
                      (modify_user_data->user_id,
                       &modify_user_data->name,
                       modify_user_data->new_name,
                       ((modify_user_data->modify_password
                         && modify_user_data->password)
                         ? modify_user_data->password
                         /* Leave the password as it is. */
                         : NULL),
                       modify_user_data->hosts,
                       modify_user_data->hosts_allow,
                       modify_user_data->ifaces,
                       modify_user_data->ifaces_allow,
                       modify_user_data->sources,
                       modify_user_data->groups, &fail_group_id,
                       modify_user_data->roles, &fail_role_id,
                       &errdesc))
                {
                  case 0:
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_user"));
                    break;
                  case 1:
                    if (send_find_error_to_client
                         ("modify_user", "group", fail_group_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 2:
                    if (send_find_error_to_client
                         ("modify_user", "user",
                          modify_user_data->user_id ?: modify_user_data->name,
                          omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 3:
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_user"));
                    log_event ("user", "User", modify_user_data->name,
                               "raised to Admin role");
                    break;
                  case 4:
                    SEND_TO_CLIENT_OR_FAIL (XML_OK ("modify_user"));
                    log_event ("user", "User", modify_user_data->name,
                               "downgraded from Admin role");
                    break;
                  case 5:
                    if (send_find_error_to_client
                         ("modify_user", "role", fail_role_id, omp_parser))
                      {
                        error_send_to_client (error);
                        return;
                      }
                    break;
                  case 6:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_user",
                                        "Error in host specification"));
                    log_event_fail ("user", "User", NULL, "modified");
                    break;
                  case 7:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_user",
                                        "Error in user name"));
                    log_event_fail ("user", "User", NULL, "modified");
                    break;
                  case 8:
                    SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                            ("modify_user",
                                             "User with name exists already"));
                    log_event_fail ("user", "User", NULL, "modified");
                    break;
                  case 99:
                    SEND_TO_CLIENT_OR_FAIL
                     (XML_ERROR_SYNTAX ("modify_user",
                                        "Permission denied"));
                    break;
                  case -2:
                    SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                            ("modify_user", "Unknown role"));
                    break;
                  case -3:
                    SEND_TO_CLIENT_OR_FAIL (XML_ERROR_SYNTAX
                                            ("modify_user", "Error in SOURCES"));
                    break;
                  case -1:
                    if (errdesc)
                      {
                        char *buf = make_xml_error_syntax ("modify_user", errdesc);
                        SEND_TO_CLIENT_OR_FAIL (buf);
                        g_free (buf);
                        break;
                      }
                  /* Fall through.  */
                  default:
                    SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("modify_user"));
                    break;
                }
              g_free (errdesc);
            }
          modify_user_data_reset (modify_user_data);
          set_client_state (CLIENT_AUTHENTIC);
          break;
        }
      CLOSE (CLIENT_MODIFY_USER, GROUPS);
      CLOSE (CLIENT_MODIFY_USER_GROUPS, GROUP);
      CLOSE (CLIENT_MODIFY_USER, HOSTS);
      CLOSE (CLIENT_MODIFY_USER, IFACES);
      CLOSE (CLIENT_MODIFY_USER, NAME);
      CLOSE (CLIENT_MODIFY_USER, NEW_NAME);
      CLOSE (CLIENT_MODIFY_USER, PASSWORD);
      CLOSE (CLIENT_MODIFY_USER, ROLE);
      case CLIENT_MODIFY_USER_SOURCES:
        assert (strcasecmp ("SOURCES", element_name) == 0);
        array_terminate (modify_user_data->sources);
        set_client_state (CLIENT_MODIFY_USER);
        break;
      case CLIENT_MODIFY_USER_SOURCES_SOURCE:
        assert (strcasecmp ("SOURCE", element_name) == 0);
        array_add (modify_user_data->sources,
                   g_strdup (modify_user_data->current_source));
        g_free (modify_user_data->current_source);
        modify_user_data->current_source = NULL;
        set_client_state (CLIENT_MODIFY_USER_SOURCES);
        break;

      case CLIENT_TEST_ALERT:
        if (test_alert_data->alert_id)
          {
            switch (manage_alert (test_alert_data->alert_id,
                                  MANAGE_EXAMPLE_TASK_UUID,
                                  EVENT_TASK_RUN_STATUS_CHANGED,
                                  (void*) TASK_STATUS_DONE))
              {
                case 0:
                  SEND_TO_CLIENT_OR_FAIL (XML_OK ("test_alert"));
                  break;
                case 1:
                  if (send_find_error_to_client
                       ("test_alert", "alert", test_alert_data->alert_id,
                        omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("test_alert",
                                      "Permission denied"));
                  break;
                case 2:
                case -1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("test_alert"));
                  break;
                case -2:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("test_alert",
                                      "Failed to find report format for"
                                      " alert"));
                  break;
                case -3:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("test_alert",
                                      "Failed to find filter for alert"));
                  break;
                default: /* Programming error. */
                  assert (0);
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_INTERNAL_ERROR ("test_alert"));
                  break;
              }
          }
        else
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("test_alert",
                              "TEST_ALERT requires an alert_id"
                              " attribute"));
        test_alert_data_reset (test_alert_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_RESTORE:
        if (restore_data->id)
          {
            switch (manage_restore (restore_data->id))
              {
                case 0:
                  SEND_TO_CLIENT_OR_FAIL (XML_OK ("restore"));
                  log_event ("resource", "Resource", restore_data->id,
                             "restored");
                  break;
                case 1:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("restore",
                                      "Resource refers into trashcan"));
                  break;
                case 2:
                  if (send_find_error_to_client ("restore", "resource",
                                                 restore_data->id, omp_parser))
                    {
                      error_send_to_client (error);
                      return;
                    }
                  break;
                case 3:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("restore",
                                      "A resource with this name exists"
                                      " already"));
                  break;
                case 4:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("restore",
                                      "A resource with this UUID exists"
                                      " already"));
                  break;
                case 99:
                  SEND_TO_CLIENT_OR_FAIL
                   (XML_ERROR_SYNTAX ("restore",
                                      "Permission denied"));
                  break;
                default:  /* Programming error. */
                  assert (0);
                case -1:
                  SEND_TO_CLIENT_OR_FAIL (XML_INTERNAL_ERROR ("restore"));
                  break;
              }
          }
        else
          SEND_TO_CLIENT_OR_FAIL
           (XML_ERROR_SYNTAX ("restore",
                              "RESTORE requires an id attribute"));
        restore_data_reset (restore_data);
        set_client_state (CLIENT_AUTHENTIC);
        break;

      case CLIENT_RESUME_TASK:
        if (resume_task_data->task_id)
          {
            if (forked == 2)
              /* Prevent the forked child from forking again, as then both
               * forked children would be using the same server session. */
              abort (); /** @todo Respond with error or something. */
            else
              {
                char *report_id;
                switch (resume_task (resume_task_data->task_id, &report_id))
                  {
                    case 0:
                      {
                        gchar *msg;
                        msg = g_strdup_printf
                   