/**********************************************************************
 * LeechCraft - modular cross-platform feature rich internet client.
 * Copyright (C) 2006-2009  Georg Rudoy
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 **********************************************************************/

#include <stdexcept>
#include <algorithm>
#include <QApplication>
#include <QDir>
#include <QStringList>
#include <QtDebug>
#include <QMessageBox>
#include <QMainWindow>
#include <QCryptographicHash>
#include <plugininterface/util.h>
#include <interfaces/iinfo.h>
#include <interfaces/iplugin2.h>
#include <interfaces/ipluginready.h>
#include <interfaces/ipluginadaptor.h>
#include <interfaces/ihaveshortcuts.h>
#include "core.h"
#include "pluginmanager.h"
#include "mainwindow.h"
#include "xmlsettingsmanager.h"
#include "coreproxy.h"

using namespace LeechCraft;
using LeechCraft::Util::MergeModel;

LeechCraft::PluginManager::DepTreeItem::DepTreeItem ()
: Plugin_ (0)
, Initialized_ (false)
{
}

void LeechCraft::PluginManager::DepTreeItem::Print (int margin)
{
	QString pre (margin, ' ');
	qDebug () << pre << Plugin_ << Initialized_;
	qDebug () << pre << "Needed:";
	QList<DepTreeItem_ptr> items = Needed_.values ();
	Q_FOREACH (DepTreeItem_ptr item, items)
		item->Print (margin + 2);
	qDebug () << pre << "Used:";
	items = Used_.values ();
	Q_FOREACH (DepTreeItem_ptr item, items)
		item->Print (margin + 2);
}

LeechCraft::PluginManager::Finder::Finder (QObject *o)
: Object_ (o)
{
}

bool LeechCraft::PluginManager::Finder::operator() (LeechCraft::PluginManager::DepTreeItem_ptr item) const
{
	return Object_ == item->Plugin_;
}

LeechCraft::PluginManager::PluginManager (QObject *parent)
: QAbstractItemModel (parent)
{
	Headers_ << tr ("Name")
		<< tr ("File");
	FindPlugins ();
}

LeechCraft::PluginManager::~PluginManager ()
{
}

int LeechCraft::PluginManager::columnCount (const QModelIndex&) const
{
	return Headers_.size ();
}

QVariant LeechCraft::PluginManager::data (const QModelIndex& index, int role) const
{
	if (!index.isValid () ||
			index.row () >= GetSize ())
		return QVariant ();

	switch (index.column ())
	{
		case 0:
			switch (role)
			{
				case Qt::DisplayRole:
					{
						QSettings settings (QCoreApplication::organizationName (),
								QCoreApplication::applicationName ());
						settings.beginGroup ("Plugins");
						settings.beginGroup (AvailablePlugins_.at (index.row ())->fileName ());
						QVariant result = settings.value ("Name");
						settings.endGroup ();
						settings.endGroup ();
						return result;
					}
				case Qt::DecorationRole:
					{
						QSettings settings (QCoreApplication::organizationName (),
								QCoreApplication::applicationName ());
						settings.beginGroup ("Plugins");
						settings.beginGroup (AvailablePlugins_.at (index.row ())->fileName ());
						QVariant result = settings.value ("Icon");
						settings.endGroup ();
						settings.endGroup ();
						return result;
					}
				case Qt::CheckStateRole:
					{
						QSettings settings (QCoreApplication::organizationName (),
								QCoreApplication::applicationName ());
						settings.beginGroup ("Plugins");
						settings.beginGroup (AvailablePlugins_.at (index.row ())->fileName ());
						bool result = settings.value ("AllowLoad", true).toBool ();
						settings.endGroup ();
						settings.endGroup ();
						return result ? Qt::Checked : Qt::Unchecked;
					}
				case Qt::ForegroundRole:
					return QApplication::palette ()
						.brush (AvailablePlugins_.at (index.row ())->isLoaded () ?
								QPalette::Normal :
								QPalette::Disabled,
							QPalette::WindowText);
				default:
					return QVariant ();
			}
		case 1:
			if (role == Qt::DisplayRole)
				return AvailablePlugins_.at (index.row ())->fileName ();
			else if (role == Qt::ForegroundRole)
				return QApplication::palette ()
					.brush (AvailablePlugins_.at (index.row ())->isLoaded () ?
							QPalette::Normal :
							QPalette::Disabled,
						QPalette::WindowText);
			else
				return QVariant ();
		default:
			return QVariant ();
	}
}

Qt::ItemFlags LeechCraft::PluginManager::flags (const QModelIndex& index) const
{
	Qt::ItemFlags result = Qt::ItemIsEnabled | Qt::ItemIsSelectable;
	if (index.column () == 0)
		result |= Qt::ItemIsUserCheckable;
	return result;
}

QVariant LeechCraft::PluginManager::headerData (int column, Qt::Orientation orient, int role) const
{
	if (role != Qt::DisplayRole ||
			orient != Qt::Horizontal)
		return QVariant ();

	return Headers_.at (column);
}

QModelIndex LeechCraft::PluginManager::index (int row, int column, const QModelIndex&) const
{
	return createIndex (row, column);
}

QModelIndex LeechCraft::PluginManager::parent (const QModelIndex&) const
{
	return QModelIndex ();
}

int LeechCraft::PluginManager::rowCount (const QModelIndex& index) const
{
	if (!index.isValid ())
		return AvailablePlugins_.size ();
	else
		return 0;
}

bool LeechCraft::PluginManager::setData (const QModelIndex& index,
		const QVariant& data, int role)
{
	if (index.column () != 0 ||
			role != Qt::CheckStateRole)
		return false;

	QSettings settings (QCoreApplication::organizationName (),
			QCoreApplication::applicationName ());
	settings.beginGroup ("Plugins");
	settings.beginGroup (AvailablePlugins_.at (index.row ())->fileName ());
	settings.setValue ("AllowLoad", data.toBool ());
	settings.endGroup ();
	settings.endGroup ();

	emit dataChanged (index, index);

	return true;
}

LeechCraft::PluginManager::Size_t LeechCraft::PluginManager::GetSize () const
{
	return AvailablePlugins_.size ();
}

void LeechCraft::PluginManager::Init ()
{
	CheckPlugins ();
	CalculateDependencies ();
	InitializePlugins ();
}

void LeechCraft::PluginManager::Release ()
{
	while (Roots_.size ())
	{
		try
		{
			Release (Roots_.takeAt (0));
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO << e.what ();
		}
		catch (...)
		{
			QMessageBox::warning (0,
					tr ("LeechCraft"),
					tr ("Release of one or more plugins failed."));
		}
	}

	Plugins_.clear ();
}

QString LeechCraft::PluginManager::Name (const LeechCraft::PluginManager::Size_t& pos) const
{
	return (qobject_cast<IInfo*> (Plugins_ [pos]->instance ()))->GetName ();
}

QString LeechCraft::PluginManager::Info (const LeechCraft::PluginManager::Size_t& pos) const
{
	return qobject_cast<IInfo*> (Plugins_ [pos]->instance ())->GetInfo ();
}

QObjectList LeechCraft::PluginManager::GetAllPlugins () const
{
	QObjectList result;
	for (PluginsContainer_t::const_iterator i = Plugins_.begin ();
			i != Plugins_.end (); ++i)
		result << (*i)->instance ();
	return result;
}

QObject* LeechCraft::PluginManager::GetProvider (const QString& feature) const
{
	if (!FeatureProviders_.contains (feature))
		return 0;
	return (*FeatureProviders_ [feature])->instance ();
}

void LeechCraft::PluginManager::Unload (QObject *plugin)
{
	Unload (Find (plugin));
}

void LeechCraft::PluginManager::FindPlugins ()
{
#ifdef Q_WS_WIN
	ScanDir (QApplication::applicationDirPath () + "/plugins/bin");
#else
	ScanDir ("/usr/local/lib/leechcraft/plugins");
	ScanDir ("/usr/lib/leechcraft/plugins");
#endif
}

void LeechCraft::PluginManager::ScanDir (const QString& dir)
{
	QSettings settings (QCoreApplication::organizationName (),
			QCoreApplication::applicationName ());
	settings.beginGroup ("Plugins");

	QDir pluginsDir = QDir (dir);
	Q_FOREACH (QFileInfo fileinfo,
			pluginsDir.entryInfoList (QStringList ("*leechcraft_*"),
				QDir::Files))
	{
		QString name = fileinfo.canonicalFilePath ();
		settings.beginGroup (name);

		QPluginLoader_ptr loader (new QPluginLoader (name));
		if (settings.value ("AllowLoad", true).toBool ())
			Plugins_.push_back (loader);

		AvailablePlugins_.push_back (loader);

		settings.endGroup ();
	}

	settings.endGroup ();
}

void LeechCraft::PluginManager::CheckPlugins ()
{
	QSettings settings (QCoreApplication::organizationName (),
			QCoreApplication::applicationName ());
	settings.beginGroup ("Plugins");

	for (int i = 0; i < Plugins_.size (); ++i)
	{
		QPluginLoader_ptr loader = Plugins_.at (i);

		QString file = loader->fileName ();

		if (!QFileInfo (loader->fileName ()).isFile ())
		{
			qWarning () << "A plugin isn't really a file, aborting load:"
				<< file;
			Plugins_.removeAt (i--);
			continue;
		}

		loader->load ();
		if (!loader->isLoaded ())
		{
			qWarning () << "Could not load library:"
				<< file
				<< ";"
				<< loader->errorString ();
			Plugins_.removeAt (i--);
			continue;
		}

		QObject *pluginEntity;
		try
		{
			pluginEntity = loader->instance ();
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to construct the instance with"
				<< e.what ()
				<< "for"
				<< file;
			Plugins_.removeAt (i--);
			continue;
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to construct the instance for"
				<< file;
			Plugins_.removeAt (i--);
			continue;
		}

		IInfo *info = qobject_cast<IInfo*> (pluginEntity);
		if (!info)
		{
			qWarning () << "Casting to IInfo failed:"
					<< file;
			Plugins_.removeAt (i--);
			continue;
		}

		QString name;
		QIcon icon;
		try
		{
			name = info->GetName ();
			icon = info->GetIcon ();
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get name/icon"
				<< e.what ()
				<< "for"
				<< file;
			Plugins_.removeAt (i--);
			continue;
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get name/icon"
				<< file;
			Plugins_.removeAt (i--);
			continue;
		}
		settings.beginGroup (file);
		settings.setValue ("Name", name);
		settings.setValue ("Icon", icon.pixmap (48, 48));
		settings.endGroup ();
	}

	settings.endGroup ();
}

QList<LeechCraft::PluginManager::PluginsContainer_t::iterator>
	LeechCraft::PluginManager::FindProviders (const QString& feature)
{
	QList<PluginsContainer_t::iterator> result;
	for (PluginsContainer_t::iterator i = Plugins_.begin ();
			i != Plugins_.end (); ++i)
	{
		try
		{
			if (qobject_cast<IInfo*> ((*i)->instance ())->
					Provides ().contains (feature))
				result << i;
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get providers with"
				<< e.what ()
				<< "for"
				<< (*i)->fileName ();
			Unload (i);
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get providers"
				<< (*i)->fileName ();
			Unload (i);
		}
	}
	return result;
}

QList<LeechCraft::PluginManager::PluginsContainer_t::iterator>
	LeechCraft::PluginManager::FindProviders (const QByteArray& expected)
{
	QList<PluginsContainer_t::iterator> result;
	for (PluginsContainer_t::iterator i = Plugins_.begin ();
			i != Plugins_.end (); ++i)
	{
		IPlugin2 *ip2 = qobject_cast<IPlugin2*> ((*i)->instance ());
		try
		{
			if (ip2 &&
					ip2->GetPluginClass () == expected)
				result << i;
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get class with"
				<< e.what ()
				<< "for"
				<< (*i)->fileName ();
			Unload (i);
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get class"
				<< (*i)->fileName ();
			Unload (i);
		}
	}
	return result;
}

LeechCraft::PluginManager::DepTreeItem_ptr
	LeechCraft::PluginManager::GetDependency
		(QObject *entity)
{
	struct Finder
	{
		QObject *Plugin_;
		DepTreeItem_ptr Result_;

		Finder (QObject *plugin)
		: Plugin_ (plugin)
		{
		}

		bool operator() (DepTreeItem_ptr item)
		{
			QList<DepTreeItem_ptr> deps =
				item->Needed_.values () + item->Used_.values ();
			std::sort (deps.begin (), deps.end ());
			QList<DepTreeItem_ptr>::const_iterator last =
				std::unique (deps.begin (), deps.end ());

			for (QList<DepTreeItem_ptr>::const_iterator i = deps.begin ();
					i != last; ++i)
				if ((*i)->Plugin_ == Plugin_)
				{
					Result_ = *i;
					return true;
				}
			for (QList<DepTreeItem_ptr>::const_iterator i = deps.begin ();
					i != last; ++i)
				if (operator() (*i))
					return true;

			return false;
		}
	};

	Finder finder (entity);
	Q_FOREACH (DepTreeItem_ptr dep, Roots_)
	{
		if (dep->Plugin_ == entity)
		{
			Roots_.removeAll (dep);
			return dep;
		}
		else
		{
			if (finder (dep))
				return finder.Result_;
		}
	}

	return DepTreeItem_ptr ();
}

void LeechCraft::PluginManager::CalculateDependencies ()
{
	for (PluginsContainer_t::iterator i = Plugins_.begin ();
			i < Plugins_.end (); ++i)
		if (!GetDependency ((*i)->instance ()))
		{
			try
			{
				Roots_ << CalculateSingle (i);

				IPluginAdaptor *ipa = qobject_cast<IPluginAdaptor*> ((*i)->instance ());
				if (ipa)
				{
					QList<QObject*> plugins = ipa->GetPlugins ();
					Q_FOREACH (QObject *child, plugins)
						Roots_ << CalculateSingle (child, Plugins_.end ());
				}
			}
			catch (...)
			{
				qWarning () << Q_FUNC_INFO
					<< "CalculateSingle failed";
			}
		}
}

LeechCraft::PluginManager::DepTreeItem_ptr
	LeechCraft::PluginManager::CalculateSingle
		(LeechCraft::PluginManager::PluginsContainer_t::iterator i)
{
	QObject *entity = (*i)->instance ();
	try
	{
		return CalculateSingle (entity, i);
	}
	catch (const std::exception& e)
	{
		qWarning () << Q_FUNC_INFO
			<< "failed to get required stuff with"
			<< e.what ()
			<< "for"
			<< (*i)->fileName ();
		Unload (i);
		throw;
	}
	catch (...)
	{
		qWarning () << Q_FUNC_INFO
			<< "failed to get required stuff"
			<< (*i)->fileName ();
		Unload (i);
		throw;
	}
}

LeechCraft::PluginManager::DepTreeItem_ptr
	LeechCraft::PluginManager::CalculateSingle (QObject *entity,
			LeechCraft::PluginManager::PluginsContainer_t::iterator pos)
{
	DepTreeItem_ptr possibly = GetDependency (entity);
	if (possibly)
		return possibly;

	IInfo *info = qobject_cast<IInfo*> (entity);
	QStringList needs;
	QStringList uses;
	needs = info->Needs ();
	uses = info->Uses ();

#ifdef QT_DEBUG
	qDebug () << "new item" << info->GetName ();
#endif
	DepTreeItem_ptr newDep (new DepTreeItem ());
	newDep->Plugin_ = entity;

	Q_FOREACH (QString need, needs)
	{
		QList<PluginsContainer_t::iterator> providers = FindProviders (need);
		Q_FOREACH (PluginsContainer_t::iterator p,
				providers)
		{
			// It's initialized already.
			if (p < pos)
			{
				DepTreeItem_ptr depprov = GetDependency ((*p)->instance ());
				// TODO register entity in depprov->Belongs_
				if (depprov)
					newDep->Needed_.insert (need, depprov);
			}
			else if (p > pos)
				newDep->Needed_.insert (need, CalculateSingle (p));
		}
	}

	Q_FOREACH (QString use, uses)
	{
		QList<PluginsContainer_t::iterator> providers = FindProviders (use);
		Q_FOREACH (PluginsContainer_t::iterator p,
				providers)
		{
			// It's initialized already.
			if (p < pos)
			{
				DepTreeItem_ptr depprov = GetDependency ((*p)->instance ());
				if (depprov)
					newDep->Used_.insert (use, depprov);
			}
			else if (p > pos)
				newDep->Used_.insert (use, CalculateSingle (p));
		}
	}

	IPluginReady *ipr = qobject_cast<IPluginReady*> (entity);
	if (ipr)
	{
		QList<PluginsContainer_t::iterator> providers =
			FindProviders (ipr->GetExpectedPluginClass ());
		Q_FOREACH (PluginsContainer_t::iterator p,
				providers)
		{
			// It's initialized already.
			if (p < pos)
			{
				DepTreeItem_ptr depprov = GetDependency ((*p)->instance ());
				if (depprov)
					newDep->Used_.insert ("__lc_plugin2", depprov);
			}
			else if (p > pos)
				newDep->Used_.insert ("__lc_plugin2", CalculateSingle (p));
		}
	}

	return newDep;
}

void LeechCraft::PluginManager::InitializePlugins ()
{
	Q_FOREACH (DepTreeItem_ptr item, Roots_)
		InitializeSingle (item);
}

bool LeechCraft::PluginManager::InitializeSingle (LeechCraft::PluginManager::DepTreeItem_ptr item)
{
	QList<QString> keys = item->Needed_.uniqueKeys ();
	Q_FOREACH (QString key, keys)
	{
		QList<DepTreeItem_ptr> providers = item->Needed_.values (key);
		bool wasSuccessful = false;
		for (QList<DepTreeItem_ptr>::const_iterator i = providers.begin (),
				end = providers.end (); i != end; ++i)
		{
			if (!(*i)->Initialized_)
				InitializeSingle (*i);

			if (!(*i)->Initialized_)
				item->Needed_.remove (key, *i);
			else
			{
				wasSuccessful = true;

				qobject_cast<IInfo*> (item->Plugin_)->
					SetProvider ((*i)->Plugin_, key);;
			}
		}

		if (!wasSuccessful)
			return false;
	}

	keys = item->Used_.uniqueKeys ();
	Q_FOREACH (QString key, keys)
	{
		QList<DepTreeItem_ptr> providers = item->Used_.values (key);
		for (QList<DepTreeItem_ptr>::const_iterator i = providers.begin (),
				end = providers.end (); i != end; ++i)
		{
			if (!(*i)->Initialized_)
				InitializeSingle (*i);

			if (!(*i)->Initialized_)
				item->Used_.remove (key, *i);
			else
			{
				if (key == "__lc_plugin2")
					qobject_cast<IPluginReady*> (item->Plugin_)->
						AddPlugin ((*i)->Plugin_);
				else
					qobject_cast<IInfo*> (item->Plugin_)->
						SetProvider ((*i)->Plugin_, key);
			}
		}
	}

	try
	{
		IInfo *ii = qobject_cast<IInfo*> (item->Plugin_);
		QString name = ii->GetName ();
		qDebug () << "Initializing" << name << "...";
		ii->Init (ICoreProxy_ptr (new CoreProxy ()));
		item->Initialized_ = true;

		Core::Instance ().Setup (item->Plugin_);
	}
	catch (const std::exception& e)
	{
		qWarning () << Q_FUNC_INFO << 2 << e.what ();
		Unload (Find (item));
		return false;
	}
	catch (...)
	{
		qWarning () << Q_FUNC_INFO << 2;
		Unload (Find (item));
		return false;
	}
	return true;
}

void LeechCraft::PluginManager::Release (DepTreeItem_ptr item)
{
	QList<DepTreeItem_ptr> deps =
		item->Needed_.values () + item->Used_.values ();
	std::sort (deps.begin (), deps.end ());
	QList<DepTreeItem_ptr>::const_iterator last =
		std::unique (deps.begin (), deps.end ());

	if (item->Initialized_)
	{
		IInfo *ii = qobject_cast<IInfo*> (item->Plugin_);
#ifdef QT_DEBUG
		try
		{
			qDebug () << Q_FUNC_INFO << ii->GetName ();
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to get the name of the unloading object";
		}
#endif

		PluginsContainer_t::iterator i = Find (item);
		try
		{
			ii->Release ();
			item->Initialized_ = false;
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "failed to release the unloading object with"
				<< e.what ()
				<< "for"
				<< (*i)->fileName ();
		}
		catch (...)
		{
			PluginsContainer_t::iterator i = Find (item);
			qWarning () << Q_FUNC_INFO
				<< "failed to release the unloading object"
				<< (*i)->fileName ();
		}
		Unload (i);
	}

	for (QList<DepTreeItem_ptr>::const_iterator i = deps.begin ();
			i != last; ++i)
		Release (*i);
}

void LeechCraft::PluginManager::DumpTree ()
{
	Q_FOREACH (DepTreeItem_ptr item, Roots_)
		item->Print ();
}

namespace
{
	struct LoaderFinder
	{
		QObject *Object_;

		LoaderFinder (QObject *o)
		: Object_ (o)
		{
		}

		bool operator() (const boost::shared_ptr<QPluginLoader>& ptr) const
		{
			return ptr->instance () == Object_;
		}
	};
};

LeechCraft::PluginManager::PluginsContainer_t::iterator
	LeechCraft::PluginManager::Find (DepTreeItem_ptr item)
{
	return Find (item->Plugin_);
}

LeechCraft::PluginManager::PluginsContainer_t::iterator
	LeechCraft::PluginManager::Find (QObject *item)
{
	return std::find_if (Plugins_.begin (), Plugins_.end (),
			LoaderFinder (item));
}

void LeechCraft::PluginManager::Unload (PluginsContainer_t::iterator i)
{
	if (i == Plugins_.end ())
		return;

	if (!UnloadQueue_.contains (i))
		UnloadQueue_ << i;

	DepTreeItem_ptr dep = GetDependency ((*i)->instance ());
	if (dep)
	{
		QPluginLoader_ptr pluginLoader = *i;
		Q_FOREACH (DepTreeItem_ptr belongs, dep->Belongs_)
			Unload (Find (belongs));

		if (dep->Initialized_)
		{
			try
			{
				qobject_cast<IInfo*> (dep->Plugin_)->Release ();
			}
			catch (const std::exception& e)
			{
				PluginsContainer_t::iterator i = Find (dep);
				qWarning () << Q_FUNC_INFO
					<< "failed to release the unloading object with"
					<< e.what ()
					<< "for"
					<< pluginLoader->fileName ();
			}
			catch (...)
			{
				PluginsContainer_t::iterator i = Find (dep);
				qWarning () << Q_FUNC_INFO
					<< "failed to release the unloading object"
					<< pluginLoader->fileName ();
			}
		}
	}

	/** TODO understand why app segfaults on exit if it's uncommented.
	if (UnloadQueue_.size () == 1)
		processUnloadQueue ();
	*/
}

void LeechCraft::PluginManager::processUnloadQueue ()
{
	for (int i = 0; i < UnloadQueue_.size (); ++i)
	{
		QPluginLoader_ptr loader = *UnloadQueue_.at (i);
		IInfo *ii = qobject_cast<IInfo*> (loader->instance ());
		QString name;
		try
		{
			name = ii->GetName ();
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "unable to get name for the unload"
				<< loader->instance ()
				<< e.what ();
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "unable to get name for the unload"
				<< loader->instance ();
		}

		try
		{
#ifdef QT_DEBUG
			qDebug () << Q_FUNC_INFO
				<< name;
#endif
			if (!loader->unload ())
				qWarning () << Q_FUNC_INFO
					<< "unable to unload"
					<< loader->instance ()
					<< loader->errorString ();
		}
		catch (const std::exception& e)
		{
			qWarning () << Q_FUNC_INFO
				<< "unable to unload with exception"
				<< loader->instance ()
				<< e.what ();
		}
		catch (...)
		{
			qWarning () << Q_FUNC_INFO
				<< "unable to unload with exception"
				<< loader->instance ();
		}
	}
	std::sort (UnloadQueue_.begin (), UnloadQueue_.end ());
	while (UnloadQueue_.size ())
	{
		Plugins_.erase (UnloadQueue_.last ());
		UnloadQueue_.pop_back ();
	}
}

