/*  wmbday
 *  Copyright (C) 2003-2005 astratis <steffen@x-berg.de>
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public Licensse as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 */

#include "ring.h"
#include "date.h"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>


static struct ring* start = NULL; /* first element in ring, so first birthday in year */
static struct ring* end = NULL; /* last element in ring*/
static struct ring* current = NULL; /* points to person, whose birthday is next */
static int count = 0; /* number of elements in ring */


/* adds a new birthday to the ring */
void ring_add(char* name, int day, int month, int year) {
  struct ring* newelement = NULL;
  struct ring* temp = NULL;
  struct ring* temp_old = NULL;
  int i;

  if((newelement = malloc(sizeof(struct ring))) == NULL) {
    fprintf(stderr, "Couldn't allocate memory\n");
    exit(1);
  }

  memset(newelement->name, '\0', sizeof(newelement->name));
  strncpy(newelement->name, name, sizeof(newelement->name)-1);
  newelement->day = day;
  newelement->month = month;
  newelement->year = year;


  /* there is no ring x-isting */
  if(count == 0) {
    start = newelement;
    end = newelement;
    newelement->next = NULL;
    ++count;
    return;
  }

  /* there is already a ring, so we need to find the right place */
  i = 0;
  temp = start;

  while(i != count && compare_dates(temp->day, temp->month, day, month) == -1) {
    temp_old = temp;
    temp = temp->next;
    ++i;
  }

  /* new start */
  if(i == 0) {
    if(start->next == NULL) {
      start->next = newelement;
    }
    newelement->next = start;
    start = newelement;
    end->next = start;
    ++count;
    return;
  }

  /* new end */
  if(i == count) {
    temp_old->next = newelement;
    newelement->next = start;
    end = newelement;
    ++count;
    return;
  }

  /* put it somewhere in the middle */
  temp_old->next = newelement;
  newelement->next = temp;
  ++count;

  return;
}


/* makes "current" point to the person, whose birthday is next */
void ring_goto_date(int day, int month) {
  int i = 0;

  current = start;

  while(i < count && compare_dates(current->day, current->month, day, month) == -1) {
    if(current->next == NULL)
        break;

    current = current->next;

    ++i;
  }

  return;
}


int ring_check_bday(int day, int month) {
  int i = 0;
  int bday_count = 0;
  struct ring* temp;

  if(current == NULL)
    return -1;

  temp = current;

  while(i < count && compare_dates(temp->day, temp->month, day, month) == 0) {
    ++bday_count;
    temp = temp->next;
  }

  return bday_count;
}


/* puts next "count" birthdays into "target" */
void ring_get_next(struct ring* target[], int number) {
  struct ring* temp;
  int i = 0;

  temp = current;

  while(i<number && i<count) {
    target[i] = temp;
    ++i;
    if(temp->next == NULL)
        break;

    temp = temp->next;
  }

  while(i<number) {
    target[i] = NULL;
     ++i;
  }
  return;
}


/* clean up whole list */
void ring_clear(void) {
  struct ring* temp;

  temp = start;

  while(count < 0) {
    free(temp);
    --count;
  }

  start = end = current = NULL;
  count = 0;

  return;
}
