/*
 *   fbdump - dump visible framebuffer contents to stdout as a PPM file
 *
 *   main.c
 *
 *   (C) Richard Drummond 2001-2007
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2 of the License, or
 *   (at your option) any later version.
 */

#include "defs.h"
#include <errno.h>
#include <string.h>
#include "options.h"
#include "surface.h"
#include "fb.h"
#include "vt.h"


#define DEFAULT_DEVICE  "/dev/fb0"


/*
 * Container for parsed command-line options
 */
typedef struct
{
   char *fbdev;  // Path to the framebuffer
   int   vt;	 // VT number to switch to
   int   delay;  // Delay before grabbing
   int   help;   // Whether to print help text
} Options;


#define OPT_OFFSET( opt_element )  ((char *)(&(((Options*)0)->opt_element))-(char *)0)

/*
 * Template for parsing command-line options
 */
static const OptionTemplate opt_template[] =
{
  { "-fb",    OPT_TYPE_STRING,  OPT_OFFSET( fbdev ),  "Path to the framebuffer device to dump is <string>" },
  { "-vt",    OPT_TYPE_INT,     OPT_OFFSET( vt ) ,    "Bring virtual terminal no. <n> to the foreground before dumping" },
  { "-delay", OPT_TYPE_INT,     OPT_OFFSET( delay ),  "Wait <n> seconds before dumping" },
  { "--help", OPT_TYPE_BOOLEAN, OPT_OFFSET( help ),   "Show this help" },
  { 0, 0, 0, 0 }
};


void print_help( FILE *stream )
{
  fprintf( stream, "%s %s\n\n", PACKAGE, VERSION );
  fprintf( stream, "(c) Richard Drummond 2001-2007\n\n" );
  fprintf( stream, "Dumps the contents of the framebuffer to stdout as a PPM file.\n" );
}


int main( int argc, char *argv[] )
{
  Options  opts = {
    .fbdev = DEFAULT_DEVICE,
    .vt    = 0,
    .delay = 0,
    .help  = 0
  };
  FBHandle fb;
  VTHandle vt;
  int      old_vt_num = 0;
  Surface  surface;
  int      ret = EXIT_FAILURE;

  /* If FRAMEBUFFER variable is specified, use that as default path to fbdev  */
  if( getenv( "FRAMEBUFFER" ) != NULL )
  {
    opts.fbdev = getenv( "FRAMEBUFFER" );
  }

  /* Parse any command line options */
  if( Options_parse( opt_template, &opts, argc, argv ) != 0 )
    return 0;

  if( opts.help )
  {
    print_help( stderr );
    Options_printUsage( opt_template, stderr );
    exit( EXIT_SUCCESS );
  }

  /* Did the user ask to switch VT? */
  if( opts.vt )
  {
    /* Get a handle on our terminal */
    if( VT_obtain( &vt ) != RETURN_FAILURE )
    {
      /* Detach our process from the terminal. We need to do this when we
       * switch VTs because otherwise all the framebuffer ioctls() will still
       * act on the VT that our process is running in.
       */
      VT_detach( &vt );

      /* Now we can switch VTs */
      if( (old_vt_num = VT_switch( &vt, opts.vt)) == RETURN_FAILURE )
      {
        fprintf( stderr, "Failed to switch to virtual terminal no. %d: %s\n", opts.vt, strerror( errno ) );
        exit( EXIT_FAILURE );
      }
    }
    else
    {
      fprintf( stderr, "Failed to open console: %s\n", strerror( errno ) );
      exit( EXIT_FAILURE );
    }
  }

    if( opts.delay )
      sleep( opts.delay );

  /* Now open framebuffer */
  if( FB_open( opts.fbdev, &fb ) != RETURN_FAILURE )
  {
    /* If user asked for a delay, then sleep */


    /* Find the visible region */
    FB_getVisibleSurface( &fb, &surface );

    /* And dump it to stdout */
    Surface_dumpAsPPM( &surface, stdout );

    /* All done */
    FB_close( &fb );

    /* Success */
    ret = EXIT_SUCCESS;
  }
  else
    fprintf( stderr, "Failed to open %s: %s\n", opts.fbdev, strerror( errno ) );

  /* If we switched vt, now switch back */
  if( old_vt_num != -1 )
  {
    VT_switch( &vt, old_vt_num);
    VT_release( &vt );
  }

  return ret;
}
