#!/usr/bin/env python2
# -*- coding: utf-8 -*-
###############################################################################
# conkyForecast.py is a (not so) simple (anymore) python script to gather 
# details of the current weather for use in conky.
#
#  Author: Kaivalagi
# Created: 06/11/2011

from datetime import datetime, timedelta
from optparse import OptionParser
import codecs
import gettext
import json
import locale
import math
import os
import socket
import sys
import time
import traceback
import urllib2

# not sure on these, might create more trouble, but here goes...
reload(sys)
sys.setdefaultencoding('utf-8')

# cPickle is a pickle class implemented in C - so its faster
# in case its not available, use regular pickle
try:
    import cPickle as pickle
except ImportError:
    import pickle

app_name = "conkyForecastWU"
app_version = 2.24
app_path = os.path.dirname(os.path.abspath(__file__))
module_name = __file__.replace(os.path.dirname (__file__) + "/", "").replace(".pyc","").replace(".py", "")

# default to standard locale translation
domain = __file__.replace(os.path.dirname (__file__) + "/", "").replace(".py", "")
locale_directory = os.path.dirname(os.path.abspath(__file__)) + "/locale"
gettext.bindtextdomain(domain, locale_directory)
gettext.textdomain(domain)
gettext.install(domain)

class CommandLineParser:

    parser = None

    def __init__(self):

        self.parser = OptionParser()
        self.parser.add_option("-C", "--config", dest="config", default="~/.conkyForecastWU.config", type="string", metavar="FILE", help=u"[default: %default] The path to the configuration file, allowing multiple config files to be used.")
        self.parser.add_option("-l", "--location", dest="location", type="string", metavar="CODE", help=u"location code for weather data [default set in config]. Normally in the form \"COUNTRY/CITY\"")
        self.parser.add_option("-d", "--datatype", dest="datatype", default="HT", type="string", metavar="DATATYPE", help=u"[default: %default] The data type options are: DW (Day of Week), WI (Weather Icon Path), LT (Forecast:Low Temp,Current:Feels Like Temp), HT (Forecast:High Temp,Current:Current Temp), CT (Conditions Text), AT (Alert Text), AM (Alert Message), PC (Precipitation Chance), HM (Humidity), VI (Visibility), WD (Wind Direction), WA (Wind Angle - in degrees), WS (Wind Speed), WG (Wind Gusts), BF (Bearing Font), BI (Bearing Icon Path), BS (Bearing font with Speed), CN (City Name), CO (Country), OB (Observatory), BR (Barometer Reading), BD (Barometer Description), UI (UV Index), UT (UV Text), DP (Dew Point), SR (SunRise), SS (SunSet), DL (DayLight), MP (Moon Phase), MI (Moon Icon Path), LU (Last Update), LF (Last Fetch). Not applicable at command line when using templates.")
        self.parser.add_option("-s", "--startday", dest="startday", type="int", metavar="NUMBER", help=u"define the starting day number, if omitted current conditions are output. Not applicable at command line when using templates.")
        self.parser.add_option("-e", "--endday", dest="endday", type="int", metavar="NUMBER", help=u"define the ending day number, if omitted only starting day data is output. Not applicable at command line when using templates.")
        self.parser.add_option("-S", "--spaces", dest="spaces", type="int", default=1, metavar="NUMBER", help=u"[default: %default] Define the number of spaces between ranged output. Not applicable at command line when using templates.")
        self.parser.add_option("-t", "--template", dest="template", type="string", metavar="FILE", help=u"define a template file to generate output in one call. A displayable item in the file is in the form [--datatype=HT --startday=1]. The following are possible options within each item: --location,--datatype,--startday,--endday,--night,--shortweekday,--imperial,--beaufort,--metrespersecond,--hideunits,--hidedegreesymbol,--spaces,--minuteshide. Note that the short forms of the options are not supported! If any of these options is set from the commandline, it sets the default value of the option for all template items.")
        self.parser.add_option("-L", "--locale", dest="locale", type="string", help=u"override the system locale for language output (bg=bulgarian, cs=czech, de=german, es=spanish, en=english, es=spanish, fj=fijian, fr=french, it=italian, nl=dutch, pl=polish, ro=romanian, sk=slovak, more to come)")
        self.parser.add_option("-i", "--imperial", dest="imperial", default=False, action="store_true", help=u"request imperial units, if omitted output is in metric.")
        self.parser.add_option("-b", "--beaufort", dest="beaufort", default=False, action="store_true", help=u"request beaufort scale for wind speeds, if omitted output is either metric/imperial.")
        self.parser.add_option("-M", "--metrespersecond", dest="metrespersecond", default=False, action="store_true", help=u"request metres per second for wind speeds, if omitted output is either metric/imperial.")
        self.parser.add_option("-n", "--night", dest="night", default=False, action="store_true", help=u"switch output to night data, if omitted day output will be output.")
        self.parser.add_option("-w", "--shortweekday", dest="shortweekday", default=False, action="store_true", help=u"Shorten the day of week data type to 3 characters.")
        self.parser.add_option("-u", "--hideunits", dest="hideunits", default=False, action="store_true", help=u"Hide units such as mph or C, degree symbols (°) are still shown.")
        self.parser.add_option("-x", "--hidedegreesymbol", dest="hidedegreesymbol", default=False, action="store_true", help=u"Hide the degree symbol used with temperature output, this is only valid if used in conjunction with --hideunits.")
        self.parser.add_option("-m", "--minuteshide", dest="minuteshide", type="int", metavar="NUMBER", help=u"Works only with LU and LF. If present, hides the date part of the LU or LF timestamp if the day of the timestamp is today. The time part is also hidden, if the timestamp is older than minutes specified in this argument. If set to 0, the time part is always shown. If set to -1, the value EXPIRY_MINUTES from the config file is used.")
        self.parser.add_option("-c", "--centeredwidth", dest="centeredwidth", type="int", metavar="WIDTH", help=u"If used the output will be centered in a string of the set width, padded out with spaces, if the output width is greater than the setting it will be truncated")
        self.parser.add_option("-r", "--refetch", dest="refetch", default=False, action="store_true", help=u"Fetch data regardless of data expiry.")
        self.parser.add_option("-v", "--verbose", dest="verbose", default=False, action="store_true", help=u"Request verbose output, not a good idea when running through conky!")
        self.parser.add_option("-V", "--version", dest="version", default=False, action="store_true", help=u"Displays the version of the script.")
        self.parser.add_option("-D", "--debug", dest="debug", default=False, action="store_true", help=u"Request debug operation, will not fetch new data but use existing json file for example")
        self.parser.add_option("--errorlogfile", dest="errorlogfile", type="string", metavar="FILE", help=u"If a filepath is set, the script appends errors to the filepath.")
        self.parser.add_option("--infologfile", dest="infologfile", type="string", metavar="FILE", help=u"If a filepath is set, the script appends info to the filepath.")                

    def parse_args(self):
        (options, args) = self.parser.parse_args()
        return (options, args)

    def print_help(self):
        return self.parser.print_help()
    
# N.B: The below class member values are defaults and should be left alone, they
#      are there to provide a working script if the script is called without all
#      the expected input. Any issues raised where these values have been
#      changed will get the simple "put the .py file back to it's original
#      state" reply
class ForecastConfig:
    BASE_WU_JSON_URL = "http://api.wunderground.com/api/<WU_LICENCE_KEY>/geolookup/conditions/forecast/q/<LOCATION>.json"
    BASE_WU_ICON_URL = "http://icons.wxug.com/i/c/<ICON_SET>/<ICON>.gif" 
    CACHE_FOLDERPATH = "/tmp/"
    PERSISTED_CACHE_FOLDERPATH = "~/"
    ICON_SET = "a"
    CONNECTION_TIMEOUT = 5
    EXPIRY_MINUTES = 30
    TIME_FORMAT = "%H:%M"
    DATE_FORMAT = "%Y-%m-%d"
    LOCALE = "" # with no setting the default locale of the system is used
    WU_ANVIL_LICENCE_KEY = ""
    WU_STRATUS_LICENCE_KEY = ""
    WU_CUMULUS_LICENCE_KEY = ""
    DEFAULT_LOCATION = "UK/Norwich"
    MAXIMUM_DAYS_FORECAST = 7
    AUTO_NIGHT = False
    PROXY_HOST = None
    PROXY_PORT = 8080
    PROXY_USERNAME = None
    PROXY_PASSWORD = None

class ForecastDataset:
    def __init__(self, last_update, day_of_week, low, high, condition_url, condition_text, alert_text, alert_message, precip, humidity, wind_dir, wind_dir_numeric, wind_speed, wind_gusts, timezone, bar_read, bar_desc, uv_index, uv_text, dew_point, observatory, visibility, city, country, sunrise, sunset, moon_age, moon_percent_illuminated):
        self.last_update = last_update
        self.day_of_week = day_of_week
        self.low = low
        self.high = high
        self.condition_url = condition_url
        self.condition_text = condition_text
        self.alert_text = alert_text
        self.alert_message = alert_message
        self.precip = precip
        self.humidity = humidity
        self.wind_dir = wind_dir
        self.wind_dir_numeric = wind_dir_numeric
        self.wind_speed = wind_speed
        self.wind_gusts = wind_gusts
        self.timezone = timezone
        self.bar_read = bar_read
        self.bar_desc = bar_desc
        self.uv_index = uv_index
        self.uv_text = uv_text
        self.dew_point = dew_point
        self.observatory = observatory
        self.visibility = visibility
        self.city = city
        self.country = country
        self.sunrise = sunrise
        self.sunset = sunset
        self.moon_age = moon_age
        self.moon_percent_illuminated = moon_percent_illuminated     
        
class ForecastLocation:
    timestamp = None
    
    def __init__(self, current, day, night):
        self.current = current
        self.day = day
        self.night = night
        self.timestamp = datetime.today()
        
    def outdated(self, mins):
        if datetime.today() > self.timestamp + timedelta(minutes=mins):
            return True
        else:
            return False

# start ignoring translations required at runtime
def _(text): return text

class ForecastText:

    # translatable dictionaries
    conditions_text = {
        "0": _(u"Drizzle"),
        "1": _(u"Rain"),
        "2": _(u"Snow"),
        "3": _(u"Snow Grains"),
        "4": _(u"Ice Crystals"),
        "5": _(u"Ice Pellets"),
        "6": _(u"Hail"),
        "7": _(u"Mist"),
        "8": _(u"Fog"),
        "9": _(u"Smoke"),
        "10": _(u"Volcanic Ash"),
        "11": _(u"Widespread Dust"),
        "12": _(u"Sand"),
        "13": _(u"Haze"),
        "14": _(u"Spray"),
        "15": _(u"Dust Whirls"),
        "16": _(u"Sandstorm"),
        "17": _(u"Low Drifting Snow"),
        "18": _(u"Low Drifting Widespread Dust"),
        "19": _(u"Low Drifting Sand"),
        "20": _(u"Blowing Snow"),
        "21": _(u"Blowing Widespread Dust"),
        "22": _(u"Blowing Sand"),
        "23": _(u"Rain Mist"),
        "24": _(u"Rain Showers"),
        "25": _(u"Snow Showers"),
        "26": _(u"Ice Pellet Showers"),
        "27": _(u"Hail Showers"),
        "28": _(u"Small Hail Showers"),
        "29": _(u"Thunderstorm"),
        "30": _(u"Thunderstorms and Rain"),
        "31": _(u"Thunderstorms and Snow"),
        "32": _(u"Thunderstorms and Ice Pellets"),
        "33": _(u"Thunderstorms with Hail"),
        "34": _(u"Thunderstorms with Small Hail"),
        "35": _(u"Freezing Drizzle"),
        "36": _(u"Freezing Rain"),
        "37": _(u"Freezing Fog"),
        "38": _(u"Light Drizzle"),
        "39": _(u"Light Rain"),
        "40": _(u"Light Snow"),
        "41": _(u"Light Snow Grains"),
        "42": _(u"Light Ice Crystals"),
        "43": _(u"Light Ice Pellets"),
        "44": _(u"Light Hail"),
        "45": _(u"Light Mist"),
        "46": _(u"Light Fog"),
        "47": _(u"Light Smoke"),
        "48": _(u"Light Volcanic Ash"),
        "49": _(u"Light Widespread Dust"),
        "50": _(u"Light Sand"),
        "51": _(u"Light Haze"),
        "52": _(u"Light Spray"),
        "53": _(u"Light Dust Whirls"),
        "54": _(u"Light Sandstorm"),
        "55": _(u"Light Low Drifting Snow"),
        "56": _(u"Light Low Drifting Widespread Dust"),
        "57": _(u"Light Low Drifting Sand"),
        "58": _(u"Light Blowing Snow"),
        "59": _(u"Light Blowing Widespread Dust"),
        "60": _(u"Light Blowing Sand"),
        "61": _(u"Light Rain Mist"),
        "62": _(u"Light Rain Showers"),
        "63": _(u"Light Snow Showers"),
        "64": _(u"Light Ice Pellet Showers"),
        "65": _(u"Light Hail Showers"),
        "66": _(u"Light Small Hail Showers"),
        "67": _(u"Light Thunderstorm"),
        "68": _(u"Light Thunderstorms and Rain"),
        "69": _(u"Light Thunderstorms and Snow"),
        "70": _(u"Light Thunderstorms and Ice Pellets"),
        "71": _(u"Light Thunderstorms with Hail"),
        "72": _(u"Light Thunderstorms with Small Hail"),
        "73": _(u"Light Freezing Drizzle"),
        "74": _(u"Light Freezing Rain"),
        "75": _(u"Light Freezing Fog"),
        "76": _(u"Heavy Drizzle"),
        "77": _(u"Heavy Rain"),
        "78": _(u"Heavy Snow"),
        "79": _(u"Heavy Snow Grains"),
        "80": _(u"Heavy Ice Crystals"),
        "81": _(u"Heavy Ice Pellets"),
        "82": _(u"Heavy Hail"),
        "83": _(u"Heavy Mist"),
        "84": _(u"Heavy Fog"),
        "85": _(u"Heavy Smoke"),
        "86": _(u"Heavy Volcanic Ash"),
        "87": _(u"Heavy Widespread Dust"),
        "88": _(u"Heavy Sand"),
        "89": _(u"Heavy Haze"),
        "90": _(u"Heavy Spray"),
        "91": _(u"Heavy Dust Whirls"),
        "92": _(u"Heavy Sandstorm"),
        "93": _(u"Heavy Low Drifting Snow"),
        "94": _(u"Heavy Low Drifting Widespread Dust"),
        "95": _(u"Heavy Low Drifting Sand"),
        "96": _(u"Heavy Blowing Snow"),
        "97": _(u"Heavy Blowing Widespread Dust"),
        "98": _(u"Heavy Blowing Sand"),
        "99": _(u"Heavy Rain Mist"),
        "100": _(u"Heavy Rain Showers"),
        "101": _(u"Heavy Snow Showers"),
        "102": _(u"Heavy Ice Pellet Showers"),
        "103": _(u"Heavy Hail Showers"),
        "104": _(u"Heavy Small Hail Showers"),
        "105": _(u"Heavy Thunderstorm"),
        "106": _(u"Heavy Thunderstorms and Rain"),
        "107": _(u"Heavy Thunderstorms and Snow"),
        "108": _(u"Heavy Thunderstorms and Ice Pellets"),
        "109": _(u"Heavy Thunderstorms with Hail"),
        "110": _(u"Heavy Thunderstorms with Small Hail"),
        "111": _(u"Heavy Freezing Drizzle"),
        "112": _(u"Heavy Freezing Rain"),
        "113": _(u"Heavy Freezing Fog"),
        "114": _(u"Overcast"),
        "115": _(u"Clear"),
        "116": _(u"Partly Cloudy"),
        "117": _(u"Mostly Cloudy"),
        "118": _(u"Scattered Clouds"),
        "-": _(u"N/A")
    }

    day_of_week_short = {
        "Today": _(u"Now"),
        "Monday": _(u"Mon"),
        "Tuesday": _(u"Tue"),
        "Wednesday": _(u"Wed"),
        "Thursday": _(u"Thu"),
        "Friday": _(u"Fri"),
        "Saturday": _(u"Sat"),
        "Sunday": _(u"Sun")
    }    
        
    # this now returns ascii code
    bearing_arrow_font = {
        "S": 0x31,"N/A"
        "SSW": 0x32,
        "SW": 0x33,
        "WSW": 0x34,
        "W": 0x35,
        "WNW": 0x36,
        "NW": 0x37,
        "NNW": 0x38,
        "N": 0x39,
        "NNE": 0x3a,
        "NE": 0x3b,
        "ENE": 0x3c,
        "E": 0x3d,
        "ESE": 0x3e,
        "SE": 0x3f,
        "SSE": 0x40,
    }

    bearing_icon = {
        "calm": "00",
        "VAR": "01",
        "S": "05",
        "SSW": "06",
        "SW": "07",
        "WSW": "08",
        "W": "09",
        "WNW": "10",
        "NW": "11",
        "NNW": "12",
        "N": "13",
        "NNE": "14",
        "NE": "15",
        "ENE": "16",
        "E": "17",
        "ESE": "18",
        "SE": "19",
        "SSE": "20"
    }        
    
    # some general things...
    general = {
 	"n/a": _(u"n/a"),
	'N/A': _(u"N/A"),
	'Not Available': _(u"Not Available"),
	'unknown': _(u"unknown"),
	'NONE': _(u"NONE"),
	'day': _(u"day"),
	'night': _(u"night")
    }
    
    # UV index ...
    UV_index = {
	"Extreme": _(u"Extreme"),
	"Very high": _(u"Very High"),
	"High": _(u"High"),
	"Moderate": (u"Moderate"),
	"Low": _(u"Low")
    }
    
    # tendencies used for barometric pressure
    bar_pressure = {
	"Very Low": _(u"Very Low"),
	"Moderate": _(u"Moderate"),
	"rising": _(u"rising"),
	"falling": _(u"falling"),
	"steady": _(u"steady"),
	"calm": _(u"calm")
    }


    # wind directions long
    wind_directions_long = {
	"East": _(u"East"),
	"East Northeast": _(u"East Northeast"),
	"East Southeast": _(u"East Southeast"),
	"North": _(u"North"),
	"Northeast": _(u"Northeast"),
	"North Northeast": _(u"North Northeast"),
	"North Northwest": _(u"North Northwest"),
	"Northwest": _(u"Northwest"),
	"South": _(u"South"),
	"Souteast": _(u"Southeast"),
	"South Southeast": _(u"South Southeast"),
	"South Southwest": _(u"South Southwest"),
	"Southwest": _(u"Southwest"),
	"variable": _(u"variable"),
	"West": _(u"West"),
	"West Northwest": _(u"West Northwest"),
	"West Southwest": _(u"West Southwest")
    }
    
    wind_directions_short = {
	"E": _(u"E"),
	"ENE": _(u"ENE"),
	"ESE": _(u"ESE"),
	"N": _(u"N"),
	"NE": _(u"NE"),
	"NNE": _(u"NNE"),
	"NNW": _(u"NNW"),
	"NW": _(u"NW"),
	"S": _(u"S"),
	"SE": _(u"SE"),
	"SSE": _(u"SSE"),
	"SSW": _(u"SSW"),
	"SW": _(u"SW"),
	"W": _(u"W"),
	"WNW": _(u"WNW"),
	"WSW": _(u"WSW")
    }

    days_of_week = {
	"Today": _(u"Today"),
	"Monday": _(u"Monday"),
	"Tuesday":_(u"Tuesday"),
	"Wednesday":_(u"Wednesday"),
	"Thursday": _(u"Thursday"),
	"Friday": _(u"Friday"),
	"Saturday": _(u"Saturday"),
	"Sunday": _(u"Sunday")
    }
    
    # first all about moon phases
    moon_phase = {    
    "New": _(u"New"),
    "First Quarter": _(u"First Quarter"),
    "Full": _(u"Full"),
    "Last Quarter": _(u"Last Quarter"),
    "Waning Crescent": _(u"Waning Crescent"),
    "Waning Gibbous": _(u"Waning Gibbous"),
    "Waxing Crescent": _(u"Waxing Crescent"),
    "Waxing Gibbous": _(u"Waxing Gibbous")
    }

    locale_lang_code = {
    #"" : "AF", #Afrikaans
    #"" : "AL", #Albanian
    #"" : "AR", #Arabic
    #"" : "HY", #Armenian
    #"" : "AZ", #Azerbaijani
    #"" : "EU", #Basque
    #"" : "BY", #Belarusian
    "bg" : "BU", #Bulgarian
    #"" : "LI", #British English
    #"" : "MY", #Burmese
    #"" : "CA", #Catalan
    "zh" : "CN", #Chinese - Simplified
    #"" : "TW", #Chinese - Traditional
    #"" : "CR", #Croatian
    "cs" : "CZ", #Czech
    #"" : "DK", #Danish
    #"" : "DV", #Dhivehi
    "nl" : "NL", #Dutch
    "en" : "EN", #English
    #"" : "EO", #Esperanto
    #"" : "ET", #Estonian
    #"" : "FA", #Farsi
    #"" : "FI", #Finnish
    "fr" : "FR", #French
    #"" : "FC", #French Canadian
    #"" : "GZ", #Galician
    "de" : "DL", #German
    #"" : "KA", #Georgian
    #"" : "GR", #Greek
    #"" : "GU", #Gujarati
    #"" : "HT", #Haitian Creole
    #"" : "IL", #Hebrew
    #"" : "HI", #Hindi
    #"" : "HU", #Hungarian
    #"" : "IS", #Icelandic
    #"" : "IO", #Ido
    #"" : "ID", #Indonesian
    #"" : "IR", #Irish Gaelic
    "it" : "IT", #Italian
    #"" : "JP", #Japanese
    #"" : "JW", #Javanese
    #"" : "KM", #Khmer
    #"" : "KR", #Korean
    #"" : "KU", #Kurdish
    #"" : "LA", #Latin
    #"" : "LV", #Latvian
    "lt" : "LT", #Lithuanian
    #"" : "ND", #Low German
    #"" : "MK", #Macedonian
    #"" : "MT", #Maltese
    #"" : "GM", #Mandinka
    #"" : "MI", #Maori
    #"" : "MR", #Marathi
    #"" : "MN", #Mongolian
    #"" : "NO", #Norwegian
    #"" : "OC", #Occitan
    #"" : "PS", #Pashto
    #"" : "GN", #Plautdietsch
    "pl" : "PL", #Polish
    #"" : "BR", #Portuguese
    #"" : "PA", #Punjabi
    "ro" : "RO", #Romanian
    #"" : "RU", #Russian
    #"" : "SR", #Serbian
    "sk" : "SK", #Slovak
    #"" : "SL", #Slovenian
    "es" : "SP", #Spanish
    #"" : "SI", #Swahili
    "sv" : "SW" #Swedish
    #"" : "CH", #Swiss
    #"" : "TL", #Tagalog
    #"" : "TT", #Tatarish
    #"" : "TH", #Thai
    #"" : "TR", #Turkish
    #"" : "TK", #Turkmen
    #"" : "UA", #Ukrainian
    #"" : "UZ", #Uzbek
    #"" : "VU", #Vietnamese
    #"" : "CY", #Welsh
    #"" : "SN", #Wolof
    #"" : "JI", #Yiddish - transliterated
    #"" : "YI" #Yiddish - unicode
    }
    
        
# end ignoring translations
del _

class ForecastInfo:
    
    # design time variables
    options = None
    config = None
    forecast_data = {}
    # a list of locations for which an attempt was made to load them
    # locations in this list are not loaded again (if there was an error,
    # this makes sure it doesn't reapeat over and over)
    loaded_locations = []
    error = ""
    errorfound = False
    
    # design time settings
    CACHE_FILENAME = ".conkyForecastWU-<LOCATION>.cache"
    FULLMOON_PAST_STATE_CACHE_FILENAME = ".conkyForecastWU-<LOCATION>.fullmoonpaststate.cache"
    CONDITION_IMAGE_FILENAME = ".conkyForecastWU-WI-<CONDITION>.gif"

    def __init__(self, options):

        self.options = options
                                         
        self.loadConfigData()
        
        # setup timeout for connections
        # TODO: seems like this doesn't work in all cases..
        socket.setdefaulttimeout(self.config.CONNECTION_TIMEOUT)
        
        # set the locale
        if self.options.locale == None:
            if self.config.LOCALE == "":
                self.options.locale = locale.getdefaultlocale()[0][0:2]
            else:
                self.options.locale = self.config.LOCALE

        self.logInfo("Locale set to " + self.options.locale)
        
        # if not the default "en" locale, configure the i18n language translation    
        if self.options.locale != "en":

            self.logInfo("Looking for translation file for '%s' under %s" % (self.options.locale, locale_directory))
            
            if gettext.find(domain, locale_directory, languages=[self.options.locale]) != None:
                self.logInfo("Translation file found for '%s'" % self.options.locale)
                
                try:
                    trans = gettext.translation(domain, locale_directory, languages=[self.options.locale])
                    trans.install(unicode=True)
                    self.logInfo("Translation installed for '%s'" % self.options.locale)
                    
                except Exception, e:
                    self.logError("Unable to load translation for '%s' %s" % (self.options.locale, e.__str__()))
            else:
                self.logInfo("Translation file not found for '%s', defaulting to 'en'" % self.options.locale)
                self.options.locale = "en"

        # setup location code if not set
        if self.options.location == None:
            self.options.location = self.config.DEFAULT_LOCATION           
        
        # setup a proxy if defined
        if self.config.PROXY_HOST != None:
            if self.config.PROXY_USERNAME != None and self.config.PROXY_PASSWORD != None:
                self.logInfo("Setting up proxy '%s:%d', with username and password"%(self.config.PROXY_HOST,self.config.PROXY_PORT))
                proxyurl = "http://%s:%s@%s:%d"%(self.config.PROXY_USERNAME,self.config.PROXY_PASSWORD,self.config.PROXY_HOST,self.config.PROXY_PORT)
            else:
                self.logInfo("Setting up proxy '%s:%d', without username and password"%(self.config.PROXY_HOST,self.config.PROXY_PORT))
                proxyurl = "http://%s:%d"%(self.config.PROXY_HOST,self.config.PROXY_PORT)

            try:
                proxy_support = urllib2.ProxyHandler({"http" : proxyurl})
                opener = urllib2.build_opener(proxy_support, urllib2.HTTPHandler)
                urllib2.install_opener(opener)
            except Exception, e:
                self.logError("Unable to setup proxy: %s"%e.__str__())
        else:
            self.logInfo("Not using a proxy as none is defined")

        # Check if the location is loaded, if not, load it. If it can't be loaded, there was an error
        if not self.checkAndLoad():
            self.logError("Failed to load the location cache for %s"%self.options.location)
                    
    def loadConfigData(self):
        try:         
            # load .conkyForecast.config from the options setting
            configfilepath = os.path.expanduser(self.options.config)
                                          
            if os.path.exists(configfilepath):
                
                self.config = ForecastConfig()
                
                #load the file
                fileinput = open(configfilepath)
                lines = fileinput.read().split("\n")
                fileinput.close() 

                for line in lines:
                    line = line.strip()
                    if len(line) > 0 and line[0:1] != "#": # ignore commented lines or empty ones

                        splitpos = line.find("=")
                        name = line[:splitpos-1].strip().upper() # config setting name on the left of =
                        value = line[splitpos+1:].split("#")[0].strip()
                        
                        if len(value) > 0:
                            if name == "BASE_WU_JSON_URL":
                                self.config.BASE_WU_JSON_URL = value
                            elif name == "BASE_WU_ICON_URL":
                                self.config.BASE_WU_ICON_URL = value
                            elif name == "MAXIMUM_DAYS_FORECAST":                               
                                try:
                                    self.config.MAXIMUM_DAYS_FORECAST = int(value)
                                except:
                                    self.logError("Invalid value of config option MAXIMUM_DAYS_FORECAST: " + value)                                
                            elif name == "CACHE_FOLDERPATH":
                                self.config.CACHE_FOLDERPATH = value
                            elif name == "PERSISTED_CACHE_FOLDERPATH":
                                self.config.PERSISTED_CACHE_FOLDERPATH = value                                
                            elif name == "ICON_SET":
                                self.config.ICON_SET = value
                            elif name == "CONNECTION_TIMEOUT":
                                try:
                                    self.config.CONNECTION_TIMEOUT = int(value)
                                except:
                                    self.logError("Invalid value of config option CONNECTION_TIMEOUT: " + value)
                            elif name == "EXPIRY_MINUTES":
                                try:
                                    self.config.EXPIRY_MINUTES = int(value)
                                except:
                                    self.logError("Invalid value of config option EXPIRY_MINUTES: " + value)
                            elif name == "TIME_FORMAT":
                                self.config.TIME_FORMAT = value
                            elif name == "DATE_FORMAT":
                                self.config.DATE_FORMAT = value                                    
                            elif name == "LOCALE":
                                self.config.LOCALE = value
                            elif name == "WU_ANVIL_LICENCE_KEY":
                                self.config.WU_ANVIL_LICENCE_KEY = value
                            elif name == "WU_STRATUS_LICENCE_KEY":
                                self.config.WU_STRATUS_LICENCE_KEY = value
                            elif name == "WU_CUMULUS_LICENCE_KEY":
                                self.config.WU_CUMULUS_LICENCE_KEY = value
                            elif name == "DEFAULT_LOCATION":
                                self.config.DEFAULT_LOCATION = value
                            elif name == "AUTO_NIGHT":
                                self.config.AUTO_NIGHT = self.parseBoolString(value)
                            elif name == "PROXY_HOST":
                                self.config.PROXY_HOST = value
                            elif name == "PROXY_PORT":                                
                                try:
                                    self.config.PROXY_PORT = int(value)
                                except:
                                    self.logError("Invalid value of config option PROXY_PORT: " + value)                                
                            elif name == "PROXY_USERNAME":
                                self.config.PROXY_USERNAME = value
                            elif name == "PROXY_PASSWORD":
                                self.config.PROXY_PASSWORD = value
                            else:
                                self.logError("Unknown option in config file: " + name)

                if self.options.verbose == True:
                    print >> sys.stdout,"*** CONFIG OPTIONS:"
                    print >> sys.stdout,"    BASE_WU_JSON_URL:", self.config.BASE_WU_JSON_URL
                    print >> sys.stdout,"    BASE_WU_ICON_URL:", self.config.BASE_WU_ICON_URL
                    print >> sys.stdout,"    MAXIMUM_DAYS_FORECAST:", self.config.MAXIMUM_DAYS_FORECAST
                    print >> sys.stdout,"    CACHE_FOLDERPATH:", self.config.CACHE_FOLDERPATH
                    print >> sys.stdout,"    PERSISTED_CACHE_FOLDERPATH:", self.config.PERSISTED_CACHE_FOLDERPATH                    
                    print >> sys.stdout,"    ICON_SET:", self.config.ICON_SET
                    print >> sys.stdout,"    CONNECTION_TIMEOUT:", self.config.CONNECTION_TIMEOUT
                    print >> sys.stdout,"    EXPIRY_MINUTES:", self.config.EXPIRY_MINUTES
                    print >> sys.stdout,"    TIME_FORMAT:", self.config.TIME_FORMAT
                    print >> sys.stdout,"    DATE_FORMAT:", self.config.DATE_FORMAT                
                    print >> sys.stdout,"    LOCALE:", self.config.LOCALE
                    print >> sys.stdout,"    WU_ANVIL_LICENCE_KEY:", self.config.WU_ANVIL_LICENCE_KEY
                    print >> sys.stdout,"    WU_STRATUS_LICENCE_KEY:", self.config.WU_STRATUS_LICENCE_KEY
                    print >> sys.stdout,"    WU_CUMULUS_LICENCE_KEY:", self.config.WU_CUMULUS_LICENCE_KEY
                    print >> sys.stdout,"    DEFAULT_LOCATION:", self.config.DEFAULT_LOCATION
                    
            else:
                self.logError("Config data file %s not found, using defaults (Registration info is needed though)" % configfilepath)

        except Exception, e:
            self.logError("Error while loading config data, using defaults (Registration info is needed though): " + e.__str__())


    def checkAndLoad(self):
        
        # if the location was not loaded before (or attempted to load)
        if not self.options.location in self.loaded_locations:
            # add it to the list so it doesn't get loaded again (or attempted to load)
            self.loaded_locations.append(self.options.location)
            
            # define CACHE_FILEPATH based on cache folder and location code
            CACHE_FILEPATH = os.path.expanduser(os.path.join(self.config.CACHE_FOLDERPATH, self.CACHE_FILENAME.replace("<LOCATION>", self.options.location.replace("/","-"))))

            if not self.forecast_data.has_key(self.options.location):
                if os.path.exists(CACHE_FILEPATH):
                    try:
                        self.logInfo("Loading cache file " + CACHE_FILEPATH)
                        file = open(CACHE_FILEPATH, 'rb')
                        self.forecast_data[self.options.location] = pickle.load(file)
                        file.close()
                    except Exception, e:
                        self.logError("Unable to read the cache file %s: %s" % (CACHE_FILEPATH, e.__str__()))
                        #TODO: get to the bottom of failure to load pickled cache file, is this a 2.7.1 issue?
                        self.logInfo("Deleting cache file due to loading issues, it will be prepared again")
                        os.remove(CACHE_FILEPATH)
                        #return False
        
            # check the data in the dictionary and update if outdated
            # if there was an update, store the new data in the cache file
            if self.checkAndUpdate(self.options.location) == True:
                try:
                    self.logInfo("Saving updated cache file " + CACHE_FILEPATH)
                    file = open(CACHE_FILEPATH, 'wb')
                    pickle.dump(self.forecast_data[self.options.location], file)
                    file.close()
                except Exception, e:
                    self.logError("Unable to save cache file %s: %s" % (CACHE_FILEPATH, e.__str__()))
                    return False

        # if the location is still not in cache, print an error and return false to writeOutput()
        if self.forecast_data.has_key(self.options.location):
            return True
        else:
            self.logError("Location %s is not in cache." % self.options.location) 
            return False
        

    def checkAndUpdate(self, location):
        # if the location is outdated or the refetch is forced..
        if not self.forecast_data.has_key(location) or \
           self.forecast_data[location].outdated(self.config.EXPIRY_MINUTES) or \
           self.options.refetch == True:

            # obtain current conditions data from xoap service
            try:

                # attempt to determine the WU lang code from supported locale options (should expand this if using just lang code and no translation is sufficient)
                try:
                    langcode = ForecastText.locale_lang_code[self.options.locale]
                except KeyError:
                    self.logError("Locale '%s' is not currently supported by WU lang codes, defaulting to English for the weather feed data."%self.options.locale)
                    langcode = "EN" # English
                    
                if self.options.debug == True:
                    #jsonfile = "%s/conkyForecastWU.json"%(app_path)
                    jsonfile = "%s/conkyForecastWU_%s.json"%(app_path, langcode)
                    self.logInfo("Fetching weather data from " + jsonfile)
                    jsonsrc = open(jsonfile)
                else:
                    #jsonurl = self.config.BASE_WU_JSON_URL.replace("<LOCATION>",self.options.location).replace("<WU_STRATUS_LICENCE_KEY>",self.config.WU_STRATUS_LICENCE_KEY)
                    jsonurl = self.config.BASE_WU_JSON_URL.replace("<LOCATION>",self.options.location).replace("<WU_STRATUS_LICENCE_KEY>",self.config.WU_STRATUS_LICENCE_KEY).replace("<LANGCODE>",langcode)                          
                    self.logInfo("Fetching weather data from " + jsonurl)
                    jsonsrc = urllib2.urlopen(jsonurl)
                    
                jsondata = jsonsrc.read()
                jsonsrc.close()                

            except Exception, e:
                self.logError("Server connection error: " + e.__str__())
                return False
                        
            else:
                # interrogate weather data
                try:
                    # parse the XML
                    self.weatherdic = json.loads(jsondata)

                    current = self.weatherdic["current_observation"]                    
                    
                    city = current["display_location"]["city"]
                    country = current["display_location"]["country"]
                    latitude = current["display_location"]["latitude"]
                    longitude = current["display_location"]["longitude"]

                    observatory = current["observation_location"]["city"]
                    timezone = current["local_tz_short"]
                    
                    condition_icon = current["icon"]
                    condition_url = self.config.BASE_WU_ICON_URL.replace("<ICON_SET>",self.config.ICON_SET).replace("<ICON>", condition_icon)
                    condition_text = self.properCase(current["weather"])
                                       
                    last_update = current["observation_time_rfc822"]
                    
                    day_of_week = "Today"
                                        
                    current_temp = str(current["temp_c"])

                    # try to get feels like from heatindex and windchill
                    current_temp_feels = current_temp
                    heat_index = str(current["heat_index_c"])
                    if heat_index != "NA":
                        current_temp_feels = heat_index

                    wind_chill = str(current["windchill_c"])
                    if wind_chill != "NA":
                        current_temp_feels = wind_chill
                        
                    precip = _(u"N/A")  
                    wind_direction = current["wind_dir"]
                    wind_direction_numeric = str(current["wind_degrees"])
                    wind_speed = str(current["wind_mph"])
                    wind_gusts = str(current["wind_gust_mph"])
                    humidity = current["relative_humidity"]                    
                    
                    bar_read = str(current["pressure_mb"])
                    bar_desc = current["pressure_trend"]
                    uv_index = str(current["UV"])
                    uv_text = str(current["UV"])
                    dew_point = str(current["dewpoint_c"])
                    visibility = str(current["visibility_km"])

                    alert_text = ""
                    alert_message = ""
                    alerts = self.weatherdic["alerts"]      
                    for alert in alerts:
                        alert_text = alert_text + alert['description'].strip() + "\n"
                        alert_message = alert_message + alert['message'].strip() + "\n"
                        
                    astronomy = self.weatherdic["moon_phase"]
                    sunrise = astronomy["sunrise"]["hour"]+":"+astronomy["sunrise"]["minute"]
                    sunset = astronomy["sunset"]["hour"]+":"+astronomy["sunset"]["minute"]
                    moon_age = astronomy["ageOfMoon"]
                    moon_percent_illuminated = astronomy["percentIlluminated"]                    

                    self.setFullMoonPastState(moon_age, moon_percent_illuminated)

                    current_forecast_data = ForecastDataset(last_update, day_of_week, current_temp_feels, current_temp, condition_url, condition_text, alert_text, alert_message, precip, humidity, wind_direction, wind_direction_numeric, wind_speed, wind_gusts, timezone, bar_read, bar_desc, uv_index, uv_text, dew_point, observatory, visibility, city, country, sunrise, sunset, moon_age, moon_percent_illuminated)
                    
                    bar_read = _(u"N/A")
                    bar_desc = _(u"N/A")
                    visibility = _(u"N/A")
                    uv_index = _(u"N/A")
                    uv_text = _(u"N/A")
                    dew_point = _(u"N/A")
                    alert_text = _(u"N/A")
                    
                    sunrise = _(u"N/A")
                    sunset = _(u"N/A")
                    moon_age = _(u"N/A")
                    moon_percent_illuminated = _(u"N\A")
                                            
                    day_forecast_data_list = []
                    night_forecast_data_list = []

                    forecastlist = self.weatherdic["forecast"]["simpleforecast"]["forecastday"]
                    for forecastday in forecastlist:
                        last_update = time.strftime('%Y-%m-%d %I:%M:%S %p %Z',time.localtime(float(forecastday['date']['epoch'])))
                        daynumber = forecastday["period"]
                        day_of_week = forecastday["date"]["weekday"]
                        day_high_temp = str(forecastday["high"]["celsius"])
                        night_high_temp = str(forecastday["low"]["celsius"])                        

                        condition_icon = forecastday["icon"]
                        condition_url = self.config.BASE_WU_ICON_URL.replace("<ICON_SET>",self.config.ICON_SET).replace("<ICON>", condition_icon)
                        condition_text = self.properCase(forecastday["conditions"])                        
                        
                        precip = str(forecastday["pop"])
        
                        wind_speed = str(forecastday["avewind"]["mph"])
                        wind_gusts = str(forecastday["maxwind"]["mph"])
                        wind_direction = str(forecastday["avewind"]["dir"])
                        wind_direction_numeric = str(forecastday["avewind"]["degrees"])
                        
                        humidity = str(forecastday["avehumidity"])
                            
                        #day_low_temp = self.getApparentTemp(day_high_temp, wind_speed, humidity)
                        #night_low_temp = self.getApparentTemp(night_high_temp, wind_speed, humidity)
                        
                        day_low_temp = night_high_temp
                        night_low_temp = night_high_temp
                                                
                        day_forecast_data = ForecastDataset(last_update, day_of_week, day_low_temp, day_high_temp, condition_url, condition_text, alert_text, alert_message, precip, humidity, wind_direction, wind_direction_numeric, wind_speed, wind_gusts, timezone, bar_read, bar_desc, uv_index, uv_text, dew_point, observatory, visibility, city, country, sunrise, sunset, moon_age, moon_percent_illuminated)
                        day_forecast_data_list.append(day_forecast_data)
    
                        # no night data available at the mo....so just populating with day data
                        night_forecast_data = ForecastDataset(last_update, day_of_week, night_low_temp, night_high_temp, condition_url, condition_text, alert_text, alert_message, precip, humidity, wind_direction, wind_direction_numeric, wind_speed, wind_gusts, timezone, bar_read, bar_desc, uv_index, uv_text, dew_point, observatory, visibility, city, country, sunrise, sunset, moon_age, moon_percent_illuminated)
                        night_forecast_data_list.append(night_forecast_data)
                        
                    self.forecast_data[location] = ForecastLocation(current_forecast_data, day_forecast_data_list, night_forecast_data_list)
    
                    return True
            
                except Exception, e:
                    self.logError("Error reading weather data: " + e.__str__())
                    return False

    def getTimestampOutput(self, timestamp, minuteshide):            
        # minuteshide:
        # None = disabled
        # -1 = hide days and use config.EXPIRY_MINUTES
        # 0 = hide days and always show hours
        
        output = u""
        
        today = datetime.today()
        days = today.day - timestamp.day
        if days or minuteshide == None:
            output += timestamp.strftime(self.config.DATE_FORMAT)
        
        if minuteshide == -1:
            minuteshide = self.config.EXPIRY_MINUTES
            
        delta = today - timestamp
        if days or minuteshide == None or minuteshide == 0 or delta.seconds > minuteshide * 60:
            if (len(output) > 0):
                output += " "
            output += timestamp.strftime(self.config.TIME_FORMAT)
        
        return output


    def getDatatypeFromSet(self, datatype, set, shortweekday, imperial, beaufort, metrespersecond, tempunit, speedunit, distanceunit, pressureunit, minuteshide, centeredwidth):
        output = u""

        try:
            if datatype == "LU":
                output = set.last_update.strip()                    
            elif datatype == "LF":
                output = self.getTimestampOutput(self.forecast_data[self.options.location].timestamp, minuteshide)
            elif datatype == "DW":
                if shortweekday == True:
                    output = _(ForecastText.day_of_week_short[set.day_of_week])
                else:
                    output = _(set.day_of_week)
            elif datatype == "WI":
                output = self.getImageSrcForCondition(set.condition_url, set.condition_text)
            elif datatype == "LT":
                if self.isNumeric(set.low) == True:
                    if imperial == True:
                        string = self.convertCelsiusToFahrenheit(set.low)
                    else:
                        string = self.roundOff(set.low)
                    string = string + tempunit
                else:
                    string = _(set.low)
                output = string
            elif datatype == "HT":
                if self.isNumeric(set.high) == True:
                    if imperial == True:
                        string = self.convertCelsiusToFahrenheit(set.high)
                    else:
                        string = self.roundOff(set.high)
                    string = string + tempunit
                else:
                    string = _(set.high)
                output = string                      
            elif datatype == "CT":
                output = _(set.condition_text)
            elif datatype == "AT":
                output = _(set.alert_text)
            elif datatype == "AM":
                output = _(set.alert_message)   
            elif datatype == "PC":
                if self.isNumeric(set.precip) == True:
                    string = set.precip + u"%"
                else:
                    string = _(set.precip)
                output = string
            elif datatype == "HM":
                if self.isNumeric(set.humidity) == True:
                    string = set.humidity + u"%"
                else:
                    string = _(set.humidity)
                output = string
            elif datatype == "WD":
                output = _(set.wind_dir)
            elif datatype == "BF":
                if set.wind_speed.lower() == "calm":
                    output = chr(0x25)
                else:
                    if (set.wind_dir == "VAR"):
                        output = chr(0x22) # 2nd level var arrow
                    else:
                        try:
                            # for the old datatype, add 0x10, that makes the output in the A-P range,
                            # which is the 2nd level arrow
                            output = chr(ForecastText.bearing_arrow_font[set.wind_dir] + 0x10)
                        except KeyError:
                            # if the value wasn't found in ForecastText.bearing_arrow_font, use space
                            output = "-"
            elif datatype == "BS":
                if set.wind_speed.lower() == "calm":
                    output = chr(0x25)
                elif self.isNumeric(set.wind_speed) == True:
                    if (set.wind_dir == "VAR"):
                        output = chr(0x21 + self.getWindLevel(set.wind_speed))
                    else:
                        try:
                            output = chr(ForecastText.bearing_arrow_font[set.wind_dir] + self.getWindLevel(set.wind_speed) * 0x10)
                        except KeyError:
                            # if the value wasn't found in ForecastText.bearing_arrow_font, use N/A
                            output = "-"
                else:
                    try:
                        # if the speed is not "calm" but also not a number, add 0x10
                        # that makes the output in the A-P range, the 2nd level arrow
                        output = chr(ForecastText.bearing_arrow_font[set.wind_dir] + 0x10)
                    except KeyError:
                        # if the value wasn't found in ForecastText.bearing_arrow_font, use N/A
                        output = "-"
            elif datatype == "BI":
                if set.wind_speed.lower() == "calm":
                    output = self.getImagePathForBearing(ForecastText.bearing_icon["calm"])
                elif self.isNumeric(set.wind_speed) == True:
                    if (set.wind_dir == "VAR"):
                        output = self.getImagePathForBearing(int(ForecastText.bearing_icon[set.wind_dir]) + self.getWindLevel(set.wind_speed))
                    else:
                        try:
                            output = self.getImagePathForBearing(int(ForecastText.bearing_icon[set.wind_dir]) + self.getWindLevel(set.wind_speed)*16)
                        except KeyError:
                            # if the value wasn't found in ForecastText.bearing_icon, use calm code
                            output = self.getImagePathForBearing(ForecastText.bearing_icon["calm"])
                                        
            elif datatype == "WA":
                output = _(set.wind_dir_numeric)
            elif datatype == "WS":
                if self.isNumeric(set.wind_speed) == True:
                    if beaufort == True:
                        string = self.convertMPHtoBeaufort(set.wind_speed)
                    elif metrespersecond == True:
                        string = self.convertMPHtoMS(set.wind_speed)
                    elif imperial == True:
                        string = self.roundOff(set.wind_speed)
                    else:
                        string = self.convertMilesToKilometres(set.wind_speed)
                        
                    string = string + speedunit
                else:
                    string = _(set.wind_speed.lower())
                output = string
            elif datatype == "WG":
                if self.isNumeric(set.wind_gusts) == True:
                    if beaufort == True:
                        string = self.convertMPHtoBeaufort(set.wind_gusts)
                    elif metrespersecond == True:
                        string = self.convertMPHtoMS(set.wind_gusts)
                    elif imperial == True:
                        string = self.roundOff(set.wind_gusts)
                    else:
                        string = self.convertMilesToKilometres(set.wind_gusts)

                    string = string + speedunit
                else:
                    string = _(set.wind_gusts) # need to define translations
                output = string              
            elif datatype == "BR":
                if self.isNumeric(set.bar_read) == True:
                    if imperial == True:
                        string = self.convertMillibarsToInches(set.bar_read,2)
                    else:
                        string = self.roundOff(set.bar_read)
                    string = string + pressureunit
                else:
                    string = _(set.bar_read)
                output = string
            elif datatype == "BD":
                output = _(set.bar_desc) # need to define translations
            elif datatype == "UI":
                output = _(set.uv_index)
            elif datatype == "UT":
                output = _(set.uv_text)
            elif datatype == "DP":
                if self.isNumeric(set.dew_point) == True:
                    if imperial == True:
                        string = self.convertCelsiusToFahrenheit(set.dew_point)
                    else:
                        string = self.roundOff(set.dew_point)
                    string = string + tempunit
                else:
                    string = _(set.dew_point)
                output = string
            elif datatype == "OB":
                output = set.observatory
            elif datatype == "VI":
                if self.isNumeric(set.visibility) == True:
                    if imperial == True:
                        string = self.convertKilometresToMiles(set.visibility,1)
                    else:
                        string = self.roundOff(set.visibility)
                    string = string + distanceunit
                else:
                    string = _(set.visibility)
                output = string            
            elif datatype == "CN":
                output = set.city
            elif datatype == "CO":
                output = set.country                
            elif datatype == "SR":
                try:
                    srtime = datetime.strptime(set.sunrise, "%H:%M")
                    output = srtime.strftime(self.config.TIME_FORMAT)
                except:
                    self.logError("Failed to extract sunrise datetime from data using standard code" + traceback.format_exc())
                    try:
                        self.logError("Attempting to extract sunrise datetime from data using python 2.4 compliant code")
                        srtime = datetime(*(time.strptime(set.sunrise, "%H:%M"))[0:3])
                        output = srtime.strftime(self.config.TIME_FORMAT)
                    except:
                        self.logError("Failed to extract sunrise datetime from data using python 2.4 compliant code" + traceback.format_exc())
                        output = set.sunrise                                
            elif datatype == "SS":
                try:
                    sstime = datetime.strptime(set.sunset, "%H:%M")
                    output = sstime.strftime(self.config.TIME_FORMAT)
                except:
                    self.logError("Failed to extract sunset datetime from data using standard code" + traceback.format_exc())
                    try:
                        self.logError("Attempting to extract sunset datetime from data using python 2.4 compliant code")
                        sstime = datetime(*(time.strptime(set.sunset, "%H:%M"))[0:3])
                        output = sstime.strftime(self.config.TIME_FORMAT)
                    except:
                        self.logError("Failed to extract sunset datetime from data using python 2.4 compliant code" + traceback.format_exc())
                        output = set.sunset                        
            elif datatype == "DL":
                srtime = None
                sstime = None
                
                try:
                    srtime = datetime.strptime(set.sunrise, "%H:%M")
                    sstime = datetime.strptime(set.sunset, "%H:%M")                  
                except:
                    self.logError("Failed to extract sunrise/sunset from data using standard code" + traceback.format_exc())
                    try:
                        self.logError("Attempting to extract sunrise/sunset from data using python 2.4 compliant code")
                        srtime = datetime(*(time.strptime(set.sunrise, "%H:%M"))[0:3])
                        sstime = datetime(*(time.strptime(set.sunset, "%H:%M"))[0:3])
                    except:
                        self.logError("Failed to extract sunrise/sunset from data using python 2.4 compliant code" + traceback.format_exc())

                if srtime != None and sstime != None:
                    delta = sstime - srtime              
                    output = self.getFormattedTimeFromSeconds(delta.seconds)
                else:
                    output = "??:??"
            elif datatype == "MP":
                output = _(self.getMoonPhase(set.moon_age, set.moon_percent_illuminated)) # need to define translations
            elif datatype == "MI":            
                output = self.getMoonImageURL(set.moon_age, set.moon_percent_illuminated)     
            else:
                self.logError("Unknown datatype requested: " + datatype)

        except KeyError, e:
            self.logError("Unknown value %s encountered for datatype '%s'! Please report this!" % (e.__str__(), datatype))
        
        # set the width if it is set, either left trimming or centering the text in spaces as requested
        if centeredwidth != None and self.isNumeric(centeredwidth) == True:
            if centeredwidth < len(output):
                output = output[:centeredwidth]
            else:
                output = output.center(int(centeredwidth))    
                            
        return output


    def getDatasetOutput(self, datatype, startday, endday, night, shortweekday, imperial, beaufort, metrespersecond, hideunits, hidedegreesymbol, spaces, minuteshide, centeredwidth):

        output = u""

        # Check if the location is loaded, if not, load it. If it can't be loaded, there was an error
        if not self.checkAndLoad():
            self.logError("Failed to load the location cache")
            return u""
        
        # define current units for output
        if hideunits == False:
            if imperial == False:
                tempunit = _(u"°C")
                speedunit = _(u"kph")
                distanceunit = _(u"km")
                pressureunit = _(u"mb")
            else:
                tempunit = _(u"°F")
                speedunit = _(u"mph")
                distanceunit = _(u"m")
                pressureunit = _(u"in")
                
            # override speed units if beaufort selected
            if beaufort == True:
                speedunit = u""
                
            if metrespersecond == True:
                speedunit = u"m/s"
        else:
            # remove degree symbol if not required
            if hidedegreesymbol == False:
                tempunit = u"°"
            else:
                tempunit = u""
                
            speedunit = u""
            distanceunit = u""
            pressureunit = u""

        if startday == None:
            output += self.getDatatypeFromSet(datatype, self.forecast_data[self.options.location].current, shortweekday, imperial, beaufort, metrespersecond, tempunit, speedunit, distanceunit, pressureunit, minuteshide, centeredwidth)
        else: # forecast data

            # ensure startday and enday are within the forecast limit
            
            if startday < 0:
                startday = 0
                self.logError("--startday set beyond forecast limit, reset to minimum of 0")
            elif startday > self.config.MAXIMUM_DAYS_FORECAST:
                startday = self.config.MAXIMUM_DAYS_FORECAST
                self.logError("--startday set beyond forecast limit, reset to maximum of " + str(self.config.MAXIMUM_DAYS_FORECAST))
                
            if endday == None: # if no endday was set use startday
                endday = startday
            elif endday < 0:
                endday = 0
                self.logError("--endday set beyond forecast limit, reset to minimum of 0")
            elif endday > self.config.MAXIMUM_DAYS_FORECAST:
                endday = self.config.MAXIMUM_DAYS_FORECAST
                self.logError("--endday set beyond forecast limit, reset to maximum of " + str(self.config.MAXIMUM_DAYS_FORECAST))
                
            for daynumber in range(startday, endday + 1):
                
                # if AUTO_NIGHT config is true then handle N/A output, by using the night option between 2pm and 2am, when the startday = 0. 
                if self.config.AUTO_NIGHT == True and daynumber == 0:
                    now = datetime.now()
                    hour = now.hour
                    if hour > 13 or hour < 2:
                        night = True
                
                if night == True:
                    output += self.getDatatypeFromSet(datatype, self.forecast_data[self.options.location].night[daynumber], shortweekday, imperial, beaufort, metrespersecond, tempunit, speedunit, distanceunit, pressureunit, minuteshide, centeredwidth)
                else:
                    output += self.getDatatypeFromSet(datatype, self.forecast_data[self.options.location].day[daynumber], shortweekday, imperial, beaufort, metrespersecond, tempunit, speedunit, distanceunit, pressureunit, minuteshide, centeredwidth)
                    
                if daynumber != endday:
                    output += self.getSpaces(spaces)

        return output

    def getTemplateItemOutput(self, template_text):
        
        # keys to template data
        LOCATION_KEY = "location"
        DATATYPE_KEY = "datatype"
        STARTDAY_KEY = "startday"
        ENDDAY_KEY = "endday"
        NIGHT_KEY = "night"
        SHORTWEEKDAY_KEY = "shortweekday"
        IMPERIAL_KEY = "imperial"
        BEAUFORT_KEY = "beaufort"
        METRESPERSECOND_KEY = "metrespersecond"
        HIDEUNITS_KEY = "hideunits"
        HIDEDEGREESYMBOL_KEY = "hidedegreesymbol"
        SPACES_KEY = "spaces"
        MINUTESHIDE_KEY = "minuteshide"
        CENTEREDWIDTH_KEY = "centeredwidth"
        
        location = self.options.location
        datatype = self.options.datatype
        startday = self.options.startday
        endday = self.options.endday
        night = self.options.night
        shortweekday = self.options.shortweekday
        imperial = self.options.imperial
        beaufort = self.options.beaufort
        metrespersecond = self.options.metrespersecond
        hideunits = self.options.hideunits
        hidedegreesymbol = self.options.hidedegreesymbol
        spaces = self.options.spaces
        minuteshide = self.options.minuteshide
        centeredwidth = self.options.centeredwidth
        
        for option in template_text.split('--'):
            if len(option) == 0 or option.isspace():
                continue
            
            # not using split here...it can't assign both key and value in one call, this should be faster
            x = option.find('=')
            if (x != -1):
                key = option[:x].strip()
                value = option[x + 1:].strip()
                if value == "":
                    value = None
            else:
                key = option.strip()
                value = None
            
            try:
                if key == LOCATION_KEY:
                    location = value
                elif key == DATATYPE_KEY:
                    datatype = value
                elif key == STARTDAY_KEY:
                    startday = int(value)
                elif key == ENDDAY_KEY:
                    endday = int(value)
                elif key == NIGHT_KEY:
                    night = True
                elif key == SHORTWEEKDAY_KEY:
                    shortweekday = True
                elif key == IMPERIAL_KEY:
                    imperial = True
                elif key == BEAUFORT_KEY:
                    beaufort = True
                elif key == METRESPERSECOND_KEY:
                    metrespersecond = True
                elif key == HIDEUNITS_KEY:
                    hideunits = True
                elif key == HIDEDEGREESYMBOL_KEY:
                    hidedegreesymbol = True
                elif key == SPACES_KEY:
                    spaces = int(value)
                elif key == MINUTESHIDE_KEY:
                    if value != None:
                        minuteshide = int(value)
                    else:
                        minuteshide = -1
                elif key == CENTEREDWIDTH_KEY:
                    centeredwidth = value
                else:
                    self.logError("Unknown template option: " + option)

            except (TypeError, ValueError):
                self.logError("Cannot convert option argument to number: " + option)
                return u""

        #REMOVED
        # Check if the location is loaded, if not, load it. If it can't be loaded, there was an error
        #if not self.checkAndLoad(location):
        #    self.logError("Failed to load the location cache")
        #    return u""        

        # store the location as the option for this moment in time
        self.options.location = location
        
        if datatype != None:
            return self.getDatasetOutput(datatype, startday, endday, night, shortweekday, imperial, beaufort, metrespersecond, hideunits, hidedegreesymbol, spaces, minuteshide, centeredwidth)
        else:
            self.logError("Template item does not have datatype defined")
            return u""

    def getOutputFromTemplate(self, template):
        output = u""
        end = False
        a = 0
        
        # a and b are indexes in the template string
        # moving from left to right the string is processed
        # b is index of the opening bracket and a of the closing bracket
        # everything between b and a is a template that needs to be parsed
        while not end:
            b = template.find('[', a)
            
            if b == -1:
                b = len(template)
                end = True
            
            # if there is something between a and b, append it straight to output
            if b > a:
                output += template[a : b]
                # check for the escape char (if we are not at the end)
                if template[b - 1] == '\\' and not end:
                    # if its there, replace it by the bracket
                    output = output[:-1] + '['
                    # skip the bracket in the input string and continue from the beginning
                    a = b + 1
                    continue
                    
            if end:
                break
            
            a = template.find(']', b)
            
            if a == -1:
                self.logError("Missing terminal bracket (]) for a template item")
                return u""
            
            # if there is some template text...
            if a > b + 1:
                output += self.getTemplateItemOutput(template[b + 1 : a])
            
            a = a + 1

        return output

    def writeOutput(self):
                
        if self.options.template != None:
            #load the file
            try:
                fileinput = codecs.open(os.path.expanduser(self.options.template), encoding='utf-8')
                template = fileinput.read()
                fileinput.close()
            except Exception, e:
                self.logError("Error loading template file: " + e.__str__())
            else:
                output = self.getOutputFromTemplate(template)
        else:         
            
            output = self.getDatasetOutput(self.options.datatype, self.options.startday, self.options.endday, self.options.night, self.options.shortweekday, self.options.imperial, self.options.beaufort, self.options.metrespersecond, self.options.hideunits, self.options.hidedegreesymbol, self.options.spaces, self.options.minuteshide, self.options.centeredwidth)
            
        print output.encode("utf-8")

    def logInfo(self, text):
        if self.options.verbose == True:
            print >> sys.stdout, "INFO: " + text

        if self.options.infologfile != None:
            datetimestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S") 
            fileoutput = open(self.options.infologfile, "ab")
            fileoutput.write(datetimestamp+" INFO: "+text+"\n")
            fileoutput.close()
            
    def logError(self, text):
        print >> sys.stderr, "ERROR: " + text
        
        if self.options.errorlogfile != None:
            datetimestamp = datetime.now().strftime("%Y-%m-%d %H:%M:%S") 
            fileoutput = open(self.options.errorlogfile, "ab")
            fileoutput.write(datetimestamp+" ERROR: "+text+"\n")
            fileoutput.close()

    def getText(self, parentNode, name):
        try:
            node = parentNode.getElementsByTagName(name)[0]
        except IndexError:
            raise Exception, "Data element <%s> not present under <%s>" % (name, parentNode.tagName)

        rc = ""
        for child in node.childNodes:
            if child.nodeType == child.TEXT_NODE:
                rc = rc + child.data
        return rc
    
    def getChild(self, parentNode, name, index = 0):
        try:
            return parentNode.getElementsByTagName(name)[index]
        except IndexError:
            raise Exception, "Data element <%s> is not present under <%s> (index %i)" % (name, parentNode.tagName, index)

    def getSpaces(self, spaces=1):
        string = u""
        for dummy in range(0, spaces):
            string = string + u" "
        return string

    def isNumeric(self, string):
        try:
            dummy = float(string)
            return True
        except:
            return False
        
    def parseBoolString(self, string):
        #return string[0].upper()=="T"
        
        if string is True or string is False:
            return string
        string = str(string).strip().lower()
        return not string in ['false','f','n','0','']

    def convertCelsiusToFahrenheit(self, temp, dp=0):
        value = ((float(temp) * 9.0) / 5.0) + 32
        return self.roundOff(value, dp)

    def convertKilometresToMiles(self, dist, dp=0):
        value = float(dist) * 0.621371192
        return self.roundOff(value, dp)

    def convertMilesToKilometres(self, dist, dp=0):
        value = float(dist) * 1.609344001
        return self.roundOff(value, dp)
        
    def convertKPHtoBeaufort(self, kph, dp=0):
        value = pow(float(kph) * 0.332270069, 2.0 / 3.0)
        return self.roundOff(value, dp)
    
    def convertMPHtoBeaufort(self, mph, dp=0):
        value = pow(float(self.convertMilesToKilometres(mph)) * 0.332270069, 2.0 / 3.0)
        return self.roundOff(value, dp)
            
    def convertKPHtoMS(self, kph, dp=0):
        value = float(kph) * 0.27777778
        return self.roundOff(value, dp)

    def convertMPHtoMS(self, mph, dp=0):
        value = float(self.convertMilesToKilometres(mph)) * 0.27777778
        return self.roundOff(value, dp)
                
    def convertMillibarsToInches(self,mb,dp=0):
        value = float(mb)/33.8582
        return self.roundOff(value, dp)
    
    def roundOff(self, value, dp=0):
        if self.isNumeric(value):
            if dp == 0:
                return str(int(round(float(value),dp))) # lose the dp
            else:
                return str(round(float(value),dp))
        else:
            return value

    def properCaseWord(self, word):
        skipList = ["a", "an", "the", "am"]

        if word not in skipList:
            return word.capitalize()
        else:
            return word
        
    def properCase(self, text):
        wordList = text.split(" ") 
        newWordList = map(self.properCaseWord, wordList)
        return " ".join(newWordList)   
        
    # work in progress...                
    def getApparentTemp(self, temp, windspeed, humidity):
        
        temp = float(temp)
        windspeed = float(windspeed)
        humidity = float(humidity)
        
        windchill = 13.12 + (0.6215*float(temp)) - (11.37 * math.pow(float(windspeed),0.16)) + (0.3965 * temp * math.pow(windspeed,0.16))
    
        if windchill < temp:    
            return int(windchill)
        else:
            return int(temp)
    
    def getWindLevel(self, speed):
        beaufort = int(self.convertMPHtoBeaufort(speed))
        if beaufort < 4:
            return 0
        elif beaufort < 7:
            return 1
        elif beaufort < 10:
            return 2
        else:
            return 3

    def getFormattedTimeFromSeconds(self,seconds,showseconds=False):
        time = int(seconds)
        hours, time = divmod(time, 60*60)
        minutes, seconds = divmod(time, 60)
        
        if showseconds == True:
            output = "%02d:%02d:%02d"%(hours, minutes, seconds)
        else:
            output = "%02d:%02d"%(hours, minutes)
            
        return output
    
    def getMoonPhase(self, ageOfMoon, percentIlluminated):
        
        if self.isNumeric(ageOfMoon):
            ageOfMoon = int(ageOfMoon)
        else:
            return "N/A"
        
        if self.isNumeric(percentIlluminated):
            percentIlluminated = int(percentIlluminated)
        else:
            return "N/A"
        
        fullMoonPastState = self.getFullMoonPastState()
        
        if percentIlluminated == 0:
            return "New Moon"
        elif percentIlluminated == 100:
            return "Full Moon"                    
        #elif (ageOfMoon < 17):
        elif fullMoonPastState == False:
            if percentIlluminated < 50:
                return "Waxing Crescent"
            elif percentIlluminated == 50:
                return "First Quarter"
            elif percentIlluminated > 50:
                return "Waxing Gibbous"                
        #elif (ageOfMoon >= 17):
        elif fullMoonPastState == True:        
            if percentIlluminated < 50:
                return "Waning Crescent"                        
            elif percentIlluminated == 50:
                return "Last Quarter"
            elif percentIlluminated > 50:
                return "Waning Gibbous"            
        else: # prevents crash if non-valid data is given as input (ex. ageOfMoon = 0 and percentIlluminated != 0)
            # does not fit model
            return "N/A"
    
    def getMoonImageURL(self, ageOfMoon, percentIlluminated):

        if self.isNumeric(ageOfMoon):
            ageOfMoon = int(ageOfMoon)
        else:
            ageOfMoon = 0 # default new moon
            
        if self.isNumeric(percentIlluminated):
            percentIlluminated = int(percentIlluminated)
        else:
            percentIlluminated = 0 # default new moon
                        
        fullMoonPastState = self.getFullMoonPastState()
                        
        if percentIlluminated == 0:
            moonImage = 24
        elif percentIlluminated == 100:
            moonImage = 13          
        #elif (ageOfMoon < 17):
        elif fullMoonPastState == False:
            if percentIlluminated < 50:
                # Image selection tested works
                if percentIlluminated < 9: # 1-8
                    moonImage = 1
                elif percentIlluminated < 17: # 9-16
                    moonImage = 2
                elif percentIlluminated < 25: # 17-24
                    moonImage = 3
                elif percentIlluminated < 33: # 25-32
                    moonImage = 4
                elif percentIlluminated < 41: # 33-40
                    moonImage = 5
                else: # 41-49
                    moonImage = 6
                #endif                
            elif percentIlluminated == 50:
                moonImage = 7
            elif percentIlluminated > 50:
                # Image selection tested works
                if percentIlluminated < 61: # 51-60
                    moonImage = 8
                elif percentIlluminated < 71: # 61-70
                    moonImage = 9
                elif percentIlluminated < 81: # 71-80
                    moonImage = 10
                elif percentIlluminated < 91: # 81-90
                    moonImage = 11
                else: # 91-99
                    moonImage = 12
                #endif
        #elif (ageOfMoon >= 17):
        elif fullMoonPastState == True:        
            if percentIlluminated < 50:
                # Image selection tested works
                if percentIlluminated > 36: # 49-37
                    moonImage = 20
                elif percentIlluminated > 24: # 36-25
                    moonImage = 21
                elif percentIlluminated > 12: # 24-13
                    moonImage = 22
                elif percentIlluminated > 0: # 12-1
                    moonImage = 23
                #endif                   
            elif percentIlluminated == 50:
                moonImage = 19
            elif percentIlluminated > 50:
                # Image selection tested works
                if percentIlluminated > 90: # 99-91
                    moonImage = 14
                elif percentIlluminated > 80: # 90-81
                    moonImage = 15
                elif percentIlluminated > 70: # 80-71
                    moonImage = 16
                elif percentIlluminated > 60: # 70-61
                    moonImage = 17
                elif percentIlluminated > 50: # 60-51
                    moonImage = 18
                #endif
        else: # prevents crash if non-vaild data is given as input (ex. ageOfMoon = 0 and percentIlluminated != 0)
            # does not fit model
            moonImage = 24
                                
        moonImagePath = "%s/images/moonicons/%s.png"%(app_path, str(moonImage).rjust(2,"0"))
        
        return moonImagePath

    def setFullMoonPastState(self, ageOfMoon, percentIlluminated):

        if self.isNumeric(ageOfMoon):
            ageOfMoon = int(ageOfMoon)
        else:
            ageOfMoon = 0 # default new moon
            
        if self.isNumeric(percentIlluminated):
            percentIlluminated = int(percentIlluminated)
        else:
            percentIlluminated = 0 # default new moon
                  
        # handle setting of full moon past state variable based on illumination percentage and age where possible
        if percentIlluminated == 0 or ageOfMoon < 15:
            fullMoonPastState = False
        elif percentIlluminated == 100 or ageOfMoon > 17:
            fullMoonPastState = True
        else:
            # get the past state as not determined above
            fullMoonPastState = self.getFullMoonPastState()

        FULLMOON_PAST_STATE_CACHE_FILEPATH = os.path.expanduser(os.path.join(self.config.PERSISTED_CACHE_FOLDERPATH, self.FULLMOON_PAST_STATE_CACHE_FILENAME.replace("<LOCATION>", self.options.location.replace("/","-"))))
        
        self.logInfo("Storing full moon past state in %s"%FULLMOON_PAST_STATE_CACHE_FILEPATH)
                
        fullMoonPastStateFile = open(FULLMOON_PAST_STATE_CACHE_FILEPATH, 'wb')
        pickle.dump(fullMoonPastState, fullMoonPastStateFile)
        fullMoonPastStateFile.close()
                    
    def getFullMoonPastState(self):
        
        fullMoonPastState = False

        FULLMOON_PAST_STATE_CACHE_FILEPATH = os.path.expanduser(os.path.join(self.config.PERSISTED_CACHE_FOLDERPATH, self.FULLMOON_PAST_STATE_CACHE_FILENAME.replace("<LOCATION>", self.options.location.replace("/","-"))))
        
        if os.path.exists(FULLMOON_PAST_STATE_CACHE_FILEPATH):
            self.logInfo("Fetching full moon past state from %s"%FULLMOON_PAST_STATE_CACHE_FILEPATH)
            fullMoonPastStateFile = open(FULLMOON_PAST_STATE_CACHE_FILEPATH, 'rb')
            fullMoonPastState = pickle.load(fullMoonPastStateFile)
            fullMoonPastStateFile.close()
        
        return fullMoonPastState
    
    def getImageSrcForCondition(self, url, condition):
        imagesrc = ""
        imgfilepath = os.path.join(self.config.CACHE_FOLDERPATH, self.CONDITION_IMAGE_FILENAME.replace("<CONDITION>",condition.replace(" ","-")))       

        if os.path.exists(imgfilepath) == False:

            try:
                self.logInfo("Fetching image from " + imagesrc)
    
                usock = urllib2.urlopen(url)
                img = usock.read()
            except Exception, e:
                self.logError("Error downloading the image file: " + e.__str__()+"\n"+traceback.format_exc())
            else:
                # save the image locally
                
                imgfile = open(imgfilepath,'wb')
                imgfile.write(img)
                self.logInfo("Saved image to " + imgfilepath)
     
            finally:
                usock.close()
                imgfile.close()

        return imgfilepath
    
    def getImagePathForBearing(self, bearingcode):
        #TODO: Once gif supported properly in conky re-enable gif output
        #if int(bearingcode) > 0 and int(bearingcode) <= 4:
        #    fileext = "gif" # use animated gif for VAR output
        #else:
        #    fileext = "png"
            
        fileext = "png" #force to always be png until animated gifs are supported
        imagesrc = "%s/images/bearingicons/%s.%s"%(app_path, str(bearingcode).rjust(2,"0"),fileext)
        return imagesrc
    
def main():

    parser = CommandLineParser()
    (options, args) = parser.parse_args()

    if options.version == True:
        print >> sys.stdout,"%s v%s"%(app_name, app_version)
    else:
        if options.verbose == True:
            print >> sys.stdout, "*** INITIAL OPTIONS:"
            print >> sys.stdout, "    config:", options.config
            print >> sys.stdout, "    location:", options.location
            print >> sys.stdout, "    datatype:", options.datatype
            print >> sys.stdout, "    start day:", options.startday
            print >> sys.stdout, "    end day:", options.endday
            print >> sys.stdout, "    spaces:", options.spaces
            print >> sys.stdout, "    template:", options.template
            print >> sys.stdout, "    locale:", options.locale
            print >> sys.stdout, "    imperial:", options.imperial
            print >> sys.stdout, "    beaufort:", options.beaufort
            print >> sys.stdout, "    metrespersecond:", options.metrespersecond
            print >> sys.stdout, "    night:", options.night
            print >> sys.stdout, "    shortweekday:", options.shortweekday
            print >> sys.stdout, "    hideunits:", options.hideunits
            print >> sys.stdout, "    hidedegreesymbol:", options.hidedegreesymbol
            print >> sys.stdout, "    minuteshide:", options.minuteshide
            print >> sys.stdout, "    centeredwidth:", options.centeredwidth
            print >> sys.stdout, "    refetch:", options.refetch
            print >> sys.stdout, "    verbose:", options.verbose
            print >> sys.stdout, "    debug:", options.debug            
            print >> sys.stdout, "    errorlogfile:",options.errorlogfile
            print >> sys.stdout, "    infologfile:",options.infologfile        
    
        forecastinfo = ForecastInfo(options)
        forecastinfo.writeOutput()

if __name__ == '__main__':
    main()
    sys.exit()
    
