
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rodent.h"
#include "rfm_modules.h"
#include "rodent_expose.i"

void
rodent_expose_all (view_t * view_p) {
    GtkAllocation *allocation_p = rfm_get_allocation();
    TRACE("rodent_expose_all: %d,%d %d,%d\n",
	    allocation_p->x,
	    allocation_p->y,
	    allocation_p->width,
	    allocation_p->height
	    );
    rfm_expose_rect(view_p, allocation_p);
}

static
population_t *find_grid_element(view_t * view_p, gdouble x, gdouble y) {
    if(!view_p->population_pp)  return NULL;

    // Margin considerations:
    x -= MARGIN_LEFT(view_p);
    y -= MARGIN_TOP(view_p); 
    if(x < 0 || y < 0)  return NULL;

    // rectangle x,y coordinates
    gint row = y / CELLHEIGHT(view_p);
    gint column = x / CELLWIDTH(view_p);
    gint grid_rows = GRID_ROWS(view_p);
    gint grid_columns = GRID_COLUMNS(view_p);
    if(column >= grid_columns || row >= grid_rows) return NULL;

    gint items = 0;
    while (view_p->population_pp[items]) items++;


    gint element;
    if (view_p->flags.type == ICONVIEW_TYPE) {
	element= row * grid_columns + column;
    } else { // transposed
	element= column * grid_rows + row;
    }
    if(element >= items) return NULL;
    if(element >= MAX_ELEMENTS(view_p)) return NULL;

    population_t *population_p = *(view_p->population_pp + element);

    if (population_p && 
		(population_p->row != row || population_p->column != column)) {
	// This is a sanity hack test. Should never happen in a perfect world.
	NOOP ("THIS IS WRONG: row, column: current (%d,%d) correct (%d,%d) element=%d columns=%d\n",
	    population_p->row, population_p->column,
	    row, column, element,
	    grid_columns);
	population_p->row = row;
	population_p->column = column;
    }

    return population_p;
}


const 
population_t *
rodent_find_in_population (view_t * view_p, gdouble x, gdouble y) {


    population_t *population_p = find_grid_element(view_p, x, y);
    if(!population_p) return NULL;

#if 0
    // this should work, but doesn't...
    GdkRectangle icon_rect;
    if (!rfm_get_population_rect(view_p, population_p, &icon_rect)) {
	DBG("rodent_find_in_population: !rfm_get_population_rect\n");
	return FALSE;
    }
    if (x >= icon_rect.x && x < icon_rect.x + icon_rect.width &&
	y >= icon_rect.y && y < icon_rect.y + icon_rect.height) {
	return population_p;
    }
    return NULL;
#else
    gint icon_size = ICON_SIZE(view_p);
    gint cellwidth = CELLWIDTH(view_p);
    gint icon_width = icon_size;
    gint icon_height = icon_size;
    gint x_spacing = (icon_size >= SMALL_ICON_SIZE)?(cellwidth - icon_width) / 2: 0;
    if(x_spacing < 0) x_spacing = 0;
    gint X = (gint)(x-MARGIN_LEFT(view_p)) % cellwidth;

    if(X < x_spacing || X > x_spacing + icon_width) return NULL;


    gint Y = (gint)(y-MARGIN_TOP(view_p)) % CELLHEIGHT(view_p);
    gint y_spacing = (icon_size - icon_height)/2;

    if(Y < y_spacing || Y > y_spacing + icon_height) return NULL;
    return population_p;
#endif
}

const
population_t *
rodent_find_in_labels (view_t * view_p, gdouble x, gdouble y) {

	
    // Find grid element.
    population_t *population_p = find_grid_element(view_p, x, y);
    if(!population_p) return NULL;
;

    GdkRectangle label_rect;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect))return NULL;
    if(x >= label_rect.x && x < label_rect.x + label_rect.width 
	    &&
       y >= label_rect.y && y < label_rect.y + label_rect.height)
    {
	return population_p;
    }
    return NULL;
}

    

// Find elements (or icons, specified by the respective gboolean)
// within an expose rectangle. The icons restriction is to do rubberband
// selecting considering only the icons to select.
static GSList *
find_items_in_rectangle(view_t * view_p, GdkRectangle *rect, gboolean icons) {

    if(!rect->width || !rect->height) {
	NOOP("find_items_in_rectangle(): !rect->width || !rect->height\n");
	return NULL;
    }
    if (!rfm_population_try_read_lock(view_p, "find_items_in_rectangle")) {
	NOOP("find_items_in_rectangle(): unable to get readlock\n");
	return NULL;
    }
    if(!view_p->population_pp){
	rfm_population_read_unlock (view_p, "find_items_in_rectangle");
        return NULL;
    }

    GSList *list=NULL;

    gint cellwidth = CELLWIDTH(view_p);
    gint cellheight = CELLHEIGHT(view_p);
    NOOP(stderr, "x: %d-%d, y:%d-%d w:%d H: %d\n",
	    rect->x, rect->x+rect->width,
	    rect->y, rect->y+rect->height,
	    cellwidth, cellheight);

    gint low_X = rect->x;
    gint low_Y = rect->y;
	// Margin considerations:
	low_X -= MARGIN_LEFT(view_p);
	low_Y -= MARGIN_TOP(view_p);
    gint high_X = low_X + rect->width - 1;
    gint high_Y = low_Y + rect->height - 1;

    if(low_X < 0){ low_X = 0; }
    if(low_Y < 0){ low_Y = 0; }

    gint low_column  = low_X / cellwidth;
    gint low_row     = low_Y / cellheight;
    gint high_column = high_X / cellwidth;
    gint high_row    = high_Y / cellheight;

   // if ( low_X % cellwidth) low_column--;
   // if ( low_Y % cellheight) low_row--;

   
    gint grid_rows = GRID_ROWS(view_p);
    gint grid_columns = GRID_COLUMNS(view_p);
    if (low_column < 0) low_column = 0;
    if (low_row < 0) low_row = 0;

    if (high_column > grid_columns - 1) {
	NOOP("find_items_in_rectangle(): logic error, high_column (%d) > grid_columns - 1 (%d)\n",

		high_column, grid_columns - 1);
	high_column = grid_columns - 1;
    }
    if (high_row > grid_rows - 1) {
	NOOP("find_items_in_rectangle(): logic error, high_row (%d) > grid_rows - 1 (%d)\n",
		high_row, grid_rows - 1);
	high_row = grid_rows - 1;
    }

    NOOP("find_items_in_rectangle(): rows = %d-%d, columns = %d-%d cellwidth=%d  cellheight=%d\n", 
	    low_row, high_row, low_column, high_column,
	    cellwidth, cellheight);

    // Just check if the element corresponding to the given
    // row and column is within the selected rows and columns.
    gint column;
    gint row;
    gint max_elements = MAX_ELEMENTS(view_p);

    for (column=low_column; column<=high_column; column++){
	for (row=low_row; row<=high_row; row++) {
	    gint element;
	    if (view_p->flags.type == ICONVIEW_TYPE) {
		element = row * grid_columns + column;
	    } else { // transposed
		element = column * grid_rows + row;
	    }
	    if (element >= max_elements) {
		// This may occur when bottom row is not full.
		continue;
	    }
	    gint items = 0;
	    while (view_p->population_pp[items]) items++;
	    if (element >= items) continue;
	    population_t *population_p = view_p->population_pp[element];
	    if (!population_p) continue;
	    NOOP("element: %d/%d (%s)\n",
		    element, view_p->max_elements-1,
		    (population_p->en)?population_p->en->path:NULL);
	    // Ok, we item is within the rows and columns, but are we only interested
	    // in the icons? 
	    gboolean found = FALSE;
	    if (!icons){
		
		found = TRUE;
	    } else 
	    {
		// Icon test.
		GdkRectangle icon_rect;
		if (!rfm_get_population_icon_rect(view_p, population_p, &icon_rect)) continue;
		gint low_x = icon_rect.x;
		gint high_x = icon_rect.x + icon_rect.width;
		gint low_y = icon_rect.y;
		gint high_y = icon_rect.y + icon_rect.height;
		gboolean X_in=FALSE;
		gboolean Y_in=FALSE;
		X_in = (low_x >= low_X+MARGIN_LEFT(view_p) && 
			low_x < high_X+MARGIN_LEFT(view_p)) ||
		       (high_x >= low_X+MARGIN_LEFT(view_p) &&
			high_x < high_X+MARGIN_LEFT(view_p));
		Y_in = (low_y >= low_Y+MARGIN_TOP(view_p) && 
			low_y < high_Y)+MARGIN_TOP(view_p) ||
		       (high_y >= low_Y+MARGIN_TOP(view_p) && 
			high_y < high_Y+MARGIN_TOP(view_p));
		if(X_in && Y_in){
		    found = TRUE;
 		}
	    }
	    if (found) {
		NOOP(stderr, "find_items_in_rectangle(): found item %d/%d\n", element, max_elements);
		if (population_p->flags & LABEL_SATURATED){
		    list=g_slist_append(list, population_p);
		    //list=g_slist_prepend(list, population_p);
		} else list=g_slist_prepend(list, population_p);	
		//} else list=g_slist_append(list, population_p);	
	    } else {
		// clear column and row.
	    }
	}
    }
    rfm_population_read_unlock(view_p, "find_items_in_rectangle");
    return list;
}

GSList *rodent_find_icons_in_rectangle(view_t * view_p, GdkRectangle *rect) {
    return find_items_in_rectangle(view_p, rect, TRUE);
}
GSList *rodent_find_items_in_rectangle(view_t * view_p, GdkRectangle *rect) {
    return find_items_in_rectangle(view_p, rect, FALSE);
}


void * 
rodent_clean_paper(gpointer data){
    widgets_t *widgets_p = data;
    cairo_t *gdk_context =  
	gdk_cairo_create(gtk_widget_get_window(widgets_p->paper));
    view_t *view_p = widgets_p->view_p;
    GdkRectangle allocation;
    gtk_widget_get_allocation(widgets_p->paper, &allocation);


    
    rodent_clean_image (view_p, &allocation, 0, 0, allocation.width, allocation.height, gdk_context);
	cairo_destroy(gdk_context);
    
    return NULL;
}

void 
rodent_threaded_clean_paper(widgets_t *widgets_p)
{
    if (rfm_get_gtk_thread() ==g_thread_self()){
	rodent_clean_paper (widgets_p);
	return;
    }
    rfm_context_function(rodent_clean_paper, widgets_p);
    return;
}
    
    
static void *
queue_an_expose(void *data){
    rfm_threadwait();
    void **arg = data;
    view_t *view_p = arg[0];
    void *area_p = arg[1];
    g_free(arg);
    g_thread_yield();
	
    rfm_expose_rect (view_p, (GdkRectangle *)area_p);
    
    g_free(area_p);
    return NULL;
}


// Function to set/unset hold on signal generated expose callback
void rodent_set_expose_hold(view_t *view_p, gboolean expose_hold){
    view_p->expose_hold = expose_hold;
    if (expose_hold){
	// might be called by a thread or main thread.
	rodent_threaded_clean_paper(&(view_p->widgets));	
    }
}

// XXX: 2 bugs identified. FIXME
// 1) when you expose label after unsaturation, the area overlaps item below, which is not redrawn because test is whether icon, not cell, is in expose rectangle...fix, either do not expose excess area (preferred) or regen icon if top part of cell is in expose area (not too good solution).
//
// 2) bug in deskview, columns is giving max columns, not actual columns in usage.
//
static gboolean 
conditional_clean_rectangle(view_t *view_p, GdkRectangle *area){
#if 10
    gint grid_rows;
    gint grid_columns;
    // erase incomplete columns/columns
    if (view_p->flags.type == DESKVIEW_TYPE)
    {
	gint elements = rfm_layout_get_max_elements(view_p);
	grid_rows = rfm_layout_get_grid_rows(view_p);
	grid_columns = elements / grid_rows;

	gint cellwidth = rfm_layout_get_cellwidth(view_p);
	gint cellheight = rfm_layout_get_cellheight(view_p);
	gint margin_top = rfm_layout_get_margin_top(view_p);
	gint margin_left = rfm_layout_get_margin_left(view_p);

	gboolean in_grid;
	    in_grid =
		area->x >= margin_left &&
		area->y >= margin_top &&
		area->x + area->width <= grid_columns*cellwidth+margin_left &&
		area->y + area->height <= grid_rows*cellheight+margin_top ;

	if (in_grid) {
	    NOOP(stderr, "in_grid: (%d) max columns=%d max rows=%d (%d,%d - %d,%d) %d>=%d, %d>=%d  %d<=%d, %d<=%d\n",
		    elements, grid_columns, grid_rows,
		    area->x,area->y,area->x+area->width,area->y+area->height,
		    area->x, margin_left,
		    area->y, margin_top,
		    area->x+area->width, grid_columns*cellwidth+margin_left,
		    area->y+area->height,grid_rows*cellheight+margin_top
		    );

	    return FALSE;
	}
    }




#endif
    
    cairo_t *gdk_context =  
	    gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
    // clean  expose area.
    // (do this only if the area to expose is beyond cell scope)
    rodent_set_draw_clip(view_p, gdk_context); // gtk-3.12 fix

    rodent_clean_rectangle (view_p, 
		 area->x,  area->y, 
		 area->width,
		 area->height,
		 gdk_context);
    cairo_destroy(gdk_context);
    return TRUE;
}

static void
expose_draw(view_t *view_p, GdkRectangle *area){
    if(!view_p || !view_p->widgets.paper)  return;
    rfm_global_t *rfm_global_p = rfm_global();
    gint status = rfm_global_p->status;
    if (status == STATUS_EXIT) return;
    if(!view_p || !view_p->widgets.paper)  return;
    status = view_p->flags.status;
    if (status == STATUS_EXIT) return;
    //static int kk=1;
    NOOP("%d) on_expose area= %d, %d width=%d, height=%d\n", kk++,
	    area->x, area->y, area->width, area->height); 
    gboolean layout_ready = rfm_layout_is_setup(view_p);


#if 10
    // 1. Clean expose area.
    // 2. If expose action not currently possible, put the expose in a queue (threaded).

    if (view_p->expose_hold || !layout_ready || !rfm_population_try_read_lock(view_p, "expose_draw")) {
	if (view_p->flags.type != DESKVIEW_TYPE){
	    // Since the expose could not proceed, keep generating
	    // expose signal until it works. Expose signal is generated
	    // in its own thread to avoid any deadlock potential.

	    // Clean the area anyways, for that we don't need a readlock.
	    // (otherwise we'll get a grayout of the area)
	    cairo_t *gdk_context =  
		gdk_cairo_create(gtk_widget_get_window(view_p->widgets.paper));
            rodent_set_draw_clip(view_p, gdk_context); // gtk-3.12 fix
            
	    rodent_clean_image (view_p, area,
		    area->x, area->y, 
		    area->width, area->height,
		    gdk_context);
		cairo_destroy(gdk_context);
	}

	NOOP(stderr, "rodent_expose(): dumping an expose signal now (write lock is on)\n");
	void **arg = (void **)malloc(2*sizeof(void *));
	if (!arg) g_error("malloc: %s\n", strerror(errno));
	GdkRectangle *area_p = (GdkRectangle *)malloc(sizeof(GdkRectangle));
	if (!area_p) g_error("malloc: %s\n", strerror(errno));
	memcpy(area_p, area, sizeof(GdkRectangle));
	arg[0] = view_p;
	arg[1] = area_p;
	rfm_view_thread_create(view_p, queue_an_expose, arg, "queue_an_expose");
	return;
    }
#endif

    // 3. Expose action proceeds from here.
    //
    // check for background pixbuf (deskview)
    // this check will create background image when changed
    background_test(view_p);

  
    // Find items which need to be redrawn.
    // take into consideration margin displacement

    // First we do the icons and later the labels.
    
    // Clean  expose area.
    // (do this only if the area to expose is beyond cell scope)
    conditional_clean_rectangle(view_p, area);
    
#if 0
    // This does not work right with gtk2 when pane is resized...
    GSList *render_list=rodent_find_icons_in_rectangle(view_p, area);
    render_pixbuf_list(view_p, render_list);
    g_slist_free(render_list);
#endif
    GSList *render_list=rodent_find_items_in_rectangle(view_p, area);
    render_item_list(view_p, render_list);
    //render_label_list(view_p, render_list);
    g_slist_free(render_list);
    // Remove readlock now.
    rfm_population_read_unlock (view_p, "expose_draw");
}

#if GTK_MAJOR_VERSION==2
void
rodent_expose (GtkWidget * widget, GdkEventExpose * event, gpointer data) {
    NOOP (stderr, "GTK2, rodent_expose(): x,y=%d,%d w,h=%d,%d\n", event->area.x, event->area.y, event->area.width, event->area.height);
    expose_draw((view_t *) data, &(event->area));
}

#else

gboolean
rodent_draw (GtkWidget * widget, cairo_t *cr, gpointer data) {
     NOOP( "rodent_draw...\n");
    
    double x1, y1, x2, y2;
    cairo_clip_extents (cr, &x1, &y1, &x2, &y2);
    GdkRectangle area;
    area.x = x1;
    area.y = y1;
    area.width = x2 - x1;
    area.height = y2 - y1;

    NOOP ("rodent_draw(): x,y=%lf,%lf x2,y2=%lf,%lf\n", 
	    x1, y1, x2, y2);
    NOOP ( "rodent_draw(): area x,y=%d,%d w,h=%d,%d\n", area.x, area.y, area.width, area.height);
    expose_draw((view_t *) data, &(area));
    return TRUE; 
}

#endif


void
rodent_recalc_population_geometry (view_t * view_p) {
   if (!view_p || view_p->population_pp==NULL || *view_p->population_pp==NULL || MAX_ELEMENTS(view_p) == 0){
       NOOP("rodent_recalc_population_geometry: skipping ...\n");
       return;
   }
    gint grid_rows = GRID_ROWS(view_p);
    gint grid_columns = GRID_COLUMNS(view_p);
   NOOP("rodent_recalc_population_geometry(): grid_columns=%d grid_rows=%d max_elements=%d\n", 
	    grid_columns, grid_rows, MAX_ELEMENTS(view_p));

   gint row;
   gint column;
   gint max_elements = MAX_ELEMENTS(view_p);
   for(column = 0; column < grid_columns; column++) {
       for(row = 0; row < grid_rows; row++) {
	   gint element;
	   if (view_p->flags.type == ICONVIEW_TYPE){
	       element = row * grid_columns + column;
	   } else { //transposed...
		element = column * grid_rows + row;
	   }
	   if (element >= max_elements) break;
	   if (view_p->population_pp[element] == NULL) break;
	   // if element is repeated as first column of next row, skip it 
	   if(element == (row + 1) * grid_columns){
	       NOOP("skipping %s\n",view_p->population_pp[element]->en->path);
	       continue;
	   }                
	   view_p->population_pp[element]->column = column;
	   view_p->population_pp[element]->row = row;

       }
   }
   return;
}

static void *
redraw_item_f(void *data){
    void **arg=data;
    view_t *view_p = arg[0];
    population_t *population_p = arg[1];
    gint flags = GPOINTER_TO_INT(arg[2]);
    
    if (!rfm_population_try_read_lock(view_p, "redraw_item_f")) return NULL;
    widgets_t *widgets_p = &(view_p->widgets);
    cairo_t *gdk_context = gdk_cairo_create(gtk_widget_get_window(widgets_p->paper));
    if (flags & 0x01) insert_pixbuf (view_p, population_p, gdk_context);
    if (flags & 0x02) insert_layout (view_p, population_p, gdk_context);
	
    rfm_population_read_unlock(view_p, "redraw_item_f");
    cairo_destroy(gdk_context);
    gdk_flush();
    return NULL;
}

gboolean
rodent_valid_population_p(view_t *view_p, const population_t *population_p){
    gboolean found = FALSE;
    population_t **pp=view_p->population_pp;
    for (; pp && *pp; pp++){
	if (*pp == population_p) found = TRUE;
    }
    return found;
}

void  
rodent_redraw_item(view_t *view_p, population_t *population_p){
    if (!rodent_valid_population_p(view_p, population_p)) return;
    void *arg[]={view_p, population_p, GINT_TO_POINTER(0x01|0x02)};
    rfm_context_function(redraw_item_f, arg);
    return ;
}

void  
rodent_redraw_icon(view_t *view_p, population_t *population_p){
    void *arg[]={view_p, population_p, GINT_TO_POINTER(0x01)};
    rfm_context_function(redraw_item_f, arg);
    return ;
}

void  
rodent_redraw_label(view_t *view_p, population_t *population_p){
    void *arg[]={view_p, population_p, GINT_TO_POINTER(0x02)};
    rfm_context_function(redraw_item_f, arg);
    return ;
}


static void *
redraw_items_f(void *data){
    view_t *view_p = data;
    //widgets_t *widgets_p = &(view_p->widgets);
    gint grid_columns = GRID_COLUMNS(view_p);
    gint grid_rows = GRID_ROWS(view_p);
    gint low_column = 0;
    gint low_row = 0;
    gint cellwidth = rfm_layout_get_cellwidth(view_p);
    gint cellheight = rfm_layout_get_cellheight(view_p);
    gint margin_top = rfm_layout_get_margin_top(view_p);
    gint margin_left = rfm_layout_get_margin_left(view_p);
    gint active_count = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.paper), "active_count"));
    if (view_p->flags.type == ICONVIEW_TYPE) {
	GtkScrolledWindow *scrolled_window = g_object_get_data(G_OBJECT(view_p->widgets.paper), "scrolled_window");
	double pos = gtk_adjustment_get_value (
		gtk_scrolled_window_get_vadjustment(scrolled_window));
	double r = floor(pos / cellheight);
	low_row = r;
    }
    GtkAllocation *allocation_p = rfm_get_allocation();
    gdouble c = allocation_p->height;
    c = ceil(c / cellheight);
    gint high_row = low_row + c;  

    gint column;
    gint max_elements = MAX_ELEMENTS(view_p);
    population_t **population_pp = view_p->population_pp;
    NOOP (stderr, "max=%d drawing columns %d,%d and rows %d, %d\n",
	    max_elements, low_column, grid_columns-1, low_row, high_row-1);
    gint row;
    for (row=low_row; row < high_row; row++) {
	for (column=low_column; column < grid_columns; column++){
	    gint element;
	    if (view_p->flags.type == ICONVIEW_TYPE) {
		element = row * grid_columns + column;
	    } else { // transposed
		element = column * grid_rows + row;
	    }
	    if (element > max_elements-1 && element <= active_count-1) {
  		GdkRectangle rect;
		rect.x = column * cellwidth + margin_left;
		rect.y = row * cellheight + margin_top;
		rect.width = cellwidth;
		rect.height = cellheight;
		NOOP (stderr, "Ghost erase:(%d) x=%d, y=%d, w=%d,h=%d\n",
		    element, rect.x, rect.y, rect.width, rect.height);
		rfm_expose_rect(view_p, &rect); // this will erase ghosts
		// This may occur when bottom row is not full.
		continue;
	    }
	    if (element >= max_elements) continue; // when active count is 0
	    
	    rodent_redraw_item(view_p, population_pp[element]);
	    NOOP(stderr, "draw item %d\n", element);
	}
    }
    g_object_set_data(G_OBJECT(view_p->widgets.paper), "active_count", GINT_TO_POINTER(max_elements));
    return NULL;
}

void 
rodent_redraw_items(view_t *view_p){
    if (!rfm_population_read_lock (view_p, "rodent_redraw_items")) return;
//    while (!rfm_population_try_read_lock (view_p)) rfm_threadwait();
    rfm_context_function(redraw_items_f, view_p);
    rfm_population_read_unlock (view_p, "rodent_redraw_items");
}

