/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */

//////////////////////////////////////////////////////////////////////7

#define RFM_PRIMARY_LAYOUT_C

#ifdef HAVE_CONFIG_H
# include <config.h>
#endif

#include "rfm.h"
#include "rfm_modules.h"

static GtkAllocation rfm_allocation;
GtkAllocation *rfm_get_allocation(void){return &rfm_allocation;}
void rfm_set_allocation(GtkAllocation *allocation){
    memcpy(&rfm_allocation, allocation, sizeof(GtkAllocation));
    return;
}

static const gchar *font_sizes_v[] = {
    "6",
    "8",
    "9",
    "10",
    "11",
    "12",
    "14",
    "18",
    "20",
    "22",
    "24",
    NULL
};

static const gchar *icon_sizes_v[] = {
    "Normal",
    "Details",
    "Compact",
    "Big",
    "Huge",
    NULL
};

#include "primary-layout.i"

const gchar **rfm_get_icon_sizes(void){return icon_sizes_v;}
const gchar **rfm_get_font_sizes(void){return font_sizes_v;}

#define LAYOUT_ICON_SIZE (0x01)
#define LAYOUT_VPANE_ALLOCATION (0x01<<1)
#define LAYOUT_CELLWIDTH (0x01<<2)
#define LAYOUT_CELLHEIGHT (0x01<<3)
#define LAYOUT_ROOT_PARAMETERS (0x01<<4)
#define LAYOUT_ROOT_CONFIGURED (0x01<<5)

void
rfm_layout_population_grid_row (view_t * view_p, population_t *population_p, gint element){
    gint grid_rows = GRID_ROWS(view_p);
    gint grid_columns = GRID_COLUMNS(view_p);
    if (view_p->flags.type == ICONVIEW_TYPE) {
	if(grid_columns) {
	    population_p->row = element / grid_columns;
	    population_p->column = element % grid_columns;
	} else {
	    population_p->row = 0;
	    population_p->column = element;
	}
    } else { // transposed matrix for desktop
	if(grid_rows) {
	    population_p->row = element % grid_rows;
	    population_p->column = element / grid_rows;
	} else {
	    population_p->row = element;
	    population_p->column = 0;
	}
    }
}

gint rfm_layout_get_name_width(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->name_width;
}

gint rfm_layout_get_text_height(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->text_height;
}

gint rfm_layout_set_root_width(view_t *view_p, gint value){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->root_w = value;
}

gint rfm_layout_set_root_height(view_t *view_p, gint value){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->root_h = value;
}

gint rfm_layout_get_root_width(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->root_w;
}

gint rfm_layout_get_root_height(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->root_h;
}

gint rfm_layout_get_root_depth(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->root_d;
}

gint rfm_layout_get_max_elements(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->max_elements;
}

gint rfm_layout_set_margin_top(view_t *view_p, gint value){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_top = value;
}

gint rfm_layout_set_margin_bottom(view_t *view_p, gint value){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_bottom = value;
}

gint rfm_layout_set_margin_right(view_t *view_p, gint value){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_right = value;
}

gint rfm_layout_set_margin_left(view_t *view_p, gint value){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_left = value;
}


gint rfm_layout_get_margin_top(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_top;
}

gint rfm_layout_get_margin_bottom(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_bottom;
}

gint rfm_layout_get_margin_right(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_right;
}

gint rfm_layout_get_margin_left(view_t *view_p){
    if (!view_p) return FALSE;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    return view_layout_p->margin_left;
}


gboolean rfm_layout_is_setup(view_t *view_p){
    if (!view_p) {
	DBG("rfm_layout_is_setup(): !view_p\n");
	return FALSE;
    }
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ICON_SIZE)) {
	DBG("rfm_layout_is_setup(): LAYOUT_ICON_SIZE\n");
	return FALSE;
    }
    GtkWidget *vpane = g_object_get_data(G_OBJECT(view_p->widgets.paper), "vpane");
    if (vpane && !(view_layout_p->tripwire & LAYOUT_VPANE_ALLOCATION)) {
	DBG("rfm_layout_is_setup(): LAYOUT_VPANE_ALLOCATION\n");
	return FALSE;
    }
    if (!(view_layout_p->tripwire & LAYOUT_CELLWIDTH)) {
	TRACE("rfm_layout_is_setup(): LAYOUT_CELLWIDTH\n");
	return FALSE;
    }
    if (!(view_layout_p->tripwire & LAYOUT_CELLHEIGHT)) {
	DBG("rfm_layout_is_setup(): LAYOUT_CELLHEIGHT\n");
	return FALSE;
    }
    if (!(view_layout_p->tripwire & LAYOUT_ROOT_PARAMETERS)) {
	DBG("rfm_layout_is_setup(): LAYOUT_ROOT_PARAMETERS\n");
	return FALSE;
    }
    if (!(view_layout_p->tripwire & LAYOUT_ROOT_CONFIGURED)) {
	DBG("rfm_layout_is_setup(): LAYOUT_ROOT_CONFIGURED\n");
	return FALSE;
    }
    return TRUE;
}


gint rfm_layout_get_grid_rows(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_grid_rows: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ROOT_CONFIGURED))
	g_error("rfm_layout_get_grid_rows: view_layout_p item LAYOUT_ROOT_CONFIGURED is not set\n");
    return view_layout_p->grid_rows;
}

gint rfm_layout_get_grid_columns(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_grid_colums: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ROOT_CONFIGURED))
	g_error("rfm_layout_get_grid_colums: view_layout_p item LAYOUT_ROOT_CONFIGURED is not set\n");
    return view_layout_p->grid_columns;
}

static void *layout_set_root_parameters(void *data){
    view_t *view_p = data;
    gint width;
    gint height;
    gint depth;
    rfm_global_t *rfm_global_p = rfm_global();
    rfm_get_drawable_geometry (rfm_global_p->root_Xwindow, NULL, NULL, 
	    &width, &height, &depth);
    view_layout_t *view_layout_p = &(view_p->view_layout);
    view_layout_p->root_w = width;
    view_layout_p->root_h = height;
    view_layout_p->root_d = depth;
    return NULL;
}

void rfm_layout_set_root_parameters(view_t *view_p){
    if (!view_p) g_error("rfm_layout_set_root_parameters: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    //rfm_global_t *rfm_global_p = rfm_global();
    if (rfm_get_gtk_thread() != g_thread_self()){
	rfm_context_function(layout_set_root_parameters, view_p);
    } else {
	layout_set_root_parameters(view_p);
    }
    view_layout_p->tripwire |= LAYOUT_ROOT_PARAMETERS;
    return;
}

void rfm_layout_get_root_parameters(view_t *view_p, gint *width, gint *height, gint *depth){
    if (!view_p) g_error("rfm_layout_get_root_parameters: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ROOT_PARAMETERS))
	g_error("rfm_layout_get_root_parameters: view_layout_p item LAYOUT_ROOT_PARAMETERS is not set\n");
    if (width) *width = view_layout_p->root_w;
    if (height) *height = view_layout_p->root_h;
    if (depth) *depth = view_layout_p->root_d;
    return;
}

gint rfm_layout_get_icon_size(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_cellheight: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ICON_SIZE))
	g_error("rfm_layout_get_icon_size: view_layout_p item LAYOUT_ICON_SIZE is not set\n");
    if (view_layout_p->icon_size) return view_layout_p->icon_size;
    return TINY_ICON_SIZE;
}

gint rfm_layout_get_icon_size_id(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_cellheight: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ICON_SIZE))
	g_error("rfm_layout_get_icon_size: view_layout_p item LAYOUT_ICON_SIZE is not set\n");
    return view_layout_p->icon_size;
}

gboolean rfm_layout_is_details_size(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_cellheight: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_ICON_SIZE))
	g_error("rfm_layout_get_icon_size: view_layout_p item LAYOUT_ICON_SIZE is not set\n");
    if (view_layout_p->icon_size) return FALSE;
    return TRUE;
}

gint rfm_layout_get_cellheight(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_cellheight: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_CELLHEIGHT))
	g_error("rfm_layout_get_cellheight: view_layout_p item LAYOUT_CELLHEIGHT is not set\n");
    return view_layout_p->cellheight;
}

gint rfm_layout_get_cellwidth(view_t *view_p){
    if (!view_p) g_error("rfm_layout_get_cellwidth: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_CELLWIDTH))
	g_error("rfm_layout_get_cellwidth: view_layout_p item LAYOUT_CELLWIDTH is not set\n");
    return view_layout_p->cellwidth;
}

void rfm_get_drawable_geometry(Drawable drawable, 
	gint *x, gint *y, gint *w, gint *h, gint *d){
    rfm_global_t *rfm_global_p = rfm_global();
    if (rfm_get_gtk_thread() != g_thread_self()){
	g_error("rfm_get_drawable_geometry is a main thread function\n");
    }
    gint x_return, y_return;
    guint w_return, h_return, d_return, border_return;
    Window root_return;
    Display *display;
    if (rfm_global_p) display = rfm_global_p->Xdisplay;
    else display = gdk_x11_display_get_xdisplay(gdk_display_get_default());

    XGetGeometry(display, drawable, &root_return,
	    &x_return, &y_return, 
	    &w_return, &h_return, 
	    &border_return, 
	    &d_return);
    if (x) *x = x_return;
    if (y) *y = y_return;
    if (w) *w = w_return;
    if (h) *h = h_return;
    if (d) *d = d_return;
    return;
}

gint
rfm_get_default_size(void){
    gint default_size = -1;
    const gchar *rfm_default_icon_size = getenv("RFM_DEFAULT_ICON_SIZE");
    if (rfm_default_icon_size && strlen(rfm_default_icon_size) != 0){
	const gchar **p=icon_sizes_v;
	for (; p && *p; p++){
	    if (strcmp(*p, rfm_default_icon_size)==0){
		// Gotcha. If value is not valid, it will be ignored.
		if (strcmp(*p, "Normal")==0) {
		    default_size = SMALL_ICON_SIZE;
		} else
		if (strcmp(*p, "Compact")==0) {
		    default_size = TINY_ICON_SIZE;
		} else
		if (strcmp(*p, "Details")==0) {
		    default_size = LIST_ICON_SIZE;
		} else
		if (strcmp(*p, "Big")==0) {
		    default_size = MEDIUM_ICON_SIZE;
		} else
		if (strcmp(*p, "Huge")==0) {
		    default_size = BIG_ICON_SIZE;
		} 
	    }
	}
    }
    // Fallback condition
    if (default_size == -1) {
	NOOP("cannot determine default icon size\n");
	return SMALL_ICON_SIZE;
    }
    return default_size;
}

static
view_preferences_t *
get_view_preferences_path (gint type, const gchar * path) {
    NOOP ("rodent_get_view_preferences()\n");
    DBHashTable *preferences;
    const gchar *key;
    GString *gs;
    gchar *f;
    if (type==ICONVIEW_TYPE) {
	f = g_build_filename (GRID_PREFERENCES_FILE, NULL);
    } else {
	f = g_build_filename (DESK_PREFERENCES_FILE, NULL);
    }
        
    TRACE("opening %s...\n",f); 
    preferences = dbh_new (f, NULL, DBH_READ_ONLY|DBH_THREAD_SAFE|DBH_PARALLEL_SAFE);
    TRACE("open %s.\n",f); 
 
    if(!preferences) {
        NOOP ("Preferences table does not yet exist: %s\n", f);
        g_free (f);
        return NULL;
    }
    dbh_set_parallel_lock_timeout(preferences, 3);
    dbh_mutex_lock(preferences);

    if(path){
        key = path;
    } else {
        key = "RODENT_ROOT";
    }
    NOOP ("looking for preferences with key=%s\n", key);

    gs = g_string_new (key);
    sprintf ((char *)DBH_KEY (preferences), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    view_preferences_t *view_preferences_p=NULL;
    if(dbh_load (preferences)) {
        view_preferences_p=
		(view_preferences_t *)malloc(sizeof(view_preferences_t));
	if (!view_preferences_p) g_error("malloc: %s\n", strerror(errno));
	memcpy (view_preferences_p, DBH_DATA (preferences), 
		sizeof (view_preferences_t));

	NOOP ("got preferences with key=%s  (preferences=0x%x, sort_column%d)\n",
           key, view_preferences_p->preferences, view_preferences_p->sortcolumn);
    }
    dbh_mutex_unlock(preferences);
    dbh_close (preferences);
    g_free (f);
    return view_preferences_p;
}


view_preferences_t *
rfm_get_view_preferences (gint type, record_entry_t *target_en) {

    const gchar *key=NULL;
    if (target_en) {
	if (target_en->module){
	    key=(const gchar *)rfm_void(PLUGIN_DIR, target_en->module,
		    "module_preferences_key");
	}
	if (!key) key=target_en->path;

    } else {
	key="RODENT_ROOT";
    }
    view_preferences_t *view_preferences_p = get_view_preferences_path (type, key);
    return view_preferences_p;
}


void
rfm_save_view_preferences (view_t * view_p, record_entry_t *target_en) {
    view_preferences_t iconview_preferences;
    DBHashTable *preferences;
    GString *gs;
    gchar *f;
    const gchar *key=NULL;
    if(!view_p)
        return;

	NOOP("rfm_save_view_preferences: %s\n", key);
    
    if(!target_en){
        key = "RODENT_ROOT";
    } else if (target_en->module) {
        key=(const gchar *)rfm_void(PLUGIN_DIR, target_en->module, "module_preferences_key");
	NOOP("getting preferences key from module: %s\n", key);
    }
    if (!key){
	if (target_en->path==NULL){
	    key = "RODENT_ROOT";
	} else {
	    key = target_en->path;
	}
    }

    if (view_p->flags.type==ICONVIEW_TYPE) {
	f = g_build_filename (GRID_PREFERENCES_FILE, NULL);
    } else {
	f = g_build_filename (DESK_PREFERENCES_FILE, NULL);
    }
    gchar *tmp_f=NULL;
    

    TRACE("opening %s...\n",f); 
    preferences = dbh_new (f, NULL, DBH_PARALLEL_SAFE|DBH_THREAD_SAFE);
    if(!preferences) {
        NOOP ("creating file: %s", f);
	tmp_f =g_strdup_printf("%s-%d", f, (gint)getpid());
	unsigned char keylength=11;
        gchar *directory = g_path_get_dirname(f);
        if (!g_file_test(directory, G_FILE_TEST_IS_DIR)){
            g_mkdir_with_parents(directory, 0700);
        }
        g_free(directory);
        preferences = dbh_new (f, &keylength, DBH_PARALLEL_SAFE|DBH_THREAD_SAFE|DBH_CREATE);
    }
    TRACE("open %s.\n",f); 
    if(!preferences) {
        DBG ("cannot open file: %s\n", f);
        g_free (f);
        g_free (tmp_f);
        return;
    }
    dbh_set_parallel_lock_timeout(preferences, 3);
    dbh_mutex_lock(preferences);

    gs = g_string_new (key);
    sprintf ((char *)DBH_KEY (preferences), "%10u", g_string_hash (gs));
    g_string_free (gs, TRUE);

    view_layout_t *view_layout_p = &(view_p->view_layout);

    iconview_preferences.preferences = view_p->flags.preferences;
    iconview_preferences.sortcolumn = view_p->flags.sortcolumn;
    iconview_preferences.icon_size = view_layout_p->icon_size;
    NOOP("saving iconsize = %d\n", iconview_preferences.icon_size);

    memcpy (DBH_DATA (preferences), &iconview_preferences, sizeof (view_preferences_t));
    dbh_set_recordsize (preferences, sizeof (view_preferences_t));

    if(!dbh_update (preferences)) {
        DBG ("!dbh_update(preferences)\n");
    }
    dbh_mutex_unlock(preferences);
    dbh_close (preferences);
    if (tmp_f){
	if (rename(tmp_f, f) < 0){
	    DBG("rename(%s, %s) failed: %s\n",
		    tmp_f, f, strerror(errno));
	}
    }
    NOOP ("saved preferences with key=%s iconsize=%d \n", 
	    key, ICON_SIZE(view_p));
    g_free (f);
    g_free (tmp_f);
    return;
}


void
rfm_set_view_preferences (view_t *view_p, view_preferences_t *view_preferences_p) {
    if(!view_preferences_p){
	// No user defined preferences for the "path". We shall now
    	// fall back to the default settings.
	NOOP( "default settings\n");
	if (view_p->en && IS_LOCAL_TYPE(view_p->en->type)){
	    view_p->flags.preferences = DEFAULT_VIEW_PREFERENCES;
	} else {
	    view_p->flags.preferences = DEFAULT_REMOTE_PREFERENCES;
	}

        view_p->flags.sortcolumn = DEFAULT_SORT_COLUMN;  
	rfm_layout_set_icon_size(view_p, rfm_get_default_size());
	return;
    }

    // We were successful in retrieving user preferences for the "path".
    view_p->flags.preferences = view_preferences_p->preferences;
    view_p->flags.sortcolumn = view_preferences_p->sortcolumn;
    rfm_layout_set_icon_size(view_p, view_preferences_p->icon_size);
    if (view_p->en) {
	if(view_p->flags.preferences & __SHOW_HIDDEN){
	    NOOP( "rfm_set_view_preferences: shows hidden...\n");
	    SET_SHOWS_HIDDEN (view_p->en->type);
	} else {
	    NOOP( "rfm_set_view_preferences: does *not* show hidden...\n");
	    UNSET_SHOWS_HIDDEN (view_p->en->type);
	}
	if(view_p->flags.preferences & __SHOW_IMAGES){
	    SET_SHOWS_IMAGES (view_p->en->type);
	} else {
	    UNSET_SHOWS_IMAGES (view_p->en->type);
	}
	SET_USER_PREFERENCES(view_p->flags.preferences);
    }

    return;
}

static void *
get_pango_layout_height(void *data)
{
    if (rfm_get_gtk_thread() != g_thread_self()){
	DBG("get_pango_layout_height is a main thread function main=0x%x self=0x%x\n",GPOINTER_TO_INT(rfm_get_gtk_thread()), GPOINTER_TO_INT(g_thread_self()));
    }
    view_t *view_p = data;
    if (!view_p->widgets.paper) g_error("view_p->widgets.paper is NULL\n");
    PangoLayout *layout;
    PangoRectangle logical_rect;
    const gchar *tag = "RIDE A BIKE TODAY";
    layout = gtk_widget_create_pango_layout (view_p->widgets.paper, tag);
    layout_set_variable_font(layout);
    pango_layout_get_pixel_extents (layout, NULL, &(logical_rect));
    gint height = logical_rect.height;
    g_object_unref (layout);
    return GINT_TO_POINTER(height);
}

static void *
set_paper_size(gpointer data){
    if (rfm_get_gtk_thread() != g_thread_self()){
	g_error("set_paper_size is a main thread function\n");
    }
    view_t *view_p = data;
    view_layout_t *view_layout_p = &(view_p->view_layout);
    NOOP(stderr, "********   set_paper_size: %d, %d\n", 
	    view_layout_p->paperX, view_layout_p->paperY);
    gtk_widget_set_size_request (view_p->widgets.paper, view_layout_p->paperX, view_layout_p->paperY);
    g_object_set_data(G_OBJECT(view_p->widgets.paper), "paperX", 
	    GINT_TO_POINTER(view_layout_p->paperX));
    g_object_set_data(G_OBJECT(view_p->widgets.paper), "paperY", 
	    GINT_TO_POINTER(view_layout_p->paperY));
    return NULL;
}


static void *
context_sync(gpointer data){
    if (rfm_get_gtk_thread() != g_thread_self()){
	g_error("context_sync is a main thread function\n");
    }
    gdk_display_sync (gdk_display_get_default ());    
    while (gtk_events_pending()) gtk_main_iteration();
    if (data) return data;
    return NULL;
}


static void *
do_layout_f(void *data){
    void **arg = data;
    view_t *view_p = arg[0];
    population_t *population_p = arg[1];
    if (population_p->layout == NULL) rfm_do_layout (view_p, population_p);
    return NULL;
}

//
void 
rfm_layout_pango_layout_setup(view_t *view_p){
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (view_layout_p->icon_size > TINY_ICON_SIZE) return;

    // Get longest label:
    
    population_t **pp = view_p->population_pp;
    gint max_length = 0;
    population_t *population_p = NULL;
    for (; pp && *pp; pp++){
	gchar *label = g_strdup((*pp)->label);
	if (label && strlen(label)>max_length) {
	    max_length = strlen(label);
	    population_p = *pp;
	}
	g_free(label);
    }
    if (population_p) {
	void *arg[]={view_p, population_p};
	rfm_context_function(do_layout_f, arg);
	view_layout_p->name_width = population_p->logical_rect.width;
	GdkRectangle rect;
	rfm_layout_get_vpane_allocation(view_p, &rect);
	view_layout_p->details_width = rect.width - view_layout_p->name_width;
    } 

    NOOP(stderr, "+++ rfm_layout_details_setup : name width=%d, detail width=%d\n",
		view_layout_p->name_width,view_layout_p->details_width);
}


void 
rfm_layout_get_vpane_allocation(view_t *view_p, GdkRectangle *rect){
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (view_p->flags.type == DESKVIEW_TYPE){
	rect->x = rect->y = 0;
	rect->width = view_layout_p->window_width;
	rect->height = view_layout_p->window_height;
	return;
    }
    if (!(view_layout_p->tripwire & LAYOUT_VPANE_ALLOCATION)){
	g_error("rfm_layout_get_vpane_allocation: LAYOUT_VPANE_ALLOCATION is not set\n");
    }
    if (!view_p || !rect) g_error("!view_p || !rect\n");
    memcpy(rect, &(view_layout_p->vpane_allocation), sizeof(GdkRectangle));
    return;
}
void 
rfm_layout_set_vpane_allocation(view_t *view_p){
    if (rfm_get_gtk_thread() != g_thread_self()) 
	g_error("rfm_layout_set_vpane_allocation is a main thread function\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (view_p->flags.type == DESKVIEW_TYPE) {
	view_layout_p->vpane_allocation.width = view_layout_p->window_width;
	view_layout_p->vpane_allocation.height = view_layout_p->window_height;
	return;
    }
    GtkWidget *vpane = g_object_get_data(G_OBJECT(view_p->widgets.paper), "vpane");
    if (!vpane){
	// deskview should not have entered this function.
	g_error("rfm_layout_set_vpane_allocation: vpane==NULL\n");
    }
    gtk_widget_get_allocation (vpane, &(view_layout_p->vpane_allocation));
    view_layout_p->tripwire |= LAYOUT_VPANE_ALLOCATION;
}

// This function sets the iconsize scale in gridview windows to the 
// size saved for the particular directory in the view.
void *
rfm_layout_set_icon_size_scale(void * data){
    view_t *view_p = data;
    GtkWidget *size_scale = g_object_get_data(G_OBJECT(view_p->widgets.paper), "size_scale");
    if (size_scale == NULL){
	return NULL;
    }
    gtk_range_set_value (GTK_RANGE(size_scale), view_p->view_layout.icon_size);
    g_object_set_data(G_OBJECT(size_scale), "oldvalue", GINT_TO_POINTER(view_p->view_layout.icon_size));
    
    gint sizeid = GPOINTER_TO_INT(g_object_get_data(G_OBJECT(size_scale), "sizeid"));
    if (!sizeid) 
        g_object_set_data(G_OBJECT(size_scale), "sizeid", GINT_TO_POINTER(1+ICON_SIZE_ID(view_p)));
    else if (sizeid-1 != ICON_SIZE_ID(view_p)) {
        gchar *t=NULL;
        switch (ICON_SIZE_ID(view_p)){
            case BIG_ICON_SIZE: t = g_strdup_printf(_("Huge (%sx%s)"), "96", "96"); break;
            case MEDIUM_ICON_SIZE: t = g_strdup_printf(_("Large (%sx%s)"), "72", "72"); break;
            case SMALL_ICON_SIZE: t = g_strdup_printf(_("Medium (%sx%s)"), "48", "48"); break;
            case TINY_ICON_SIZE: t = g_strdup_printf(_("Small (%sx%s)"), "24", "24"); break;
            case LIST_ICON_SIZE: t =  g_strdup(_("List View")); break;
        }
        rfm_diagnostics(&(view_p->widgets), "rodent",_("Icon size:")," ", t, "\n", NULL);
        g_free(t);
        g_object_set_data(G_OBJECT(size_scale), "sizeid", GINT_TO_POINTER(1+ICON_SIZE_ID(view_p)));
    }
    
    return NULL;
}


void 
rfm_layout_set_icon_size(view_t *view_p, gint size){
    NOOP(stderr, "--- icon size set to %d\n", size);
    switch (size){
	case BIG_ICON_SIZE:
	case MEDIUM_ICON_SIZE:
	case SMALL_ICON_SIZE:
	case TINY_ICON_SIZE:
	case LIST_ICON_SIZE:
	    break;
	default:
	    g_error("rfm_layout_set_icon_size: incorrect size (%d)\n", size);
    }
    view_layout_t *view_layout_p = &(view_p->view_layout);
    view_layout_p->tripwire |= LAYOUT_ICON_SIZE;
    view_layout_p->icon_size = size;  

}

void 
rfm_layout_set_icon_size_full(view_t *view_p, gint size){
    if (!view_p) g_error("rfm_layout_set_icon_size_full: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!view_layout_p->max_elements) g_error("rfm_layout_set_icon_size_full: !view_layout_p->max_elements\n");

    rfm_layout_set_icon_size(view_p, size);
    // this should be set already... rfm_layout_set_vpane_allocation(view_p);
    rfm_layout_configure(view_p, view_layout_p->max_elements);
}

static void 
layout_reset_columns(view_t *view_p){
    view_layout_t *view_layout_p = &(view_p->view_layout);
    
    // Set the column width (required for cellwidth)
    if (view_layout_p->icon_size >= SMALL_ICON_SIZE) {
	view_layout_p->column_width = view_layout_p->icon_size * 2; 
    } else if (view_layout_p->icon_size >= TINY_ICON_SIZE) {
	view_layout_p->column_width = SMALL_ICON_SIZE * 4; 
    } else {
	if (view_p->flags.type == DESKVIEW_TYPE) {
	    view_layout_p->column_width = rfm_allocation.width - 
	    view_layout_p->margin_right - view_layout_p->margin_left; 
	} else {
	    view_layout_p->column_width = view_layout_p->vpane_allocation.width; 
	}
    }
    // Set the text_height (required for cellheight)
    if (rfm_get_gtk_thread() == g_thread_self()){
	view_layout_p->text_height = GPOINTER_TO_INT(get_pango_layout_height(view_p));
    } else {
	view_layout_p->text_height = GPOINTER_TO_INT(rfm_context_function(get_pango_layout_height, view_p));
    }

    // Now set cellwidth and cell height

    view_layout_p->cellwidth = view_layout_p->column_width;
    if (view_layout_p->icon_size >= SMALL_ICON_SIZE){
	view_layout_p->cellheight = view_layout_p->icon_size + (2*(view_layout_p->text_height + TEXTSPACING)); 
    } else {
	// icon_size may be zero for details, so use the hardcode value, TINY_ICON_SIZE.
	view_layout_p->cellheight = TINY_ICON_SIZE + TEXTSPACING; 
    }
    view_layout_p->tripwire |= (LAYOUT_CELLWIDTH|LAYOUT_CELLHEIGHT);

    // Width is equal to window width, minus margin.
    // Margin is (CELLWIDTH(view_p) / 2.
    gint width = rfm_allocation.width - (view_layout_p->cellwidth / 2);

    // Now get number of grid rows and columns to fit view_layout_p->max_elements.
    GtkWidget *vpane = g_object_get_data(G_OBJECT(view_p->widgets.paper), "vpane");
    if (vpane == NULL) { // transposed...
	gint height = rfm_allocation.height - 
	    view_layout_p->margin_bottom - view_layout_p->margin_top;
	width = rfm_allocation.width - 
	    view_layout_p->margin_right - view_layout_p->margin_left;


	view_layout_p->grid_rows = ((double)height / view_layout_p->cellheight) + 0.5;
	view_layout_p->grid_columns = ((double)width / view_layout_p->cellwidth) + 0.5;
	if (view_layout_p->grid_rows * view_layout_p->grid_columns < view_layout_p->max_elements) {
	    view_layout_p->max_elements = view_layout_p->grid_rows * view_layout_p->grid_columns;
	}
    } else  {// (iconview)
	view_layout_p->grid_columns = ((double)width / view_layout_p->cellwidth) + 0.5;
	view_layout_p->grid_rows = (view_layout_p->max_elements) / view_layout_p->grid_columns;
	if(view_layout_p->max_elements % view_layout_p->grid_columns)
	    view_layout_p->grid_rows++;

	// Correct for number of rows in case the paper must also fill empty space.

	gint items=(view_layout_p->cellwidth==0 || view_layout_p->cellheight==0)?-1: 
		   view_layout_p->vpane_allocation.width * 
		   view_layout_p->vpane_allocation.height /
		   view_layout_p->cellwidth/ view_layout_p->cellheight;
	NOOP(stderr, "init_grid (%d):items=%d pane allocation:w=%d h=%d \n", 
		view_layout_p->max_elements, items,
		view_layout_p->vpane_allocation.width,
		view_layout_p->vpane_allocation.height);
	gint total_rows = items;

	if (view_layout_p->grid_columns > 0) {
	    gint rows = total_rows / view_layout_p->grid_columns;
	    if (total_rows % view_layout_p->grid_columns != 0){
		// remainder...
		rows++;
	    }
	    total_rows = rows;
	} else total_rows = -1;
	if (view_layout_p->max_elements <= items - view_layout_p->grid_columns) {
	    // hacky correction for when population does not fill paper completely
	    total_rows = -1;
	}
       
	if (total_rows > 0 && total_rows > view_layout_p->grid_rows) {
	    view_layout_p->grid_rows = total_rows;
	    NOOP( "changing rows to %d (items = %d)\n", total_rows, items);
	} 
    }	
    NOOP (stderr, "layout_reset_columns: rfm_allocation.width=%d rfm_allocation.height=%d\n",rfm_allocation.width,rfm_allocation.height);
    NOOP (stderr, "layout_reset_columns: view_layout_p->max_elements=%d cellwidth=%d cellheight=%d\n",
	    view_layout_p->max_elements, view_layout_p->cellwidth, view_layout_p->cellheight);
    NOOP (stderr, "layout_reset_columns: columns=%d rows=%d\n",
	    view_layout_p->grid_columns, view_layout_p->grid_rows);
   view_layout_p->grid_area = view_layout_p->grid_rows * view_layout_p->grid_columns;
}

static void 
layout_configure(view_t *view_p, gint pathc){
   if (!view_p) g_error("layout_configure: !view_p\n");
    rfm_global_t *rfm_global_p = rfm_global();
    if(!rfm_global_p) g_error ("layout_configure: rfm_global_p==NULL\n");
    if(!rfm_global_p->window) g_error ("layout_configure: rfm_global_p->window==NULL\n");

    if (pathc == 0){
	DBG("init_grid: pathc==0, skipping...\n");
	return;
    }
    if(!rfm_global_p->window) g_error ("layout_configure: rfm_global_p->window==NULL\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);

    if (!(view_layout_p->tripwire & LAYOUT_ICON_SIZE))
	g_error("view_layout_p item LAYOUT_ICON_SIZE is not set\n");
    GtkWidget *vpane = g_object_get_data(G_OBJECT(view_p->widgets.paper), "vpane");
    if (vpane && !(view_layout_p->tripwire & LAYOUT_VPANE_ALLOCATION))
	g_error("view_layout_p item LAYOUT_VPANE_ALLOCATION is not set\n");
    
    NOOP(stderr, "layout_configure(): pathc=%d\n", pathc);
    view_layout_p->max_elements = pathc;
    layout_reset_columns(view_p);

     if (view_layout_p->icon_size < TINY_ICON_SIZE){
        view_layout_p->paperX = 192; // hack : 192 is maximum cellwidth
	//view_layout_p->name_width + view_layout_p->details_width;
    } else 
    {
        view_layout_p->paperX = view_layout_p->cellwidth;
    }

    view_layout_p->paperY = view_layout_p->grid_rows *  view_layout_p->cellheight;
    view_layout_p->tripwire |= LAYOUT_ROOT_CONFIGURED;
    rfm_layout_set_window_size(view_p, rfm_allocation.width, rfm_allocation.height);
    
    return;    
}

gboolean rfm_layout_set_window_size(view_t *view_p, gint width, gint height){
    if (!view_p) g_error("rfm_layout_set_window_size: !view_p\n");
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (width == view_layout_p->window_width && height == view_layout_p->window_height){
	return FALSE;
    }
    view_layout_p->window_width = width;
    view_layout_p->window_height = height;
    return TRUE;
}

void 
rfm_layout_configure(view_t *view_p, gint pathc){
    NOOP(stderr, "+++rfm_layout_configure\n");
    // (configure view layout)
    layout_configure(view_p, pathc);

    // check paper size
    view_layout_t *view_layout_p = &(view_p->view_layout);
#if 10
    gboolean setup_not_ok = 
	!view_layout_p->paperX || !view_layout_p->paperY ||
	view_layout_p->paperX != 
	GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.paper), "paperX"))
	||
	view_layout_p->paperY != 
	GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.paper), "paperY"));
#else
    // This does not work (at present).
    // Does not present too much advantages either, memorywise.
    gboolean setup_not_ok = 
	!view_layout_p->paperX || !view_layout_p->paperY ||
	view_layout_p->paperX < 
	GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.paper), "paperX"))
	||
	view_layout_p->paperY < 
	GPOINTER_TO_INT(g_object_get_data(G_OBJECT(view_p->widgets.paper), "paperY"));

#endif
    
    if (setup_not_ok){ 

	//static gint c=1;NOOP(stderr, "rfm_layout_configure(): reset paper size (%d)...\n)",c); 

	// set paper size
	g_object_set_data(G_OBJECT(view_p->widgets.paper), "paper_setup", NULL);
	rfm_context_function(set_paper_size, view_p);
	rfm_context_function(context_sync, NULL);
	

	// now that we have sync'd, activate reload on size changes.
	g_object_set_data(G_OBJECT(view_p->widgets.paper), "paper_setup", GINT_TO_POINTER(1));
	GtkWidget *size_scale = g_object_get_data(G_OBJECT(view_p->widgets.paper), "size_scale");
	if (size_scale != NULL){
	    // Disable reload by scale change signal,
	    g_object_set_data(G_OBJECT(view_p->widgets.paper), "size_scale_disabled", GINT_TO_POINTER(1));
	    rfm_context_function(rfm_layout_set_icon_size_scale, view_p);
	    // sync
	    rfm_context_function(context_sync, NULL);
	    // and now reactivate reload by scale change signal.
	    g_object_set_data(G_OBJECT(view_p->widgets.paper), "size_scale_disabled", NULL);
	}
    }
}


////////////////////////////////////////////////////////////////////////////////////////////

gboolean 
rfm_get_population_rect(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_CELLWIDTH))
	g_error("view_layout_p item LAYOUT_CELLWIDTH is not set\n");
    if (!(view_layout_p->tripwire & LAYOUT_CELLHEIGHT))
	g_error("view_layout_p item LAYOUT_CELLHEIGHT is not set\n");

    if (!population_p || !rect) {
	DBG("rfm_get_population_rect() called with !population_p || !rect\n");
	return FALSE;
    }
    if (!rfm_population_try_read_lock(view_p, "rfm_get_population_rect")) return FALSE;
    gint cellwidth = view_layout_p->cellwidth;
    gint cellheight = view_layout_p->cellheight;
    rect->x = population_p->column * cellwidth + MARGIN_LEFT(view_p);
    rect->y = population_p->row * cellheight + MARGIN_TOP(view_p);
    rect->width = cellwidth;
    rect->height = cellheight;

    rfm_population_read_unlock(view_p, "rfm_get_population_rect");
    return TRUE;
}

gboolean 
rfm_get_population_icon_rect(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_CELLWIDTH))
	g_error("view_layout_p item LAYOUT_CELLWIDTH is not set\n");
    if (!(view_layout_p->tripwire & LAYOUT_CELLHEIGHT))
	g_error("view_layout_p item LAYOUT_CELLHEIGHT is not set\n");
    gint cellwidth = view_layout_p->cellwidth;
    gint cellheight = view_layout_p->cellheight;
    if (!rfm_population_try_read_lock(view_p, "rfm_get_population_icon_rect")) {
	DBG("rfm_get_population_icon_rect(): failed to get read lock\n");
	return FALSE;
    }
    rect->x = population_p->column * cellwidth + MARGIN_LEFT(view_p);
    rect->y = population_p->row * cellheight + MARGIN_TOP(view_p);
 

    gint icon_size = view_layout_p->icon_size;
    if (!icon_size) icon_size = TINY_ICON_SIZE;
    rect->width =  icon_size;
    rect->height = icon_size;
    
    rfm_population_read_unlock(view_p, "rfm_get_population_icon_rect");
    return TRUE;
}

gboolean 
rfm_get_population_label_rect(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    view_layout_t *view_layout_p = &(view_p->view_layout);
    if (!(view_layout_p->tripwire & LAYOUT_CELLWIDTH))
	g_error("view_layout_p item LAYOUT_CELLWIDTH is not set\n");
    if (!(view_layout_p->tripwire & LAYOUT_CELLHEIGHT))
	g_error("view_layout_p item LAYOUT_CELLHEIGHT is not set\n");
    gint cellwidth = view_layout_p->cellwidth;
    gint cellheight = view_layout_p->cellheight;
    if (!rfm_population_try_read_lock(view_p, "rfm_get_population_label_rect")) {
	DBG("rfm_get_population_label_rect(): failed to get read lock\n");
	return FALSE;
    }
    rect->x = population_p->column * cellwidth + MARGIN_LEFT(view_p);
    rect->y = population_p->row * cellheight + MARGIN_TOP(view_p);
     
    rect->height=0;
    rect->width=0;
    if(population_p->layout) {
	rect->height += population_p->logical_rect.height;
	rect->width = population_p->logical_rect.width;
    }
    gint x_offset = (view_layout_p->icon_size >= SMALL_ICON_SIZE)?
	(cellwidth - population_p->logical_rect.width) / 2:
	 TINY_ICON_SIZE + 2;
    // Details icon_size (0) does not have a layout2.
    if (view_layout_p->icon_size && population_p->layout2) {
	rect->height += population_p->logical_rect2.height;
	if (population_p->logical_rect2.width > population_p->logical_rect.width){
	    rect->width = population_p->logical_rect2.width;
	}
	gint x_offset2 = (view_layout_p->icon_size >= SMALL_ICON_SIZE)?
		(cellwidth - population_p->logical_rect2.width) / 2:
		TINY_ICON_SIZE + 2;

	if (x_offset2 < x_offset) {
	    x_offset = x_offset2;
	}
    }

    if (rect->x + x_offset<0) x_offset=0;
    else  rect->x += x_offset;	    
    if (view_layout_p->icon_size >= SMALL_ICON_SIZE) {
	rect->y += (TEXTSPACING + ICON_SIZE(view_p));	  
    } else {
	rect->y += TEXTSPACING;	  
    }

    rfm_population_read_unlock(view_p, "rfm_get_population_label_rect");

        // test code
    //if (population_p->flags & LABEL_SATURATED){
	//rect->x -= rect->width; if (rect->x < 0) rect->x = 0;
	//rect->width *= 2; rect->width +=2;
    //}

    return TRUE;
}

gboolean
rfm_get_population_label_rect_full(view_t * view_p, const population_t * population_p, GdkRectangle *rect){
    if (!view_p || ! population_p) {
	    g_error("incorrect function call: layout_full_dimensions()");
    }
    if (!rfm_population_try_read_lock(view_p, "rfm_get_population_label_rect_full")) {
	DBG("rfm_get_population_label_rect(): failed to get read lock\n");
	return FALSE;
    }
	    
    GdkRectangle item_rect;
    GdkRectangle label_rect;
    if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) {
	DBG("rfm_get_population_label_rect_full: !rfm_get_population_label_rect\n");
	return FALSE;
    }
    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) {
	DBG("rfm_get_population_label_rect_full: !rfm_get_population_rect\n");
	return FALSE;
    }
    view_layout_t *view_layout_p = &(view_p->view_layout);
    rect->width = population_p->logical_rect_full.width+4;
    rect->height = population_p->logical_rect_full.height+4;
    rect->y = label_rect.y+2;
    // with texts under the icons, center the saturated text.
    if (view_layout_p->icon_size >= SMALL_ICON_SIZE) {
	gint center = item_rect.x + (item_rect.width/2);
	rect->x = center - (rect->width/2);
    } else {
	// texts to the right of the icon, use label.x
	rect->x = label_rect.x;
    }
	
    // correct for overlap
    gint scrollbar_width = 25;
    gint area_width;
    GtkWidget *vpane = g_object_get_data(G_OBJECT(view_p->widgets.paper), "vpane");
    if (vpane) {
	area_width = view_layout_p->vpane_allocation.width;
    } else {
	area_width = rfm_allocation.width;
    }
	
    NOOP(stderr, "overlap: x=%d, w=%d (%d), aw=%d nx=%d\n",rect->x,rect->width,rect->x+rect->width,area_width,area_width - rect->width);

    if (rect->x + rect->width + scrollbar_width >= area_width) {
	rect->x = area_width - rect->width - scrollbar_width;
    }
    // correct for underlap
    if (rect->x < MARGIN_LEFT(view_p)) rect->x = MARGIN_LEFT(view_p);
    //if (rect->x < 0) rect->x = 0;
 
#if 0
    gint offset = 3;
    if (ICON_SIZE(view_p) >= SMALL_ICON_SIZE &&
		    population_p->logical_rect_full.width > CELLWIDTH(view_p)) {
	    GdkRectangle item_rect;
	    if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;
	offset = abs(label_rect.x - item_rect.x) + 5;
    }
#endif
    rfm_population_read_unlock(view_p, "rfm_get_population_label_rect_full");
    return TRUE;
}
#if 0
layout_full_dimensions(view_t * view_p,
	const population_t * population_p,
	gint *width, gint *height){

	if (!view_p || ! population_p) {
		g_error("incorrect function call: layout_full_dimensions()");
		return;
	}
    gint offset = 3;
   	if (ICON_SIZE(view_p) >= SMALL_ICON_SIZE &&
			population_p->logical_rect_full.width > CELLWIDTH(view_p)) {
		GdkRectangle item_rect;
		GdkRectangle label_rect;
		if (!rfm_get_population_label_rect(view_p, population_p, &label_rect)) return;
		if (!rfm_get_population_rect(view_p, population_p, &item_rect)) return;
	    offset = abs(label_rect.x - item_rect.x) + 5;
	}
	// XXX logical_rect_full.width is not always truthful...
	if (width) *width =  offset + population_p->logical_rect_full.width;
	if (height) *height =  population_p->logical_rect_full.height;
}
#endif
////////////////////////////////////////////////////////////////////////////////////////////

