#ifdef COPYRIGHT_INFORMATION
#include "gplv3.h"
#endif
/*
 * Copyright (C) 2002-2012 Edscott Wilson Garcia
 * EMail: edscott@users.sf.net
 *
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; 
 */
//
//

/********/
#define MNTTYPE_PROCFS	"proc"
#define MNTTYPE_SMBFS	"smbfs"
#define MNTTYPE_DEV	"devpts"
#define MNTTYPE_SHM	"tmpfs"
#define	MNTTYPE_CDFS	"iso9660"
#define MNTTYPE_SWAP	"swap"
#define MNTTYPE_NFS	"nfs"

#define MNTTYPE_CACHEFS	"cachefs"       // proc
#define MNTTYPE_HSFS	"hsfs"  // proc

#define MNTTYPE_CODAFS	"coda"  //nfs
#define MNTTYPE_CDFS_AIX "cdrfs"        /* AIX */
#define MNTTYPE_KERNFS	"kernfs"        // proc
#define MNTTYPE_MFS	"mfs"   //proc

static void *private_is_mounted (void *p);
static gboolean is_user_type (const gchar * mnt_point);
static void * fstab_is_in_fstab (void *p);



static GHashTable *stat_hash;
static GHashTable *count_hash;
static GHashTable *data_hash;

static GMutex *
get_stat_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}

static GMutex *
get_count_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}

static GMutex *
get_data_hash_mutex(void){
    static GMutex *mutex = NULL;
    static gsize initialized = 0;
    if (g_once_init_enter (&initialized)){
	rfm_mutex_init(mutex);
      g_once_init_leave (&initialized, 1);
    }
    return mutex;
}


static int
get_fstab_type (const char *type) {

    if(strstr (type, MNTTYPE_NFS)) {
        NOOP ("    got %s type for fstab item %s\n", MNTTYPE_NFS, type);
        return __NFS_TYPE;
    } else if(strstr (type, MNTTYPE_SMBFS)) {
        NOOP ("    got %s type for fstab item %s\n", MNTTYPE_SMBFS, type);
        return __SMB_TYPE;
    } else if(strstr (type, MNTTYPE_PROCFS)) {
        NOOP ("    got %s type for fstab item %s\n", MNTTYPE_PROCFS, type);
        return __PROC_TYPE;
    } else if(strstr (type, MNTTYPE_DEV)) {
        NOOP ("    got %s type for fstab item %s\n", MNTTYPE_DEV, type);
        return __PROC_TYPE;
    } else if(strstr (type, MNTTYPE_SHM)) {
        NOOP ("    got %s type for fstab item %s\n", MNTTYPE_SHM, type);
        return __PROC_TYPE;
    } else if(strstr (type, MNTTYPE_CDFS)) {
        NOOP ("    got %s type for fstab item%s\n", MNTTYPE_CDFS, type);
        return __CDFS_TYPE;
    } else if(strstr (type, MNTTYPE_CDFS_AIX))
        return __CDFS_TYPE;
    else if(strstr (type, MNTTYPE_CACHEFS))
        return __PROC_TYPE;
    else if(strstr (type, MNTTYPE_HSFS))
        return __PROC_TYPE;
    else if(strstr (type, MNTTYPE_KERNFS))
        return __PROC_TYPE;
    else if(strstr (type, MNTTYPE_MFS))
        return __PROC_TYPE;
    else if(strstr (type, MNTTYPE_MFS))
        return __NFS_TYPE;

    return 1;
}

///////////   common code:

static void *
fstab_do_properties (void *p, void *q) {
    
    widgets_t *widgets_p = p;
    if (!q){
	DBG("!q\n");
	return NULL;
    }
    GList * select_list=q;
    NOOP("do_prop\n");
    
    record_entry_t *en = select_list->data;

    
    if(IS_PARTITION_TYPE (en->type)) {
        gchar *sudo = g_find_program_in_path ("sudo");
        if(!sudo){
	    DBG("sudo command not found\n");
            goto done;
	}
        g_free (sudo);

        gchar *device = g_strdup (en->path);
	NOOP("*device = %s", device);
        while(strlen (device) && isdigit (device[strlen (device) - 1])) {
            device[strlen (device) - 1] = 0;
	    NOOP("device => %s", device);
        }
        gchar *argv[] = { "sudo", "-A", "fdisk", "-l", device, NULL };
	NOOP("%s -A fdisk -l %s", sudo, device);
	rfm_threaded_show_text (widgets_p);
	

        rfm_thread_run_argv (widgets_p, argv, FALSE);
        g_free (device);
        goto done;
    } else {
	NOOP("IS_PARTITION_TYPE (en->type) == FALSE\n");
        return NULL;
    }
done:

    return GINT_TO_POINTER (1);
}

static void *
fstab_hide_local_menu_items (void *p, void *q) {
    record_entry_t *en = q;
    widgets_t *widgets_p = p;
    view_t *view_p=widgets_p->view_p;
    if(!q)
        return NULL;

    // nothing selected:
    gchar *symbols_null[] = {
        "paste_menuitem",
        NULL
    };


    gchar *symbols[] = {
        "cut_menuitem",
        "rename_menuitem",
        "autotype_Prun",
        "paste_menuitem",
        NULL
    };

    gchar *symbols_partition[] = {
        "open_with_menuitem",
        "copy_menuitem",
        "duplicate_menuitem",
        "symlink_menuitem",
        "touch_menuitem",
        "navigation_separator",
        "file_separator",
        
        //"properties2",
        NULL
    };
    gchar *symbols_multiple_select[] = {
        "copy_menuitem",
        "remove_menuitem",
        "properties2", "find2",
        "open_with_menuitem",
        "ejecutar1", "ejecutar2", "ejecutar3",
        "ejecutar4", "ejecutar5", "ejecutar6", "ejecutar7",
        "ejecutar8", "ejecutar9", "ejecutar10",
        "navigation_separator", 
        "file_separator",

        NULL
    };
/*    gchar *symbols_mixed[] = { 
	"copy_menuitem",
	"remove_menuitem",
	NULL
    };*/

    gchar *symbols_dir[] = {
        "module1_menu",
        NULL
    };

    gchar **pp;
    if(!p) {
        for(pp = symbols_null; *pp; pp++) {
            HIDE_IT (*pp);
        }
        return NULL;
    }

    for(pp = symbols; *pp; pp++) {
        HIDE_IT ( *pp);
    }
    for(pp = symbols_partition; *pp; pp++) {
        if(IS_PARTITION_TYPE (en->type)) {
            HIDE_IT ( *pp);
        }
    }
    for(pp = symbols_dir; *pp; pp++) {
        if(!IS_PARTITION_TYPE (en->type)) {
            HIDE_IT ( *pp);
        }
    }
    if(g_slist_length (view_p->selection_list) > 1) {
        for(pp = symbols_multiple_select; *pp; pp++) {
            HIDE_IT ( *pp);
        }
    }
    return GINT_TO_POINTER (1);
}

static gchar *
get_mnt_point(widgets_t *widgets_p, record_entry_t *en) {
    gchar *mount_point = rfm_natural(RFM_MODULE_DIR, "callbacks", en, "callback_mnt_point");
    NOOP( "fstab: getting mount point; %s\n", mount_point);
    // XXX if mount_point is NULL tell user something!
    return mount_point;
}

#ifdef THIS_IS_LINUX
static gchar *types[]={ "auto", "vfat", "iso9660","ntfs", "ntfs-3g",
         "ext", "ext2", "ext3", "ext4",
        "adfs",  "affs",  "autofs",  "cifs",  "coda",  
	"cramfs", "debugfs", "devpts", "efs", 
        "hfs", "hfsplus", "hpfs",  "jfs", "minix", "msdos", "ncpfs",
        "nfs", "nfs4",  "proc", "qnx4", "ramfs", "reiserfs",
        "romfs", "squashfs",  "smbfs",  "sysv",  "tmpfs",  "ubifs", "udf", 
        "ufs", "umsdos", "usbfs", "xfs", "xiafs", NULL};
#else
# ifdef THIS_IS_BSD
static gchar *types[]={"auto", "msdosfs","cd9660", "ntfs","mfs", 
       	"nfs",  "nwfs", "nullfs", "oldnfs", "portalfs",
        "smbfs", "udf", "unionfs", NULL};

static pthread_mutex_t mntmutex = PTHREAD_MUTEX_INITIALIZER;
static gchar *
get_bsd_partition(const gchar *p){
    NOOP("get_bsd_partition: %s\n", p);
    if (!p) return NULL;
    gchar *mnt_point = realpath((gchar *)p, NULL);
    if (!mnt_point) return NULL;

    pthread_mutex_lock(&mntmutex);
    struct statfs *mnt_buf;
    size_t mnt_items = getmntinfo(&mnt_buf, MNT_NOWAIT);

    gint i=0;
    gchar *mnt_partition = NULL;
    for (;i<mnt_items; i++){
        if(strcmp (mnt_point, (mnt_buf+i)->f_mntonname) == 0 ||
           strcmp (mnt_point, (mnt_buf+i)->f_mntfromname) == 0) {
	    mnt_partition = g_strdup((mnt_buf+i)->f_mntfromname);
	    break;
	}
    }
    pthread_mutex_unlock(&mntmutex);
    g_free(mnt_point);
    NOOP("get_bsd_partition: %s -> %s\n", p, mnt_partition);
    return mnt_partition;
}

# else
# error "This should not happen"
# endif
#endif

static gchar *
get_mnt_type(widgets_t * widgets_p, gchar *mnt_device){
    NOOP( "fstab: getting radio response\n");
    gchar *title_txt = g_strdup_printf("%s %s", _("Mount"), mnt_device);
    gchar *label_txt = g_strdup( _("Type"));
    gchar *type = rfm_complex(RFM_MODULE_DIR, "callbacks", 
	    title_txt, label_txt, types, "get_radio_response");
    g_free(title_txt);
    g_free(label_txt);
    // XXX if mount_point is NULL tell user something!
    NOOP( "selected type is %s\n", type);

    return type;
}

static void *
private_fstab_mount (widgets_t *widgets_p, record_entry_t *en) {
    gchar *argument[MAX_COMMAND_ARGS];
    gchar **ap;
    gchar *umount = "umount";
    gchar *mount = "mount";
    gchar *mnt_src = NULL;
    gchar *mnt_tgt = NULL;

    SETWD();

    if(!widgets_p) return NULL;

    if(!en) return NULL;

    mnt_src = realpath(en->path, NULL);
    if (!mnt_src) mnt_src = g_strdup (en->path);
    mnt_tgt = NULL;

    // user option check for mount point
    // this is necesary for mount/umount command outside of
    // the fstab module directory
    if(is_user_type (mnt_src)) {
        SET_USER_TYPE (en->type);
    }

    gchar *options = NULL;
    gchar *mount_type = NULL;
    NOOP("private_fstab_mount: %s\n", mnt_src);
    if(IS_PARTITION_TYPE (en->type) && private_is_mounted ((void *)mnt_src) == NULL && !IS_USER_TYPE (en->type)) {
	mount_type = get_mnt_type (widgets_p, mnt_src);
	if (!mount_type) {
	    return NULL; // cancel from mount type dialog.
	}
#ifdef THIS_IS_LINUX
	if (strcmp(mount_type, "vfat")==0)
	    options = g_strdup_printf ("uid=%d", geteuid ());
#else
# ifdef THIS_IS_BSD
	if (strcmp(mount_type, "msdosfs")==0)
	    options = g_strdup_printf ("-m=644,-M=755");
# else
# error "This should not happen"
# endif
#endif
	if (strcmp(mount_type, "auto")==0){
	    g_free(mount_type);
	    mount_type = NULL;
	}

	mnt_tgt=get_mnt_point(widgets_p, en);
	if (mnt_tgt == NULL) {
            g_free (mnt_src);
            return NULL;
	}
    } // end of partition type condition.

    // isofs:
    if (is_iso_image(en)) {
	mnt_tgt=get_mnt_point(widgets_p, en);
	if (mnt_tgt == NULL) {
            g_free (mnt_src);
            return NULL;
	}
	options = g_strdup("loop");
    }
    NOOP ("MOUNT: mnt_src=%s mnt_tgt=%s\n", mnt_src, mnt_tgt);

    ap = argument;

    // Sudo check...
    // 
    // BSD not necessary if vfs.usermount?
#ifdef THIS_IS_LINUX
    gboolean use_sudo = TRUE;
#else
# ifdef THIS_IS_BSD
    int old_p=0;
    gboolean use_sudo = FALSE;
    size_t size=sizeof(int);

    if (sysctlbyname("vfs.usermount", &old_p, &size, NULL, 0) < 0){
	NOOP(stderr, "use_sudo: failed command: sysctlbyname(\"vfs.usermount\"\n");
	// Check access privileges to device
	use_sudo = TRUE;
    } else {
	if (old_p != 1){
	    NOOP(stderr, "use_sudo: old_p != 1\n");
	    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-warning", NULL);
	    rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strconcat("sysctl vfs.usermount=0", "\n", NULL));
	    rfm_threaded_diagnostics(widgets_p, "xffm/stock_help", NULL);
	    rfm_threaded_diagnostics(widgets_p, "xffm_tag/green", 
			g_strconcat(_("Use sysctl or add this line to \"/etc/sysctl.conf\":"),"\n", NULL));
	    rfm_threaded_diagnostics(widgets_p, "xffm_tag/magenta", 
		    g_strconcat("   ","vfs.usermount=1","\n", NULL));
	    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-info", NULL);
	    rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strconcat(_("Using sudo"), "...", "\n", NULL));

	   use_sudo = TRUE;
	} else {
	    struct stat st;
	    // get mount partition.
            NOOP("private_fstab_mount(%s)\n", mnt_src);
	    gchar *mnt_partition = get_bsd_partition(mnt_src);
	    stat((mnt_partition)?mnt_partition:mnt_src, &st);
	    if (!(st.st_mode & S_IWGRP)){
		rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-warning", NULL);
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strconcat(_("No write access to "),
			mnt_src, "\n", NULL));
		rfm_threaded_diagnostics(widgets_p, "xffm/stock_help", NULL);
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/green", 
			g_strconcat(_("Add these lines to /etc/devfs.rules:"),"\n", NULL));
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/magenta", 
			g_strconcat("[localrules=5]","\n", NULL));
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/magenta", 
			g_strconcat(_("add path 'da*' mode 0660 group operator"),"\n", NULL));
		rfm_threaded_diagnostics(widgets_p, "xffm/stock_help", NULL);
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/green", 
			g_strconcat(_("Add this line to /etc/rc.conf:"),"\n", NULL));
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/magenta", 
			g_strconcat("   ","devfs_system_ruleset=\"localrules\"","\n", NULL));
		rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-info", NULL);
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", g_strconcat(_("Using sudo"), "...", "\n", NULL));
		NOOP("No write access to %s\n",mnt_src);
		NOOP(stderr, "use_sudo: No write access to %s\n",mnt_src);
		use_sudo = TRUE;
	    }
	}
    }
# else
# error "This should not happen"
# endif

#endif
    // not for root
    if(!getuid ()){
        use_sudo = FALSE;
    }
    // not for general user mounts
    else if(IS_USER_TYPE (en->type) && fstab_is_in_fstab(en->path)){
        use_sudo = FALSE;
    }
    // sudo not installed
    else {
        char *p = g_find_program_in_path ("sudo");
        if(p == NULL) {
            use_sudo = FALSE;
            rfm_threaded_diagnostics (widgets_p, "xffm/stock_dialog-warning", g_strconcat(strerror (ENOENT), ": sudo", "\n", NULL));
        } else {
            g_free (p);
        }
    }

    if(use_sudo) {
        *ap++ = "sudo";
        *ap++ = "-A";
    }


#if 10
    if(private_is_mounted ((void *)mnt_src)) {
        *ap++ = umount;
    } else {
        *ap++ = mount;
	// Here we set the mounted type. This remains set until
	// monitor unsets it along with the greenball...
	SET_MOUNTED_TYPE(en->type);
	if (mount_type) {
            *ap++ = "-t";
            *ap++ = mount_type;

	}
        if(options) {
            *ap++ = "-o";
            *ap++ = options;
        }
    }
    memset(en->st, 0, sizeof(struct stat));

    *ap++ = (gchar *) mnt_src;
    if(mnt_tgt) {
        *ap++ = mnt_tgt;
    }
    *ap++ = NULL;
#if 0
    // XXX use threaded version here (?)
    NOOP ("MOUNT:");
    rfm_diagnostics (widgets_p, "xffm/stock_dialog-info", NULL);
    int i;
    for(i = 0; argument[i]; i++) {
        NOOP (" %s", argument[i]);
        if(strstr (argument[i], "username="))
            rfm_diagnostics (widgets_p, NULL, " username=*****", NULL);
        else
            rfm_diagnostics (widgets_p, NULL, " ", argument[i], NULL);
    }
    NOOP ("\n");
    rfm_diagnostics (widgets_p, NULL, "\n", NULL);
#endif

#endif

    rfm_global_t *rfm_global_p = rfm_global();
    rfm_cursor_wait (rfm_global_p->window);
    rfm_thread_run_argv (widgets_p, argument, FALSE);
    rfm_cursor_reset (rfm_global_p->window);
    //g_free (bash_command);



    NOOP ("fstab_mount %s done ", mnt_src);
    g_free (mnt_src);
    g_free (mnt_tgt);
    g_free (options);
    g_free (mount_type);
    return GINT_TO_POINTER (1);
}

static void *
fstab_double_click (void *p, void *q) {
    record_entry_t *en = (record_entry_t *) q;
    NOOP(stderr, "fstab doubleclick... %s\n", en->tag);
    widgets_t *widgets_p = (widgets_t *) p;
    if(IS_PARTITION_TYPE (en->type)) {
	if (!entry_is_mounted(en)){
	    rfm_threaded_show_text (widgets_p);
	    gchar *text = g_strdup_printf(_("The volume '%s' is not mounted."),
		    en->path);
	    rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-info", NULL);
	    rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", 
		    g_strconcat(text, "\n", NULL));
	    g_free(text);
	    return GINT_TO_POINTER (1);
	}
	set_mounts_info(en);
	
	if (en->tag && g_path_is_absolute(en->tag)){
	    if (access(en->tag, R_OK|X_OK )!=0){
		rfm_threaded_show_text (widgets_p);
		gchar *text = g_strdup_printf("%s: '%s'",
			strerror(EACCES), en->path);
		rfm_threaded_diagnostics(widgets_p, "xffm/stock_dialog-info", 
			 NULL);
		rfm_threaded_diagnostics(widgets_p, "xffm_tag/stderr", 
			g_strconcat(text, "\n", NULL));
		g_free(text);
		return GINT_TO_POINTER (1);
	    }
	    rodent_push_view_go_history ();
	    record_entry_t *new_en = rfm_stat_entry ((gchar *) en->tag, 0);
	    if (!rodent_refresh (widgets_p, new_en)){ rfm_destroy_entry(new_en); }
	}
        return GINT_TO_POINTER (1);
    }
    return NULL;
}


// gboolean
// This function does a module specific stat operation. This function
// is used by the Rodent monitor thread to determine if stat information
// for the view's pointer entry record has changed, as that will trigger
// a soft reload.
// Parameter p is the path to stat.
// Parameter q is a pointer to a struct stat record to place stat information.
// if p is NULL, then the function checks the top level stat file (module defined).
// if q is NULL, then the function just checks to see if the stat is possible.
static void *
fstab_module_stat (void *p, void *q) {
    NOOP("fstab_module_stat now, p=%s q=0x%x\n", (gchar *)p, GPOINTER_TO_INT(q));
    const gchar *path = p;
    // if p is NULL, then the function checks the top level stat file
    // and extra fstab file. (two files, yes)
    if(path == NULL && q != NULL) {
	struct stat st[2];
	if(stat ("/etc/fstab", st) < 0) {
	    DBG ("stat(%s): %s", "/etc/fstab\n", strerror (errno));
	    return NULL;
	}
	if(stat ("/etc/mtab", st+1) < 0) {
	    DBG ("stat(%s): %s", "/etc/mtab\n", strerror (errno));
	    return NULL;
	}
	st->st_mtime += (st+1)->st_mtime;
	st->st_ctime += (st+1)->st_ctime;
	st->st_size += (st+1)->st_size;
	st->st_mode += (st+1)->st_mode;
	st->st_nlink += (st+1)->st_nlink;
	st->st_uid += (st+1)->st_uid;
	st->st_gid += (st+1)->st_gid;
	memcpy(q, st, sizeof(struct stat));
	return GINT_TO_POINTER (1);
    } else if (path == NULL) {
	path="/etc/mtab";
    }
    struct stat st_v;
    struct stat *st = q;
    // if q is NULL, then the function just checks if the stat is possible.
    if(st == NULL) {
        st = &st_v;
    }
    if(!rfm_g_file_test (path, G_FILE_TEST_EXISTS)){
	// this happens when usb item is unplugged, so
	// it merits not a warning....
	// DBG("%s does not exist\n", path);
        return NULL;
    }
    // May block: (fat chance)
    if(stat (path, st) < 0) {
        DBG ("stat(%s): %s\n", path, strerror (errno));
        return NULL;
    }
    return GINT_TO_POINTER (1);
}

static gint 
malloc_items(xfdir_t *xfdir_p, gint count){
    gint first = 1;
    /* up item: */
    xfdir_p->pathc = count + first;
    /* fuse item: */
    if (rfm_void(PLUGIN_DIR, "fuse", "module_active")){
	xfdir_p->pathc++;
	first++;
    }

    xfdir_p->gl = (dir_t *) malloc (xfdir_p->pathc * sizeof (dir_t));
    if (!xfdir_p->gl) g_error("malloc: %s\n", strerror(errno));
    memset (xfdir_p->gl, 0, xfdir_p->pathc * sizeof (dir_t));
    
    xfdir_p->gl[0].en = NULL;
    xfdir_p->gl[0].pathv = g_strdup (g_get_host_name ());
    if (rfm_void(PLUGIN_DIR, "fuse", "module_active")){
	xfdir_p->gl[1].en=rfm_mk_entry(0);
	xfdir_p->gl[1].en->st = NULL;
	xfdir_p->gl[1].en->parent_module = MODULE_NAME;
	xfdir_p->gl[1].en->module = "fuse";
	SET_ROOT_TYPE(xfdir_p->gl[1].en->type);
	gchar *g = rfm_void(PLUGIN_DIR, "fuse", "module_label");
	if (!g) g=g_strdup_printf("FIXME: no module label for \"%s\"", "fuse");
	xfdir_p->gl[1].en->path=g;
	xfdir_p->gl[1].pathv = g_strdup(g);
	// Up icon qualifier (for special icon treatment)
	// SET_UP_TYPE (xfdir_p->gl[1].en->type);
	// Dummy type qualifier (for paper placement at 0,0 treatment)
	SET_DUMMY_TYPE (xfdir_p->gl[1].en->type);
    }
    return first;
}
#if 0

void *
is_mounted_with_timeout (gchar *path) {
    return private_is_mounted(path);
}

#else

// XXX does not do the trick if running under gdb:
// The thread exit will be held back by blockage, 
// and the holding back of the exit will block the main thread...

typedef struct heartbeat_t{
    gboolean condition;
    GMutex *mutex;
    GCond *signal;
    GThread *thread;
    gchar *path;
} heartbeat_t;

static void *
heartbeat_is_mounted(gpointer data){
    heartbeat_t *heartbeat_p = data;
    NOOP("heartbeat_is_mounted: %s\n", heartbeat_p->path);
    // This function call may block
    void *retval = private_is_mounted(heartbeat_p->path);
    
 	    
    g_mutex_lock(heartbeat_p->mutex);
    heartbeat_p->condition = TRUE;
    g_cond_signal(heartbeat_p->signal);
    g_mutex_unlock(heartbeat_p->mutex);
    return retval;

}

static 
void *wait_on_thread(gpointer data){
    heartbeat_t *heartbeat_p = data;
    void *value = g_thread_join(heartbeat_p->thread);

    rfm_mutex_free(heartbeat_p->mutex);
    rfm_cond_free(heartbeat_p->signal);
    g_free (heartbeat_p->path);
    g_free (heartbeat_p);
    return value;
}



G_MODULE_EXPORT
void *
is_mounted_with_timeout (gchar *path) {
    heartbeat_t *heartbeat_p = (heartbeat_t *)malloc(sizeof(heartbeat_t));
    if (!heartbeat_p) g_error("malloc heartbeat_p: %s\n",strerror(errno));
    memset(heartbeat_p, 0, sizeof(heartbeat_t));

    rfm_mutex_init(heartbeat_p->mutex);
    rfm_cond_init(heartbeat_p->signal);
    heartbeat_p->condition = 0;
    heartbeat_p->path = g_strdup(path);

    g_mutex_lock(heartbeat_p->mutex);
    NOOP("Creating thread with wait for heartbeat_is_mounted: %s\n",
	    path);
    // This thread does not affect view nor window and need not
    // be wait controlled.
    heartbeat_p->thread =
	rfm_thread_create ("heartbeat_is_mounted", heartbeat_is_mounted, heartbeat_p, TRUE);
    if (!heartbeat_p->thread) {
	    rfm_time_out(rfm_get_widget("widgets_p"), path);
    }
    if (!heartbeat_p->condition) {
	gint load_timeout = 1;
	if (!rfm_cond_timed_wait(heartbeat_p->signal, heartbeat_p->mutex, load_timeout))

	{
	    g_mutex_unlock(heartbeat_p->mutex);
	    NOOP("dead heartbeat: is_mounted\n");
	    // Dead heartbeat:
	    // Fire off a thread to do the wait free the heartbeat_p.
	    // This does not affect view nor window.
	    rfm_thread_create ("wait_on_thread", wait_on_thread, heartbeat_p, FALSE);
	    NOOP("dead heartbeat: returning NULL\n");
	    rfm_time_out(rfm_get_widget("widgets_p"), path);
	    return GINT_TO_POINTER(-1);
	}
    }
    g_mutex_unlock(heartbeat_p->mutex);
    return (wait_on_thread(heartbeat_p));
}
#endif

